//********************************************************************************
//* File       : FMgr.hpp                                                        *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2005-2025 Mahlon R. Smith, The Software Samurai   *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 22-Jun-2025                                                     *
//* Version    : (see FMgrVersion string in FMgr.cpp)                            *
//*                                                                              *
//* Description: This class implements a wrapper for Linux system file           *
//* management.  Although C++ has an extensive library of file management        *
//* routines, this is more fun AND more educational.                             *
//*                                                                              *
//*                                                                              *
//* Developed using GNU G++ (Gcc v: 4.4.2)                                       *
//*  under Fedora Release 12, Kernel Linux 2.6.31.5-127.fc12.i686.PAE            *
//********************************************************************************
//* Version History (most recent first):                                         *
//*   See version history in FMgr.cpp                                            *
//*                                                                              *
//********************************************************************************
//* Programmer's Notes:                                                          *
//*                                                                              *
//********************************************************************************

#ifndef FMGR_INCLUDED
#define FMGR_INCLUDED

#include <unistd.h>           //* For Linux/UNIX file access
#include <fstream>            //* For C++ file I/O
#include <thread>             //* std::thread definition (and POSIX thread defs)
#include <condition_variable> //* 
#include <chrono>             //* timers for temporarily putting threads to sleep
#include <mutex>              //* for access locks on critical data
#include <exception>          //* for capturing system exceptions
#include <errno.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <wordexp.h>          //* For environment-variable substitution
#include <limits.h>           //* for realpath() and PATH_MAX
#include <stdlib.h>

class tnFName ;               // dummy declaration
#include "FMgrDispData.hpp"   //* Definition of DispData class

//* Definition for callback-method pointer. See FMgrConfigOptions class below. *
typedef void (DMSG)( const char* msg ) ;
//* Flag to enable/disable saving FMgr-class debugging data to a log file *
#define ENABLE_FMgrDebugLog (0)

//* TreeNode Allocation log.                      *
//* TreeNode and tnFName allocation/release log.  *
//* Debugging only. Used to test for memory leaks.*
#define TNA_LOG (0)

//* Encapsulate enhancement under construction for intelligent  *
//* scan recursion to avoid recursing into external filesystems.*
//* Conditional compile will be removed when algorithm stable.  *

const short MAX_PATH  = PATH_MAX ;     //* Size of buffer to hold path/filename
const short MAX_FNAME = 256 ;          //* Size of buffer to hold a filename

//* User name or group name field size *
const short USERNAME_SIZE = 32 ;

//* Super-user (root) user ID *
const uid_t superUID  = (0x0000) ;

//* Root directory path. (during processing, this is often a special case) *
const wchar_t* const ROOT_PATH = L"/" ;
const wchar_t fSLASH = L'/' ;

const wchar_t* const noAccess   = L" (rdacc)" ; // read-access-error message
const wchar_t* const extFileSys = L" (extfs)" ; // external-filesystem message

//* Index value for referencing the top node in a TreeNode list (BASE NODE) *
const UINT BNODE = ZERO ;

//* File info display format options *
enum DispFormat { dfDEFAULT } ;

//* Parameters for file size display field width, Min, Max and Default values *
#define MIN_fsFIELD (8)
#define MAX_fsFIELD (13)
#define DFLT_fsFIELD (10)

//* Size of string-data members for 'fileSystemStats' class.*
//* Also used for 'usbDevice' class.                        *
const short fssLEN = 64 ;        // size of string-data members

//* Local definitions for file type encoded in the "st_mode" element of the stat{} structure *
// NOTE: There are a few other file types (system specific) such as:
//       "high-performance" (contiguous data)
//       "door" and "port" (Solaris)
//       "off-line (migrated)" (Cray)
//       "network special" (HP-UX)
//       The FMgr class handles these as fmUNKNOWN_TYPE.
enum fmFType : short { fmDIR_TYPE,   fmREG_TYPE,  fmLINK_TYPE, fmCHDEV_TYPE, 
                       fmBKDEV_TYPE, fmFIFO_TYPE, fmSOCK_TYPE, fmUNKNOWN_TYPE, fmTYPES } ;

//* Sorting options for directory listings *
enum fmSort : short { fmNO_SORT, fmNAMEr_SORT, fmNAME_SORT, fmDATEr_SORT, fmDATE_SORT, 
                      fmSIZEr_SORT, fmSIZE_SORT, fmTYPEr_SORT, fmTYPE_SORT, 
                      fmEXTr_SORT, fmEXT_SORT, fmSORT_OPTIONS } ;

//* The following c-type structures defined in the standard header files 
//* need to be C++ data types: stat{} dirent{}, etc.
typedef struct stat64 FileStats ;
typedef struct dirent64 deStats ;
typedef struct tm Tm ;
typedef struct passwd PwdInfo ;
typedef struct group GrpInfo ;
typedef struct utimbuf uTime ;

//* Month-name strings for file info formatting *
const wchar_t wMonthString[13][4] =
{
   { L"   " },
   { L"Jan" }, { L"Feb" }, { L"Mar" }, { L"Apr" }, { L"May" }, { L"Jun" },
   { L"Jul" }, { L"Aug" }, { L"Sep" }, { L"Oct" }, { L"Nov" }, { L"Dec" }
} ;

//* Specifies the character or group of characters to be "escaped" *
//* by a call to the EscapeSpecialChars() method.                  *
enum escGroup : short
{
   escDQ,      // escape only double-quote character ( " )
   escMIN,     // escape the four most critical characters: ( $ \ ` " )
   escLINUX,   // escMIN plus the Linux shell special characters ( \ ' " ? : ; & > < | * )
   escMAX,     // escLINUX plus all remaining special characters
               // # + = % ! ~ { } @ ( ) [ ] and space(' ' 20h)
   escWCHAR    // escape only the specified character
} ;

//* Structure for passing configuration data to the FMgr constructor *
class FMgrConfigOptions
{
   public:
   //* Pointer to an array of color attributes,          *
   //* one color for each supported file type (fmTYPES)  *
   attr_t*  ftColors ;
   //* Path specification for directory where temporary  *
   //* files are to be created.                          *
   const char* tfPath ;
   //* Optional path specification for initial start-up  *
   //* directory.                                        *
   const char* stPath ;
   fmSort   sortOption ;      //* File sorting option
   bool     caseSensitive ;   //* Case sensitive/insensitive sort
   bool     showHidden ;      //* Show/hide 'hidden' files
   bool     rootScan ;        //* Enable full root directory scan

   //* Callback method pointer for debugging messages    *
   //* from FMgr class to caller's code.                 *
   DMSG* dmCallback ;
} ;

//* Class for reporting the system local date/time.                            *
//* ---------------------------------------------------------------------------*
#define DECODE_TZ (1)   // if non-zero, decode time-zone fields of localTime
const int64_t maxETIME32 = 0x07FFFFFFF,      // largest 32-bit time_t
              minETIME32 = -maxETIME32 - 1 ; // smallest 32-bit time_t
const short ltFMT_LEN = 16 ;                 // length of time string buffers
//* Important note about the year 2038 bug:                                    *
//* Because 'time_t' is defined as a signed, 32-bit value on 32-bit systems,   *
//* time reporting will break down for date/time values                        *
//*                     >= 19 Jan 2038 at 11:14:08                             *
//* For this reason and for others, we have defined our own local-time class.  *
//* This class uses a signed, 64-bit value for storing the epoch time.         *
//* It is hoped that all Linux systems will have been updated long before this *
//* becomes an issue; however, even now (2013), some applications are using    *
//* date/time manipulations that exceed the 32-bit limit. BEWARE!!             *
//* NOTE: time arithmetic should always be performed on signed 64-bit values.  *
//*                                                                            *
class localTime
{
   public:
   localTime ( void ) { this->reset() ; }
   void reset ( void )
   {  //* Set to epoch date: Thursday, 01 Jan. 1970 @ 08:00:00
      this->sysepoch = ZERO ;
      this->epoch = ZERO ;
      this->day = 4 ;
      this->year = 1970 ;
      this->month = this->date = 1 ;
      this->hours = 8 ;
      this->minutes = this->seconds = ZERO ;
      this->julian = ZERO ;
      #if DECODE_TZ != 0   // time-zone data
      *this->timezone = NULLCHAR ;
      *this->utc_zone  = NULLCHAR ;
      this->gmtoffset = ZERO ;
      this->dst       = false ;
      #endif   // DECODE_TZ
   }
   int64_t  epoch ;        //* Seconds since the epoch (01-Jan-1970)
   time_t   sysepoch ;     //* system epoch time (see note)
   USHORT   day ;          //* Day of the week (0 == Sun, ..., 6 == Sat)
   USHORT   date ;         //* Day of the month (1-[28|29|30|31])
   USHORT   month ;        //* Month of the year (1 == Jan, ... 12 == Dec)
   USHORT   year ;         //* Year (four digit year)
   USHORT   hours ;        //* Time of day, hour (0-23)
   USHORT   minutes ;      //* Time of day, minutes (0-59)
   USHORT   seconds ;      //* Time of day, seconds (0-59)
   USHORT   julian ;       //* Completed days since beginning of year

   #if DECODE_TZ != 0   // time-zone data
   // See DecodeEpochTime method for more information.
   char     timezone[ltFMT_LEN] ; //* string representing the local time zone (useless)
   char     utc_zone[ltFMT_LEN] ; //* string representing the UTC offset for local time zone
   short    gmtoffset ;    //* Expressed as (signed) number of seconds offset from UTC+0:00
   bool     dst ;          //* 'true' if daylight savings time in effect,
                           //* 'false if not DST or info not available
   #endif   // DECODE_TZ
} ;

//* Count of directory and non-directory files in a *
//* node of the filesystem tree structure. Used for *
//* preliminary scan to determine resources needed. *
//* See DirectoryCount() method.                    *
class nodeCount
{
   public:
   ~nodeCount ( void ) {}           // destructor
   nodeCount ( void )               // default constructor
   {
      this->reset() ;
   }
   nodeCount ( UINT currLevel )     // initialization constructor
   {
      this->reset() ;
      this->level = currLevel ;
   }
   void reset ( void )
   {
      this->dirCnt = ZERO ;
      this->regCnt = ZERO ;
      this->level  = ZERO ;
      this->trgPath[0] = NULLCHAR ;
   }

   //*** Data Members ***
   //* Number of subdirectories at this     *
   //* level of the filesystem tree.        *
   UINT dirCnt ;

   //* Number of NON-subdirectory files     *
   //* at this level of the filesystem tree.*
   UINT regCnt ;

   //* Level of subdirectory tree below the *
   //* Current-Working-Directory. (0==CWD)  *
   UINT level ;

   //* Full path of target directory.       *
   char trgPath[MAX_PATH] ;

} ;

//* Class definition for each individual file of the TreeNode class            *
//* An array of these is attached to TreeNode.tnFiles for each directory       *
class tnFName
{
   public:
   ~tnFName ( void )          //* Destructor
   { /* nothing to do */ }
   tnFName ( void )           //* Constructor
   { this->ReInit () ; }

   void ReInit(void)          //* Set all members to default values
   {
      fName[0] = NULLCHAR ;
      fType    = fmTYPES ;
      fBytes   = ZERO ;
      modTime.day = modTime.date = modTime.month = modTime.year 
                  = modTime.hours = modTime.minutes = modTime.seconds = ZERO ;
      readAcc  = false ;
      writeAcc = false ;
      fsMatch  = true ;
      // NOTE: data member, rawStats is left unitialized
   }

   //* Data Members *
   char        fName[MAX_FNAME] ; //* Filename string
   fmFType     fType ;     //* File type (enum fmFType)
   UINT64      fBytes ;    //* File size (in bytes)
   localTime   modTime ;   //* Date/Time file last modified (human readable)
   FileStats   rawStats ;  //* Copy of system's "stat" structure (all file info)
   bool        readAcc ;   //* 'true' if user has read access to file
   bool        writeAcc ;  //* 'true' if user has write access to file
   bool        fsMatch  ;  //* 'true' if file is on the same filesystem as base directory
                           //*        (directories only, ignored for all other file types)
   bool        spare ;     //* (currently unused)
} ;

//* Class definition for a construct used for traversing a subdirectory tree *
class TreeNode
{
public:
   virtual ~TreeNode () ;     //* Destructor
   TreeNode ( void ) ;        //* Constructor
   void ReInit ( void ) ;     //* Reinitialize data members
   void operator += ( const TreeNode& src ) ; // add src data to accumulator members
   void GetActual ( UINT& nlnodes, UINT& tnfcount ) const ; // DEBUG ONLY !!
   void Increment_nlnodes ( void ) ;   // increment the private member (use with caution)
   void Increment_tnfcount ( void ) ;  // increment the private member (use with caution)

   tnFName  dirStat ;   //* Directory file's vital statistics
   UINT     totFiles,   //* Total number of files in directory
            dirFiles,   //* Number of (sub)directory files (fmDIR_TYPE)
            regFiles,   //* Number of regular files (fmREG_TYPE)
            linkFiles,  //* Number of symbolic-link files (fmLINK_TYPE)
            cdevFiles,  //* Number of character-device files (fmCHDEV_TYPE)
            bdevFiles,  //* Number of block-device files (fmBKDEV_TYPE)
            fifoFiles,  //* Number of named-pipe files (fmFIFO_TYPE)
            sockFiles,  //* Number of socket files (fmSOCK_TYPE)
            unkFiles ;  //* Number of unknown-type files (fmUNKNOWN_TYPE)
   UINT64   totBytes,   //* Total number of bytes in directory contents
            dirBytes,   //* Total number of bytes in fmDIR_TYPE files
            regBytes,   //* Total number of bytes in fmREG_TYPE files
            linkBytes,  //* Total number of bytes in fmLINK_TYPE files
            cdevBytes,  //* Total number of bytes in fmCHDEV_TYPE files
            bdevBytes,  //* Total number of bytes in fmBKDEV_TYPE files
            fifoBytes,  //* Total number of bytes in fmFIFO_TYPE files
            sockBytes,  //* Total number of bytes in fmSOCK_TYPE files
            unkBytes ;  //* Total number of bytes in fmUNKNOWN_TYPE files

   UINT     wprotFiles ;   //* Number of write-protected files
   UINT     violations ;   //* Number of access violations encountered during 
                           //* directory scan (read protection, privledge, etc.)
   UINT     allocBytes ;   //* Number of dynamically allocated bytes controlled 
                           //* by this structure (incl. the structure itself)
   tnFName* tnFiles ;      //* Pointer to an array of data structures, one for
                           //* each file in the tree node 
   UINT     tnFCount ;     //* Number of elements in tnFiles array

   short    level ;        //* Depth of node in subdir tree (0 == start node)

   TreeNode*   nextLevel ; //* Pointer to array of node pointers for directories 
                           //* at next lower level (NULL if no more levels)
   TreeNode*   prevLevel ; //* Pointer to parent node (NULL if top of list)

   private:                //* Used by constructor and destructor
   UINT     nlnodes ;      //* Size of array controlled by 'nextLevel' member
   UINT     tnfcount ;     //* Size of array controlled by 'tnFiles' member
friend class FMgr ;        //* FMgr class has full access to class members
} ;

//* Used for sorting file display information *
class ReSort
{
   public:
   ReSort( char*& txPtr, attr_t& cattr, tnFName*& fnPtr ) : 
         tPtr(txPtr), cAttr(cattr), dPtr(fnPtr) {}
   ReSort()
   {
      tPtr  = NULL ;
      cAttr = ZERO ;
      dPtr  = NULL ;
   }
   char*       tPtr ;            // pointer to display text string
   attr_t      cAttr ;           // color attribute
   tnFName*    dPtr ;            // pointer to file's information structure
} ;

const char* const gvfsDriver = "gvfsd-fuse" ;   // virtual device-driver type
const char* const mtpURI  = "mtp://" ;    // initial URI substring
const char* const mtpHost = "mtp:host=" ; // replaces 'mtpURI' in directory name
const char* const optiURI = "://sr" ;     // identifies an optical-drive virtual filesystem
const char* const optiHost = ":host=sr" ; // replaces optiURI
//* Contitional compile flag:                                       *
//*  Set to '0' for production. Captures only useful information.   *
//*  Set to '1' for debugging. Captures comprehensive optical-drive *
//*             info provided by the 'gio' utility.                 *
#define DVD_CAPTURE_ALL (0)

//********     Multi-purpose filesystem information object.     ********
//* 1) Filesystem information for virtual (MTP/GVFS) filesystems.      *
//*    Can be used to collect basic information on any USB device, but *
//*    some fields will be initialized only for mtp/gvfs filesystems.  *
//*    See USB_DeviceStats() method for additional information.        *
//* 2) Filesystem information for optical drives (DVD/CD/Blu-Ray).     *
//*    If the "DVD_CAPTURE_ALL" conditional-compile flag is enabled,   *
//*    additional diagnostic data will be captured.                    *
//*    See the DVD_DeviceStats() method for additional information.    *
class usbDevice
{
// CZONE - CONSIDER MERGING WITH filesystemStats CLASS.
   public:
   usbDevice ( void )         // default constructor
   {
      this->reset () ;
   }
   void reset ( void )        // set all data members to default values
   {
      this->mntpath[0] = NULLCHAR ;
      this->mntdir[0]  = NULLCHAR ;
      this->uri[0]     = NULLCHAR ;
      this->desc[0]    = NULLCHAR ;
      this->devfmt[0]  = NULLCHAR ;
      this->devpath[0] = NULLCHAR ;
      this->fstype[0]  = NULLCHAR ;
      this->uuid[0]    = NULLCHAR ;
      this->label[0]   = NULLCHAR ;
      this->backend[0] = NULLCHAR ;
      this->totalbytes = this->usedbytes = this->freebytes = ZERO ;
      this->bus = this->dev = ZERO ;
      this->vendor = this->product = ZERO ;
      this->readonly = this->remote = this->mounted = this->init = false ;

      //* Flags for optical media only *
      this->optiDrive = this->has_media = this->can_eject = this->can_mount = 
      this->can_unmount = false ;

      #if DVD_CAPTURE_ALL != 0
      //* Optional diagnostic data for optical drives and media *
      this->dfltloc[0] = this->icons[0] = this->sstype[0] = 
      this->sortkey[0] = this->mimetype[0] = NULLCHAR ;
      this->rem = this->mrem = this->mcheck = this->mpoll = 
      this->can_start = this->can_stop = this->automnt = false ;
      #endif   // DVD_CAPTURE_ALL
   }

   //* Public data members *
   char     mntpath[MAX_PATH] ;  // full mountpoint filespec
   char     mntdir[MAX_FNAME] ;  // mountpoint directory name
   char     uri[MAX_FNAME] ;     // Universal Resource Identifier ("activation_root")
   char     desc[fssLEN] ;       // description of attached device
   char     devfmt[fssLEN] ;     // formatted bus/device string ex: "/003/011"
   char     devpath[fssLEN] ;    // full device-driver filespec
   char     fstype[fssLEN] ;     // filesystem type (usually "mtpfs")
   char     uuid[fssLEN] ;       // Universally-Unique-IDdentifier
   char     label[fssLEN] ;      // media-volume description
   char     backend[fssLEN] ;    // gvfs backend processor (usually "mtp")
   uint64_t totalbytes ;         // total bytes of storage space
   uint64_t usedbytes ;          // used storage bytes
   uint64_t freebytes ;          // free (available) storage bytes
   int16_t  bus ;                // USB bus number
   int16_t  dev ;                // USB device number
   uint16_t vendor ;             // vendor (hex)  ID part A  ex: 04c2:01ba
   uint16_t product ;            // product (hex) ID part B
   bool     readonly ;           // 'true' if device mounted as read-only
   bool     remote ;             // 'true' if remote (network) device
   bool     mounted ;            // 'true' if attached device mounted
   bool     optiDrive ;          // 'true' if verified as optical drive
   bool     has_media ;          // 'true' if media inserted
   bool     can_eject ;          // can eject media
   bool     can_mount ;          // can be mounted
   bool     can_unmount ;        // can be un-mounted
   bool     init ;               // 'true' if record initialized

   #if DVD_CAPTURE_ALL != 0
   //* Optional diagnostic data for optical drives and media *
   char     dfltloc[MAX_PATH] ;  // default mountpoint path
   char     icons[fssLEN] ;      // "themed icons" (may contain the word "optical")
   char     sstype[fssLEN] ;     // start/stop type (dynamic or on shutdown)
   char     sortkey[fssLEN] ;    // plug-in-type/fixed-removable/driver-name
   char     mimetype[fssLEN] ;   // "x_content_type"
   bool     rem ;                // removable drive
   bool     mrem ;               // removable media
   bool     mcheck ;             // auto-check for media
   bool     mpoll ;              // can poll for media insertion
   bool     can_start ;          // 'true' if drive can be manually started
   bool     can_stop ;           // 'true' if drive can be manually stopped
   bool     automnt ;            // should auto-mount
   #endif   // DVD_CAPTURE_ALL
} ;   // usbDevice

//* Used as the parameter for GetFilesystemStats() method and is initialized  *
//* with various information about the target filesystem.                     *
class fileSystemStats
{
   public:
   fileSystemStats()
   {
      this->reset() ;
   }

   void reset ( void )
   {
      this->fsType[0] = this->seLinux[0] = 'u' ;
      this->fsType[1] = this->seLinux[1] = 'n' ;
      this->fsType[2] = this->seLinux[2] = 'k' ;
      this->fsType[3] = this->seLinux[3] = NULLCHAR ;
      this->mntPoint[0] = this->uuid[0] = this->label[0] = this->device[0] = NULLCHAR ;
      this->freeBytes = this->usedBytes = this->systemID = ZERO ;
      this->blockTotal = this->blockFree = this->blockAvail = 
      this->inodeTotal = this->inodeAvail = ZERO ;
      this->fsTypeCode = ZERO ;
      this->blockSize = this->fblockSize = ZERO ;
      this->nameLen = ZERO ;
      this->isMounted = this->isMountpoint = false ;
   }

   //****************
   //* Data Members *
   //****************
   char     mntPoint[MAX_PATH] ; // 'lsblk' mountpoint path
   char     uuid[fssLEN] ;       // 'lsblk' UUID (requires 37 bytes, 49 for gvfs)
   char     fsType[fssLEN] ;     // 'lsblk' filesystem type string
   char     label[fssLEN] ;      // 'lsblk' disc label
   char     device[fssLEN] ;     // 'df' device name (block-device-driver path)
   char     seLinux[fssLEN] ;    // 'stat' format codes, %C
   UINT64   systemID ;           // %i (hex)
   UINT64   freeBytes ;          // (calculated: blockAvail * fblockSize)
   UINT64   usedBytes ;          // (calculated: (blockTotal - blockFree) * fblockSize)
   ULONG    blockTotal ;         // %b
   ULONG    blockFree ;          // %f
   ULONG    blockAvail ;         // %a
   ULONG    inodeTotal ;         // %c
   ULONG    inodeAvail ;         // %d
   UINT     fsTypeCode ;         // %t (hex)
   UINT     blockSize ;          // %s
   UINT     fblockSize ;         // %S
   USHORT   nameLen ;            // %l
   bool     isMounted ;          // 'true'  target drive is mounted
                                 //         i.e. drive info available
                                 // 'false' drive info not available
                                 // (See notes in FileSytemStats() method.)
   bool     isMountpoint ;       // 'true'  target file == mountpoint
                                 // 'false' target file != mountpoint 
                                 //         (or insufficient information)
} ;   // (fileSystemStats)

//* Information on the user account under which the application is running. *
const short uiMAX_sGID = 32 ;
const short uiMAX_STRING = 1024 ;
class UserInfo
{
   public:
   UserInfo ()
   {
      *stringAlloc = NULLCHAR ;
      userName = realName = shellProg = homeDir = 
      passWord = grpName = grpPWord = stringAlloc ;
      userID = grpID = ggrpID = (-1) ;
      suppGroups = ZERO ;
      goodData = false ;
   }
   char*   userName ;         //* User name
   char*   realName ;         //* User's real name
   char*   grpName ;          //* Group name
   char*   shellProg ;        //* User's shell program
   char*   homeDir ;          //* Path to user's home directory
   char*   passWord ;         //* User's password (encrypted)
   char*   grpPWord ;         //* Group password (encrypted)
   gid_t   suppGID[uiMAX_sGID];//* List of supplimentary groups belonged to 
                              //* (first uiMAX_sGID only, probably far more)
                              //* than enough except for 'root' user       )
   uid_t    userID ;          //* User ID
   gid_t    grpID ;           //* User's group ID
   gid_t    ggrpID ;          //* Group ID from Group file
   short    suppGroups ;      //* Number of supplimentary groups that user belongs to
   bool     goodData ;        //* true if data are valid, else false

   private:
   char  stringAlloc[uiMAX_STRING];//* Memory for storing string data
friend class FMgr ;     // FMgr class has full access to class members
} ;

//* Decoded file protection bits structure *
struct FileProt
{
   char        read ;               //* 'r' if bit set, else '-'
   char        write ;              //* 'w' if bit set, else '-'
   char        exec ;               //* 'x' if bit set, else '-'
} ;
//* Expanded and formatted version of the tnFName class. *
//* Used for inspection and display of file stats.       *
class ExpStats
{
   public:
   char        fileName[MAX_FNAME] ;      //* Filename string
   char        filePath[MAX_PATH] ;       //* Path string (excl. filename)
   char        ownerName[USERNAME_SIZE] ; //* File owner's name
   char        groupName[USERNAME_SIZE] ; //* File group's name
   uint64_t    fileSize ;                 //* File size in bytes
   fmFType     fileType ;                 //* File type, member of enum fmFType
   uint64_t    iNode ;                    //* File's inode number (unique id)
   uint64_t    hLinks ;                   //* Number of hard links
   uid_t       userID ;                   //* User's (owner's) ID (32 bits)
   gid_t       grpID ;                    //* Group's ID          (32 bits)
   bool        setUID ;                   //* Set UID bit on execution
   bool        setGID ;                   //* Set GID bit on execution
   bool        sticky ;                   //* Restricted-deletion flag (only owner can delete)
   FileProt    usrProt ;                  //* User (owner) file-protection bits
   FileProt    grpProt ;                  //* Group file-protection bits
   FileProt    othProt ;                  //* Others file-protection bits
   localTime   modTime ;                  //* Date/Time file last modified
   localTime   accTime ;                  //* Date/Time file last accessed
   localTime   staTime ;                  //* Date/Time file status last modified
   FileStats   rawStats ;                 //* Copy of system's "stat" structure (all file info)
   ExpStats*   symbTarg ;     //* If current file is a symbolic link, this pointer references 
                              //* the stats for the link-target file, else NULL. 
} ;



//***************************
//** FMgr class definition **
//***************************
class FMgr
{
public:
   virtual ~FMgr () ;                        //* Destructor

//* Default constructor.                                                       *
//*                                                                            *
//* Input  : fileTypeColors: pointer to an array of color attributes,          *
//*           one attribute for each supported file type.                      *
//*           See enum fmFType in FMgr.hpp for supported file types.           *
//*           See NCurses.hpp for a list of color attributes. Be sure that     *
//*           the colors selected do not conflict with the 'select',           *
//*           'copy', or 'cut' colors.                                         *
//* Returns: nothing (but constructor returns a pointer to class object)       *
FMgr ( FMgrConfigOptions& cfg ) ;

//* Set the current-working directory (CWD). This method does not update the   *
//* data stored in the class members. It simply changes the CWD.               *
//*                                                                            *
//* Input  : newCwd : path of new directory                                    *
//*          oldCwd : (optional, NULL pointer by default)                      *
//*                   if specified, receives the old (previous) CWD            *
//* Returns: OK  if successful                                                 *
//*          ERR if target directory does not exist or is inaccessible         *
short SetCWD ( const char* newCwd, gString* oldCwd = NULL ) ;
short SetCWD ( const gString& newCwd, gString* oldCwd = NULL ) ;

//* Returns the path of user's current working directory.                      *
//* This is not necessarily the directory associated with the data stored in   *
//* our class members.                                                         *
//*                                                                            *
//* Input  : cdir : pointer to buffer to hold path string                      *
//*                 (buffer must be >= MAX_PATH bytes)                         *
//* Returns: OK if successful                                                  *
//*          ERR if target directory does not exist or is inaccessible         *
short GetCWD ( gString& cdir ) ;
short GetCWD ( char* cdir ) ;

//* Returns a copy of the path string associated with the directory data       *
//* stored in our class data members.                                          *
//*                                                                            *
//* Input  : cdir : pointer to buffer to hold path string                      *
//*                 (must be >= MAX_PATH bytes)                                *
//* Returns: nothing                                                           *
void  GetCurrDir ( char* cdir ) ;
void  GetCurrDir ( gString& cdir ) ;
   
//* Re-read the current directory, format and sort the display data.           *
//*                                                                            *
//* Input  : dData: (by reference) partially initialized instance of the       *
//*                 DispData class:                                            *
//*                  dData.strWidth must contain the number of display columns *
//*                  for the display strings that will be created.             *
//* Returns: If successful, returns 'true' and on return, the remaining        *
//*            members of the DispData class object will have been             *
//*            initialized.                                                    *
//*          If operation failed, returns 'false' dData.fileCount == ZERO      *
//*            and remaining members of DispData class object will be          *
//*            NULL pointers and ZERO values.                                  *
bool  RefreshCurrDir ( DispData& dData ) ;

//* Scan the directory tree, starting at the specified directory.              *
//* The scan proceeds from the specified directory, downward, scanning each    *
//* subdirectory in the tree and capturing data on each directory and the      *
//* files they contain. (with certain exceptions)                              *
//*                                                                            *
//* Produce a linked list of TreeNode class objects containing a summary of    *
//* the data for each directory in the tree.                                   *
//*                                                                            *
//* This method uses a multi-threaded approach to reading the directory tree.  *
//*                                                                            *
//* IMPORTANT NOTE: If called, this method allocates persistent memory.        *
//*                 It is the caller's responsibility to release that memory   *
//*                 when it is no longer needed (see ReleaseDirTree()).        *
//*                                                                            *
//*                                                                            *
//* Input  : dPath: full path specification of top-level directory             *
//*          dData: (by reference) DispData-class object for controlling       *
//*                  access to shared data.                                    *
//*          recurse: (optional, true by default)                              *
//*                 if 'true',  scan all files and directories in the tree     *
//*                 if 'false', scan only the top-level contents of the        *
//*                             specified directory                            *
//*                      Programmer's Note: See also data member 'irRootscan'. *
//*          dironly: (optional, default: false)                               *
//*                 if 'false', collect data for all objects, both filenames   *
//*                             and directory names                            *
//*                 if 'true',  collect directory names only                   *
//*                             Note: A count of non-directory files returned  *
//*                                   but no detail for the individual files.  *
//*                                                                            *
//* Returns: pointer to the head of a doubly-linked list of TreeNode class     *
//*          objects containing data on each directory in the tree and the     *
//*          files contained in those directories.                             *
//*          NOTE: Returns a NULL pointer if target is not a directory name    *
//*           or target does not exist or user does not have read permission   *
TreeNode* CaptureDirTree ( const gString& dPath, DispData& dData, 
                           bool recurse = true, bool dironly = false ) ;

//* Change directory to a child directory below current directory.             *
//*                                                                            *
//* Input  : dPath: (by reference)                                             *
//*                 Initially contains name of target child directory.         *
//*                 On return, will contain current-working-directory path.    *
//*          dData: (by reference) partially initialized instance of the       *
//*                 DispData class:                                            *
//*                  dData.strWidth must contain the number of display columns *
//*                  for the display strings that will be created.             *
//* Returns: If successful, returns 'true' and the remaining members of the    *
//*            DispData class object will have been initialized.               *
//*          If operation fails, returns 'false', dData.fileCount == ZERO      *
//*            and remaining members of DispData class object will be          *
//*            NULL pointers and ZERO values.                                  *
bool  CdChild ( gString& dPath, DispData& dData ) ;

//* Change directory to parent directory above current directory.              *
//* If already at 'root' directory, simply returns a copy of current data.     *
//*                                                                            *
//* Input  : dPath: (by reference, initial value ignored)                      *
//*                 On return, will contain current-working-directory path.    *
//*          dData: (by reference) partially initialized instance of the       *
//*                 DispData class:                                            *
//*                  dData.strWidth must contain the number of display columns *
//*                  for the display strings that will be created.             *
//* Returns: If successful, returns 'true' and the remaining members of the    *
//*            DispData class object will have been initialized.               *
//*           If operation fails, returns 'false', dData.fileCount == ZERO     *
//*            and remaining members of DispData class object will be          *
//*            NULL pointers and ZERO values.                                  *
bool  CdParent ( gString& dPath, DispData& dData ) ;

//* Change directory to directory specified by full path string.               *
//*                                                                            *
//* Input  : dPath: full path specification for new target directory           *
//*          dData: (by reference) partially initialized instance of the       *
//*                 DispData class:                                            *
//*                  dData.strWidth must contain the number of display columns *
//*                  for the display strings that will be created.             *
//* Returns: If successful, returns 'true' and the remaining members of the    *
//*            DispData class object will have been initialized.               *
//*           If operation fails, returns 'false', dData.fileCount == ZERO     *
//*            and remaining members of DispData class object will be          *
//*            NULL pointers and ZERO values.                                  *
bool  CdPath ( gString& dPath, DispData& dData ) ;

//* Perform a 'stat' ('lstat') on the file specified by the full path string.  *
//* The specified file may, or may not be in the current file list.            *
//* By default the stats for the specified file are returned, but if           *
//* getTarg != false && target is a symbolic link, the symbolic link is        *
//* followed i.e. stats for link target are returned.                          *
//*                                                                            *
//* Input  : tnFile : tnFName class object (by reference) to hold 'stat' data  *
//*          fPath  : full path/filename specification                         *
//*          getTarg: (optional, false by default) if true, symbolic links are *
//*                   followed. Else, do not follow links                      *
//* Returns: OK if successful, all data members of tnFile will be initialized  *
//*          Else returns system 'errno' value                                 *
//*           If error during system call:                                     *
//*            - tnFile.readAcc == false and all other members are undefined   *
//*             Exception:                                                     *
//*             If 'getTarg' != false AND error is a broken symbolic link      *
//*             - tnFile.fName == name of link target if known, else "unknown" *
//*             - tnFile.readAcc == false                                      *
//*             - tnFile.fType == fmTYPES                                      *
//*             - All other members of 'tnFile' are undefined                  *
short GetFileStats ( tnFName& tnFile, const gString& fPath, bool getTarg = false ) ;

//* For a symbolic link source file, return the stat information for the       *
//* file the sym-link points to i.e. the sym-link target.                      *
//*                                                                            *
//* Input  : lnkPath  : full path/filename specification of symbolic link file *
//*          ltrgStats: (by reference, initial contents ignored)               *
//*                     tnFName class object (by reference) to hold 'stat' data*
//*          ltrgPath : (by reference, initial contents ignored)               *
//*                     receives absolute path/filename of link-target file    *
//*                                                                            *
//* Returns: OK if successful,                                                 *
//*            - ltrgPath gets the absolute path/filename of link target file. *
//*            - all data members of ltrgStats will be initialized             *
//*          Else if specified source file is not a symbolic link, the errno   *
//*            code EINVAL (invalid argument) is returned.                     *
//*            - ltrgPath is initialized to "unknown"                          *
//*            - ltrgStats.readAcc == false and all other members are undefined*
//*          Else system 'errno' value is returned                             *
//*            If error during system call:                                    *
//*            - ltrgPath is initialized to "unknown"                          *
//*            - ltrgStats.readAcc == false and all other members are undefined*
//*               Exception:                                                   *
//*               If the cause of the error is a broken symbolic link i.e.     *
//*               (link target file not found), then:                          *
//*               - ltrgPath == path/filename of missing link target if known, *
//*                 else "unknown"                                             *
//*               - ltrgStats.fName == name of missing link target if known,   *
//*                 else "unknown"                                             *
//*               - ltrgStats.readAcc == false                                 *
//*               - ltrgStats.fType == fmTYPES                                 *
//*               - All other members of ltrgStats are undefined               *
short GetLinkTargetStats ( const gString& lnkPath, tnFName& ltrgStats, gString& ltrgPath ) ;

//* Expand and format for human readability the 'stat' info for the file       *
//* specified by fs structure.                                                 *
//*                                                                            *
//* Input  : fs   : pointer to FileStats structure (stat64) returned by        *
//*                 the GNU C library stat64() and lstat64() functions.        *
//*                 (Note: for displayed files and files on the    )           *
//*                 (clipboard, this structure is the 'rawStats'   )           *
//*                 (member of the tnFName class object.           )           *
//*          es   : ExpStats structure for holding expanded statistics         *
//*                 a) caller initializes es.fileName with the name of         *
//*                    the target file.                                        *
//*                 b) caller (optionally) initializes es.filePath with        *
//*                    the path string for the file (excluding filename)       *
//*                    (This is needed to 'stat' a symbolic link target.)      *
//*                 c) if the specified file is a symbolic link,               *
//*                    caller (optionally) initializes es.symbTarg             *
//*                    referencing a second ExpStats structure to              *
//*                    hold the info for the symbolic link's target.           *
//*                 (other fields of es initialized on return)                 *
//* Returns: OK if successful, else ERR                                        *
//*             ERR indicates a problem with the symbolic link target:         *
//*               a) invalid es.filePath, or                                   *
//*               b) specified file is a symbolic link, but es.symbTarg        *
//*                  does not point to an ExpStats structure to hold           *
//*                  target file's information, or                             *
//*               c) broken symbolic link                                      *
//*                  Note: if ERR returned, and cause of error was a           *
//*                   broken symbolic link, then                               *
//*                es.symbTarg->fileType == fmTYPES,                           *
//*                es.symbTarg->fileName == filename of missing target         *
//*                                         if available, else "unknown"       *
//*                es.symbTarg->filePath == path of missing target             *
//*                                         if available, else "unknown"       *
//*                   and all other fields of es.symbTarg are undefined.       *
short ExpandStats ( const FileStats* fs, ExpStats& es ) ;

//* Expand and format for human readability the 'stat' info for the file       *
//* specified by tnFName class object.                                         *
//*                                                                            *
//* Same as above, except that source data are from an initialized tnFName     *
//* class object AND that es.fileName need not be initialized                  *
//*                                                                            *
//* Input  : tf : pointer to initialized tnFName class object.                 *
//* Returns: OK if successful, else ERR                                        *
short ExpandStats ( const tnFName* tf, ExpStats& es ) ;

//* Convert GNU/UNIX epoch time code (seconds since the epoch: Jan. 01, 1970)  *
//* to local time in a structured format (class localTime).                    *
//* See the info page for the 'stat' command for more information.             *
//* See also the notes in DirEntryDisplayFormat().                             *
//*                                                                            *
//* Input  : eTime  : epoch time (this is a 64-bit value)                      *
//*                   (See note in definition of localTime class.)             *
//*          ftTime : (by reference, initial values ignored)                   *
//* Returns: nothing, but all members of ftTime will be initialized            *
void  DecodeEpochTime ( int64_t eTime, localTime& ftTime ) ;

//* Convert human-readable time to GNU/UNIX epoch time code:                   *
//* (seconds since the epoch: Jan. 01, 1970)                                   *
//*                                                                            *
//* Input  : ftTime : (by reference)                                           *
//*                   members 'sysepoch' and 'epoch' are initialized           *
//* Returns: copy of member 'epoch'                                            *
int64_t EncodeEpochTime ( localTime& ftTime ) ;

//* Set file permissions for the specified file: USR rwx, GRP rwx, OTHER rwx,  *
//* sUID, sGID, and sticky bit.                                                *
//* (requires either superuser access or ownership of the file.)               *
//* See <sys/stat.h> and <bits/stat.h> for mask definitions.                   *
//* NOTE: For a symbolic link source file, setting or resetting permissions    *
//*       always occurs for the link target, not the link itself.              *
//*                                                                            *
//* Input  : pathBuff: full path/filename specification                        *
//*          newMode : encoded bit word (unsigned int)                         *
//* Returns: OK if successful, else code from 'errno'.                         *
short SetFilePermissions ( const char* PathFilename, mode_t newPermissions ) ;

//* Set file Modify and/or Access date/timestamps.                             *
//* As a side effect, cTime will be set to current system time.                *
//* (requires file write permission)                                           *
//* Note: date string format: "dd mmm yyyy hh:mm:ss"                           *
//*                                                                            *
//* Input  : pathBuff  : full path/filename specification                      *
//*          ft        : structure must be correctly initialized               *
//*                      (except for 'day' which is ignored)                   *
//*          updateMod : (optional, 'true' by default)                         *
//*                      if true, update modification time                     *
//*          updateAcc : (optional, 'true' by default)                         *
//*                      if true, update access time                           *
//* Returns: OK if successful, else code from 'errno'                          *
short SetFileTimestamp ( const char* PathFilename, const localTime& newDate, 
                         bool updateMod = true, bool updateAcc = true ) ;

//* Set file owner (userID) and/or file group (grpID).                         *
//* Requires superuser access to set owner.                                    *
//* Requires either superuser access or ownership of the file to set group.    *
//* (For owner to set group, owner must be a member of the target group.)      *
//* As a side effect, cTime will be set to current system time.                *
//*                                                                            *
//* Input  : pathBuff: full path/filename specification                        *
//*          newOwner: new User ID   (if -1, then no change)                   *
//*          newGroup: new Group ID  (if -1, then no change)                   *
//* Returns: OK if successful, else code from 'errno'.                         *
short SetFileOwnerGroup ( const char* PathFilename, uid_t newOwner, gid_t newGroup ) ;

//* Determine whether user has 'read' permission for the specified file,       *
//* and optionally enable owner/group read permission.                         *
//*                                                                            *
//* IMPORTANT NOTE:                                                            *
//* True read permission on a directory name actually requires BOTH 'Read' and *
//* 'Execute' permission.                                                      *
//* a) 'Execute' only, will allow us to set the directory to CWD, but          *
//*    will not allow us to read its contents.                                 *
//* b) 'Read' only, will allow us to read the NAMES of the files contained in  *
//*    the directory (but NOT the other stats), but will not allow us to set   *
//*    the directory as the CWD.                                               *
//* Therefore, if caller wants a report of read access on a directory name,    *
//* we indicate read access ONLY if user has both 'Read' and 'Execute'         *
//* permission. If caller wants to enable read access on a directory name, we  *
//* enable BOTH 'Read' and 'Execute' permissions.                              *
//*                                                                            *
//* Input : sStats: (by reference) stats of file to be tested                  *
//*                 sStats.readAcc will be set/reset according to result       *
//*                 and if 'enable', then st_mode bits will be updated,        *
//*                 other members remain unchanged                             *
//*         sPath : path/filename of file to be tested                         *
//*         enable: (optional, false by default)                               *
//*                 if 'true', enable 'owner' and 'group' read access          *
//*                 if 'false', simply report user's current read access       *
//* Returns: 'true' if user has read permission, else 'false'                  *
bool  ReadPermission ( tnFName& sStats, const gString& sPath, bool enable = false ) ;
bool  ReadPermission ( tnFName& sStats, const char* sPath, bool enable = false ) ;

//* Determine whether user has 'write' permission for the specified file,      *
//* and optionally enable owner/group write permission.                        *
//*                                                                            *
//* Input : sStats: (by reference) stats of file to be tested                  *
//*                 sStats.writeAcc will be set/reset according to result      *
//*                 and if 'enable', then st_mode bits will be updated,        *
//*                 other members remain unchanged                             *
//*         sPath : path/filename of file to be tested                         *
//*         enable: (optional, false by default)                               *
//*                 if 'true', enable 'owner' and 'group' write access         *
//*                 if 'false', simply report user's current write access      *
//* Returns: 'true' if user has write permission, else 'false'                 *
bool  WritePermission ( tnFName& sStats, const gString& sPath, bool enable = false ) ;
bool  WritePermission ( tnFName& sStats, const char* sPath, bool enable = false ) ;

//* Determine whether user has 'execute' permission for the specified file,    *
//* and optionally set file as executable for owner/group/others.              *
//* This method follows symbolic links and reports or modifies the link-target *
//* file (if it exists).                                                       *
//*                                                                            *
//* Input : sPath : path/filename of file to be modified                       *
//*         enable: if 'true', enable 'execute' permission                     *
//*                 if 'false', simply report current 'executable' status      *
//* Returns: 'true' if user has execute permission, else 'false'               *
bool  ExecutePermission ( const gString& sPath, bool enable ) ;

//* Write-protect the specified file. Resets owner/group/others                *
//* write-permission bits.                                                     *
//*                                                                            *
//* Input : sPath : path/filename of file to be modified                       *
//*         sStats: (by reference) stats of file to be modified                *
//*                 if successful, sStats.rawStats.st_mode will be updated and *
//*                 sStats.writeAcc will be reset                              *
//* Returns: OK if successful, else 'errno' code                               *
short WriteProtect ( const gString& sPath, tnFName& sStats ) ;

//* Format a file-display text string based upon file information in its       *
//* tnFName class object.                                                      *
//* Note that maxWidth parameter is assumed to be at least large enough for    *
//* the filesize, date, time, spaces and an 8.3 filename format.               *
//* For the default formula this would be 11 + 8 + 5 + 13 = 37 columns.        *
//*                                                                            *
//* Input  : fnptr    : pointer to a tnFName object containing                 *
//*                     information to be displayed                            *
//*          fmtText  : receives formatted display data                        *
//*          maxWidth : maximum display columns                                *
//*          fmt      : format option                                          *
//*          fullWidth: (optional, 'true' by default)                          *
//*                     if true, pad with trailing spaces until maxWidth       *
//* Returns: nothing                                                           *
void  DirEntryDisplayFormat ( tnFName* fnptr, gString& fmtText, short maxwidth, 
                              DispFormat fmt, bool fullWidth = true ) ;

//* Create a copy of the specified source file.                                *
//* Preserves all permission bits and date/timestamps.                         *
//* WARNING: Do not call this method to copy a 'special' file. (see below)     *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*          followLink: (optional, default==false)                            *
//*                      If source is a symbolic link the following            *
//*                      applies; else the 'followLink' has no effect.         *
//*                      if false, copy the link itself                        *
//*                      if true, copy the file link points to                 *
//* Returns: OK if successful, else returns errno value                        *
short CopyFile ( const char* srcPath, const char* trgPath, bool followLink = false ) ;

//* Create a copy of the specified Directory-type source file.                 *
//* Directories are a special case in that:                                    *
//*  - if target directory does not exist, create the directory, preserving    *
//*    all the source file's permission bits and both 'modified' and 'access'  *
//*    date/timestamps are set to source file's mod date/time.                 *
//*  - if a target file of the same name already exists, do not overwrite it.  *
//*    - if the existing target file is of Directory-type AND user has write   *
//*      access to it, do nothing and report that file was copied successfully.*
//*    - if the existing target file is of Directory-type BUT user does not    *
//*      have write access to it,                                              *
//*      return 'EACCES - File write protected'                                *
//*    - if existing target file is not Directory-type,                        *
//*      return 'ENOTDIR - Not a directory'                                    *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//* Returns: OK if successful, else returns errno value                        *
short CopyDirName ( const char* srcPath, const char* trgPath ) ;

//* Note: A standard 'copy' usually cannot be performed for 'special' files.   *
//* Therefore, instead of 'copying' the file, these methods create a new file  *
//* with the same permission and timestamp atributes as the source file.       *
//* The user will be the 'owner' and 'group' of the new file.                  *
//* These operations are not fool-proof - don't expect miracles.               *
//*                                                                            *
//* Create a copy of the specified FIFO (named pipe) file.                     *
short CopyFifo ( const char* srcPath, const char* trgPath ) ;
//* Create a copy of the specified Character Device file. (NOT YET IMPLEMENTED)*
short CopyCharDevice ( const char* srcPath, const char* trgPath ) ;
//* Create a copy of the specified Block Device file.  (NOT YET IMPLEMENTED)   *
short CopyBlockDevice ( const char* srcPath, const char* trgPath ) ;
//* Create a copy of the specified Socket file.  (NOT YET IMPLEMENTED)         *
short CopySocket ( const char* srcPath, const char* trgPath ) ;
//* Create a copy of the specified Unsupported-type file.                      *
short CopyUnsupportedType ( const char* srcPath, const char* trgPath ) ;

//* Create a symbolic link (shortcut file) referencing the source file.        *
//* A symbolic link can reference a file of any type.                          *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//* Returns: OK if successful, else returns errno value                        *
short CreateSymbolicLink ( const char* srcPath, const char* trgPath ) ;

//* Create a 'hard link' to the source file.                                   *
//* A hard link can be created for a file of any type EXCEPT a Directory.      *
//* A hard link to a symbolic-link source makes no sense - don't do it.        *
//* A hard link cannot generally be made across filesystems.                   *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//* Returns: OK if successful, else returns errno value                        *
short CreateHardLink ( const char* srcPath, const char* trgPath ) ;

//* Delete (unlink) the specified source file.                                 *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//* Returns: OK if successful, else returns errno value                        *
short DeleteFile ( const char* srcPath ) ;

//* Delete the specified directory. (directory must be empty)                  *
//*                                                                            *
//* Input  : dirPath : full path specification of source                       *
//* Returns: OK if successful, else returns errno value                        *
short DeleteEmptyDir ( const char* dirPath ) ;

//* Rename the specified source file to specified destination file.            *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//* Returns: OK if successful, else returns errno value                        *
short RenameFile ( const char* srcPath, const char* trgPath ) ;

//* Set the directory-sort option, and re-sort the existing display data.      *
//* If there is currently file-display data, sort it according to the new      *
//* sort option. (Does not update the display.)                                *
//*                                                                            *
//* Input  : sOption: member of enum fmSort                                    *
//* Returns: OK if valid input, else ERR                                       *
short SetSortOption ( fmSort sort_option ) ;

//* Returns the current sort option.                                           *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: member of enum fmSort                                             *
fmSort GetSortOption ( void ) ;

//* Set or reset the case-sensitive flag for sorting display data              *
//* alphabetically, and re-sort the existing display data (if any).            *
//* THIS METHOD DOES NOT RE-DRAW THE DISPLAY.                                  *
//* This flag affects the following sort options: fmNAME_SORT, fmNAMEr_SORT,   *
//* fmEXT_SORT, fmEXTr_SORT.                                                   *
//*                                                                            *
//* Input  : 'true' for case-sensitive sort (default on instantiation)         *
//*          'false' for case-insensitive sort                                 *
//* Returns: OK                                                                *
short CaseSensitiveAlphaSort ( bool caseSensitive ) ;

//* Create a new sub-directory in the current directory.                       *
//*                                                                            *
//* Input  : buffer containing name of new directory                           *
//* Returns: OK if successful, else errno value                                *
short CreateSubDirectory ( const char* dirName ) ;

//* Create a new sub-directory corresponding to the specified path/filename.   *
//*                                                                            *
//* IMPORTANT NOTE: If the target filesystem is an MTP/GVFS virtual filesystem,*
//* then call CreateDirectory_gvfs() instead.                                  *
//*                                                                            *
//* Input  : buffer containing path/filename of new directory                  *
//* Returns: OK if successful, else errno value                                *
short CreateDirectory ( const char* dirPath ) ;

//* Scans the contents of the specified directory and returns 'true' if        *
//* directory contains no files (other than "." and "..").                     *
//*                                                                            *
//* Input  : dirPath: full path of directory to be scanned                     *
//* Returns: 'true' if empty directory, file is not a directory, OR            *
//*            file not found                                                  *
//*          else 'false'                                                      *
bool  isEmptyDir ( const char* dirPath ) ;

//* Returns the width setting for the filesize display field.                  *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: returns width of display field                                    *
short GetFileSizeFieldWidth ( void ) ;
 
//* Set the width of the filesize display field. If invalid input value,       *
//* value is unchanged and ERR is returned.                                    *
//*                                                                            *
//* Input  : field width (range: 8 through 13)                                 *
//* Returns: returns OK if successful, else ERR                                *
short SetFileSizeFieldWidth ( short width ) ;

//* Enable comma-separation formatting for the filesize display field.         *
//* (default==enabled)                                                         *
//*              formatting off:  1234567890                                   *
//*              formatting on :  1,234,567,890                                *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK                                                                *
short EnableFileSizeFormatting ( void ) ;

//* Disable comma-separation formatting for the filesize display field.        *
//*              formatting off:  1234567890                                   *
//*              formatting on :  1,234,567,890                                *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: OK                                                                *
short DisableFileSizeFormatting ( void ) ;

//* Test the specified character to determine whether it may be used as a      *
//* character in a filename or directory name.                                 *
//*                                                                            *
//* Input  : character to test (wchar_t)                                       *
//* Returns: true if acceptable, else false                                    *
bool  IsFNameChar ( wchar_t ch ) ;

//* Set the flag indicating whether 'hidden' files will be reported in the     *
//* file-display list.                                                         *
//* Will take effect the next time a directory is read.                        *
//*                                                                            *
//* Input  : true  = show hidden files                                         *
//*          false = omit hidden files from the list (default)                 *
//* Returns: OK                                                                *
short ShowHiddenFiles ( bool show ) ;

//* Get a summary of the contents of the specified directory.                  *
//* Summary contains: the number of files and subdirectories in the            *
//* directory and the combined size (in bytes) of all files and                *
//* subdirectories in the specified top-level directory (but see 'recurse'     *
//* parameter, below.)                                                         *
//*                                                                            *
//* Input  : dPath : full path specification of directory to be scanned        *
//*          fCount (by reference): initial value ignored                      *
//*          totBytes (by reference): initial value ignored                    *
//*          recurse (optional, 'false' by default)                            *
//*            if 'true' read specified directory and contents of all          *
//*            subdirectories it contains                                      *
//* Returns: 'true' if directory read successfully                             *
//*                 fCount and totBytes are initialized                        *
//*          'false' if directory does not exist or no read access or path     *
//*                 links not resolved. fCount and totBytes == ZERO            *
bool  DirectorySummary ( const char* dPath, UINT& fileCount, 
                         UINT64& totalBytes, bool recursiveRead = false ) ; 

//* Walk through the previously-captured data for the specified TreeNode, whose*
//* base directory is indicated by tnfPtr. Collect file count and file size    *
//* information for the entire node list, INCLUDING the directory itself.      *
//*                                                                            *
//* Input  : tnfPtr   : pointer to tnFName object describing base directory    *
//*          fileCount: (by reference): initial value ignored                  *
//*                     on return, contains number of files in the node        *
//*          totBytes : (by reference): initial value ignored                  *
//*                     on return, contains combined size of all files in node *
//* Returns: 'true' if count was successful                                    *
//*                 fileCount and totalBytes are initialized                   *
//*          'false' if specified tnFName object does not describe a directory *
//*                 or directory data is not in displayed-file list            *
//*                 fileCount and totalBytes == ZERO                           *
bool  TreeNodeSummary ( tnFName* tnfPtr, UINT& fileCount, UINT64& totalBytes ) ;

//* Walk through the previously-captured data beginning at the specified       *
//* TreeNode object, and collect file count and file size information for      *
//* the entire node list.                                                      *
//*                                                                            *
//* Input  : tnPtr    : pointer to TreeNode object                             *
//*          fCount   : file-count accumulator (by reference)                  *
//*          tBytes   : file-size accumulator (by reference)                   *
//*                                                                            *
//* Returns: 'true'   (fCount and tBytes have been updated)                    *
bool  TreeNodeSummary ( const TreeNode* tnPtr, UINT& fCount, UINT64& tBytes ) ;

//* Walk through the previously-captured data beginning at the specified       *
//* TreeNode object, and collect accumulated information for the entire        *
//* node list.                                                                 *
//* The following data members of the accumulator node receive the totals.     *
//* (remaining data members are set to default values)                         *
//* totFiles   totBytes  |  cdevFiles  cdevBytes  |  unkFiles    unkBytes      *
//* dirFiles   dirBytes  |  bdevFiles  bdevBytes  |  wprotFiles                *
//* regFiles   regBytes  |  fifoFiles  fifoBytes  |  violations                *
//* linkFiles  linkBytes |  sockFiles  sockBytes  |  allocBytes                *
//*                                                                            *
//* Input  : tnPtr    : pointer to TreeNode object                             *
//*          tnAcc    : TreeNode instance to act as data accumulator           *
//* Returns: 'true'   (tnAcc accumulator members have been updated)            *
bool  TreeNodeSummary ( const TreeNode* tnPtr, TreeNode& tnAcc ) ;

//* Create a copy of the TreeNode list beginning with the base directory which *
//* is indicated by tnfPtr.                                                    *
//*                                                                            *
//* Input  : tnfPtr : pointer to tnFName object describing base directory      *
//*          tnBase : TreeNode object (by reference) which will act as the     *
//*                   base node for the new list.                              *
//* Returns: 'true' if count was successful                                    *
//*                 fileCount and totalBytes are initialized                   *
//*          'false' if specified tnFName object does not describe a directory *
//*                 or directory data is not in displayed-file list or memory- *
//*                 allocation error.                                          *
bool  CopyDirTree ( tnFName* tnfPtr, TreeNode& tnBase ) ;

//* Create a copy of the specified TreNode object and all lower-level nodes    *
//* and file arrays attached to it.                                            *
//*                                                                            *
//* Input  : tnSrc : pointer to source TreeNode object                         *
//*          tnTarg: uninitialized instance (by reference) to receive data     *
//*          nondir: (optional, 'true' by default)                             *
//*                  'true' : copy all directory AND non-directory records     *
//*                  'false': copy ONLY directory records                      *
//* Returns: 'true' (TreeNode list has been copied)                            *
bool  CopyDirTree ( const TreeNode* tnSrc, TreeNode& tnTarg, bool nondir = true ) ;

//* Release the directory tree information list.                               *
//* NOTE: This is a complex data allocation that cannot be released using      *
//* 'delete' [] (TreeNode*)xyz;  Prevent memory leaks, play by the rules.      *
//*                                                                            *
//* Recursively walk to the bottom of the allocation tree, then release the    *
//* tree starting with the bottom branch and working back toward the top.      *
//*                                                                            *
//* NOTE: The TreeNode object referenced by caller's pointer IS NOT deleted    *
//* because it may be a member of an array; however, all lower-level nodes ARE *
//* deleted and the 'nextLevel' pointer which controlled them is set to NULL.  *
//*                                                                            *
//* Input  : pointer to TreeNode object                                        *
//*           Note that each object may have a list of objects attached to it. *
//* Returns: nothing,                                                          *
void  ReleaseDirTree ( TreeNode* tnPtr ) ;

//* Get a pointer to the base node of the captured directory tree.             *
//* Allows for filename search of the directory tree. No data may be modified. *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: const pointer to base node of TreeNode array                      *
const TreeNode* GetBaseNode ( void ) ;

//* Scan the TreeNode list of displayed files and return a pointer to the      *
//* TreeNode object to which the specified directory name belongs.             *
//*                                                                            *
//* Input  : pointer to tnFName object which describes a directory name in     *
//*           the list of displayed files.                                     *
//* Returns: pointer to the TreeNode object associated with this directory.    *
//*           Returns a NULL pointer if the specified file is not a            *
//*           directory or if the tnFName object is not a member of the        *
//*           displayed-files list.                                            *
const TreeNode* DirName2TreeNode ( tnFName* dirName ) ;

//* Read file system stats for the filesystem containing CURRENT directory.    *
//*                                                                            *
//* Input  : fsStats: (by reference) unitialized instance of                   *
//*                   fileSystemStats class: receives stat data                *
//* Returns: OK if successful or 'errno' value if stat failed                  *
//*           if 'OK', all fields of fsStats have been initialized             *
short GetFilesystemStats ( fileSystemStats& fsStats ) ;

//* Read file system stats for filesystem containing specified path/filename.  *
//*                                                                            *
//* Input  : trgPath: full path/filename for any file on the target file system*
//*          fsStats: (by reference) unitialized instance of                   *
//*                   fileSystemStats class: receives stat data                *
//* Returns: OK if successful or 'errno' value if stat failed                  *
//*           All members of 'fsStats' for which the target filesystem         *
//*           provides data have been initialized.                             *
//*           Note: Some filesystems do not provide data for all fields.       *
short GetFilesystemStats ( const gString& trgPath, fileSystemStats& fsStats ) ;
short GetFilesystemStats ( const char* trgPath, fileSystemStats& fsStats ) ;

//* Read the ID of the filesystem containing specified path/filename.          *
//*                                                                            *
//* Input  : trgPath: full path/filename for any file on the target file system*
//*          trgID  : (by reference) receives filesystem ID  (hex)             *
//*          trgType: (by reference) receives filesystem type  (hex)           *
//* Returns: OK if successful, else ERR                                        *
short FilesystemID ( const char* trgPath, UINT64& trgID, UINT& trgType ) ;

//* Returns the path of the device driver or other unique identifier           *
//* associated with the filesystem which contains the specified file.          *
//*                                                                            *
//* Input  : trgPath: full path/filename for any file on the target file system*
//*                   (normally the filespec of a directory)                   *
//*          devPath: (by reference) receives the requested filespec:          *
//*                   source: filespec of associated device driver             *
//*                   target: filespec of mountpoint                           *
//*          source : (optional, 'true' by default)                            *
//*                   'true' i.e. "source"  : return the device driver         *
//*                         associated with the filesystem containing the      *
//*                         specified target path                              *
//*                   'false' i.e. "target" : return the mountpoint of the     *
//*                         filesystem which contains the target path          *
//*                                                                            *
//* Returns: OK if 'devPath' contains:                                         *
//*             - device driver for block device                               *
//*             - device driver for virtual (MTP/GVFS) device                  *
//*             - inode number for logical filesystems (tmpfs, proc, etc.)
//*             - mountpoint filespec (source == false)                        *
//*          ERR if system call fails (unlikely)                               *
short FilesystemID ( const gString& trgPath, gString& devPath, bool source = true ) ;

//* Returns the amount of free storage space on the file system which contains *
//* the file or directory specified.                                           *
//*                                                                            *
//* Input  : trgPath  : path specification to any file/directory on            *
//*                     target file system                                     *
//*          fsysID   : (initial value ignored)                                *
//*                     on return, contains file system ID code                *
//*          freeBytes: (initial value ignored)                                *
//*                     on return, contains number of free storage bytes       *
//* Returns: OK if successful or system's error status if stat failed          *
//*           if not 'OK', fsysID and freeBytes are undefined                  *
short TargetFilesystemFreespace ( const char* trgPath, 
                                  UINT64& fsysID, UINT64& freeBytes ) ;

//* Determine whether the specified filespec refers to a mountpoint directory. *
//* This method uses the 'mountpoint' system utility.                          *
//*                                                                            *
//* Input  : trgPath : target filespec                                         *
//* Returns: 'true'  if target is a mountpoint directory                       *
//*          'false' if target is not a mountpoint                             *
//*                  (or is not a directory, or does not exist)                *
bool  isMountpoint ( const char* trgPath ) ;

//* Call the 'system' C-language library function with the specified command.  *
//*                                                                            *
//* This method (optionally) captures the actual exit code of the called       *
//* program and returns it to caller. This is done through two layers of       *
//* indirection which allows us to retrieve the exit code in our own           *
//* environment.                                                               *
//* The capture of the exit code is optional because if the caller doesn't     *
//* need the exit code, then we can omit one layer of indirection and used the *
//* 'system' library call instead.                                             *
//*                                                                            *
//* Input : cmd    : command string with program name and options              *
//*         retExit: (optional, 'false' by default)                            *
//*                  if 'false', do not capture the exit code of the called    *
//*                              process.                                      *
//*                                                                            *
//* Returns: if specified, the exit code of called external program            *
//*          Else, OK (ZERO)                                                   *
short Systemcall ( const char* cmd, bool retExit = false ) ;

//* Scan the source string (typically a filespec). If the string contains any  *
//* characters of the specified character group (enum escGroup) 'escape' them  *
//* with a backslash character.                                                *
//*                                                                            *
//* Do not use this method for URL (Universal Resource Locator) specs.         *
//* URL specifications require a specific protocol which is not handled here.  *
//*                                                                            *
//* Input  : pathSpec : (by reference) source text data                        *
//*                     on return the data has bee 'escaped' accoring to the   *
//*                     specified criteria                                     *
//*          group    : (member of enum escGroup) specifies the group of       *
//*                     characters to be 'escaped'                             *
//*          wch      : (optional, NULLCHAR by default)                        *
//*                     if 'group' parameter == escWCHAR, then 'wch' specifies *
//*                     a single character (NOT nullchar) to be 'escaped'      *
//* Returns: number of characters 'escaped'                                    *
short EscapeSpecialChars ( gString& pathSpec, escGroup group, wchar_t wch = NULLCHAR ) ;

//* Scan the source string (typically a filespec). If the string contains any  *
//* "escaped" characters i.e. characters preceed by a backstroke ('\'),        *
//* remove the backstrokes.                                                    *
//*                                                                            *
//* Escaped characters are typically regexp or shell "special characters".     *
//* This includes the backstroke character itself.                             *
//*                                                                            *
//* Input  : gsSrc    : (by reference) contains source text to be scanned      *
//*          retainDQ : (optional, 'false' by default)                         *
//*                     if 'true', then remove the backstroke for all escaped  *
//*                     characters EXCEPT the double-quote ( " ) character     *
//*                                                                            *
//* Returns: number of characters for which the escaping was removed           *
short RemoveCharEscapes ( gString& gsSrc, bool retainDQ = false ) ;

//* Construct the full filespec for the specified filesystem based on the      *
//* Uniform Resource Identifier (URI).                                         *
//* 1) MTP/GVfs URI:                                                           *
//*    Example URI: mtp://Android_ce012345c012345d01/                          *
//*    Filespec   : /run/user/1000/gvfs/mtp:host=Android_ce012345c012345d01    *
//* 2) Audio Disc URI:                                                         *
//*    Example URI: cdda://sr0/                                                *
//*    Filespec   : /run/user/1000/gvfs/cdda:host=sr0                          *
//*                                                                            *
//* Input  : pathSpec : (by reference) receives the target filespec            *
//*          uri      : URI for an MTP/GVFS filesystem                         *
//*                                                                            *
//* Returns: 'true' if valid filespec,       (pathSpec initialized)            *
//*          'false' if invalid or unrecognized 'uri' format (pathSpec cleared)*
bool  Uri2Filespec ( gString& pathSpec, const char* uri ) ;

//* Eject the removable media from the specified device, or if no device       *
//* specified, then eject the media from the first DVD/CD drive.               *
//*    Example: EjectMedia( "/dev/sr0" ) ;                                     *
//* Typical devices with ejectable media: CD-ROM, DVD-ROM, floppy disk, tape   *
//* drive, ZIP disk or USB disk.                                               *
//*                                                                            *
//* 1) Not all optical drives can be closed under software control.            *
//* 2) If target device is mounted and writable, be sure to close the output   *
//*    stream or unmount the device before ejecting the tray to avoid potential*
//*    loss of data.                                                           *
//* 3) If target device does not support removable media, the operation will   *
//*    silently fail.                                                          *
//* 4) The 'close' and 'toggle' flags are mutually exclusive. If 'toggle' is   *
//*    set, then 'close' is ignored.                                           *
//*                                                                            *
//* Input  : device : (optional, NULL pointer by default)                      *
//*                   If not specified, media will be ejected from the         *
//*                    system's default DVD/CD drive: "/dev/cdrom"             *
//*                   If specified, device whose media are to be ejected.      *
//*          close  : (optional, 'false' by default)                           *
//*                   If 'false', open the tray.                               *
//*                   If 'true',  close it tray.                               *
//*          toggle : (optional, 'false' by default)                           *
//*                   If 'false', ignore.                                      *
//*                   If 'true', toggle the state of the tray.                 *
//*                                                                            *
//* Returns: nothing                                                           *
void  EjectMedia ( const char* device = NULL, bool close = false, bool toggle = false ) ;

//* Perform environment-variable expansion or tilde ('~') expansion on the     *
//* specified path string.                                                     *
//*                                                                            *
//* Input  : gsPath  : (by reference) contains the string to be scanned        *
//* Returns: 'true'  if expansion successful (or no expansion needed)          *
//*          'false' if unable to expand                                       *
bool  EnvExpansion ( gString& gsPath ) ;

//* Create a unique path/filename for a temporary file.                        *
//*                                                                            *
//* Input  : tmpPath: (by reference) receives the path/filename                *
//* Returns: 'true'  if path/filename created successfully                     *
//*          'false' if library call failed                                    *
bool  CreateTempname ( gString& tmpPath ) ;

//* Delete the specified temporary file.                                       *
//*                                                                            *
//* Input  : tmpPath : full path/filename specification of source              *
//* Returns: OK if successful, else returns errno value                        *
short DeleteTempname ( const gString& tmpPath ) ;

//* Instantiate an array of TreeNode-class objects.                            *
//* The array is allocated from the memory heap.                               *
//* Do not call this method for automatic-variable instantiation.              *
//*                                                                            *
//* Access is controlled by a mutex to avoid having multiple execution threads *
//* overwhelm the system with memory-allocation requests.                      *
//*                                                                            *
//* Input  : tnCount : number of objects to allocate for the array             *
//* Returns: pointer to array of TreeNode objects                              *
TreeNode* Allocate_TreeNodes ( UINT tnCount ) ;

//* Release (delete) an array of TreeNode objects which was allocated by a     *
//* call to Allocate_TreeNodes().                                              *
//*                                                                            *
//* If an array of tnFName objects is attached to the 'tnFiles' member of any  *
//* element in the array, the tnFName array is also deleted.                   *
//*                                                                            *
//* Input  : tnPtr  : pointer to an array of TreeNode objects                  *
//*                   on return, tnPtr == NULL                                 *
//*          tnCount: number of elements in the array                          *
//* Returns: nothing                                                           *
void  Release_TreeNodes ( TreeNode*& tnPtr, UINT tnCount ) ;

//* Instantiate an array of tnFName-class objects.                             *
//* The array is allocated from the memory heap.                               *
//* Do not call this method for automatic-variable instantiation.              *
//*                                                                            *
//* Access is controlled by a mutex to avoid having multiple execution threads *
//* overwhelm the system with memory-allocation requests.                      *
//*                                                                            *
//* Input  : tnfCount: number of objects to allocate for the array             *
//* Returns: pointer to array of tnFName objects                               *
tnFName* Allocate_tnFNames ( UINT tnfCount ) ;

//* Release (delete) an array of tnFName objects which was allocated by a      *
//* call to Allocate_tnFNames().                                               *
//*                                                                            *
//* Input  : tnfPtr  : pointer to an array of tnFName objects                  *
//*                    on return, tnfPtr == NULL                               *
//*          tnfCount: number of elements in the array                         *
//*                     (primarily for debugging memory leaks)                 *
//* Returns: nothing                                                           *
void  Release_tnFNames ( tnFName*& tnfPtr, UINT tnfCount ) ;

//* Allow/disallow scan of full directory tree when CWD is the root directory. *
//*                                                                            *
//* Input  : enable     : if 'true',  recursive tree scan from root is enabled *
//*                       if 'false', recursive tree scan from root is disabled*
//*          reportOnly : (optional, 'false' by default)                       *
//*                       if 'false', flag is set/reset as 'enable' indicates  *
//*                       if 'true',  state of flag is is not modified,        *
//*                                   only reported                            *
//* Returns: current state of recursive-scan flag                              *
bool  ScanFromRoot ( bool enable, bool reportOnly = false ) ;

//* Returns error code for the most recent file management error. The value    *
//* returned is from the system 'errno' variable indicating the file access    *
//* or memory access error encountered.                                        *
//*                                                                            *
//* Input  : none                                                              *
//* Returns: most recently captured value from global 'errno' variable         *
short GetErrorCode ( void ) ;

//* Returns a short error message string for the specified errno code.         *
//*                                                                            *
//* Input  : code value returned to caller from this->recentErrno.             *
//* Returns: (const char*) pointer to message string                           *
const char* GetErrnoMessage ( short errCode ) ;

//* Concatenate a path string with a filename string to create a path/filename *
//* specification.                                                             *
//*                                                                            *
//* Input  : pgs  : gString object (by reference) to hold the path/filename    *
//*                 On return, pgs contains the path/filename string in both   *
//*                 UTF-8 and wchar_t formats.                                 *
//*          uPath: pointer to UTF-8 path string                               *
//*          uFile: pointer to UTF-8 filename string                           *
//*               OR                                                           *
//*          wPath: pointer to wchar_t path string                             *
//*          wFile: pointer to wchar_t filename string                         *
//* Returns: OK if success                                                     *
//*          ERR if string truncated                                           *
short CatPathFname ( gString& pgs, const char* uPath, const char* uFile ) ;
short CatPathFname ( gString& pgs, const wchar_t* wPath, const wchar_t* wFile ) ;

//* Given a string that may contain one of the following:                      *
//*  - a filename                                                              *
//*  - a relative path/filename                                                *
//*  - an absolute path/filename                                               *
//* extract the filename string from the input string and return it to caller. *
//*                                                                            *
//* Input  : nameBuff: buffer to hold filename                                 *
//*          pfSource: source filename or path/filename                        *
//* Returns: 'true' if filename successfully isolated                          *
//*          'false' in special case of pfSource=="/" (at root directory)      *
//*                  OR if invalid input                                       *
bool  ExtractFilename ( char nameBuff[MAX_FNAME], const char* pfSource ) ;
bool  ExtractFilename ( char nameBuff[MAX_FNAME], const gString& pfSource ) ;
bool  ExtractFilename ( gString& nameBuff, const gString& pfSource ) ;
bool  ExtractFilename ( gString& nameBuff, const char* pfSource ) ;

//* Given a string that may contain one of the following:                      *
//*  - a filename                                                              *
//*  - a path/filename                                                         *
//* extract the filename-extension including the leading '.' from the input    *
//* string and return it to caller.                                            *
//*                                                                            *
//* Input  : extBuff : buffer to hold extension                                *
//*                    If source filename does not contain an extension, then  *
//*                    a null string is returned (extBuff.gschars() == 1)      *
//*          pfSource: source filename or path/filename                        *
//* Returns: nothing                                                           *
void  ExtractExtension ( gString& extBuff, const gString& pfSource ) ;
void  ExtractExtension ( gString& extBuff, const char* pfSource ) ;
void  ExtractExtension ( gString& extBuff, const wchar_t* pfSource ) ;

//* Given a string that contains a full path/filename specification, extract   *
//* the pathname, i.e. truncate the string to remove the filename.             *
//*                                                                            *
//* Input  : pathBuff: buffer to contain path string                           *
//*          pfSource: full path/filename specification                        *
//* Returns: nothing                                                           *
void  ExtractPathname ( char pathBuff[MAX_PATH], const char* pfSource ) ;
void  ExtractPathname ( char pathBuff[MAX_PATH], const gString& pfSource ) ;
void  ExtractPathname ( gString& pathBuff, const gString& pfSource ) ;
void  ExtractPathname ( gString& pathBuff, const char* pfSource ) ;

//* Efficiently copy a UTF-8 string or a wchar_t (wide) string from source     *
//* to target.                                                                 *
//*                                                                            *
//* Replaces the C library strncpy() and wcsncpy() functions, which are safer  *
//* than strcpy() and wcscpy() BUT are grossly inefficient for copying         *
//* NULL-terminated strings which are shorter than the target buffer.          *
//*                                                                            *
//* Behavior is undefined if source and target overlap. DON'T DO IT!           *
//*                                                                            *
//* Input  : trg   : pointer to target buffer (caller must verify that target  *
//*                  target is large enough to hold source data.)              *
//*          src   : pointer to source data to be copied                       *
//*          size  : maximum number of bytes (char*) or characters (wchar_t*)  *
//*                  to copy INCLUDING the NULL terminator                     *
//*                  NOTE: Target will always be NULL terminated.              *
//* Returns: index into 'trg' of first free position after NULL terminator     *
int   strnCpy ( char* trg, const char* src, int size ) ;
int   strnCpy ( wchar_t* trg, const wchar_t* src, int size ) ;

//* Get a copy of application user's information.                              *
//*                                                                            *
//* Input  : (by reference, initial data ignored) receives user information    *
//* Returns: true if successful, else false                                    *
bool  IdentifyUser ( UserInfo& ui ) ;


     //*****************************************************************
     //**                   MTP/GVFS Method Group                     **
     //*****************************************************************

//* Create a copy of the specified source file.                                *
//* Preserve all permission bits and date/timestamps that are supported by the *
//* target filesystem.                                                         *
//*                                                                            *
//* WARNING: Do not call this method to copy a 'special' file.                 *
//*          ('regular', 'symlink' and 'FIFO' files only)                      *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*          followLink: (optional, default==false)                            *
//*                      If source is a symbolic link the following            *
//*                      applies; else the 'followLink' has no effect.         *
//*                      if false, copy the link itself                        *
//*                      if true, copy the file link points to                 *
//*                                                                            *
//* Returns: OK if successful                                                  *
//*          returns errno value if available, else EASSES                     *
short CopyFile_gvfs ( const char* srcPath, const char* trgPath, bool followLink = false ) ;

//* Delete the specified source file.                                          *
//* The 'gio' utility silently overrides write protection, so caller must test *
//* for write access.                                                          *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*                                                                            *
//* Returns: OK if successful, else returns errno value                        *
short DeleteFile_gvfs ( const char* srcPath ) ;

//* Rename the specified source file to specified destination file.            *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: OK if successful, else returns errno value                        *
short RenameFile_gvfs ( const char* srcPath, const char* trgPath ) ;

//* Create a new subdirectory.                                                 *
//* The mode bits will be set to the default for the filesystem.               *
//*    Example: all-access-for-user: drwx---------                             *
//* The timestamp will be the current local time.                              *
//*                                                                            *
//* Input  : dirPath : full path/filename specification for new directory      *
//*                                                                            *
//* Returns: OK if successful, else returns errno value                        *
short CreateDirectory_gvfs ( const char* dirPath ) ;

//* Scans the contents of the specified directory and returns 'true' if        *
//* directory contains no files (other than "." and "..").                     *
//*                                                                            *
//* Input  : dirPath : full path specification of source                       *
//*                                                                            *
//* Returns: 'true' if directory is empty, else 'false'                        *
bool  isEmptyDir_gvfs ( const char* dirPath ) ;

//* Delete the specified directory.                                            *
//* The 'gio' utility does not perform recursive operations such as deletion   *
//* of a directory tree, so the specified directory must be empty.             *
//*                                                                            *
//* Input  : dirPath : full path specification of source                       *
//*                                                                            *
//* Returns: OK if successful, else returns errno value                        *
short DeleteEmptyDir_gvfs ( const char* dirPath ) ;

//* Create a copy of the specified Directory-type source file.                 *
//* This method, insofar as possible, mirrors the corresponding method for     *
//* "real" filesystems:                                                        *
//*  - If target directory does not exist, create it.                          *
//*  - If target directory exists, do nothing.                                 *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*          trgPath : full path/filename specification of target              *
//*                                                                            *
//* Returns: OK if successful, else returns errno value                        *
short CopyDirName_gvfs ( const char* srcPath, const char* trgPath ) ;

//* Report whether the current base directory lies within a GVFS filesystems   *
//* i.e. whether the operation requires the "MTP" protocol.                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if target lies within (i.e. BELOW) the gvfs anchor point  *
//*          'false' otherwise                                                 *
bool  isGvfsPath ( void ) ;

//* Determine whether the specified filespec lies on the mount path for        *
//* GVFS filesystems i.e. whether the operation requires the "MTP" protocol.   *
//*                                                                            *
//* Input  : trgPath : target filespec                                         *
//*                                                                            *
//* Returns: 'true'  if target lies within (i.e. BELOW) the gvfs anchor point  *
//*          'false' otherwise                                                 *
bool  isGvfsPath ( const gString& trgPath ) ;

//* Returns the version number of the 'gio' utility (if installed).            *
//*                                                                            *
//* If the static variable 'gioInstalled' is not set, call the 'gioAvailable()'*
//* method to determine whether the 'gio' utility is installed and to          *
//* initialize the version string in the static 'gioVersion' variable at the   *
//* top of this module.                                                        *
//*                                                                            *
//* Input  : gvfsVersion : (by reference) receives the 'gio' utility version   *
//*                        or empty string if 'gio' not installed              *
//*                                                                            *
//* Returns: 'true' if GVfs tools are accessible, else 'false'                 *
bool  gioGetVersion ( gString& gvfsVersion ) ;

//* Scan for devices attached to the Universal Serial Bus (USB).               *
//* Capture filesystem and support information for the devices.                *
//* This method is designed primarily for smartphones, tablets and other       *
//* virtual filesystems attached to the USB bus. Devices may, or may not be    *
//* mounted and visible to the user.                                           *
//*                                                                            *
//* The "all" parameter:                                                       *
//* --------------------                                                       *
//* Optionally, this method can report the basic bus, device, ID and           *
//* descriptive text information for each device attached to the USB bus.      *
//* This option is not used by the FileMangler application, but may be helpful *
//* in other applications.                                                     *
//*                                                                            *
//* Important Note: This method allocates dynamic memory. It is the caller's   *
//*                 responsibility to release this memory when it is no longer *
//*                 needed. Example: delete [] usbdPtr ;                       *
//*                                                                            *
//* Input  : usbdPtr  : (by reference, initial value ignored)                  *
//*                     receives pointer to a dynamically-allocated array of   *
//*                     usbDevice objects. Set to NULL pointer if no USB       *
//*                     devices matching the criteria are found.               *
//*          testmnt  : (optional, 'false' by default)                         *
//*                     if 'false', do not test for mounted status             *
//*                     if 'true',  for each MTP/GVFS device determine whether *
//*                                 it is currently mounted                    *
//*          all      : (optional, 'false' by default)                         *
//*                     if 'false', report only MTP/GVFS devices               *
//*                     if 'true',  summary report of all devices attached     *
//*                                 to USB bus                                 *
//*                                                                            *
//* Returns: count of USB devices captured (elements in usbDevice array)       *
short USB_DeviceStats ( usbDevice*&, bool testmnt = false, bool all = false ) ;

//* Scan for drives attached to the local system.                              *
//* Capture filesystem and support information for the devices.                *
//*                                                                            *
//* This method is designed primarily for locating optical drives              *
//* (DVD,CD,BLU-RAY, etc). Data discs contained in the drive may, or may not   *
//* be mounted and visible to the user.                                        *
//*                                                                            *
//* Optionally, this method can report information for all attached drives.    *
//* This option is not used by the FileMangler application, but may be helpful *
//* in other applications.                                                     *
//*                                                                            *
//* Important Note: This method allocates dynamic memory. It is the caller's   *
//*                 responsibility to release this memory when it is no longer *
//*                 needed. Example: delete [] usbDev ;                        *
//*                                                                            *
//* Input  : usbdPtr  : (by reference, initial value ignored)                  *
//*                     receives pointer to a dynamically-allocated array of   *
//*                     usbDevice objects. Set to NULL pointer if no optical   *
//*                     devices matching the criteria are found.               *
//*          testmnt  : (optional, 'false' by default)                         *
//*                     if 'false', do not test for mounted status             *
//*                     if 'true',  for each optical drive identified,         *
//*                                 determine whether it is currently mounted  *
//*          all      : (optional, 'false' by default)                         *
//*                     if 'false', report only optical drive devices          *
//*                     if 'true',  summary report of all drives identified    *
//*                                                                            *
//* Returns: count of optical drives captured (elements in usbDevice array)    *
short DVD_DeviceStats ( usbDevice*& usbdPtr, bool testmnt = false, bool all = false ) ;


               //*****************************
               //*     Debugging Methods     *
               //*****************************
//* For debugging only. Used only to test for memory leaks *
void  GetTreeNode_Allocation ( UINT& tnObjects, UINT& tnfObjects ) ;

//* Report information specific to MTP/GVFS filesystems *
short GetFilesystemStats_gvfs ( const char* trgPath, usbDevice& usbDev ) ;

//* Report information specific to optical-drive (DVD/CD/Blu-Ray) filesystems *
short GetFilesystemStats_opti ( const char* trgPath, usbDevice& usbDev ) ;

//* Get FMgr-class version number string *
const char* Get_FMgr_Version ( void ) ;


private:
   //********************************************
   //* Private methods to support public access *
   //********************************************

   //* Allocate and release temporary storage for directory-file data          *
   bool  AllocateDynamicStorage ( UINT fCount ) ;
   void  ReleaseDynamicStorage ( bool releaseAll = true ) ;

   //* Methods used for sorting display data for the files *
   void  ReSortDirData ( fmSort sOption ) ;
   void  rsddName_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddNameR_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddDate_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddDateR_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddSize_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddSizeR_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddType_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddTypeR_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddExt_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddExtR_Sort ( ReSort iList[], UINT fCount, bool sensi ) ;
   void  rsddUn_Sort ( ReSort iList[], UINT fCount ) ;
   UINT  rsddIsolateDirNames ( ReSort iList[], UINT fCount, 
                               fmSort sOption, bool sensi );
   bool  IsolateFileExtension ( gString& fname ) ;
   void  rsddRandomize ( ReSort iList[], UINT fCount ) ;



   short GetFileStats ( tnFName& tnFile, const char* fPath, bool getTarg = false ) ;
   void  FormatExpandedStats ( ExpStats& es ) ;
   void  DecodePermissions ( ExpStats& es ) ;
   fmFType DecodeFileType ( UINT mode ) ;
   bool  IdentifyUser ( void ) ;                   // get info on application user

   void  BrokenLinkTargetPath ( const char* linkPath, 
              char targetPath[MAX_PATH], char targetName[MAX_FNAME] ) ;
   void  cdtStripNonDirFiles ( TreeNode* tnPtr ) ;
   short ScanDirTree ( const nodeCount& nodecnt, TreeNode* tnPtr, 
                       bool recurse = true, bool fsTest = false, DispData* dData = NULL ) ;
   UINT  procFlatscan ( nodeCount& nodecnt, TreeNode* tnPtr ) ;
   //* Programmer's Note: Do not overload method names for thread targets.*
   //* The compiler will not be able to match parameters correctly.       *
   void  sdtMgrthread ( const gString& dPath, const nodeCount& basecnt,
                        TreeNode* baseNode, DispData* dData ) ;
   UINT  sdtMgrthread_Level2 ( const nodeCount& nextTotal, TreeNode* baseNode, 
                               DispData* dData ) ;
   UINT  sdtMgrthread_Level1 ( const nodeCount& basecnt, 
                               TreeNode* baseNode, DispData* dData ) ;
   UINT  sdtMgrthread_LevelR ( const nodeCount& basecnt, 
                               TreeNode* baseNode, DispData* dData ) ;
   //UINT  sdtMgrthread_LevelG ( const nodeCount& basecnt, 
   //                            TreeNode* baseNode, DispData* dData ) ;
   //void  sdtGvfsthread ( const gString& dPath, TreeNode* tnPtr, DispData* dData ) ;
   void  sdtSubthread ( const gString& dPath, TreeNode* tnPtr, DispData* dData ) ;
   void  sdtSubthread_run ( const gString& dPath, TreeNode* tnPtr, DispData* dData ) ;
   void  sdtSubthread_proc ( const gString& dPath, TreeNode* tnPtr, DispData* dData ) ;
   bool  FilesystemMatch ( const gString& trgPath ) ;
   bool  isMountpath ( const gString& trgPath ) ;
   bool  isRootScan ( void ) ;
   void  ReleaseDirTree ( void ) ;                 // release the whole tree
   void  TNSum ( const TreeNode* tnPtr, TreeNode& tnAcc ) ;
   UINT  DirectoryCount ( nodeCount& nodecnt ) ;
   TreeNode* DirName2TreeNode ( tnFName* dirName, TreeNode* topNode ) ;
   DIR*  Safe_opendir ( const char* dirPath ) ;    // open a directory file

   //* MTP/GVFS group *
   UINT  GvfsFlatScan ( nodeCount& nodecnt, TreeNode* tnPtr ) ;
   short GetFileStats_gvfs ( tnFName& tnFile, const char* fPath, bool getTarg = false ) ;
   bool  GetFilesystemStats_aux ( const char* trgPath, usbDevice& usbDev ) ;
   bool  isMountpoint_gvfs ( const gString& trgPath ) ;
   bool  gioAvailable ( const gString& tmpSpec ) ;


   //* For debugging only *
   void DebugMsg ( const char* msg, short pause = ZERO ) ;
   void DebugLog ( const char* msg, short err = (-1), const char* aux = NULL ) ;
   void UserAlert ( short beeps = 1 ) ;

   //************************
   //* Private Data Members *
   //************************
   char     currDir[MAX_PATH] ;     //* Path string for currently-displayed directory
   TreeNode* deHead ;               //* Pointer to head of TreeNode list of directory entries
   char**   textData ;              //* Pointer to array of display-string pointers
   attr_t*  colorData ;             //* Pointer to array of color attributes for display strings
   tnFName**  deList ;              //* Pointer to array of pointers to file-data structures
   char*    rawText ;               //* Pointer to actual display-string data
   void*    dlBlockPtr ;            //* Pointer to memory allocated for display-list data
   attr_t   colorLookup[fmTYPES] ;  //* Color attributes for each supported file type
   const char* tfPath ;             //* Path spec for directory to write temp files

   UserInfo userInfo ;              //* Information on application-user's account
   fmSort   sortOption ;            //* Sort option for list of directory entries
   bool     caseSensitive ;         //* Sub-option for sorting directory entries by name
   bool     showHidden ;            //* If true, include 'hidden files' in file list
   USHORT   fileSizeFieldWidth ;    //* Field width for display of filesize (range: 8-13)
   short    recentErrno ;           //* Most recent 'errno' returned from system
   bool     fileSizeCommaFormat ;   //* If true, format filesize with commas, else no commas
   bool     groupDirectories ;      //* If true, group directories during directory-entry sort

   //* Group of members to control 'Intelligent Recursion' into different filesystems.*
   char     irID[MAX_FNAME] ;       //* Unique identifier for target filesystem
   bool     irMntpath ;             //* 'true' if base node of scan in on one of the fs mount paths
   bool     irOnesys ;              //* 'true' if all nodes of the tree are on the same filesystem
   bool     irVirtual ;             //* 'true' if base node of is on a virtual (MTP/GVFS) filesystem
   bool     irRootscan ;            //* If true, allow full tree scan from root directory

   //* Mutex (access control) for instantiation of TreeNode and tnFName *
   //* objects as well as other instantiations. Relieves pressure on    *
   //* the system resources for multi-threaded requests for dynamic     *
   //* memory allocation from the heap.                                 *
   mutex heapLock ;

   int      treenodeObjects ;       //* Track TreeNode objects instantiated/released
   int      tnfnameObjects ;        //* Track tnFName objects instantiated/released
   #if TNA_LOG != 0                 //* For debugging only:
   ofstream tnalogofs ;             //* Log file to track instantiate/release
   gString  tnalogFile,             //* of TreeNode and tnFName objects
            tnalogOut ;
   #endif  // TNA_LOG

   #if ENABLE_FMgrDebugLog != 0     //* For debugging only:
   ofstream dbgofs ;                //* Log file to capture system exceptions
   gString  dbgFile ;               //* and misc debug data
   #endif   // ENABLE_FMgrDebugLog

   //* Pointer to callback method for FMgr-class debugging messages *
   DMSG*    DebugMsgCB ;
} ;


#endif   // FMGR_INCLUDED

