//******************************************************************************
//* File       : EcLang.hpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2021 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 11-Jun-2021                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Language support data and display text for Exercalc           *
//*              application's supported languages.                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

#ifndef ECLANG_INCLUDED
#define ECLANG_INCLUDED

//* For LTR (left-to-right) languages, reset. *
//* For RTL (right-to-left) languages, set.   *
const bool AppDirection[] =      // [SUPPORT FOR RTL LANGUAGES NOT YET IMPLEMENTED]
{
   false,         // English,   LTR
   false,         // Espanol,   LTR
   false,         // Zhongwen,  LTR
   false,         // TiengViet, LTR
   false,         // language taken from locale (if locale language is RTL, set flag)
} ;

const short LangCount = 4 ;      // number of supported UI languages
const short ExerTypes = 4 ;      // number of exercise types (secondary arrays)

//* Display names for members of enum AppLang. Keep them synchronized. *
const wchar_t* const alStrings[] = 
{
   L"English",
   L"Español",
   L"Zhōngwén",
   L"TiếngViệt",
   L"(from locale)",
} ;

//* X offset for display of User's General Data *
//* Programmer's Note: X offset for display of Header for User's General Data is constant *
const short UserGenlDataOffset[LangCount] = 
{
   15,   //** English **
   17,   //** Español **
   20,   //** Zhōngwén (中文) **
   17,   //** TiếngViệt **
} ;

//* X offset for display of Header for User's Current Data *
const short UserCurrHdrOffset[LangCount] = 
{
   28,   //** English **
   28,   //** Español **
   30,   //** Zhōngwén (中文) **
   28,   //** TiếngViệt **
} ;

//* X offset for display of User's Current Data *
const short UserCurrDataOffset[LangCount] = 
{
   17,   //** English **
   21,   //** Español **
   18,   //** Zhōngwén (中文) **
   23,   //** TiếngViệt **
} ;

const wchar_t* const TitleDesc[LangCount] = 
{
   //** English **
   L"Calculate Daily Exercise Parameters",
   //** Español **
   L"Calcular Parámetros de Ejercicio Diario",
   //** Zhōngwén (中文) **
   L"计算日常运动参数",
   //** TiếngViệt **
   L"Tính Toán Các Thông Số Tập Thể Dục Hàng Ngày",
} ;

const wchar_t* const UserDlgTitle[LangCount] =
{
   //** English **
   L"  Exercise Statistics  ",
   //** Español **
   L"  Estadísticas de Ejercicio  ",
   //** Zhōngwén (中文) **
   L"      锻炼统计      ",
   //** TiếngViệt **
   L"  Thống Kê Bài Tập  ",
} ;

//* Display data for user-interface dialog control objects *
enum dlgControls : short { cloPB = ZERO, valTB, selDD, savPB, optMW, hlpMW, 
                           #if DEBUG_MENU != 0
                           dbgMW,
                           #endif   // DEBUG_MENU
                           CtrlCount } ;  // number of controls in user dialog

//* User-interface control labels *
const char* const UserDlgLabels[LangCount][CtrlCount] =
{
   //** English **
   {
   "     Close Dialog      ",
   " << Enter a Value",
   " Select a Parameter To Be Modified",
   "   Save To Log File    ",
   "   Other Operations    ",
   " ",                                   // dummy value for sub-menu
   },

   //** Español **
   {
   "    Cerrar Diálogo     ",
   " << Ingrese un Valor",
   "Seleccione Parámetro Para Modificar",
   "  Guardar en Registro  ",
   "   Otras Operaciones   ",
   " ",                                   // dummy value for sub-menu
   },

   //** Zhōngwén (中文) **
   {
   "      关闭对话框       ",
   " << 输入值",
   "         选择要修改的参数",
   "  将数据保存到日志文件 ",
   "       其他操作        ",
   " ",                                   // dummy value for sub-menu
   },

   //** TiếngViệt **
   {
   "    Đóng Hộp Thoại     ",
   " << Nhập giá trị",
   "  Chọn một tham số để sửa đổi",
   "  Lưu vào tệp nhật ký  ",
   "  Các hoạt động khác   ",
   " ",                                   // dummy value for sub-menu
   },
} ;


//* Data for initialization of the Options Menu.*
enum optParm : short { optCopy = ZERO,    // copy data to clipboard
                       optLogV,           // logfile view ('less')
                       optLogY,           // logfile summary
                       optLogW,           // logfile write record
                       optLogR,           // logfile read record
                       optLogE,           // logfile erase
                       optComp,           // compare records
                       optRest,           // restore record
                       optHelp,           // invoke Help
                       #if DEBUG_MENU != 0
                       optDebug,          // debugging sub-menu
                       #endif   // DEBUG_MENU
                       optITEMS      // number of menu items
                     } ;
const short optCOLS  = 37 ;      // display columns for each item
const char optparmData[LangCount][optITEMS][optCOLS+15] = 
{
   //** English **
   {
   "Copy Data to Clipboard               ",
   "View Log                             ",
   "Log Summary                          ",
   "Write Record to Log                  ",
   "Read Record from Log                 ",
   "Erase Log File Contents              ",
   "Compare Data Sets                    ",
   "Restore Original Data Set            ",
   "Help                              ...",
   #if DEBUG_MENU != 0
   "Debugging Options                 ...",
   #endif   // DEBUG_MENU
   },

   //** Español **
   {
   "Copiar Datos al Portapapeles         ",     // Copy data to clipboard
   "Ver Archivo de Registro              ",     // View log file
   "Resumen del Archivo de Registro      ",     // Log file summary
   "Escribir Registro Para Registrar     ",     // Write record to log
   "Leer Registro del Archivo de Registro",     // Read record from log
   "Borrar Contenido del Registro        ",     // Compare data sets
   "Comparar Conjuntos de Datos          ",     // Compare data sets
   "Restaurar Conjunto de Satos Original ",     // Restore original data set
   "Ayuda                             ...",     // Help
   #if DEBUG_MENU != 0
   "Opciones de Depuración            ...",     // Debugging options
   #endif   // DEBUG_MENU
   },

   //** Zhōngwén (中文) **
   {
   "将数据复制到剪贴板                   ",       // Copy data to clipboard
   "查看日志文件                         ",     // View log file
   "日志文件摘要                         ",     // Log file summary
   "将记录写入日志文件                   ",      // Write record to log
   "从日志文件中读取一条记录             ",       // Read record from log
   "清除日志文件的内容                   ",      // Erase log file contents
   "比较数据集                           ",     // Compare data sets
   "恢复原始数据集                       ",      // Restore original data set
   "救命                              ...",    // Help
   #if DEBUG_MENU != 0
   "调试选项                          ...",     // Debugging options
   #endif   // DEBUG_MENU
   },

   //** TiếngViệt **
   {
   "Sao chép dữ liệu vào khay nhớ tạm    ",     // Copy data to clipboard
   "Xem dữ liệu tệp nhật ký              ",     // View log file
   "Tóm tắt dữ liệu nhật ký              ",     // Log file summary
   "Ghi bản ghi vào tệp nhật ký          ",     // Write record to log
   "Đọc một mục từ tệp nhật ký           ",     // Read record from log
   "Xóa nội dung tệp nhật ký             ",     // Erase log file contents
   "So sánh các tập dữ liệu              ",     // Compare data sets
   "Khôi phục tập dữ liệu gốc            ",     // Restore original data set
   "Tài liệu trợ giúp                 ...",     // Help
   #if DEBUG_MENU != 0
   "Tùy chọn gỡ lỗi                   ...",     // Debugging options
   #endif   // DEBUG_MENU
   },
} ;


//* Data for initialization of the Help Sub-menu *
enum hlpParm : short { hlpInfo, hlpHtml, hlpAbout, hlpITEMS } ;
const short hlpCOLS  = 23 ;      // display columns for each item
const char hlpparmData[LangCount][hlpITEMS][hlpCOLS+10] = 
{
   //** English **
   {
      "Info Reader Help       ",
      "HTML Format Help       ",
      "About Exercalc         ",
   },

   //** Español **
   {
      "Info Ayuda del lector  ",
      "Ayuda de formato HTML  ",
      "Acerca del Programa    ",
   },

   //** Zhōngwén (中文) **
   {
      "Info 读者帮助          ",
      "HTML 格式帮助          ",
      "关于 Exercalc         .",
   },

   //** TiếngViệt **
   {
      "Info Trợ giúp Người đọc",
      "Trợ giúp Định dạng HTML",
      "Về Exercalc            ",
   },
} ;

//* Message relating to clipboard access *
enum copy2Cb : short { c2cbTemplate = ZERO, c2cbError, c2cbAvail, c2cbITEMS } ;
const wchar_t* const Copy2CbMsgs[LangCount][c2cbITEMS] = 
{
   //** English **
   {
   L"  %S bytes written to clipboard.  ",
   L"  Error writing to clipboard.  ",
   L" System clipboard is not accessible. ",
   },
   //** Español **
   {
   L"  %S bytes escritos en el portapapeles.  ",// X bytes written to clipboard.
   L"  Error al escribir en el portapapeles.  ",// Error writing to clipboard.
   L" El portapapeles del sistema no es accesible. ",// System clipboard is not accessible.
   },
   //** Zhōngwén (中文) **
   {
   L"  %S 个字节写入剪贴板。  ",                    // X bytes written to clipboard.
   L"  写入剪贴板时出错。  ",                       // Error writing to clipboard.
   L" 无法访问系统剪贴板。 ",                       // System clipboard is not accessible.
   },
   //** TiếngViệt **
   {
   L"  %S byte được ghi vào khay nhớ tạm.  ",   // X bytes written to clipboard.
   L"  Lỗi khi ghi vào khay nhớ tạm.  ",        // Error writing to clipboard.
   L" Không thể truy cập khay nhớ tạm hệ thống. ",// System clipboard is not accessible.
   },
} ;

//* Messages for display of data for user's exercise goal *
enum goalMsgs : short { gmDaily, gmWeekly, gmAge, gmGender,
                        gmMale, gmFemale, gmOther, gmCOUNT } ;
const wchar_t* const GoalMsgs[LangCount][gmCOUNT] = 
{
   //** English **
   {
   L"Daily Goal",                      // daily goal
   L"Weekly Goal",                     // weekly goal
   L"Age",                             // Age
   L"Gender",                          // Gender
   L"Male",                            // Male
   L"Female",                          // Female
   L"Other",                           // Other
   },

   //** Español **
   {
   L"Objetivo Diario",                 // daily goal
   L"Objetivo semanal",                // weekly goal
   L"Edad",                            // Age
   L"Género",                          // Gender
   L"Masculino",                       // Male
   L"Hembras",                         // Female
   L"Otro",                            // Other
   },

   //** Zhōngwén (中文) **
   {
   L"每日目标",                         // daily goal
   L"每周目标",                         // weekly goal
   L"年龄",                            // Age
   L"性别",                            // Gender
   L"男性",                            // Male
   L"女性",                            // Female
   L"其他",                            // Other
   },

   //** TiếngViệt **
   {
   L"Mục tiêu hàng ngày",              // daily goal
   L"Mục tiêu hàng tuần",              // weekly goal
   L"Tuổi tác",                        // Age
   L"Giới tính",                       // Gender
   L"Nam giới",                        // Male
   L"Giống cái",                       // Female
   L"Khác",                            // Other
   },
} ;


#if DEBUG_MENU != 0
//* Data for initialization of the Debug Sub-menu *
enum dbgParm : short { dbgColor = ZERO, dbgCmap, dbgSample, 
                       dbgConfig, dbgRtl, dbgWidget, dbgITEMS } ;
const short dbgCOLS  = 26 ;      // display columns for each item
const char dbgparmData[LangCount][dbgITEMS][dbgCOLS+14]
{
   //** English **
   {
      "Test Color Scheme         ",
      "Display Color Registers   ",
      "Create Sample Logfile     ",
      "Display Config Options    ",
      "RTL Example Dialog        ",
      "Chart-widget Test         ",
   },

   //** Español **
   {
      "Pruebe el Esquema de Color",
      "Mostrar Registros de Color",
      "Crear Archivo de Muestra  ",
      "Mostrar Configuración     ",
      "Ejemplo Diálogo de RTL    ",
      "Prueba de widget gráfico  ",
   },

   //** Zhōngwén (中文) **
   {
      "测试配色方案              ",
      "显示颜色寄存器            ",
      "创建样本日志文件          ",
      "显示配置选项              ",
      "RTL 示例对话框            ",
      "图表小部件测试            ",
   },

   //** TiếngViệt **
   {
      "Kiểm tra phối màu         ",
      "Hiển thị thanh ghi màu    ",
      "Tạo tệp nhật ký mẫu       ",
      "Tùy chọn cấu hình         ",
      "Hộp thoại ví dụ RTL       ",
      "Kiểm tra widget biểu đồ   ",
   },
} ;
#endif   // DEBUG_MENU

//* Enumerated type for deoding selections from    *
//* 'selparmData' array, below. (keep synchronized)*
enum selParm : short { spNone = ZERO,     // no selection (prompt) 
                       spXType,           // exercise type
                       spTime,            // elapsed time
                       spKm,              // distance travelled (km)
                       //spMi,              // distance travelled (mi)
                       spMets,            // METs (intensity of exercise)
                       spKg,              // body mass (kg)
                       spKc,              // kilocalories burned
                       spGoal,            // calculate daily/weekly goal
                       spDiscard,         // discard modifications
                       spITEMS            // number of dropdown items
                     } ;
//* Data for initialization of dctDROPDOWN control *
const short spWIDTH = 36 ;
const char selparmData[LangCount][spITEMS][spWIDTH+16] =
{
   //** English **
   {
   "-Select a Parameter To Be Modified-",
   "Exercise Type                      ",
   "Elapsed Time                       ",
   "Distance Travelled (km or mi)      ",
   "METs  (intensity of exercise)      ",
   "Body Mass (weight in kg)           ",
   "Kilocalories Burned                ",
   "Display Exercise Goal              ",
   "Discard Modifications              ",
   },

   //** Español **
   {
   "Seleccione Parámetro Para Modificar",    // Select a Parameter To Be Modified
   "Tipo de Ejercicio                  ",    // Exercise Type
   "Tiempo Transcurrido                ",    // Elapsed Time
   "Distancia Recorrida (km o mi)      ",    // Distance Travelled (km)
   "METs  (intensidad del ejercicio)   ",    // METs  (intensity of exercise)
   "Masa Corporal (peso en kg o libras)",    // Body Mass (weight in kg)
   "Kilocalorías Quemadas              ",    // kilocalories burned
   "Mostrar Objetivo de Ejercicio      ",    // calculate daily/weekly goal
   "Descartar Modificaciones           ",    // Discard Modifications
   },

   //** Zhōngwén (中文) **
   {
   "--------- 选择要修改的参数 --------",      // Select a Parameter To Be Modified
   "运动方式                           ",    // Exercise Type
   "经过时间                           ",    // Elapsed Time
   "行驶距离（公里或英里）             ",       // Distance Travelled (km)
   "METs （运动强度）                  ",     // METs  (intensity of exercise)
   "身体质量 （重量，公斤或磅）        ",        // Body Mass (weight in kg)
   "千卡烧                             ",   // kilocalories burned
   "显示锻炼目标                       ",    // calculate daily/weekly goal
   "放弃修改                           ",    // Discard Modifications
   },

   //** TiếngViệt **
   {
   "--- Chọn một tham số để sửa đổi ---", // Select a Parameter To Be Modified
   "Loại bài tập                       ", // Exercise Type
   "Thời gian trôi qua                 ", // Elapsed Time
   "Quãng đường đã đi (km hoặc là mi) .", // Distance Travelled (km)
   "METs  (cường độ tập thể dục)       ", // METs  (intensity of exercise)
   "Khối lượng cơ thể (kg hoặc là lb) .", // Body Mass (weight in kg)
   "Kilocalories bị đốt cháy           ", // kilocalories burned
   "Hiển thị mục tiêu tập thể dục      ", // calculate daily/weekly goal
   "Hủy sửa đổi                        ", // Discard Modifications
   },
} ;

//* Prompt messages when user selects a parameter to be modified from *
//* the dropdown control. Note that not all options require a prompt, *
//* so these get a placeholder null string.                           *
const wchar_t* const selectPrompt[LangCount][spITEMS] = 
{
   //** English **
   {
   L"",                                                  // (title)
   L"1)Walking 2)Bicycling 3)Running 4)General",         // exercise type
   L"Example 1:30 (or minutes only 90)",                 // elapsed time
   L"Example(km) 14.50 (or miles, 9.01 mi)",             // distance travelled
   L"Value from activity table (or other value)",        // METs (intensity)
   L"Example(kg) 64.0 (or pounds, 141.1 lb)",            // body mass
   L"",                                                  // kilocalories burned
   L"1) Daily goal  2) Weekly goal",                     // calculate daily/weekly goal
   L"",                                                  // discard modifications
   },
   //** Español **
   {
   L"",                                                  // (title)
   L"1)Caminando 2)Bicicleta 3)Corriendo 4)Genérico",    // exercise type
   L"Ejemplo 1:30  (o solo minutos 90)    ",             // elapsed time
   L"Ejemplo(km) 14.50  (o millas, 9.01 mi)",            // distance travelled
   L"Valor de la tabla de actividad (u otro valor)",     // METs (intensity)
   L"Ejemplo(kg) 64.0  (o libras, 141.1 lb)",            // body mass
   L"",                                                  // kilocalories burned
   L"1) Objetivo diario  2) Objetivo semanal",           // calculate daily/weekly goal
   L"",                                                  // discard modifications
   },
   //** Zhōngwén (中文) **
   {
   L"",                                                  // (title)
   L"1)步行  2)单车  3)跑步  4)普通运动   ",                 // exercise type
   L"示例 1:30（或仅分钟 90）             ",                 // elapsed time
   L"示例（公里）14.50（或英里，9.01 mi）  ",                  // distance travelled
   L"活动表中的值（或其他值）             ",                  // METs (intensity)
   L"例子（千克）64.0（或磅, 141.1 lb）   ",                  // body mass
   L"",                                                  // kilocalories burned
   L"1）每日目标  2）每周目标",                              // calculate daily/weekly goal
   L"",                                                  // discard modifications
   },
   //** TiếngViệt **
   {
   L"",                                                  // (title)
   L"1)Đi Dạo 2)Đi Xe Đạp 3)Đang Chạy 4)Bài Tập Chung",  // exercise type
   L"Ví dụ 1:30  (hoặc phút chỉ 90)       ",             // elapsed time
   L"Ví dụ (km) 14.50  (hoặc dặm, 9.01 dặm)",            // distance travelled
   L"Giá trị từ bảng hoạt động (hoặc giá trị khác)",     // METs (intensity)
   L"Ví dụ (kg) 64.0 (hoặc pound, 141.1 lb)",            // body mass
   L"",                                                  // kilocalories burned
   L"1) Mục tiêu hàng ngày  2) Mục tiêu hàng tuần",      // calculate daily/weekly goal
   L"",                                                  // discard modifications
   },
} ;


//* Error message if user selection or user-supplied value is out-of-range.*
enum badUser : short { bumXType = ZERO, bumGoal, bumRange, bumMSGS } ;
const wchar_t* const BadUserMsg[LangCount][bumMSGS] = 
{
   //** English **
   L"Sorry, invalid selection. Enter 1-4 Only. (TAB to quit)",
   L"Sorry, invalid selection. Enter 1-2 Only. (TAB to quit)",
   L"Sorry, value out of range. Try again. (TAB to quit)",

   //** Español **
   L"Lo sentimos, selección inválida. Ingrese 1-4 solamente.",
   L"Lo sentimos, selección inválida. Ingrese 1-2 solamente.",
   L"Lo siento, valor fuera de rango. Inténtalo de nuevo.",

   //** Zhōngwén (中文) **
   L"抱歉，选择无效。 仅输入1-4。 （TAB键退出）",
   L"抱歉，选择无效。 仅输入1-2。 （TAB键退出）",
   L"抱歉，值超出范围。 再试一次。 （TAB键退出）",

   //** TiếngViệt **
   L"Xin lỗi, lựa chọn không hợp lệ. Chỉ nhập 1-4. (TAB = thoát)",
   L"Xin lỗi, lựa chọn không hợp lệ. Chỉ nhập 1-2. (TAB = thoát)",
   L"Xin lỗi, giá trị nằm ngoài phạm vi. Thử lại. (TAB = thoát)",
} ;

//* If user has discarded experimental    *
//* parameters and restored original data *
const wchar_t* const parmsRestored[LangCount] = 
{
   //** English **
   L" Original parameters restored. ",
   //** Español **
   L" Parámetros originales restaurados. ",
   //** Zhōngwén (中文) **
   L"  原始参数已恢复。 ",
   //** TiếngViệt **
   L" Đã khôi phục các thông số ban đầu. ",
} ;

const wchar_t* const UserGenlTitle[LangCount] =
{
   //** English **
   L" Defined Parameters \n",
   //** Español **
   L"  Parámetros Definidos  \n",
   //** Zhōngwén (中文) **
   L"        定义参数        \n",
   //** TiếngViệt **
   L"   Tham Số Xác Định   \n",
} ;

//* User General Data (Metric Units) *
const wchar_t* const UserGenlData[LangCount][5] =
{
   //** English **
   {
   L"Walking METs:\n"
    " Km/hr      :\n"
    " Mi/hr      :\n"
    " kCal/hr    :\n",
   L"Bicycle METs:\n"
    " Km/hr      :\n"
    " Mi/hr      :\n"
    " kCal/hr    :\n",
   L"Running METs:\n"
    " Km/hr      :\n"
    " Mi/hr      :\n"
    " kCal/hr    :\n",
   L"General METs:\n"
    " kCal/hr    :\n",
   L"Height (cm) :\n"
    "       (in) :\n"
    "Mass   (kg) :\n"
    "       (lb) :\n"
    "BMI (kg/m²) :\n",
   },

   //** Español **
   {  // equivalente metabólico de la tarea (metabolic equivalent of task MET)
   L"Caminando METs:\n"
    " Km/hora      :\n"
    " millas/hora  :\n"
    " kiloCal/hora :\n",
   L"Bicicleta METs:\n"
    " Km/hora      :\n"
    " millas/hora  :\n"
    " kiloCal/hora :\n",
   L"Corriendo METs:\n"
    " Km/hora      :\n"
    " millas/hora  :\n"
    " kiloCal/hora :\n",
   L"General METs  :\n"
    " kiloCal/hora :\n",
   L"Altura (cm)   :\n"
    "       (in)   :\n"
    "Masa   (kg)   :\n"
    "       (lb)   :\n"
    "BMI (kg/m²)   :\n",
   },

   //** Zhōngwén (中文) **
   {
   L"步行 代谢当量    :\n"     // Walking  (代谢当量任务 Metabolic Equivalents for Task)
    " 公里/小时       :\n"    // Km/hr
    " 英里/小时       :\n"    // mi/hr
    " 每小时千卡      :\n",   // kCal/hr
   L"自行车 代谢当量  :\n"     // Bicycling
    " 公里/小时       :\n"    // Km/hr
    " 英里/小时       :\n"    // mi/hr
    " 每小时千卡      :\n",   // kCal/hr
   L"跑步 代谢当量    :\n"     // Running
    " 公里/小时       :\n"    // Km/hr
    " 英里/小时       :\n"    // mi/hr
    " 每小时千卡      :\n",   // kCal/hr
   L"普通运动 代谢当量:\n"     // General    (代谢当量任务)
    " 每小时千卡      :\n",   // kCal/hr
   L"高度 (厘米)      :\n"     // height (cm)
    "       (寸)      :\n"    // (in)
    "弥撒 (公斤)      :\n"     // body mass (kg)
    "       (磅)      :\n"    // (lb)
    "体重指数(公斤/m²):\n", // 体重指数 (BMI)
   },

   //** TiếngViệt **
   {
   L"Đi dạo METs   :\n"       // Walking METs
    " km/giờ       :\n"       // Km/hr
    " Mi/hr        :\n"       // mi/hr
    " kCal/hr      :\n",      // kCal/hr
   L"Xe đạp METs   :\n"       // Bicycling
    " km/giờ        :\n"      // Km/hr
    " Mi/hr        :\n"       // mi/hr
    " kCal/hr      :\n",      // kCal/hr
   L"Đang chạy METs:\n"       // Running
    " km/giờ       :\n"       // Km/hr
    " dặm/giờ      :\n"       // mi/hr
    " kCal/hr      :\n",      // kCal/hr
   L"Chung METs    :\n"       // General METs
    " kCal/hr      :\n",      // kCal/hr
   L"Chiều cao (cm):\n"       // height (cm)
    "          (in):\n"       // (in)
    "Khối lượng(kg):\n"       // body mass (kg)
    "          (lb):\n"       // (lb)
    "BMI (kg/m²)   :\n",      // BMI
   },
} ;

const wchar_t* const UserCurrTitle[LangCount] =
{
   //** English **
   L"        Workout Results       \n",
   //** Español **
   L"  Resultados del Entrenamiento  \n",
   //** Zhōngwén (中文) **
   L"           锻炼结果            \n",
   //** TiếngViệt **
   L"       Kết Quả Tập Luyện      \n",
} ;

const wchar_t* const UserCurrData[LangCount] =
{
   //** English **
   L"Exercise Type  :\n"
    "Elapsed Time   :\n"
    "   (minutes)   :\n"
    "Distance  (km) :\n"
    "          (mi) :\n"
    "MET Minutes    :\n"
    "kCal Burned    :\n"
    "   (mass) (kg) :\n"
    "          (lb) :\n",

   //** Español **
   L"Tipo de Ejercicio  :\n"
    "Tiempo Transcurrido:\n"
    "   (minutos)       :\n"
    "Distancia (km)     :\n"
    "          (millas) :\n"
    "MET Minutos        :\n"
    "Kcal Quemado       :\n"
    "   (masa) (kg)     :\n"
    "          (libras) :\n",

   //** Zhōngwén (中文) **
   L"运动方式        :\n"
    "经过时间        :\n"
    "  (分钟)        :\n"
    "行驶距离 (公里) :\n"
    "         (英里) :\n"
    "代谢当量 分钟   :\n"
    "大卡消耗        :\n"
    "    弥撒 (公斤) :\n"
    "           (磅) :\n",

   //** TiếngViệt **
   L"Loại bài tập         :\n"
    "Thời gian đã trôi qua:\n"
    "   (phút)            :\n"
    "Khoảng cách (km)     :\n"
    "            (dặm)    :\n"
    "MET Phút             :\n" // trao đổi chất tương đương với nhiệm vụ (metabolic equivalent of task)
    "k-calo bị đốt cháy   :\n"
    "   (khối lượng) (kg) :\n" // khối lượng cơ thể (body mass)
    "                (lb) :\n",
} ;

//* Exercise-type Names (referenced in multiple places) *
const wchar_t* const UserExerType[LangCount][ExerTypes + 1] = 
{
   //** English **
   {
      L"Walking",
      L"Bicycling",
      L"Running",
      L"General",
      L"unknown",    // (should never be referenced)
   },

   //** Español **
   {
      L"Caminando",
      L"Bicicleta",
      L"Corriendo",
      L"Genérico",
      L"desconocido",
   },

   //** Zhōngwén (中文) **
   {
      L"步行",      // Walking
      L"单车",      // Bicycling
      L"跑步",      // Running
      L"普通运动",   // General
      L"未知",      // unknown
   },

   //** TiếngViệt **
   {
      L"Đi Dạo",           // Walking
      L"Đi Xe Đạp",        // Bicycling
      L"Đang Chạy",        // Running
      L"Bài Tập Chung",    // General
      L"không biết",       // unknown
   },
} ;

const wchar_t* const UserNoData[LangCount] = 
{
   //** English **
   L" No input values were specified. ",
   //** Español **
   L" No se especificaron valores de entrada. ",
   //** Zhōngwén (中文) **
   L" 未指定输入值。 ",
   //** TiếngViệt **
   L" Không có giá trị đầu vào nào được chỉ định. ",
} ;

//* Display data and formatting for LogfileSummary() method.*
enum sumRec : short { srhTitle = ZERO, srhTimes, srhTotHdr, srhAvgHdr, 
                      srhAvgData, srhTotData, srhITEMS } ;
const wchar_t* const summRecHdr[LangCount][srhITEMS] = 
{
   //** English **
   {
   L"   Logfile Summary   ",           // dialog title
   L"Logfile: \n"                    // filename, record count and timestamp range
    "Records: \n"
    "   From: \n"
    "     To: \n\n\n",
   L" Total Exercise              ",   // 'totals' column heading
   L" Average Daily Exercise      ",   // 'averages' column heading
   L"Body Mass (kg): \n"          // 'averages' base data
    "          (lb): \n"
    "Kilometers/hr : \n"
    "         (mph): \n"
    "BMI           : \n",
   L"Kilometers    : \n"          // 'totals'/'averages' item names
    "       (miles): \n"
    "Elapsed Time  : \n"
    "MET Minutes   : \n"
    "kCal Burned   : \n",
   },

   //** Español **
   {
   L"   Resumen del archivo de registro   ", // dialog title
   L"Nombre  : \n"                     // filename, record count and timestamp range
    "Archivos: \n"
    "   Desde: \n"
    "   Hasta: \n\n\n",
   L" Ejercicio Total             ",   // 'totals' column heading
   L" Ejercicio Diario Promedio   ",   // 'averages' column heading
   L"Masa      (kg): %-.3f\n"          // 'averages' base data
    "          (lb): %-.3f\n"
    "Km/hora       : %-.3f\n"
    " (millas/hora): %-.3f\n"
    "BMI (kg/m²)   : %-.3f\n",
   L"Kilómetros    : \n"               // 'totals'/'averages' item names
    "      (millas): \n"
    "Tiempo Total  : \n"
    "MET Minutos   : \n"
    "Kcal Quemado  : \n",
   },

   //** Zhōngwén (中文) **
   {
   L"    日志文件摘要    ",              // dialog title
   L"文档名称: \n"                      // filename, record count and timestamp range
    "记录    : \n"
    "      从: \n"
    "      至: \n\n\n",
   L" 锻炼总数                    ",    // 'totals' column heading
   L" 平均每日运动                ",      // 'averages' column heading
   L"弥撒   (公斤): %-.3f\n"            // 'averages' base data
    "         (磅): %-.3f\n"
    "公里/小时    : %-.3f\n"
    " (英里/小时) : %-.3f\n"
    "体重指数     : %-.3f\n",
   L"公里数       : \n"                // 'totals'/'averages' item names
    "    (英里)   : \n"
    "经过时间     : \n"
    "代谢当量 分钟: \n"
    "千卡烧       : \n",
   },

   //** TiếngViệt **
   {
   L"   Tóm Tắt Tệp Nhật Ký   ",       // dialog title
   L"Tên tệp: \n"                      // filename, record count and timestamp range
    "Văn khố: \n"
    "     Từ: \n"
    "    Đến: \n\n\n",
   L" Tổng Bài Tập                ",   // 'totals' column heading
   L" Bài Tập Trung Bình Hàng Ngày ",  // 'averages' column heading
   L"Khối lượng (kg)   : %-.3f\n"      // 'averages' base data
    "           (lb)   : %-.3f\n"
    "Kilomét / giờ     : %-.3f\n"
    "          (mph)   : %-.3f\n"
    "BMI (kg/m²)       : %-.3f\n",
   L"Kilomét           : \n"           // 'totals'/'averages' item names
    "      (dặm)       : \n"
    "Thời gian trôi qua: \n"
    "MET Phút          : \n"
    "Calo bị đốt cháy  : \n",
   },
} ;

//* Summary graph, horizontal axis label *
enum alHoriz : short { alhDist = ZERO, alhEner, alhTime, alhITEMS } ;
const wchar_t* const axisLabelH[LangCount][alhITEMS] = 
{
   //** English **
   {
   L"  Data Range:%.2f-%.2f kilometers  -----  %.0f Days",
   L"  Data Range:%.2f-%.2f kilocalories  -----  %.0f Days",
   L"  Data Range:%.2f-%.2f minutes  -----  %.0f Days",
   },
   //** Español **
   {
   L"  Rango de datos:%.2f-%.2f kilómetros  -----  %.0f Dias",
   L"  Rango de datos:%.2f-%.2f kilocalorías  -----  %.0f Dias",
   L"  Rango de datos:%.2f-%.2f minutos  -----  %.0f Dias",
   },
   //** Zhōngwén (中文) **
   {
   L"  数据范围:%.2f-%.2f 公里  -----  %.0f 天",
   L"  数据范围:%.2f-%.2f 千卡  -----  %.0f 天",
   L"  数据范围:%.2f-%.2f 分钟  -----  %.0f 天",
   },
   //** TiếngViệt **
   {
   L"  Dải dữ Liệu:%.2f-%.2f cây số  -----  %.0f Ngày",
   L"  Dải dữ Liệu:%.2f-%.2f kilocalories  -----  %.0f Ngày",
   L"  Dải dữ Liệu:%.2f-%.2f phút  -----  %.0f Ngày",
   },
} ;

//* Summary graph, "Goal" label (data-formatting template) *
enum alGoal : short
{
   algDist = ZERO,      // Kilometers
   algEner,             // Kilocalories'
   algTime,             // Minutes
   algITEMS,            // number of items in axisLabelG[]
} ;
const wchar_t* const axisLabelG[LangCount][algITEMS] = 
{
   //** English **
   L" Distance Goal: %.2f km/day ",
   L" Energy Goal: %.2f kCal/day ",
   L" Time Goal: %02hd:%02hd hr/day ",
   //** Español **
   L" Objetivo de distancia: %.2f km/día ",
   L" Objetivo energético: %.2f kCal/day ",
   L" Objetivo de tiempo: %02hd:%02hd horas/día ",
   //** Zhōngwén (中文) **
   L" 距离目标: %.2f 公里/天 ",
   L" 能源目标: %.2f 千卡/天 ",
   L" 时间目标: %02hd:%02hd 小时/天 ",
   //** TiếngViệt **
   L" Mục tiêu khoảng cách: %.2f km/ngày ",
   L" Mục tiêu năng lượng: %.2f kcal/ngày ",
   L" Mục tiêu thời gian: %02hd:%02hd giờ/ngày ",
} ;

//* Display data and formatting for LoadRecord() method.*
enum loadRec : short { ldlLoad, ldlCancel, ldlList, ldlTitle, 
                       ldlLabel, ldlInstr, ldlITEMS } ;
const char* const LoadDlgLabels[LangCount][ldlITEMS] = 
{
   //** English **
   {
   "        Select         ",          // 'Select' pushbutton
   "        Cancel         ",          // 'Cancel' pushbutton
   "   Date     Duration   ExerciseType   Distance  METs   kCal\n", // 'List' column headings
   "   Select Log File Record   ",     // dialog title
   "Number of Records: ",              // record-count label
   "Highlight the desired record and then press \"Select\"." // instructions
   },

   //** Español **
   {
   "      Seleccione       ",          // 'Select' pushbutton
   "       Cancelar        ",          // 'Cancel' pushbutton
   "  Fecha     Duración   TipoEjercicio  Distancia METs   kCal\n", // 'List' column headings
   "   Seleccione un Registro de Archivo de Registro   ",     // dialog title
   "Número de Registros: ",            // record-count label
   "Resalte el registro deseado y luego presione \"Seleccione\"." // instructions
   },

   //** Zhōngwén (中文) **
   {
   "         选择          ",           // 'Select' pushbutton
   "         取消          ",           // 'Cancel' pushbutton
   "   日期       期间     运动类型         距离    METs   大卡\n", // 'List' column headings
   "   选择日志文件记录   ",              // dialog title
   "记录数： ",                          // record-count label
   "突出显示所需的记录，然后按 \"选择\"."    // instructions
   },

   //** TiếngViệt **
   {
   "          Lựa          ",          // 'Select' pushbutton
   "        Huỷ Bỏ         ",          // 'Cancel' pushbutton
   "   Ngày    Thời lượng  Loại Bài tập Khoảng Cách METs   kCal\n", // 'List' column headings

   "   Chọn bản ghi tệp nhật ký   ",   // dialog title
   "Số lượng bản ghi: ",               // record-count label
   "Đánh dấu bản ghi mong muốn rồi nhấn \"Lựa\"." // instructions
   },
} ;

//* Display data and formatting for LogfileAppend()/lfaConfirm() methods.*
enum saveRec : short { sdlSave, sdlCancel, sdlComm, sdlTitle, 
                       sdlInstr, sdlITEMS } ;
const char* const SaveDlgLabels[LangCount][sdlITEMS] = 
{
   //** English **
   {
   "      Save Record      ",          // 'Save' pushbutton
   "        Cancel         ",          // 'Cancel' pushbutton
   "Comment Text",                     // 'Comment' textbox
   "  Save Record To Log File  ",      // dialog title
   // Instructions
   "Save the currently-displayed record to the log file?\n"
   "Optionally enter a comment to be saved with the record.\n",
   },

   //** Español **
   {
   "   Guardar Registro    ",          // 'Select' pushbutton
   "       Cancelar        ",          // 'Cancel' pushbutton
   "Texto de Comentario",              // 'Comment' textbox
   "  Guardar Registro en Archivo de Registro  ", // dialog title
   // Instructions
   "¿Guardar el registro que se muestra actualmente en el archivo\n"
   "de registro? Opcionalmente, ingrese un comentario para\n"
   "guardarlo con el registro.\n",
   },

   //** Zhōngwén (中文) **
   {
   "        保存记录       ",          // 'Save' pushbutton
   "         取消          ",          // 'Cancel' pushbutton
   "评论文字",                         // 'Comment' textbox
   "  将记录保存到日志文件  ",           // dialog title
   // Instructions
   "             是否将当前显示的记录保存到日志文件？\n"
   "             （可选）输入要与记录一起保存的注释。\n",
   },

   //** TiếngViệt **
   {
   "      Lưu Bản Ghi      ",          // 'Save' pushbutton
   "        Huỷ Bỏ         ",          // 'Cancel' pushbutton
   "Văn bản Nhận xét",                 // 'Comment' textbox
   "  Lưu bản ghi vào tệp nhật ký  ",  // dialog title
   // Instructions
   "Bạn có muốn lưu bản ghi được hiển thị vào tệp nhật ký không?\n"
   "Nếu muốn, hãy viết bình luận để được đưa vào tệp nhật ký.\n",
   },
} ;


//* Display data and formatting for CompareRecords() method.*
enum crHdr : short { crhTitle = ZERO, crhLeft, crhRight, crhVnames, crhITEMS } ;
const wchar_t* const chrTemplate =  // formatting template for compRecHdr data
   L"%-.3f\n"        // Height
    "%-.2f\n"        // Mass
    "%-.2f\n"        // BMI
    "%S\n"           // Exercise Type
    "%02hd:%02hd\n"  // Elapsed Time
    "%-.3f\n"        // Kilometers
    "%-.3f\n"        // KPH
    "%-.3f\n"        // METs Ratio
    "%-.3f\n"        // MET Minutes
    "%-.3f\n" ;      // Kilocalories

const wchar_t* const compRecHdr[LangCount][crhITEMS] = 
{
   //** English **
   {
   L"   Compare Data Records   ",      // dialog title
   L"   Original               ",      // left-column header
   L" Retrieved    ",                  // right-column header
   L"Height    (m) :\n"                // variable names
    "Body Mass (kg):\n"
    "BMI           :\n"
    "Exercise Type :\n"
    "Elapsed Time  :\n"
    "Kilometers    :\n"
    "Km per Hour   :\n"
    "METs Ratio    :\n"
    "MET Minutes   :\n"
    "Kilocalories  : ",
   },
   //** Español **
   {
   L"   Comparar Registros de Datos   ",  // dialog title
   L"   Original                       ", // left-column header
   L" Recuperado   ",                     // right-column header
   L"Altura        (m)      :\n"    // height (meters)
    "Masa Corporal (kg)     :\n"    // body mass (kg)
    "Índice de masa corporal:\n"    // BMI
    "Tipo de ejercicio      :\n"    // exercise type
    "Tiempo Transcurrido    :\n"    // elapsed time
    "Kilómetros             :\n"    // kilometers
    "Kilómetros por Hora    :\n"    // kph
    "METs Proporción        :\n"    // METs Ratio
    "MET Minutos            :\n"    // MET Minutes
    "Kilocalorías           : ",    // kilocalories
   },
   //** Zhōngwén (中文) **
   {
   L"   比较数据记录   ",                // dialog title
   L"   原始记录              ",        // left-column header
   L" 检索到的记录    ",                 // right-column header
   L"高度     (米)  :\n"                // variable names
    "弥撒     (公斤):\n"    // body mass (kg)
    "体重指数       :\n"    // BMI
    "运动类型       :\n"    // exercise type
    "经过时间       :\n"    // elapsed time
    "公里           :\n"   // kilometers
    "每小时公里     :\n"    // kph
    "代谢当量比     :\n"    // METs Ratio
    "代谢当量 分钟  :\n"    // MET Minutes
    "千卡           : ",   // kilocalories
   },
   //** TiếngViệt **
   {
   L"   So Sánh Các Bản Ghi Dữ Liệu   ",    // dialog title
   L"   Nguyên                            ",// left-column header
   L" Lấy Lại      ",                       // right-column header
   L"Chiều cao         (m) :\n"             // variable names
    "khối lượng cơ thể (kg):\n"     // body mass (kg
    "BMI                   :\n"     // BMI
    "Loại bài tập          :\n"     // exercise type
    "Thời gian trôi qua    :\n"     // elapsed time
    "Cây số                :\n"     // kilometers
    "Ki lô mét mỗi giờ     :\n"     // kph
    "METs Tỉ Lệ            :\n"     // METs Ratio
    "MET Phút              :\n"     // MET Minutes
    "Kilocalories          : ",     // kilocalories
   },
} ;

//* User messages for logfile operations *
enum logMsgs : short
{
   lmGoodWrite = ZERO,        // successful write to log
   lmReinit,                  // log file reinitialized
   lmAbortWrite,              // user aborted write to logfile
   lmNoTarget,                // unable to find log file
   lmNoCreate,                // cannot create file OR user does not have access
   lmNoAccess,                // unable to find or open log file
   lmNoRegular,               // target is not a 'regular' file
   lmNoRecords,               // logfile is valid but contains no records
   lmBadFormat,               // specified log file has invalid record format
   lmMSGS                     // number of messages in array
} ;

//* Various message for user interaction with log file methods.*
const wchar_t* const LogMsgs[LangCount][lmMSGS] = 
{
   //** English **
   {
      L" Data written to log file. ",
      L" Log File Reinitialized. ",
      L" Operation cancelled. Record not written. ",
      L" Unable to find log file. ",
      L" Cannot create log or no write access. ",
      L" File not found or no read access. ",
      L" Target is not a \"Regular\" file. ",
      L" Log file contains no data. ",
      L" Incorrect format for log file. ",
   },
   //** Español **
   {
      L" Datos escritos en el archivo de registro. ", // Data written to log file.
      L" Archivo de registro reinicializado. ",       // Log File Reinitialized.
      L" Operación cancelada. Registro no escrito. ",  // Operation cancelled. Record not written.
      L" No se pudo encontrar el archivo de registro. ", // Unable to find log file.
      L" No se puede crear un registro o no hay acceso de escritura. ", // Cannot create log or no write access.
      L" Archivo no encontrado o sin acceso de lectura. ", // File not found or no read access.
      L" Target no es un archivo \"Regular\". ",      // Target is not a "Regular" file.
      L" El archivo de registro no contiene datos. ", // Log file contains no data.
      L" Formato incorrecto para el archivo de registro. ", // Incorrect format for log file.
   },
   //** Zhōngwén (中文) **
   {
      L" 数据写入日志文件。 ",             // Data written to log file.
      L" 日志文件已重新初始化。 ",          // Log File Reinitialized.
      L" 操作已取消。 记录未写。 ",         // Operation cancelled. Record not written.
      L" 找不到日志文件。 ",              // Unable to find log file.
      L" 无法创建日志或没有写访问权限。 ",   // Cannot create log or no write access.
      L" 找不到文件或没有读取权限。 ",      // File not found or no read access.
      L" 目标不是“常规”文件。 ",          // Target is not a "Regular" file.
      L" 日志文件不包含任何数据。 ",       // Log file contains no data.
      L" 日志文件格式错误。 ",            // Incorrect format for log file.
   },
   //** TiếngViệt **
   {
      L" Dữ liệu được ghi vào tệp nhật ký. ",      // Data written to log file.
      L" Tệp nhật ký được khởi động lại. ",        // Log File Reinitialized.
      L" Đã hủy hoạt động. Dữ liệu không được ghi. ", // Operation cancelled. Record not written.
      L" Không thể tìm thấy tệp nhật ký. ",        // Unable to find log file.
      L" Không thể tạo nhật ký hoặc không có quyền ghi. ", // Cannot create log or no write access.
      L" Không tìm thấy tệp hoặc không có quyền đọc. ", // File not found or no read access.
      L" Mục tiêu không phải là tệp \"Đều đặn\". ",// Target is not a "Regular" file.
      L" Tệp nhật ký không chứa dữ liệu. ",        // Log file contains no data.
      L" Định dạng không chính xác cho tệp nhật ký. ", // Incorrect format for log file.
   },
} ;

//* Logfile Identifier (first line of file) *
const short LogID_CompChars = 8 ; // number of chars to compare for file validation
const char* const LogID[LangCount] = 
{
   //** English **
   "# EC_LOG: Exercalc: Log File",
   //** Español **
   "# EC_LOG: Exercalc: Archivo de Registro",
   //** Zhōngwén (中文) **
   "# EC_LOG: Exercalc: 日志文件",
   //** TiếngViệt **
   "# EC_LOG: Exercalc: Tệp nhật ký",
} ;

//* Warning message if info/html help file not found.*
const wchar_t* const NoHelp_info[LangCount] = 
{  // "  ALERT! - ¡Notificación! - 警报! - Thông báo!  ",
   L" Help file 'exercalc.info' was not found. ",
   L" No se encontró el archivo de ayuda 'exercalc.info'. ",
   L" 无法找到帮助文件 'exercalc.info'。 ",
   L" Không tìm thấy tệp trợ giúp 'exercalc.info'. ",
} ;
const wchar_t* const NoHelp_html[LangCount] = 
{
   L" Help file 'exercalc.html' was not found. ",
   L" No se encontró el archivo de ayuda 'exercalc.html'. ",
   L" 无法找到帮助文件 'exercalc.html'。 ",
   L" Không tìm thấy tệp trợ giúp 'exercalc.html'. ",
} ;

//* 'Yes'/'No' and 'OK' labels *
const char* const Yes[LangCount] = 
{
   //** English **
   "    YES    ",
   //** Español **
   "    SI    ",
   //** Zhōngwén (中文) **
   "    是    ",
   //** TiếngViệt **
   "    Đúng    ",
} ;
const char* const No[LangCount] = 
{
   //** English **
   "    NO     ",
   //** Español **
   "    NO    ",
   //** Zhōngwén (中文) **
   "   没有   ",
   //** TiếngViệt **
   "   Không   ",
} ;
const char* const Ok[LangCount] = 
{
   //** English **
   "   OK   ",
   //** Español **
   "  OKAY  ",
   //** Zhōngwén (中文) **
   "   好   ",
   //** TiếngViệt **
   "  Đồng ý  ",
} ;

const wchar_t* const StatInfoTitle[LangCount] = 
{
   //** English **
   L"   Metabolic Equivalent of Task Table (METs)   \n",
   //** Español **
   L"   Tabla de Equivalentes Metabólicos de la Tarea (METs)   \n",
   //** Zhōngwén (中文) **
   L"   任务代谢当量表  (METs)   \n",
   //** TiếngViệt **
   L"   Biểu Đồ Tương Đương Trao Đổi Chất (METs)   \n",
} ;

//* Error message: Terminal window too small *
const char* const termsizeError[LangCount][2] = 
{
   //** English **
   {
   "Terminal window is too small: (%2hd x %3hd)",
   "Please resize window to at least %hd rows by %hd columns.",
   },
   //** Español **
   {
   "La ventana de la terminal es demasiado pequeña: (%2hd x %3hd)",
   "Cambie el tamaño de la ventana a al menos %hd filas por %hd columnas."
   },
   //** Zhōngwén (中文) **
   {
   "终端窗口太小： (%2hd x %3hd)",
   "将窗口大小调整为至少 %hd 行乘 %hd 列。",
   },
   //** TiếngViệt **
   {
   "Cửa sổ đầu cuối quá nhỏ: (%2hd x %3hd)",
   "Vui lòng thay đổi kích thước cửa sổ thành ít nhất %hd hàng x %hd cột.",
   },
} ;

//* Error message: Temporary-files directory *
const wchar_t* const tmpdirError[LangCount] = 
{
   //** English **
   L"ERROR: Directory for creating temporary files is inaccessible.",
   //** Español **
   L"ERROR: El directorio para crear archivos temporales es inaccesible.",
   //** Zhōngwén (中文) **
   L"错误：无法访问用于创建临时文件的目录。",
   //** TiếngViệt **
   L"Lỗi: Không thể truy cập thư mục được sử dụng để tạo tệp tạm thời.",
} ;


//* Column headings for display of lookup tables *
const wchar_t* const StatInfoHdr[LangCount][ExerTypes] = 
{
   //** English **
   {
   L" METs  Description -- WALKING ACTIVITY",
   L" METs  Description -- BICYCLING ACTIVITY",
   L" METs  Description -- RUNNING ACTIVITY",
   L" METs  Description -- GENERAL ACTIVITY",
   },
   //** Español **
   {
   L" METs  Descripción -- ACTIVIDAD A PIE",
   L" METs  Descripción -- ACTIVIDAD DE BICICLETA",
   L" METs  Descripción -- ACTIVIDAD DE CORRER",
   L" METs  Descripción -- ACTIVIDAD FÍSICA GENERAL",
   },
   //** Zhōngwén (中文) **
   {
   L" METs  描述  --  步行活动",
   L" METs  描述  --  骑车活动",
   L" METs  描述  --  跑步活动",
   L" METs  描述  --  常见的运动类型",
   },
   //** TiếngViệt **
   {
   L" METs  Sự miêu tả -- HOẠT ĐỘNG ĐI BỘ",
   L" METs  Sự miêu tả -- HOẠT ĐỘNG XE ĐẠP",
   L" METs  Sự miêu tả -- HOẠT ĐỘNG CHẠY",
   L" METs  Sự miêu tả -- HOẠT ĐỘNG THỂ CHẤT CHUNG",
   },
} ;

//**********************
//* METs Lookup Tables *
//**********************
// Programmer's Note: When creating METs table text for RTL languages, it 
// will be necessary to reverse the numeric data within the source text so 
// it will be rendered correctly when written to the screen as RTL.
const wchar_t* const StatInfo[LangCount][ExerTypes] = 
{
   //** English **
   {
   //* Walking *
   //n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxxx
   L"2.0   walking, <3.2 kph (<2.0 mph), level, very slow\n"
    "2.5   walking, 3.22 kph (2.0 mph), level, firm surface, slow\n"
    "2.8   walking, 4.02 kph (2.5 mph), downhill\n"
    "3.0   walking, 4.02 kph (2.5 mph), firm surface\n"
    "3.3*  walking, 4.83 kph (3.0 mph), level, firm, moderate\n"
    "3.5   walking, for pleasure, relaxation\n"
    "3.8   walking, 5.63 kph (3.5 mph), level,firm,brisk,exercise\n"
    "4.0   walking, to work or to class, focused\n"
    "5.0   walking, using crutches\n"
    "5.0   walking, grass track\n"
    "5.0   walking, 6.44 kph (4.0 mph), level, firm, very brisk\n"
    "6.0   walking, 5.63 kph (3.5 mph), uphill\n"
    "6.0   hiking, cross country\n"
    "6.3   walking, 7.24 kph (4.5 mph), level,firm,extremely brisk\n"
    "6.5   race walking\n"
    "7.0   backpacking\n"
    "8.0   walking, 8.05 kph (5.0 mph)\n"
    "\n"
    " * default value\n",

   //* Bicycling *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 3.0   stationary, 50 watts, very light effort\n"
    " 4.0   <16.1 kph (<10 mph), leisure, to work or for pleasure\n"
    " 5.0   unicycling\n"
    " 5.5   stationary, 100 watts, light effort\n"
    " 6.0   16-19.2 kph (10-11.9 mph), leisure, slow, light effort\n"
    " 7.0   stationary, general\n"
    " 7.0   stationary, 150 watts, moderate effort\n"
    " 8.0*  general\n"
    " 8.0   19-22.4 kph (12-13.9 mph), leisure, moderate effort\n"
    " 8.5   BMX or mountain\n"
    "10.0   22-25.6 kph (14-15.9 mph), fast, vigorous effort\n"
    "10.5   stationary, 200 watts, vigorous effort\n"
    "12.5   stationary, 250 watts, very vigorous effort\n"
    "12.0   25-30.6 kph (16-19 mph), racing, general, very fast\n"
    "16.0   >32 kph (>20 mph), racing, not drafting\n"
    "\n"
    " * default value\n",

   //* Running *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 3.0   running, wheeling, general\n"
    " 4.5   jogging on a mini-tramp\n"
    " 6.0   jog/walk combination (jogging component <10 minutes)\n"
    " 7.0*  jogging, general\n"
    " 8.0   jogging, in place\n"
    " 8.0   running\n"
    " 8.0   running, in place\n"
    " 8.0   running, 8.05 kph (7.45 min/km), 5 mph (12 min/mi)\n"
    " 8.0   running, training, pushing a wheelchair\n"
    " 9.0   running, 8.37 kph (7.17 min/km), 5.2 mph (11.5 min/mi)\n"
    " 9.0   running, cross country\n"
    "10.0   running, 9.66 kph (6.21 min/km), 6 mph (10 min/mi)\n"
    "10.0   running, on a track, team practice\n"
    "11.0   running, 10.78 kph (5.56 min/km), 6.7 mph (9 min/mi)\n"
    "11.5   running, 11.27 kph (5.33 min/km), 7 mph (8.5 min/mi)\n"
    "12.5   running, 12.07 kph (4.97 min/km), 7.5 mph (8 min/mi)\n"
    "13.5   running, 12.87 kph (4.66 min/km), 8 mph (7.5 min/mi)\n"
    "14.0   running, 13.84 kph (4.34 min/km), 8.6 mph (7 min/mi)\n"
    "15.0   running, 14.48 kph (4.14 min/km), 9 mph (6.5 min/mi)\n"
    "15.0   running, stairs, up\n"
    "16.0   running, 16.09 kph (3.73 min/km), 10 mph (6 min/mi)\n"
    "18.0   running, 17.54 kph (3.42 min/km), 10.9 mph (5.5 min/mi)\n"
    "\n"
    " * default value\n",

   //* General *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 1.0   conditioning  whirlpool, sitting\n"
    " 2.5   conditioning  stretching, hatha yoga\n"
    " 2.5   conditioning  mild stretching\n"
    " 3.0   conditioning  weight lifting, light or moderate effort\n"
    " 3.5   conditioning  calisthenics, light or moderate effort\n"
    " 3.5   conditioning  rowing, stationary, light effort\n"
    " 4.0   conditioning  water aerobics, water calisthenics\n"
    " 5.5*  conditioning  health club exercise, general\n"
    " 6.0   conditioning  weight lifting, vigorous effort\n"
    " 6.0   conditioning  slimnastics,jazzercise\n"
    " 7.0   conditioning  rowing, stationary, moderate effort\n"
    " 7.0   conditioning  ski machine, general\n"
    " 8.0   conditioning  calisthenics, heavy, vigorous effort\n"
    " 8.5   conditioning  rowing, stationary, vigorous effort\n"
    " 9.0   conditioning  stair-treadmill ergometer, general\n"
    "12.0   conditioning  rowing, stationary, very vigorous effort\n"
    " 5.0   dancing       aerobic, low impact\n"
    " 6.5   dancing       aerobic, general\n"
    " 7.0   dancing       aerobic, high impact\n"
    " 8.5   dancing       aerobic, step, with 15-20cm (6-8in) step\n"
    "10.0   dancing       aerobic, step, with 25-30cm (10-12in)step\n"
    " 2.5   lawn/garden   mowing lawn, riding mower\n"
    " 4.5   lawn/garden   mowing lawn, power mower\n"
    " 5.0   lawn/garden   digging, spading, garden, composting\n"
    " 5.5   lawn/garden   mowing lawn, walk, power mower, general\n"
    " 5.5   lawn/garden   mowing lawn, walk, power mower\n"
    " 6.0   lawn/garden   mowing lawn, walk, hand mower\n"
    "\n"
    " * default value\n",
   },

   //** Español **
   {
    //* Walking *
   //n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxxx
   L"2.0   caminando, <3,2 kph (2,0mph), nivel, muy lento"
    "2.5   caminando, 3,2 kph (2,0mph), nivelado, firme, lento\n"
    "2.8   caminando, 4,0 kph (2,5mph), cuesta abajo\n"
    "3.0   caminando, 4,0 kph (2,5mph), superficie firme\n"
    "3.3*  caminando, 4,8 kph (3,0mph), Nivel, firme, moderado\n"
    "3.5   caminando, por placer, relajación\n"
    "3.8   caminando, 5,6 kph (3,5mph), nivel, firme, enérgico\n"
    "4.0   caminando, al trabajo oa la clase, enfocado\n"
    "5.0   caminando, usando muletas\n"
    "5.0   caminando, en una pista de hierba\n"
    "5.0   caminando, 6,4 kph (4,0mph), Nivel, firme, muy enérgico\n"
    "6.0   caminando, 5,6 kph (3,5mph), cuesta arriba\n"
    "6.0   senderismo, campo a través\n"
    "6.3   caminando, 7,2 kph (4,5mph), Nivel, firme, muy enérgico\n"
    "6.5   carrera caminando\n"
    "7.0   mochilero\n"
    "8.0   caminando, 8,0 kph (5,0mph)\n"
    "\n"
    " * valor por defecto\n",

    //* Bicycling *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 3.0   Estacionario, 50 vatios, esfuerzo muy ligero.\n"
    " 4.0   <16.1 kph (<10mph), ocio, trabajo o placer\n"
    " 5.0   monociclo\n"
    " 5.5   Estacionario, 100 vatios, esfuerzo ligero\n"
    " 6.0   16-19.2 kph (10-11.9mph), ocio, esfuerzo lento, ligero\n"
    " 7.0   Estacionario, general\n"
    " 7.0   estacionario, 150 vatios, esfuerzo moderado\n"
    " 8.0*  general\n"
    " 8.0   19-22.4 kph (12-13.9mph), ocio, esfuerzo moderado\n"
    " 8.5   BMX o montaña\n"
    "10.0   22-25.6 kph (14-15.9mph), esfuerzo rápido y vigoroso\n"
    "10.5   stationary, 200 watts, vigorous effort\n"
    "12.5   stationary, 250 watts, very vigorous effort\n"
    "12.0   25-30.6 kph (16-19mph), carreras, general, muy rápido\n"
    "16.0   >32 kph (>20mph), carreras, campo abierto\n"
    "\n"
    " * valor por defecto\n",

    //* Running *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 3.0   corriendo, casual, general\n"
    " 4.5   trotar en un mini-trampolín\n"
    " 6.0   combinación trotar/caminar (trotar <10 minutos)\n"
    " 7.0*  trotar, general\n"
    " 8.0   trotar en el lugar\n"
    " 8.0   corriendo\n"
    " 8.0   corriendo en su lugar\n"
    " 8.0   corriendo, 8.05 kph (7.45 minutos/km)   [5 mph]\n"
    " 8.0   corriendo, entrenando, empujando una silla de ruedas\n"
    " 9.0   corriendo, 8.37 kph (7.17 minutos/km)   [5.2 mph]\n"
    " 9.0   corriendo, a campo traviesa\n"
    "10.0   corriendo, 9.66 kph (6.21 minutos/km)   [6 mph]\n"
    "10.0   corriendo, en una pista, práctica en equipo\n"
    "11.0   corriendo, 10.78 kph (5.56 minutos/km)  [6.7 mph]\n"
    "11.5   corriendo, 11.27 kph (5.33 minutos/km)  [7 mph]\n"
    "12.5   corriendo, 12.07 kph (4.97 minutos/km)  [7.5 mph]\n"
    "13.5   corriendo, 12.87 kph (4.66 minutos/km)  [8 mph]\n"
    "14.0   corriendo, 13.84 kph (4.34 minutos/km)  [8.6 mph]\n"
    "15.0   corriendo, 14.48 kph (4.14 minutos/km)  [9 mph]\n"
    "15.0   corriendo, por las escaleras\n"
    "16.0   corriendo, 16.09 kph (3.73 minutos/km)  [10 mph]\n"
    "18.0   corriendo, 17.54 kph (3.42 minutos/km)  [10.9 mph]\n"
    "\n"
    " * valor por defecto\n",

    //* General *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 1.0   condición física  Jacuzzi, sentado\n"
    " 2.5   condición física  estiramiento, hatha yoga\n"
    " 2.5   condición física  estiramiento suave\n"
    " 3.0   condición física  levantamiento de pesas, moderado\n"
    " 3.5   condición física  calistenia, ligero o moderado\n"
    " 3.5   condición física  remo, estacionario, esfuerzo ligero\n"
    " 4.0   condición física  aeróbic acuático, calistenia acuática\n"
    " 5.5*  condición física  ejercicio del club de salud, general\n"
    " 6.0   condición física  levantamiento de pesas, vigoroso\n"
    " 6.0   condición física  ejercicio para adelgazar, jazzercise\n"
    " 7.0   condición física  remo, estacionario, esfuerzo moderado\n"
    " 7.0   condición física  máquina de esquí, general\n"
    " 8.0   condición física  calistenia, esfuerzo pesado, vigoroso\n"
    " 8.5   condición física  remando, estacionario, vigoroso\n"
    " 9.0   condición física  ergómetro cinta rodante, general\n"
    "12.0   condición física  remando, parado, muy vigoroso\n"
    " 5.0   bailando          aeróbico de bajo impacto\n"
    " 6.5   bailando          aeróbico, general \n"
    " 7.0   bailando          aeróbico de alto impacto\n"
    " 8.5   bailando          aeróbicos de paso, con 15-20cm paso\n"
    "10.0   bailando          aeróbicos de paso, con 25-30cm paso\n"
    " 2.5   Césped y jardín   podar el cesped, cortacésped\n"
    " 4.5   Césped y jardín   podar el cesped, autopropulsada\n"
    " 5.0   Césped y jardín   excavación, pala, jardín, compostaje\n"
    " 5.5   Césped y jardín   podar el cesped, motorizado, general\n"
    " 5.5   Césped y jardín   podar el cesped, caminar, motorizado\n"
    " 6.0   Césped y jardín   podar el cesped, empujar la podadora\n"
    "\n"
    " * valor por defecto\n",
   },

   //** Zhōngwén (中文) **
   {
   //* Walking *
   //n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxxx
   L"2.0   步行, <3.2 千米/小时 [<2.0mph], 在水平地面上，非常慢\n"
    "2.5   步行, 3.2 千米/小时 [2.0mph], 在水平地面上缓慢\n"
    "2.8   步行, 4.0 千米/小时 [2.5mph], 走下坡路\n"
    "3.0   步行, 4.0 千米/小时 [2.5mph], 在水平地面上\n"
    "3.3*  步行, 4.8 千米/小时 [3.0mph], 在水平地面上，步伐适中\n"
    "3.5   步行, 为了娱乐或放松\n"
    "3.8   步行, 5.6 千米/小时 [3.5mph], 水平，坚硬表面，轻快，运动\n"
    "4.0   步行, 上班或上课，集中精力\n"
    "5.0   用拐杖\n"
    "5.0   步行, 在草地上\n"
    "5.0   步行, 6.4 千米/小时 [4.0mph], 水平，表面牢固，快速\n"
    "6.0   步行, 5.6 千米/小时 [3.5mph], 上山\n"
    "6.0   远足越野\n"
    "6.3   步行, 7.2 千米/小时 [4.5mph], 水平,表面牢固,非常快\n"
    "6.5   竞走\n"
    "7.0   背包旅行\n"
    "8.0   步行, 8.0 千米/小时 [5.0mph]\n"
    "\n"
    " * 默认值\n",

   //* Bicycling *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 3.0   固定，50瓦，省力\n"
    " 4.0   <16.1 千米/小时 [<10mph], 休闲，骑车上班, 休息时间\n"
    " 5.0   单轮车\n"
    " 5.5   固定，100瓦，省力\n"
    " 6.0   16-19.2 千米/小时 [10-11.9mph], 休闲，缓慢，轻松\n"
    " 7.0   固定，普通型\n"
    " 7.0   固定，150瓦，省力\n"
    " 8.0*  普通型\n"
    " 8.0   19-22.4 千米/小时 [12-13.9mph], 休闲，适度的努力\n"
    " 8.5   小轮车或山地自行车\n"
    "10.0   22-25.6 千米/小时 [14-15.9mph], 快速而努力\n"
    "10.5   固定，200瓦，费力\n"
    "12.5   固定的，250瓦，非常费力\n"
    "12.0   25-30.6 千米/小时 [16-19mph], 竞速，普通型, 非常快\n"
    "16.0   >32 千米/小时 [>20mph], 竞争, 在公开的轨道上\n"
    "\n"
    " * 默认值\n",

   //* Running *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 3.0   跑步，休闲，一般\n"
    " 4.5   在蹦床上慢跑\n"
    " 6.0   慢跑与步行组合\n"
    " 7.0*  跑步，普通型\n"
    " 8.0   慢跑而不向前运动\n"
    " 8.0   跑步\n"
    " 8.0   跑步， 运行到位\n"
    " 8.0   跑步， 8.05 每小时公里 (7.45 每公里分钟)   [5 mph]\n"
    " 8.0   跑步， 训练，推轮椅\n"
    " 9.0   跑步， 8.37 每小时公里 (7.17 每公里分钟)   [5.2 mph]\n"
    " 9.0   跑步， 越野跑\n"
    "10.0   跑步， 9.66 每小时公里 (6.21 每公里分钟),  [6 mph]\n"
    "10.0   跑步， 在轨道上，团队实践\n"
    "11.0   跑步， 10.78 每小时公里 (5.56 每公里分钟), [6.7 mph]\n"
    "11.5   跑步， 11.27 每小时公里 (5.33 每公里分钟), [7 mph]\n"
    "12.5   跑步， 12.07 每小时公里 (4.97 每公里分钟), [7.5 mph]\n"
    "13.5   跑步， 12.87 每小时公里 (4.66 每公里分钟), [8 mph]\n"
    "14.0   跑步， 13.84 每小时公里 (4.34 每公里分钟), [8.6 mph]\n"
    "15.0   跑步， 14.48 每小时公里 (4.14 每公里分钟), [9 mph]\n"
    "15.0   跑步， 上楼去\n"
    "16.0   跑步， 16.09 每小时公里 (3.73 每公里分钟), [10 mph]\n"
    "18.0   跑步， 17.54 每小时公里 (3.42 每公里分钟), [10.9 mph]\n"
    "\n"
    " * 默认值\n",

   //* General *
   // n.n  xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxxx
   L" 1.0  身体调节    漩涡浴，静静地坐着\n"
    " 2.5  身体调节    伸展运动，哈他瑜伽\n"
    " 2.5  身体调节    轻度拉伸\n"
    " 3.0  身体调节    举重，轻度或中等力度\n"
    " 3.5  身体调节    健美操，轻度或中度的努力\n"
    " 3.5  身体调节    划艇，平稳的，小努力\n"
    " 4.0  身体调节    水中有氧运动\n"
    " 5.5* 身体调节    健身俱乐部锻炼，普通型\n"
    " 6.0  身体调节    举重，努力\n"
    " 6.0  身体调节    减肥运动，爵士舞\n"
    " 7.0  身体调节    划艇，平稳的，适度的努力\n"
    " 7.0  身体调节    滑雪机，普通型\n"
    " 8.0  身体调节    健美操，努力\n"
    " 8.5  身体调节    划艇，平稳的，努力\n"
    " 9.0  身体调节    楼梯跑步机 测功计，普通型\n"
    "12.0  身体调节    划艇，平稳的，很努力\n"
    " 5.0  跳舞        健美操，影响低\n"
    " 6.5  跳舞        健美操，普通型\n"
    " 7.0  跳舞        健美操，重大影响\n"
    " 8.5  跳舞        健美操，踏板操，使用 15 – 20 厘米楼梯\n"
    "10.0  跳舞        健美操，踏板操，使用 25 – 30 厘米楼梯\n"
    " 2.5  草坪和花园  修剪草坪，骑马割草机\n"
    " 4.5  草坪和花园  修剪草坪，自推式割草机\n"
    " 5.0  草坪和花园  用铁锹挖，园艺工作，堆肥\n"
    " 5.5  草坪和花园  修剪草坪, 步行, 自推式割草机，普通型\n"
    " 5.5  草坪和花园  修剪草坪, 步行, 自推式割草机\n"
    " 6.0  草坪和花园  修剪草坪, 步行, 割草机\n"
    "\n"
    " * 默认值\n",
   },

   //** TiếngViệt **
   {
   //* Walking *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L"2.0   đi dạo, <3.2 kph (<2.0 mph), bề mặt bằng phẳng, rất chậm\n"
    "2.5   đi dạo, 3.22 kph (2.0 mph), bằng phẳng, chắc chắn, chậm\n"
    "2.8   đi dạo, 4.02 kph (2.5 mph), xuống dốc\n"
    "3.0   đi dạo, 4.02 kph (2.5 mph), Bề mặt vững chắc\n"
    "3.3*  đi dạo, 4.83 kph (3.0 mph), mặt bằng, chắc chắn,vừa phải\n"
    "3.5   đi dạo, cho niềm vui hoặc thư giãn\n"
    "3.8   đi dạo, 5.63 kph (3.5 mph), bằng phẳng,tốc độ nhanh\n"
    "4.0   đi dạo, đi làm hoặc đến lớp, tập trung\n"
    "5.0   đi dạo, sử dụng nạng\n"
    "5.0   đi dạo, con đường cỏ\n"
    "5.0   đi dạo, 6.44 kph (4.0 mph), bằng phẳng, tốc độ rất nhanh\n"
    "6.0   đi dạo, 5.63 kph (3.5 mph), lên dốc\n"
    "6.0   đi bộ đường dài, đi xuyên đất nước\n"
    "6.3   đi dạo, 7.24 kph (4.5 mph), bằng phẳng,tốc độ cực nhanh\n"
    "6.5   cuộc thi đi bộ\n"
    "7.0   ba lô\n"
    "8.0   đi dạo, 8.05 kph (5.0 mph)\n"
    "\n"
    " * giá trị mặc định\n",

   //* Bicycling *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 3.0   xe đạp tại chỗ, 50 oát, nỗ lực rất nhẹ\n"
    " 4.0   <16.1 kph (<10 mph), nhàn nhã, đi làm, vì niềm vui\n"
    " 5.0   xe đạp một bánh\n"
    " 5.5   xe đạp tại chỗ, 100 oát, nỗ lực nhẹ\n"
    " 6.0   16-19.2 kph (10-11.9 mph), nhàn nhã, nhẹ nhàng\n"
    " 7.0   xe đạp tại chỗ, chung\n"
    " 7.0   xe đạp tại chỗ, 150 oát, nỗ lực vừa phải\n"
    " 8.0*  chung\n"
    " 8.0   19-22.4 kph (12-13.9 mph), nhàn nhã, nỗ lực vừa phải\n"
    " 8.5   BMX hoặc xe đạp leo núi\n"
    "10.0   22-25.6 kph (14-15.9 mph), nỗ lực nhanh chóng, mạnh mẽ\n"
    "10.5   xe đạp tại chỗ, 200 oát, nỗ lực mạnh mẽ\n"
    "12.5   xe đạp tại chỗ, 250 oát, nỗ lực rất mạnh mẽ\n"
    "12.0   25-30.6 kph (16-19 mph), đua xe, nói chung, rất nhanh\n"
    "16.0   >32 kph (>20 mph), đua xe, đường đua mở\n"
    "\n"
    " * giá trị mặc định\n",

   //* Running *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 3.0   chạy, giải trí, nói chung\n"
    " 4.5   chạy bộ trên tấm bạt lò xo mini\n"
    " 6.0   chạy bộ/đi bộ sự phối hợp (chạy bộ <10 phút)\n"
    " 7.0*  chạy bộ, nói chung\n"
    " 8.0   chạy bộ tại chỗ\n"
    " 8.0   đang chạy\n"
    " 8.0   đang chạy, tại chỗ\n"
    " 8.0   đang chạy, 8.05 km mỗi giờ (7.45 phút mỗi km), [5mph]\n"
    " 8.0   đang chạy, đào tạo, đẩy xe lăn\n"
    " 9.0   đang chạy, 8.37 km mỗi giờ (7.17 phút mỗi km), [5.2mph]\n"
    " 9.0   đang chạy, xuyên quốc gia\n"
    "10.0   đang chạy, 9.66 km mỗi giờ (6.21 phút mỗi km)  [6mph]\n"
    "10.0   đang chạy, trên đường đua, luyện tập theo nhóm\n"
    "11.0   đang chạy, 10.78 km mỗi giờ (5.56 phút mỗi km),[6.7mph]\n"
    "11.5   đang chạy, 11.27 km mỗi giờ (5.33 phút mỗi km), [7mph]\n"
    "12.5   đang chạy, 12.07 km mỗi giờ (4.97 phút mỗi km),[7.5mph]\n"
    "13.5   đang chạy, 12.87 km mỗi giờ (4.66 phút mỗi km), [8mph]\n"
    "14.0   đang chạy, 13.84 km mỗi giờ (4.34 phút mỗi km),[8.6mph]\n"
    "15.0   đang chạy, 14.48 km mỗi giờ (4.14 phút mỗi km), [9mph]\n"
    "15.0   đang chạy, lên cầu thang\n"
    "16.0   đang chạy, 16.09 km mỗi giờ (3.73 phút mỗi km), [10mph]\n"
    "18.0   đang chạy, 17.54 km mỗi giờ (3.42 phút mỗi km), [11mph]\n" // [10.9mph]
    "\n"
    " * giá trị mặc định\n",

   //* General *
   // n.n   xxxxxxxxxxxxxx Max Display Columns xxxxxxxxxxxxxxxxxxxx
   L" 1.0   Điều kiện thể chất  bồn tắm nước xoáy, ngồi lặng lẽ\n"
    " 2.5   Điều kiện thể chất  kéo dài, hatha yoga\n"
    " 2.5   Điều kiện thể chất  kéo dài nhẹ\n"
    " 3.0   Điều kiện thể chất  Cử tạ, nỗ lực nhẹ đến vừa phải\n"
    " 3.5   Điều kiện thể chất  bài tập thể dục, nhẹ đến trung bình\n"
    " 3.5   Điều kiện thể chất  máy chèo thuyền, nhẹ nhàng\n"
    " 4.0   Điều kiện thể chất  thể dục nhịp điệu dưới nước\n"
    " 5.5*  Điều kiện thể chất  câu lạc bộ sức khỏe,chung\n"
    " 6.0   Điều kiện thể chất  Cử tạ, nỗ lực mạnh mẽ\n"
    " 6.0   Điều kiện thể chất  bài tập khiêu vũ,nhạc jazzercise\n"
    " 7.0   Điều kiện thể chất  máy chèo thuyền, vừa sức\n"
    " 7.0   Điều kiện thể chất  máy trượt tuyết, chung\n"
    " 8.0   Điều kiện thể chất  bài tập thể dục, nỗ lực mạnh mẽ\n"
    " 8.5   Điều kiện thể chất  máy chèo thuyền, nỗ lực mạnh mẽ\n"
    " 9.0   Điều kiện thể chất  Máy chạy bộ cầu thang, chung\n"
    "12.0   Điều kiện thể chất  máy chèo thuyền, cực kỳ nỗ lực\n"
    " 5.0   khiêu vũ      hiếu khí, tác động thấp\n"
    " 6.5   khiêu vũ      hiếu khí, chung\n"
    " 7.0   khiêu vũ      hiếu khí, tác động cao\n"
    " 8.5   khiêu vũ      bước thể dục nhịp điệu, 15-20cm (6-8in)\n"
    "10.0   khiêu vũ      bước thể dục nhịp điệu, 25-30cm (10-12in)\n"
    " 2.5   cừu con/vườn  cắt cỏ, cưỡi máy cắt cỏ\n"
    " 4.5   cừu con/vườn  cắt cỏ, máy cắt cỏ tự cung cấp năng lượng\n"
    " 5.0   cừu con/vườn  digging, spading, garden, composting\n"
    " 5.5   cừu con/vườn  cắt cỏ, máy cắt cỏ đi bộ, chung\n"
    " 5.5   cừu con/vườn  cắt cỏ, máy cắt cỏ đi bộ\n"
    " 6.0   cừu con/vườn  cắt cỏ, đi bộ, đẩy máy cắt cỏ\n"
    "\n"
    " * giá trị mặc định\n",
   },
} ;

   //** English **
   //** Español **
   //** Zhōngwén (中文) **
   //** TiếngViệt **


#endif   // ECLANG_INCLUDED
