//******************************************************************************
//* File       : EcConfig.hpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2021 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 27-Apr-2021                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Support data for the Exercalc application:                    *
//*              a) Configuration data and dialog definitions.                 *
//*              b) Definitions and data for file access.                      *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

#ifndef ECCONFIG_INCLUDED
#define ECCONFIG_INCLUDED

//****************
//* Header Files *
//****************

//***************
//* Definitions *
//***************

const short MIN_ROWS  = 36 ;        //* Minimum lines needed for dialog window
const short MIN_COLS  = 132 ;       //* Minimum columns needed for dialog window
const short USER_ROWS = 11 ;        //* Minimum rows in user-data area

//* Configuration information. Used for parsing the configuration file.*

//* String-comparison object definition *
// Usage Note: Not all members are used in all circumstances.
//       'str' used for each instance
//       'len' used for each instance (needed for comparisons)
//    'argcnt' used only in CfgParm[] : indicates number of args in attached array
//      'dflt' used only in CfgParm[] : indicates default member in attached array
//      'args' used only in CfgParm[] : attach point for argument array
class CfgComp
{
   public:
   const wchar_t* str ;    // pointer to command string
   short len ;             // command-string length (not incl. NULLCHAR)
   short argcnt ;          // number of valid arguments for command (1 if dynamic value)
   short dflt ;            // 'default' member of the argument array
   const CfgComp* args ;   // pointer to array of arg description objects
} ;


//********
//* Data *
//********

//* Names for accessing members of CfgParm[] array (see below) *
enum cciArguments : short 
{ 
   cciHT = ZERO,  // Height
   cciBM,         // Body Mass
   cciAG,         // Age
   cciGE,         // Gender
   cciET,         // Exercise Type
   cciKB,         // Kph Bicycling
   cciKW,         // Kph Walking
   cciKR,         // Kph Running
   cciMW,         // WalkIntensity (METs)
   cciMB,         // BikeIntensity (METs)
   cciMR,         // RunIntensity  (METs)
   cciMG,         // GeneralIntensity (METs)
   cciGL,         // Goal for exercise
   cciLP,         // Logfile Path
   cciLS,         // Logfile auto-save
   cciLA,         // Language
   cciLO,         // Locale
   cciCS,         // ColorScheme
   cciEM,         // EnableMouse
   cciEC,         // EndConfigFile
   cciArgs        // (number of records in array)
} ;

const CfgComp geArgs[] =                  // 'Gender' arguments
{
   { L"Male",    4, 0, 0, NULL },
   { L"Female",  6, 0, 0, NULL },
   { L"Other",   5, 0, 0, NULL },
} ;

const CfgComp etArgs[] =                  // 'ExerciseType' arguments
{
   { L"Walk",    4, 0, 0, NULL },
   { L"Bike",    4, 0, 0, NULL },
   { L"Run",     3, 0, 0, NULL },
   { L"General", 7, 0, 0, NULL },
} ;

static const CfgComp glArgs[] =           // 'Goal' arguments
{
   { L"Weekly",  6, 0, 0, NULL },
   { L"Daily",   5, 0, 0, NULL },
} ;

static const CfgComp laArgs[] =           // 'Language' arguments
{  //* NOTE: These are in the same order as 'enum AppLang' *
   { L"English",    7, 0, 0, NULL },
   { L"Espanol",    7, 0, 0, NULL },
   { L"Zhongwen",   8, 0, 0, NULL },
   { L"TiengViet",  9, 0, 0, NULL },
   { L"locale",     6, 0, 0, NULL },      // (default)
} ;

static const CfgComp loArgs[] =           // 'Locale' arguments
{
   { L"locale",     6, 0, 0, NULL },      // (default)
   // Otherwise, a valid locale in the form: 'nn_NN.utf8'
} ;

static const CfgComp csArgs[] =           // 'ColorScheme' arguments
{
   { L"Black",     5, 0, 0, NULL },
   { L"Red",       3, 0, 0, NULL },
   { L"Green",     5, 0, 0, NULL },
   { L"Brown",     5, 0, 0, NULL },       // (default)
   { L"Blue",      4, 0, 0, NULL },
   { L"Magenta",   7, 0, 0, NULL },
   { L"Cyan",      4, 0, 0, NULL },
   { L"Gray",      4, 0, 0, NULL },
} ;

const CfgComp emArgs[] =                  // 'EnableMouse' and 'LogfileAuto' arguments
{
   { L"disable", 7, 0, 0, NULL },         // (default)
   { L"enable",  6, 0, 0, NULL },
} ;

const CfgComp CfgParm[cciArgs] =   // Configuration Parameters
{
   { L"Height",                6,  1, 0, NULL   },
   { L"BodyMass",              8,  1, 0, NULL   },
   { L"Age",                   3,  1, 0, NULL   },
   { L"Gender",                6,  3, 0, geArgs },
   { L"ExerciseType",         12,  4, 0, etArgs },
   { L"AverageVelocity_Bike", 20,  1, 0, NULL   },
   { L"AverageVelocity_Walk", 20,  1, 0, NULL   },
   { L"AverageVelocity_Run",  19,  1, 0, NULL   },
   { L"Walk_METs",             9,  1, 0, NULL   },
   { L"Bike_METs",             9,  1, 0, NULL   },
   { L"Run_METs",              8,  1, 0, NULL   },
   { L"General_METs",         12,  1, 0, NULL   },
   { L"Goal",                  4,  2, 0, glArgs },
   { L"LogfilePath",          11,  1, 0, NULL   },
   { L"LogfileAuto",          11,  2, 0, emArgs },
   { L"Language",              8,  5, 0, laArgs },
   { L"Locale",                6,  1, 0, loArgs },
   { L"ColorScheme",          11,  8, 3, csArgs },
   { L"EnableMouse",          11,  2, 0, emArgs },
   { L"EndConfigFile",        13,  1, 0, NULL   },
} ;

//* Default values for numeric config options for which no value is specified.*
const float DEFAULT_Kph_Walk  =  5.63 ; //* Walking  :  3.5 kph (3.8 METS) ( 3.5mph)
const float DEFAULT_Kph_Bike  = 19.31 ; //* Bicycling: 12.0 kph (8.0 METS) (12.0mph)
const float DEFAULT_Kph_Run   = 12.87 ; //* Running  :  8.0 kph (8.0 METS) ( 8.0mph)
const float DEFAULT_Mets_Walk =  3.30 ; //* Walking  :  3.0 mph, moderate pace, firm surface
const float DEFAULT_Mets_Bike =  8.00 ; //* Bicycling:  bicycling general
const float DEFAULT_Mets_Run  =  7.00 ; //* Running  :  jogging general
const float DEFAULT_Mets_Gen  =  5.50 ; //* General  :  health club general


//*******************************
//*** File-access definitions ***
//*******************************

//* Local definitions for file type encoded in the "st_mode" *
//* element of the stat64{} structure                        *
enum fmFType : short
{ fmDIR_TYPE,   fmREG_TYPE,  fmLINK_TYPE, fmCHDEV_TYPE, 
  fmBKDEV_TYPE, fmFIFO_TYPE, fmSOCK_TYPE, fmUNKNOWN_TYPE, fmTYPES } ;

//* The following c-type structures defined in the standard header files 
//* need to be C++ data types: stat{} dirent{}, etc.
typedef struct stat64 FileStats ;
//typedef struct dirent64 deStats ;
typedef struct tm Tm ;
//typedef struct passwd PwdInfo ;
//typedef struct group GrpInfo ;
//typedef struct utimbuf uTime ;

const short MAX_PATH = PATH_MAX ;      //* Size of buffer to hold path/filename
const short MAX_FNAME = 255 ;          //* Size of buffer to hold filename (POSIX)

//* Name of default configuration file.                         *
//* File must be located in the same director as the executable.*
const char* const ecCfg = "ecalc.cfg" ;


//* Class for reporting the system local date/time.                            *
//* ---------------------------------------------------------------------------*
#define DECODE_TZ (0)   // if non-zero, decode time-zone fields of localTime
const int64_t maxETIME32 = 0x07FFFFFFF,      // largest 32-bit time_t
              minETIME32 = -maxETIME32 - 1 ; // smallest 32-bit time_t
const short ltFMT_LEN = 16 ;                 // length of time string buffers
const short ltISO_LEN = 24 ;
//* Important note about the year 2038 bug:                                    *
//* Because 'time_t' is defined as a signed, 32-bit value on 32-bit systems,   *
//* time reporting will break down for date/time values                        *
//*                     >= 19 Jan 2038 at 11:14:08                             *
//* For this reason and for others, we have defined our own local-time class.  *
//* This class uses a signed, 64-bit value for storing the epoch time.         *
//* It is hoped that all Linux systems will have been updated long before this *
//* becomes an issue; however, even now (2013), some applications are using    *
//* date/time manipulations that exceed the 32-bit limit. BEWARE!!             *
//* NOTE: time arithmetic should always be performed on signed 64-bit values.  *
//*                                                                            *
class localTime
{
   public:
   localTime ( void ) { this->reset() ; }
   void reset ( void )
   {  //* Set to epoch date: Thursday, 01 Jan. 1970 @ 08:00:00
      this->sysepoch = ZERO ;
      this->epoch = ZERO ;
      this->day = 4 ;
      this->year = 1970 ;
      this->month = this->date = 1 ;
      this->hours = 8 ;
      this->minutes = this->seconds = ZERO ;
      this->julian = ZERO ;
      *this->iso = NULLCHAR ;
      #if DECODE_TZ != 0   // time-zone data
      *this->timezone = NULLCHAR ;
      *this->utc_zone  = NULLCHAR ;
      this->gmtoffset = ZERO ;
      this->dst       = false ;
      #endif   // DECODE_TZ
   }
   int64_t  epoch ;        //* Seconds since the epoch (01-Jan-1970)
   time_t   sysepoch ;     //* system epoch time (see note)
   uint16_t day ;          //* Day of the week (0 == Sun, ..., 6 == Sat)
   uint16_t date ;         //* Day of the month (1-[28|29|30|31])
   uint16_t month ;        //* Month of the year (1 == Jan, ... 12 == Dec)
   uint16_t year ;         //* Year (four digit year)
   uint16_t hours ;        //* Time of day, hour (0-23)
   uint16_t minutes ;      //* Time of day, minutes (0-59)
   uint16_t seconds ;      //* Time of day, seconds (0-59)
   uint16_t julian ;       //* Completed days since beginning of year
   char iso[ltISO_LEN] ;   //* String representation: ISO 8601-1:2019 "yyyy-mm-ddThh:mm:ss"

   #if DECODE_TZ != 0   // time-zone data
   // See DecodeEpochTime method for more information.
   char     timezone[ltFMT_LEN] ; //* string representing the local time zone (useless)
   char     utc_zone[ltFMT_LEN] ; //* string representing the UTC offset for local time zone
   short    gmtoffset ;    //* Expressed as (signed) number of seconds offset from UTC+0:00
   bool     dst ;          //* 'true' if daylight savings time in effect,
                           //* 'false if not DST or info not available
   #endif   // DECODE_TZ
} ;

//* Class definition for human-readable file stats.                            *
class tnFName
{
   public:
   ~tnFName ( void ) {}       //* Destructor
   tnFName ( void )           //* Constructor
   { this->ReInit () ; }
   void ReInit(void)
   {
      fName[0] = NULLCHAR ;
      fType    = fmTYPES ;
      fBytes   = ZERO ;
      modTime.day = modTime.date = modTime.month = modTime.year 
                  = modTime.hours = modTime.minutes = modTime.seconds = ZERO ;
      readAcc  = false ;
      writeAcc = false ;
      // NOTE: data member, rawStats is left unitialized
   }
   char        fName[MAX_FNAME] ; //* Filename string
   fmFType     fType ;     //* File type (enum fmFType)
   UINT64      fBytes ;    //* File size (in bytes)
   localTime   modTime ;   //* Date/Time file last modified (human readable)
   FileStats   rawStats ;  //* Copy of system's "stat" structure (all file info)
   bool        readAcc ;   //* 'true' if user has read access to file
   bool        writeAcc ;  //* 'true' if user has write access to file 
} ;

#endif   // ECCONFIG_INCLUDED
