//******************************************************************************
//* File       : EcConfig.cpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2020-2021 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 01-Jul-2021                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Application setup and configuration methods.                  *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "Exercalc.hpp"    //* Class definition


//***************
//* Definitions *
//***************

//***************
//* Prototypes  *
//***************

//**********
//*  Data  *
//**********
extern InitNcDialog tdInit ;     // defined locally in Exercalc.cpp
extern InitNcDialog sdInit ;
extern InitNcDialog udInit ;


//*************************
//*       Configure       *
//*************************
//******************************************************************************
//* Read the configuration file and initialize data members.                   *
//* Configuration file must be in same directory as executable.                *
//*                                                                            *
//* Input  : cfgMsg  : pointer to an array of uninitialized diagnostic records *
//*                    Diagnostic messages are stored in the array, and will   *
//*                    be displayed by caller after NCurses Engine is started. *
//*          cfgMsgs : (by reference, initial value ignored)                   *
//*                    receives number of messages written to 'cfgMsg'         *
//*          verbose : (optional, 'false' by default)                          *
//*                    if 'true', output verbose diagnostics                   *
//*                                                                            *
//* Returns: ZERO if config file exists and all data valid                     *
//*          ERR  if config file not found or invalid format                   *
//*          else, number of syntax errors found                               *
//******************************************************************************
//* User selects the appropriate parameters for calculating energy expended:   *
//* 1) Height in centimeters (stored as meters for calculations)               *
//* 2) Body Mass in kilograms                                                  *
//* 3) Average velocity during exercise. (used for calculations)               *
//*    a) Bicycling                                                            *
//*    b) Walking                                                              *
//*    c) Running                                                              *
//* B) MET (Metaboloc Equivalent for Task)                                     *
//*    Taken from "Compendium Of Physical Activities"                          *
//*      http://prevention.sph.sc.edu/tools/docs/documents_compendium.pdf      *
//*    All values converted from English measurements to metric.               *
//*                                                                            *
//*    From: https://www.healthline.com/health/what-are-mets#examples          *
//*    A MET is a ratio of your working metabolic rate relative to your resting*
//*    metabolic rate. Metabolic rate is the rate of energy expended per unit  *
//*    of time. It’s one way to describe the intensity of an exercise or       *
//*    activity. One MET is the energy you spend sitting at rest — your resting*
//*    or basal metabolic rate. So, an activity with a MET value of 4 means    *
//*    you’re exerting four times the energy than you would if you were sitting*
//*    still. To put it in perspective, a brisk walk at 3 or 4 miles per hour  *
//*    has a value of 4 METs. Jumping rope, which is a more vigorous activity, *
//*    has a MET value of 12.3.                                                *
//*    To better understand METs, it’s helpful to know a little about how your *
//*    body uses energy.                                                       *
//*    -- The cells in your muscles use oxygen to help create the energy needed*
//*       to move your muscles. One MET is approximately 3.5 milliliters of    *
//*       oxygen consumed per kilogram (kg) of body weight per minute.         *
//*    -- So, for example, if you weigh 160 pounds (72.5 kg), you consume      *
//*       about 254 milliliters of oxygen per minute while you’re at rest      *
//*       (72.5 kg x 3.5 mL).                                                  *
//*    -- Energy expenditure may differ from person to person based on several *
//*       factors, including your age and fitness level. For example, a young  *
//*       athlete who exercises daily won’t need to expend the same amount of  *
//*       energy during a brisk walk as an older, sedentary person.            *
//*    -- For most healthy adults, MET values can be helpful in planning an    *
//*       exercise regimen, or at least gauging how much you’re getting out    *
//*       of your workout routine.                                             *
//*                                                                            *
//*                                                                            *
//* Kilocalorie calculations are based on:                                     *
//*    METs * 3.5 * mass(kg) / 200 == kCal burned per minute                   *
//*                                                                            *
//* METs x 3.5 x mass (kg) / 200 = Kcal/min.                                   *
//* For example, Shane is a 40-year old male who weighs 195 pounds.            *
//* You can use this formula to determine how many calories per minute he      *
//* uses during some of his regular activities:                                *
//* a) 2 hours of bicycling @ 12.0 mph (METs: 8.0)                             *
//*    8.0 x 3.5mL x 88.6kg / 200 = 12.4 Kcal/min x 120min = 1488 Kcal         *
//* b) 45 minutes of resistance training – explosive effort  (METs: 5.0)       *
//*    5.0 x 3.5mL x 88.6kg / 200 = 7.8 Kcal/min x 45min = 351 Kcal            *
//* c) 1 hour of sitting: (METs: 1.0)                                          *
//*    1.0 x 3.5mL x 88.6kg / 200 = 1.6 Kcal/min x 1min = 1.6 Kcal             *
//*                                                                            *
//* 1 MET = 3.5 x 75kg ÷ 200 x 1 min = 1.31 calories burned per minute         *
//* 1.31 calories x 60 minutes x 24 hours = 1,886 calories                     *
//* 12.5 METS x 3.5 x 75kg ÷ 200 x 30 mins = 492 calories                      *
//* One pound of fat contains approximately 3,500 calories of energy.          *
//*                                                                            *
//* METs per minute calculation:                                               *
//*    METs * 60 == MET minutes                                                *
//* HealthLine.com recommends 500-1000 MET minutes per week.                   *
//*                                                                            *
//* An individual’s capacity is the highest MET number he or she can sustain   *
//* for a few minutes, Earnest said. You can increase this capacity by getting *
//* fitter.                                                                    *
//* A healthy 50-year-old man should have a capacity of at least 9.2 METs;     *
//* a healthy 50-year-old woman should clock in at 8.2 METs or higher,         *
//* according to a recent study on women’s fitness in the New England Journal  *
//* of Medicine. For men age 20, 13.5 METs; age 30, 11.4 METs; age 40,         *
//* 10.3 METs. For women age 20, 12.1 METs; age 30, 10.8 METs; age 40,         *
//* 9.5 METs.                                                                  *
//* https://www.latimes.com/archives/la-xpm-2005-oct-24-he-mets24-story.html   *
//*                                                                            *
//* High Accuracy Calorie Calculator:                                          *
//*   https://keisan.casio.com/exec/system/1350891527                          *
//*   93.0kg: walk at 3.5mph for 1.0 hour yields 4.30 METs/hr and 321.0 kCal   *
//*                                                                            *
//* METs/Calorie Calculator:                                                   *
//*   https://metscalculator.com/                                              *
//*   93.0kg: walk at 3.5mph for 1.0 hour yields 4.30 METs/hr and 399.9 kCal   *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short Exercalc::Configure ( CfgMsg* cfgMsg, short& cfgMsgs, bool verbose )
{
   const char* const ConfigFileID = "ECALC" ;   // config file identifier
   const wchar_t wCOMMENT = wHASH ;             // comment follows

   gString cfgPath,           // config file path/filename
           cfgName,           // config file name
           gs, gsa, gsv ;     // work buffers
   short   status = ZERO ;    // return value

   //* If alternate config file specified *
   if ( this->cfg.cfgPath[0] != NULLCHAR )
   {
      cfgPath = this->cfg.cfgPath ;
      short nli = cfgPath.findlast( L'/' ) ;
      if ( nli >= ZERO ) ++nli ;
      else               nli = ZERO ;
      cfgName.compose( "\"%S\"", &cfgPath.gstr()[nli] ) ;
   }
   //* Else construct default filespec *
   else
   {
      cfgPath.compose( "%s/%s", this->cfg.appPath, ecCfg ) ;
      cfgName.compose( "\"%s\"", ecCfg ) ;
   }

   //* Verify that the target file exists *
   FileStats rawStats ;
   if ( (lstat64 ( cfgPath.ustr(), &rawStats )) == ZERO )
      cfgPath.copy( this->cfg.cfgPath, MAX_PATH ) ;
   else
      status = ERR ;

   gsv = "Read configuration file: " ;
   gsv.copy( cfgMsg[cfgMsgs].msg, MAX_FNAME ) ;
   cfgMsg[cfgMsgs].attr = &this->suGood ;
   cfgMsg[cfgMsgs++].nl = false ;
   cfgName.copy( cfgMsg[cfgMsgs].msg, MAX_FNAME ) ;
   cfgMsg[cfgMsgs++].attr = &this->suNorm ;

   //* If config file exists *
   if ( status == ZERO )
   {
      char  lineData[gsMAXBYTES] ;  // raw UTF-8 input
      short lineNum = 1 ;           // source-line counter
      bool  done = false ;          // loop control

      //* Open the file *
      ifstream ifs ( cfgPath.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         //* Verify that file is a valid Exercalc config file *
         ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
         if ( ifs.good() && (!(strncmp ( lineData, ConfigFileID, 5 ))) )
            ++lineNum ;
         else
         {
            status = ERR ;
            done = true ;
         }

         //* Read the parameters and store values in member variables *
         float fIn ;                // command argument value
         short ccIndex = ZERO,      // index into command array
               aaIndex = ZERO,      // index into argument array
               aIndex ;             // index into command string
         bool  badArg,              // internal syntax-error flag
               useDflt ;            // 'true' if no parameter value given

         while ( ! done )
         {
            //* Read a line from the file.*
            ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( (ifs.good()) || (ifs.gcount()) > ZERO )
            {
               gs = lineData ;             // load the data line
               gs.strip() ;                // strip leading and trailing whitespace

               //* If a comment line or an empty line, discard it.*
               if ( ((gs.find( wCOMMENT )) == ZERO)
                    || ((gs.gschars()) == 1) )
               { ++lineNum ; continue ; }

               //* Potentially valid configuration command.    *
               //* Scan the list of valid commands for a match.*
               for ( ccIndex = ZERO ; ccIndex < cciArgs ; ++ccIndex )
               {
                  if ( (gs.compare( CfgParm[ccIndex].str, true, 
                                    CfgParm[ccIndex].len )) == ZERO )
                  { break ; }
               }     // for(;;)

               //* If a valid configuration command *
               if ( ccIndex < cciArgs )
               {
                  //* If end of config file reached *
                  if ( ccIndex == cciEC )
                  {
                     done = true ;
                  }

                  //* Else, isolate the command argument (if any) *
                  else
                  {
                     //* Index the argument (if any) *
                     if ( (aIndex = gs.after( wEQUAL )) >= ZERO )
                     {
                        //* Discard the configuration-command token   *
                        //* and strip leading and trailing whitespace *
                        gsa = &gs.gstr()[aIndex] ;
                        gsa.strip() ;

                        badArg = useDflt = true ; // expect the worst

                        //* For config options which have const string *
                        //* arguments, scan for matching argument.     *
                        if ( CfgParm[ccIndex].args != NULL )
                        {
                           for ( aaIndex = ZERO ; aaIndex < CfgParm[ccIndex].argcnt ; 
                                                                      ++aaIndex )
                           {  //* (This test IS NOT case sensitive) *
                              if ( (gsa.compare( CfgParm[ccIndex].args[aaIndex].str,
                                     false, CfgParm[ccIndex].args[aaIndex].len )) == ZERO )
                              {
                                 badArg = false ;
                                 break ;     // found a match
                              }
                           }
                        }

                        //* If a constant-value argument not found, *
                        //* check for a numeric argument.           *
                        if ( badArg )
                        {
                           //* If a numeric argument specified *
                           if ( (gsa.gscanf( " %f", &fIn )) == 1 )
                              useDflt = false ;
                        }
                     }     // ('=' found)
                     else  // no argument (or bad syntax)
                        gsa.clear() ;

                     //* Process the configuration options *
                     switch ( ccIndex )
                     {
                        case cciHT:       // user's height
                           //* Capture the value (if any) and perform *
                           //* units conversion as necessary.         *
                           //* If no value specified, 'heightM'==0.0  *
                           this->DecodeLength2Meters ( gsa, this->ud.heightM ) ;
                           break ;

                        case cciBM:       // user's body-mass index
                           //* Capture the value (if any) and perform *
                           //* units conversion as necessary.         *
                           //* If no value specified, 'massKg'==0.0   *
                           this->DecodeMass2Kilograms ( gsa, this->ud.massKg ) ;
                           break ;

                        case cciAG:       // user's age
                           if ( useDflt )
                           {
                              //* An age of 0.0 indicates that age should *
                              //* not be used as a factor in calculations.*
                              this->ud.ageY = 0.0 ;
                           }
                           else
                           {
                              //* Years rounded to nearest tenth *
                              this->ud.ageY = (rintf ( fIn * 10.0 )) / 10.0 ;
                           }
                           break ;

                        case cciGE:       // user's gender
                           switch ( aaIndex )
                           {
                              case 0:  this->ud.gender = goMale ;    break ;
                              case 1:  this->ud.gender = goFemale ;  break ;
                              default: this->ud.gender = goNone ;    break ;
                           }
                           break ;

                        case cciKW:       // kph for walking
                           //* If parameter value not specified *
                           //* enter the default value.         *
                           if ( useDflt )
                              this->ud.kphWalk = DEFAULT_Kph_Walk ;
                           //* Convert the argument to KPH *
                           else
                              this->DecodeVelocity2Kph ( gsa, this->ud.kphWalk ) ;
                           break ;

                        case cciKB:       // kph for bicycling
                           //* If parameter value not specified *
                           //* enter the default value.         *
                           if ( useDflt )
                              this->ud.kphBike = DEFAULT_Kph_Bike ;
                           //* Convert the argument to KPH *
                           else
                              this->DecodeVelocity2Kph ( gsa, this->ud.kphBike ) ;
                           break ;

                        case cciKR:       // kph for running
                           //* If parameter value not specified *
                           //* enter the default value.         *
                           if ( useDflt )
                              this->ud.kphRun = DEFAULT_Kph_Run ;
                           //* Convert the argument to KPH *
                           else
                              this->DecodeVelocity2Kph ( gsa, this->ud.kphRun ) ;
                           break ;

                        case cciET:       // exercise type
                           switch ( aaIndex )
                           {
                              case 0:  this->ud.exerType = xtWalk ;     break ;
                              case 1:  this->ud.exerType = xtBike ;     break ;
                              case 2:  this->ud.exerType = xtRun ;      break ;
                              case 3:  this->ud.exerType = xtGeneral ;  break ;
                              default: this->ud.exerType = xtWalk ;     break ;
                           } ;
                           break ;

                        case cciMW:       // walking METs
                           //* Save METs value if specified, else it will   *
                           //* be calculated from 'kphWalk' (if available), *
                           //* or set to default value.                     *
                           if ( useDflt )  this->ud.metsWalk = DEFAULT_Mets_Walk ;
                           else            this->ud.metsWalk = fIn ;
                           break ;

                        case cciMB:       // bicycling METs
                           //* Save METs value if specified, else it will   *
                           //* be calculated from 'kphBike' (if available), *
                           //* or set to default value.                     *
                           if ( useDflt )  this->ud.metsBike = DEFAULT_Mets_Bike ;
                           else            this->ud.metsBike = fIn ;
                           break ;

                        case cciMR:       // running METs
                           //* Save METs value if specified, else it will   *
                           //* be calculated from 'kphRun' (if available),  *
                           //* or set to default value.                     *
                           if ( useDflt )  this->ud.metsRun = DEFAULT_Mets_Run ;
                           else            this->ud.metsRun = fIn ;
                           break ;

                        case cciMG:       // general exercise METs
                           //* Save METs value if specified, else it will   *
                           //* be set to default value.                     *
                           if ( useDflt )  this->ud.metsGen = DEFAULT_Mets_Gen ;
                           else            this->ud.metsGen = fIn ;
                           break ;

                        case cciGL:       // Goal for exercise
                           //* Decode the value and units and intensity     *
                           //* for the specified time period.               *
                           //* 'aaIndex' == Weekly or Daily                 *
                           //*   (default values set during instantiation)  *
                           // Programmer's Note: This is a multi-token entry,
                           // and thus requires special processing.
                           {
                           //* Initialize the daily/weekly flag *
                           if ( aaIndex == ZERO )  this->gWeek = true ;
                           else                    this->gWeek = false ;
                           //* Step past the Weekly/Daily parameter *
                           short indx ;
                           if ( (indx = gsa.after( wSPACE )) > ZERO )
                           {
                              //* Goal exercise type == preferred exercise type*
                              this->gXType = this->ud.exerType ;

                              //* Index the second parameter. This will be  *
                              //* a numeric (or time) value with units.     *
                              indx = gsa.scan( indx ) ;
                              gsa.shiftChars( -(indx) ) ;
                              if ( (fIn = this->Time2Minutes ( gsa.ustr())) > ZERO )
                              {
                                 this->gValue = fIn ; // save the captured value
                                 //* Step over the numeric data to units.*
                                 //* S/B no whitespace between value and *
                                 //* units, but we test for it anyway.   *
                                 indx = ZERO ;
                                 while ( (gsa.gstr()[indx] < L'A') &&
                                         (gsa.gstr()[indx] != L' ') &&
                                         (gsa.gstr()[indx] != L'\0') )
                                    ++indx ;
                                 indx = gsa.scan( indx ) ;
                                 gsa.shiftChars( -(indx) ) ;
                                 if ( (gsa.find( "mi" )) == ZERO )
                                    this->gUnits = vuMiles ;
                                 else if ( (gsa.find( "km" )) == ZERO )
                                    this->gUnits = vuKmeters ;
                                 else if ( (gsa.find( "kcal" )) == ZERO )
                                    this->gUnits = vuKcalories ;
                                 else if ( (gsa.find( "met" )) == ZERO )
                                 {
                                    this->gUnits = vuKcalories ;
                                    this->gValue = 
                                       this->Mets2kCal ( this->gValue, 
                                                         this->ud.massKg, 1 ) ;
                                 }
                                 else     // default units == time
                                    this->gUnits = vuMinutes ;

                                 //* Step over the second parameter.        *
                                 //* Capture the Intensity parameter (METs).*
                                 //* If METs value not provided, initialized*
                                 //* default will be retained.              *
                                 if ( (indx = gsa.find( L' ' )) > ZERO )
                                 {
                                    indx = gsa.scan( indx) ;
                                    gsa.shiftChars( -(indx) ) ;
                                    fIn = this->Time2Minutes ( gsa.ustr() ) ;
                                    if ( fIn > 0.0 )
                                       this->gMets = fIn ;
                                 }
                              }
                           }     // if()
                           }     // (compile trick)
                           break ;

                        case cciLP:       // log-file path
                           {
                           gString gstrg ;
                           if ( (this->LogfileCompose ( gsa, gstrg )) )
                              gstrg.copy( this->cfg.logPath, MAX_PATH ) ;
                           }
                           break ;

                        case cciLS:       // auto-save of log-file on exit
                           //* If parameter value not specified *
                           //* enter the default value.         *
                           if ( (aaIndex == CfgParm[cciLS].dflt) ||
                                (aaIndex == CfgParm[cciLS].argcnt) )
                              this->cfg.logAuto = false ;
                           else
                              this->cfg.logAuto = true ;
                           break ;

                        case cciLA:       // user-interface language
                           switch ( aaIndex )
                           {
                              case 0:
                                 this->cfg.lang = enLang ;  break ; // English
                              case 1:
                                 this->cfg.lang = esLang ;  break ; // Espanol
                              case 2:
                                 this->cfg.lang = zhLang ;  break ; // Zhongwen
                              case 3:
                                 this->cfg.lang = viLang ;  break ; // TiengViet
                              default:
                                 this->cfg.lang = locLang ; break ; // Default
                           } ;
                           //* Test for RTL language *
                           if ( AppDirection[this->cfg.lang] != false )
                              this->cfg.rtl = true ;
                           break ;

                        case cciLO:       // user-interface locale
                           //* If we arrive here, the config option either     *
                           //* specifies a non-default locale (verified to     *
                           //* support UTF-8 encoding) OR no argument specified*
                           if ( (gsa.gschars() > 1) && 
                                ((gsa.compare( "locale" )) != ZERO) )
                           {
                              if ( (nc.VerifyLocale ( gsa.ustr() )) == OK )
                                 gsa.copy( this->cfg.altLocale, MAX_FNAME ) ;
                              else     // trigger the error message
                                 ccIndex = cciArgs ;
                           }
                           break ;

                        case cciCS:       // select color scheme
                           //* If color scheme not specified on command line, *
                           //* but non-default color scheme is specified here,*
                           //* update the color scheme.                       *
                           if ( this->cfg.scheme == dfltCS )
                           {
                              switch ( aaIndex )
                              {
                                 case 0: this->cfg.scheme = ncbcBK ; break ; // Black
                                 case 1: this->cfg.scheme = ncbcRE ; break ; // Red
                                 case 2: this->cfg.scheme = ncbcGR ; break ; // Green
                                 case 3: this->cfg.scheme = ncbcBR ; break ; // Brown
                                 case 4: this->cfg.scheme = ncbcBL ; break ; // Blue
                                 case 5: this->cfg.scheme = ncbcMA ; break ; // Magenta
                                 case 6: this->cfg.scheme = ncbcCY ; break ; // Cyan
                                 case 7: this->cfg.scheme = ncbcGY ; break ; // Gray
                                 default:this->cfg.scheme = dfltCS ; break ; // Default
                              } ;
                              //* Test for "reverse" indicator *
                              if ( (gsa.find( ",r" )) > ZERO )
                                 this->cfg.rscheme = true ;
                           }
                           break ;

                        case cciEM:       // enable mouse support
                           //* If parameter value not specified *
                           //* enter the default value.         *
                           if ( (aaIndex == CfgParm[cciEM].dflt) ||
                                (aaIndex == CfgParm[cciEM].argcnt) )
                              this->cfg.enableMouse = false ;
                           else
                              this->cfg.enableMouse = true ;
                           break ;
                     } ;      // switch()
                  }     // (argument test)
               }        // (good ccIndex)
               //* If invalid command, increment error count *
               else
                  ++status ;

               //* If specified, produce verbose output *
               if ( verbose || (ccIndex == cciArgs) )
               {
                  gsv.compose( "LINE %3hd: '%S'", &lineNum, gs.gstr() ) ;
                  gsv.copy( cfgMsg[cfgMsgs].msg, MAX_FNAME ) ;
                  if ( ccIndex < cciArgs )   cfgMsg[cfgMsgs++].attr = &this->suBold ;
                  else                       cfgMsg[cfgMsgs++].attr = &this->suError ;
                  if ( cfgMsgs >= CFGMSG_COUNT - 1 )   // too many errors
                     done = true ;
               }
            }           // valid data read
            else        // EOF reached
               done = true ;

            ++lineNum ;
         }              // while()
         ifs.close() ;                 // close the source file

      }
   }

   if ( status == ZERO )
   {
      gsv = "Configuration parameters initialized." ;
      gsv.copy( cfgMsg[cfgMsgs].msg, MAX_FNAME ) ;
      cfgMsg[cfgMsgs++].attr = &this->suGood ;
   }
   else
   {
      if ( status > ZERO )
         gsv.compose( "%hd syntax error%sreading configuration file."
                        " Continuing with default%s",
                        &status, (status > 1 ? "s " : " "),
                        (status > 1 ? "s." : ".") ) ;
      else
         gsv = "Configuration file not found, or invalid format."
               " Continuing with defaults." ;
      gsv.copy( cfgMsg[cfgMsgs].msg, MAX_FNAME ) ;
      cfgMsg[cfgMsgs++].attr = &this->suError ;
   }

   return status ;

}  //* End Configure() *

//*************************
//* CommandLineOverrides  *
//*************************
//******************************************************************************
//* Override configuration parameters with specified command-line arguments.   *
//*                                                                            *
//* For command-line arguments which DO NOT contain the default value,         *
//* override the value set during configuration.                               *
//*                                                                            *
//* Values for member variables which are calculated dynamically will be       *
//* initialized by caller.                                                     *
//*                                                                            *
//* The following application-setup overrides were handled at a higher level:  *
//*   ca.altLocale                                                             *
//*   ca.appLanguage                                                           *
//*   ca.scheme                                                                *
//*   ca.mouse                                                                 *
//*                                                                            *
//* Input  : ca  : (by reference) copy of command-line arguments.              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::CommandLineOverrides ( const commArgs& ca )
{
   this->ud.inpUnits = ca.inpUnits ;
   this->ud.inpValue = ca.inpValue ;

   if ( ca.heightM > ZERO )      this->ud.heightM  = ca.heightM ;
   if ( ca.massKg > ZERO )       this->ud.massKg   = ca.massKg ;
   if ( ca.ageY > 0.0 )          this->ud.ageY     = ca.ageY ;
   if ( ca.gender != goNone )    this->ud.gender   = ca.gender ;
   if ( ca.kphBike > ZERO )      this->ud.kphBike  = ca.kphBike ;
   if ( ca.kphWalk > ZERO )      this->ud.kphWalk  = ca.kphWalk ;
   if ( ca.kphRun > ZERO )       this->ud.kphRun   = ca.kphRun ;
   if ( ca.exerType != xtNone )  this->ud.exerType = ca.exerType ;
   if ( ca.logPath[0] != NULLCHAR )
   {
      gString gssrc = ca.logPath,
              gstrg ;
      if ( (this->LogfileCompose ( gssrc, gstrg )) )
         gstrg.copy( this->cfg.logPath, MAX_PATH ) ;
   }
   if ( ca.logAuto )             this->cfg.logAuto = ca.logAuto ;

}  //* End CommandLineOverrides() *

//*************************
//*    SetColorScheme     *
//*************************
//******************************************************************************
//* Set the color-attribute members to the specified color scheme.             *
//*                                                                            *
//* Input  : scheme : color-scheme code                                        *
//*          reverse: if 'true',  configure for a dark terminal background     *
//*                   if 'false', configure for a light terminal background    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* We would like to be able to query the terminal settings for the            *
//* foreground/background colors and automagically adjust the color scheme for *
//* the existing background; however, that information is not reliably         *
//* available nor easy to obtain. Overall, it is best to put the burden on     *
//* the poor user to tell us if it is a dark background.                       *
//*                                                                            *
//* See:   'gconftool-2' or 'gsettings' for accessing the desired information. *
//******************************************************************************

void Exercalc::SetColorScheme ( NcBaseColors scheme, bool reverse )
{
   attr_t spdAttr = nc.bw ;   // attribute for 'spDiscard' item of dropdown list

   //* Set the scheme *
   this->cfg.scheme = scheme ;
   this->cfg.rscheme = reverse ;

   //* Set color attributes for start-up messages *
   this->suNorm  = nc.bw ;
   this->suBold  = nc.bl ;
   this->suGood  = nc.gr ;
   this->suError = nc.reB ;

   //* If user has specified, reverse the foreground/background *
   //* for the basic eight(8) color pairs.                      *
   if ( this->cfg.rscheme )
   {
      nc.SetDefaultBackground ( ncbcGY ) ;
   }

   //* Set color attributes for individual color schemes *
   switch ( this->cfg.scheme )
   {
      case ncbcBK:      // black
         this->cfg.ub = nc.bk ;        // user-interface dialog border
         this->cfg.bb = nc.bw ;        // other borders and backgrounds
         this->cfg.tt = this->cfg.bb ; // title dialog color
         this->cfg.sd = nc.bw ;        // sub-dialog color
         this->cfg.sb = nc.blB ;       // sub-dialog Bold color
         this->cfg.em = nc.brbl ;      // sub-dialog Emphasis color
         this->cfg.er = nc.grbk ;      // Emphasis color for 'reverse' bkgnd
         this->cfg.mn = nc.recy ;      // menu border (without focus) color
         this->cfg.mf = nc.bkcy ;      // menu border (with focus) color
         this->cfg.dn = nc.bkcy ;      // dropdown (without focus) color
         this->cfg.df = nc.recy ;      // dropdown (with focus) color
         this->cfg.pn = nc.cygy ;      // pushbutton (without focus) color
         this->cfg.pf = nc.reG ;       // pushbutton (with focus) color
         this->cfg.tn = this->cfg.bb ; // text box (without focus) color
         this->cfg.tf = nc.brgr ;      // text box (with focus) color
         spdAttr      = nc.brcy ;      // 'spDiscard' item
         break ;
      case ncbcRE:      // red
         this->cfg.ub = nc.reR ;       // user-interface dialog border
         this->cfg.bb = nc.re ;        // other borders and backgrounds
         this->cfg.tt = this->cfg.bb ; // title dialog color
         this->cfg.sd = nc.br ;        // sub-dialog color
         this->cfg.sb = nc.blB ;       // sub-dialog Bold color
         this->cfg.em = nc.brbl ;      // sub-dialog Emphasis color
         this->cfg.er = nc.grre ;      // Emphasis color for 'reverse' bkgnd
         this->cfg.mn = nc.grbl ;      // menu border (without focus) color
         this->cfg.mf = nc.blR ;       // menu border (with focus) color
         this->cfg.dn = nc.blR ;       // dropdown (without focus) color
         this->cfg.df = nc.grbl ;      // dropdown (with focus) color
         this->cfg.pn = nc.gyR ;       // pushbutton (without focus) color
         this->cfg.pf = nc.grG ;       // pushbutton (with focus) color
         this->cfg.tn = this->cfg.bb ; // text box (without focus) color (invisible)
         this->cfg.tf = nc.bkcy ;      // text box (with focus) color
         spdAttr      = this->cfg.em ; // 'spDiscard' item
         break ;
      case ncbcGR:      // green
         this->cfg.ub = nc.grR ;       // user-interface dialog border
         this->cfg.bb = nc.gr ;        // other borders and backgrounds
         this->cfg.tt = this->cfg.bb ; // title dialog color
         this->cfg.sd = nc.br ;        // sub-dialog color
         this->cfg.sb = nc.blB ;       // sub-dialog Bold color
         this->cfg.em = nc.brbl ;      // sub-dialog Emphasis color
         this->cfg.er = nc.brgr ;      // Emphasis color for 'reverse' bkgnd
         this->cfg.mn = nc.rebl ;      // menu border (without focus) color
         this->cfg.mf = nc.blR ;       // menu border (with focus) color
         this->cfg.dn = nc.blR ;       // dropdown (without focus) color
         this->cfg.df = nc.rebl ;      // dropdown (with focus) color
         this->cfg.pn = nc.gyR ;       // pushbutton (without focus) color
         this->cfg.pf = nc.reG ;       // pushbutton (with focus) color
         this->cfg.tn = this->cfg.bb ; // text box (without focus) color (invisible)
         this->cfg.tf = nc.brR ;       // text box (with focus) color
         spdAttr      = this->cfg.em ; // 'spDiscard' item
         break ;
      case ncbcBL:      // blue
         this->cfg.ub = nc.blR ;       // user-interface dialog border
         this->cfg.bb = nc.bl ;        // other borders and backgrounds
         this->cfg.tt = this->cfg.bb ; // title dialog color
         this->cfg.sd = nc.bl ;        // sub-dialog color
         this->cfg.sb = nc.br ;        // sub-dialog Bold color
         this->cfg.em = nc.brR ;       // sub-dialog Emphasis color
         this->cfg.er = nc.grbl ;      // Emphasis color for 'reverse' bkgnd
         this->cfg.mn = nc.recy ;      // menu border (without focus) color
         this->cfg.mf = nc.bkcy ;      // menu border (with focus) color
         this->cfg.dn = nc.bkcy ;      // dropdown (without focus) color
         this->cfg.df = nc.recy ;      // dropdown (with focus) color
         this->cfg.pn = nc.gyR ;       // pushbutton (without focus) color
         this->cfg.pf = nc.reG ;       // pushbutton (with focus) color
         this->cfg.tn = this->cfg.bb ; // text box (without focus) color (invisible)
         this->cfg.tf = nc.maR ;       // text box (with focus) color
         spdAttr      = nc.brcy ;      // 'spDiscard' item
         break ;
      case ncbcMA:      // magenta
         this->cfg.ub = nc.maR ;       // user-interface dialog border
         this->cfg.bb = nc.ma ;        // other borders and backgrounds
         this->cfg.tt = this->cfg.bb ; // title dialog color
         this->cfg.sd = nc.br ;        // sub-dialog color
         this->cfg.sb = nc.bl ;        // sub-dialog Bold color
         this->cfg.em = nc.brR ;       // sub-dialog Emphasis color
         this->cfg.er = nc.brma ;      // Emphasis color for 'reverse' bkgnd
         this->cfg.mn = nc.brbk ;      // menu border (without focus) color
         this->cfg.mf = nc.blR ;       // menu border (with focus) color
         this->cfg.dn = nc.bk ;        // dropdown (without focus) color
         this->cfg.df = nc.brbk ;      // dropdown (with focus) color
         this->cfg.pn = nc.gyR ;       // pushbutton (without focus) color
         this->cfg.pf = nc.brbk ;      // pushbutton (with focus) color
         this->cfg.tn = this->cfg.bb ; // text box (without focus) color (invisible)
         this->cfg.tf = nc.brgr ;      // text box (with focus) color
         spdAttr      = this->cfg.df ; // 'spDiscard' item
         break ;
      case ncbcCY:      // cyan
         this->cfg.ub = nc.cyR ;       // user-interface dialog border
         this->cfg.bb = nc.cy ;        // other borders and backgrounds
         this->cfg.tt = this->cfg.bb ; // title dialog color
         this->cfg.sd = nc.br ;        // sub-dialog color
         this->cfg.sb = nc.bl ;        // sub-dialog Bold color
         this->cfg.em = nc.brbl ;      // sub-dialog Emphasis color
         this->cfg.er = nc.bkcy ;      // Emphasis color for 'reverse' bkgnd
         this->cfg.mn = nc.rebl ;      // menu border (without focus) color
         this->cfg.mf = nc.blR ;       // menu border (with focus) color
         this->cfg.dn = nc.blR ;       // dropdown (without focus) color
         this->cfg.df = nc.rebl ;      // dropdown (with focus) color
         this->cfg.pn = nc.gyR ;       // pushbutton (without focus) color
         this->cfg.pf = nc.reG ;       // pushbutton (with focus) color
         this->cfg.tn = this->cfg.bb ; // text box (without focus) color (invisible)
         this->cfg.tf = nc.brR ;       // text box (with focus) color
         spdAttr      = this->cfg.em ; // 'spDiscard' item
         break ;
      case ncbcGY:      // gray
         this->cfg.ub = nc.gyR ;       // user-interface dialog border
         this->cfg.bb = nc.gy ;        // other borders and backgrounds
         this->cfg.tt = nc.bw ;        // title dialog color
         this->cfg.sd = nc.br ;        // sub-dialog color
         this->cfg.sb = nc.bl ;        // sub-dialog Bold color
         this->cfg.em = nc.brbl ;      // sub-dialog Emphasis color
         this->cfg.er = nc.bkgy ;      // Emphasis color for 'reverse' bkgnd
         this->cfg.mn = nc.rebl ;      // menu border (without focus) color
         this->cfg.mf = nc.blR ;       // menu border (with focus) color
         this->cfg.dn = nc.blR ;       // dropdown (without focus) color
         this->cfg.df = nc.rebl ;      // dropdown (with focus) color
         this->cfg.pn = nc.cyR ;       // pushbutton (without focus) color
         this->cfg.pf = nc.reG ;       // pushbutton (with focus) color
         this->cfg.tn = this->cfg.bb ; // text box (without focus) color (invisible)
         this->cfg.tf = nc.brgr ;      // text box (with focus) color
         spdAttr      = this->cfg.em ; // 'spDiscard' item
         break ;
      case ncbcBR:      // brown (default)
      default:
         this->cfg.ub = nc.brR ;       // user-interface dialog border
         this->cfg.bb = nc.br ;        // other borders and backgrounds
         this->cfg.tt = this->cfg.bb ; // title dialog color
         this->cfg.sd = nc.br ;        // sub-dialog color
         this->cfg.sb = nc.blB ;       // sub-dialog Bold color
         this->cfg.em = nc.brbl ;      // sub-dialog Emphasis color
         this->cfg.er = nc.grbr ;      // Emphasis color for 'reverse' bkgnd
         this->cfg.mn = nc.rebl ;      // menu border (without focus) color
         this->cfg.mf = nc.blR ;       // menu border (with focus) color
         this->cfg.dn = nc.blR ;       // dropdown (without focus) color
         this->cfg.df = nc.rebl ;      // dropdown (with focus) color
         this->cfg.pn = nc.gyR ;       // pushbutton (without focus) color
         this->cfg.pf = nc.reG ;       // pushbutton (with focus) color
         this->cfg.tn = this->cfg.bb ; // text box (without focus) color (invisible)
         this->cfg.tf = nc.brgr ;      // text box (with focus) color
         spdAttr      = this->cfg.em ; // 'spDiscard' item
         break ;
   } ;

   //* Finish color assignments by initializing color arrays *
   this->cfg.mono[0] = attrDFLT ;              // individual items of sub-menus
   this->cfg.mono[1] = this->cfg.mf ;
   for ( short i = ZERO ; i < spITEMS ; ++i )  // individual items of Drowdown control
      this->cfg.dattr[i] = this->cfg.mf ;
   this->cfg.dattr[spDiscard] = spdAttr ;
   for ( short i = ZERO ; i < optITEMS ; ++i ) // individual items of Menuwin control
      this->cfg.mattr[i] = this->cfg.mf ;

}  //* End SetColorScheme() *

//*************************
//*       DiagMsg         *
//*************************
//******************************************************************************
//* Display start-up diagnostic message in console window.                     *
//* DO NOT call this method after dialog window has opened.                    *
//*                                                                            *
//* Input  : msg    : message to be displayed                                  *
//*          color  : color attribute for message                              *
//*          newLine: (optional, 'true' by default)                            *
//*                   if 'true'  move cursor to next message position          *
//*                   if 'false' leave cursor at end of message                *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::DiagMsg ( const char* msg, attr_t color, bool newLine )
{
   static short xCol = ZERO, xColBaseY = 11 ;

   this->suPos = nc.WriteString ( this->suPos.ypos, this->suPos.xpos, msg, color ) ;
   if ( newLine )
   {
      this->suPos = { short(this->suPos.ypos + 1), xCol  } ;
      if ( xCol == ZERO && (this->suPos.ypos >= this->cfg.termRows) )
      {
         xCol = this->cfg.termCols / 2 + 1 ;
         this->suPos = { short(xColBaseY + 1), xCol } ;
      }
   }

}  //* End DiagMsg() *

//*************************
//*      ClearWindow      *
//*************************
//******************************************************************************
//* Clear the display and set cursor position to origin (0,0).                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::ClearWindow ( void )
{

   nc.ClearScreen () ;

}  //* End ClearWindow() *

//*************************
//*      DrawPicture      *
//*************************
//******************************************************************************
//* Draw a pseudo-picture into the Picture dialog.                             *
//* The image drawn is indicated by the 'cfg.image' member.                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::DrawPicture ( void )
{
   //* Block character definitions for both horizontal and vertical *
   //* progress-bar action.                                         *
   const short cellDIV = 8 ;     // character cell divided into eighths
   //const wchar_t hblock[cellDIV] =
   //{
   //   0x0258F,    // ▏
   //   0x0258E,    // ▎
   //   0x0258D,    // ▍
   //   0x0258C,    // ▌
   //   0x0258B,    // ▋
   //   0x0258A,    // ▊
   //   0x02589,    // ▉
   //   0x02588,    // █
   //} ;
   const wchar_t vblock[cellDIV] =
   {
      0x02581,    // ▁
      0x02582,    // ▂
      0x02583,    // ▃
      0x02584,    // ▄
      0x02585,    // ▅
      0x02586,    // ▆
      0x02587,    // ▇
      0x02588,    // █
   } ;

   gString gs ;                     // text formatting
   winPos wp = this->pictArea ;     // cursor position

#if 1    // EXPERIMENTAL
   wp = this->userDlg->WriteParagraph ( wp,
         L"                                                      ▇        \n"
          "                                                     ▆▆▆       \n"
          "                                                    ▄▄▄▄▄      \n"
          "                                                   ▂▂▂▂▂▂▂     \n"
          "                                                     ▂▂▂       \n"
          "                                                     ▂▂▂       \n"
          "                                                     ▂▂▂       \n"
          "                                                     ▂▂▂       \n"
          "                                                   ▂▂▂▂▂▂▂     \n"
          "                                                    ▄▄▄▄▄      \n"
          "                                                     ▆▆▆       \n"
          "                                                      ▇        \n"
          , this->cfg.ub ) ;
#endif   // EXPERIMENTAL

   //* Trim the bottom row *
   // Programmer's Note: Cursor position will not auto-increment beyond dialog.
   wp.ypos = this->pictArea.ypos + this->pictRows - 1 ;
   wp.xpos = ZERO ;
   for ( wp.xpos = ZERO ; wp.xpos < this->pictCols ; ++wp.xpos )
      this->userDlg->WriteChar ( wp.ypos, wp.xpos, vblock[3], this->cfg.ub ) ;

}  //* End DrawPicture() *

//*************************
//*     TermsizeError     *
//*************************
//******************************************************************************
//* Called if application cannot continue because terminal window is too small.*
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::TermsizeError ( void )
{
   gString gsOut( termsizeError[this->cfg.lang][0], 
                  &this->cfg.termRows, &this->cfg.termCols ) ;
   this->DiagMsg ( gsOut.ustr(), this->suError ) ;
   gsOut.compose( termsizeError[this->cfg.lang][1],
                  &MIN_ROWS, &MIN_COLS ) ;
   this->DiagMsg ( gsOut.ustr(), this->suError ) ;

}  //* End TermsizeError() *

//*************************
//*     TempdirError      *
//*************************
//******************************************************************************
//* Called if application cannot continue because temp-file directory could    *
//* not be found.                                                              *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void Exercalc::TempdirError ( void )
{

   gString gsOut( tmpdirError[this->cfg.lang] ) ;
   this->DiagMsg ( gsOut.ustr(), this->suError ) ;

}  //* End TempdirError() *

//*************************
//*   DebugInfo2StatDlg   *
//*************************
//******************************************************************************
//* FOR DEBUGGING ONLY: Write startup information to the Stats dialog window. *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

#if DEBUG_SETUP_PARMS != 0 || DEBUG_MENU != 0
void Exercalc::DebugInfo2StatDlg ( const commArgs& ca )
{
   //* Redraw the window and clear the display area *
   this->statsDlg->DrawBox ( ZERO, ZERO, sdInit.dLines, sdInit.dColumns, this->cfg.bb ) ;
   this->statsDlg->SetDialogTitle ( 
               "  Debugging Only - Display Configuration Parameters  ", 
               this->cfg.bb | ncuATTR ) ;
   this->statsDlg->ClearWin () ;

   gString gsOut( 
       "Command-line Arguments\n"
       "----------------------\n"
       "inpValue   : %-7.03f\n"
       "inpUnits   : %s\n"
       "exerType   : %s\n"
       "heightM    : %-7.3f\n"
       "massKg     : %-7.3f\n"
       "ageY       : %-5.1f\n"
       "gender     : %s\n"
       "kphBike    : %-7.3f\n"
       "kphWalk    : %-7.3f\n"
       "kphRun     : %-7.3f\n"
       "logPath    : %s\n"
       "logAuto    : %s\n"
       "altLocale  : %s\n"
       "appLanguage: %s\n"
       "colorScheme: %s%s\n"
       "mouse      : %s\n",
       &ca.inpValue,
       (ca.inpUnits == vuMiles ? "vuMiles" :
        ca.inpUnits == vuKmeters ? "vuKmeters" :
        ca.inpUnits == vuMinutes ? "vuMinutes" :
        ca.inpUnits == vuKcalories ? "vuKcalories" : "vuNone"),
       (ca.exerType == xtWalk ? "xtWalk" :
        ca.exerType == xtRun ? "xtRun" :
        ca.exerType == xtBike ? "xtBike" :
        ca.exerType == xtGeneral ? "xtGeneral" : "xtNone"),
       &ca.heightM, &ca.massKg, &ca.ageY,
       (ca.gender == goMale ? "goMale" : 
        ca.gender == goFemale ? "goFemale" :
        ca.gender == goOther ? "goOther" : "goNone"),
       &ca.kphBike, &ca.kphWalk, &ca.kphRun,
       ca.logPath,
       (ca.logAuto ? "enable" : "disable"),
       ca.altLocale,
       (ca.appLanguage == enLang ? "English" :
        ca.appLanguage == esLang ? "Espanol" :
        ca.appLanguage == zhLang ? "Zhongwen" :
        ca.appLanguage == viLang ? "Tieng Viet" :
        ca.appLanguage == locLang ? "locale" : "unknown"),
        // Programmer's Note: Compiler error (g++ v:10.2.1):
        // Cannot read non-ASCII strings in trinary arguments.
       (ca.scheme == ncbcBK ? "black" :
        ca.scheme == ncbcRE ? "red" :
        ca.scheme == ncbcGR ? "green" :
        ca.scheme == ncbcBR ? "brown" :
        ca.scheme == ncbcBL ? "blue" :
        ca.scheme == ncbcMA ? "magenta" :
        ca.scheme == ncbcCY ? "cyan" : 
        ca.scheme == ncbcGY ? "gray" : "default"),
       (ca.rscheme ? ",r" : " "),
       (ca.mouse ? "enable" : "disable") ) ;
   this->statsDlg->WriteParagraph ( 2, 2, gsOut, this->cfg.bb, true ) ;

   winPos wp( 2, 32 ) ;
   gsOut.compose( 
       "Configuration Parameters\n"
       "------------------------\n"
       "Height      : %6.3fm\n"
       "Mass        : %6.3fkg\n"
       "BodyMassIndx: %5.2f\n"
       "Age         : %-5.1f\n"
       "Gender      : %s\n"
       "Avg Kph Bike: %5.2f\n"
       "Avg Kph Walk: %5.2f\n"
       "Avg Kph Run : %5.2f\n"
       "ExerciseType: %s\n"
       "metsWalk    : %6.3f\n"
       "metsBike    : %6.3f\n"
       "metsRun     : %6.3f\n"
       "metsGeneral : %6.3f\n",
       &this->ud.heightM, &this->ud.massKg, &this->ud.bmi, &this->ud.ageY,
       (this->ud.gender == goMale ? "goMale" : 
        this->ud.gender == goFemale ? "goFemale" :
        this->ud.gender == goOther ? "goOther" : "goNone"),
       &this->ud.kphBike, &this->ud.kphWalk, &this->ud.kphRun,
       (this->ud.exerType == xtWalk ? "xtWalk" :
        this->ud.exerType == xtBike ? "xtBike" :
        this->ud.exerType == xtRun ? "xtRun" :
        this->ud.exerType == xtGeneral ? "xtGeneral" : 
        this->ud.exerType == xtNone ? "xtNone" : "unknown"),
       &this->ud.metsWalk, &this->ud.metsBike, &this->ud.metsRun, &this->ud.metsGen ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, this->cfg.bb, true ) ;
   gsOut.compose( 
       "gValue      : %6.3f\n"
       "gUnits      : %s\n"
       "gXType      : %s\n"
       "gMets       : %6.3f\n"
       "gWeek       : %s\n"
       "altLocale   : %s\n"
       "appLanguage*: %s\n"
       "colorScheme : %s%s\n"
       "enableMouse : %s\n"
       "tmpPath     : %s\n"
       "logAuto     : %s\n"
       "logPath     :\n",
       &this->gValue,
       (this->gUnits == vuMinutes ? "vuMinutes" :
        this->gUnits == vuKmeters ? "vuKmeters" :
        this->gUnits == vuMiles   ? "vuMiles" :
        this->gUnits == vuKcalories ? "vuKcalories" : "unknown"),
       (this->gXType == xtWalk    ? "xtWalk"    :
        this->gXType == xtBike    ? "xtBike"    :
        this->gXType == xtRun     ? "xtRun"     :
        this->gXType == xtGeneral ? "xtGeneral" :
        this->gXType == xtNone    ? "xtNone" : "unknown"),
       &this->gMets,
       (this->gWeek ? "true" : "false"),
       this->cfg.altLocale,
       (this->cfg.lang == enLang ? "English" :
        this->cfg.lang == esLang ? "Espanol" :
        this->cfg.lang == zhLang ? "Zhongwen" :
        this->cfg.lang == viLang ? "Tieng Viet" :
        this->cfg.lang == locLang ? "locale" : "unknown"),
        // Programmer's Note: Compiler error (g++ v:10.2.1):
        // Cannot read non-ASCII strings in trinary arguments.
       (this->cfg.scheme == ncbcBK ? "black" :
        this->cfg.scheme == ncbcRE ? "red" :
        this->cfg.scheme == ncbcGR ? "green" :
        this->cfg.scheme == ncbcBR ? "brown" :
        this->cfg.scheme == ncbcBL ? "blue" :
        this->cfg.scheme == ncbcMA ? "magenta" :
        this->cfg.scheme == ncbcCY ? "cyan" :
        this->cfg.scheme == ncbcGY ? "gray" : "default"),
       (this->cfg.rscheme ? ",r" : " "),
       (this->cfg.enableMouse ? "enable" : "disable"),
       this->cfg.tmpPath,
       (this->cfg.logAuto ? "enable" : "disable") ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, this->cfg.bb, true ) ;
   short maxwidth = sdInit.dColumns - wp.xpos - 1, indx = ZERO ;
   gsOut.compose( "%s\n", this->cfg.logPath ) ;
   while ( ((gsOut.gscols()) > maxwidth) && (indx >= ZERO) )
   {
      indx = gsOut.after( L'/' ) ;
      if ( indx > ZERO )
         gsOut.shiftCols( -(indx) ) ;
   }
   wp = this->statsDlg->WriteParagraph ( wp, gsOut, this->cfg.bb, true ) ;
   //this->statsDlg->WriteParagraph ( wp.ypos, 2, this->cfg.logPath, this->cfg.bb, true ) ;

}  //* End DebugInfo2StatDlg() *
#endif   // DEBUG_SETUP_PARMS || DEBUG_MENU

//*************************
//*  ColorScheme2StatDlg  *
//*************************
//******************************************************************************
//* FOR DEBUGGING ONLY: Display examples of all color schemes.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

#if DEBUG_MENU != 0
void Exercalc::ColorScheme2StatDlg ( void )
{
   const short boxCount = 9 ;
   const short boxrows = 7, boxcols = 32 ;
   NcBaseColors ncBC[boxCount] = 
   { ncbcBK, ncbcRE, ncbcGR, ncbcBR, ncbcBL, ncbcMA, ncbcCY, ncbcGY, ncbcBW } ;

   const wchar_t *boxTemplate = L" %S %S " ;
   const wchar_t *boxTitle[LangCount] = 
   { L"Color Scheme",  L"Esquema Colores",  L"配色方案",  L"Bảng màu" } ;
   const wchar_t *colorName[LangCount][boxCount] = 
   {
      {  // English
         L"Black",  L"Red",      L"Green",  L"Brown",
         L"Blue",   L"Magenta",  L"Cyan",   L"Gray", L"White",
      },
      {  // Español
         L"Negro",  L"Rojo",     L"Verde",  L"Marrón",
         L"Azul",   L"Magenta",  L"Cian",   L"Gris", L"Blanco",
      },
      {  // Zhōngwén (中文)
         L"黑色",    L"红",       L"绿色",    L"棕色",
         L"蓝色",    L"品红",     L"青色",    L"灰色", L"White",
      },
      {  // TiếngViệt
         L"Đen",           L"Đỏ",    L"Xanh lá",   L"Nâu",
         L"Xanh da trời",  L"Đỏ sậm",  L"Lục lam",   L"Xám", L"White",
      },
   } ;

   winPos boxwp( 1, 0 ),                           // box position
          wp ;                                     // cursor position
   gString gs ;                                    // text formatting
   NcBaseColors origScheme = this->cfg.scheme ;    // save original color scheme
   short dlgRows, dlgCols, dlgCenter ;             // dialog dimensions
   this->statsDlg->GetDialogDimensions ( dlgRows, dlgCols ) ;
   dlgCenter = dlgCols / 2 + 1 ;
   this->statsDlg->ClearWin () ;                   // clear the dialog window
   gs.compose( "   %S   ", boxTitle[this->cfg.lang] ) ; // set dialog title
   this->statsDlg->SetDialogTitle ( gs, this->cfg.bb | ncuATTR ) ;

   //* Display color schemes for white (light) backgrounds. *
   short indx = ZERO ;
   for ( ; indx < (boxCount - 1) ; ++indx )
   {
      //* Initialize the color variables for this scheme *
      this->SetColorScheme ( ncBC[indx], this->cfg.rscheme ) ;
      //* Compose the box title *
      gs.compose( boxTemplate, boxTitle[this->cfg.lang], 
                  colorName[this->cfg.lang][indx] ) ;
      //* Draw the box *
      this->statsDlg->DrawBox ( boxwp.ypos, boxwp.xpos, boxrows, boxcols, 
                                this->cfg.ub, gs.ustr() ) ;
      //* Draw the objects of the interior of the box *
      wp = { short(boxwp.ypos + 1), short(boxwp.xpos + 1) } ;
      wp = this->statsDlg->WriteParagraph ( wp, "Color : ", this->cfg.bb ) ;
      wp = this->statsDlg->WriteParagraph ( wp, "176.50 ", this->cfg.sb ) ;
      wp = this->statsDlg->WriteParagraph ( wp, " cm ", this->cfg.em ) ;
      ++wp.xpos ;
      wp = this->statsDlg->WriteParagraph ( wp, "x", this->cfg.er ) ;
      ++wp.xpos ;
      wp = this->statsDlg->WriteParagraph ( wp, "Textbox \n", this->cfg.tf ) ;
      wp.xpos = boxwp.xpos + 2 ;
      wp = this->statsDlg->WriteParagraph ( wp, " Push ", this->cfg.pn ) ;
      wp = this->statsDlg->WriteParagraph ( wp, " Push \n", this->cfg.pf ) ;
      wp.xpos = boxwp.xpos + 2 ;
      wp = this->statsDlg->WriteParagraph ( wp, 
                       "┌──────────┐\n│-Dropdown-◆\n└──────────┘", this->cfg.dn ) ;
      wp.ypos -= 3 ;
      ++wp.xpos ;
      wp = this->statsDlg->WriteParagraph ( wp, "  Menu  \n", this->cfg.mn ) ;
      wp = this->statsDlg->WriteParagraph ( wp, "┌──────────────┐\n"
                                                "│xxxxxxxxxxxxxx│\n"
                                                "│xxxxxxxxxxxxxx│\n"
                                                "└──────────────┘\n", this->cfg.mf ) ;
      wp.ypos -= 3 ;
      ++wp.xpos ;
      wp = this->statsDlg->WriteParagraph ( wp, "xxxxxxxxxxxxxx\n"
                                                "xxxxxxxxxxxxxx", this->cfg.mattr[1] ) ;

      //* Advance the cursor position to the next box *
      if ( (indx % 2) == ZERO )
         boxwp.xpos = dlgCenter ;
      else
         boxwp = { short(boxwp.ypos += boxrows + 1), ZERO } ;
   }

   //* Restore original color scheme *
   this->SetColorScheme ( origScheme, this->cfg.rscheme ) ;

   this->statsDlg->RefreshWin () ;

}  //* End ColorScheme2StatDlg() *
#endif   // DEBUG_MENU

//*************************
//*   ColorMap2StatDlg    *
//*************************
//******************************************************************************
//* FOR DEBUGGING ONLY: Display the first thirty-two(32) color-attribute pairs *
//*                     and RGB register values.                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

#if DEBUG_MENU != 0
void Exercalc::ColorMap2StatDlg ( void )
{
   const wchar_t *dlgTitle[LangCount] = 
   { L"Color Register Map",  L"Mapa de Registro de Color",  
     L"彩色套准图",  L"Bản đồ đăng ký màu" } ;

   NcColorMap ncmap ;                              // get color-register map
   nc.GetColorMap ( ncmap ) ;
   gString gs ;                                    // text formatting
   winPos wp( 2, 1 ) ;                             // cursor position
   this->statsDlg->ClearWin () ;                   // clear the dialog window
   gs.compose( "   %S   ", dlgTitle[this->cfg.lang] ) ; // set dialog title
   this->statsDlg->SetDialogTitle ( gs, this->cfg.bb | ncuATTR ) ;

   //* Display the first 16 RGB register sets (8 for older hardware) *
   wp = this->statsDlg->WriteParagraph ( wp, "REG  RED  GREEN  BLUE\n", 
                                         this->cfg.bb | ncuATTR ) ;
   for ( short i = ZERO ; (i < ncmap.rgbCount) && (i < 16) ; ++i )
   {
      gs.compose( "%2hd)  %04hX  %04hX  %04hX\n", &i,
                  &ncmap.rgbMap[i].r, &ncmap.rgbMap[i].g, &ncmap.rgbMap[i].b ) ;
      wp = this->statsDlg->WriteParagraph ( wp, gs, this->cfg.bb ) ;
   }

   //* Display misc color setup info *
   ++wp.ypos ;
   gs.compose( "bkgndColor: %hd\n"
               "pairCount : %hd\n"
               "rgbCount  : %hd\n"
               "supColors : %hd\n"
               "modPairs  : %s\n"
               "modRGB    : %s\n\n",
               &ncmap.bkgndColor, &ncmap.pairCount, &ncmap.rgbCount,
               &ncmap.supColors,
               (ncmap.canModifyPairs ? "yes" : "no"),
               (ncmap.canModifyRGB ? "yes" : "no"),
               &nc.bw, &nc.bl, 
               &nc.bk, &nc.ma, 
               &nc.re, &nc.cy, 
               &nc.gr, &nc.gy,
               &nc.br ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gs, this->cfg.bb ) ;
   #if 1    // DISABLE
   gs.compose( "bw:%08X bl:%08X\n"
               "bk:%08X ma:%08X\n"
               "re:%08X cy:%08X\n"
               "gr:%08X gy:%08X\n"
               "br:%08X\n",
               &nc.bw, &nc.bl, &nc.bk, &nc.ma, &nc.re, &nc.cy, 
               &nc.gr, &nc.gy, &nc.br ) ;
   wp = this->statsDlg->WriteParagraph ( wp, gs, this->cfg.bb ) ;
   #endif   // DISABLE

   //* Display color pairs *
   wp = { 2, 24 } ;
   wp = this->statsDlg->WriteParagraph ( wp, "REG FG BG ..  "
                                             "REG FG BG ..  "
                                             "REG FG BG ..\n",
                                         this->cfg.bb | ncuATTR ) ;
   for ( short i = ZERO ; (i < ncmap.pairCount) && (i <= 86) ; ++i )
   {
      gs.compose( "%2hd) %2hd %2hd\n", &i,
                  &ncmap.pairMap[i].fgnd, &ncmap.pairMap[i].bkgnd ) ;
      wp = this->statsDlg->WriteParagraph ( wp, gs, this->cfg.bb ) ;
      if ( i == 21 ) wp = { 3, short(wp.xpos + 14) } ;
      if ( i == 43 ) wp = { 3, short(wp.xpos + 14) } ;
      if ( i == 63 ) break ;   // API initializes only the first 64 pairs
   }
   wp = { 3, 34 } ;
   attr_t attr ;
   gs = "xo\n" ;
   for ( short i = ZERO ; i < 64 ; ++i )
   {
      switch ( i )
      {
         case  0:  attr = nc.bw ;    break ;
         case  1:  attr = nc.re ;    break ;
         case  2:  attr = nc.gr ;    break ;
         case  3:  attr = nc.br ;    break ;
         case  4:  attr = nc.bl ;    break ;
         case  5:  attr = nc.ma ;    break ;
         case  6:  attr = nc.cy ;    break ;
         case  7:  attr = nc.gy ;    break ;

         case  8:  attr = nc.bkre ;  break ;
         case  9:  attr = nc.bkgr ;  break ;
         case 10:  attr = nc.bkbr ;  break ;
         case 11:  attr = nc.bkbl ;  break ;
         case 12:  attr = nc.bkma ;  break ;
         case 13:  attr = nc.bkcy ;  break ;
         case 14:  attr = nc.bkgy ;  break ;

         case 15:  attr = nc.rebk ;  break ;
         case 16:  attr = nc.regr ;  break ;
         case 17:  attr = nc.rebr ;  break ;
         case 18:  attr = nc.rebl ;  break ;
         case 19:  attr = nc.rema ;  break ;
         case 20:  attr = nc.recy ;  break ;
         case 21:  attr = nc.regy ;  break ;

         case 22:  attr = nc.grbk ;  break ;
         case 23:  attr = nc.grre ;  break ;
         case 24:  attr = nc.grbr ;  break ;
         case 25:  attr = nc.grbl ;  break ;
         case 26:  attr = nc.grma ;  break ;
         case 27:  attr = nc.grcy ;  break ;
         case 28:  attr = nc.grgy ;  break ;

         case 29:  attr = nc.brbk ;  break ;
         case 30:  attr = nc.brre ;  break ;
         case 31:  attr = nc.brgr ;  break ;
         case 32:  attr = nc.brbl ;  break ;
         case 33:  attr = nc.brma ;  break ;
         case 34:  attr = nc.brcy ;  break ;
         case 35:  attr = nc.brgy ;  break ;

         case 36:  attr = nc.blbk ;  break ;
         case 37:  attr = nc.blre ;  break ;
         case 38:  attr = nc.blgr ;  break ;
         case 39:  attr = nc.blbr ;  break ;
         case 40:  attr = nc.blma ;  break ;
         case 41:  attr = nc.blcy ;  break ;
         case 42:  attr = nc.blgy ;  break ;

         case 43:  attr = nc.mabk ;  break ;
         case 44:  attr = nc.mare ;  break ;
         case 45:  attr = nc.magr ;  break ;
         case 46:  attr = nc.mabr ;  break ;
         case 47:  attr = nc.mabl ;  break ;
         case 48:  attr = nc.macy ;  break ;
         case 49:  attr = nc.magy ;  break ;

         case 50:  attr = nc.cybk ;  break ;
         case 51:  attr = nc.cyre ;  break ;
         case 52:  attr = nc.cygr ;  break ;
         case 53:  attr = nc.cybr ;  break ;
         case 54:  attr = nc.cybl ;  break ;
         case 55:  attr = nc.cyma ;  break ;
         case 56:  attr = nc.cygy ;  break ;

         case 57:  attr = nc.gybk ;  break ;
         case 58:  attr = nc.gyre ;  break ;
         case 59:  attr = nc.gygr ;  break ;
         case 60:  attr = nc.gybr ;  break ;
         case 61:  attr = nc.gybl ;  break ;
         case 62:  attr = nc.gyma ;  break ;
         case 63:  attr = nc.gycy ;  break ;

         default: attr = nc.bwR ;   break ;
      } ;
      wp = this->statsDlg->WriteParagraph ( wp, gs, attr ) ;
      if ( i == 21 ) wp = { 3, short(wp.xpos + 14) } ;
      if ( i == 43 ) wp = { 3, short(wp.xpos + 14) } ;
   }


   this->statsDlg->RefreshWin () ;

}  //* End ColorMap2StatDlg() *
#endif   // DEBUG_MENU

//*************************
//*  LogfileDebugSample   *
//*************************
//******************************************************************************
//* FOR DEBUGGING ONLY: Create a sample logfile for testing various            *
//* functionality for parsing and analysis of logfile data.                    *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

#if DEBUG_MENU != 0
void Exercalc::LogfileDebugSample ( void )
{
   const short RECORDS = 56 ;       // number of logfile records to create
   const float distFLOOR   = 2.0 ;  // data range and starting value
   const float distCEILING = 8.0 ;     // (Adjust these values to change the )
   const float distBASE    = 6.5 ;     // (shape of the graph.               )
   const float distINCR    = 0.3 ;
   const float enerFLOOR   = 300.0 ;
   const float enerCEILING = 1800.0 ;
   const float enerBASE    = 1200.0 ;
   const float enerINCR    = 185.0 ;
   const float timeFLOOR   = 40.0 ;
   const float timeCEILING = 80.0 ;
   const float timeBASE    = 70.0 ;
   const float timeINCR    = 4.0 ;
   chrono::duration<short>aWhile( 3 ) ;

   //* The 'ud' member will be overwritten, so   *
   //* save the currently-displayed data record. *
   UserData udBackup = this->ud ;

   //* Indicates which data member of the record is variable *
   //* Choices are: Distance     (vuKmeters)
   //*              Energy       (vuKcalories)
   //*              Elapsed Time (vuMinutes)
   valUnits dataType = this->gUnits ;
   //* Indicates type of exercise being recorded:
   //* Choices are: Walk         (xtWalk)
   //*              Bike         (xtBike)
   //*              Run          (xtRun)
   //*              General      (xtGeneral)
   XType exType = this->ud.exerType ;

   //* If desired, adjust the "Goal" value, so graph   *
   //* will place the goal line within the value range.*
   //* (not critical, just for fun)                    *
   //this->gValue = 10.0 ;

   float floor, ceiling, baseval, increment ;
   if ( dataType == vuKmeters )
   {
      floor     = distFLOOR ;
      ceiling   = distCEILING ;
      baseval   = distBASE ;
      increment = -(distINCR) ;
   }
   else if ( dataType == vuKcalories )
   {
      floor     = enerFLOOR ;
      ceiling   = enerCEILING ;
      baseval  = enerBASE ;
      increment = -(enerINCR) ;
   }
   else  // (dataType == vuMinutes)
   {
      floor = timeFLOOR ;
      ceiling = timeCEILING ;
      baseval = timeBASE ;
      increment = -(timeINCR) ;
   }
   gString gs ;                  // text formatting
   int64_t tstamp ;              // timestamp (epoch time)
   int64_t tsInc = oneDay ;      // timestamp increment
   short indx ;                  // offset into udAlloc[] array

   //* Get the current time and subtract RECORDS days (in seconds) *
   this->GetLocalTime ( tstamp ) ;
   tstamp -= oneDay * RECORDS ;

   //* Reinitialize the log file *
   gString gsTrg = this->cfg.logPath ;
   if ( (this->LogfileCreate ( gsTrg )) )
   {
      //* Open the logfile for append *
      ofstream ofs( gsTrg.ustr(), ofstream::out | ofstream::app ) ;
      if ( ofs.is_open() )
      {
         //* Create a series of records, one per day *
         for ( short i = ZERO ; i < RECORDS ; ++i )
         {
            //* Set an elapsed-time value and update the displayed record *
            this->ud.inpValue = baseval ;
            this->ud.inpUnits = dataType ;
            this->ud.exerType = exType ;
            this->Recalculate ( false ) ;

            //* Format the record for output *
            this->LogfileFormatRecord ( this->ud ) ;

            //* Synthesize the timestamp: scan to epoch and increment by *
            //* one day. (The human-readable timestamp is not updated at *
            //* this time.)                                              *
            for ( indx = ZERO ; (this->udAlloc[indx] != '\0') && (indx < 8192) ; ++indx ) ;
            while ( (this->udAlloc[indx] != ':') && (indx > ZERO) )
               --indx ;
            gs = &this->udAlloc[indx] ;
            gs.gscanf( ":%llX", &tstamp ) ;
            tstamp += tsInc ;
            tsInc += oneDay ;
            gs.compose( ":%llX\n\n", &tstamp ) ;
            gs.copy( &this->udAlloc[indx], 24 ) ;

            //* Save record to the logfile and release the dynamic allocation *
            ofs << this->udAlloc ;
            if ( this->udAlloc != NULL )
            { delete [] this->udAlloc ; this->udAlloc = NULL ; }

            //* Modulate the daily increment *
            if ( ((baseval += increment) < floor) || (baseval > ceiling) )
               baseval += increment = -(increment) ;
         }
         ofs.flush() ;        // flush the output buffer
         ofs.close() ;        // close the logfile
         this->UserMessage ( "  Test Logfile Created.  ", this->cfg.em, true ) ;
      }

      //* Pause so user can read the message *
      this_thread::sleep_for( aWhile ) ;
      this->UserMessage ( "", this->cfg.bb ) ;
   }  // (logfile reinitialized)

   //* Restore the 'ud' member data *
   this->ud = udBackup ;

}  //* End LogfileDebugSample() *
#endif   // DEBUG_MENU

//*************************
//*    DebugRtlExample    *
//*************************
//******************************************************************************
//* FOR DEBUGGING ONLY: Instantiate a sample dialog which demonstrates RTL     *
//* user-interface languages. This dialog is used in the application           *
//* documentation.                                                             *
//*                                                                            *
//* We take a number of shortcuts for this dialog. For instance, we do not     *
//* reference the 'this->cfg.lang' member, but instead hard-code two languages,*
//* English and Hebrew. In addition, we define REV4DOC to artificially invert  *
//* the RTL text so it will be captured canonically during the screen capture. *
//*                                                                            *
//* For the plain-text capture of text for the info reader documentation, we   *
//* capture artificially-reversed text because the Gnometerm and Konsole       *
//* terminal emulators insists on being "helpful" with RTL text. The text data *
//* are reversed using the gString class textReverse() method.                 *
//*                                                                            *
//* For capture in HTML format, we have choices:                               *
//*   1) We can capture the text canonically and block the browser's           *
//*      helpfulness with RTL data. This is done using the HTML5 sequence:     *
//*                     <bdo dir="ltr"> wxyz </bdo>                            *
//*   2) We can capture artificially-reversed text and allow the brower to     *
//*      apply its brain-damaged "helpfulness".                                *
//* We choose option 1 because the browser's algorithm is not reliable, AND    *
//* because both Firefox and Opera "helpfully" get the punctuation wrong.      *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

#if DEBUG_MENU != 0
void Exercalc::DebugRtlExample ( void )
{
   #define REV4DOC (0)     // artificially reverse RTL text for screen capture
   #if REV4DOC != 0
   gString gs0, gs1, gs2, gs3, gs4, gs5, gs6 ;  // control labels
   #endif   // REV4DOC

   //* Simulated language index *
   const short lang = this->cfg.rtl ? 1 : 0 ;

   //* Named indices *
   enum dre : short { dreClosePB = ZERO, dreChocRB, dreVaniRB, dreStrawRB,
                      dreCherryRB, dreLemonRB, dreCondiTB, dreITEMS,
                      dreTitle, dreStatic, dreTbText, 
                      drePrice, dreKilo, dreEuro, txtITEMS } ;

   //* Display text *
   const char* DlgText[2][txtITEMS] = 
   {
   //** English **
   {
     "    Close Dialog    ",                       // Close pushbutton
     "Chocolate",                                  // XOR Radiobutton group
     "Vanilla",
     "Strawberry",
     "Cherry",
     "Lemon Sherbet",
     "Condiments",                                 // Condiment textbox
     "DUMMY ELEMENT",
     "  My Favorite Ice Cream Flavor  ",           // Title
     "Please select your favorite ice cream flavor from\n"  // Static text
     "the radio button group, below. If you enjoy toppings\n"
     "with your ice cream, enter the names of your favorite\n"
     "condiments in the text box.\n",
     "I like caramel sauce!",                      // Textbox contents
     "    Sale! Today Only.    ",                  // Price-list heading
     "Kilogram",                                   // Kilogram
     "€",                                          // Euro
   },
   //** Español ** // omitted
   //** Zhōngwén (中文) ** // omitted
   //** TiếngViệt ** // omitted
   //** Hebrew **
   {  // Programmer's Note: Appears reversed in the text editor, 
      // but SHOULD be displayed correctly in the dialog.
     "  סגור את הדיאלוג   ",                       // Close pushbutton
     "שוקולד",                                     // XOR Radiobutton group
     "וניל",
     "גלידת תות",
     "דובדבן",
     "שרבט לימון",
     "תבלינים ותבלינים",                           // Condiment textbox
     "DUMMY ELEMENT",
     "  טעם הגלידה החביב עלי  ",                   // Title
     "אנא בחר את טעם הגלידה המועדף עליך מקבוצת לחצני\n"  // Static text
     "הבחירה, למטה. אם אתה נהנה מתוספות עם הגלידה שלך,\n"
     "הזן את שמות התבלינים האהובים עליך בתיבת הטקסט.\n"
     "זה אולי לא כשר, אבל בהחלט טעים!\n",
     "אני אוהב רוטב קרמל!",                        // Textbox contents
     "     מְכִירָה! רק היום.     ",                  // Price-list heading
     "קִילוֹגרָם",                                    // Kilogram
     "₪",                                          // Shekel
   },
   } ;
   // exchange rate 1 shekel == $0.31 == 0.25 euro
   // ice cream is approx. 1.50 euro/kilo
   const short PRICES = 5 ;
   float DlgVolume[PRICES] = 
   { 0.250, 0.500, 1.000, 2.000, 5.000 } ;
   float DlgPriceE[PRICES] =     // prices in Euros
   { 0.40, 0.75, 1.50, 2.80, 7.00 } ;
   float DlgPriceS[PRICES] =     // prices in Shekels
   { 1.60, 3.00, 6.00, 11.20, 28.00 } ;

   winPos wp = this->statsDlg->GetDialogPosition () ;
   const short dlgROWS = 24,
               dlgCOLS = 57,
               dlgULY  = wp.ypos,
               dlgULX  = wp.xpos ;
   attr_t dColor = (this->cfg.rtl ? nc.gybl | ncbATTR : nc.blR),
          bColor = nc.brbl,
          rnColor = bColor,         // Radiobutton colors
          rfColor = nc.rebl,
          tnColor = nc.brgr,        // Textbox colors
          tfColor = nc.blG,
          pnColor = nc.gyR,         // Pushbutton colors
          pfColor = (this->cfg.rtl ? nc.gyre | ncbATTR : nc.reR) ;
   gString gsOut, gsKio, gsVal ;
   wp = { 2, 2 } ;         // position of static text

   InitCtrl ic[dreITEMS] = 
   {
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - -  dreDonePB *
      dctPUSHBUTTON,                // type:      control type
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dlgROWS - 2),           // ulY:       upper left corner in Y
      short(dlgCOLS / 2 - 10),      // ulX:       upper left corner in X
      1,                            // lines:     control lines
      20,                           // cols:      control columns
      DlgText[lang][dreClosePB],    // dispText:  display text
      pnColor,                      // nColor:    non-focus color
      pfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dreChocRB],               // nextCtrl:  link in next structure
   },

   {  //* 'CHOCOLATE' Radiobutton - - - - - - - - - - - - - - - - -  dreChocRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      7,                            // ulY:       upper left corner in Y
      8,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      DlgText[lang][dreChocRB],     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dreVaniRB]                // nextCtrl:  link in next structure
   },

   {  //* 'VANILLA' Radiobutton - - - - - - - - - - - - - - - - - -  dreVaniRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      true,                         // rbSelect:  default selection
      short(ic[dreChocRB].ulY + 1), // ulY:       upper left corner in Y
      ic[dreChocRB].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      DlgText[lang][dreVaniRB],     // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dreStrawRB]               // nextCtrl:  link in next structure
   },

   {  //* 'STRAWBERRY' Radiobutton  - - - - - - - - - - - - - - -   dreStrawRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[dreVaniRB].ulY + 1), // ulY:       upper left corner in Y
      ic[dreVaniRB].ulX,            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      DlgText[lang][dreStrawRB],    // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dreCherryRB]              // nextCtrl:  link in next structure
   },

   {  //* 'CHERRY' Radiobutton  - - - - - - - - - - - - - - - - -  dreCherryRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[dreStrawRB].ulY + 1),// ulY:       upper left corner in Y
      ic[dreStrawRB].ulX,           // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      DlgText[lang][dreCherryRB],   // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dreLemonRB]               // nextCtrl:  link in next structure
   },

   {  //* 'LEMON' Radiobutton  - - - - - - - - - - - - - - - - - -  dreLemonRB *
      dctRADIOBUTTON,               // type:      
      rbtS3s,                       // rbSubtype: standard, 3 chars wide
      false,                        // rbSelect:  default selection
      short(ic[dreCherryRB].ulY + 1),// ulY:       upper left corner in Y
      ic[dreCherryRB].ulX,          // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a)
      rnColor,                      // nColor:    non-focus color
      rfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      DlgText[lang][dreLemonRB],    // label:
      ZERO,                         // labY:      
      5,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[dreCondiTB]               // nextCtrl:  link in next structure
   },

   {  //* 'CONDIMENT' Textbox  - - - - - - - - - - - - - - - - - -  dreCondiTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(ic[dreLemonRB].ulY + 2),// ulY:       upper left corner in Y
      ic[dreLemonRB].ulX,           // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      short(ic[dreClosePB].cols * 2),// cols:      control columns
      DlgText[lang][dreTbText],     // dispText:  
      tnColor,                      // nColor:    non-focus color
      tfColor,                      // fColor:    focus color
      tbPrint,                      // filter:    all printing claracters
      DlgText[lang][dreCondiTB],    // label:
      1,                            // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
   } ;

   //* Adjust control definitions for RTL data *
   if ( this->cfg.rtl )
   {
      #if REV4DOC != 0  // Reverse the text for screen capture
      gs0 = DlgText[lang][dreClosePB] ;
      gs0.textReverse() ;
      ic[dreClosePB].dispText = gs0.ustr() ;
      gs1 = DlgText[lang][dreChocRB] ;
      gs1.textReverse() ;
      ic[dreChocRB].label = gs1.ustr() ;
      gs2 = DlgText[lang][dreVaniRB] ;
      gs2.textReverse() ;
      ic[dreVaniRB].label = gs2.ustr() ;
      gs3 = DlgText[lang][dreStrawRB] ;
      gs3.textReverse() ;
      ic[dreStrawRB].label = gs3.ustr() ;
      gs4 = DlgText[lang][dreCherryRB] ;
      gs4.textReverse() ;
      ic[dreCherryRB].label = gs4.ustr() ;
      gs5 = DlgText[lang][dreLemonRB] ;
      gs5.textReverse() ;
      ic[dreLemonRB].label = gs5.ustr() ;
      gs6 = DlgText[lang][dreCondiTB] ;
      gs6.textReverse() ;
      ic[dreCondiTB].label = gs6.ustr() ;
      gsOut = DlgText[lang][dreTbText] ;
      gsOut.textReverse( true ) ;
      ic[dreCondiTB].dispText = gsOut.ustr() ;
      #endif   // REV4DOC
      ic[dreChocRB].ulX += ic[dreCondiTB].cols - 3 ;  // position radiobuttons
      ic[dreVaniRB].ulX = ic[dreChocRB].ulX ;
      ic[dreStrawRB].ulX = ic[dreChocRB].ulX ;
      ic[dreCherryRB].ulX = ic[dreChocRB].ulX ;
      ic[dreLemonRB].ulX = ic[dreChocRB].ulX ;
      ic[dreChocRB].labX = - 3 ;                      // radiobutton label offsets
      ic[dreVaniRB].labX = - 3 ;
      ic[dreStrawRB].labX = - 3 ;
      ic[dreCherryRB].labX = - 3 ;
      ic[dreLemonRB].labX = - 3 ;
      ic[dreCondiTB].labX = ic[dreCondiTB].cols - 1 ; // textbox label offset
   }

   //* Save the dialog window which will be obscured *
   this->statsDlg->SetDialogObscured () ;

   //* Define the dialog *
   InitNcDialog dInit
   (
    dlgROWS,         // number of display lines
    dlgCOLS,         // number of display columns
    dlgULY,          // Y offset from upper-left of terminal 
    dlgULX,          // X offset from upper-left of terminal 
    NULL,            // dialog title
    ncltDUAL,        // border line-style
    bColor,          // border color attribute
    dColor,          // interior color attribute
    ic               // pointer to list of control definitions
   ) ;

   //* Instantiate the dialog *
   NcDialog *dp = new NcDialog ( dInit ) ;

   //* If user interface language is an RTL language, *
   //* set the internal NcDialog flag for RTL output. *
   if ( this->cfg.rtl )
   {
      wp.xpos = dlgCOLS - 3 ;    // position of static text
      dp->DrawLabelsAsRTL () ;
      dp->DrawContentsAsRTL ( dreCondiTB, true ) ;    // textbox edit
   }

   if ( (dp->OpenWindow()) == OK )  // open the stats dialog
   {
      gsOut = DlgText[lang][dreTitle] ;
      #if REV4DOC != 0  // Reverse the text for screen capture
      if ( this->cfg.rtl ) gsOut.textReverse() ;
      #endif   // REV4DOC
      dp->SetDialogTitle ( gsOut.ustr(), dInit.borderColor | ncuATTR ) ;

      //* Write the static text *
      gsOut = DlgText[lang][dreStatic] ;
      #if REV4DOC != 0  // Reverse the text for screen capture
      if ( this->cfg.rtl ) gsOut.textReverse( true, true ) ;
      #endif   // REV4DOC
      wp = dp->WriteParagraph ( wp, gsOut, dColor, false, this->cfg.rtl ) ;

      //* Price List *
      wp = { short(ic[dreCondiTB].ulY + 2), short(dlgCOLS / 2) } ;
      gsOut = DlgText[lang][drePrice] ;
      #if REV4DOC != 0  // Reverse the text for screen capture
      if ( this->cfg.rtl ) gsOut.textReverse() ;
      #endif   // REV4DOC
      gsOut.append( L'\n' ) ;
      wp = dp->WriteParagraph ( wp, gsOut, bColor | ncuATTR, 
                                false, this->cfg.rtl ) ;
      wp.xpos += (this->cfg.rtl ? -2 : 2) ;
      for ( short i = ZERO ; i < 5 ; ++i )
      {
         gsOut.compose( "%1.2f", &DlgVolume[i] ) ; // amount in kilos
         gsVal.compose( "%1.2f", &DlgPriceE[i] ) ; // price in Euros
         gsKio = DlgText[lang][dreKilo] ;          // "kilogram"
         if ( this->cfg.rtl )
         {
            gsOut.textReverse( false, false, true ) ; // reverse the amount
            gsVal.compose( "%1.2f", &DlgPriceS[i] ) ; // price in Shekels
            gsVal.textReverse( false, false, true ) ; // reverse price
            #if REV4DOC != 0  // Reverse the text for screen capture
            gsKio.textReverse() ;                     // reverse "kilogram"
            #endif   // REV4DOC
         }
         gsOut.append( " %S %S%s\n", gsKio.gstr(), gsVal.gstr(), 
                                     DlgText[lang][dreEuro] ) ;
         wp = dp->WriteParagraph ( wp, gsOut, dColor, false, this->cfg.rtl ) ;
      }

      //* Group all radio buttons in the dialog *
      short XorGroup[] = { dreChocRB, dreVaniRB, dreStrawRB, 
                           dreCherryRB, dreLemonRB, -1 } ;
      dp->GroupRadiobuttons ( XorGroup ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control

      while ( ! done )
      {
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               #if 0    // TEMP - CAPTURE
               dp->CaptureDialog ( "capturedlg.txt" ) ;
               dp->CaptureDialog ( "capturedlg.html", true, false, 
                                   "infodoc-styles.css", 4, false, nc.blR ) ;
               #endif   // CAPTURE
               done = true ;
            }
         }

         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey )
               Info.HotData2Primary () ;
            else
               icIndex = dp->EditRadiobutton ( Info ) ;
         }

         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            Info.viaHotkey = false ;
            icIndex = dp->EditTextbox ( Info ) ;
         }

         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }  // while()
   }

   if ( dp != NULL )                // delete (close) the dialog
      delete dp ;

   //* Restore the obscured dialog *
   this->statsDlg->RefreshWin () ;

}  //* End DebugRtlExample() *
#endif   // DEBUG_MENU

//*************************
//*   DebugChartWidget    *
//*************************
//******************************************************************************
//* FOR DEBUGGING ONLY: Exercise the "Chart" class. That class will eventually *
//* be moved into the NcDialog API and will replace the hard-coded summary     *
//* chart in this application.                                                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

#if DEBUG_MENU != 0
void Exercalc::DebugChartWidget ( void )
{
const int32_t Data1Count = 200 ;
const double  Data1[Data1Count] // All data are positive values
{
    1.0,  1.5,  2.0,  2.5,  3.0,  3.5,  4.0,  4.5,  5.0,  5.5, // increase
    6.0,  6.5,  7.0,  7.5,  8.0,  8.5,  9.0,  9.5, 10.0, 10.5,
   11.0, 11.5, 12.0, 12.5, 13.0, 13.5, 14.0, 14.5, 15.0, 15.5,
   16.0, 16.5, 17.0, 17.5, 18.0, 18.5, 19.0, 19.5, 20.0, 20.5,

   20.3, 20.1, 19.9, 19.7, 19.5, 19.3, 19.1, 18.9, 18.7, 18.5, // decrease
   18.3, 18.1, 17.9, 17.7, 17.5, 17.3, 17.1, 16.9, 16.7, 16.5,

   15.0, 15.1, 15.2, 15.3, 15.4, 15.5, 15.6, 15.4, 15.2, 15.0, // fluctuate
   14.8, 14.6, 14.4, 14.2, 14.0, 14.5, 15.0, 15.5, 15.3, 15.1,

   15.0, 14.0, 15.0, 14.8, 14.6, 14.4, 14.2, 14.0, 13.8, 13.6, // decrease
   13.4, 13.2, 13.0, 12.8, 12.6, 12.4, 12.2, 12.0, 11.8, 11.6,
   11.4, 11.2, 11.0, 10.8, 10.6, 10.4, 10.2, 10.0,  9.8,  9.6,
    9.4,  9.2,  9.0,  8.8,  8.6,  8.4,  8.2,  8.0,  7.8,  7.6,
    7.4,  7.2,  7.0,  6.8,  6.6,  6.4,  6.2,  6.0,  6.8,  6.6,
    6.4,  6.2,  6.0,  5.8,  5.6,  5.4,  5.2,  5.0,  4.8,  4.6,
    4.4,  4.2,  4.0,  3.8,  3.6,  3.4,  3.2,  3.0,  2.8,  2.6,
    2.4,  2.2,  2.0,  1.8,  1.6,  1.4,  1.2,  1.0,  0.8,  0.6,
    
    0.4,  0.2,  0.0,  0.5,  1.0,  1.5,  2.0,  2.5,  3.0,  2.5, // step up
    2.0,  2.5,  3.0,  3.5,  4.0,  4.5,  5.0,  4.5,  4.0,  4.5,
    5.0,  5.5,  6.0,  6.5,  7.0,  7.5,  8.0,  8.5,  9.0,  9.5,
   10.0, 11.0, 12.0, 13.0, 13.5, 14.0, 14.5, 15.0, 15.5, 14.0,
} ;
attr_t Attr1[Data1Count]
{
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl,

   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
   nc.br, nc.cy, nc.br, nc.ma, nc.br, nc.bl, nc.br, nc.bl, nc.br, nc.bl, 
} ;
const int32_t Data2Count = 200 ;
const double  Data2[Data2Count] // Mixed positive and negative data values
{
    // first page, wander about
    5.0,  5.5,  6.0,  6.5,  5.5,  4.5,  4.0,  2.0,  1.5,  1.0,
    0.2,  0.0, -0.3, -0.7, -1.1, -1.6, -2.2, -2.8, -3.4, -4.0,
   -4.8, -5.1, -5.5, -6.0, -6.2, -6.4, -6.8, -7.4, -7.5, -7.5,
   -7.4, -7.2, -6.9, -6.5, -6.0, -5.4, -4.7, -3.9, -2.0, -1.0,
    0.1,  1.3,  2.4,  3.5,  4.6,  4.0,  3.0,  2.0,  1.0,  0.5,
   -1.5, -2.0, -2.5, -3.0, -2.4, -2.1, -1.6, -1.1, -0.8, -0.4,
    // increase from zero to max and back to zero
    0.0,  0.2,  0.4,  0.7,  0.9,  1.1,  1.4,  1.6,  1.8,  2.1,
    2.3,  2.5,  2.8,  3.0,  3.2,  3.5,  3.7,  3.9,  4.2,  4.4,
    4.6,  4.9,  5.1,  5.3,  5.6,  5.8,  6.0,  6.3,  6.5,  6.7,
    7.0,  7.2,  7.4,  7.5,  7.4,  7.2,  7.0,  6.7,  6.5,  6.3,
    6.0,  5.8,  5.6,  5.3,  5.1,  4.9,  4.6,  4.4,  4.2,  3.9,
    3.7,  3.5,  3.2,  3.0,  2.8,  2.5,  2.3,  2.1,  1.8,  1.6,
    1.4,  1.1,  0.9,  0.7,  0.4,  0.2,  0.0, -0.2, -0.4, -0.7,
   // decrease from zero to min and back to zero
   -0.9, -1.1, -1.4, -1.6, -1.8, -2.1, -2.3, -2.5, -2.8, -3.0,
   -3.2, -3.5, -3.7, -3.9, -4.2, -4.4, -4.6, -4.9, -5.1, -5.3,
   -5.6, -5.8, -6.0, -6.3, -6.5, -6.7, -7.0, -7.2, -7.4, -7.5,
   -7.4, -7.2, -7.0, -6.7, -6.5, -6.3, -6.0, -5.8, -5.6, -5.3,
   -5.1, -4.9, -4.6, -4.4, -4.2, -3.9, -3.7, -3.5, -3.2, -3.0,
   -2.8, -2.5, -2.3, -2.1, -1.8, -1.6, -1.4, -1.1, -0.9, -0.7,
   -0.4, -0.2,  0.0,  0.5,  1.0,  1.5,  2.0,  1.5,  2.0,  2.5,
} ;

const int32_t IntCount = 50 ;
const float FloatData[IntCount]
{
   15.0,  12.0,   9.0,   6.0,   3.0,   1.0,   0.0,  -2.0,  -4.0,  -6.0,
   -8.0, -10.0, -12.0, -14.0, -15.0, -14.5, -14.0, -13.0, -12.5, -12.0,
  -11.5, -11.0, -10.5, -10.0,  -9.0,  -8.5,  -7.0,  -6.0,  -5.0,  -4.0,
   -3.0,  -2.0,  -1.0,  -0.0,   1.0,   2.5,   4.0,   5.5,   7.0,   8.5,
   10.0,  11.0,  11.5,  12.0,  12.5,  13.0,  13.5,  14.0,  14.5,  15.0,
} ;
const char sByteData[IntCount]
{
   15,  12,   9,   6,   3,   1,   0,  -2,  -4,  -6,
   -8, -10, -12, -14, -15, -14, -14, -13, -12, -12,
  -11, -11, -10, -10,  -9,  -8,  -7,  -6,  -5,  -4,
   -3,  -2,  -1,  -0,   1,   2,   4,   5,   7,   8,
   10,  11,  11,  12,  12,  13,  13,  14,  14,  15,
}  ;
const unsigned char uByteData[IntCount]
{
    0,   2,   4,   6,   8,  10,  12,  14,  16,  18,
   20,  22,  24,  26,  28,  30,  32,  31,  30,  29,
   28,  27,  26,  25,  24,  23,  22,  21,  20,  19,
   18,  17,  16,  15,  14,  13,  12,  11,  10,   9,
    8,   7,   6,   5,   4,   3,   4,   3,   2,   1,
}  ;
const short sShortData[IntCount]
{
   15,  12,   9,   6,   3,   1,   0,  -2,  -4,  -6,
   -8, -10, -12, -14, -15, -14, -14, -13, -12, -12,
  -11, -11, -10, -10,  -9,  -8,  -7,  -6,  -5,  -4,
   -3,  -2,  -1,  -0,   1,   2,   4,   5,   7,   8,
   10,  11,  12,  11,  12,  13,  14,  13,  14,  15,
}  ;
const unsigned short uShortData[IntCount]
{
    0,   2,   4,   6,   8,  10,  12,  14,  16,  18,
   20,  22,  24,  26,  28,  30,  32,  31,  30,  29,
   28,  27,  26,  25,  24,  23,  22,  21,  20,  19,
   18,  17,  16,  15,  14,  13,  12,  11,  10,   9,
    8,   7,   6,   5,   4,   3,   4,   3,   2,   1,
}  ;
const int sIntData[IntCount]
{
  150,  120,   90,   60,   30,   10,    0,  -20,  -40,  -60,
  -80, -100, -120, -140, -150, -140, -135, -125, -120, -115,
 -110, -105, -100,  -80,  -70,  -60,  -50,  -40,  -30,  -20,
  -10,   -5,    0,    5,   10,   20,   30,   40,   45,   50,
   55,   60,   65,   70,   75,   80,   85,   90,  100,  115,
}  ;
const unsigned int uIntData[IntCount]
{
    0,  20,  40,  60,  80, 100, 120, 140, 160, 180,
  200, 220, 240, 260, 280, 300, 320, 310, 300, 290,
  280, 270, 260, 250, 240, 230, 220, 210, 200, 190,
  180, 170, 160, 150, 140, 130, 120, 110, 105, 100,
   90,  80,  70,  60,  55,  50,  45,  40,  35,  30,
}  ;
const long int sLongData[IntCount]
{
  150,  200,  300,  400,  500,  550,  600,   700,  800,  900,
 1000,  850,  700,  550,  400,  250,  100,   -50, -200, -350,
 -500, -650, -800, -950, -975, -985, -995, -1000, -900, -800,
 -750, -650, -500, -400, -200, -100,    0,    50,  100,  150,
  200,  250,  300,  350,  400,  450,  500,   550,  600,  700,
}  ;
const unsigned long int uLongData[IntCount]
{
     0,  200,  400,  600,  800, 1000, 1200, 1400, 1600, 1800,
  2000, 2200, 2400, 2600, 2800, 3000, 3200, 3100, 3000, 2900,
  2800, 2700, 2600, 2500, 2400, 2300, 2200, 2100, 2000, 1900,
  1800, 1700, 1600, 1500, 1400, 1300, 1200, 1100, 1050, 1000,
   900,  800,  700,  600,  550,  500,  450,  400,  350,  300,
}  ;
const long long int sLongLongData[IntCount]
{
  150,  200,  300,  400,  500,  550,  600,   700,  800,  900,
 1000,  850,  700,  550,  400,  250,  100,   -50, -200, -350,
 -500, -650, -800, -950, -975, -985, -995, -1000, -900, -800,
 -750, -650, -500, -400, -200, -100,    0,    50,  100,  150,
  200,  250,  300,  350,  400,  450,  500,   550,  600,  700,
}  ;
const unsigned long long int uLongLongData[IntCount]
{
     0,  200,  400,  600,  800, 1000, 1200, 1400, 1600, 1800,
  2000, 2200, 2400, 2600, 2800, 3000, 3200, 3100, 3000, 2900,
  2800, 2700, 2600, 2500, 2400, 2300, 2200, 2100, 2000, 1900,
  1800, 1700, 1600, 1500, 1400, 1300, 1200, 1100, 1050, 1000,
   900,  800,  700,  600,  550,  500,  450,  400,  350,  300,
}  ;
//* Coordinate pairs for use by Cartesian charts *
const int32_t CartDataCount = 408 ;    // Two hundred four X/Y Pairs
const double CartData1[CartDataCount]  // Range in X: -20.0 through 20.0
{                                      // Range in Y: ?
   //* First and Second Quadrants *
     1.0,  1.0,     // 00
     1.0,  2.0,     // 01
     2.0,  2.0,     // 02
     3.0,  0.0,     // 03
     4.0,  0.0,     // 04
     4.0,  1.0,     // 05
     4.0,  3.0,     // 06
     5.0,  2.0,     // 07
     5.0,  4.0,     // 08
     6.0, -1.0,     // 09
     6.0,  0.0,     // 10
     6.0,  2.0,     // 11
     7.0,  1.0,     // 12
     7.0,  3.0,     // 13
     7.0,  5.0,     // 14
     8.0, -2.0,     // 15
     8.0,  0.0,     // 16
     8.0,  3.0,     // 17
     8.0,  6.0,     // 18
     9.0,  1.0,     // 19
     9.0,  4.0,     // 20
    10.0, -3.0,     // 21
    10.0,  0.0,     // 22
    10.0,  4.0,     // 23
    10.0,  7.0,     // 24
    11.0,  1.0,     // 25
    11.0,  3.0,     // 26
    11.0,  5.0,     // 27
    11.0,  8.0,     // 28
    12.0, -4.0,     // 29
    12.0,  0.0,     // 30
    12.0,  5.0,     // 31
    12.0,  6.0,     // 32
    13.0,  2.0,     // 33
    13.0,  4.0,     // 34
    13.0,  7.0,     // 35
    14.0, -5.0,     // 36
    14.0,  0.0,     // 37
    14.0,  4.0,     // 38
    14.0,  6.0,     // 39
    15.0,  2.0,     // 40
    15.0,  5.0,     // 41
    15.0,  7.0,     // 42
    16.0, -6.0,     // 43
    16.0,  4.0,     // 44
    16.0,  7.0,     // 45
    17.0,  0.0,     // 46
    17.0,  2.0,     // 47
    17.0,  5.0,     // 48
    17.0,  7.0,     // 49
    18.0,  3.0,     // 50
    18.0,  6.0,     // 51
    18.0,  7.0,     // 52
    19.0,  0.0,     // 53
    19.0,  2.0,     // 54
    20.0, -8.0,     // 55
    20.0,  7.0,     // 56
    19.0, -2.0,     // 57
    18.0, -3.0,     // 58
    18.0, -6.0,     // 59
    17.0, -2.0,     // 60
    17.0, -5.0,     // 61
    17.0, -7.0,     // 62
    16.0, -7.0,     // 63
    16.0, -4.0,     // 64
    16.0,  6.0,     // 65
    15.0, -2.0,     // 66
    15.0, -5.0,     // 67
    15.0, -7.0,     // 68
    14.0,  5.0,     // 69
    14.0, -4.0,     // 70
    14.0, -6.0,     // 71
    13.0, -2.0,     // 72
    13.0, -4.0,     // 73
    13.0, -8.0,     // 74
    12.0, -6.0,     // 75
    12.0, -5.0,     // 76
    12.0,  4.0,     // 77
    11.0, -1.0,     // 78
    11.0, -3.0,     // 79
    11.0, -5.0,     // 80
    11.0, -8.0,     // 81
    10.0, -7.0,     // 82
    10.0, -4.0,     // 83
    10.0,  3.0,     // 84
     9.0, -1.0,     // 85
     9.0, -4.0,     // 86
     8.0, -6.0,     // 87
     8.0, -3.0,     // 88
     8.0,  2.0,     // 89
     7.0, -1.0,     // 90
     7.0, -3.0,     // 91
     7.0, -5.0,     // 92
     6.0, -2.0,     // 93
     6.0,  1.0,     // 94
     5.0, -2.0,     // 95
     5.0, -4.0,     // 96
     4.0, -3.0,     // 97
     4.0, -1.0,     // 98
     2.0, -1.0,     // 99
     2.0, -2.0,     //100
     1.0, -1.0,     //101

   //* Third and Fourth Quadrants *
    -1.0, -1.0,     //102
    -1.0, -2.0,     //103
    -2.0, -2.0,     //104
    -3.0,  0.0,     //105
    -4.0,  0.0,     //106
    -4.0, -1.0,     //107
    -4.0, -3.0,     //108
    -5.0, -2.0,     //109
    -5.0, -4.0,     //110
    -6.0,  1.0,     //111
    -6.0, -0.0,     //112
    -6.0, -2.0,     //113
    -7.0, -1.0,     //114
    -7.0, -3.0,     //115
    -7.0, -5.0,     //116
    -8.0,  2.0,     //117
    -8.0, -0.0,     //118
    -8.0, -3.0,     //119
    -8.0, -6.0,     //120
    -9.0, -1.0,     //121
    -9.0, -4.0,     //122
   -10.0,  3.0,     //123
   -10.0, -0.0,     //124
   -10.0, -4.0,     //125
   -10.0, -7.0,     //126
   -11.0, -1.0,     //127
   -11.0, -3.0,     //128
   -11.0, -5.0,     //129
   -11.0, -8.0,     //130
   -12.0,  4.0,     //131
   -12.0, -0.0,     //132
   -12.0, -5.0,     //133
   -12.0, -6.0,     //134
   -13.0, -2.0,     //135
   -13.0, -4.0,     //136
   -13.0, -7.0,     //137
   -14.0,  5.0,     //138
   -14.0, -0.0,     //139
   -14.0, -4.0,     //140
   -14.0, -6.0,     //141
   -15.0, -2.0,     //142
   -15.0, -5.0,     //143
   -15.0, -7.0,     //144
   -16.0,  6.0,     //145
   -16.0, -4.0,     //146
   -16.0, -7.0,     //147
   -17.0, -0.0,     //148
   -17.0, -2.0,     //149
   -17.0, -5.0,     //150
   -17.0, -7.0,     //151
   -18.0, -3.0,     //152
   -18.0, -6.0,     //153
   -18.0, -7.0,     //154
   -19.0, -0.0,     //155
   -19.0, -2.0,     //156
   -20.0,  8.0,     //157
   -20.0, -7.0,     //158
   -19.0,  2.0,     //159
   -18.0,  3.0,     //160
   -18.0,  6.0,     //161
   -17.0,  2.0,     //162
   -17.0,  5.0,     //163
   -17.0,  7.0,     //164
   -16.0,  7.0,     //165
   -16.0,  4.0,     //166
   -16.0, -6.0,     //167
   -15.0,  2.0,     //168
   -15.0,  5.0,     //169
   -15.0,  7.0,     //170
   -14.0, -5.0,     //171
   -14.0,  4.0,     //172
   -14.0,  6.0,     //173
   -13.0,  2.0,     //174
   -13.0,  4.0,     //175
   -13.0,  8.0,     //176
   -12.0,  6.0,     //177
   -12.0,  5.0,     //178
   -12.0, -4.0,     //179
   -11.0,  1.0,     //180
   -11.0,  3.0,     //181
   -11.0,  5.0,     //182
   -11.0,  8.0,     //183
   -10.0,  7.0,     //184
   -10.0,  4.0,     //185
   -10.0, -3.0,     //186
    -9.0,  1.0,     //187
    -9.0,  4.0,     //188
    -8.0,  6.0,     //189
    -8.0,  3.0,     //190
    -8.0, -2.0,     //191
    -7.0,  1.0,     //192
    -7.0,  3.0,     //193
    -7.0,  5.0,     //194
    -6.0,  2.0,     //195
    -6.0, -1.0,     //196
    -5.0,  2.0,     //197
    -5.0,  4.0,     //198
    -4.0,  3.0,     //199
    -4.0,  1.0,     //200
    -2.0,  1.0,     //201
    -2.0,  2.0,     //202
    -1.0,  1.0,     //203
} ;
//* These data bracket the endpoints of the X and Y axes.*
// Programmer's Note: These values were empirically constructed to test the 
// limits of the overlay using the default chart dimensions. If the "alternate"
// chart dimensions or borders are used, the datapoint layout will be less pretty.
const int32_t CartOverlayCount = 76 ;
const double CartOverlayData[CartOverlayCount]
{
     0.0,   9.0,    // 00 (off-scale above, should be discarded)
     1.5,   8.0,    // 01
     2.0,   7.0,    // 02
     2.5,   6.0,    // 03
     2.0,   5.0,    // 04
     1.5,   4.0,    // 05
    -1.0,   8.0,    // 06
    -1.5,   7.0,    // 07
    -2.5,   6.0,    // 08
    -1.5,   5.0,    // 09
    -1.0,   4.0,    // 10

     0.0,  -9.0,    // 11 (off-scale below, should be discarded)
     1.5,  -8.0,    // 12
     2.0,  -7.0,    // 13
     2.5,  -6.0,    // 14
     2.0,  -5.0,    // 15
     1.5,  -4.0,    // 16
    -1.0,  -8.0,    // 17
    -1.5,  -7.0,    // 18
    -2.5,  -6.0,    // 19
    -1.5,  -5.0,    // 20
    -1.0,  -4.0,    // 21

    21.5,   0.0,    // 22 (off-scale right, should be discarded)
    21.0,   1.0,    // 23
    20.0,   2.0,    // 24
    19.5,   1.0,    // 25
    19.0,   0.0,    // 26
    21.0,  -1.0,    // 27
    20.0,  -2.0,    // 28
    19.5,  -1.0,    // 29

   -21.0,   0.0,    // 30 (off-scale left, should be discarded)
   -20.0,   1.0,    // 31
   -19.5,   2.0,    // 32
   -19.0,   1.0,    // 33
   -18.5,   0.0,    // 34
   -20.0,  -1.0,    // 35
   -19.5,  -2.0,    // 36
   -19.0,  -1.0,    // 37
} ;

const char* const horizLabel = "Horizontal-axis Label" ;
const char* const vertLabel  = "Vertical-axis Label" ;
const char* const headText   = 
   " This chart displays some random dataset in a way that users \n"
   " could find interesting. -- (users are such simple creatures)" ;
const char* const footText   =
   "  These data were recorded by people in white lab coats and  \n"
   "       bad haircuts, so you know it must be accurate.        " ;
const short COL1_ITEMS = 28 ;    // Prompt, column 1 items
const char* const PromptCol1 =   // Prompt, column 1
            "0) ctLowerLeft\n"               // 0
            "1) ctLowerRight\n"              // 1
            "2) ctUpperLeft\n"               // 2
            "3) ctUpperRight\n"              // 3
            "4) ctLowerCenter\n"             // 4
            "5) ctUpperCenter\n"             // 5
            "6) ctCenterLeft\n"              // 6
            "7) ctCenterRight\n"             // 7
            "8) ctCartesian\n"               // 8
            "   -----\n"                     // 9
            "s) Grid: ncltSINGLE\n"          //10
            "d) Grid: ncltDUAL\n"            //11
            "   -----\n"                     //12
            "b) Border: ncltSINGLE\n"        //13
            "B) Border: ncltDUAL\n"          //14
            "x) Border: none\n"              //15
            "l) Draw horizontal line\n"      //16
            "   -----\n"                     //17
            "h) Set Header text\n"           //18
            "f) Set Footer text\n"           //19
            "t) Set Title\n"                 //20
            "   -----\n"                     //21
            "c) Custom shift keys\n"         //22
            "a) Audible shift alert\n"       //23
            "   -----\n"                     //24
            "n) Mixed Pos/Neg data\n"        //25
            "N) Alt color for neg values\n"  //26
            "   -----\n\n" ;                 //27
const short COL2_ITEMS = 21 ;    // Prompt, column 2 items
const char* const PromptCol2 =   // Prompt, column 2
            "i) Create NcDialog window\n"       // 0
            "   -----\n"                        // 1
            "v) Vertical Bars\n"                // 2
            "z) Horizontal Bars\n"              // 3
            "m) Multi-color Bars\n"             // 4
            "T) Bar tips only\n"                // 5
            "w) Narrow bar width\n"             // 6
            "W) Widely spaced bars\n"           // 7
            "C) Alt. Cartesian character\n"     // 8
            "---Input Data Formats---\n"        // 9
            "Alt+a)   Double\n"                 //10
            "Alt+A)   Float\n"                  //11
            "Alt+b,B) Byte(signed,unsigned)\n"  //12
            "Alt+c,C) Short\n"                  //13
            "Alt+d,D) Integer\n"                //14
            "Alt+e,E) Long Int\n"               //15
            "Alt+f,F) LongLong\n"               //16
            "---Misc Adjustments---\n"          //17
            "Alt+g) Cartesian overlay\n"        //18
            "Alt+G) Odd rows, even cols\n"      //19
            "   -----\n" ;                      //20
const char* ShiftInstrDflt = "Shift Data using:\n"
                             "RightArrow :+1,  LeftArrow :-1\n"
                             "Shift+Right:+5,  Shift+Left:-5\n"
                             "Ctrl+Right :+10, Ctrl+Left :-10\n"
                             "PageDown, PageUp, Home, End,\n"
                             "(any other key to close chart)\n" ;
const char* ShiftInstrCust = "'f' first page  'l' last page\n"
                             "'n' next page   'p' prev page\n"
                             "'a' : +1    'A' : -1\n"
                             "'b' : +5    'B' : -5\n"
                             "'c' : +10   'C' : -10\n"
                             "(any other key to close chart)\n" ;
const char* ParentDlgTitle = "  Display Chart in Parent Dialog  " ;
const char* ChildDlgTitle  = "  Local Chart Dialog Window  " ;

   Chart *cp = NULL ;            // pointer to Chart instance
   gString cVersion ;            // Chart-class version
   winPos  cvPos ;               // display position for chart-class version

   //* Define the chart parameters *
   chartDef cdef = 
   {
      this->statsDlg,      // dPtr:        dialog pointer
      2,                   // ulY:         Y origin
      1,                   // ulX:         X origin
      20,                  // rows:        rows
      short(sdInit.dColumns-2),// cols:    columns
      1,                   // yOff:        Y offset
      1,                   // xOff:        X offset
      3,                   // yFoot:       footer rows
      cellDIV,             // barWidth:    bar width (divisions)
      ZERO,                // barSpace:    bar spacing (0 or 1)
      ctLowerLeft,         // chType:      chart type (see below)
      this->cfg.bb,        // borderColor: border color
      this->cfg.bb,        // titleColor:  title color
      this->cfg.bb,        // textColor:   text color
      this->cfg.sb,        // boldColor:   bold color
      this->cfg.bb,        // gridColor:   grid color
      this->cfg.bb,        // barColor:    bar color (all values, or positive for centered axes)
      this->cfg.bb,        // negColor:    bar color (negative values for centered axes)
      Data1Count,          // dataCount:   data element count
      ZERO,                // dataOffset:  offset into data array
      Data1,               // dataPtr:     data array
      idtDouble,           // dataType:    type of data referenced by 'dataPtr'
      NULL,                // attrPtr:     bar-color array
      ncltSINGLE,          // borderStyle: border style (line type)
      ncltSINGLE,          // gridStyle:   grid style (see below)
      vertLabel,           // vaxisLabel:  vertical-axis label
      horizLabel,          // haxisLabel:  horizontal-axis label
      NULL,                // title:       dialog title text
      NULL,                // headText:    header text
      NULL,                // footText:    footer text
      false,               // horizBars:   'true' if horizontal bars, 'false' if vertical bars
      false,               // barTips:     'false' if full bar, 'true' if pseudo-line
      false,               // border:      'true' if area border, 'false' if no border
   } ;

   //* User selects configuration options *
   winPos   wp1( 2, 3 ),            // prompt, column 1
            wp2( 2, short(sdInit.dColumns / 2) ), // prompt, column 2
            wpShiftInstr( 6, 31 ),  // Y/X offset for shift instructions
            wpFooter,               // receives position of footer
            wpx ;                   // item ticks
   wkeyCode wk ;                    // user input
   wchar_t  selChar ;               // active/inactive item in menu
   short    ftrRows, ftrCols ;      // dimensions of the footer area
   bool customShift = false,        // 'true' if custom data-shift keys
        audibleShift = true,        // 'true' if audible beep on shift-key error
        horizLine = false,          // 'true' if horizontal line in footer area
        altcartChar = false,        // 'true' if alternate char for Cartesian datapoints
        cartOverlay = false,        // 'true' if secondary data overlay for Cartesian chart
        altDimensions = false,      // 'true' if alternate grid dimensioning
        done = false ;              // loop control

   //* Set the dialog title *
   this->statsDlg->SetDialogTitle ( "  Debug Chart Class Object  ", 
                                    this->cfg.bb | ncuATTR ) ;

#if 0    // TEMP - EXPERIMENTAL
int bs = sizeof(char),
    bu = sizeof(unsigned char),
    ss = sizeof(short),
    su = sizeof(unsigned short),
    is = sizeof(int),
    iu = sizeof(unsigned int),
    ls = sizeof(long),
    lu = sizeof(unsigned long),
    Ls = sizeof(long long),
    Lu = sizeof(unsigned long long),
    f  = sizeof(float),
    D  = sizeof(double) ;
gString gx( "b:%d,%d s:%d,%d i:%d,%d l:%d,%d L:%d,%d f:%d D:%d",
            &bs, &bu, &ss, &su, &is, &iu, &ls, &lu, &Ls, &Lu, &f, &D ) ;
this->statsDlg->WriteString ( 1, 1, gx, nc.gr, true ) ;
nckPause();
#endif   // TEMP - EXPERIMENTAL
   while ( ! done )
   {
      //* Display the prompt *
      this->statsDlg->ClearWin () ;          // clear the target window
      cvPos = this->statsDlg->WriteParagraph ( wp1, PromptCol1, this->cfg.bb ) ;
      if ( cVersion.gschars() > 1 )
         this->statsDlg->WriteString ( cvPos, cVersion, this->cfg.bb ) ;
      this->statsDlg->WriteParagraph ( wp2, PromptCol2, this->cfg.bb ) ;

      this->statsDlg->WriteParagraph ( (wp1.ypos + 28), (wp1.xpos + 28),
                                       "SPACE (or ENTER) to display chart\n"
                                       "Q) (or TAB) to Quit\n", this->cfg.bb ) ;

      wpx = { wp1.ypos, short(wp1.xpos - 1) } ;
      for ( short i = ZERO ; i < COL1_ITEMS ; ++i )
      {
         switch ( i )
         {
            case  0: selChar = cdef.chType == ctLowerLeft   ? L'*' : L' ' ;   break ;
            case  1: selChar = cdef.chType == ctLowerRight  ? L'*' : L' ' ;   break ;
            case  2: selChar = cdef.chType == ctUpperLeft   ? L'*' : L' ' ;   break ;
            case  3: selChar = cdef.chType == ctUpperRight  ? L'*' : L' ' ;   break ;
            case  4: selChar = cdef.chType == ctLowerCenter ? L'*' : L' ' ;   break ;
            case  5: selChar = cdef.chType == ctUpperCenter ? L'*' : L' ' ;   break ;
            case  6: selChar = cdef.chType == ctCenterLeft  ? L'*' : L' ' ;   break ;
            case  7: selChar = cdef.chType == ctCenterRight ? L'*' : L' ' ;   break ;
            case  8: selChar = cdef.chType == ctCartesian   ? L'*' : L' ' ;   break ;
            case  9: selChar = L' ' ; break ;
            case 10: selChar = cdef.gridStyle == ncltSINGLE ? L'*' : L' ' ;   break ;
            case 11: selChar = cdef.gridStyle == ncltDUAL   ? L'*' : L' ' ;   break ;
            case 12: selChar = L' ' ; break ;
            case 13:
            case 14:
               selChar = L' ' ;
               if ( cdef.border )
               {
                  if ( (i == 13 && cdef.borderStyle == ncltSINGLE) ||
                       (i == 14 && cdef.borderStyle == ncltDUAL) )
                  {
                     selChar = L'*' ;
                  }
               }
               break ;
            case 15: selChar = cdef.border ? L' ' : L'*' ;            break ;
            case 16: selChar = horizLine ? L'*' : L' ' ;              break ;
            case 17: selChar = L' ' ;                                 break ;
            case 18:
               if ( cdef.headText != NULL )  selChar = L'*' ;
               else                          selChar = L' ' ;
               break ;
            case 19:
               if ( cdef.footText != NULL )  selChar = L'*' ;
               else                          selChar = L' ' ;
               break ;
            case 20: selChar = (cdef.title == NULL ? L' ' : L'*') ;   break ;
            case 21: selChar = L' ' ;                                 break ;
            case 22: selChar = (customShift ? L'*' : L' ') ;          break ;
            case 23: selChar = (audibleShift ? L'*' : L' ') ;         break ;
            case 24: selChar = L' ' ;                                 break ;
            case 25: selChar = (cdef.dataPtr != Data1 ? L'*' : L' ') ; break ;
            case 26: selChar = (cdef.negColor != cdef.barColor ? L'*' : L' ') ; break ;
            case 27: selChar = L' ' ;                                 break ;
         } ;
         this->statsDlg->WriteChar ( wpx.ypos++, wpx.xpos, selChar, this->cfg.bb ) ;
      }
      wpx = { wp2.ypos, short(wp2.xpos - 1) } ;
      for ( short i = ZERO ; i < COL2_ITEMS ; ++i )
      {
         switch ( i )
         {
            case  0: selChar = (cdef.dPtr == NULL ? L'*' : L' ') ;    break ;
            case  1: selChar = L' ' ;                                 break ;
            case  2: selChar = (cdef.horizBars ? L' ' : L'*') ;       break ;
            case  3: selChar = (cdef.horizBars ? L'*' : L' ') ;       break ;
            case  4: selChar = (cdef.attrPtr == NULL ? L' ' : L'*') ; break ;
            case  5: selChar = (cdef.barTips ? L'*' : L' ') ;         break ;
            case  6: selChar = (cdef.barWidth == cellDIV ? L' ' : L'*') ; break ;
            case  7: selChar = (cdef.barSpace == 1 ? L'*' : L' ') ;   break ;
            case  8: selChar = (altcartChar ? L'*' : L' ') ;          break ;
            //* Input Data Formats *
            case  9: selChar = L' ' ;                                 break ;
            case 10:
               selChar = (cdef.dataType == idtDouble) ? L'*' : L' ' ;
               break ;
            case 11:
               selChar = (cdef.dataType == idtFloat) ? L'*' : L' ' ;
               break ;
            case 12:
               selChar = ((cdef.dataType == idtByte_s) ||
                          (cdef.dataType == idtByte_u)) ? L'*' : L' ' ;
               break ;
            case 13:
               selChar = ((cdef.dataType == idtShort_s) ||
                          (cdef.dataType == idtShort_u)) ? L'*' : L' ' ;
               break ;
            case 14:
               selChar = ((cdef.dataType == idtInt_s) ||
                          (cdef.dataType == idtInt_u)) ? L'*' : L' ' ;
               break ;
            case 15:
               selChar = ((cdef.dataType == idtLong_s) ||
                          (cdef.dataType == idtLong_u)) ? L'*' : L' ' ;
               break ;
            case 16:
               selChar = ((cdef.dataType == idtLongLong_s) ||
                          (cdef.dataType == idtLongLong_u)) ? L'*' : L' ' ;
               break ;
            case 17: selChar = L' ' ;                                 break ;
            case 18: selChar = cartOverlay ? L'*' : L' ' ;            break ;
            case 19: selChar = altDimensions ? L'*' : L' ' ;          break ;
            case 20: selChar = L' ' ;                                 break ;
         } ;
         this->statsDlg->WriteChar ( wpx.ypos++, wpx.xpos, selChar, this->cfg.bb ) ;
      }
      this->statsDlg->RefreshWin () ;

      this->statsDlg->GetKeyInput ( wk ) ;
      if ( wk.type == wktFUNKEY && wk.key == nckENTER )
         wk = { nckSPACE, wktPRINT } ;

      if ( wk.type == wktPRINT )
      {
         //* For non-Cartesian chart types, verify that     *
         //* 'dataPtr' DOES NOT point to the Cartesian data.*
         if ( (cdef.dataPtr == CartData1) && (wk.key >= L'0') && (wk.key <= L'7') )
         { cdef.dataPtr = Data1 ; cdef.dataCount = Data1Count ; }

         switch ( wk.key )
         {
            case L'0': cdef.chType = ctLowerLeft ;    break ;
            case L'1': cdef.chType = ctLowerRight ;   break ;
            case L'2': cdef.chType = ctUpperLeft ;    break ;
            case L'3': cdef.chType = ctUpperRight ;   break ;
            case L'4': cdef.chType = ctLowerCenter ;  break ;
            case L'5': cdef.chType = ctUpperCenter ;  break ;
            case L'6': cdef.chType = ctCenterLeft ;   break ;
            case L'7': cdef.chType = ctCenterRight ;  break ;
            case L'8':
               cdef.chType    = ctCartesian ;
               cdef.dataPtr   = CartData1 ;
               cdef.dataCount = CartDataCount ;
               break ;
            case L's': cdef.gridStyle = ncltSINGLE ;  break ;
            case L'd': cdef.gridStyle = ncltDUAL ;    break ;
            case L'b':
               cdef.borderStyle = ncltSINGLE ;
               cdef.border = true ;
               break ;
            case L'B':
               cdef.borderStyle = ncltDUAL ;
               cdef.border = true ;
               break ;
            case L'x':
               cdef.borderStyle = ncltSINGLE ;
               cdef.border = false ;
               break ;
            case L'l': horizLine = horizLine ? false : true ;        break ;
            case L'h':
               if ( cdef.headText == NULL )
               { cdef.headText = headText ; cdef.yOff += 2 ; }
               else
               { cdef.headText = NULL ; cdef.yOff -= 2 ; }
               break ;
            case L'f':
               if ( cdef.footText == NULL )
               { cdef.footText = footText ;  }
               else
               { cdef.footText = NULL ; }
               break ;
            case L't':
               if ( cdef.title == NULL )
               {
                  if ( cdef.dPtr == NULL ) cdef.title = ChildDlgTitle ;
                  else                     cdef.title = ParentDlgTitle ;
               }
               else
                  cdef.title = NULL ;
               break ;
            case L'c': customShift = customShift ? false : true ;    break ;
            case L'a': audibleShift = audibleShift ? false : true ;  break ;
            case L'n':
               if ( cdef.dataPtr == Data1 )
               { cdef.dataPtr = Data2 ; cdef.dataCount = Data2Count ; }
               else
               { cdef.dataPtr = Data1 ; cdef.dataCount = Data1Count ; }
               break ;
            case L'N':
               cdef.negColor = (cdef.negColor == cdef.barColor) ? 
                                                  this->cfg.sb : cdef.barColor ;
               break ;
            case L'i':
               if ( cdef.dPtr == NULL )   // point to application dialog window
               {
                  cdef.dPtr = this->statsDlg ;
                  cdef.ulY = 2 ;                    // position inside dialog window
                  cdef.ulX = 1 ;
                  cdef.rows = 20 ;                  // dimensions inside dialog window
                  cdef.cols = sdInit.dColumns - 2 ;
                  cdef.yFoot = 3 ;                  // free space below grid
                  horizLine = false ;               // disable horizontal line
                  if ( cdef.title != NULL )         // if dialog title is active
                     cdef.title = ParentDlgTitle ;  //  be sure it's the right title
               }
               else                       // create custom dialog window
               {
                  cdef.dPtr = NULL ;
                  cdef.ulY = sdInit.dYoffset ;      // at application dialog origin 
                  cdef.ulX = sdInit.dXoffset ;
                  cdef.rows = sdInit.dLines + 2 ;   // beyond application dialog window
                  cdef.cols = sdInit.dColumns + 8 ; // beyond application dialog window
                  cdef.yFoot = 14 ;                 // free space below grid
                  horizLine = true ;                // enable horizontal line
                  if ( cdef.title != NULL )         // if dialog title is active
                     cdef.title = ChildDlgTitle ;   //  be sure it's the right title
               }
               break ;
            case L'v': cdef.horizBars = false ;    break ;
            case L'z': cdef.horizBars = true ;     break ;
            case L'm': cdef.attrPtr = cdef.attrPtr == NULL ? Attr1 : NULL ; break ;
            case L'T': cdef.barTips = cdef.barTips ? false : true ;  break ;
            case L'w': 
               if ( cdef.barWidth == cellDIV )
                  cdef.barWidth = (cellDIV / 2) ;
               else
                  cdef.barWidth = cellDIV ;
               break ;
            case L'W': cdef.barSpace = (cdef.barSpace == ZERO) ? 1 : ZERO ; break ;
            case L'C': altcartChar = altcartChar ? false : true ;           break ;
            case L'q':
            case L'Q':
               done = true ;
               break ;
            case nckSPACE:
               this->statsDlg->ClearWin () ;       // clear the target window
               if ( cp != NULL )                   // if existing instance, delete it
               { delete cp ; cp = NULL ; }
               #if 1    // Automatic refresh
               cp = new Chart( cdef, true ) ;
               #else    // Delayed refresh
               cp = new Chart( cdef ) ;
               this->statsDlg->UserAlert ( 2 ) ;
               nckPause();
               cp->refresh() ;
               #endif   // Refresh testing

               if ( ! audibleShift )               // silence warning beep
                  cp->AudibleShift( false ) ;
               if ( horizLine )                    // split area with horizontal line
                  cp->DrawHorizontalLine ( (cdef.yFoot > 3 ? 2 : 0), 
                                           cdef.borderStyle, 
                                           cdef.borderColor, true ) ;
               if ( altcartChar )                  // alternate Cartesian character
                  cp->SetCartesianChar ( L'▪' ) ;
               //* Apply a secondary dataset (for Cartesian charts only) *
               if ( cartOverlay && (cdef.chType == ctCartesian) )
               {
                  cp->OverlayCartesianDataset ( CartOverlayData, CartOverlayCount,
                                                this->cfg.sb ) ;
               }
               if ( cVersion.gschars() == 1 )      // get class version
                  cVersion.compose( "Chart v:%s", cp->GetVersion() ) ;

               //* Display the shift-keys prompt in the target dialog window.*
               cp->GetFooterPosition ( wpFooter, ftrRows, ftrCols ) ;
               wpFooter.ypos += wpShiftInstr.ypos + (cdef.dPtr == NULL ? 1 : 0) 
                                + ((cdef.border && cdef.dPtr != NULL) ? 1 : 0) ;
               wpFooter.xpos += wpShiftInstr.xpos ;
               cp->DrawExtendedFooter ( wpFooter, 
                  (customShift ? ShiftInstrCust : ShiftInstrDflt), this->cfg.sb, true ) ;

               //* Default Chart-class user interface key definitions *
               if ( ! customShift )
               {
                  cp->ShiftData ( wk ) ;
                  //* If all data currently displayed, just pause.*
                  if ( wk.key == nckNULLCHAR )
                     nckPause();
               }     // default definitions
               else     // Custom user interface key definitions
               {
                  const short sDEFS = 10 ;
                  ShiftDef sdef[sDEFS] =
                  { // wk.key wk.type    sb           sc
                     { {L'f', wktPRINT}, sbFirstPage, ZERO },
                     { {L'l', wktPRINT}, sbLastPage,  ZERO  },
                     { {L'n', wktPRINT}, sbNextPage,  ZERO  },
                     { {L'p', wktPRINT}, sbPrevPage,  ZERO  },
                     { {L'a', wktPRINT}, sbNoShift,      1  },
                     { {L'A', wktPRINT}, sbNoShift,     -1  },
                     { {L'b', wktPRINT}, sbNoShift,      5  },
                     { {L'B', wktPRINT}, sbNoShift,     -5  },
                     { {L'c', wktPRINT}, sbNoShift,     10  },
                     { {L'C', wktPRINT}, sbNoShift,    -10  },
                  } ;
                  cp->ShiftData ( wk, sDEFS, sdef ) ;
                  //* If all data currently displayed, just pause.*
                  if ( wk.key == nckNULLCHAR )
                     nckPause();
               }     // custom definitions

               delete cp ;
               cp = NULL ;

               //* If parent dialog was saved, restore it now *
               if ( cdef.dPtr == NULL )
                  this->statsDlg->RefreshWin () ;
               break ;
            default:
               this->statsDlg->UserAlert () ;
               break ;
         } ;
      }        // (wk.type == wktPRINT)
      else if ( wk.type == wktEXTEND )    // ALT+keycode
      {
         switch ( wk.key )
         {
            //* Input Data Formats *
            case L'a':     // Double (default)
               cdef.dataType  = idtDouble ;
               cdef.dataPtr   = Data1 ;
               cdef.dataCount = Data1Count ;
               break ;
            case L'A':     // Float
               cdef.dataType  = idtFloat ;
               cdef.dataPtr   = FloatData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'b':     // Byte (signed)
               cdef.dataType  = idtByte_s ;
               cdef.dataPtr   = sByteData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'B':     // Byte (unsigned)
               cdef.dataType  = idtByte_u ;
               cdef.dataPtr   = uByteData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'c':     // Short (signed)
               cdef.dataType  = idtShort_s ;
               cdef.dataPtr   = sShortData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'C':     // Short (unsigned)
               cdef.dataType  = idtShort_u ;
               cdef.dataPtr   = uShortData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'd':     // Integer (signed)
               cdef.dataType  = idtInt_s ;
               cdef.dataPtr   = sIntData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'D':     // Integer (unsigned)
               cdef.dataType  = idtInt_u ;
               cdef.dataPtr   = uIntData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'e':     // Long Int (signed)
               cdef.dataType  = idtLong_s ;
               cdef.dataPtr   = sLongData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'E':     // Long Int (unsigned)
               cdef.dataType  = idtLong_u ;
               cdef.dataPtr   = uLongData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'f':     // Long Long Int (signed)
               cdef.dataType  = idtLongLong_s ;
               cdef.dataPtr   = sLongLongData ;
               cdef.dataCount = IntCount ;
               break ;
            case L'F':     // Long Long Int (unsigned)
               cdef.dataType  = idtLongLong_u ;
               cdef.dataPtr   = uLongLongData ;
               cdef.dataCount = IntCount ;
               break ;

            case L'g':     // overlay Cartesian chart with addition dataset
               cartOverlay = cartOverlay ? false : true ;
               break ;
            case L'G':     // test the grid using alternate dimensions
               if ( altDimensions )    // return to standard dimensions
               {
                  ++cdef.rows ;
                  ++cdef.cols ;
                  altDimensions = false ;
               }
               else                    // set alternate dimensions
               {
                  --cdef.rows ;
                  --cdef.cols ;
                  altDimensions = true ;
               }
               break ;
         } ;
      }        // (wk.type == wktEXTEND)
      else     // any other keycode
         done = true ;
   }

   if ( cp != NULL )             // call destructor for Chart object
      delete cp ;

   this->DisplayMETsTable () ;   // restore the target window

}  //* End DebugChartWidget() *
#endif   // DEBUG_MENU

//*************************
//*  GetCommandLineArgs   *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//* Valid Arguments: see DisplayHelp() method.                                 *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: 'false' if normal startup                                         *
//*          'true'  if request for help or invalid command-line argument      *
//******************************************************************************

bool Exercalc::GetCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCA (0)

   const char EQUAL = '=' ;

   //* Get the application executable's directory path.        *
   //* Save the actual, absolute path specification, replacing *
   //* symbolic links and relative path specifications.        *
   if ( (realpath ( ca.argList[0], ca.appPath )) != NULL )
   {
      gString gs( ca.appPath ) ;
      gs.limitChars( (gs.findlast( L'/' )) ) ;
      gs.copy( ca.appPath, MAX_PATH ) ;
   }

   //* If user provided command-line arguments *
   if ( ca.argCount > 1 )
   {
      gString gs ;               // text formatting
      const char* argPtr ;       // pointer to option argument
      float fIn ;                // temp parameter value
      short j = ZERO ;           // for multiple arguments in same token
      char subchar,              // sub-option
           indx ;                // index into parameter string
      bool multiarg = false ;    // if concatenated arguments

      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            multiarg = false ;
            ++j ;

            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               gs = ca.argList[i] ;
               indx = gs.after( wEQUAL ) ;   // index the argument

               //* User's height *
               if ( (gs.find( L"--height" )) == ZERO )
               {
                  if ( indx > ZERO )
                     gs.shiftChars( -(indx) ) ;
                  else
                     gs = ca.argList[++i] ;
                  if ( !(this->DecodeLength2Meters ( gs, ca.heightM )) )
                     ca.helpFlag = true ;
               }

               //* User's mass *
               else if ( (gs.find( L"--mass" )) == ZERO )
               {
                  if ( indx > ZERO )
                     gs.shiftChars( -(indx) ) ;
                  else
                     gs = ca.argList[++i] ;
                  if ( !(this->DecodeMass2Kilograms ( gs, ca.massKg )) )
                     ca.helpFlag = true ;
               }

               //* User's age *
               else if ( (gs.find( L"--age" )) == ZERO )
               {
                  if ( indx > ZERO )
                     gs.shiftChars( -(indx) ) ;
                  else
                     gs = ca.argList[++i] ;
                  if ( (ca.ageY = this->Time2Minutes ( gs.ustr() )) <= 0.0 )
                  {
                     ca.ageY = 0.0 ;
                     ca.helpFlag = true ;
                  }
               }

               //* User's gender *
               else if ( (gs.find( L"--gender" )) == ZERO )
               {
                  if ( indx > ZERO )
                     gs.shiftChars( -(indx) ) ;
                  else
                     gs = ca.argList[++i] ;
                  subchar = tolower ( gs.gstr()[0] ) ;
                  switch ( subchar )
                  {
                     case 'm':   ca.gender = goMale ;    break ;
                     case 'f':   ca.gender = goFemale ;  break ;
                     case 'o':   ca.gender = goOther ;   break ;
                     default:    ca.gender = goNone ; ca.helpFlag = true ; break ;
                  }
               }

               //* Specify log-file path
               else if ( (gs.find( L"--cfg" )) == ZERO )
               {
                  if ( indx > ZERO )
                     gs.shiftChars( -(indx) ) ;
                  else
                     gs = ca.argList[++i] ;
                  gs.copy( ca.cfgPath, MAX_PATH ) ;
                  // Programmer's Note: Caller must perform filename validation.
               }

               //* Specify log-file path
               else if ( (gs.find( L"--log" )) == ZERO )
               {
                  if ( indx > ZERO )
                     gs.shiftChars( -(indx) ) ;
                  else
                     gs = ca.argList[++i] ;
                  gs.copy( ca.logPath, MAX_PATH ) ;
                  // Programmer's Note: Caller must perform filename validation.
               }

               #if DEBUG_RTL != 0
               //* For debugging only: simulate RTL data *
               // Programmer's Note: RTL debugging code is optimised ONLY for English.
               // Other languages may show some text positions incorrectly.
               else if ( (gs.find( L"--rtl" )) == ZERO )
                  this->cfg.rtl = true ;
               #endif   // DEBUG_RTL

               //* Request for application version number *
               else if ( (gs.compare( L"--version" )) == ZERO )
                  ca.helpFlag = ca.verFlag = true ;

               //* Long-form request for Help *
               else if ( (gs.compare( L"--help" )) == ZERO )
                  ca.helpFlag = true ;

               else  // invalid argument
               { ca.helpFlag = true ; break ; }

               continue ;     // finished with this argument
            }  // (double dash)

            //** Short-form Arguments **
            else
            {
               char argLetter = ca.argList[i][j] ;

               //* Input values *
               if ( argLetter == 'i' || argLetter == 'I' )
               {
                  subchar = tolower ( ca.argList[i][++j] ) ;
                  switch ( subchar )
                  {
                     case 'm':   ca.inpUnits = vuMiles ;     break ;
                     case 'k':   ca.inpUnits = vuKmeters ;   break ;
                     case 'h':   ca.inpUnits = vuMinutes ;   break ;
                     case 'c':   ca.inpUnits = vuKcalories ; break ;
                     default:
                        ca.inpUnits = vuKmeters ;
                        ca.helpFlag = true ;
                        break ;
                  }

                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument

                  gs = argPtr ;
                  gs.strip() ;
                  ca.inpValue = this->DecodeInputValues ( gs, ca.inpUnits ) ;

                  if ( ca.inpValue < 0.0 )
                  { ca.helpFlag = true ; break ; }

                  continue ;     // finished with this argument
               }

               //* Type of exercise *
               else if ( argLetter == 't' || argLetter == 'T' )
               {
                  subchar = tolower ( ca.argList[i][++j] ) ;
                  switch ( subchar )
                  {
                     case 'w':   ca.exerType = xtWalk ;     break ;
                     case 'r':   ca.exerType = xtRun ;      break ;
                     case 'b':   ca.exerType = xtBike ;     break ;
                     case 'g':   ca.exerType = xtGeneral ;  break ;
                     default:    ca.helpFlag = true ;       break ;
                  }
               }

               //* Velocity (for walk/run/bike) *
               else if ( argLetter == 'v' || argLetter == 'V' )
               {
                  subchar = tolower ( ca.argList[i][++j] ) ;
                  switch ( subchar )      // validate the exercise type
                  {
                     case 'w':
                     case 'r':
                     case 'b':
                        break ;
                     default: ca.helpFlag = true ; break ;
                  }

                  //* Get the numeric argument for velocity *
                  if ( ! ca.helpFlag )
                  {
                     if ( ca.argList[i][++j] == EQUAL )
                        argPtr = &ca.argList[i][++j] ;
                     else if ( i < (ca.argCount - 1) )
                        argPtr = ca.argList[++i] ;
                     else
                     { ca.helpFlag = true ; break ; }    // invalid syntax for argument

                     gs = argPtr ;
                     gs.strip() ;
                     if ( (this->DecodeVelocity2Kph ( gs, fIn )) )
                     {
                        switch ( subchar )
                        {
                         case L'w': ca.kphWalk = fIn ; break ;
                         case L'b': ca.kphBike = fIn ; break ;
                         case L'r': ca.kphRun  = fIn ; break ;
                        } ;
                     }
                     else        // invalid syntax for argument
                        ca.helpFlag = true ;
                  }
                  else           // invalid syntax for sub-option
                     break ;

                  if ( fIn < 0.0 )  // time machine module not installed
                  { ca.helpFlag = true ; break ; }

                  continue ;     // finished with this argument
               }

               //* Display Exercise Goal (-g[d|w] option)
               else if ( argLetter == 'g' || argLetter == 'G' )
               {
                  subchar = tolower ( ca.argList[i][++j] ) ;
                  if ( subchar == 'd' )      ca.showGoal = sgDaily ;  // daily goal
                  else if ( subchar == 'w' ) ca.showGoal = sgWeekly ; // weekly goal
                  else                       // invalid syntax for argument
                  { ca.helpFlag = true ; break ; }
               }

               //* Application Locale (uppercase 'L' only) *
               else if ( argLetter == 'L' )
               {
                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
                  gs = argPtr ;
                  gs.strip() ;
                  gs.copy( ca.altLocale, MAX_FNAME ) ;
                  continue ;     // finished with this argument
               }

               //* Application Language (lowercase 'l' only) *
               else if ( argLetter == 'l' )
               {
                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
                  gs = argPtr ;
                  gs.strip() ;
                  if ( (gs.compare( L"en", false, 2 )) == ZERO )
                     ca.appLanguage = enLang ;
                  else if ( (gs.compare( L"es", false, 2 )) == ZERO )
                     ca.appLanguage = esLang ;
                  else if ( (gs.compare( L"zh", false, 2 )) == ZERO )
                     ca.appLanguage = zhLang ;
                  else if ( (gs.compare( L"vi", false, 2 )) == ZERO )
                     ca.appLanguage = viLang ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
                  continue ;     // finished with this argument
               }

               //* Color Scheme (uppercase 'C' only) *
               else if ( argLetter == 'C' )
               {
                  if ( ca.argList[i][++j] == EQUAL )
                     argPtr = &ca.argList[i][++j] ;
                  else if ( i < (ca.argCount - 1) )
                     argPtr = ca.argList[++i] ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
                  gs = argPtr ;
                  gs.strip() ;
                  // Programmer's Note: We need only the first three(3) characters
                  // to identify the color. (note alternate spelling: grey/gray)
                  if ( (gs.find( "bla" )) == ZERO )         ca.scheme = ncbcBK ;
                  else if ( (gs.find( "red" )) == ZERO )    ca.scheme = ncbcRE ;
                  else if ( (gs.find( "gree" )) == ZERO )   ca.scheme = ncbcGR ;
                  else if ( (gs.find( "bro" )) == ZERO )    ca.scheme = ncbcBR ;
                  else if ( (gs.find( "blu" )) == ZERO )    ca.scheme = ncbcBL ;
                  else if ( (gs.find( "mag" )) == ZERO )    ca.scheme = ncbcMA ;
                  else if ( (gs.find( "cya" )) == ZERO )    ca.scheme = ncbcCY ;
                  else if ( ((gs.find( "gray" )) == ZERO) ||
                            ((gs.find( "grey" )) == ZERO) ) ca.scheme = ncbcGY ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
                  //* Test for "reverse" indicator *
                  if ( (gs.find( ",r" )) > ZERO )
                     ca.rscheme = true ;
                  continue ;     // finished with this argument
               }

               //* Automatic log-file save *
               else if ( argLetter == 'a' || argLetter == 'A' )
               {
                  subchar = tolower ( ca.argList[i][++j] ) ;
                  if ( subchar == 'e' )         ca.logAuto = true ;
                  else if ( subchar == 'd' )    ca.logAuto = false ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               }

               //* Mouse Support *
               else if ( argLetter == 'm' || argLetter == 'M' )
               {
                  subchar = tolower ( ca.argList[i][++j] ) ;
                  if ( subchar == 'e' )         ca.mouse = true ;
                  else if ( subchar == 'd' )    ca.mouse = false ;
                  else
                  { ca.helpFlag = true ; break ; }    // invalid syntax for argument
               }

               //* Pause after start-up sequence so user can see diagnostics *
               else if ( argLetter == 'p' || argLetter == 'P' )
               {
                  subchar = tolower ( ca.argList[i][j + 1] ) ;
                  if ( subchar == 'v' )
                  {
                     ca.diagPause = 2 ;
                     ++j ;
                  }
                  else
                     ca.diagPause = 1 ;
               }

               //* Else, is either 'h', 'H' or an invalid argument.*
               //* Either way, invoke help.                        *
               else
               {
                  ca.helpFlag = true ;
                  if ( argLetter != 'h' && argLetter != 'H' )
                     break ;
               }
            }

            //* If separate tokens have been concatenated *
            if ( ca.argList[i][j + 1] != nckNULLCHAR )
               multiarg = true ;
         }
         else  // invalid argument, token does not begin with a DASH character
         { ca.helpFlag = true ; break ; }

      }     // for(;;)

      #if DEBUG_GCA != 0  // debugging only
      gs.compose( L"%hd", &ca.argCount ) ;
      wcout << L"argCount   : " << gs.gstr() << endl ;
      for ( short i = ZERO ; i < ca.argCount ; i++ )
      {
         gs.compose( L"argList[%hd] : '%s'", &i, ca.argList[i] ) ;
         wcout << gs.gstr() << endl ;
      }
      gs.compose( L"appPath    : '%s'", ca.appPath ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"cfgPath    : '%s'", ca.cfgPath ) ;
      wcout << gs.gstr() << endl ;
      gs.compose( L"inpValue   : %-7.03f\n"
                   "inpUnits   : %s\n"
                   "exerType   : %s\n"
                   "heightM    : %-7.3f\n"
                   "massKg     : %-7.3f\n"
                   "ageY       : %-5.1f\n"
                   "gender     : %s\n"
                   "kphBike    : %-7.3f\n"
                   "kphWalk    : %-7.3f\n"
                   "kphRun     : %-7.3f\n"
                   "showGoal   : %s\n"
                   "logPath    : %s\n"
                   "logAuto    : %s\n"
                   "altLocale  : %s\n"
                   "appLanguage: %s\n"
                   "colorScheme: %s%s\n"
                   "mouse      : %s\n"
                   "diagPause  : %hd\n"
                   "verFlag    : %hhd\n"
                   "helpFlag   : %hhd\n\n"
                   " Press Enter", 
                  &ca.inpValue,
                  (ca.inpUnits == vuMiles ? "vuMiles" :
                   ca.inpUnits == vuKmeters ? "vuKmeters" :
                   ca.inpUnits == vuMinutes ? "vuMinutes" :
                   ca.inpUnits == vuKcalories ? "vuKcalories" : "vuNone"),
                  (ca.exerType == xtWalk ? "xtWalk" :
                   ca.exerType == xtRun ? "xtRun" :
                   ca.exerType == xtBike ? "xtBike" :
                   ca.exerType == xtGeneral ? "xtGeneral" : "xtNone"),
                  &ca.heightM, &ca.massKg, &ca.ageY,
                  (ca.gender == goMale ? "goMale" : 
                   ca.gender == goFemale ? "goFemale" :
                   ca.gender == goOther ? "goOther" : "goNone"),
                  &ca.kphBike, &ca.kphWalk, &ca.kphRun,
                  (ca.showGoal == sgNoshow ? "sgNoshow" :
                   ca.showGoal == sgWeekly ? "sgWeekly" : "sgDaily"),
                  ca.logPath,
                  (ca.logAuto ? "enable" : "disable"),
                  ca.altLocale,
                  (ca.appLanguage == enLang ? "English" :
                   ca.appLanguage == esLang ? "Espanol" :
                   ca.appLanguage == zhLang ? "Zhongwen" :
                   ca.appLanguage == viLang ? "Tieng Viet" :
                   ca.appLanguage == locLang ? "locale" : "unknown"),
                   // Programmer's Note: Compiler error (g++ v:10.2.1):
                   // Cannot read non-ASCII strings in trinary arguments.
                  (ca.scheme == ncbcBK ? "black" :
                   ca.scheme == ncbcRE ? "red" :
                   ca.scheme == ncbcGR ? "green" :
                   ca.scheme == ncbcBR ? "brown" :
                   ca.scheme == ncbcBL ? "blue" :
                   ca.scheme == ncbcMA ? "magenta" :
                   ca.scheme == ncbcCY ? "cyan" : 
                   ca.scheme == ncbcGY ? "gray" : "default"),
                  (ca.rscheme ? ",r" : " "),
                  (ca.mouse ? "enable" : "disable"), 
                  &ca.diagPause, &ca.verFlag, &ca.helpFlag ) ;
      wcout << gs.gstr() << endl ;
      getchar () ;
      #endif   // DEBUG_GCA
   }
   return ca.helpFlag ;

   #undef DEBUG_GCA
}  //* End GetCommandLineArgs() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

