//******************************************************************************
//* File       : TrshFile.hpp                                                  *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2015-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 17-Jan-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module contains methods for accessing files to be        *
//* processed, extracting information about each item and creating the         *
//* modified output.                                                           *
//*                                                                            *
//* These methods are adapted from a small subset of the FMgr class written    *
//* for the FileMangler file-management project by the same author.            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* =================                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include <fstream>         // for file i/o
#include <sys/stat.h>      // for 'stat' info on target files
#include <dirent.h>        // for reading contents of 'directory type' files
#include <wordexp.h>       // for expansion of environment variables
#include <utime.h>         // for setting file timestamps

//***************
//* Definitions *
//***************

//** Handy data types **
typedef unsigned int       UINT ;
typedef unsigned long int  ULONG ;
//typedef unsigned _int64    UINT64 ;     // Microsloth
typedef unsigned long long int  UINT64 ;// Linux
typedef unsigned short int USHORT ;
typedef unsigned char      UCHAR ;

//** Definitions **
#define ZERO         0
#define SPACE        (' ')
#define DASH         ('-')
#define COMMA        (',')
#define PERIOD       ('.')
#define COLON        (':')
#define TILDE        ('~')
#define UNDERSCORE   ('_')
#define CARET        ('^')
#define TAB          ('\t')
#define NEWLINE      ('\n')
#define NULLCHAR     ('\0')
#define ESC          (0x1B)
#define BELL         ('\x07')    // noise-maker character

const short OK  = ZERO ;
const short ERR = (-1) ;

const short MAX_PATH  = PATH_MAX ; //* Size of buffer to hold path/filename
const short MAX_FNAME = 256 ;      //* Size of buffer to hold a filename
const UINT64 FULL64 = 0xFFFFFFFFFFFFFFFF ; // max value for UINT64

//* Define the C structures as something C++ *
typedef struct dirent64 deStats ;
typedef struct stat64 FileStats ;
typedef struct tm Tm ;
typedef struct utimbuf uTime ;

//* File type encoded in the "st_mode" element of the stat{} structure *
enum fmFType : short 
{ fmDIR_TYPE,   fmREG_TYPE,  fmLINK_TYPE, fmCHDEV_TYPE, 
  fmBKDEV_TYPE, fmFIFO_TYPE, fmSOCK_TYPE, fmUNKNOWN_TYPE, fmTYPES } ;

//* For trashcan or other directories, these are the permission *
//* bits that must be set to ensure user read/write access.     *
const mode_t dirMode = (S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) ;

//* Class for reporting the system local date/time.                            *
//* ---------------------------------------------------------------------------*
const int64_t maxETIME32 = 0x07FFFFFFF,      // largest 32-bit time_t
              minETIME32 = -maxETIME32 - 1 ; // smallest 32-bit time_t
const short ltFMT_LEN = 16 ;                 // length of time string buffers
//*                                                                            *
//* Important note about the year 2038 bug:                                    *
//* Because 'time_t' is defined as a signed, 32-bit value on 32-bit systems,   *
//* time reporting will break down for date/time values                        *
//*                     >= 19 Jan 2038 at 11:14:08                             *
//* For this reason and for others, we have defined our own local-time class.  *
//* This class uses a signed, 64-bit value for storing the epoch time.         *
//* It is hoped that all Linux systems will have been updated long before this *
//* becomes an issue; however, even now (2013), some applications are using    *
//* date/time manipulations that exceed the 32-bit limit. BEWARE!!             *
//* NOTE: time arithmetic should always be performed on signed 64-bit values.  *
//*                                                                            *
class localTime
{
   public:
   localTime ( void ) { this->reset() ; }
   void reset ( void )
   {  //* Set to epoch date: Thursday, 01 Jan. 1970 @ 08:00:00
      this->sysepoch = ZERO ;
      this->epoch = ZERO ;
      this->day = 5 ;
      this->year = 1970 ;
      this->month = this->date = 1 ;
      this->hours = 8 ;
      this->minutes = this->seconds = ZERO ;
      #if 0    // not currently used
      *this->timezone = NULLCHAR ;
      *this->utc_zone  = NULLCHAR ;
      this->gmtoffset = ZERO ;
      this->dst       = false ;
      #endif   // not currently used
   }
   int64_t  epoch ;        //* Seconds since the epoch (01-Jan-1970)
   time_t   sysepoch ;     //* system epoch time (see note)
   USHORT   day ;          //* Day of the week (1 == Sun, ..., 7 == Sat)
   USHORT   date ;         //* Day of the month (1-[28|29|30|31])
   USHORT   month ;        //* Month of the year (1 == Jan, ... 12 == Dec)
   USHORT   year ;         //* Year (four digit year)
   USHORT   hours ;        //* Time of day, hour (0-23)
   USHORT   minutes ;      //* Time of day, minutes (0-59)
   USHORT   seconds ;      //* Time of day, seconds (0-59)

   #if 0    // additional fields not currently used
   // See ctfDecodeTimestamp method for more information.
   char     timezone[ltFMT_LEN] ; //* string representing the local time zone (useless)
   char     utc_zone[ltFMT_LEN] ; //* string representing the UTC offset for local time zone
   short    gmtoffset ;    //* Expressed as (signed) number of seconds offset from UTC+0:00
   bool     dst ;          //* 'true' if daylight savings time in effect,
                           //* 'false if not DST or info not available
   #endif   // not currently used
} ;

//* Class containing decoded file statistics *
class DFileStats
{
   public:
   DFileStats ( void )
   {
      this->reset() ;
   }
   void reset ( void )
   {
      *this->fName = NULLCHAR ;
      this->fType  = fmUNKNOWN_TYPE ;
      this->fBytes = ZERO ;
      this->modTime.reset() ;
      this->rAccess = this->wAccess = this->xAccess = false ;
   }
   char       fName[MAX_FNAME] ; // filename string (UTF-8)
   fmFType    fType ;            // file type (enum fmFType)
   UINT64     fBytes ;           // file size (in bytes)
   localTime  modTime ;          // Date/Time file last modified (human readable)
   FileStats  rawStats ;         // Copy of system's raw FileStats data
   bool       rAccess ;          // 'true' if user has read access
   bool       wAccess ;          // 'true' if user has write access 
   bool       xAccess ;          // 'true' if user has execute access
} ;

#if 0    // NOT CURRETLY USED IN THIS APPLICATION
//* Class definition describing a file. *
class tnFName
{
   public:
   tnFName ( void )           //* Constructor
   {
      this->ReInit () ;       // initialize data members
   }
   void ReInit(void)
   {
      fName[0] = NULLCHAR ;
      fType    = fmTYPES ;
      fBytes   = ZERO ;
      modTime.day = modTime.date = modTime.month = modTime.year 
                  = modTime.hours = modTime.minutes = modTime.seconds = ZERO ;
      readAcc  = false ;
      writeAcc = false ;
      // NOTE: data member, rawStats is left unitialized
   }
   char        fName[MAX_FNAME] ; //* Filename string
   fmFType     fType ;     //* File type (enum fmFType)
   UINT64      fBytes ;    //* File size (in bytes)
   localTime   modTime ;   //* Date/Time file last modified (human readable)
   FileStats   rawStats ;  //* Copy of system's "stat" structure (all file info)
   bool        readAcc ;   //* 'true' if user has read access to file
   bool        writeAcc ;  //* 'true' if user has write access to file 
                              // writeAcc is initially set as false when record
                              // is read. Must explicitly be calculated if needed.
} ;
#endif   // NOT CURRETLY USED IN THIS APPLICATION

//* Used as the parameter for FileSystemStats() method to contain information  *
//* about the filesystem. See below.                                           *
class fileSystemStats
{
   public:
   fileSystemStats()
   {
      seLinuxCode[0] = fsType[0] = 'u' ;
      seLinuxCode[1] = fsType[1] = 'n' ;
      seLinuxCode[2] = fsType[2] = 'k' ;
      seLinuxCode[3] = fsType[3] = NULLCHAR ;
      freeBytes = usedBytes = systemID = ZERO ;
      blockTotal = blockFree = blockAvail = inodeTotal = inodeAvail = ZERO ;
      fsTypeCode = ZERO ;
      nameLen = blockSize = fblockSize = ZERO ;
   }                          // 'stat' format code
   char     seLinuxCode[64] ; // %C
   char     fsType[64] ;      // %T
   UINT64   systemID ;        // %i (hex)
   UINT64   freeBytes ;       // (calculated: blockAvail * fblockSize)
   UINT64   usedBytes ;       // (calculated: (blockTotal - blockFree) * fblockSize)
   ULONG    blockTotal ;      // %b
   ULONG    blockFree ;       // %f
   ULONG    blockAvail ;      // %a
   ULONG    inodeTotal ;      // %c
   ULONG    inodeAvail ;      // %d
   UINT     fsTypeCode ;      // %t (hex)
   USHORT   nameLen ;         // %l
   USHORT   blockSize ;       // %s
   USHORT   fblockSize ;      // %S
} ;


