//******************************************************************************
//* File       : cTrashCan.cpp                                                 *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2015-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 17-Jan-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description:                                                               *
//* This module contains methods related to management of data in the          *
//* 'Trashcan':  a) send files or directory trees to the Trashcan              *
//*              b) restore files or directory trees from the Trashcan         *
//*              c) empty the Trashcan (delete Trashcan contents)              *
//*              d) report current contents of Trashcan                        *
//*                                                                            *
//* Copyright Notice:                                                          *
//* =================                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History: See cTrash.cpp.                                           *
//*                                                                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//******************************************************************************
//* Using the trashcan for GNOME, KDE or any trashcan (loosly) based on the    *
//* freedesktop.org trashcan specification.                                    *
//*                                                                            *
//* The trashcan for the desktop environment is located at                     *
//*  ~/.local/share/Trash                                                      *
//* and has three subdirectories                                               *
//*   files    (deleted files)                                                 *
//*   info     (info about deleted files)                                      *
//*   expunged (not used by FileMangler)                                       *
//* Older versions of the desktop may have different locations for the trashcan*
//* for example:  ~/.Trash                                                     *
//* Also, there are system-level directories for trash, created by Admin,      *
//* but we aren't interested in those -- only the user-local trash.            *
//*                                                                            *
//* The specification says that if the trashcan subdirectories do not exist,   *
//* they should be automagically created without delay. This is done at a      *
//* higher (application) level.                                                *
//*                                                                            *
//* When a file is moved to the trashcan, two things happen                    *
//* 1) The file is moved to ~/.local/share/Trash/files subdirectory.           *
//*    - Note that if a file of that name already exists, then the target is   *
//*      renamed to avoid overwriting an existing file.                        *
//*      - For Nautilus, the new name will take the format originalname.2.ext  *
//*        so we follow that format.                                           *
//*                                                                            *
//* 2) A file with the extension of '.trashinfo' is created in the             *
//*    ~/.local/share/Trash/info subdirectory with a base name corresponding   *
//*    to the name of the file in the 'files' subdirectory.                    *
//*    - This file contains three entries:                                     *
//*      a) a header line                                                      *
//*      b) the full path/filename of the source                               *
//*      c) a date/timestamp for when the file was moved to the trash          *
//*    - Note that for subdirectory trees, a '.trashinfo' file is created ONLY *
//*      for the top-level directory name. Contents of the subdirectory are    *
//*      simply moved to the trashcan, and no associated '.trashinfo' file     *
//*      is created for them.                                                  *
//*                                                                            *
//* Example:                                                                   *
//*  - Use Nautilus to move the file 'cTrash.o' to the trashcan.               *
//*    Two files appear:                                                       *
//*      ~/.local/share/Trash/files/cTrash.o                                   *
//*      ~/.local/share/Trash/info/cTrash.o.trashinfo                          *
//*    The first is the original file.                                         *
//*    The second is information about that file i.e.                          *
//*      [Trash Info]                                                          *
//*      Path=/home/sam/Software/cTrash/cTrash.o                               *
//*      DeletionDate=2015-08-20T10:29:43                                      *
//*                                                                            *
//*  - Recompile the source file to create another copy of cTrash.o            *
//*    then again:                                                             *
//*  - Use Nautilus to move the file 'cTrash.o' to the trashcan.               *
//*    Two additional files appear:                                            *
//*      ~/.local/share/Trash/files/cTrash.2.o                                 *
//*      ~/.local/share/Trash/info/cTrash.2.o.trashinfo                        *
//*    The first is the original file.                                         *
//*    The second is information about that file i.e.                          *
//*      [Trash Info]                                                          *
//*      Path=/home/sam/Software/cTrash/cTrash.o                               *
//*      DeletionDate=2015-08-20T10:30:36                                      *
//*                                                                            *
//* Example:                                                                   *
//*  - Use Nautilus to move the file 'ctrash' to the trashcan.                 *
//*    (Note tha 'ctrash' has no filename extension.)                          *
//*    1st)                                                                    *
//*      ~/.local/share/Trash/files/ctrash                                     *
//*      ~/.local/share/Trash/info/ctrash.trashinfo                            *
//*    2nd)                                                                    *
//*      ~/.local/share/Trash/files/ctrash.2                                   *
//*      ~/.local/share/Trash/info/ctrash.2.trashinfo                          *
//*    3nd)                                                                    *
//*      ~/.local/share/Trash/files/ctrash.3                                   *
//*      ~/.local/share/Trash/info/ctrash.3.trashinfo                          *
//*    Note that for files with no extension:                                  *
//*      second occurrance: '.2' is appended to the item                       *
//*      third occurrance : '.2' is REPLACED by '.3'                           *
//*                                                                            *
//* See the cttNextTargetName method for our implementation.                                                                           *
//*                                                                            *
//*                                                                            *
//* Note: If source file is a Symbolic Link, the link, NOT link target is      *
//*       moved to the Trashcan                                                *
//* Note: Trashcan operations for certain 'special' files, Character Device,   *
//*       Block Device, Socket, or others MAY not be supported.                *
//*       Copy or delete of these files may require super-user privlege OR     *
//*       these operations may not be possible under normal circumstances.     *
//*       For these, the user will be alerted if the operation is not          *
//*       supported or not permitted.                                          *
//*                                                                            *
//* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *  *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "cTrash.hpp"      //* General definitions

//****************
//* Local Data   *
//****************
static const char* const trashinfoHdr  = "[Trash Info]" ;
static const char* const trashinfoPath = "Path=" ;
static const char* const trashinfoDate = "DeletionDate=" ;
static const wchar_t* const trashinfoExt  = L".trashinfo" ;
static const wchar_t* const dirtagTemplate = L"[DIR: %u files] " ;
static const wchar_t* const statSerial   = L"%3u) " ;
static const wchar_t* const statTemplate = L"%S\n"
                                            "     %S%S  -- (%S bytes)" ;

//***********************
//* cttSendItemToTrash  *
//***********************
//******************************************************************************
//* Move the specified item to the trashcan.                                   *
//*                                                                            *
//* -- If item moved successfully, the source item is deleted.                 *
//* -- If there is an error, source item is not modified, and any              *
//*    intermediate files written to the trashcan are removed.                 *
//*                                                                            *
//*                                                                            *
//* Input  : fspec  : (by reference) contains information on source item       *
//*          tStamp : (by reference) contains deletion-date timestamp          *
//*          eFile  : (by reference)                                           *
//*                   If a processing error occurs, receives the filespec of   *
//*                   the file which caused the error.                         *
//*                   -- If a top-leve item caused the error, receives fSpec.  *
//*                   -- If source is a non-empty directory, AND if the        *
//*                      top-level directory was created successfully, then    *
//*                      eFile receives the filespec for the file which caused *
//*                      the error.                                            *
//*                                                                            *
//* Returns: 'true'  if operation successful                                   *
//*          'false' if processing error                                       *
//******************************************************************************
//* Programmer's Note: Processing errors are unlikely unless the target        *
//* filesystem fills up or source/target filesystem gets unmounted in          *
//* mid-operation.                                                             *
//*                                                                            *
//******************************************************************************

bool cTrash::cttSendItemToTrash ( const FSpec& fspec, const gString& tStamp, gString& eFile )
{
   eFile.clear() ;               // initialize the error return
   bool status = false ;         // return value

   gString srcSpec = fspec.fSpec,   // path/filename of source
           trgSpec,                 // path/filename of target
           infoSpec ;               // path/filename for information file
   this->cttNextTargetName ( srcSpec, trgSpec, infoSpec ) ;

   //* Create the .trashinfo record *
   if ( (status = this->cttCreateInfoRecord ( srcSpec, infoSpec, tStamp )) != false )
   {
      //* Copy source to target *
      bool targetCreated ;
      if ( fspec.fStat.fType == fmDIR_TYPE )
         targetCreated = this->ctfCopyDirName ( srcSpec, trgSpec, fspec.fStat ) ;
      else
         targetCreated = this->ctfRenameFile ( srcSpec, trgSpec ) ;

      if ( targetCreated != false )
      {  //* If source is a directory, copy its contents to trashcan *
         if ( fspec.fStat.fType == fmDIR_TYPE )
         {
            UINT contents = this->ctfCopyDirContents ( srcSpec, trgSpec ) ;
            //* Update the timestamp in case item is restored in future.      *
            //* (Note that only the mod time will retain the specified value.)*
            this->ctfSetTimestamps ( trgSpec, fspec.fStat.rawStats.st_mtime,
                                     fspec.fStat.rawStats.st_atime ) ;
            if ( contents == fspec.dFiles )
               status = true ;

            //* If successful, delete source directory and its contents.*
            if ( status != false )
            {
               UINT delCount = this->ctfDeleteDirContents ( srcSpec ) ;
               if ( (this->ctfDeleteDirectory ( srcSpec )) != false )
                  ++delCount ;
            }
            //* If copy of contents failed, delete whatever we've    *
            //* copied so far, and then delete the .trashinfo record.*
            else
            {
               this->ctfDeleteDirContents ( trgSpec ) ;
               this->ctfDeleteDirectory ( trgSpec ) ;
               this->ctfDeleteFile ( infoSpec ) ;  // delete the .trashinfo record
            }
         }
      }
      else
      {
         this->ctfDeleteFile ( infoSpec ) ;  // delete the .trashinfo record
         eFile = srcSpec ;
         status = false ;
      }
   }
   return status ;

}  //* Endd cttSendItemtoTrash() *

//***********************
//*  cttNextTargetName  *
//***********************
//******************************************************************************
//* Determine the target filespec for the item to be copied into the           *
//* Trash/files directory.                                                     *
//*                                                                            *
//* a) If target DOES NOT exist, then return a filespec using source filename. *
//* b) If target DOES exist, then find the lowest serialized filename which is *
//*    not in use.                                                             *
//* Please see implementation notes at the top of this module.                 *
//*                                                                            *
//*                                                                            *
//* Input  : srcSpec : (by reference) filespec of source file                  *
//*          trgSpec : (by reference) receives the target filespec             *
//*          infoSpec: (by reference) receives the info record filespec        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void cTrash::cttNextTargetName ( const gString& srcSpec, gString& trgSpec, gString& infoSpec )
{
   gString tfBase( "%S/%S", this->tcDir.gstr(), tcfileDir ),
           tiBase( "%S/%S", this->tcDir.gstr(), tcinfoDir ),
           tName, tExt ;
   this->ctfExtractFilename ( tName, srcSpec ) ;   // get filename incl. extension
   this->ctfExtractFileExtension ( tExt, tName ) ; // get extension
   this->ctfCatPathFilename ( trgSpec, tfBase, tName.gstr() ) ; // create initial test spec
   short ei = tName.find( tExt.gstr() ) ;          // remove extension from filename
   if ( ei >= ZERO )
      tName.limitChars( ei ) ;
   short cerealNum = 1 ;                           // filename serialization

   while ( (this->ctfTargetExists ( trgSpec, false, true )) != false )
   {
      ++cerealNum ;
      trgSpec.compose( L"%S/%S.%hd%S", tfBase.gstr(), tName.gstr(), &cerealNum, tExt.gstr() ) ;
   }

   //* Create filespec for .trashinfo record *
   this->ctfExtractFilename ( tName, trgSpec ) ;
   infoSpec.compose( "%S/%S%S", tiBase.gstr(), tName.gstr(), trashinfoExt ) ;

}  //* End cttNextTargetName() *

//***********************
//* cttCreateInfoRecord *
//***********************
//******************************************************************************
//* Create .trashinfo record file describing the item sent to the trashcan.    *
//*                                                                            *
//* Note that if there is an existing target file, then it has been orphaned,  *
//* at some previous time, so we overwrite it.                                 *
//*                                                                            *
//* Input  : srcSpec  : (by reference) path/filename of source item            *
//*          infoSpec : (by reference) path/filename for .trashinfo record file*
//*          tStamp   : deletion date entry for record file                    *
//*                                                                            *
//* Returns: 'true' if log file created, else 'false'                          *
//******************************************************************************

bool cTrash::cttCreateInfoRecord ( const gString& srcSpec, 
                                   const gString& infoSpec, const gString& tStamp )
{
   bool status = false ;

   ofstream ofs( infoSpec.ustr(), ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
   {
      ofs << trashinfoHdr << '\n'
          << trashinfoPath << srcSpec.ustr() << '\n'
          << trashinfoDate << tStamp.ustr() << endl ;

      ofs.close() ;
      status = true ;
   }
   return status ;

}  //* End cttCreateInfoRecord() *

//***************************
//* cttRestoreItemFromTrash *
//***************************
//******************************************************************************
//* Restore the specified item in the trashcan to its original position OR     *
//* to the alternate positon specified by tcdi.trgPath.                        *
//*                                                                            *
//* 1) 'tcdi' contains the source path, target path and '.trashinfo' path,     *
//*    as well as statistics on the source item.                               *
//* 2) Caller has verified that the target's parent directory exists and that  *
//*    we have r/w/x permission for it.                                        *
//* 3) Caller has verified that we have r/w permission on any existing file    *
//*    targets, and r/w/x permission on any existing directory targets.        *
//* 4) User has given either direct ('y' to prompt) or indirect (--no-confirm  *
//*    option) permission to overwrite existing targets.                       *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : tcdi   : record describing the item to be restored                *
//*                   and the restoration target path/filename                 *
//*                                                                            *
//* Returns: 'true'  if item successfully restored                             *
//*          'false' if unable to restore item                                 *
//******************************************************************************
//* Notes:                                                                     *
//* 1) Ideally, we would not write anything to the display in this method;     *
//*    however, the caller receives only success or failure of the operation.  *
//*    So in the case of a serious (but unlikely) error while restoring the    *
//*    contents of a directory tree, we alert the user.                        *
//*                                                                            *
//*    The caller has verified that there are no existing targets which        *
//*    are protected, so the most likely cause of failure is that the target   *
//*    filesystem has run out of space, has been unmounted or some other       *
//*    event beyond our control.                                               *
//*                                                                            *
//*    We have a design decision when this happens. In an ideal world, we      *
//*    would remove whatever we have successfully copied, but that would be    *
//*    a lot of work, even if the target filesystem is still accessible.       *
//*    Our decision is to alert the user that the restore was only partially   *
//*    successful, AND to reassure him/her/it that the source item in the      *
//*    Trashcan is intact.                                                     *
//*                                                                            *
//******************************************************************************

bool cTrash::cttRestoreItemFromTrash ( const tcDetailInfo& tcdi )
{
   bool srcDeleted = false,         // true if source item successfully deleted
        itemRestored = false ;      // return value

   //* Copy the file or top-level directory name to target *
   if ( (this->ctfCopyFile ( tcdi.srcPath, tcdi.trgPath, tcdi.dfs )) != false )
   {
      //* If source is a single file, it has *
      //* been copied to the target position.*
      if ( tcdi.fType != fmDIR_TYPE )
      {
         itemRestored = true ;
      }

      //* Else, source is a directory name. Copy its contents. *
      else
      {
         UINT filesCopied = this->ctfCopyDirContents ( tcdi.srcPath, tcdi.trgPath ) ;
         if ( filesCopied == (tcdi.files - 1) )
         {
            //* Set target timestamp to source timestamp.                     *
            //* (Note that only the mod time will retain the specified value.)*
            this->ctfSetTimestamps ( tcdi.trgPath, tcdi.dfs.modTime.sysepoch,
                                     tcdi.dfs.modTime.sysepoch ) ;
            itemRestored = true ;
         }
         else
         {  //* Source directory was not fully restored. *
            //* This is rather unlikely. See note above. *
            if ( !this->silent )
            {
               this->textOut ( L" SYSTEM ERROR! The following item was only partially restored:\n"
                                "    ", false ) ;
               this->textOut ( tcdi.trgPath ) ;
               this->textOut ( 
                  L" The most likely cause is that the target filesystem is full, or has\n"
                   " become inaccessible. Trashcan source item is fully intact.\n" ) ;
            }
         }
      }

      //* If source item successfully copied to target, then *
      //* remove the source item and its 'trashinfo' record. *
      if ( itemRestored )
      {
         if ( tcdi.fType == fmDIR_TYPE )
         {
            UINT filesDeleted = this->ctfDeleteDirContents ( tcdi.srcPath ) ;
            if ( filesDeleted == (tcdi.files - 1) )
               srcDeleted = this->ctfDeleteDirectory ( tcdi.srcPath ) ;
         }
         else
         {
            srcDeleted = this->ctfDeleteFile ( tcdi.srcPath ) ;
         }

         if ( srcDeleted != false )
         {  //* Source item successfully deleted.                         *
            //* Remove the '.trashinfo' record. If this fails, it does no *
            //* real harm, since it describes an item which no longer     *
            //* exists, so reporting it would only confuse the user.      *
            this->ctfDeleteFile ( tcdi.infoPath ) ;
         }
         else
         {  //* Source item deletion failed. This means that the target *
            //* item has been properly restored, BUT that the trashcan  *
            //* record has a problem. This is fairly unlikely because   *
            //* source permission was verified at a higher level.       *
            //*       CURRENTLY, WE IGNORE THIS KIND OF ERROR.          *
         }
      }
   }
   return itemRestored ;

}  //* End cttRestoreItemFromTrash() *

//**************************
//* cttDeleteItemFromTrash *
//**************************
//******************************************************************************
//* Delete an item from the trashcan.                                          *
//*                                                                            *
//* Input  : tcdi   : record describing the item to be deleted                 *
//*                                                                            *
//* Returns: 'true'  if item successfully deleted (or item does not exist)     *
//*          'false' if unable to delete item                                  *
//******************************************************************************
bool cTrash::cttDeleteItemFromTrash ( const tcDetailInfo& tcdi )
{
   UINT dFiles = ZERO ;          // files deleted
   bool itemDeleted = false ;    // return value

   //* If the target is a directory *
   if ( tcdi.fType == fmDIR_TYPE )
   {
      //* If the directory is not empty, delete its contents *
      if ( tcdi.files > 1 )
         dFiles = this->ctfDeleteDirContents ( tcdi.srcPath ) ;

      //* If we deleted all contents or if the directory was already empty, *
      //* everything is fine. If we deleted ANYTHING, but not EVERYTHING,   *
      //* then the data are now corrupted, so we must continue anyway.      *
      if ( dFiles > ZERO || tcdi.files == 1 )
      {
         if ( (this->ctfDeleteDirectory ( tcdi.srcPath )) != false )
            ++dFiles ;
      }
   }
   else
   {
      if ( (this->ctfDeleteFile ( tcdi.srcPath )) != false )
         ++dFiles ;
   }

   //* Remove the item's record file *
   if ( dFiles > ZERO )
   {
      this->ctfDeleteFile ( tcdi.infoPath ) ;
      itemDeleted = true ;
   }
   return itemDeleted ;

}  //* End cttDeleteItemFromTrash() *

//***********************
//*    cttEmptyTrash    *
//***********************
//******************************************************************************
//* Delete all data from the trashcan.                                         *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: 'true'  if successful                                             *
//*          'false' if processing error                                       *
//******************************************************************************
bool cTrash::cttEmptyTrash ( void )
{
   bool status = false ;

   gString targDir ;
   this->ctfCatPathFilename ( targDir, this->tcDir, tcfileDir ) ;
   this->ctfDeleteDirContents ( targDir ) ;
   this->ctfCatPathFilename ( targDir, this->tcDir, tcinfoDir ) ;
   this->ctfDeleteDirContents ( targDir ) ;

   //* Re-scan the trashcan, and if both      *
   //* subdirectories are empty, then success.*
   tcSummaryInfo tcsumm ;
   this->cttSummaryScan ( tcsumm ) ;
   if ( tcsumm.iFiles == ZERO && tcsumm.files == ZERO )
      status = true ;

   return status ;

}  //* End cttEmptyTrash() *

//***********************
//*   cttSummaryScan    *
//***********************
//******************************************************************************
//* Get summary statistics for items in the Trashcan.                          *
//*                                                                            *
//* Input  : tcSummary : (by reference) receives summary data                  *
//*                                                                            *
//* Returns: 'true'  if scan successful                                        *
//*          'false' if processing error                                       *
//******************************************************************************
//* Note on calculating disc space occupied by trashed files:                  *
//*  This is only an approximate value based on the fact that files occupy an  *
//*  integral number of logical allocation blocks on the disc. Files smaller   *
//*  than the size of a block will occupy one block. This includes all the     *
//*  'xxx.trashinfo' files and probably at least some of the trashed files.    *
//*  Files that are larger than one block, will likely waste some amount of    *
//*  space in the last block allocated to that file.                           *
//*                                                                            *
//* Our Solution:                                                              *
//*  We don't care about the exact amount, so we calculate it as:              *
//*   total file size +                                                        *
//*   fileCount * average wasted space per file +                              *
//*   one full block for each 'xxx.trashinfo' file                             *
//******************************************************************************

bool cTrash::cttSummaryScan ( tcSummaryInfo& tcSummary )
{
   fileSystemStats fsStats ;           // logical block size && free disc space
   gString filePath, infoPath ;        // Trashcan subdir paths
   UINT   itemCount,                   // interim stats
          fileCount,
          infoCount ;
   UINT64 itemSize,
          totalSize ;
   bool status = false ;               // return value

   //* Initialize caller's data *
   tcSummary.reset() ;

   //* 1) Create the paths to the Trashcan subdirectories.                 *
   //* 2) Get filesystem freespace and allocation-block size.              *
   //* 3) Get number of files in the 'info' subdirectory.                  *
   //* 4) Get number of items AND size of items in 'files' subdirectory.   *
   //* 5) Get number of files AND size of files in 'files' subdirectory.   *
   this->ctfCatPathFilename ( filePath, this->tcDir, tcfileDir ) ;
   this->ctfCatPathFilename ( infoPath, this->tcDir, tcinfoDir ) ;
   if ( ((this->ctfFilesystemStats ( this->tcDir, fsStats )) != false) &&
        ((this->ctfDirectorySummary ( infoPath, infoCount, itemSize )) != false) &&
        ((this->ctfDirectorySummary ( filePath, itemCount, itemSize )) != false) &&
        ((this->ctfDirectorySummary ( filePath, fileCount, totalSize, true )) != false) )
   {
      tcSummary.items  = itemCount ;
      tcSummary.files  = fileCount ;
      tcSummary.iFiles = infoCount ;
      tcSummary.bytes  = totalSize ;
      tcSummary.fSpace = fsStats.freeBytes ;
      if ( tcSummary.bytes > ZERO && fileCount > ZERO ) // (avoid math problems)
      {
         double avgfSize = (double)tcSummary.bytes / (double)fileCount,
                fWaste = (avgfSize >= fsStats.fblockSize) ? 
                         (tcSummary.bytes * 0.05) : 
                         ((double)fileCount * (double)fsStats.fblockSize - avgfSize) ;
         tcSummary.tSpace = (UINT64)(tcSummary.bytes + fWaste + infoCount * fsStats.fblockSize) ;
      }
      status = true ;
   }

   return status ;

}  //* End cttSummaryScan() *

//***********************
//*    cttDetailScan    *
//***********************
//******************************************************************************
//* Scan the files: 'Trash/info/*.trashinfo' and translate their data into     *
//* human readable form. Sort the records according to current sort option.    *
//*                                                                            *
//* NOTE: To avoid memory leaks, be sure to release any previous allocation    *
//*       attached to this pointer before calling this method.                 *
//*                                                                            *
//* Input  : iCount : number of .trashinfo records reported by summary scan    *
//*        : tcdi   : receives a pointer to an array of data records, one for  *
//*                   each _verified_ record in the Trash/info subdirectory    *
//*                                                                            *
//* Returns: number of '.trashinfo' files scanned and verified                 *
//*          i.e. the number of valid records written to tcdi                  *
//*          .trashinfo records without matching item in trash are not reported*
//*          items in trash without matching .trashinfo record are not reported*
//******************************************************************************
//* deStats (aka struct dirent64) -- from bits/dirent.h                        *
//*                                                                            *
//* struct dirent                   // used with readdir/readdir_r             *
//* {                                                                          *
//*    #ifndef __USE_FILE_OFFSET64                                             *
//*    __ino_t d_ino;               // file serial number, 32-bit              *
//*    __off_t d_off;               // file size in bytes, 32-bit              *
//*    #else                                                                   *
//*    __ino64_t d_ino;             // file serial number, 64-bit              *
//*    __off64_t d_off;             // file size in bytes, 64-bit              *
//*    #endif                                                                  *
//*    unsigned short int d_reclen; // size of the directory entry itself      *
//*    unsigned char d_type;        // file type                               *
//*    char d_name[256];            // file name                               *
//* } ;                                                                        *
//*                                                                            *
//* #ifdef __USE_LARGEFILE64                                                   *
//* struct dirent64                 // use with readdir64_r                    *
//* {                                                                          *
//*    __ino64_t d_ino;             // file serial number                      *
//*    __off64_t d_off;             // file size in bytes                      *
//*    unsigned short int d_reclen; // size of the directory entry itself      *
//*    unsigned char d_type;        // file type                               *
//*    char d_name[256];            // file name                               *
//* } ;                                                                        *
//* #endif                                                                     *
//*                                                                            *
//******************************************************************************

UINT cTrash::cttDetailScan ( UINT iCount, tcDetailInfo*& tcdi )
{
   UINT goodiCount = ZERO ;      // return value

   //* Allocate an array of records *
   tcdi = new tcDetailInfo[iCount + 1] ;
   if ( iCount > ZERO )
   {
      gString  filePath,            // Trashcan subdir paths
               infoPath, 
               trimPath,            // trimmed filespec for display
               intFmt,              // formatted-integer string
               dirTag,              // directory-identification tag
               srcData ;            // data from source '.trashinfo' file
      this->ctfCatPathFilename ( filePath, this->tcDir, tcfileDir ) ;
      this->ctfCatPathFilename ( infoPath, this->tcDir, tcinfoDir ) ;
      short afterPrefix, extIndex ;

      //* Open the two subdirectories for reading *
      DIR* idir = this->ctfOpendir ( infoPath ) ;
      DIR* fdir = this->ctfOpendir ( filePath ) ;
      if ( idir != NULL && fdir != NULL )
      {
         deStats*  destat ;
         while ( (destat = readdir64 ( idir )) != NULL )
         {
            //* Do not include 'current dir' and 'parent dir' names.           *
            if ( destat->d_name[ZERO] == PERIOD )
            {
               if (   destat->d_name[1] == NULLCHAR 
                   || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
                  continue ;     // go to next record
            }

            //* Create the full path/filename spec to the info file *
            this->ctfCatPathFilename ( tcdi[goodiCount].infoPath, infoPath, destat->d_name ) ;

            //* Open the file and extract the information *
            ifstream dataIn( tcdi[goodiCount].infoPath.ustr(), ifstream::in ) ;
            if ( dataIn.is_open() )
            {  //* Read and discard the header line *
               if ( (this->ctfReadLine ( dataIn, srcData )) != false )
               {
                  //* Read and format the original filespec *
                  if ( (this->ctfReadLine ( dataIn, srcData )) != false )
                  {  //* Step past the prefix *
                     if ( (afterPrefix = srcData.after( trashinfoPath )) > ZERO )
                     {  //* Save filespec of original file *
                        tcdi[goodiCount].trgPath = &srcData.gstr()[afterPrefix] ;
                        //* Verify that the matching data file exists *
                        this->ctfCatPathFilename ( tcdi[goodiCount].srcPath, 
                                                   filePath, destat->d_name ) ;
                        extIndex = tcdi[goodiCount].srcPath.find( trashinfoExt ) ;
                        tcdi[goodiCount].srcPath.limitChars( extIndex ) ;

                        if ( (this->ctfGetFileStats ( tcdi[goodiCount].srcPath, 
                                                      tcdi[goodiCount].dfs )) != false )
                        {  //* Source item exists.               *
                           //* Read and format the deletion date *
                           if ( (this->ctfReadLine ( dataIn, srcData )) != false )
                           {  //* Format the data for display.              *
                              //* -- If source is a directory name, count   *
                              //*    the number of files it contains.       *
                              //* -- Trim the original filespec to fit      *
                              //*    display.                               *
                              dirTag.clear() ;
                              tcdi[goodiCount].files = 1 ;
                              tcdi[goodiCount].size = tcdi[goodiCount].dfs.fBytes ;
                              tcdi[goodiCount].fType = tcdi[goodiCount].dfs.fType ;
                              if ( tcdi[goodiCount].fType == fmDIR_TYPE )
                              {
                                 UINT   dirFiles ;
                                 UINT64 dirBytes ;
                                 this->ctfDirectorySummary ( 
                                       tcdi[goodiCount].srcPath, 
                                       dirFiles, dirBytes, true ) ;
                                 dirTag.compose( dirtagTemplate, &dirFiles ) ;
                                 tcdi[goodiCount].size += dirBytes ;
                                 tcdi[goodiCount].files += dirFiles ;
                              }
                              trimPath = tcdi[goodiCount].trgPath ;
                              this->ctfTrimPathString ( trimPath, this->maxStatCols ) ;
                              intFmt.formatInt( tcdi[goodiCount].size, 5, true ) ; 

                              tcdi[goodiCount].dRecord.compose( statTemplate,
                                     trimPath.gstr(), dirTag.gstr(),
                                     srcData.gstr(), intFmt.gstr() ) ;

                              //* Parse the deletion date into its components *
                              this->cttDecodeDate ( srcData, tcdi[goodiCount].tDate ) ;

                              tcdi[goodiCount].select = false ; // reset the selection flag
                              ++goodiCount ;    // we have a good record
                           }
                        }
                        else
                        { /* Ignore records with invalid format */ }
                     }
                     else
                     { /* Ignore records with invalid format */ }
                  }
               }
               dataIn.close() ;
            }
         }
      }
      if ( idir != NULL )
         closedir ( idir ) ;
      if ( fdir != NULL )
         closedir ( fdir ) ;

      //* Sort the output *
      this->cttSortRecords ( tcdi, goodiCount, this->sortOption ) ;

      //* Insert the serial number for each item *
      gString gsrc ;
      UINT recNum ;
      for ( UINT i = ZERO ; i < goodiCount ; ++i )
      {
         recNum = i + 1 ;
         gsrc.compose( statSerial, &recNum ) ;
         tcdi[i].dRecord.insert( gsrc.gstr() ) ;
      }
   }
   return goodiCount ;

}  //* End cttDetailScan() *

//*************************
//*    cttSortRecords     *
//*************************
//******************************************************************************
//* Sort detailed trashcan records.                                            *
//*                                                                            *
//* Input  : tcrec   : an array of records, one for each item in trashcan      *
//*          tccnt   : number of records in tcRecord array                     *
//*          sOption : indicates which sort option to use                      *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* This method uses a two-level sort, known as a cocktail sort or             *
//* bi-directional bubble sort; and although the algorithm is not fast,        *
//* the data set is usually less than 100 items.                               *
//*                                                                            *
//* We do a sub-sort for groups which have identical primary comparisons.      *
//* You may disagree with selection of sub-sort type, but this is our best     *
//* guess on making it easier for the user to visually scan the list.          *
//******************************************************************************

void cTrash::cttSortRecords ( tcDetailInfo* tcrec, int tccnt, OutSort sOption )
{
   tcDetailInfo saveLow, saveHigh ; // swap buffers
   int lIndex, gIndex,              // low and high limits search indices
       i ;                          // intermediate index

   //*****************************************
   //*** Sort by deletion date, ascending  ***
   //*****************************************
   if ( tccnt > 1 && sOption == osDATE )
   {
      for ( lIndex = 0, gIndex = tccnt-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            if ( tcrec[i].tDate.epoch < tcrec[lIndex].tDate.epoch )
            {
               saveLow = tcrec[lIndex] ;
               tcrec[lIndex] = tcrec[i] ;
               tcrec[i] = saveLow ;
            }
            if ( i < gIndex )
            {
               if ( tcrec[gIndex].tDate.epoch < tcrec[i].tDate.epoch )
               {
                  saveHigh = tcrec[gIndex] ;
                  tcrec[gIndex] = tcrec[i] ;
                  tcrec[i] = saveHigh ;
                  
                  if ( tcrec[i].tDate.epoch < tcrec[lIndex].tDate.epoch )
                  {
                     saveLow = tcrec[lIndex] ;
                     tcrec[lIndex] = tcrec[i] ;
                     tcrec[i] = saveLow ;
                  }
               }
            }
         }
      }

      //* Scan for items with same date and sub-sort by name, ascending *
      int top, bot ;
      for ( top = ZERO, bot = 1 ; bot < tccnt ; )
      {
         while ( (bot < tccnt) &&
                 tcrec[bot].tDate.epoch == tcrec[top].tDate.epoch )
            ++bot ;
         //* If we found at least two files of the same date *
         if ( (bot - top) > 1 ) 
         {
            --bot ;     // range it
            int cnt = bot - top + 1 ;
            this->cttSortRecords ( &tcrec[top], cnt, osNAME ) ;
            top = bot + 1 ;
         }
         else
            top = bot ;
         bot = top + 1 ;
         if ( top >= tccnt )
            break ;
      }
   }

   //*****************************************
   //*** Sort by deletion date, descending ***
   //*****************************************
   else if ( tccnt > 1 && sOption == osDATE_R )
   {
      for ( lIndex = 0, gIndex = tccnt-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            if ( tcrec[i].tDate.epoch > tcrec[lIndex].tDate.epoch )
            {
               saveLow = tcrec[lIndex] ;
               tcrec[lIndex] = tcrec[i] ;
               tcrec[i] = saveLow ;
            }
            if ( i < gIndex )
            {
               if ( tcrec[gIndex].tDate.epoch > tcrec[i].tDate.epoch )
               {
                  saveHigh = tcrec[gIndex] ;
                  tcrec[gIndex] = tcrec[i] ;
                  tcrec[i] = saveHigh ;
                  
                  if ( tcrec[i].tDate.epoch > tcrec[lIndex].tDate.epoch )
                  {
                     saveLow = tcrec[lIndex] ;
                     tcrec[lIndex] = tcrec[i] ;
                     tcrec[i] = saveLow ;
                  }
               }
            }
         }
      }

      //* Scan for items with same date and sub-sort by name, ascending *
      int top, bot ;
      for ( top = ZERO, bot = 1 ; bot < tccnt ; )
      {
         while ( (bot < tccnt) &&
                 tcrec[bot].tDate.epoch == tcrec[top].tDate.epoch )
            ++bot ;
         //* If we found at least two files of the same date *
         if ( (bot - top) > 1 ) 
         {
            --bot ;     // range it
            int cnt = bot - top + 1 ;
            this->cttSortRecords ( &tcrec[top], cnt, osNAME ) ;
            top = bot + 1 ;
         }
         else
            top = bot ;
         bot = top + 1 ;
         if ( top >= tccnt )
            break ;
      }
   }

   //*****************************************
   //*** Sort by item size, ascending      ***
   //*****************************************
   else if ( tccnt > 1 && sOption == osSIZE )
   {
      for ( lIndex = 0, gIndex = tccnt-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            if ( tcrec[i].size < tcrec[lIndex].size )
            {
               saveLow = tcrec[lIndex] ;
               tcrec[lIndex] = tcrec[i] ;
               tcrec[i] = saveLow ;
            }
            if ( i < gIndex )
            {
               if ( tcrec[gIndex].size < tcrec[i].size )
               {
                  saveHigh = tcrec[gIndex] ;
                  tcrec[gIndex] = tcrec[i] ;
                  tcrec[i] = saveHigh ;
                  
                  if ( tcrec[i].size < tcrec[lIndex].size )
                  {
                     saveLow = tcrec[lIndex] ;
                     tcrec[lIndex] = tcrec[i] ;
                     tcrec[i] = saveLow ;
                  }
               }
            }
         }
      }

      //* Scan for items with same size and sub-sort by name, ascending *
      int top, bot ;
      for ( top = ZERO, bot = 1 ; bot < tccnt ; )
      {
         while ( (bot < tccnt) &&
                 tcrec[bot].size == tcrec[top].size )
            ++bot ;
         //* If we found at least two files of the same date *
         if ( (bot - top) > 1 ) 
         {
            --bot ;     // range it
            int cnt = bot - top + 1 ;
            this->cttSortRecords ( &tcrec[top], cnt, osNAME ) ;
            top = bot + 1 ;
         }
         else
            top = bot ;
         bot = top + 1 ;
         if ( top >= tccnt )
            break ;
      }
   }

   //*****************************************
   //*** Sort by item size, descending     ***
   //*****************************************
   else if ( tccnt > 1 && sOption == osSIZE_R )
   {
      for ( lIndex = 0, gIndex = tccnt-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            if ( tcrec[i].size > tcrec[lIndex].size )
            {
               saveLow = tcrec[lIndex] ;
               tcrec[lIndex] = tcrec[i] ;
               tcrec[i] = saveLow ;
            }
            if ( i < gIndex )
            {
               if ( tcrec[gIndex].size > tcrec[i].size )
               {
                  saveHigh = tcrec[gIndex] ;
                  tcrec[gIndex] = tcrec[i] ;
                  tcrec[i] = saveHigh ;
                  
                  if ( tcrec[i].size > tcrec[lIndex].size )
                  {
                     saveLow = tcrec[lIndex] ;
                     tcrec[lIndex] = tcrec[i] ;
                     tcrec[i] = saveLow ;
                  }
               }
            }
         }
      }

      //* Scan for items with same size and sub-sort by name, ascending *
      int top, bot ;
      for ( top = ZERO, bot = 1 ; bot < tccnt ; )
      {
         while ( (bot < tccnt) &&
                 tcrec[bot].size == tcrec[top].size )
            ++bot ;
         //* If we found at least two files of the same date *
         if ( (bot - top) > 1 ) 
         {
            --bot ;     // range it
            int cnt = bot - top + 1 ;
            this->cttSortRecords ( &tcrec[top], cnt, osNAME ) ;
            top = bot + 1 ;
         }
         else
            top = bot ;
         bot = top + 1 ;
         if ( top >= tccnt )
            break ;
      }
   }

   //*****************************************
   //*** Sort by item name, ascending      ***
   //*****************************************
   else if ( tccnt > 1 && sOption == osNAME )
   {
      gString lName, gName, iName ;
      for ( lIndex = 0, gIndex = tccnt-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            this->ctfExtractFilename ( iName, tcrec[i].trgPath ) ;
            this->ctfExtractFilename ( lName, tcrec[lIndex].trgPath ) ;
            if ( (iName.compare( lName.gstr() )) < ZERO )
            {
               saveLow = tcrec[lIndex] ;
               tcrec[lIndex] = tcrec[i] ;
               tcrec[i] = saveLow ;
            }
            if ( i < gIndex )
            {
               this->ctfExtractFilename ( gName, tcrec[gIndex].trgPath ) ;
               this->ctfExtractFilename ( iName, tcrec[i].trgPath ) ;
               if ( (gName.compare( iName.gstr() )) < ZERO )
               {
                  saveHigh = tcrec[gIndex] ;
                  tcrec[gIndex] = tcrec[i] ;
                  tcrec[i] = saveHigh ;

                  iName = gName ;
                  this->ctfExtractFilename ( lName, tcrec[lIndex].trgPath ) ;
                  if ( (iName.compare( lName.gstr() )) < ZERO )
                  {
                     saveLow = tcrec[lIndex] ;
                     tcrec[lIndex] = tcrec[i] ;
                     tcrec[i] = saveLow ;
                  }
               }
            }
         }
      }

      //* Scan for items with same name and sub-sort by date, ascending *
      int top, bot ;
      for ( top = ZERO, bot = 1 ; bot < tccnt ; )
      {
         this->ctfExtractFilename ( lName, tcrec[top].trgPath ) ;
         while ( bot < tccnt )
         {
            this->ctfExtractFilename ( iName, tcrec[bot].trgPath ) ;
            if ( (lName.compare( iName.gstr() )) == ZERO )
               ++bot ;
            else
               break ;
         }

         //* If we found at least two files of the same date *
         if ( (bot - top) > 1 ) 
         {
            --bot ;     // range it
            int cnt = bot - top + 1 ;
            this->cttSortRecords ( &tcrec[top], cnt, osDATE ) ;
            top = bot + 1 ;
         }
         else
            top = bot ;
         bot = top + 1 ;
         if ( top >= tccnt )
            break ;
      }
   }

   //*****************************************
   //*** Sort by item name, descending     ***
   //*****************************************
   else if ( tccnt > 1 && sOption == osNAME_R )
   {
      gString lName, gName, iName ;
      for ( lIndex = 0, gIndex = tccnt-1 ; (gIndex-lIndex) > ZERO ; lIndex++, gIndex-- )
      {
         for ( i = lIndex+1 ; i <= gIndex ; i++ )
         {
            this->ctfExtractFilename ( iName, tcrec[i].trgPath ) ;
            this->ctfExtractFilename ( lName, tcrec[lIndex].trgPath ) ;
            if ( (iName.compare( lName.gstr() )) > ZERO )
            {
               saveLow = tcrec[lIndex] ;
               tcrec[lIndex] = tcrec[i] ;
               tcrec[i] = saveLow ;
            }
            if ( i < gIndex )
            {
               this->ctfExtractFilename ( gName, tcrec[gIndex].trgPath ) ;
               this->ctfExtractFilename ( iName, tcrec[i].trgPath ) ;
               if ( (gName.compare( iName.gstr() )) > ZERO )
               {
                  saveHigh = tcrec[gIndex] ;
                  tcrec[gIndex] = tcrec[i] ;
                  tcrec[i] = saveHigh ;

                  iName = gName ;
                  this->ctfExtractFilename ( lName, tcrec[lIndex].trgPath ) ;
                  if ( (iName.compare( lName.gstr() )) > ZERO )
                  {
                     saveLow = tcrec[lIndex] ;
                     tcrec[lIndex] = tcrec[i] ;
                     tcrec[i] = saveLow ;
                  }
               }
            }
         }
      }

      //* Scan for items with same name and sub-sort by date, ascending *
      int top, bot ;
      for ( top = ZERO, bot = 1 ; bot < tccnt ; )
      {
         this->ctfExtractFilename ( lName, tcrec[top].trgPath ) ;
         while ( bot < tccnt )
         {
            this->ctfExtractFilename ( iName, tcrec[bot].trgPath ) ;
            if ( (lName.compare( iName.gstr() )) == ZERO )
               ++bot ;
            else
               break ;
         }

         //* If we found at least two files of the same date *
         if ( (bot - top) > 1 ) 
         {
            --bot ;     // range it
            int cnt = bot - top + 1 ;
            this->cttSortRecords ( &tcrec[top], cnt, osDATE ) ;
            top = bot + 1 ;
         }
         else
            top = bot ;
         bot = top + 1 ;
         if ( top >= tccnt )
            break ;
      }
   }

}  //* End cttSortRecords() *

//***********************
//*    cttDecodeDate    *
//***********************
//******************************************************************************
//* Parse the deletion-date string from the '.trashinfo' file.                 *
//*                                                                            *
//* Input  : trashedDate : (by reference) original deletion-date string        *
//*          tDate       : (by reference) receives decoded datestamp           *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* The string SHOULD BE in the following format:                              *
//*                    DeletionDate=yyyy-mm-ddThh:mm:ss                        *
//* If it is not, we return a dummy datestamp.                                 *
//******************************************************************************

void cTrash::cttDecodeDate ( const gString& trashedDate, localTime& tDate )
{
   tDate.reset() ;            // initialize to epoch value
   short aIndex ;
   if ( (aIndex = trashedDate.after( L"=" )) >= ZERO )
   {
      const wchar_t* wptr = &trashedDate.gstr()[aIndex] ;
      int convCount =swscanf ( wptr, L"%hu-%hu-%huT%hu:%hu:%hu",
                               &tDate.year, &tDate.month, &tDate.date, 
                               &tDate.hours, &tDate.minutes, &tDate.seconds ) ;

      if ( convCount == 6 )
      {  //* Convert formatted time to simple epoch time *
         Tm tm ;
         tm.tm_wday = tDate.day - 1 ;          // (this field ignored)
         tm.tm_mday = tDate.date ;
         tm.tm_mon  = tDate.month - 1 ;
         tm.tm_year = tDate.year - 1900 ;
         tm.tm_hour = tDate.hours ;
         tm.tm_min  = tDate.minutes ;
         tm.tm_sec  = tDate.seconds ;
         tDate.sysepoch = timelocal ( &tm ) ;     // (see mktime())
         tDate.epoch = (int64_t)tDate.sysepoch ;  // (see note above)
      }
      else
         tDate.reset() ;
   }

}  //* End cttDecodeDate() *

//***********************
//*    cttEncodeDate    *
//***********************
//******************************************************************************
//* Given the current local time, create a timestamp string for the            *
//* '.trashinfo' record.                                                       *
//*                                                                            *
//* Input  : lt      : (by reference) current system local time                *
//*          tStamp  : (by reference) receives encoded timestamp               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note:                                                         *
//* Format of the string returned is: DeletionDate=yyyy-mm-ddThh:mm:ss         *
//*                                                                            *
//******************************************************************************

void cTrash::cttEncodeDate ( const localTime& lt, gString& tStamp )
{
   tStamp.compose( L"%04hu-%02hu-%02huT%02hu:%02hu:%02hu",
                   &lt.year, &lt.month, &lt.date, 
                   &lt.hours, &lt.minutes, &lt.seconds ) ;
}  //* End cttEncodeDate() *

//***********************
//*   cttTestFiletype   *
//***********************
//******************************************************************************
//* Test the specified filetype against the list of supported filetypes.       *
//*                                                                            *
//* Input  : fType  : type of file                                             *
//*                                                                            *
//* Returns: 'true'  if one of the supported filetypes                         *
//*          'false' if unsupported filetype                                   *
//******************************************************************************

bool cTrash::cttTestFiletype ( fmFType fType )
{
   bool suppType = false ;

   if ( (fType == fmDIR_TYPE)  ||
        (fType == fmREG_TYPE)  ||
        (fType == fmLINK_TYPE) ||
        (fType == fmFIFO_TYPE) )
      suppType = true ;

   return suppType ;

}  //* End cttTestFiletype() *

#if DEBUG_METHODS != 0        // Development and debugging only
//*************************
//*   cttDisplayRecord    *
//*************************
//******************************************************************************
//* FOR DEVELOPMENT ONLY!                                                      *
//* Display the contents of a tcDetailInfo record.                             *
//*                                                                            *
//* Input  : tcdi  : record to be displayed                                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void cTrash::cttDisplayRecord ( const tcDetailInfo& tcdi )
{
   const wchar_t* fileTYPE[] = 
   {
      L"fmDIR_TYPE  ",
      L"fmREG_TYPE  ",
      L"fmLINK_TYPE ",
      L"fmCHDEV_TYPE",
      L"fmBKDEV_TYPE",
      L"fmFIFO_TYPE ",
      L"fmSOCK_TYPE ",
      L"fmUNKNOWN_TYPE",
   } ;
   gString gsOut, fmtFiles, fmtSize ;

   this->textOut ( "tcDetailInfo Record\n"
                   "-------------------------------------------" ) ;
   this->textOut ( tcdi.srcPath ) ;
   this->textOut ( tcdi.trgPath ) ;
   this->textOut ( tcdi.infoPath ) ;
   this->textOut ( tcdi.dRecord ) ;
   fmtFiles.formatInt( tcdi.files, 6, true ) ;
   fmtSize.formatInt( tcdi.size, 6, true ) ;
   gsOut.compose( "Del Date   : %04hd-%02hd-%02hd_%02hd:%02hd:%02hd\n"
                  "File Type  : %S  File Count: %-6S  File Size: %S\n"
                  "Permissions: %S/%S/%S         Select    : %S",
                  &tcdi.tDate.year, &tcdi.tDate.month, &tcdi.tDate.date, 
                  &tcdi.tDate.hours, &tcdi.tDate.minutes, &tcdi.tDate.seconds, 
                  fileTYPE[tcdi.fType], fmtFiles.gstr(), fmtSize.gstr(),
                  (tcdi.dfs.rAccess ? L"r" : L"-"),
                  (tcdi.dfs.wAccess ? L"w" : L"-"),
                  (tcdi.dfs.xAccess ? L"x" : L"-"),
                  (tcdi.select ? L"yes" : L"no") ) ;
   this->textOut ( gsOut ) ;
   this->textOut ( L"" ) ;

}  //* End cttDisplayRecord() *
#endif                        // DEBUG_METHODS

//***********************
//*                     *
//***********************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

