//******************************************************************************
//* File       : Trsh.hpp                                                      *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2015-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 17-Jan-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Definitions and data for Command-line Trashcan utility,       *
//* 'trsh' which allows the command-line user AND command-line utilities to    *
//* access the system's local (user-owned) Trashcan directories.               *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* =================                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include <iostream>        //* Standard I/O definitions
#include <sstream>         //* Definitions for iostringstream classes
#include <iomanip>         //* Output formatting specifications
#include <cstring>         //* String manipulation from C
#include <unistd.h>        //* UNIX system interface
#include <cctype>          //* Character testing
#include <cmath>           //* Math library
#include <cstdlib>         //* Misc. functionality
#include <ctime>           //* Access to system time
#include <thread>          //* std::thread definition (and POSIX thread defs)
#include <condition_variable> //* Additional info for threads
#include <chrono>          //* Timers for temporarily putting threads to sleep
#include <mutex>           //* For access locks on critical data
#include <locale>          //* Locale handling for character encoding, etc. 
#include <langinfo.h>      //* Support for world languages identification/conversion
#include <fstream>         // for file i/o
#include <sys/stat.h>      // for 'stat' info on target files
#include <dirent.h>        // for reading contents of 'directory type' files

using namespace std ;      //* Scope quailfier

#include "gString.hpp"     //* gString class definition
#include "cTrashFile.hpp"  //* Data definitions for file manipulation


#define DEBUG_METHODS (0)  // Enable Development and Debugging methods

//* Specify language for output: *
//* 0 == English                 *
//* 1 == Español                 *
//* 2 == Tiếng Việt              *
//* 3 == 汉语（简体）              *
#define CT_LANG (0)


//***********************************************
//* Constant Data:                              *
//***********************************************

//* Application version string. Keep it current! *
const wchar_t* const AppVersion = L"0.0.06" ;

//* Application title *
const wchar_t* const AppTitle1 = L"ctrash v:" ;
const wchar_t* const AppTitle2 = L" (c)2015-2020 The Software Samurai" ;
const wchar_t* const AppTitle3 = // (this string == minTERMCOLS)
L"-------------------------------------------------" ;

const wchar_t CR = 0x000D ;   // carriage-return character
const wchar_t STAR = ('*') ;  // asterisk
const wchar_t SLASH = ('/') ; // forward slash
const wchar_t BSLASH = ('\\') ; // back slash
const wchar_t SGLQUOTE = ('\'') ;// single quote
const wchar_t DBLQUOTE = ('"') ;// double quotation mark
const wchar_t HASH = ('#') ;  // hash mark, pound/number sign

//* Minimum value for '--remove' option (number of minutes in 24 hours) *
//* Maximum value for '--remove' option (number of minutes in 1 year)   *
const int MIN_REM = -(24 * 60) ;
const int MAX_REM = (366 * 24 * 60) ;

//* Trashcan subdirectory names *
const wchar_t* const tcfileDir = L"files" ;
const wchar_t* const tcinfoDir = L"info" ;

//* Filetype strings for error reporting.*
const wchar_t* const ftStrings[] = 
{
   L"DIR",
   L"REG",
   L"SYM_LINK",
   L"CHAR_DEV",
   L"BLOCK_DEV",
   L"FIFO",
   L"SOCKET",
   L"UNKNOWN",
} ;

//* Sort option for detailed report of Trashcan contents, *
//* and for interactive item restore.                     *
enum OutSort : short
{
   osDATE,        // sort by deletion date, ascending (default)
   osDATE_R,      // sort by deletion date, descending
   osSIZE,        // sort by item size (bytes) ascending
   osSIZE_R,      // sort by item size (bytes) descending
   osNAME,        // sort by _original_ item name, ascending
   osNAME_R,      // sort by _original_ item name, descending
   osNOSORT,      // items are displayed in the order stored on disk
} ;

//* 'rm' command emulation: interactive confirmation options *
enum rmIteract : short
{
   rmINT_NEVER,   // never ask for confirmation
   rmINT_ONCE,    // ask for confirmation once before batch move
   rmINT_EACH     // ask for confirmation before each item moved
} ;

//* Holds captured command-line arguments *
class commArgs
{
   public:
   commArgs ( int argc, char** argv, char** argenv ) :
               argCount(argc), argList(argv), envList(argenv)
   {
      this->reset() ;
   }
   void reset ( void )
   {
      this->reset_flags() ;
      this->errMsg = "ERROR: " ;
      *this->trashDir = NULLCHAR ;
      *this->altTarget = NULLCHAR ;
      this->remTime = ZERO ;
      this->itemCount = this->dispWidth = ZERO ;
      this->sortOption = osDATE ;
   }
   void reset_flags ( void )
   {
      this->textFlag = this->confirm = true ;
      this->restore  = this->empty   = this->preconf = this->report  = 
      this->details  = this->silent  = this->quiet   = 
      this->verbose  = this->permdel = 
      this->verFlag  = this->helpFlag = false ;

      //* 'rm' emulation flags *
      this->rm_mode = this->rm_dir = this->rm_rec = false ;
      this->rm_int = rmINT_NEVER ;
   }
   short   argCount ;   // command-line arguments
   char**  argList ;    // list of argument strings
   char**  envList ;    // pointer to terminal environment
   gString errMsg ;     // if command-line option error, then description of error
   char    trashDir[gsMAXBYTES] ; // path to trashcan directory
   char    altTarget[gsMAXBYTES]; // alternate target directory for restore operation
   int     remTime ;    // For remove-from-trash option, deleting according to time in trash
   short   itemCount ;  // Number of filenames in 'fileList'
   short   dispWidth ;  // Optional display width for formatting output
   OutSort sortOption ; // Optional sort option (see above)
   rmIteract rm_int ;   // 'rm' emulation interactive option
   bool    textFlag ;   // 'true' if text-only, 'false' if interactive dialog
   bool    restore ;    // 'true' if restore-from-trash, 'false' if send-to-trash
   bool    empty ;      // 'true' if request to empty the trash
   bool    preconf ;    // 'true' if confirm BEFORE sending files to trashcan
   bool    confirm ;    // 'true' if confirm BEFORE actions that cannot be undone
   bool    report ;     // 'true' if reporting on contents of trashcan
   bool    details ;    // 'true' if detailed or interactive operation
   bool    silent ;     // 'true' if writing no output to display
   bool    quiet ;      // 'true' if writing no output to display EXCEPT error reports
   bool    verbose ;    // 'true' if verbose (detailed) output to display
   bool    permdel ;    // 'true' if bypassing trashcan and permanently deleting targets
   bool    rm_mode ;    // 'true' if 'rm' emulation mode
   bool    rm_dir ;     // 'rm' emulation empty-directory option
   bool    rm_rec ;     // 'rm' emulation non-empty directory option
   bool    verFlag ;    // 'true' if application version request (overrides all others)
   bool    helpFlag ;   // 'true' if command-line help (overrides all except verFlag)
} ;

//* Holds a path/filename specification for file to be processed.*
class FSpec
{
   public:
   gString    fSpec ;   // filespec (path/filename)
   DFileStats fStat ;   // file statistics
   UINT64     fsysID ;  // if filetype==fmDIR_TYPE, filesystem ID (hex)
   UINT64     dBytes ;  // if filetype==fmDIR_TYPE, bytes in contents
   UINT       dFiles ;  // if filetype==fmDIR_TYPE, files in contents
} ;

//* Record for Trashcan scan (summary) *
class tcSummaryInfo
{
   public:
   void reset ( void )
   {
      this->items = this->files = this->iFiles = ZERO ;
      this->bytes = this->tSpace = this->fSpace = ZERO ;
   }

   UINT   items ;          // number of top-level items
   UINT   files ;          // number of data files
   UINT   iFiles ;         // number of information records
   UINT64 bytes ;          // total size of data files
   UINT64 tSpace ;         // total combined size of data and info files on disk
   UINT64 fSpace ;         // free disk space (bytes)
} ;

//* Record for Trashcan scan (item detail)  *
class tcDetailInfo
{
   public:
   gString srcPath ;    // path/filename of item in 'Trash/files' directory
   gString trgPath ;    // path/filename of item in its original location
   gString infoPath ;   // path/filename of '.trashinfo' file in 'Trash/info'
   gString dRecord ;    // data record formatted for output (132 columns, optimal)
   DFileStats dfs ;     // stat data for srcPath
   localTime tDate ;    // datestamp for when item was trashed
   fmFType fType ;      // file type of source item (dir or non-dir)
   UINT   files ;       // total number of files in the item (non-directory==1)
                        // if 'fType' == fmDIR_TYPE, number of files item
                        // contains (including the item name)
   UINT64 size ;        // total size of item in 'Trash/files' directory
   bool select ;        // 'true' if item is selected for restore or delete
} ;

enum fspecWidth : short // Max column width for displayed filespec
{                       // (see cTrash::maxStatCols)
   fswDEFAULT = 132,
   fswMIN     =  80,
   fswMAX     = 512
} ;


//********************************
//* Application class definition *
//********************************
class cTrash
{
   public:
   virtual ~cTrash () ;             // destructor
   cTrash ( commArgs& ca ) ;        // constructor
   short ctProcStatus ( void )      // returns 'OK' if all files processed successfully
   { return this->procStatus ; }

   //***-----------------------------------------***
   //***   Public Trashcan management methods    ***
   //***-----------------------------------------***
   //* Report the contents of the trashcan directory.                          *
   //*                                                                         *
   //* Input  : details    : 'false' summary report                            *
   //*                       'true'  detailed report                           *
   //*                                                                         *
   //* Returns: number if items currently in trashcan (not the file count)     *
   //*          ERR (-1) if one or more pre-processing errors                  *
   short ctReportTrashcan ( bool details ) ;

   //* Move file(s) from the original position on the filesystem to the user's *
   //* local trashcan (or to the alternate trashcan directory, if specified).  *
   //*                                                                         *
   //* User access is verified for ALL items before ANY item is moved.         *
   //*                                                                         *
   //*                                                                         *
   //* Input  : confirm   : 'false' no confirmation required                   *
   //*                      'true'  ask for user confirmation before operation *
   //*          permDelete: 'false' move source files to trashcan              *
   //*                      'true'  bypass trashcan and unlink the file(s)     *
   //*          altTarget : (optional, NULL pointer by default)                *
   //*                      -- If NULL pointer (or empty string) indicates     *
   //*                         restore to original position.                   *
   //*                      -- If specified:                                   *
   //*                         a) path of alternate target directory           *
   //*                            (directory must exist)                       *
   //*                         b) full path/filename for alternate target      *
   //*                            (if target already exists, confirm overwrite)*
   //*                                                                         *
   //* Returns: number of items succesfully moved to trashcan (not file count) *
   //*          ZERO if user aborted the operation                             *
   //*          ERR (-1) if one or more pre-processing errors                  *
   short ctMoveToTrash ( bool confirm, bool permDelete, const char* altTarget = NULL ) ;

   //* Restore files from trashcan to their original position (or to alternate *
   //* target directory, if specified).                                        *
   //*                                                                         *
   //* Input  : interact  : 'true;  interactively select item(s) to be restored*
   //*                      'false' restore most-recently deleted item(s)      *
   //*          preconfirm: 'true'  get user confirmation before restore       *
   //*                      'false' no confirmation required for restoration   *
   //*          altTarget : (optional, NULL pointer by default)                *
   //*                      -- If NULL pointer (or empty string) indicates     *
   //*                         restore to original position.                   *
   //*                      -- If specified:                                   *
   //*                         a) path of alternate target directory           *
   //*                            (directory must exist)                       *
   //*                         b) full path/filename for alternate target      *
   //*                            (if target already exists, confirm overwrite)*
   //*                                                                         *
   //* Returns: number of items succesfully restored from trashcan             *
   //*            (not the file count)                                         *
   //*          ZERO if user aborted the operation                             *
   //*          ERR (-1) if one or more pre-processing errors                  *
   short ctRestoreFromTrash ( bool interact, bool preconfirm, 
                              const char* altTarget = NULL ) ;

   //* Empty the trashcan: 1) completely, 2) interactively, 3) by freshness    *
   //* date. This operation cannot be undone.                                  *
   //*                                                                         *
   //* NOTE: This method is not called by the cTrash application itself.       *
   //*       In order to get better timestamp resolution, cTrash calls a       *
   //*       private overload of ctEmptyTrash. This method is for use by       *
   //*       applications which integrate the cTrash source code into their    *
   //*       own code.                                                         *
   //*                                                                         *
   //* Input  : interact  : 'false' delete all files from the trashcan         *
   //*                      'true'  display trashcan contents and prompt user  *
   //*                              for files to be deleted                    *
   //*          days      : (optional, 'true' by default)                      *
   //*                      (ignored if remTime==ZERO or 'interact' != false)  *
   //*                      indicates whether 'remTime' should be interpreted  *
   //*                      as days or as hours                                *
   //*                      'true'  : interpret 'remTime' as days              *
   //*                      'false' : interpret 'remTime' as hours             *
   //*          remTime   : (optional, ZERO by default, and        )           *
   //*                      (ignored if ZERO or 'interact' != false)           *
   //*                      if specified and 'days' != false:                  *
   //*                        Items which have been in the trashcan for        *
   //*                        >= remTime days will be deleted                  *
   //*                        Range: 1 - 366 days (1 day through 1 year)       *
   //*                      if specified and 'days' == false:                  *
   //*                        Items which have been in the trashcan for        *
   //*                        <= remTime hours will be deleted                 *
   //*                        Range: 1 - 24 hours                              *
   //*                                                                         *
   //* Returns: number of items succesfully deleted from trashcan              *
   //*            (not the file count)                                         *
   //*          ZERO if preprocessing error or user aborted the operation      *
   short ctEmptyTrash ( bool interact, bool days = true, short remTime = ZERO ) ;

   //* Scan the files: 'Trash/info/*.trashinfo' and translate their data into  *
   //* human readable form. Sort the records according to current sort option. *
   //*                                                                         *
   //* IMPORTANT NOTE:                                                         *
   //* This method performs dynamic memory allocation for the records attached *
   //* to the 'tcdi' parameter. To avoid memory leaks, be sure to release any  *
   //* allocation attached to 'tcdi' pointer:  delete [] tcdi ;                *
   //*                                                                         *
   //* NOTE: This method is not called by the cTrash application itself.       *
   //*       In order to better respond to error conditions, cTrash calls the  *
   //*       summary-scan and detail-scan methods separately.                  *
   //*       This method is for use by applications which integrate the cTrash *
   //*       source code into their own code and want to optionally format the *
   //*       output in their own style.                                        *
   //*                                                                         *
   //* Input  : tcsi  : receives summary data for trashcan contents            *
   //*          tcdi  : receives a pointer to an array of data records, one for*
   //*                  each _verified_ record in the Trash/info subdirectory  *
   //*                                                                         *
   //* Returns: number of '.trashinfo' files scanned and verified              *
   //*          i.e. the number of valid records written to tcdi               *
   //*          -- If return value != tcsi.iFiles, then trashcan is partially  *
   //*             corrupt. '.trashinfo' records without matching item in      *
   //*             trashcan are not reported. Items in trashcan without        *
   //*             matching '.trashinfo' record are not reported.              *
   UINT  ctScanTrashcan ( tcSummaryInfo& tcsi, tcDetailInfo*& tcdi ) ;
   
   //* Query whether Trashcan has been located and is properly configured.     *
   //*                                                                         *
   //* Input  : none                                                           *
   //* Returns: 'true'  if Trashcan operations are available                   *
   //*          'false' if Trashcan operations have been disabled              *
   bool  ctTrashcanConfigured ( void ) { return ( this->trConfigured ) ; }

   //* Write text to the display. All output goes through these methods.       *
   void  textOut ( const gString& tOut, bool newln = true ) ;
   void  textOut ( const char*    tOut, bool newln = true ) ;
   void  textOut ( const wchar_t* tOut, bool newln = true ) ;


   private:

   //*********************
   //** Private Methods **
   //*********************
   //* Interpret user's command options and gather specified source-filenames. *
   bool  ctGetCommandLineArgs ( commArgs& ca ) ;
   bool  ctGetRM_Args ( commArgs& ca ) ;
   //* For 'rm' emulation mode: Move file(s) to trashcan.                      *
   short ctrmMoveToTrash ( rmIteract intOption, bool emptyDirs, bool recurse ) ;
   //* For restore operations, prompt user for which item(s) to restore.       *
   //* For empty-trash operations, prompt user for which item(s) to delete.    *
   UINT  ctPromptInteractive ( tcDetailInfo* tcDetail, UINT iTotal, bool restore ) ;
   //* For restore operations, scan all items for most-recently-deleted item(s)*
   UINT  ctrScanMostRecent ( tcDetailInfo* tcDetail, UINT iTotal ) ;
   //* For move-to-trashcan operations, validate a source item.                *
   bool  ctmValidateSourceItem ( short flIndex, UINT& fTotal, UINT& fPass, UINT64& totBytes ) ;
   //* For move-to-trashcan operations, test freespace on trashcan filesystem. *
   bool  ctmPromptFreespace ( UINT64 tBytes ) ;
   //* For move-to-trashcan operations, prompt for confirmation BEFORE moving. *
   bool  ctmPromptPreconfirm ( const gString& iCount, const gString& iFiles, 
                               const gString& iBytes, bool preConf, bool permDel ) ;
   bool  ctmPromptPreconfirm ( void ) ;
   //* For restore operations, prompt for confirmation BEFORE restoration.     *
   bool  ctrPromptPreconfirm ( const tcDetailInfo* tcDetail, UINT iTotal, UINT eCount ) ;
   //* For restore operations, validate alternate target if specified.         *
   bool  ctrResolveAltTarget ( const char* altTarget, gString& altPath, gString& altFName, 
                               fmFType& altFType, bool& altExist, UINT rTotal ) ;
   //* For restore operations, possible insufficient target freespace.         *
   bool  ctrPromptFreespace ( const gString& trgNode, UINT64 rBytes ) ;
   //* For restore operations, test for overwrite of existing target(s).       *
   bool  ctrTestOverwrite ( const tcDetailInfo* tcDetail, UINT iTotal, UINT& eCount, 
                            UINT& pCount, const gString& altDir, const gString& altName ) ;
   bool  ctrTestDirPermission ( const gString& srcPath, const gString& trgPath, UINT& eCount ) ;
   //* For restore operations, display restoration summary.                    *
   void  ctrSummaryReport ( UINT itemsRestored, UINT filesRestored ) ;
   //* Empty the trashcan: 1)completely, 2)interactively, 3)by freshness date. *
   short ctEmptyTrash ( bool interact, int remTime = ZERO ) ;
   //* Get user response to text-mode prompts.                                 *
   bool  ctUserResponse ( gString& gsIn ) ;
   //* Display the application's title, version and copyright info.            *
   void  ctDisplayAppVersion ( void ) ;
   //* Display command-line options *
   void  ctDisplayCommandLineHelp ( void ) ;


   //***-----------------------------------------***
   //***   Private Trashcan management methods   ***
   //***-----------------------------------------***
   //* Parse the deletion-date string from the '.trashinfo' file.              *
   void cttDecodeDate ( const gString& trashedDate, localTime& tDate ) ;
   //* Get summary statistics for items in the Trashcan.                       *
   bool  cttSummaryScan ( tcSummaryInfo& tcSummary ) ;
   //* Scan and translate the files: 'Trash/info/*.trashinfo'                  *
   UINT  cttDetailScan ( UINT iCount, tcDetailInfo*& tcdi ) ;
   //* Sort detailed trashcan records.                                         *
   void  cttSortRecords ( tcDetailInfo* tcrec, int tccnt, OutSort sOption ) ;
   //* Move the specified item to the trashcan.                                *
   bool  cttSendItemToTrash ( const FSpec& fSpec, const gString& tStamp, gString& eFile ) ;
   //* Restore an item from the trashcan                                       *
   bool  cttRestoreItemFromTrash ( const tcDetailInfo& tcdi ) ;
   //* Delete an item from the trashcan.                                       *
   bool  cttDeleteItemFromTrash ( const tcDetailInfo& tcdi ) ;
   //* Delete all data from the trashcan.                                      *
   bool  cttEmptyTrash ( void ) ;
   //* Determine the target filespec for item to be copied into Trash/files    *
   void  cttNextTargetName ( const gString& srcSpec, gString& trgSpec, gString& infoSpec ) ;
   //* Create .trashinfo record file describing the item sent to the trashcan. *
   bool  cttCreateInfoRecord ( const gString& srcSpec, 
                               const gString& infoSpec, const gString& tStamp ) ;
   //* Create a timestamp string to be used in the '.trashinfo' record.        *
   void  cttEncodeDate ( const localTime& lt, gString& tStamp ) ;
   //* Test the specified filetype against the list of supported filetypes.    *
   bool  cttTestFiletype ( fmFType fType ) ;

   //***-----------------------------------------***
   //***    Methods to read and modify files     ***
   //***-----------------------------------------***
   //* Get the user's current working directory.                               *
   short ctfGetCWD ( gString& dPath ) ;
   //* Determine whether target file exists AND if it is of the allowed type.  *
   bool  ctfTargetExists ( const gString& fPath, bool isDir = false, bool phys = false, 
                           UINT64* fsPtr = NULL, fmFType* ftPtr = NULL ) ;
   //* Stat (lstat) the target file, and return the formatted information.     *
   bool  ctfGetFileStats ( const gString& trgPath, DFileStats& dfs ) ;
   //* Extract file type from the file 'mode'.                                 *
   fmFType ctfDecodeFileType ( UINT mode ) ;
   //* Convert an 'epoch' (time_t) time code to its date/time components.      *
   void  ctfDecodeTimestamp ( localTime& lTime, time_t epochTime ) ;
   //* Verify that the file's attributes meet the criteria.                    *
   bool  ctfVerifyFileStats ( const DFileStats& dfs ) ;
   //* Convert specified path/filename to full path/filename.                  *
   bool  ctfRealpath ( const gString& rawPath, gString& realPath) ;
   //* Validate the path and component directories of the trashcan.            *
   bool  ctfValidateTrashPath ( const gString& rawPath, gString& realPath ) ;
   //* Perform environment or tilde ('~') expansion on the string.             *
   bool  ctfEnvExpansion ( gString& gsPath ) ;
   //* Create a unique path/directory name for application's temporary files.  *
   bool  ctfCreateTemppath ( void ) ;
   //* Create a unique path/filename for a temporary file.                     *
   bool  ctfCreateTempname ( gString& tmpPath ) ;
   //* Stub for C++17 function fs::temp_directory_path().                      *
   bool  ctfGetTempdirPath ( gString& tdPath ) ;
   //* Get the system timecode and convert it to localTime format.             *
   bool  ctfGetLocalTime ( localTime& lt ) ;
   //* Concatenate path and filename to create a full file spec.               *
   bool  ctfCatPathFilename ( gString& pgs, const gString& wPath, const char* uFile ) ;
   bool  ctfCatPathFilename ( gString& pgs, const gString& wPath, const wchar_t* wFile ) ;
   //* Trim (shorten) a path/filename string to fit the display space.         *
   bool  ctfTrimPathString ( gString& gsPath, short maxWidth ) ;
   //* Read file system stats for filesystem containing specified path/filename.*
   bool  ctfFilesystemStats ( const gString& trgPath, fileSystemStats& fsStats ) ;
   //* Read file system ID (hex)                                               *
   UINT64 ctfFileSystemID ( const gString& trgPath ) ;
   //* Get a summary of the contents of the specified directory.               *
   bool  ctfDirectorySummary ( const gString& dPath, UINT& fCount, 
                               UINT64& totBytes, bool recurse = false ) ;
   //* Verify user acces to all directory contents.                            *
   UINT  ctfValidateDirContents ( FSpec& fspec ) ;
   //* Open a directory file for reading.                                      *
   DIR*  ctfOpendir ( const gString& dirPath ) ;
   //* Extract the path from the path/filename provided.                       *
   void ctfExtractPathname ( gString& ePath, const gString& fPath ) ;
   //* Extract the filename from the path/filename provided.                   *
   void ctfExtractFilename ( gString& eName, const gString& fPath ) ;
   //* Extract the filename extension (including the '.') from  path/filename  *
   void ctfExtractFileExtension ( gString& eExt, const gString& fPath ) ;
   //* Copy a file of valid type, retaining all timestamps and persmissions.   *
   bool  ctfCopyFile ( const gString& srcPath, const gString& trgPath ) ;
   bool  ctfCopyFile ( const gString& srcPath, const gString& trgPath, const DFileStats& dfs ) ;
   //* Copy a regular or symlink file, retaining all timestamps/persmissions.  *
   bool  ctfCopyRegFile ( const gString& srcPath, const gString& trgPath, 
                          bool followLink = false ) ;
   //* Copy a top-level directory file (not its contents).                     *
   bool  ctfCopyDirName ( const gString& srcPath, const gString& trgPath, 
                          const DFileStats& srcStats ) ;
   //* Copy a FIFO-type file, retaining all timestamps and persmissions.       *
   bool  ctfCopyFifo ( const gString& srcPath, const gString& trgPath, 
                       const DFileStats& srcStats ) ;
   //* Rename (or move) the specified file.                                    *
   bool  ctfRenameFile ( const gString& srcPath, const gString& trgPath ) ;
   //* Delete ('unlink') the specified file.                                   *
   bool  ctfDeleteFile ( const gString& trgPath ) ;
   //* Create an empty directory file.                                         *
   bool  ctfCreateDirectory ( const gString& trgPath, const DFileStats* dfs = NULL ) ;
   //* Delete an empty directory file (or optionally, 1st delete any contents).*
   bool  ctfDeleteDirectory ( const gString& trgPath, bool delContents = false ) ;
   //* Copy the contents of source directory to target directory.              *
   UINT  ctfCopyDirContents ( const gString& srcPath, const gString& trgPath ) ;
   //* Delete the contents of the target directory.                            *
   UINT  ctfDeleteDirContents ( const gString& trgPath ) ;
   //* Set both Modify and Access timestamps for the target file.              *
   bool  ctfSetTimestamps ( const gString& trgPath, time_t modTime, time_t accTime ) ;
   //* Change directory to directory specified by full path string.            *
   bool  ctfCdPath ( const gString& dPath ) ;
   //* Read one line of text from the specified input stream.                  *
   bool  ctfReadLine ( ifstream& ifs, gString& gs ) ;

   #if DEBUG_METHODS != 0        // Development and debugging only
   void  cttDisplayRecord ( const tcDetailInfo& tcdi ) ;
   void  ctfDisplayFilesystemStats ( const gString& fPath, const fileSystemStats& fss ) ;
   #endif                        // DEBUG_METHODS

   //******************
   //** Private Data **
   //******************
   gString  cwDir ;        // Current-working-directory on start-up
   gString  tcDir ;        // Base trashcan directory (contains 'files' and 'info')
   gString  tmpDir ;       // Path/filename for directory containing temporary files
   gString  listFile ;     // Path/filename of list of files to be processed
   FSpec*   fileList ;     // Points to dynamic memory allocation for list of files to process
   int      procStatus ;   // Move to Trashcan: number of items (not files) successfully moved
                           // Restore from Trashcan: number of items (not files) successfully restored
                           // Empty Trashcan: number of items (not files) successfully removed
                           // Trashcan Report: number of items (not files) currently in Trashcan
   short    itemCount ;    // Number of filespecs in 'fileList'
   short    maxStatCols ;  // Maximum number of display columns for a displayed filespec
   OutSort  sortOption ;   // Sort option for display of Trashcan records
   bool     textMode ;     // 'true' if Text Mode (default), 'false' if Interactive (dialog) Mode
   bool     trConfigured ; // 'true' if Trashcan properly configured
   bool     silent ;       // 'true' if writing no output to display
   bool     quiet ;        // 'true' if writing no output to display EXCEPT error reports
   bool     verbose ;      // 'true' if verbose (detailed) output to display
   bool     confirm ;      // 'true' if confirm before operations that cannot be undone

} ;   // End cTrash class

