//******************************************************************************
//* File       : cTrash.cpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2015-2020 Mahlon R. Smith, The Software Samurai *
//*                 GNU GPL copyright notice below                             *
//* Date       : 18-Jan-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: Main module for Command-line Trashcan utility, 'ctrash'.      *
//* which allows the command-line user AND command-line utilities to access    *
//* the system's local (user-owned) Trashcan directories.                      *
//*                                                                            *
//* The full documentation for 'ctrash' is located in 'ctrash.info' and        *
//* 'ctrash.html'.                                                             *
//*    Texinfo-format docs: info -f ctrash.info                                *
//*    HTML-format docs   : open 'ctrash.html' in your favorite browser        *
//*                                                                            *
//* Copyright Notice:                                                          *
//* =================                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.0.06 17-Jan-2020                                                      *
//*   -- Update compiler to g++ v:9.2.1:                                       *
//*      -- New warning that library functions 'tempnam' and 'tmpnam_r are     *
//*         dangerous. While this is technically true, this application uses   *
//*         them in such a way that negates the danger.                        *
//*      -- Still, we seriously dislike compiler warnings, even when we are    *
//*         "sure" that they do not apply to us. See ctfGetTempdirPath(),      *
//*         ctfCreateTemppath() and ctfCreateTempname() methods for our        *
//*         solution.                                                          *
//*      -- Import updated version of the gString class from NcDialogAPI.      *
//*         No change in functionality.                                        *
//*   -- Update the copyright message displayed by the '--version' option.     *
//*   -- Documentation update.                                                 *
//*   -- Posted to website 17-Jan 2020                                         *
//*                                                                            *
//* v: 0.0.05 26-Feb-2016                                                      *
//*    -- Update the copyright message displayed by the '--version' option.    *
//*       (not pubically released)                                             *
//*                                                                            *
//* v: 0.0.04 07-Oct-2015                                                      *
//*   - Generate error message if filename specified by '--file-list' option   *
//*     was not found.                                                         *
//*   - Update calls to 'formatInt' and 'compare' methods of the gString class *
//*     to match changes to gString prototypes (no functionality changed).     *
//*   - Update algorithm for creating temp-file names, to reduce the likelihood*
//*     of duplicate filenames.                                                *
//*   - Update copyright message.                                              *
//*   - Bug fix: When trimming path strings for display, a filename wider than *
//*     the available field width was causing an out-of-bounds error.          *
//*   - Posted to website 17-Jan-2016                                          *
//*                                                                            *
//* v: 0.0.03 06-Oct-2015                                                      *
//*   - Complete implementation of 'rm' compatibility mode.                    *
//*   - Several bug fixes in restoration algorithm.                            *
//*   - Integrate suggestions from testers.                                    *
//*   - Documentation update.                                                  *
//*   - Posted to website 06-Oct-2015                                          *
//*                                                                            *
//* v: 0.0.02 02-Sep-2015                                                      *
//*   - Restructure source item validation to significantly reduce the number  *
//*     of 'lstat' calls.                                                      *
//*   - Complete recursive move to and from trashcan for directory trees.      *
//*   - Sort detailed-record output (cttSortRecords method).                   *
//*   - Pretest free space on target filesystem (trashcan's filesystem         *
//*     and filesystems for directory tree restoration).                       *
//*                                                                            *
//* v: 0.0.01 22-Apr-2015                                                      *
//*   - Adapt the Trashcan-specific code from the FileMangler utility to       *
//*     produce a simple command-line utility for accessing the Trashcan.      *
//*   - All output passes through a single point, so that an NcDialog user     *
//*     interface can be added later. This is also useful in case we want to   *
//*     adapt the code as a link library for other applications.               *
//*                                                                            *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* -- See cTrashCan.cpp for background information of how trashcan data are   *
//*    generated and processed.                                                *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *
//* =================                                                          *
//* == To-do List: ==                                                          *
//* =================                                                          *
//* -- Automate the build and install using a script file.                     *
//* -- Structure application to make it easy to build as a link library.       *
//*    -- Alternate application-layer methods that produce data on the         *
//*       operation, but produce no direct output.                             *
//*    -- redirect textOut() to memory or to a file?                           *
//* -- Add optional ANSI color to output (unneeded if NcDialog used)           *
//*    Additional parameter to enable/disable color output                     *
//*    This can be done in one or both:                                        *
//*     -- Add color parameter to textOut() methods.                           *
//*        See Wayclip app for an example.                                     *
//*     -- Embed ANSI color directly into the strings.                         *
//*    '--color-yes' option  (ls uses --color[=[ | none | auto | always]]      *
//*    '--color-no' option                                                     *
//*    'ls' uses blue for directories                                          *
//*              brown for fifos                                               *
//*              black for regulars                                            *
//*              green for non-dir executables                                 *
//*              cyan for symlinks                                             *
//*              brown-on-black for char devices                               *
//*              yellow-on-black for block devices                             *
//*              and some sticky-bit crap colors                               *
//* -- Add support for additional languages:                                   *
//*    -- Zhongwen                                                             *
//*    -- Espanol                                                              *
//* --                                                                         *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "cTrash.hpp"


//*********************
//* Local Definitions *
//*********************
#define DEBUG_OPTIONS (0)
#if DEBUG_OPTIONS != 0
void EncodeDelDate ( localTime& lt ) ;
short sddval = ZERO ;
char  sddunit = ' ' ;
#endif // DEBUG_OPTIONS

//****************
//* Local Data   *
//****************
static const char* const defaultTRASHDIR = "%S/.local/share/Trash" ;
static const char* const NoFiles = "No source filename(s) specified." ;
static const wchar_t* const badLIST = L"Unable to verify list of files to be processed. " ;


//*************************
//*         main          *
//*************************
//******************************************************************************
//* Program entry point.                                                       *
//*                                                                            *
//* Command-line Usage:                                                        *
//*  See ctDisplayCommandLineHelp method for list of valid arguments.          *
//*  See ctGetCommandLineArgs method for command-line argument processing.     *
//*                                                                            *
//* Returns: number of items (NOT the number of files) successfully processed  *
//*           Note: If there is a processing error, then some specified items  *
//*                 may remain unprocessed.                                    *
//*                                                                            *
//*          Operation               Returns                                   *
//*          ----------------------  ----------------------------------------  *
//*          Move to Trashcan:        number of items successfully moved       *
//*          Restore from Trashcan:   number of items successfully restored    *
//*          Empty Trashcan:          number of items successfully deleted     *
//*          Trashcan Report:         number of items currently in Trashcan    *
//*                                                                            *
//*          Returns (-1, i.e. pre-processing error) if:                       *
//*            a) one or more invalid command-line options, or invalid         *
//*               sub-option for options that require them                     *
//*            b) one or more specified source files not found, or inaccessible*
//*            c) user aborted the operation before processing                 *
//*            d) location for creating temporary files not found (unlikely)   *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   //* Gather our entry-sequence data *
   commArgs clArgs( argc, argv, argenv ) ;

   //* Create the application class object *
   cTrash ctrash( clArgs ) ;

   return ( int(ctrash.ctProcStatus()) ) ;

}  //* End main() *

//*************************
//*        ~cTrash        *
//*************************
//******************************************************************************
//* Destructor. Return all resources to the system.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

cTrash::~cTrash ( void )
{
   //* If we have a dynamic allocation of      *
   //* filespec records, release it.           *
   if ( this->fileList != NULL )
      delete [] this->fileList ;

   //* Delete all temporary files and the      *
   //* temporary directory which contains them.*
   if ( (this->ctfTargetExists ( this->tmpDir )) != false )
   {
      this->ctfDeleteDirectory ( this->tmpDir, true ) ;
   }

}  //* End ~cTrash() *

//*************************
//*        cTrash         *
//*************************
//******************************************************************************
//* Default constructor.                                                       *
//*                                                                            *
//* Input  : commArgs class object (by reference)                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

cTrash::cTrash ( commArgs& ca )
{
   //* Set the locale to the environment's choice i.e. UTF-8 encoding.*
   //* This affects all input and output streams.                     *
   std::locale::global(std::locale(""));
   #if 0    // DEBUGGING ONLY
   wcout << L"ioLocale:" << std::locale().name().c_str() << endl ;
   #endif   // DEBUGGING ONLY

   //* Initialize our data members *
   this->procStatus = (-1) ;     // initially indicates a pre-processing error
   this->textMode = true ;       // currently the ONLY mode
   this->trConfigured = false ;  // trashcan configuration not yet verified
   this->sortOption = osDATE ;   // default sort option
   this->silent = this->quiet = this->verbose = false ;
   this->confirm = true ;
   this->fileList = NULL ;
   this->itemCount = ZERO ;
   //* Limit width of a displayed filespec.            *
   //* (5 col for output prefix and 2 cols for safety) *
   this->maxStatCols = (fswDEFAULT - 7) ;

   //* Determine where temp files are stored on the system and create a  *
   //* tempfile subdirectory there. We must do this before reading the   *
   //* command-line args because that is where the list of files to be   *
   //* processed will live. Then read the command-line arguments.        *
   bool validArgs = false ;
   if ( (this->ctfCreateTemppath ()) != false )
      validArgs = this->ctGetCommandLineArgs ( ca ) ;
   else
      ca.errMsg.append( "Directory for creating temporary file is inaccessible." ) ;

   //* If valid user input, and not a cry for help *
   if ( (validArgs != false) && (ca.helpFlag == false) && (ca.verFlag == false) ) 
   {  //* Finish initialization of our data members.*
      this->ctfGetCWD ( this->cwDir ) ;   // get path of current-working directory
      this->tcDir = ca.trashDir ;         // copy trashcan path to our private data member
      this->trConfigured = true ;         // trashcand configuration has been verified
      this->textMode = ca.textFlag ;
      this->sortOption = ca.sortOption ;
      this->silent   = ca.silent ;
      this->quiet    = ca.quiet ;
      this->verbose  = ca.verbose ;
      this->confirm  = ca.confirm ;
      if ( (ca.dispWidth >= fswMIN) && (ca.dispWidth <= fswMAX) )
         this->maxStatCols = (ca.dispWidth - 7) ;
      this->procStatus = ZERO ;           // prepare for a successful operation

      //* Application title *
      if ( !this->silent && !this->quiet )
      {
         gString gsOut( "\n%S%S%S\n%S", AppTitle1, AppVersion, AppTitle2, AppTitle3 ) ;
         this->textOut ( gsOut ) ;
      }

      //* Capture the list of items to be processed *
      if ( ca.itemCount > ZERO )
      {  //* Allocate an array of records to hold the filespecs.        *
         //* (we may allocate more records than we need, but not fewer) *
         this->fileList = new FSpec[ca.itemCount] ;
         ifstream ifs( this->listFile.ustr(), ifstream::in ) ;
         if ( ifs.is_open() )
         {
            gString gs ;
            while ( (this->itemCount < ca.itemCount) && 
                    ((this->ctfReadLine ( ifs, gs )) != false) )
            {
               while ( (gs.gstr()[0]) == SPACE )   // eliminate leading whitespace
                  gs.shiftChars( -1 ) ;
               if ( (gs.gschars() > 1) && (*gs.gstr() != L'#') )
                  this->fileList[this->itemCount++].fSpec = gs ;
            }
            ifs.close() ;
         }
         else
         {  //* This is rather unlikely.*
            validArgs = false ;
            if ( !this->silent )
            {
               ca.errMsg.append( badLIST ) ;
               ca.errMsg.append( L'\n' ) ;
               this->textOut ( ca.errMsg ) ;
            }
         }
      }


      //******************************
      //** Grant the user's request **
      //** ( see option logic in  ) **
      //** ( ctGetCommandLineArgs ) **
      //******************************

      //* Report contents of trashcan, summary or detailed. *
      if ( ca.report != false )
         this->procStatus = this->ctReportTrashcan ( ca.details ) ;

      //* Restore files from the trashcan, either most recent *
      //* item(s) or through interactive item selection.      *
      else if ( ca.restore != false )
         this->procStatus = this->ctRestoreFromTrash ( ca.details, ca.preconf, 
                                                       ca.altTarget ) ;

      //* If operation is to send files to the trashcan, then *
      //* one or more source files must have been specified.  *
      else if ( this->itemCount > ZERO )
      {  //* Call the appropriate move-to-trashcan method for     *
         //* cTrash standard mode, or 'rm' command emulation mode.*
         if ( ca.rm_mode )
            this->procStatus = this->ctrmMoveToTrash ( ca.rm_int, ca.rm_dir, ca.rm_rec ) ;
         else
            this->procStatus = this->ctMoveToTrash ( ca.preconf, ca.permdel, ca.altTarget ) ;
      }

      //* Empty the trashcan, either completely or with interactive *
      //* item selection, or according to freshness date.           *
      else if ( ca.empty != false )
      {
         #if 0    // TEMP - Testing public ctEmptyTrash method
         // Note: remTime is in minutes, but called method expects days or hours.
         bool days = true ;
         if ( ca.remTime > ZERO )
            ca.remTime = ca.remTime / (24 * 60) ;
         else if ( ca.remTime < ZERO )
         {
            ca.remTime = ca.remTime / 60 ;
            days = false ;
         }
         this->procStatus = this->ctEmptyTrash ( ca.details,  days, ca.remTime ) ;
         #else    // Production Method
         this->procStatus = this->ctEmptyTrash ( ca.details, ca.remTime ) ;
         #endif   // Testing public ctEmptyTrash method
      }

      //* Write a blank line before exit *
      if ( !this->silent )
      { gString gsOut ; this->textOut ( gsOut ) ; }
   }           // valid user input

   else        // explicit or implied cry for help
   {
      if ( ca.verFlag != false )
      {
         this->ctDisplayAppVersion () ;
         this->procStatus = ZERO ;     // version request is not an error
      }
      else if ( ca.helpFlag != false )
      {
         this->ctDisplayCommandLineHelp () ;
         if (validArgs != false)       // _explicit_ help request is not an error
            this->procStatus = ZERO ;
      }
      else if ( !this->silent )        // if not 'silent' mode, report command-line errors
      {
         ca.errMsg.append( "\nFor command-line options: 'ctrash --help'\n" ) ;
         gString gsOut( "\n%S%S%S\n%S", AppTitle1, AppVersion, AppTitle2, AppTitle3 ) ;
         this->textOut ( gsOut ) ;
         this->textOut ( ca.errMsg ) ;
      }
   }

}  //* End cTrash() *

//*************************
//*   ctReportTrashcan    *
//*************************
//******************************************************************************
//* Report the contents of the trashcan directory.                             *
//*                                                                            *
//* Input  : details    : 'false' summary report                               *
//*                       'true'  detailed report                              *
//*                                                                            *
//*                                                                            *
//* Returns: number if items currently in trashcan (not the file count)        *
//*          ERR (-1) if one or more pre-processing errors                     *
//******************************************************************************

short cTrash::ctReportTrashcan ( bool details )
{
   #define DEBUG_SORT (0)  // Set to (1) to debug sort algorithm, else (0)

   const wchar_t* Hdr[] = 
   {
      L"\nTrashcan Summary: \n------------------------------------------------",
      L"LOCATION: ",
      L"       ITEMS: ",
      L"       FILES: ",
      L"  TOTAL SIZE: ",
      L"SIZE ON DISK: ",
      L"FS FREESPACE: ",
   } ;
   const short fieldWIDTH = 6 ;           // formatted-integer field width
   tcSummaryInfo tcSummary ;              // summary data
   tcDetailInfo* tcDetail = NULL ;        // pointer to detail data
   UINT  dItems ;                         // value returned by detail scan
   short hdrIndex = ZERO ;                // index into the summary header array
   gString gsOut, gsInt ;                 // data formatting
   short status = ERR ;                   // return value
   bool  tcCorrupt = false ;              // 'true' if trashcan corrupted

   if ( (this->cttSummaryScan ( tcSummary )) != false )
      status = OK ;

   if ( (status == OK) && (tcSummary.items > ZERO) && (details != false) )
   {
      dItems = this->cttDetailScan ( tcSummary.iFiles, tcDetail ) ;
      tcCorrupt = bool(dItems != tcSummary.iFiles) ;

      this->textOut ( L"Trashcan Contents\n"
                       "----------------------------------------"
                       "----------------------------------------" ) ;
      #if DEBUG_SORT == 0     // Production Build
      for ( UINT index = ZERO ; index < dItems ; ++index )
         this->textOut ( tcDetail[index].dRecord ) ;

      #else    // DEBUG_SORT
      for ( UINT index = ZERO ; index < dItems ; ++index )
      {
         gString gsn ;
         this->ctfExtractFilename ( gsn, tcDetail[index].trgPath ) ;
         gsInt.formatInt( tcDetail[index].size, fieldWIDTH ) ;
         gsOut.compose( L"%3u) %04hd-%02hd-%02hd_%02hd:%02hd:%02hd  %S  %S", 
            &index, &tcDetail[index].tDate.year, &tcDetail[index].tDate.month, 
            &tcDetail[index].tDate.date, &tcDetail[index].tDate.hours, 
            &tcDetail[index].tDate.minutes, &tcDetail[index].tDate.seconds,
            gsInt.gstr(), gsn.gstr() ) ;
         this->textOut ( gsOut ) ;
      }
      #endif   // DEBUG_SORT
   }

   #if DEBUG_SORT == 0     // Production Build
   if ( status == OK )
   {
      this->textOut ( Hdr[hdrIndex++] ) ;
      this->textOut ( Hdr[hdrIndex++], false ) ;
      this->textOut ( this->tcDir ) ;
      this->textOut ( Hdr[hdrIndex++], false ) ;
      gsInt.formatInt( tcSummary.items, fieldWIDTH ) ;
      this->textOut ( gsInt ) ;
      this->textOut ( Hdr[hdrIndex++], false ) ;
      gsInt.formatInt( tcSummary.files, fieldWIDTH ) ;
      this->textOut ( gsInt ) ;
      this->textOut ( Hdr[hdrIndex++], false ) ;
      gsInt.formatInt( tcSummary.bytes, fieldWIDTH ) ;
      this->textOut ( gsInt ) ;
      this->textOut ( Hdr[hdrIndex++], false ) ;
      gsInt.formatInt( tcSummary.tSpace, fieldWIDTH ) ;
      this->textOut ( gsInt ) ;
      this->textOut ( Hdr[hdrIndex++], false ) ;
      gsInt.formatInt( tcSummary.fSpace, fieldWIDTH ) ;
      this->textOut ( gsInt ) ;
      status = tcSummary.items ;    // return number of items scanned
   }
   if ( tcCorrupt != false )
      this->textOut ( L"NOTE: Trashcan data may be partially corrupted!" ) ;
   #endif   // Production Build

   if ( tcDetail != NULL )       // if we have a dynamic allocation, release it
      delete [] tcDetail ;

   return status ;

   #undef DEBUG_SORT
}  //* End ctReportTrashcan() *

//*************************
//*     ctMoveToTrash     *
//*************************
//******************************************************************************
//* Move file(s) from the original position on the filesystem to the user's    *
//* local trashcan (or to the alternate trashcan directory, if specified).     *
//*                                                                            *
//* User access is verified for ALL items before ANY item is moved.            *
//*                                                                            *
//*                                                                            *
//* Input  : preconfirm: 'false' no confirmation required                      *
//*                      'true'  ask for user confirmation before operation    *
//*          permDelete: 'false' move source files to trashcan                 *
//*                      'true'  bypass trashcan and unlink the file(s)        *
//*          altTarget : (optional, NULL pointer by default)                   *
//*                      -- If NULL pointer (or empty string) indicates        *
//*                         restore to original position.                      *
//*                      -- If specified:                                      *
//*                         a) path of alternate target directory              *
//*                            (directory must exist)                          *
//*                         b) full path/filename for alternate target         *
//*                            (if target already exists, confirm overwrite)   *
//*                                                                            *
//* Returns: number of items succesfully moved to trashcan (not the file count)*
//*          ZERO if user aborted the operation                                *
//*          ERR (-1) if one or more pre-processing errors                     *
//******************************************************************************

short cTrash::ctMoveToTrash ( bool preconfirm, bool permDelete, const char* altTarget )
{
   gString srcItem, gsOut, gsIn ;
   UINT64 totalBytes = ZERO ;    // total bytes of data to be moved
   UINT   totalFiles = ZERO,     // total of all items and their contents
          totalPass  = ZERO,     // total of all validated items and their contents
          itemsMoved = ZERO ;    // number of items successfully moved to trashcan
   bool   userAbort = false ;    // true if user aborted the operation (or gave us bad input)
   short  flIndex = ZERO,        // index into processing list
          status = ERR ;         // return value

   //* Step through the list of items to be processed and validate each item.  *
   //* See notes in header of ctmValidateSourceItem().                         *
   for ( flIndex = ZERO ; (flIndex < this->itemCount) && !userAbort ; ++flIndex )
   {
      userAbort = this->ctmValidateSourceItem ( flIndex, totalFiles, totalPass, totalBytes ) ;
   }

   //* If all source items validated *
   if ( !userAbort )
   {  //* Format totals for output *
      gString ti, tf, tb ;
      ti.formatInt( (UINT)this->itemCount, 5, true ) ;
      tf.formatInt( (USHORT)totalFiles, 5, true ) ;
      tb.formatInt( totalBytes, 5, true ) ;
      //* A bit of beautification *
      if ( totalBytes < 100000 )
         tb.append( L' ' ) ;
      status = ZERO ;               // hope for the best

      //* Check freespace on filesystem which contains trashcan.*
      if ( !this->silent && !this->quiet && this->confirm )
            userAbort = this->ctmPromptFreespace ( totalBytes ) ;

      //* If caller specified that we should ask for  *
      //* confirmation before moving data to trashcan,*
      //* OR if we are about to unlink the items.     *
      if ( !userAbort )
      {
         if ( !this->silent && (preconfirm || (permDelete && this->confirm)) )
         {
            userAbort = this->ctmPromptPreconfirm ( ti, tf, tb, preconfirm, permDelete ) ;
         }
         else if ( this->verbose )
         {
            //* List the individual items and the summary stats *
            for ( flIndex = ZERO ; flIndex < this->itemCount ; ++flIndex )
               this->textOut ( this->fileList[flIndex].fSpec ) ;
            gsOut.compose( "\nMoving these %S items (%S files, %Sbytes) to Trashcan.", 
                           ti.gstr(), tf.gstr(), tb.gstr() ) ;
            this->textOut ( gsOut ) ;
         }
      }

      //* Data verified and user confirmation received. *
      //* Move specified items to the trashcan.         *
      gString eFile ;               // if error, receives offending filespec
      flIndex = ZERO ;              // reset the index
      if ( !userAbort )
      {
         //* Create the deletion-date timestamp    *
         //* string for all items to be processed. *
         gString tStamp ;
         localTime lt ;
         this->ctfGetLocalTime ( lt ) ;
         #if DEBUG_OPTIONS != 0  // For Development ONLY
         EncodeDelDate ( lt ) ;
         #endif // DEBUG_OPTIONS
         this->cttEncodeDate ( lt, tStamp ) ;

         //* Move each item to the trashcan. (abort on error) *
         for ( flIndex = ZERO ; flIndex < this->itemCount ; ++flIndex )
         {
            if ( (this->cttSendItemToTrash ( this->fileList[flIndex], 
                                             tStamp, eFile )) != false )
            {
               ++itemsMoved ;
            }
            else
               break ;
         }
         status = itemsMoved ;      // return number of iems moved to trashcan
      }

      //* Report success of operation *
      if ( status >= ZERO && !this->quiet )
      {
         if ( !userAbort )
         {
            if ( !permDelete )
               gsOut.compose( "OK: %S items (%S files) moved to Trashcan.", 
                              ti.gstr(), tf.gstr() ) ;
            else
               gsOut.compose( "OK: %S items (%S files) permanently deleted.", 
                              ti.gstr(), tf.gstr() ) ;
         }
         else
            gsOut = "Operation cancelled. Source items unchanged." ;
         this->textOut ( gsOut ) ;
      }
      //* Else, report operational errors. This is moderately  *
      //* unlikely because we pre-verified all the source data.*
      else if ( status == ERR && !this->silent )
      {
         this->textOut ( L"ERROR! Operation terminated unexpectedly.\n"
                          "       Error occurred in item:\n       ", false ) ;
         this->textOut ( eFile ) ;
         UINT notMoved = this->itemCount - itemsMoved ;
         gString im, in ;
         im.formatInt( (UINT)itemsMoved, 5 ) ;
         in.formatInt( (UINT)notMoved, 5 ) ;
         gsOut.compose( "%S items moved to Trashcan\n"
                        "%S source items not processed",
                        im.gstr(), in.gstr() ) ;

         //* Report unmoved items *
         if ( this->verbose && (flIndex < this->itemCount) )
         {
            this->textOut ( L"\nUnprocessed Items:" ) ;
            while ( flIndex < this->itemCount )
               this->textOut ( this->fileList[flIndex++].fSpec.gstr() ) ;
         }
      }
   }     // (all validated)
   return status ;

}  //* End ctMoveToTrash() *

//*************************
//*    ctrmMoveToTrash    *
//*************************
//******************************************************************************
//* For 'rm' emulation mode: Move file(s) to trashcan.                         *
//*                                                                            *
//* Input  : intOption : member of enum rmInteract, confirmation option        *
//*          emptyDirs : 'false' do not process directories                    *
//*                              (unless 'recurse' != false)                   *
//*                      'true'  process empty directories as regular files    *
//*          recurse   : 'false' do not process non-empty directories          *
//*                      'true'  process all directories as regular files      *
//*                                                                            *
//* Returns: ZERO if all items successfully moved to trashcan                  *
//*          ERR (-1) if error(s) before or during processing                  *
//******************************************************************************
//* Notes:                                                                     *
//* -- If no source items, just display a message and return.                  *
//* -- 'rm' will override write-protection with user permission.               *
//*    cTrash will not override protection on items or their contents.         *
//* -- 'rm' does preconfirm only if 4 or more items.                           *
//*     cTrash always preconfirms, if specified, without regard to item count. *
//* -- 'rm' only knows the default trashcan, so no alternates allowed.         *
//*    cTrash also uses only the default trashcan in emulation mode.           *
//* -- 'rm' prompts for confirmation using:                                    *
//*         rm: remove regular file 'fname'? _                                 *
//*         rm: remove fifo 'fname'? _                                         *
//*         rm: remove symbolic link 'fname'? _                                *
//*         rm: remove directory 'dname'? _          (empty directories)       *
//*         rm: descend into directory 'dname'? _    (non-empty directories)   *
//*    cTrash follows this pattern, except that 'rm' will attempt any filetype,*
//*    but cTrash will fail on pretest for unsupported types.                  *
//* -- 'rm' returns 0 on total success                                         *
//*    cTrash also returns 0 on total success                                  *
//*    'rm' returns the value 255 (decimal) if partial or total failure        *
//*         (this is -1 for a byte value, but all coreutils should return int) *
//*    cTrash returns ERR (int -1) on partial or total failure.                *
//* -- 'rm' in verbose mode when deleting a non-empty directory, reports each  *
//*     file contained in the base directory, then reports the base directory. *
//*    cTrash reports only the base directory along with the total number of   *
//*    file removed. This is much less messy.                                  *
//* -- 'rm' isn't nearly so careful as cTrash about data protection, AND       *
//*      'rm' doesn't have to worry about the user trying to send the trashcan *
//*       or the root directory to the trash.                                  *
//*    cTrash reports such an error as a "general protection error"            *
//*      If the user wants more specific information they can run cTrash in    *
//*      standard (non-emulation) mode.                                        *
//* -- 'rm' reports the raw, relative path for the item as specified on the    *
//*    command line, while cTrash gets the full path/filename spec if the item *
//*    exists. For this reason, we capture a list of the raw specs provided    *
//*    by the user, which we then use for confirmations and reports.           *
//* -- We set the 'silent' flag so the validation routine won't write error    *
//*    messages to the display.                                                *
//* -- In order to check freespace on the filesystem containing the trashcan,  *
//*    we need to scan the list twice. We're not sure this is worth the extra  *
//*    processing, so we may decide to disable the pre-validation.             *
//*                                                                            *
//*                                                                            *
//* SAMPLE 'rm' OUTPUT                                                         *
//* ==================                                                         *
//* NOTE: rm does not abort on fail, but continues for each item in the list.  *
//*                                                                            *
//* NOTE: If no arguments specified                                            *
//  [altTarg]$ rm                                                              *
//  rm: missing operand                                                        *
//  Try 'rm --help' for more information.                                      *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: Unrecognized option                                                  *
//  [Texinfo]$ rm --quiet                                                      *
//  rm: unrecognized option '--quiet'                                          *
//  Try 'rm --help' for more information.                                      *
//  [Texinfo]$                                                                 *
//  [Texinfo]$ rm -z                                                           *
//  rm: invalid option -- 'z'                                                  *
//  Try 'rm --help' for more information.                                      *
//  [Texinfo]$                                                                 *
//*                                                                            *
//* NOTE: Unrecognized argument for '--interactive' (only first char tested)   *
//*       (Our emulated message is a bit simpler.)                             *
//  [altTarg]$ rm --interactive=jump a.txt                                     *
//  rm: invalid argument ‘jump’ for ‘--interactive’                            *
//  Valid arguments are:                                                       *
//    - ‘never’, ‘no’, ‘none’                                                  *
//    - ‘once’                                                                 *
//    - ‘always’, ‘yes’                                                        *
//  Try 'rm --help' for more information.                                      *
//  [altTarg]$                                                                 *
//*                                                                            *
//  NOTE: Source item does not exist                                           *
//  [altTarg]$ rm s.txt                                                        *
//  rm: cannot remove ‘s.txt’: No such file or directory                       *
//  [altTarg]$                                                                 *
//  [altTarg]$ -i rm s.txt                                                     *
//  rm: cannot remove ‘s.txt’: No such file or directory                       *
//  [altTarg]$                                                                 *
//                                                                             *
//  NOTE: Confirm for each item: (Abc is a non-empty dir)                      *
//  [altTarg]$ rm -i a.txt Abc                                                 *
//  rm: remove regular file ‘a.txt’? n                                         *
//  rm: cannot remove ‘Abc’: Is a directory                                    *
//*                                                                            *
//  NOTE: Confirm for each item: (Abc is a non-empty dir)                      *
//  [altTarg]$ rm -i --dir a.txt Abc                                           *
//  rm: remove regular file ‘a.txt’? n                                         *
//  rm: cannot remove ‘Abc’: Directory not empty                               *
//*                                                                            *
//  NOTE: Confirm for each item: (Abc is a non-empty dir)                      *
//  [altTarg]$ rm -i --recursive a.txt Abc                                     *
//  rm: remove regular file ‘a.txt’? n                                         *
//  rm: descend into directory ‘Abc’? n                                        *
//*                                                                            *
//  NOTE: Confirm for each item: (Abc is a non-empty dir, Bcd is empty dir)    *
//  [altTarg]$ rm -i --dir a.txt Bcd Abc                                       *
//  rm: remove regular file ‘a.txt’? n                                         *
//  rm: remove directory ‘Bcd’? n                                              *
//  rm: cannot remove ‘Abc’: Directory not empty                               *
//*                                                                            *
//* NOTE: Target is a regular file                                             *
//  [altTarg]$ rm -i a.txt                                                     *
//  rm: remove regular file ‘a.txt’? y                                         *
//*                                                                            *
//* NOTE: Target is a write-protected regular file                             *
//  [cTrash]$ rm ./altTarg/f.txt                                               *
//  rm: remove write-protected regular file ‘./altTarg/f.txt’? n               *
//  [cTrash]$                                                                  *
//*                                                                            *
//* NOTE: Target is a FIFO file.                                               *
//  [altTarg]$ rm -i ../Abc/xys.fifo                                           *
//  rm: remove fifo ‘../Abc/xys.fifo’? n                                       *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: Target is a symbolic link                                            *
//  [altTarg]$ rm -i ../Abc/xyr.link                                           *
//  rm: remove symbolic link ‘../Abc/xyr.link’? n                              *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: Target is an empty directory                                         *
//  [altTarg]$ rm -i --dir Bcd                                                 *
//  rm: remove directory ‘Bcd’? n                                              *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: Target is an non-empty directory                                     *
//  [altTarg]$ rm -i --recursive Abc                                           *
//  rm: descend into directory ‘Abc’? n                                        *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: Verbose output confirms deletion.                                    *
//  [altTarg]$ rm -iv b.txt                                                    *
//  rm: remove regular file ‘b.txt’? y                                         *
//  removed ‘b.txt’                                                            *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: Verbose output when removing an empty directory:                     *
//  [altTarg]$ rm -ivd Bcd                                                     *
//  rm: remove directory ‘Bcd’? y                                              *
//  removed directory: ‘Bcd’                                                   *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: Verbose output when removing an non-empty directory:                 *
//  [altTarg]$ rm -Ivr ../altTarg/Abc                                          *
//  rm: remove all arguments recursively? y                                    *
//*     This is followed by a report on each file and subdirectory, ending     *
//*     with a report of the base directory.                                   *
//*                                                                            *
//* NOTE: Output when removing an non-empty directory (no prompt, not verbose):*
//*       No additional output.                                                *
//  [altTarg]$ rm --recursive ../altTarg/Abc                                   *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: Verbose output when removing a FIFO file:                            *
//  [altTarg]$ rm -ivd ../altTarg/Abc/xys.fifo                                 *
//  rm: remove fifo ‘../altTarg/Abc/xys.fifo’? y                               *
//  removed ‘../altTarg/Abc/xys.fifo’                                          *
//  [altTarg]$                                                                 *
//*                                                                            *
//  NOTE: The '--interactive=once' and '-I' will not confirm for 3 or fewer    *
//        items, only with >= 4 items. (as designed, but stupid)               *
//  [altTarg]$ rm --interactive=once c.txt d.txt e.txt f.txt                   *
//  rm: remove all arguments? n                                                *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: f.txt is write-protected:                                            *
//  [altTarg]$ rm -i f.txt                                                     *
//  rm: remove write-protected regular file ‘f.txt’? n                         *
//*                                                                            *
//* NOTE: f.txt is read-protected (rm ignores this, cTrash will not)           *
//  [altTarg]$ rm -i f.txt                                                     *
//  rm: remove regular file ‘f.txt’? y                                         *
//*                                                                            *
//* NOTE: f.txt is read and write protected (with and without interaction):    *
//  [altTarg]$ rm -i f.txt                                                     *
//  rm: remove write-protected regular file ‘f.txt’? n                         *
//  [altTarg]$ rm f.txt                                                        *
//  rm: remove write-protected regular file ‘f.txt’? n                         *
//  [altTarg]$                                                                 *
//*                                                                            *
//* NOTE: The 'altTarg' directory is write-protected (a.txt is not protected): *
//  [cTrash]$ rm ./altTarg/a.txt                                               *
//  rm: cannot remove ‘./altTarg/a.txt’: Permission denied                     *
//  [cTrash]$                                                                  *
//*                                                                            *
//* RETURN VALUES:                                                             *
//* --------------                                                             *
//* success all     :    0                                                     *
//* success partial :  256                                                     *
//* failure total   :  256                                                     *
//* user denies all :    0                                                     *
//* user denies some:    0                                                     *
//******************************************************************************

short cTrash::ctrmMoveToTrash ( rmIteract intOption, bool emptyDirs, bool recurse )
{
   gString srcItem,              // source item full path/filename spec
           gsOut,                // formatted output
           gsIn,                 // user response
           eFile ;               // if error, receives offending filespec (not used here)
   UINT64 totalBytes = ZERO ;    // total bytes of data to be moved
   UINT   totalFiles = ZERO,     // total of all items and their contents
          totalPass  = ZERO,     // total of all validated items and their contents
          itemsMoved = ZERO,     // number of items successfully moved to trashcan
          failCount = ZERO ;     // number of source items that failed validation
   short flIndex = ZERO,         // index into processing list
         status = ZERO ;         // return value (assume success)
   bool  invItem,                // true if item NOT validated
         skipItem = false,       // true if user indicates to skip an item
         userAbort = false ;     // true if user aborted the operation (or gave us bad input)

   this->silent = true ;         // silence the validation method (see note above)

   #if 0    // TEMP - Development Only, set temporary trashcan
   gString gstmp( "./testTrash" ) ;
   this->ctfRealpath ( gstmp, this->tcDir ) ;
   #endif   // Development Only

   //* 'rm' does confirmations and reports using the filespec     *
   //* provided by the user, so save a list of the raw filespecs. *
   //* At the same time gather the total size of source items.    *
   char rawList[this->itemCount][gsMAXBYTES] ;
   for ( flIndex = ZERO ; (flIndex < this->itemCount) ; ++flIndex )
   {
      this->fileList[flIndex].fSpec.copy( rawList[flIndex], gsMAXBYTES ) ;
      this->ctmValidateSourceItem ( flIndex, totalFiles, totalPass, totalBytes ) ;
   }

   //* Get amount of free space on filesystem containing the trashcan *
   fileSystemStats fss ;
   if ( (this->ctfFilesystemStats ( this->tcDir, fss )) != false )
   {
      if ( fss.freeBytes < (totalBytes * 1.5) )
      {
         
         this->textOut ( L"rm: insufficient free space on target filesystem" ) ;
         userAbort = true ;   // don't enter processing loop
         status = ERR ;       // return error condition
      }
   }
   //* Reset the accumulators *
   totalFiles = totalPass = ZERO ;
   totalBytes = ZERO ;

   //* Create the deletion-date timestamp    *
   //* string for all items to be processed. *
   localTime lt ;
   this->ctfGetLocalTime ( lt ) ;
   gString tStamp ;
   this->cttEncodeDate ( lt, tStamp ) ;

   //* Step through the list of items to be processed *
   for ( flIndex = ZERO ; (flIndex < this->itemCount) && !userAbort ; ++flIndex )
   {
      //* Validate the item, and update the accumulators *
      //* a) does it exist?
      //* b) is it of a supported filetype?
      //* c) does user have r/w/(x) permission?
      //* d) item is not IN trashcan?
      //* e) item doesn't CONTAIN trashcan?
      //* f) item is not root ( "/" )
      //* g) item does not cross filesystem boundary?
      invItem = this->ctmValidateSourceItem ( flIndex, totalFiles, totalPass, totalBytes ) ;

      //* Item validated, continue. *
      if ( !invItem )
      {
         skipItem = false ;      // reset the flag

         //* If item is a directory, then do secondary tests.*
         if ( this->fileList[flIndex].fStat.fType == fmDIR_TYPE )
         {
            if ( !emptyDirs && !recurse )
            {
               gsOut.compose( "rm: cannot remove '%s': Is a directory", rawList[flIndex] ) ;
               this->textOut ( gsOut ) ;
               ++failCount ;
               skipItem = true ;
            }
            else if ( emptyDirs && this->fileList[flIndex].dFiles > ZERO )
            {
               gsOut.compose( "rm: cannot remove '%s': Directory not empty", rawList[flIndex] ) ;
               this->textOut ( gsOut ) ;
               ++failCount ;
               skipItem = true ;
            }
            else
            {
               //* OK to process directories.                 *
               //* (recurse) || (emptyDirs && dFiles == ZERO) *
            }
         }

         //* If pre-confirmation specified and this is the first item. *
         if ( !skipItem && (intOption == rmINT_ONCE) && (itemsMoved == ZERO) )
         {
            this->textOut ( L"rm: remove all arguments? ", false ) ;
            this->ctUserResponse ( gsIn ) ;
            wchar_t resp = *gsIn.gstr() ;
            if ( resp != L'y' && resp != L'Y' )
               break ;           // user has aborted the operation
         }
         //* If separate confirmation for each item *
         else if ( !skipItem && (intOption == rmINT_EACH) )
         {
            if ( this->fileList[flIndex].fStat.fType == fmDIR_TYPE )
            {
               if ( this->fileList[flIndex].dFiles == ZERO )
                  gsOut.compose( "rm: remove directory '%s'? ", rawList[flIndex] ) ;
               else
                  gsOut.compose( "rm: descend into directory '%s'? ", rawList[flIndex] ) ;
            }
            else if ( this->fileList[flIndex].fStat.fType == fmREG_TYPE )
               gsOut.compose( "rm: remove regular file '%s'? ", rawList[flIndex] ) ;
            else if ( this->fileList[flIndex].fStat.fType == fmFIFO_TYPE )
               gsOut.compose( "rm: remove fifo '%s'? ", rawList[flIndex] ) ;
            else if ( this->fileList[flIndex].fStat.fType == fmLINK_TYPE )
               gsOut.compose( "rm: remove symbolic link '%s'? ", rawList[flIndex] ) ;

            this->textOut ( gsOut, false ) ;
            this->ctUserResponse ( gsIn ) ;
            wchar_t resp = *gsIn.gstr() ;
            if ( resp != L'y' && resp != L'Y' )
               skipItem = true ;
         }

         //* Validated item, with user permission (if required) *
         if ( !skipItem )
         {
            if ( (this->cttSendItemToTrash ( this->fileList[flIndex], 
                                             tStamp, eFile )) != false )
            {
               ++itemsMoved ;       // count moved item

               if ( this->verbose )
               {
                  if ( this->fileList[flIndex].fStat.fType != fmDIR_TYPE )
                     gsOut.compose( "removed '%s", rawList[flIndex] ) ;
                  else           // Directory removed
                  {
                     gsOut.compose( "removed directory '%s'", rawList[flIndex] ) ;

                     //* If directory was not empty *
                     gString fmtFiles, fmtBytes ;
                     fmtFiles.formatInt( this->fileList[flIndex].dFiles, 6, true ) ;
                     fmtBytes.formatInt( this->fileList[flIndex].dBytes, 6, true ) ;
                     if ( this->fileList[flIndex].dFiles > ZERO )
                        gsOut.append( "  (%S files %S bytes)", 
                                      fmtFiles.gstr(), fmtBytes.gstr() ) ;
                  }
                  this->textOut ( gsOut ) ;
               }
            }
            else
               ++failCount ;        // count un-moved item
         }
      }

      //* Item failed validation. report the reason *
      else
      {
         ++failCount ;        // count unprocessed items
         status = ERR ;       // returns the bad news

         //* If source item does not exist *
         if ( this->fileList[flIndex].fStat.fType == fmUNKNOWN_TYPE )
         {
            gsOut.compose( "rm: cannot remove '%S': No such file or directory",
                           this->fileList[flIndex].fSpec.gstr() ) ;
            this->textOut ( gsOut ) ;
         }

         else if ( (this->fileList[flIndex].fStat.fType == fmDIR_TYPE) 
                    && (!this->fileList[flIndex].fStat.rAccess ||
                        !this->fileList[flIndex].fStat.wAccess ||
                        !this->fileList[flIndex].fStat.xAccess) )
         {
            gsOut.compose( "rm: cannot remove '%s': Directory (or contents) write-protected",
                           rawList[flIndex] ) ;
            this->textOut ( gsOut ) ;
         }

         else if ( ((this->fileList[flIndex].fStat.fType != fmDIR_TYPE) 
                    && (!this->fileList[flIndex].fStat.rAccess ||
                        !this->fileList[flIndex].fStat.wAccess)) )
         {
            gsOut.compose( "rm: cannot remove '%s': write-protected",
                           rawList[flIndex] ) ;
            this->textOut ( gsOut ) ;
         }

         //* General error: (see note)                        *
         //* a) if source item is not of a supported filetype *
         //* b) if source item is already in the trashcan     *
         //* c) if source item contains the trashcan          *
         //* d) if source item is the root directory ( "/" )  *
         else
         {
            gsOut.compose( "rm: cannot remove '%s': general protection error",
                           rawList[flIndex] ) ;
            this->textOut ( gsOut ) ;
         }
      }
   }

   return status ;

}  //* End ctrmMoveToTrash() *

//*************************
//* ctmValidateSourceItem *
//*************************
//******************************************************************************
//* For move-to-trashcan operations, validate a source item.                   *
//*                                                                            *
//* Input  : flIndex : index into this->fileList of item to be verified        *
//*          fTotal  : (by reference) receives total number of files for items *
//*                    and their contents tested                               *
//*          fPass   : (by reference) receives total number of items verified  *
//*          totBytes: (by reference) receives total size in bytes of all      *
//*                    items and contents tested                               *
//*                                                                            *
//* Returns: 'false' if item and contents verified                             *
//*          'true'  if validation error and we must abort                     *
//******************************************************************************
//* Programmer's Notes: For a successful move to trashcan, we verify that:     *
//* a) source item exists,                                                     *
//* b) is of an approved filetype                                              *
//* c) that user has read/write access, and that if source item is a           *
//*    directory, user has read/write access to all directory contents         *
//* d) that source item does not live in the trashcan directory.               *
//* e) that source item does not contain the trashcan directory.               *
//*    (this includes the root directory "/")                                  *
//* f) that each source item lives entirely on the same filesystem.            *
//* g) that there is nominally enough freespace on the trashcan's filesystem   *
//*    Programmer's Note: If source and target are on the same filesystem,     *
//*    then the move may fail even if there is enough room because directory   *
//*    trees are moved by copy-then-delete. (freespace is tested by caller)    *
//*                                                                            *
//******************************************************************************

bool cTrash::ctmValidateSourceItem ( short flIndex, UINT& fTotal, UINT& fPass, 
                                     UINT64& totBytes )
{
   gString srcItem,              // source item filespec
           gsOut ;               // formatted output
   bool userAbort = true ;

   //* Get full path/filename specification for the item *
   //* and verify that source item actually exists.      *
   if ( ((this->ctfRealpath ( this->fileList[flIndex].fSpec, srcItem )) != false) 
        &&
        ((this->ctfGetFileStats ( srcItem, this->fileList[flIndex].fStat )) != false) )
   {
      //* Verify filetype and permissions.                                   *
      //* Must be a supported filetype and user must have r/w/(x) permission.*
      if ( (this->ctfVerifyFileStats ( this->fileList[flIndex].fStat )) != false )
      {
         //* Verify that source item is not already in the trashcan, *
         //* and is not the trashcan directory, itself.              *
         if ( (srcItem.compare( this->tcDir.gstr(), true, 
                                (this->tcDir.gschars() - 1) )) != ZERO )
         {
            //* Verify that source item does not contain the trashcan *
            if ( (this->tcDir.compare( srcItem.gstr(), true, 
                                       (srcItem.gschars() - 1))) != ZERO )
            {
               //* Verify that source item is not the root directory *
               if ( (srcItem.compare( L"/" )) != ZERO )
               {
                  //* Verified top-level item. Save the filespec *
                  //* and update accumulators.                   *
                  this->fileList[flIndex].fSpec = srcItem ;
                  ++fTotal ;                 // update our accumulators
                  ++fPass ;
                  totBytes += this->fileList[flIndex].fStat.fBytes ;

                  //* If source item is a directory, validate its contents *
                  //* Verify that all contents live on the same filesystem *
                  if ( this->fileList[flIndex].fStat.fType == fmDIR_TYPE )
                  {
                     //* Get the filesystem ID. If any of the contents    *
                     //* of this directory are on a different filesystem, *
                     //* validatition will fail.                          *
                     this->fileList[flIndex].fsysID = 
                               this->ctfFileSystemID ( this->fileList[flIndex].fSpec ) ;
                     UINT dcPass = this->ctfValidateDirContents ( this->fileList[flIndex] ) ;
                     fTotal += this->fileList[flIndex].dFiles ;
                     fPass += dcPass ;
                     totBytes += this->fileList[flIndex].dBytes ;
                     if ( dcPass == this->fileList[flIndex].dFiles )
                     {
                        userAbort = false ;     // good record
                     }
                     else if ( !this->silent )
                     {
                        this->textOut ( "Error! One or more files in directory item\n  ", false ) ;
                        this->textOut ( srcItem ) ;
                        this->textOut ( "is inaccessible or is an unsupported file type." ) ;
                        if ( this->verbose )
                        {
                           UINT dcFail = this->fileList[flIndex].dFiles - dcPass ;
                           gString gst, gsf ;
                           gst.formatInt( this->fileList[flIndex].dFiles, 7 ) ;
                           gsf.formatInt( dcFail, 7 ) ;
                           gsOut.compose( "%S files contained in directory item\n"
                                          "%S inaccessible files",
                                          gst.gstr(), gsf.gstr() ) ;
                           this->textOut ( gsOut ) ;
                        }
                     }
                  }
                  else     // not a directory, so reset these members
                  {  // (This isn't actually necessary, but it's warm-and-fuzzy.)
                     this->fileList[flIndex].fsysID = this->fileList[flIndex].dBytes = ZERO ;
                     this->fileList[flIndex].dFiles = ZERO ;

                     userAbort = false ;        // good record
                  }
               }
               else if ( !this->silent )
               {
                  this->textOut ( L"Error! It is not possible to move the 'root' directory!" ) ;
               }
            }
            else if ( !this->silent )     // source item contains trashcan
            {
               this->textOut ( "Error! Specified source item CONTAINS the Trashcan.\n  ", false ) ;
               this->textOut ( srcItem ) ;
            }
         }
         else if ( !this->silent )        // source item is in trashcan
         {
            this->textOut ( "Error! Specified source item is ALREADY in the trashcan.\n  ", false ) ;
            this->textOut ( srcItem ) ;
         }
      }
      else if ( !this->silent )
      {
         this->textOut ( "Error! Specified source item,\n  ", false ) ;
         this->textOut ( srcItem ) ;
         this->textOut ( "is inaccessible or is an unsupported file type." ) ;
         if ( this->verbose )
         {
            const wchar_t 
                  *ra = this->fileList[flIndex].fStat.rAccess ? L"read"  : L"---",
                  *wa = this->fileList[flIndex].fStat.wAccess ? L"write" : L"---",
                  *xa = this->fileList[flIndex].fStat.xAccess ? L"exec"  : L"---" ;

            gsOut.compose( "fType:%S  Access: |%S|%S|%S|", 
                           ftStrings[this->fileList[flIndex].fStat.fType], ra, wa, xa ) ;
            this->textOut ( gsOut ) ;
         }
      }
   }
   else if ( !this->silent )
   {  //* This probably means that the source file does not exist.*
      gsOut.compose( "Error! Specified source item, '%S' was not found.",
                     this->fileList[flIndex].fSpec.gstr() ) ;
      this->textOut ( gsOut ) ;
   }

   if ( userAbort != false && !this->silent )
      this->textOut ( L"Operation terminated." ) ;

   return userAbort ;

}  //* End ctmValidateSourceItem() *

//*************************
//*  ctmPromptFreespace   *
//*************************
//******************************************************************************
//* For move-to-trashcan operations, if the free space on trashcan's filesystem*
//* may be insufficient, ask the user whether it is OK to continue.            *
//*                                                                            *
//* Input  : tBytes  : combined size of all items to be moved                  *
//*                                                                            *
//* Returns: 'true'  if user has aborted the operation                         *
//*          'false' if user wants to continue                                 *
//******************************************************************************
//* Programmer's Note: If we cannot stat the filesystem, we don't worry the    *
//* user with this test on the assumption that if there is not enough free     *
//* space, the move will fail anyway.                                          *
//******************************************************************************

bool cTrash::ctmPromptFreespace ( UINT64 tBytes )
{
   fileSystemStats fss ;
   bool userAbort = false ;

   if ( (this->ctfFilesystemStats ( this->tcDir, fss )) != false )
   {
      if ( fss.freeBytes < (tBytes * 1.5) )
      {
         gString gstotal, gsfree ;
         gstotal.formatInt( tBytes, 6, true ) ;
         gsfree.formatInt( fss.freeBytes, 6, true ) ;
         gString gsOut( "Filesystem containing the Trashcan is nearly full.\n"
                        "  Bytes Needed: %S  Available: %S\n"
                        "  continue? (y/n): ", gstotal.gstr(), gsfree.gstr() ) ;
         this->textOut ( gsOut, false ) ;
         gString gsIn ;
         this->ctUserResponse ( gsIn ) ;
         wchar_t resp = *gsIn.gstr() ;
         if ( resp != L'y' && resp != L'Y' )
            userAbort = true ;
      }
   }
   return userAbort ;

}  //* End ctmPromptFreespace() *

//*************************
//*  ctmPromptPreconfirm  *
//*************************
//******************************************************************************
//* For move-to-trashcan operations, prompt for confirmation BEFORE moving.    *
//*                                                                            *
//* a) preconfirmation of restoration specified by user                        *
//*    ('--preconfirm' or '-p' option)                                         *
//* b) confirm unlink of source data                                           *
//*    ('--delete' or '-d' option  AND NOT '--no-confirm' option)              *
//*                                                                            *
//*                                                                            *
//* Input  : iCount  : formatted number of items to be moved                   *
//*          iFiles  : formatted number of files to be moved                   *
//*          iBytes  : formatted total size of items to be moved               *
//*          preConf : 'true'  if pre-move confirmation specified by user      *
//*                    'false' if no pre-move confirmation required            *
//*          permDel : 'false' if sending items to trashcan                    *
//*                    'true'  if bypassing trashcan and unlinking source items*
//*                                                                            *
//* Returns: 'true'  if user has aborted the operation                         *
//*          'false' if user wants to continue                                 *
//******************************************************************************

bool cTrash::ctmPromptPreconfirm ( const gString& iCount, const gString& iFiles, 
                                   const gString& iBytes, bool preConf, bool permDel )
{
   gString gsOut, gsIn ;
   bool userAbort = false ;

   //* If verbose output, list the individual items. *
   if ( this->verbose != false )
   {
      //* List the individual items *
      this->textOut ( L"Items To Be Moved:" ) ;
      for ( short i = ZERO ; i < this->itemCount ; ++i )
         this->textOut ( this->fileList[i].fSpec ) ;
   }

   //* Permanent deletion is more final, so its message takes precidence *
   if ( permDel != false )
   {
      gsOut.compose(
       "\nCAUTION: You are about to permanently remove %S items (%S files)\n"
         "         from your system. This operation cannot be undone.\n"
         "         Delete files? (y/n): ", iCount.gstr(), iFiles.gstr() ) ;
   }

   //* Preconfirmation of move to trashcan *
   else
   {
      gsOut.compose( "\nMove %S items (%S files, %Sbytes) to the trash? (y/n): ", 
                     iCount.gstr(), iFiles.gstr(), iBytes.gstr() ) ;
   }
   this->textOut ( gsOut, false ) ;
   this->ctUserResponse ( gsIn ) ;
   wchar_t resp = *gsIn.gstr() ;
   if ( resp != L'y' && resp != L'Y' )
      userAbort = true ;

   return userAbort ;

}  //* End ctmPromptPreconfirm() *

//*************************
//*  ctRestoreFromTrash   *
//*************************
//******************************************************************************
//* Restore item(s) from trashcan to their original position (or to alternate  *
//* target directory, if specified).                                           *
//*                                                                            *
//*                                                                            *
//* Input  : interact  : 'true;  interactively select item(s) to be restored   *
//*                      'false' restore most-recently deleted item(s)         *
//*          preconfirm: 'true'  get user confirmation before restoring items  *
//*                      'false' no confirmation required for item restoration *
//*          altTarget : (optional, NULL pointer by default)                   *
//*                      -- If NULL pointer (or empty string) indicates        *
//*                         restore to original position.                      *
//*                      -- If specified:                                      *
//*                         a) path of alternate target directory              *
//*                            (directory must exist)                          *
//*                         b) full path/filename for alternate target         *
//*                            (if target already exists, confirm overwrite)   *
//*                                                                            *
//* Returns: number of items succesfully restored from trashcan                *
//*            (not the file count)                                            *
//*          ZERO if user aborted the operation                                *
//*          ERR (-1) if one or more pre-processing errors                     *
//******************************************************************************
//* Notes:                                                                     *
//* Validation of the source(s) and target(s) is more complex than the         *
//* restoration itself.                                                        *
//* 1) Trashcan must contain at least one item.                                *
//* 2) Trashcan may be partially corrupt, but we can work through that because *
//*    only valid items are reported and can be restored.                      *
//*    Data without .trashinfo, or .trashinfo without data will not be seen    *
//*    by the user.                                                            *
//* 3) Selection of item(s) to be restored can be accomplished in one of       *
//*    two ways:                                                               *
//*    a) restore the most-recently-deleted item(s)                            *
//*    c) allow user to specify which item(s) are to be restored               *
//* 4) Restoration Target:                                                     *
//*    a) If item(s) are to be returned to their original position,            *
//*       -- Does the containing directory already exist?                      *
//*       -- Is there potential for overwrite of existing targets?             *
//*    b) If caller has specified an alternate target (not the item's original *
//*       position), we must validate its potential existence and its filetype.*
//*       -- If alt target is a directory, it must exist and be r/w/x          *
//*          accessible, and we must determine whether there is potential for  *
//*          overwrite of existing targets, and whether overwrite is OK.       *
//*       -- If alt target is a non-directory, then there must be exactly one  *
//*          item being restored, AND if an existing target, is it OK to       *
//*          overwrite it?                                                     *
//* 5) Freespace on target:                                                    *
//*    In order to keep the operation simple, we make a huge (and potentially  *
//*    incorrect) assumption: that all source items are being restored to the  *
//*    same filesystem. This will nearly always be the case, but it is by no   *
//*    means certain.                                                          *
//*    a) Using this assumption, we select the target filesystem for the       *
//*       largest item to be restored. Then we do a single test for freespace  *
//*       on that filesystem. If items are being restored to multiple          *
//*       filesystems, this may result in an occasional restoration failure    *
//*       due to lack of target freespace.                                     *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short cTrash::ctRestoreFromTrash ( bool interact, bool preconfirm, const char* altTarget )
{
   tcSummaryInfo tcSummary ;        // trashcan summary data
   tcDetailInfo* tcDetail = NULL ;  // pointer to trashcan detail data
   UINT   iTotal = ZERO,            // number of .trashinfo files scanned
          rTotal = ZERO ;           // total number of items to be restored
   UINT64 rBytes = ZERO ;           // total bytes to be restored
   gString gsIn,                    // interactive user input
           gsOut ;                  // output formatting
   DFileStats trgdfs ;              // file stats on existing target (if any)
   short itemsRestored = ZERO ;     // number of items successfully restored (return value)
   bool userAbort = false ;         // true if direct user abort or user screw-up

   //*** Alternate target info (if any) ***
   gString    altPath,              // path of alternate target (if any)
              altName ;             // filename of alternate target (if any)
   DFileStats altdfs ;              // file stats on existing alt target (if any)
   bool       altExist    = false,  // 'true' if alt target exists
              altVerified = false ; // 'true' if alt target verified


   //* Get summary statistics on trashcan *
   if ( (this->cttSummaryScan ( tcSummary )) != false )
   {
      if ( tcSummary.items > ZERO )
      {  //* Get detailed statistics and display data on trashcan contents.*
         if ( (iTotal = this->cttDetailScan ( tcSummary.iFiles, tcDetail )) == ZERO )
            userAbort = true ;   // trashcan is corrupted AND no '.trashinfo' records
      }
      else
      {  //* Trashcan is empty *
         if ( !this->silent )
         {
            if ( this->verbose )    // Display summary statistics
               this->ctReportTrashcan ( false ) ;

            //* User is a dufus: Report empty trashcan.*
            this->textOut ( L"Restore operation terminated. Trashcan is empty." ) ;
         }
         //* It isn't really the user, but the user's *
         //* stupidity that caused the abort.         *
         userAbort = true ;
      }
   }

   //* If trashcan contains at least one validated item *
   if ( !userAbort )
   {
      //* If we are to interactively restore one or more items from the        *
      //* trashcan. Display detailed statistics and prompt user for item(s).   *
      //* to restore.                                                          *
      if ( interact != false )
      {
         if ( (rTotal = this->ctPromptInteractive ( tcDetail, iTotal, true )) == ZERO )
         {
            if ( !this->silent )
               this->textOut ( L"No items were selected. Operation cancelled." ) ;
            userAbort = true ;
         }
      }

      //* Else, we are to restore the most-recent item(s) sent to trashcan.    *
      //* Scan the .trashinfo data files for newest deletion timestamp.        *
      else
      {
         if ( (rTotal = this->ctrScanMostRecent ( tcDetail, iTotal )) == ZERO )
            userAbort = true ;
      }
   }

   //* If restoration source items have been identified, identify the *
   //* filesystem to test for freespace.This will be, either the      *
   //* alternate target directory (if specified) or the target for    *
   //* largest item to be restored. (see notes above)                 *
   if ( !userAbort )
   {
      gString trgNode ;
      UINT largeIndex = ZERO ;      // index of largest item to be restored
      bool nodeFound = true ;
      //* Find the largest item to be restored. *
      for ( UINT i = ZERO ; i < iTotal ; ++i )
      {
         if ( tcDetail[i].select != false )
         {
            rBytes += tcDetail[i].size ;
            if ( tcDetail[i].size > tcDetail[largeIndex].size )
               largeIndex = i ;
         }
      }

      //* If an alternate target was specified, get its vital info.*
      if ( altTarget != NULL && *altTarget != NULLCHAR )
      {
         //* The specified target may, or may not exist;     *
         //* however, its parent directory MUST exist.       *
         //* See criteria for success in ctrResolveAltTarget.*
         fmFType altFType ;
         userAbort = this->ctrResolveAltTarget ( altTarget, altPath, altName, 
                                                 altFType, altExist, rTotal) ;
         if ( !userAbort )
         {
            trgNode = altPath ;
            nodeFound = altVerified = true ;
         }
         else if ( !this->silent )
         {
            this->textOut ( L"Error, unable to resolve alternate target path:\n"
                             "    ", false ) ;
            this->textOut ( altTarget ) ;
         }
      }
      //* Else assumed target is original item position of largest item. *
      else
      {
         //* Target's parent directory must exist, or we will fail later *
         //* anyway. Use it as a node on the filesystem to be checked.   *
         this->ctfExtractPathname ( trgNode, tcDetail[largeIndex].trgPath ) ;
         nodeFound = this->ctfGetFileStats ( trgNode, trgdfs ) ;
      }

      //* Test freespace on target filesystem.*
      if ( !userAbort && nodeFound && !this->silent )
         userAbort = this->ctrPromptFreespace ( trgNode, rBytes ) ;

      //* Scan for existing restoration targets, and potentially *
      //* prompt for permission to overwrite them.               *
      //* Protected targets WILL NOT be overwritten.             *
      UINT eCount,      // count of existing targets
           pCount ;     // count of existing _protected_ targets (may be incomplete)
      if ( !userAbort && this->confirm != false && !this->silent )
      {  //* If verified alternate target, trgNode contains its path.*
         //* Else, use original restoration target(s).               *
         if ( !altVerified )
            trgNode.clear() ;
         userAbort = this->ctrTestOverwrite ( tcDetail, iTotal, 
                                              eCount, pCount, trgNode, altName ) ;
      }

      //* Get pre-restoration confirmation if required.*
      if ( !userAbort && ((preconfirm != false) || (this->confirm && eCount > ZERO)) )
         if ( (userAbort = this->ctrPromptPreconfirm ( tcDetail, iTotal, eCount )) != false )
            this->ctrSummaryReport ( itemsRestored, ZERO ) ;
   }

   //***********************************************************
   //* All necessary error checking and confirmations complete.*
   //* Restore the specified item(s) from trashcan to target.  *
   //***********************************************************
   if ( !userAbort )
   {
      tcDetailInfo tcdi ;
      gString trgNode ;             // full path/filename of target's parent directory
      short filesRestored = ZERO ;  // number of _files_ restored
      bool nodeFound,               // 'true' if target's parent directory exists
           goodRestore ;            // 'true' if item restoration successful

      if ( !this->quiet )
         this->textOut ( L"Restoring:" ) ;

      for ( UINT i = ZERO ; i < iTotal ; ++i )
      {
         if ( tcDetail[i].select != false )
         {
            //* Construct source information *
            tcdi = tcDetail[i] ;

            //* If an alternate target has been specified, either a *
            //* target directory or a full path/filename, then      *
            //* modify the item's 'tcdi.trgPath' to point to it.    *
            if ( altVerified )
            {
               tcdi.trgPath = altPath ;      // target directory
               //* If specified, append the alternate item name to path *
               if ( altName.gschars() > 1 )
               {
                  tcdi.trgPath.append ( "/%S", altName.gstr() ) ;
               }
               //* Else, append the original item name to path *
               else
               {
                  gString eName ;
                  this->ctfExtractFilename ( eName, tcdi.srcPath ) ;
                  tcdi.trgPath.append ( "/%S", eName.gstr() ) ;
               }
            }

            //* Verify that the target's parent directory exists.   *
            //* While this step is not strictly necessary, it gives *
            //* us a chance to display a detailed error message.    *
            this->ctfExtractPathname ( trgNode, tcdi.trgPath ) ;
            nodeFound = this->ctfGetFileStats ( trgNode, trgdfs ) ;

            //* Move source item to target *
            if ( nodeFound )
            {
               if ( (goodRestore = this->cttRestoreItemFromTrash ( tcdi )) != false )
               {  //* Update the accumulators *
                  ++itemsRestored ;
                  filesRestored += tcdi.files ;
               }
            }
            else
               goodRestore = false ;

            if ( this->verbose )
            {
               //* Display the item's formatted description *
               this->textOut ( tcDetail[i].dRecord ) ;
               if ( goodRestore )
                  this->textOut ( L"[TO] ", false ) ;
               else if ( !nodeFound )
                  this->textOut ( L"[One or more parent directories "
                                   "on target path are missing.]" ) ;
            }
            if ( !this->quiet )
            {
               gsOut = tcdi.trgPath ;
               if ( !goodRestore )
               {  //* If restore failed (trashcan copy is _probably_ intact) *
                  this->textOut ( L"[FAILED] ", false ) ;
                  this->ctfTrimPathString ( gsOut, (this->maxStatCols - 2) ) ;
               }
               //* Report each item's restored position *
               this->textOut ( gsOut ) ;
            }
         }
      }

      //* Report summary of operation *
      if ( !this->quiet )
         this->ctrSummaryReport ( itemsRestored, filesRestored ) ;
   }

   if ( tcDetail != NULL )       // if we have a dynamic allocation, release it
      delete [] tcDetail ;

   return itemsRestored ;

}  //* End ctRestoreFromTrash() *

//*************************
//*  ctPromptInteractive  *
//*************************
//******************************************************************************
//* 1) For interactive restore operations , prompt user for which item(s) to   *
//*    restore.                                                                *
//* 2) For interactive empty-trash operations, prompt user for which item(s)   *
//*    to delete.                                                              *
//*                                                                            *
//* Input  : tcDetail  : array of records describing each item in trashcan     *
//*          iTotal    : number of items in the trashcan                       *
//*          restore   : if 'true',  then display the 'restore' message        *
//*                      if 'false', then display the 'delete' message         *
//*                                                                            *
//* Returns: number of items to be restored                                    *
//*          if ZERO returned, then nothing to restore, abort operation        *
//******************************************************************************

UINT cTrash::ctPromptInteractive ( tcDetailInfo* tcDetail, UINT iTotal, bool restore )
{
   gString gsIn ;                // user input
   UINT rTotal = ZERO ;          // return value

   for ( UINT i = ZERO ; i < iTotal ; ++i )
      this->textOut ( tcDetail[i].dRecord ) ;

   if ( restore != false )
      this->textOut ( L"\nEnter the item number of each item to be restored, "
                       "separated by commas.\n"
                       "Example: 1,3,4    (or 'q' to quit)\n"
                       "  selection: ", false ) ;
   else
      this->textOut ( L"\nEnter the item number of each item to be deleted, "
                       "separated by commas.\n"
                       "Example: 1,3,4    (or 'a' (all), or 'q' to quit)\n"
                       "  selection: ", false ) ;

   this->ctUserResponse ( gsIn ) ;

   //* Decode and validate user input *
   UINT rItem ;            // numeric user input
   short ci ;              // comma index

   //* For delete (but not for restore), allow 'all' token.*
   const wchar_t* wptr = gsIn.gstr() ;  // pointer to input string
   if ( !restore && (*wptr == L'a' || *wptr == L'A') )        // select all
   {  //* Set all 'select' flags *
      for ( UINT i = ZERO ; i < iTotal ; ++i )
      {
         tcDetail[i].select = true ;
         ++rTotal ;
      }
   }
   else
   {
      do
      {
         if ( (swscanf ( gsIn.gstr(), L" %u", &rItem )) == 1 )
         {  //* Note: Data displayed to user is one-based, *
            //* but our tracking is zero-based (indices).  *
            if ( rItem != ZERO && rItem <= iTotal )
            {
               tcDetail[--rItem].select = true ;
               ++rTotal ;                    // update accumulator
               if ( (ci = gsIn.find( L',' )) > ZERO )
                  gsIn.shiftChars ( -(ci + 1) ) ;
               else     // end of user input
                  break ;
            }
            else
            {
               gString gsOut( "Error: %d is not a valid item number.", &rItem ) ;
               this->textOut ( gsOut ) ;
               rTotal = ZERO ;         // user's input out-of-range
               break ;
            }
         }
         else        // end of user input (or user error)
         {
            wptr = gsIn.gstr() ;
            if ( (*wptr == L'q') || (*wptr == L'Q') )    // user aborted
               rTotal = ZERO ;
            else if ( (*wptr != NULLCHAR) && ((*wptr < L'0') || (*wptr > L'9')) )
            {
               gString gsOut( "Error: '%S' is not a numeric value.", wptr ) ;
               this->textOut ( gsOut ) ;
               rTotal = ZERO ;         // user mis-typed
            }
            //* If abort, reset all 'select' flags.*
            if ( rTotal == ZERO )
            {
               for ( UINT i = ZERO ; i < iTotal ; ++i )
                  tcDetail[i].select = false ;
            }
            break ;
         }
      }
      while ( gsIn.gschars() > 1 ) ;
   }

   return rTotal ;

}  //* End ctPromptInteractive() *

//*************************
//*   ctrScanMostRecent   *
//*************************
//******************************************************************************
//* For restore most-recently-deleted operations (undo), scan all items in the *
//* trashcan for the most-recently-deleted item(s).                            *
//*                                                                            *
//*                                                                            *
//* Input  : tcDetail  : array of records describing each item in trashcan     *
//*          iTotal    : number of items in the trashcan                       *
//*                                                                            *
//* Returns: number of items to be restored                                    *
//*          if ZERO returned, then nothing to restore, abort operation        *
//******************************************************************************

UINT cTrash::ctrScanMostRecent ( tcDetailInfo* tcDetail, UINT iTotal )
{
   int64_t newestTimestamp = ZERO ;    // newest timestamp
   UINT rTotal = ZERO ;                // return value

   //* Identify item(s) to be restored             *
   //* First itentify record with newest timestamp *
   for ( UINT i = ZERO ; i < iTotal ; ++i )
   {
      if ( tcDetail[i].tDate.epoch > newestTimestamp )
         newestTimestamp = tcDetail[i].tDate.epoch ;
   }
   //* Then identify all items with that timestamp *
   for ( UINT i = ZERO ; i < iTotal ; ++i )
   {
      if ( tcDetail[i].tDate.epoch == newestTimestamp )
      {
         tcDetail[i].select = true ;
         ++rTotal ;
      }
   }
   return rTotal ;

}  //* End ctrScanMostRecent() *

//*************************
//*  ctrPromptPreconfirm  *
//*************************
//******************************************************************************
//* For restore operations, prompt for confirmation BEFORE restoration.        *
//*                                                                            *
//* This method called if:                                                     *
//* a) preconfirmation of restoration specified by user (--preconfirm option)  *
//* b) existing targets AND '--no-confirm' option NOT specified, AND           *
//*    the existing targets, if any, are not write protected.                  *
//* c) both preconfirmation AND existing targets                               *
//*                                                                            *
//* Input  : tcDetail  : array of records describing each item in trashcan     *
//*          iTotal    : number of items in the trashcan                       *
//*          eCount    : count of existing targets                             *
//*                                                                            *
//* Returns: 'true'  if user has aborted the operation                         *
//*          'false' if user wants to continue                                 *
//******************************************************************************

bool cTrash::ctrPromptPreconfirm ( const tcDetailInfo* tcDetail, 
                                   UINT iTotal, UINT eCount )
{
   UINT64 rBytes = ZERO ;     // total bytes to be restored
   UINT   rItems = ZERO ;     // total items to be restored
   UINT   rFiles = ZERO ;     // total files to be restored
   bool hdrOut = false,       // indicates whether verbose-data header written
        userAbort = false ;   // return value

   for ( UINT i = ZERO ; i < iTotal ; ++i )
   {
      if ( tcDetail[i].select != false )
      {
         if ( !hdrOut && !this->quiet )
         {
            this->textOut ( L"Items To Be Restored:" ) ;
            hdrOut = true ;
         }
         if ( this->verbose )
            this->textOut ( tcDetail[i].dRecord ) ;
         else if ( !this->quiet )
            this->textOut ( tcDetail[i].trgPath ) ;
         ++rItems ;
         rFiles += tcDetail[i].files ;
         rBytes += tcDetail[i].size ;
      }
   }

   if ( rItems > ZERO )
   {
      gString fmtBytes ;
      fmtBytes.formatInt( rBytes, 6, true ) ;
      gString gsOut( "\n%3u item(s), including %u files (%Sbytes) to be restored.",
                     &rItems, &rFiles, fmtBytes.gstr() ) ;
      if ( rBytes < 100000 )
      {  //* A bit of beautification *
         short i = gsOut.find( L"bytes" ) ;
         gsOut.insert( L' ', i ) ;
      }
      this->textOut ( gsOut ) ;
      if ( eCount > ZERO )
      {
         gsOut.compose( "%3u of the target item(s) already exist and will be overwritten.", &eCount ) ;
         this->textOut ( gsOut ) ;
      }
      this->textOut ( L"     continue? (y/n): ", false ) ;
      gString gsIn ;             // user input
      this->ctUserResponse ( gsIn ) ;
      wchar_t resp = *gsIn.gstr() ;
      if ( resp != L'y' && resp != L'Y' )
         userAbort = true ;
   }
   return userAbort ;

}  //* End ctrPromptPreconfirm() *

//*************************
//*  ctrResolveAltTarget  *
//*************************
//******************************************************************************
//* For restore operations, validate alternate target directory or             *
//* path/filename.                                                             *
//*                                                                            *
//* This can be a rather difficult parameter to validate, so carefully study   *
//* the criteria for success.                                                  *
//*                                                                            *
//* 1) If 'altTarget' specifies an existing directory,                         *
//*      AND if that directory has sufficient permission,                      *
//*    then selected item(s) will be restored to that directory.               *
//*    a) 'altPath'  receives the restoration path                             *
//*    b) 'altFName' receives empty string                                     *
//*    c) 'altfType' == fmDIR_TYPE                                             *
//*    d) 'altExist' == true                                                   *
//*    e) returns 'false' (no abort)                                           *
//* 2) If 'altTarget' specifies an existing non-directory file,                *
//*      AND if target is a supported filetype,                                *
//*      AND if target has sufficient permission,                              *
//*      AND if exactly one item is to be restored,                            *
//*    then the item will be restored using specified path/filename.           *
//*    a) 'altPath' receives the path of existing restoration parent directory *
//*    b) 'altFName' receives the existing target filename                     *
//*    c) 'altfType' == the file type of existing target                       *
//*       (Caller may later check that source and target are of same filetype.)*
//*    d) 'altExist' == true                                                   *
//*    e) returns 'false' (no abort)                                           *
//* 3) If 'altTarget' does not exist, BUT target's parent directory exists,    *
//*      AND if target's parent directory actually IS a directory,             *
//*      AND if target's parent directory has sufficient permission,           *
//*      AND if exactly one item is to be restored,                            *
//*    then the item will be restored using specified path/filename.           *
//*    a) 'altPath' receives the path of existing restoration parent directory *
//*    b) 'altFName' receives the non-existing target filename                 *
//*    b) 'altfType' == fmUNKNOWN_TYPE                                         *
//*    d) 'altExist' == false                                                  *
//*    c) returns 'false' (no abort)                                           *
//* 4) All other scenarios return 'true' (abort), with 'altPath', 'altFName',  *
//*   'altfType' and 'altExist undefined.                                      *
//*                                                                            *
//*                                                                            *
//* Input  : altTarget : path of alternate target (user version)               *
//*          altPath   : receives target path (see above)                      *
//*          altFName  : received non-existing target filename (see above)     *
//*          altfType  : receives filetype of existing target                  *
//*                      (or fmUNKNOWN_TYPE)                                   *
//*          altExist  : receives 'true' if existing target                    *
//*          rTotal  : number of items user has selected for restoration       *
//*                                                                            *
//* Returns: 'false' if alternate target verified                              *
//*          'true'  if invalid alt target                                     *
//******************************************************************************
//* Notes: A valid alternate path may be:                                      *
//* a) a relative path:                                                        *
//*    Ex: ../Art_backup                                                      *
//*    Ex: ./Art/scene.jpg                                                     *
//*    Ex: ./                                                                  *
//*    Ex: .                                                                   *
//* b) an absolute path:                                                       *
//*    Ex: ~/Art                                                               *
//*    Ex: ${HOME}/Art                                                         *
//*    Ex: /home/sam/Art_backup                                                *
//*    Ex: /home/sam/Art/scene.jpg                                             *
//* If the path cannot be resolved, then we must abort the restore operation,  *
//* so construct the path or path/filename carefully.                          *
//******************************************************************************

bool cTrash::ctrResolveAltTarget ( const char* altTarget, 
                                   gString& altPath, gString& altFName, 
                                   fmFType& altfType, bool& altExist, UINT rTotal )
{
   gString gstmp = altTarget ;
   DFileStats dfs ;                 // stats for targets
   bool trgExist = false,           // true if full target exists
        done = false,               // true if translation complete
        userAbort = false ;         // return value
   altFName.clear() ;               // assume that there is no terminal filename
   altfType = fmUNKNOWN_TYPE ;      // assume that target does not exist
   altExist = false ;

   //* Expand the environment-variable substitution and if *
   //* possible, get the full path to target and its stats.*
   if ( (this->ctfEnvExpansion ( gstmp )) != false )
   {
      if ( (this->ctfRealpath ( gstmp, altPath )) != false )
      {  //* Target (probably) exists. Get its stats.*
         trgExist = this->ctfGetFileStats ( altPath, dfs ) ;

         //* If existing target (see criteria above). *
         if ( trgExist && (done = ctfVerifyFileStats ( dfs )) != false )
         {
            altfType = dfs.fType ;     // target filetype
            altExist = true ;          // target exists

            //* If existing target is not a directory name, then  *
            //* there must be exactly one restoration target.     *
            if ( altfType != fmDIR_TYPE && rTotal == 1)
            {  //* Separate parent directory and target filename.*
               gstmp = altPath ;
               this->ctfExtractPathname ( altPath, gstmp ) ;
               this->ctfExtractFilename ( altFName, gstmp ) ;
            }
            //* Else, multiple source items with non-directory target *
            else if ( altfType != fmDIR_TYPE )
               userAbort = true ;
         }
         //* Target (probably) exists; however, it is either of an *
         //* unsupported filetype OR user has insufficient access. *
         else
            userAbort = true ;
      }

      //* Target does not exist. Test for existence of parent directory.*
      if ( !userAbort && !trgExist )
      {
         gString parPath ;                // parent of target path

         this->ctfExtractPathname ( altPath, gstmp ) ;
         this->ctfExtractFilename ( altFName, gstmp ) ;
         if ( altPath.gschars() > 1 )
         {
            if ( (this->ctfRealpath ( altPath, parPath )) != false )
            {  //* Target's parent (probably) exists. Get its stats.*
               if ( ((this->ctfGetFileStats ( parPath, dfs )) != false)
                    && (dfs.fType == fmDIR_TYPE) && (rTotal == 1) &&
                    ((this->ctfVerifyFileStats ( dfs )) != false) )
               {
                  //* Ultimate target does not exist, so its filetype   *
                  //* is unknown, the path is the parent path, and      *
                  //* altFName gets the filename of non-existing target.*
                  altfType = fmUNKNOWN_TYPE ;
                  altPath  = parPath ;
               }
               else
               {  //* Target's parent directory does not exist, or   *
                  //* insufficient user permission, or parent is not *
                  //* a directory, (i.e. user is a dufus), or more   *
                  //* than one source item is to be restored.        *
                  userAbort = true ;
               }
            }
            else  // target's parent directory does not exist
               userAbort = true ;
         }
         else
            userAbort = true ;
      }
   }
   else        // invalid character(s) in the source string
      userAbort = true ;

   return userAbort ;

}  //* End ctrResolveAltTarget() *

//*************************
//*  ctrPromptFreespace   *
//*************************
//******************************************************************************
//* For restore operations, if the free space on target filesystem may be      *
//* insufficient, ask the user whether it is OK to continue.                   *
//*                                                                            *
//*                                                                            *
//* Input  : trgNode : any node on the target filesystem tree                  *
//*          rBytes  : combined size of all items to be restored               *
//*                                                                            *
//* Returns: 'true'  if user has aborted the operation                         *
//*          'false' if user wants to continue                                 *
//******************************************************************************
//* Programmer's Note: If we cannot stat the filesystem, we don't worry the    *
//* user with this test on the assumption that if there is not enough free     *
//* space, the restore will fail anyway. A filesystem stat is most likely to   *
//* fail on an external device like a flash-memory or SD-card device.          *
//* This is not the fault of the device, but a timing error in the kernel.     *
//******************************************************************************

bool cTrash::ctrPromptFreespace ( const gString& trgNode, UINT64 rBytes )
{
   fileSystemStats fss ;
   bool userAbort = false ;

   if ( (this->ctfFilesystemStats ( trgNode, fss )) != false )
   {
      if ( fss.freeBytes < (rBytes * 1.5) )
      {
         gString gstotal, gsfree ;
         gstotal.formatInt( rBytes, 6, true ) ;
         gsfree.formatInt( fss.freeBytes, 6, true ) ;
         gString gsOut( "\n  Target filesystem for restore is nearly full.\n"
                          "  Bytes Needed: %S  Available: %S\n"
                          "  continue? (y/n): ", gstotal.gstr(), gsfree.gstr() ) ;
         this->textOut ( gsOut, false ) ;
         gString gsIn ;
         this->ctUserResponse ( gsIn ) ;
         wchar_t resp = *gsIn.gstr() ;
         if ( resp != L'y' && resp != L'Y' )
            userAbort = true ;
      }
   }
   return userAbort ;

}  //* End ctrPromptFreespace() *

//*************************
//*   ctrTestOverwrite    *
//*************************
//******************************************************************************
//* For restore operations, test for overwrite of existing target(s).          *
//*                                                                            *
//* 1) Scan items to be restored.                                              *
//* 2) Count existing targets.                                                 *
//* 3) Count existing, protected targets.                                      *
//*    If not in 'quiet' mode, display the protected targets along with their  *
//*    r/w/x permissions.                                                      *
//* 4) If in 'verbose' mode, display all existing targets along with their     *
//*    r/w/x permissions.                                                      *
//*                                                                            *
//* If existing targets found:                                                 *
//*  a) If targets are not protected, display a message asking for permission  *
//*     to overwrite them.                                                     *
//*  b) If one or more existing targets are procted, then signal abort of the  *
//*     restore. We do not overwrite protected data.                           *
//*                                                                            *
//* This method is called ONLY when: this->confirm && !this->silent            *
//* a) !this->silent    (not running in silent mode)                           *
//*    If user specified silent mode, then that also means that we do not      *
//*    ask for confirmations, and potential failure of the overwrite will      *
//*    indicate the situation.                                                 *
//* b) this->confirm    (confirmation required for non-reversible actions)     *
//*    If user specified that no confirmations are necessary, then there is    *
//*    no need to call this method, and again, potential failure of the        *
//*    overwrite will indicate the situation.                                  *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : tcDetail  : array of records describing each item in trashcan     *
//*          iTotal    : number of items in the trashcan                       *
//*          eCount    : (by reference) receives count of existing targets     *
//*                      both top-level items and directory contents           *
//*          pCount    : (by reference) receives count of protected targets    *
//*                      including directory contents                          *
//*          altDir    : (empty string if none)                                *
//*                      If an alternate restoration directory was specified,  *
//*                      scan that directory for existing item names.          *
//*          altName   : (empty string if none)                                *
//*                      If an alternate path/filename was specified as the    *
//*                      restoration target, combine 'altDir' and 'altName'    *
//*                      as the target for which to test                       *
//*                                                                            *
//* Returns: 'true'  if protected target(s) identified (signals abort)         *
//*          'false' no protected targets with corresponding source file       *
//******************************************************************************
//* Programmer's Note:                                                         *
//* There are a number of design decisions to be made here.                    *
//* 1) If an existing target item is protected, do we allow the operation to   *
//*    continue for other items? Our decision: NO.                             *
//*                                                                            *
//* 2) If an existing target item is of a different filetype than the source,  *
//*    do we allow the overwrite? Our decision: NO.                            *
//*                                                                            *
//* 3) Contents of an existing directory target:                               *
//*    Do we need to check target permissions against the source files to be   *
//*    restored? Our decision: A QUALIFIED YES.                                *
//*    a) If an existing file in the target directory IS NOT of a supported    *
//*       type, we ignore its permissions because it will never be overwritten *
//*       anyway.                                                              *
//*    b) If an existing file in the target directory IS of a supported type,  *
//*       AND it is protected, what should we do?                              *
//*       i)  The easy answer is not to test for a match, but to simply abort  *
//*           on the assumption that a potential corresponding source file     *
//*           MAY need to overwrite the target file.                           *
//*           This would mean aborting without verification of actual need.    *
//*       ii) The correct answer is to test for a corresponding source file,   *
//*           and to abort if there is a match (no items restored).            *
//*           The reason for this is that we don't want to leave the user with *
//*           a directory containing some restored and some unrestored files.  *
//*           That would be just too cruel. The problem is that this is a very *
//*           expensive test scenario.                                         *
//*       Our decision: We are all about the user friendliness, so if an       *
//*       existing target file within an existing target item IS PROTECTED,    *
//*       then we test for the existence of a corresponding source file and    *
//*       abort ONLY if their is a match.                                      *
//*                                                                            *
//******************************************************************************

bool cTrash::ctrTestOverwrite ( const tcDetailInfo* tcDetail, UINT iTotal, 
                                UINT& eCount, UINT& pCount, 
                                const gString& altDir, const gString& altName )
{
   const wchar_t rCHAR = L'r', wCHAR = L'w', xCHAR = L'x', zCHAR = L'-' ;
   gString trgPath, fName, gsOut ;
   DFileStats trgStat ;
   UINT   rItems = ZERO ;     // total items to be restored
   bool trgExist,
        hdrOut = false,
        userAbort = false ;
   eCount = pCount = ZERO ;   // initialize caller's accumulators
   
   for ( UINT i = ZERO ; i < iTotal ; ++i )
   {
      if ( tcDetail[i].select != false )
      {
         ++rItems ;
         //* Create target spec *
         if ( (altDir.gschars()) > 1 ) // if targets are on alternate path
         {
            if ( (altName.gschars()) > 1 )
               this->ctfCatPathFilename ( trgPath, altDir, altName.gstr() ) ;
            else
            {
               this->ctfExtractFilename ( fName, tcDetail[i].trgPath ) ;
               this->ctfCatPathFilename ( trgPath, altDir, fName.gstr() ) ;
            }
         }
         else                    // targets are on original path
         {
            trgPath = tcDetail[i].trgPath ;
         }

         //* Get stats for target spec *
         trgExist = this->ctfGetFileStats ( trgPath, trgStat ) ;
         if ( trgExist )
         {
            ++eCount ;

            //* Test for supported filetype and permissions *
            if ( (this->ctfVerifyFileStats ( trgStat )) != false )
            {  //* Compare source filetype with target filetype *
               if ( trgStat.fType == tcDetail[i].fType )
               {
                  //* If an existing target is a directory, *
                  //* test its contents.                    *
                  if ( trgStat.fType == fmDIR_TYPE )
                  {
                     UINT seCount = ZERO ;
                     userAbort = this->ctrTestDirPermission ( tcDetail[i].srcPath, 
                                                              trgPath, seCount ) ;
                     //eCount += seCount - 1 ;
                     // Programmer's Note: 'seCount' is currently not used.
                     // Note also, that during the call, the top-level directory 
                     // is counted again.

                     //* There may be more than one protected target, *
                     //* but one is enough to abort.                  *
                     if ( userAbort )
                        ++pCount ;
                  }
               }
               else     // top-level item failed verification (conflicting filetypes)
                  ++pCount ;
            }
            else     // top-level item failed verification
               ++pCount ;

            if ( this->verbose )
            {
               if ( !hdrOut )
               {
                  this->textOut ( L"\n  Existing Target Items:" ) ;
                  hdrOut = true ;
               }
               gsOut.compose( "  [%C%C%C] ", 
                              (trgStat.rAccess ? &rCHAR : &zCHAR),
                              (trgStat.wAccess ? &wCHAR : &zCHAR),
                              (trgStat.xAccess ? &xCHAR : &zCHAR) ) ;
               this->textOut ( gsOut, false ) ;
               this->textOut ( trgPath ) ;
            }
         }     // if(trgExist)
      }
   }
   if ( this->verbose && eCount > ZERO )
      this->textOut ( L"" ) ;

   //* If one or more _protected_ existing targets, OR       *
   //* an existing target is of an unsupported filetype, OR  *
   //* source and existing target are of different filetypes.*
   if ( pCount > ZERO )
   {
      this->textOut ( L"One or more existing target items are protected, or are of\n"
                       "an unsupported or conflicting file type. Operation terminated." ) ;
      userAbort = true ;
   }

   return userAbort ;

}  //* End ctrTestOverwrite() *

//*************************
//* ctrTestDirPermission  *
//*************************
//******************************************************************************
//* For each file in the specified directory, test the read/write/execute      *
//* permissions.                                                               *
//*                                                                            *
//* The top-level directory itself IS NOT included in the total count.         *
//*                                                                            *
//* Input  : srcPath  : (by reference) path to source directory (in trashcan)  *
//*          trgPath  : (by reference) path to directory to be scanned         *
//*          eCount   : (by reference) receives count of existing files        *
//*                                                                            *
//* Returns: 'false' if permissions on all existing target verified            *
//*          'true'  if one or more existing targets are protected             *
//******************************************************************************
//* Notes:                                                                     *
//* We return a count of all existing targets ( 'eCount' ).                    *
//* We signal a protected target only if the target:                           *
//*    a) is of a supported filetype                                           *
//*    b) has a corresponding source file                                      *
//* Targets which are not of a supported filetype or which have no             *
//* corresponding source file will not be reported as protected.               *
//*                                                                            *
//******************************************************************************

bool cTrash::ctrTestDirPermission ( const gString& srcPath, 
                                    const gString& trgPath, UINT& eCount )
{
   DIR*   dirPtr ;                        // pointer to directory contents
   static USHORT recursionLevel = ZERO ;  // levels of recursion
   bool   prot = false ;                  // return value
   //* Initialize caller's accumulator *
   if ( recursionLevel == ZERO )
      eCount = ZERO ;
   ++recursionLevel ;

   //* Open the directory *
   if ( (dirPtr = this->ctfOpendir ( trgPath )) != NULL )
   {
      gString    fPath,             // target filespec
                 fName,             // target filename
                 srcSpec ;          // source filespec
      DFileStats dfs,               // target file stats
                 sdfs ;             // source file stats
      deStats*   destat ;           // directory entry
      bool       srcExist ;         // true if matching source item

      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* Do not include 'current dir' and 'parent dir' names.              *
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
               continue ;           // go to next record
         }

         //* Create the target file path/filename and get its stats *
         this->ctfCatPathFilename ( fPath, trgPath, destat->d_name ) ;
         if ( (this->ctfGetFileStats ( fPath, dfs )) != false )
         {
            ++eCount ;              // count the existing file

            //* Test target filetype *
            if ( (this->cttTestFiletype ( dfs.fType )) != false )
            {
               //* Test for corresponding source file.*
               this->ctfExtractFilename ( fName, fPath ) ;
               this->ctfCatPathFilename ( srcSpec, srcPath, fName.gstr() ) ;
               srcExist = this->ctfGetFileStats ( srcSpec, sdfs ) ;

               //* Target is a supported filetype, test its permissions.*
               if ( !(this->ctfVerifyFileStats ( dfs )) )
               {  //* Target is protected.  If existing source matches *
                  //* existing PROTECTED target, report conflict.      *
                  if ( srcExist )
                     prot = true ;
               }
               else
               { /* target permisisons are OK */ }

               //* If we have not yet encountered a conflicting  *
               //* source/target file, AND if both target and    *
               //* source are directories, then test recursively.*
               if ( !prot && (dfs.fType == fmDIR_TYPE) && (sdfs.fType == fmDIR_TYPE) )
               {
                  if ( (this->ctrTestDirPermission ( srcSpec, fPath, eCount )) != false )
                     prot = true ;
               }
            }
            else
            { /* unsupported filetypes are ignored */ }
         }
         else
         {  /* lstat failed - unlikely */ }
      }
      closedir ( dirPtr ) ;      // Close the directory
   }
   else     // invalid target path or user does not have full target access
      prot = true ;

   --recursionLevel ;
   return prot ;

}  //* End ctrTestDirPermission() *

//*************************
//*   ctrSummaryReport    *
//*************************
//******************************************************************************
//* For restore operations, display restoration summary.                       *
//*                                                                            *
//* Input  : itemsRestored : items successfully restored from trashcan         *
//*          filesRestored : total number of files restored                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
void cTrash::ctrSummaryReport ( UINT itemsRestored, UINT filesRestored )
{
   
   gString gsOut( "\n%hd item(s) (%hd files) restored from Trashcan.",
                  &itemsRestored, &filesRestored ) ;
   this->textOut ( gsOut ) ;

}  //* End ctrSummaryReport() *

//*************************
//*     ctEmptyTrash      *
//*************************
//******************************************************************************
//* Public Method. Converts the parameters from days or hours to an offset     *
//* in seconds for call to our private method.                                 *
//*                                                                            *
//* Empty the trashcan: 1) completely,                                         *
//*                     2) interactively,                                      *
//*                     3) by freshness date.                                  *
//* This operation cannot be undone.                                           *
//*                                                                            *
//* NOTE: This method is not called by the cTrash application itself.          *
//*       In order to get better timestamp resolution, cTrash calls a          *
//*       private overload of ctEmptyTrash. This method is for use by          *
//*       applications which integrate the cTrash source code into their       *
//*       own code.                                                            *
//*                                                                            *
//* Input  : interact  : 'false' delete all files from the trashcan            *
//*                      'true'  display trashcan contents and prompt user     *
//*                              for files to be deleted                       *
//*          days      : (optional, 'true' by default)                         *
//*                      (ignored if remTime==ZERO or 'interact' != false)     *
//*                      indicates whether 'remTime' should be interpreted     *
//*                      as days or as hours                                   *
//*                      'true'  : interpret 'remTime' as days                 *
//*                      'false' : interpret 'remTime' as hours                *
//*          remTime   : (optional, ZERO by default, and        )              *
//*                      (ignored if ZERO or 'interact' != false)              *
//*                      if specified and 'days' != false:                     *
//*                        Items which have been in the trashcan for           *
//*                        >= remTime days will be deleted                     *
//*                        Range: 1 - 366 days (1 day through 1 year)          *
//*                      if specified and 'days' == false:                     *
//*                        Items which have been in the trashcan for           *
//*                        <= remTime hours will be deleted                    *
//*                        Range: 1 - 24 hours                                 *
//*                                                                            *
//* Returns: number of items succesfully deleted from trashcan                 *
//*            (not the file count)                                            *
//*          ZERO if preprocessing error or user aborted the operation         *
//******************************************************************************

short cTrash::ctEmptyTrash ( bool interact, bool days, short remTime )
{
   int remMinutes ;
   if ( days != false )
      remMinutes = int(remTime) * 24 * 60 ;
   else
      remMinutes = (int(remTime) * 60) ;

   short status = this->ctEmptyTrash ( interact, remMinutes ) ;
   return status ;

}  //* End ctEmptyTrash() *

//*************************
//*     ctEmptyTrash      *
//*************************
//******************************************************************************
//* Empty the trashcan: 1) completely,                                         *
//*                     2) interactively,                                      *
//*                     3) by freshness date.                                  *
//* This operation cannot be undone.                                           *
//*                                                                            *
//* Input  : interact  : 'false' delete all files from the trashcan            *
//*                      'true'  display trashcan contents and prompt user     *
//*                              for files to be deleted                       *
//*          remTime   : (optional, ZERO by default, and        )              *
//*                      (ignored if ZERO or 'interact' != false)              *
//*                      if specified, it is the offset in minutes from the    *
//*                      current system local time for establishing a          *
//*                      comparison timestamp                                  *
//*                      If > ZERO, it represents a number of days, and items  *
//*                                 will be deleted if their deletion date     *
//*                                 is <= this timestamp.                      *
//*                         Range: 1 through 1827 days                         *
//*                                                                            *
//*                      If < ZERO, it represents a number of hours, and items *
//*                                 will be deleted if their deletion date     *
//*                                 is >= this timestamp.                      *
//*                         Range: 1 through 24 hours  (absolute value)        *
//*                                                                            *
//*                                                                            *
//* Returns: number of items succesfully deleted from trashcan                 *
//*            (not the file count)                                            *
//*          ZERO if preprocessing error or user aborted the operation         *
//******************************************************************************
//* Notes:                                                                     *
//* a) There are three flavors of emptying the trashcan:                       *
//*    1. '-E' option: User interactively selects which items are to           *
//*       be deleted. (overrides the '--remove' and '-e' options)              *
//*    2. '--remove' option: Select items for deletion based on fressness      *
//*       date ('hours' parameter). (overrides '-e' option)                    *
//*    3. '-e' option: Empty all trashcan contents.                            *
//*                                                                            *
//* b) The 'select' field of the tcDetailInfo record is used to flag the       *
//*    items selected for deletion. Note that for the '-e' option, all data,   *
//*    not just valid trashcan items are deleted.                              *
//*                                                                            *
//* c) The trashcan must contain valid data records UNLESS we are completely   *
//*    emptying the trashcan. In that case we ignore record errors.            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short cTrash::ctEmptyTrash ( bool interact, int remTime )
{
   tcSummaryInfo tcSummary ;        // trashcan summary data
   tcDetailInfo* tcDetail = NULL ;  // pointer to trashcan detail data
   gString gsOut, gsIn,             // formatted output and user response
           fmtItems, fmtFiles, fmtBytes ;
   UINT64 iBytes = ZERO,            // number of bytes selected for deletion
          dBytes = ZERO ;           // number of bytes actually deleted
   UINT   iTotal  = ZERO,           // number of valid items in trashcan
          iSelect = ZERO,           // number of items selected for deletion
          dSelect = ZERO,           // number of items actually deleted (return value)
          iFiles  = ZERO,           // number of files selected for deletion
          dFiles  = ZERO ;          // number of files actually deleted
   bool userAbort = false ;         // true if user abort OR if preprocessing error

   //* Get a summary of the trashcan contents *
   if ( (this->cttSummaryScan ( tcSummary )) != false )
   {
      //* Get detailed statistics and display data on trashcan contents.*
      if ( tcSummary.items > ZERO )
      {
         if ( (iTotal = this->cttDetailScan ( tcSummary.iFiles, tcDetail )) == ZERO )
         {  //* Trashcan is corrupted, AND contains no valid records. *
            //* This constitutes an error UNLESS caller wants to      *
            //* completely empty the trashcan.                        *
            if ( interact != false || remTime != ZERO )
            {
               if ( !this->silent )
                  this->textOut ( L"Operation terminated. "
                                   "Trashcan contains no valid records." ) ;
               userAbort = true ;
            }
         }
      }
      else
      {  //* Trashcan is empty *
         if ( !this->silent )
         {
            if ( this->verbose )    // Display summary statistics
               this->ctReportTrashcan ( false ) ;

            //* User is a dufus: Report empty trashcan.*
            this->textOut ( L"Operation terminated. Trashcan is empty." ) ;
         }
         //* It isn't really the user, but the user's *
         //* stupidity that caused the abort.         *
         userAbort = true ;
      }
   }
   else
   {  //* Trashcan is unreachable. cTrash will have validated the trashcan *
      //* earlier, so we will only arrive here if this method is called    *
      //* directly by another application.                                 *
      this->textOut ( "Operation terminated. Trashcan is unreachable." ) ;
      userAbort = true ;
   }

   //* If no errors in reading trashcan contents,     *
   //* select the item(s) to be removed from trashcan.*
   if ( !userAbort )
   {
      //*************************************************
      //** If interactive selection of items specified **
      //*************************************************
      if ( interact != false )
      {
         if ( (iSelect = this->ctPromptInteractive ( tcDetail, iTotal, false )) > ZERO )
         {  //* Update the accumulators *
            for ( UINT i = ZERO ; i < iTotal ; ++i )
            {
               if ( tcDetail[i].select != false )
               {
                  iFiles += tcDetail[i].files ;
                  iBytes += tcDetail[i].size ;
               }
            }
         }
      }

      //***********************************************************
      //** If deleting items older than specified number of days **
      //***********************************************************
      else if ( remTime > ZERO )
      {
         if ( remTime <= MAX_REM )
         {
            //* Get the system local time and calculate the freshness *
            //* date. Items with deletion dates <= 'freshDate' will   *
            //* be tagged for deletion.                               *
            localTime lt ;
            this->ctfGetLocalTime ( lt ) ;
            int64_t freshDate = lt.epoch - (remTime * 60) ;
            for ( UINT i = ZERO ; i < iTotal ; ++i )
            {
               if ( tcDetail[i].tDate.epoch <= freshDate )
               {
                  tcDetail[i].select = true ;
                  ++iSelect ;
                  iFiles += tcDetail[i].files ;
                  iBytes += tcDetail[i].size ;
               }
            }
            if ( iSelect == ZERO && !this->silent )
               this->textOut ( L"No items meet date criterion. Operation terminated." ) ;
         }
         else if ( !this->silent )
         {  //* cTrash range checks all input, so this will happen only *
            //* if this method is called through the public method by   *
            //* another application.                                    *
            gsOut.compose( "Error! Specified freshness date is out-of-range.\n"
                           "Operation terminated." ) ;
            this->textOut ( gsOut ) ;
            userAbort = true ;
         }
      }

      //************************************************************
      //** If deleting items newer than specified number of hours **
      //************************************************************
      else if ( remTime < ZERO )
      {
         if ( remTime >= MIN_REM )
         {
            //* Get the system local time and calculate the freshness *
            //* date. Items with deletion dates <= 'freshDate' will   *
            //* be tagged for deletion.                               *
            localTime lt ;
            this->ctfGetLocalTime ( lt ) ;
            int64_t freshDate = lt.epoch + (remTime * 60) ;
            for ( UINT i = ZERO ; i < iTotal ; ++i )
            {
               if ( tcDetail[i].tDate.epoch >= freshDate )
               {
                  tcDetail[i].select = true ;
                  ++iSelect ;
                  iFiles += tcDetail[i].files ;
                  iBytes += tcDetail[i].size ;
               }
            }
            if ( iSelect == ZERO && !this->silent )
               this->textOut ( L"No items meet date criterion. Operation terminated." ) ;
         }
         else if ( !this->silent )
         {  //* cTrash range checks all input, so this will happen only *
            //* if this method is called through the public method by   *
            //* another application.                                    *
            gsOut.compose( "Error! Specified freshness date is out-of-range.\n"
                           "Operation terminated." ) ;
            this->textOut ( gsOut ) ;
            userAbort = true ;
         }
      }

      //***************************************************
      //** Else, we are to delete ALL trashcan contents. **
      //***************************************************
      else
      {
         //* Get pre-confirmation (if specified) *
         if ( this->confirm != false )
         {
            this->textOut ( L"Deleting all Trashcan contents. This operation cannot be undone.\n"
                             "  continue? (y/n): ", false ) ;
            this->ctUserResponse ( gsIn ) ;
            wchar_t resp = *gsIn.gstr() ;
            if ( resp != L'y' && resp != L'Y' )
            {
               this->textOut ( L"Operation cancelled. (Trashcan data unchanged.)" ) ;
               iTotal = ZERO ;         // prevent summary report
               userAbort = true ;
            }
         }

         //* This option does not rely on item selection, but deletes       *
         //* everything in the trashcan's 'files' and 'info' subdirectories.*
         if ( !userAbort )
         {
            if ( (this->cttEmptyTrash ()) != false )
            {
               dSelect = tcSummary.iFiles ;
               dFiles  = tcSummary.files ;
               dBytes  = tcSummary.tSpace ;
            }
            else if ( !this->silent )
            {  //* This kind of error is unlikely if user owns the trashcan.*
               tcSummaryInfo tcsumm ;
               this->cttSummaryScan ( tcsumm ) ;
               if ( tcsumm.tSpace > ZERO )
               {
                  this->textOut ( L"Access Error: some data may remain in Trashcan." ) ;
                  dSelect = tcSummary.iFiles - tcsumm.iFiles ;
                  dFiles  = tcSummary.files - tcsumm.files ;
                  dBytes  = tcSummary.tSpace - tcsumm.tSpace ;
               }
            }
         }
      }
   }

   //* For interactive and date-based item selection, delete specified items.*
   if ( !userAbort && iSelect > ZERO )
   {
      gString origName ;      // for reporting item names

      //* Get pre-confirmation (if specified) *
      if ( this->confirm != false )
      {
         if ( this->verbose )
         {  //* List the individual items *
            for ( UINT i = ZERO ; i < iTotal ; ++i )
            {
               if ( tcDetail[i].select != false )
               {
                  this->ctfExtractFilename ( origName, tcDetail[i].trgPath ) ;
                  gsOut.compose( "%-24S (%04hd-%02hd-%02hd %02hd:%02hd:%02hd)", 
                                 origName.gstr(),
                                 &tcDetail[i].tDate.year, &tcDetail[i].tDate.month, 
                                 &tcDetail[i].tDate.date, &tcDetail[i].tDate.hours, 
                                 &tcDetail[i].tDate.minutes, &tcDetail[i].tDate.seconds ) ;
                  this->textOut ( gsOut ) ;
               }
            }
         }

         fmtItems.formatInt( iSelect, 6, true ) ;
         fmtFiles.formatInt( iFiles, 6, true ) ;
         fmtBytes.formatInt( iBytes, 6, true ) ;
         gsOut.compose( L"\nDeleting %S items (%S files and %S bytes) from Trashcan.\n"
                         "  This operation cannot be undone.  continue? (y/n): ",
                        fmtItems.gstr(), fmtFiles.gstr(), fmtBytes.gstr() ) ;
         this->textOut ( gsOut, false ) ;
         this->ctUserResponse ( gsIn ) ;
         wchar_t resp = *gsIn.gstr() ;
         if ( resp != L'y' && resp != L'Y' )
         {
            this->textOut ( L"Operation cancelled. (Trashcan data unchanged.)" ) ;
            iTotal = ZERO ;         // prevent summary report
            userAbort = true ;
         }
      }

      for ( UINT i = ZERO ; i < iTotal ; ++i )
      {
         this->ctfExtractFilename ( origName, tcDetail[i].trgPath ) ;
         if ( tcDetail[i].select != false )
         {
            if ( (this->cttDeleteItemFromTrash ( tcDetail[i] )) != false )
            {  //* Item successfully deleted. Update accumulators. *
               ++dSelect ;
               dFiles += tcDetail[i].files ;
               dBytes += tcDetail[i].size ;

               if ( this->verbose )
               {
                  gsOut.compose( "%-24S (deleted)", origName.gstr() ) ;
                  this->textOut ( gsOut ) ;
               }
            }
            else if ( !this->quiet )   // error deleting item
            {
                  gsOut.compose( "%-24S (NOT deleted)", origName.gstr() ) ;
               this->textOut ( gsOut ) ;
            }
         }
      }
   }

   //* Report our success *
   if ( (iTotal > ZERO) && !this->silent )
   {
      if ( !this->quiet || interact )
      {
         fmtItems.formatInt( dSelect, 6, true ) ;
         fmtFiles.formatInt( dFiles, 6, true ) ;
         fmtBytes.formatInt( dBytes, 6, true ) ;
         gsOut.compose( L"\n%S items (%S files) deleted from Trashcan.\n"
                         "%S bytes of disk space recovered.",
                        fmtItems.gstr(), fmtFiles.gstr(), fmtBytes.gstr() ) ;
         this->textOut ( gsOut ) ;
      }
   }

   if ( tcDetail != NULL )       // if we have a dynamic allocation, release it
      delete [] tcDetail ;

   return dSelect ;

}  //* End ctEmptyTrash() *

//***********************
//*   ctScanTrashcan    *
//***********************
//******************************************************************************
//* Scan the files: 'Trash/info/*.trashinfo' and translate their data into     *
//* human readable form. Sort the records according to current sort option.    *
//*                                                                            *
//* IMPORTANT NOTE:                                                            *
//* This method performs dynamic memory allocation for the records attached    *
//* to the 'tcdi' parameter. To avoid memory leaks, be sure to release any     *
//* allocation attached to 'tcdi' pointer:  delete [] tcdi ;                   *
//*                                                                            *
//* NOTE: This method is not called by the cTrash application itself.          *
//*       In order to better respond to error conditions, cTrash calls the     *
//*       summary-scan and detail-scan methods separately.                     *
//*       This method is for use by applications which integrate the cTrash    *
//*       source code into their own code and want to optionally format the    *
//*       output in their own style.                                           *
//*                                                                            *
//* Input  : tcsi  : receives summary data for trashcan contents               *
//*          tcdi  : receives a pointer to an array of data records, one for   *
//*                  each _verified_ record in the Trash/info subdirectory     *
//*                                                                            *
//* Returns: number of '.trashinfo' files scanned and verified                 *
//*          i.e. the number of valid records written to tcdi                  *
//*          If return value != tcsi.iFiles, then trashcan is partially corrupt*
//*          .trashinfo records without matching item in trash are not reported*
//*          items in trash without matching .trashinfo record are not reported*
//******************************************************************************

UINT cTrash::ctScanTrashcan ( tcSummaryInfo& tcsi, tcDetailInfo*& tcdi )
{
   UINT goodiCount = ZERO ;      // return value

   //* Get summary data, and if successful, get detailed data *
   if ( ((this->cttSummaryScan ( tcsi )) != false) && tcsi.iFiles > ZERO )
      goodiCount = this->cttDetailScan ( tcsi.iFiles, tcdi ) ;

   return goodiCount ;

}  //* End ctScanTrashcan() *

//*************************
//*       textOut         *
//*************************
//******************************************************************************
//* All, or nearly all text written to the display goes through this method,   *
//* so we can control whether it goes through stdout (wide stream) or through  *
//* the NcDialog output stream.                                                *
//*                                                                            *
//* Input  : tOut   : text data to be displayed                                *
//*          newln  : (optional, true by default)                              *
//*                   if 'true', then terminate the line with an 'endl'        *
//*                   if 'false', do not termnate the line                     *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We channel as much of the stdout stream as possible     *
//* through this method to avoid future effort if we later add the NcDialog    *
//* (ncurses) interface which is incompatible with stdin/stdout.               *
//*                                                                            *
//* We may want to color-code the output, even if the NcDialog API is not      *
//* used. The following is a table of ANSI color-code sequences.               *
//* Think about this...                                                        *
//*                                                                            *
//* ANSI Colors (partial list):                                                *
//* =========== Foreground  Background   ============ Foreground  Background   *
//* Black       \033[0;30m  \033[0;40m   Dark Gray    \033[1;30m  \033[1;40m   *
//* Red         \033[0;31m  \033[0;41m   Bold Red     \033[1;31m  \033[1;41m   *
//* Green       \033[0;32m  \033[0;42m   Bold Green   \033[1;32m  \033[1;42m   *
//* Yellow      \033[0;33m  \033[0;43m   Bold Yellow  \033[1;33m  \033[1;43m   *
//* Blue        \033[0;34m  \033[0;44m   Bold Blue    \033[1;34m  \033[1;44m   *
//* Purple      \033[0;35m  \033[0;45m   Bold Purple  \033[1;35m  \033[1;45m   *
//* Cyan        \033[0;36m  \033[0;46m   Bold Cyan    \033[1;36m  \033[1;46m   *
//* Light Gray  \033[0;37m  \033[0;47m   White        \033[1;37m  \033[1;47m   *
//*                                                                            *
//******************************************************************************

void cTrash::textOut ( const gString& tOut, bool newln )
{
   if ( this->textMode != false )
   {
      wcout << tOut.gstr() ;
      if ( newln != false )
         wcout << endl ;
   }
   else
   {
      /* Interactive Mode not yet implemented. */
   }

}  //* End textOut() *
void cTrash::textOut ( const char* tOut, bool newln )
{
   gString gs( tOut ) ;
   this->textOut ( gs, newln ) ;

}  //* End textOut() *
void cTrash::textOut ( const wchar_t* tOut, bool newln )
{
   gString gs( tOut ) ;
   this->textOut ( gs, newln ) ;

}  //* End textOut() *

//*************************
//*    ctuserResponse     *
//*************************
//******************************************************************************
//* Get user response to text-mode prompts.                                    *
//*                                                                            *
//* Input  : gsIn   : (by reference) receives user input                       *
//*                                                                            *
//* Returns: 'true'  if data successfully received                             *
//*          'false' if input error (gsIn will contain an empty string)        *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* - Buffered input, Blocking read (which we hate).                           *
//*   We wait indefinitely for user input.                                     *
//* - Direct, single-token user input is straightforward: stdin expects        *
//*   character input, and token is terminated by SPACE character or ENTER key.*
//*   However, we don't see the token until ENTER is pressed.                  *
//* - Input may be redirected from a response file, but to us, there is no     *
//*   apparent difference.                                                     *
//*   - If the token comes from a response file, then the shell does not echo  *
//*     it to the display. This throws off the formatting of the prompt, but   *
//*     the functionality is not affected.                                     *
//* - Token must consist of printing characters only (we never see the ENTER). *
//*   NOTE: A printing character is an codepoint which is defined by the       *
//*         current locale to be a printing character. For the C locale, this  *
//*         is basically ASCII; however, we use the locale specified by the    *
//*         user's environment, (generally a UTF-8 locale), so we get maximum  *
//*         functionality.                                                     *
//******************************************************************************

bool cTrash::ctUserResponse ( gString& gsIn )
{
   wchar_t userInput[gsMAXCHARS] = L"" ;  // input buffer
   bool  status = true ;                  // return value

   gsIn.clear() ;             // clear caller's old data
   
#if 1    // NEW
   wcin.getline( userInput, gsMAXCHARS ) ;
#else    // OLD
   wcin >> userInput ;        // get user's response
#endif   // OLD
   for ( short i = ZERO ; userInput[i] != NULLCHAR ; i++ )
   {
      if ( !(iswprint ( userInput[i] )) )
      {
         status = false ;
         break ;
      }
   }
   if ( status != false )
      gsIn = userInput ;

   return status ;

}  //* End ctUserResponse() *

//*************************
//* ctGetCommandLineArgs  *
//*************************
//******************************************************************************
//* Capture user's command-line arguments.                                     *
//*                                                                            *
//* Valid Arguments: (see DisplayCommandLineHelp())                            *
//*                                                                            *
//* Input  : ca   : commArgs class object (by reference)                       *
//*                                                                            *
//* Returns: 'true' if all arguments are valid                                 *
//*          'false' if invalid argument(s)                                    *
//******************************************************************************
//* Text Mode:                                                                 *
//*  no args : error msg                                                       *
//*  filename(s) only: Move specified file(s) to trashcan                      *
//*  Request for info overides --silent and -q.                                *
//*  --help -h or arg error overrides all switches except --version            *
//*  --version overrides all other switches                                    *
//*  See ctDisplayCommandLineHelp method for full list of options.             *
//*                                                                            *
//* User's can occasionally be outrageously obtuse, so if conflicting options  *
//* are specified, the following logic applies.                                *
//*                                                                            *
//*  Option Logic (standard mode)                                              *
//*   a) '--version'     overrides everything else                             *
//*   b) '--help' '-h'   overrides everything except '--version'               *
//*   c) '-s' '-S'       Statistics overrides '-r', '-R', '-t', '-e', '-E'
//*   d) '-s' '-S'       Detailed statistics '-S' overrides '-s'               *
//*   e) '-r' '-R'       Restore overrides '-t', '-e', '-E'                    *
//*   f) '-r' '-R'       Interactive restore '-R' overrides restore-recent '-r'*
//*   g) '-t'            overrides empty ('-e', '-E')                          *
//*                      must be accompanied by one or more filenames          *
//*   h) '-e' '-E'       Interactive empty ('-E') overrides both empty-all     *
//*      '--remove'      ('-e') and '--remove'. '--remove' over '-e'.          *
//*   i) '--silent'      Suppresses all output for file movements/deletions,   *
//*                      but has no effect on '-s' '-S' '-E'                   *
//*   j) '--quiet' '-q'  Suppresses all output EXCEPT ERRORS,                  *
//*                      but has no effect on '-s' '-S' '-E'                   *
//*   k) '--verbose' '-v' Verbose output overrides both 'silent and 'quiet'    *
//*   l) '--preconfirm' '-p'  Pre-confirm overrides '--silent' '-q'            *
//*   m) '--delete' '-d' Least significant AND most dangerous option is        *
//*                      overridden by all other options                       *
//*   n) '--no-confirm'  No confirmation for operations that cannot be undone. *
//*                      deletions ('-e' '--delete' '-d')                      *
//*                                                                            *
//*                                                                            *
//* Interactive Mode:                                                          *
//*  a) Interactive Mode not implemented in this release.                      *
//*                                                                            *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//* Undocumented options:                                                      *
//* =====================                                                      *
//* '--sdd'  For development only. When moving items to the trashcan,          *
//*          synthesize the deletion date for later testing of sort, recovery, *
//*          and removal of data items.                                        *
//*                                                                            *
//******************************************************************************

bool cTrash::ctGetCommandLineArgs ( commArgs& ca )
{
   #define DEBUG_GCA (0)

   gString gs( ca.argList[1] ) ; // text formatting
   bool status = true ;          // return value (hope for the best)

   if ( (ca.argCount > 1) && ((gs.compare( L"--rm-emulate" )) != ZERO) )
   {
      #if DEBUG_GCA != 0  // debugging only
      wcout << L"DEBUG_GCA:" << endl ;
      for ( int i = 1 ; i < ca.argCount ; i++ )
         wcout << L"   " << ca.argList[i] << endl ;
      #endif   // DEBUG_GCA

      //* Open a temporary file to hold the filenames to be processed.*
      ofstream ofs( this->listFile.ustr(), ofstream::out | ofstream::trunc ) ;

      short j = ZERO ;           // for multiple arguments in same token
      bool multiarg = false ;

      for ( short i = 1 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            multiarg = false ;
            ++j ;
            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               gs = ca.argList[i] ;

               //* Produce detailed output to display *
               if ( (gs.compare( L"--verbose" )) == ZERO )
               {
                  ca.verbose = true ;
               }

               //* Produce displayoutput only for error conditions *
               else if ( (gs.compare( L"--quiet" )) == ZERO )
               {
                  ca.quiet = true ;
               }

               //* Produce no display output *
               else if ( (gs.compare( L"--silent" )) == ZERO )
               {
                  ca.silent = true ;
                  ca.confirm = false ;
               }

               //* Ask for user confirmation BEFORE move/restore operations *
               else if ( (gs.compare( L"--preconfirm" )) == ZERO )
               {
                  ca.preconf = true ;
               }

               //* Do not ask for user confirmation before critical operations *
               else if ( (gs.compare( L"--no-confirm" )) == ZERO )
               {
                  ca.confirm = false ;
               }

               //* Delete items from the trashcan based on freshness date *
               //* --remove=[DAYS (1 - 1827) | HOURS (-24 - -1) | 'today']
               //*   DAYS range : 1 through 1827  (placed in trash before or equal to DAYS
               //*   HOURS range: -24 through -1  (placed in trash within previous HOURS hours)
               //*   'today'    : placed in trash since 00:00:00 system local time
               else if ( (gs.compare( L"--remove", true, 8 )) == ZERO )
               {
                  bool syntaxError = false ;
                  short nIndex ;
                  if ( (nIndex = gs.find( L'=' )) > ZERO )
                  {
                     ++nIndex ;
                     gs.shiftChars( -(nIndex) ) ;
                     int units, minutes ;
                     //* Test for the 'today' token.                    *
                     if ( (gs.compare( L"today", false, 5 )) == ZERO )
                     {
                        //* Get the current system local time, and round up to *
                        //* the next whole minute. The negative of this was    *
                        //* (approximately) 00:00:00 (midnight) this morning.  *
                        localTime lt ;
                        this->ctfGetLocalTime ( lt ) ;
                        minutes = (lt.hours * 60) + lt.minutes ;
                        if ( lt.seconds > ZERO )
                           ++minutes ;
                        ca.remTime = -(minutes) ;
                        ca.empty   = true ;
                        ca.details = false ;
                     }
                     else if ( (swscanf ( gs.gstr(), L"%d", &units )) == 1 )
                     {
                        bool rangeError = false ;
                        if ( units > ZERO )  // 'units' represents days
                        {  
                           //* Convert days to minutes *
                           if ( units <= 366 )
                           {
                              ca.remTime = (units * 24 * 60) ;
                              ca.empty   = true ;
                              ca.details = false ;
                           }
                           else
                              rangeError = true ;
                        }
                        else if ( units < ZERO )   // 'units' represents hours
                        {
                           //* Convert hours to minutes *
                           if ( units >= (-24) )
                           {
                              ca.remTime = (units * 60) ;
                              ca.empty   = true ;
                              ca.details = false ;
                           }
                           else
                              rangeError = true ;
                        }
                        else
                           rangeError = true ;

                        if ( rangeError )
                        {
                           gs = "Parameter out of range: '--remove' " ;
                           ca.errMsg.append( gs.gstr() ) ;
                           status = false ;
                           break ;
                        }
                     }
                     else
                        syntaxError = true ;
                  }
                  else
                     syntaxError = true ;

                  if ( syntaxError )
                  {
                     gs = "Invalid syntax for: '--remove' " ;
                     ca.errMsg.append( gs.gstr() ) ;
                     status = false ;
                     break ;
                  }
               }

               //* Output sort option *
               else if ( (gs.compare( L"--sort", true, 6 )) == ZERO )
               {
                  bool syntaxError = false ;
                  short nIndex ;
                  if ( (nIndex = gs.find( L'=' )) > ZERO )
                  {
                     ++nIndex ;
                     gs.shiftChars( -(nIndex) ) ;
                     const wchar_t op1 = gs.gstr()[0] ;
                     switch ( op1 )
                     {
                        case L'd':  ca.sortOption = osDATE ;      break ;
                        case L'D':  ca.sortOption = osDATE_R ;    break ;
                        case L's':  ca.sortOption = osSIZE ;      break ;
                        case L'S':  ca.sortOption = osSIZE_R ;    break ;
                        case L'n':  ca.sortOption = osNAME ;      break ;
                        case L'N':  ca.sortOption = osNAME_R ;    break ;
                        default:    syntaxError = true ;          break ;
                     } ;
                  }
                  else
                     syntaxError = true ;

                  if ( syntaxError )
                  {
                     gs = "Invalid syntax for: '--sort' " ;
                     ca.errMsg.append( gs.gstr() ) ;
                     status = false ;
                     break ;
                  }
               }

               //* Read filenames from specified text file *
               else if ( (gs.compare( L"--file-list", true, 11 )) == ZERO )
               {
                  short nIndex ;
                  if ( (nIndex = gs.find( L'=' )) > ZERO )
                  {
                     ++nIndex ;
                     gs.shiftChars( -(nIndex) ) ;
                     ifstream ifs( gs.ustr(), ifstream::in ) ;
                     if ( ifs.is_open() )
                     {
                        while ( (this->ctfReadLine ( ifs, gs )) != false )
                        {
                           ofs << gs.ustr() << endl ;
                           ++ca.itemCount ;
                        }
                        ifs.close() ;
                     }
                     else
                     {
                        ca.errMsg.append( "List file not found: '%S'", gs.gstr() ) ;
                        status = false ;
                        break ;
                     }
                  }
                  else
                  {
                     gs = "Invalid syntax for: '--file-list' " ;
                     ca.errMsg.append( gs.gstr() ) ;
                     status = false ;
                     break ;
                  }
               }

               //* Alternate restoration target *
               else if ( (gs.compare( L"--alt-target", true, 12 )) == ZERO )
               {
                  bool syntaxError = false ;
                  short nIndex ;
                  if ( (nIndex = gs.find( L'=' )) > ZERO )
                  {
                     ++nIndex ;
                     gs.shiftChars( -(nIndex) ) ;
                     if ( gs.gschars() > 1 )
                        gs.copy( ca.altTarget, gsMAXBYTES ) ;
                     else
                        syntaxError = true ;
                  }
                  else
                     syntaxError = true ;

                  if ( syntaxError )
                  {
                     gs = "Invalid syntax for: '--alt-target' " ;
                     ca.errMsg.append( gs.gstr() ) ;
                     *ca.altTarget = NULLCHAR ;
                     status = false ;
                     break ;
                  }
               }

               //* If user has specified to 'unlink' the target files, *
               //* bypassing the trashcan entirely.                    *
               else if ( (gs.compare( L"--delete" )) == ZERO )
               {
                  ca.permdel = true ;
               }

               //* Specify alternate location for trashcan.            *
               //* If not specified, default location used (see below).*
               else if ( (gs.compare( L"--trash-path", true, 12 )) == ZERO )
               {
                  short nIndex ;
                  if ( (nIndex = gs.find( L'=' )) > ZERO )
                  {
                     ++nIndex ;
                     gs.shiftChars( -(nIndex) ) ;
                     gs.copy( ca.trashDir, gsMAXBYTES ) ;
                  }
                  else
                  {
                     gs = "Invalid syntax for: '--trash-path' " ;
                     ca.errMsg.append( gs.gstr() ) ;
                     status = false ;
                     break ;
                  }
               }

               //* Specify maximum width of an output line *
               else if ( (gs.compare( L"--stat-cols", true, 11 )) == ZERO )
               {
                  bool syntaxError = false ;
                  short nIndex ;
                  if ( (nIndex = gs.find( L'=' )) > ZERO )
                  {
                     ++nIndex ;
                     gs.shiftChars( -(nIndex) ) ;
                     if ( (swscanf ( gs.gstr(), L"%hd", &ca.dispWidth )) == 1 )
                     {
                        if ( ca.dispWidth < fswMIN || ca.dispWidth > fswMAX )
                        {
                           ca.dispWidth = ZERO ;
                           gs = "'--stat-cols' value out of range " ;
                           ca.errMsg.append( gs.gstr() ) ;
                           status = false ;
                           break ;
                        }
                     }
                     else
                        syntaxError = true ;
                  }
                  else
                     syntaxError = true ;

                  if ( syntaxError )
                  {
                     gs = "Invalid syntax for: '--stat-cols' " ;
                     ca.errMsg.append( gs.gstr() ) ;
                     status = false ;
                     break ;
                  }
               }

               //* '--rm-emulate' must be the first argument on command line *
               else if ( (gs.compare( L"--rm-emulate" )) == ZERO )
               {
                  ca.errMsg.append( L"'--rm-emulate' option invalid in this context " ) ;
                  status = false ;
                  break ;
               }

               //* Request for application version number *
               else if ( (gs.compare( L"--version" )) == ZERO )
               {
                  ca.helpFlag = ca.verFlag = true ;
                  break ;
               }

               //* Long-form request for Help *
               else if ( (gs.compare( L"--help" )) == ZERO )
               {
                  ca.helpFlag = true ;
                  break ;
               }

               #if DEBUG_OPTIONS != 0
               //* Development Only: set a specific deletion date, overriding *
               //* the current system local time.                             *
               //* Two arguments (see top of this module):                    *
               //* sddunit must be 'd' (days) or 'h' (hours)                  *
               //* If 'd' : sddval must be > ZERO                             *
               //* If 'h' : sddval must be > ZERO && <= 24                    *
               else if ( (gs.compare( L"--sdd", true, 5 )) == ZERO )
               {
                  short nIndex ;
                  if ( (nIndex = gs.find( L'=' )) > ZERO )
                  {
                     ++nIndex ;
                     gs.shiftChars( -(nIndex) ) ;
                     if ( (swscanf ( gs.gstr(), L"%hd%c", &sddval, &sddunit )) == 2 )
                     {
                        if ( (sddunit == 'd' && sddval > ZERO) ||
                             (sddunit == 'h' && (sddval > ZERO && sddval <= 24)) )
                           { /* good values */ }
                        else { sddval = ZERO ; sddunit = ' ' ; }
                     }
                     else { sddval = ZERO ; sddunit = ' ' ; }
                  }
               }
               #endif // DEBUG_OPTIONS

               else           // invalid switch
               {  //* Generate an error mesage *
                  gs.compose( "Unrecognized command: '%s' ", ca.argList[i] ) ;
                  ca.errMsg.append( gs.gstr() ) ;
                  status = false ;
                  break ;
               }
               continue ;     // finished with this argument
            }  // (double dash)

            char argLetter = ca.argList[i][j] ;
            switch ( argLetter )
            {
               case 't':   ca.restore = false ;                      break ;
               case 'r':   ca.restore = true ; ca.details = false ;  break ;
               case 'R':   ca.restore = ca.details = true ;          break ;
               case 'e':   ca.empty = true ; ca.details = false ;    break ;
               case 'E':   ca.empty = ca.details = true ;            break ;
               case 's':   ca.report = true ; ca.details = false ;   break ;
               case 'S':   ca.report = ca.details = true ;           break ;
               case 'p':   ca.preconf = true ;                       break ;
               case 'v':   ca.verbose = true ;                       break ;
               case 'q':   ca.quiet = true ;                         break ;
               case 'd':   ca.permdel = true ;                       break ;
               case 'h':   ca.helpFlag = true ;                      break ;
               default:
                  gs.compose( "Unrecognized command: '-%c' ", &argLetter ) ;
                  ca.errMsg.append( gs.gstr() ) ;
                  status = false ;
                  break ;
            }
            if ( (status != false) && ca.argList[i][j + 1] != NULLCHAR )
               multiarg = true ;

            //* If request for help, we're done *
            if ( ca.helpFlag != false )
               break ;
         }
         else
         {  //* Interpret argument as a filename. *
            if ( ofs.is_open() )
            {
               ofs << ca.argList[i] << endl ;
               ++ca.itemCount ;
            }
         }
      }     // for(;;)

      //* If sending item to trashcan, BUT no items specified.*
      if ( (status != false) && (ca.restore == false) && (ca.itemCount == ZERO)
           && (ca.empty == false) && (ca.report == false) )
      {
         ca.errMsg.append( NoFiles ) ;
         status = false ;
      }

      //* If alternate trashcan specified, verify its exsistence. *
      //* Otherwise, verify existence of default local trashcan.  *
      if ( status != false )
      {
         gString rawTrash, realTrash ;
         if ( *ca.trashDir != NULLCHAR )
            rawTrash = ca.trashDir ;
         else
         {
            for ( short i = ZERO ; ca.envList[i] != NULL ; ++i )
            {
               gs = ca.envList[i] ;
               if ( (gs.find( "HOME=", ZERO, true, 5 )) == ZERO )
               {
                  rawTrash.compose( defaultTRASHDIR, &gs.gstr()[5] ) ;
                  break ;
               }
            }
         }
         if ( (status = this->ctfValidateTrashPath ( rawTrash, realTrash )) != false )
            realTrash.copy( ca.trashDir, gsMAXBYTES ) ; // trashcan configuration verified
         else
         {  //* Generate an error mesage.                                 *
            //* (there is a small possibility of message truncation here) *
            gs.compose( "Trashcan directory not found or improperly configured.\n"
                        "       '%S' ", rawTrash.gstr() ) ;
            ca.errMsg.append( gs.gstr() ) ;
         }
      }

      //************************************
      //** Apply Option Logic (see above) **
      //************************************
      if ( ca.verFlag != false || ca.helpFlag != false )
      {
         bool v = ca.verFlag ;
         ca.reset() ;
         ca.verFlag = v ;
         ca.helpFlag = !v ;
      }
      if ( ca.verbose || ca.quiet || ca.silent )
      {  //* Output control flags have logical overlap *
         if ( ca.verbose )
            ca.quiet = ca.silent = false ;
         else if ( ca.quiet )
            ca.silent = false ;
      }
      if ( ca.preconf || (ca.details && (ca.restore || ca.empty)) )
      {  //* 'silent' and 'quiet' do not apply to interactive operations.*
         ca.silent = ca.quiet = false ;
      }

      //*********************************************
      //* If we have an open output file, close it. *
      //*********************************************
      if ( ofs.is_open() )
         ofs.close() ;
      else if ( !ca.verFlag && !ca.helpFlag )
      {  //* Temp file creation error. *
         ca.itemCount = ZERO ;   // report that no filenames specified
         ca.errMsg.append( badLIST ) ;
         status = false ;
      }

      #if DEBUG_GCA != 0  // debugging only
      wcout << L"\nDEBUG_GCA:" << endl ;
      gs.compose( "trashDir : %s\n"
                  "altTarget: %s\n"
                  "itemCount:%2hd  staleDays:%2hd dispWidth :%3hd  permdel : %hhd\n"
                  "textFlag : %hhd  helpFlag : %hhd   verFlag : %hhd\n"
                  " restore : %hhd     empty : %hhd   preconf : %hhd\n"
                  " confirm : %hhd    report : %hhd   details : %hhd\n"
                  "  silent : %hhd     quiet : %hhd   verbose : %hhd\n"
                  ,
                  ca.trashDir, ca.altTarget, 
                  &ca.itemCount, &ca.staleDays, &ca.dispWidth, &ca.permdel, 
                  &ca.textFlag, &ca.helpFlag, &ca.verFlag, 
                  &ca.restore, &ca.empty, &ca.preconf, 
                  &ca.confirm, &ca.report, &ca.details, 
                  &ca.silent, &ca.quiet, &ca.verbose ) ;
      wcout << gs.gstr() << endl ;
      #endif   // DEBUG_GCA
   }        // if(ca.argCount>1 && !rm-emulate)
   else if ( (gs.compare( L"--rm-emulate" )) == ZERO )
   {
      status = this->ctGetRM_Args ( ca ) ;
   }
   else                          // no arguments specified
   {
      ca.errMsg.append( NoFiles ) ;
      status = false ;
   }
   return status ;

   #undef DEBUG_GCA
}  //* End ctGetCommandLineArgs() *

#if DEBUG_OPTIONS != 0
//*************************
//*     EncodeDelDate     *
//*************************
//******************************************************************************
//* DEVELOPMENT ONLY -- NON-MEMBER METHOD                                      *
//* Synthesize a deletion timestamp which is older than the current system     *
//* local time.                                                                *
//* Used with the '-sdd' debugging option.                                     *
//* See static variables 'sddval' and 'sddunit' at the top of this module.     *
//*                                                                            *
//* Input  : lt  : contains current system local time, and                     *
//*                receives the modified time                                  *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void EncodeDelDate ( localTime& lt )
{
   if ( (sddval > ZERO) && (sddunit == 'd' || sddunit == 'h') && (lt.sysepoch >= ZERO) )
   {
      time_t offset ;      // time offset in seconds
      if ( sddunit == 'h' )
         offset = (sddval * 60 * 60) ;
      else
         offset = (sddval * 60 * 60 * 24) ;
      lt.sysepoch -= offset ;
      lt.epoch = (int64_t)lt.sysepoch ;      // promote to 64-bit

      Tm    tm ;                             // Linux time structure
      if ( (localtime_r ( &lt.sysepoch, &tm )) != NULL )
      {
         //* Translate to localTime format *
         lt.day      = ZERO ;                // day-of-week is not initialized
         lt.date     = tm.tm_mday ;          // today's date
         lt.month    = tm.tm_mon + 1 ;       // month
         lt.year     = tm.tm_year + 1900 ;   // year
         lt.hours    = tm.tm_hour ;          // hour
         lt.minutes  = tm.tm_min ;           // minutes
         lt.seconds  = tm.tm_sec ;           // seconds
      }
   }
}  //* End EncodeDelDate() *
#endif // DEBUG_OPTIONS

//*************************
//*     ctGetRM_Args      *
//*************************
//******************************************************************************
//* 'rm' command compatible parameter interface.                               *
//* All valid arguments ('rm' version 8.21) are recognized, but only partial   *
//* compliance is implemented.                                                 *
//*                                                                            *
//* 'rm' compatibility applies ONLY when moving items TO the trashcan.         *
//* It has no  effect on restoration, trashcan empty, or reporting operations. *
//*                                                                            *
//* The following logical areas are affected:                                  *
//* 1) operating on empty directories                                          *
//*    a) '--dir' and '-d'     (allow operation on empty dirs)                 *
//*       cTrash standard: all directories are treated like regular files      *
//*       ca.rm_dir == false, then refuse to move a directory                  *
//*       ca.rm_dir != false, then move an empty directory like a regular file *
//* 2) recursion into non-empty subdirectories                                 *
//*    a) '--recursive' and '-r' and '-R'   (recurse into non-empty dirs)      *
//*       cTrash standard: all directories are treated like regular files      *
//*       ca.rm_rec == false, then refuse to move a directory (see ca.rm_dir)  *
//*       ca.rm_rec != false, then move all directories like regular files     *
//* 3) confirmation prompts                                                    *
//*    a) '-I'   (prompt once)                                                 *
//*       cTrash equivalent: --preconfirm                                      *
//*       ca.rm_int == RMINT_ONCE                                              *
//*    b) '-i'   (prompt for each item)                                        *
//*       cTrash standard: (no equivalent)                                     *
//*       ca.rm_int == RMINT_EACH                                              *
//*    c) '--interactive=WHEN'   (prompt according to WHEN)                    *
//*          WHEN=='never'       (cTrash equivalent: '--no-confirm')           *
//*            ca.rm_int == RMINT_NEVER                                        *
//*          WHEN=='once'        (same as '-I')                                *
//*            ca.rm_int == RMINT_ONCE                                         *
//*          WHEN=='always'      (same as '-i')                                *
//*          WHEN==(no arg)      (same as '-i')                                *
//*            ca.rm_int == RMINT_EACH                                         *
//* 4) reporting (silent or verbose)                                           *
//*    a) '--force', 'f'                                                       *
//*       cTrash equivalent: --no-confirm                                      *
//*       Note that cTrash (unlike 'rm') will never ignore invalid option      *
//*       syntax, nor source files which were not found.                       *
//*       We're no longer just animals pooping in the woods. (sorry, Stallman) *
//*    b) '--verbose' and '-v'                                                 *
//*       cTrash equivalent: '--verbose' and '-v'                              *
//*       Note, however, that the type of verbose output may be different in   *
//*       'rm' compatibility mode.                                             *
//* 5) options which are accepted, but ignored                                 *
//*    These options make no sense for trashcan operations.                    *
//*    a) --no-preserve-root (cTrash always preserves "/")                     *
//*    b) --preserve-root    (cTrash always preserves "/")                     *
//*    c) --one-file-system (cTrash never recurses into a different filesystem)*
//* 6) options which are overridden by the equivalent cTrash options           *
//*    a) --help                                                               *
//*    b) --version                                                            *
//*                                                                            *
//* Return value: An exit status of zero indicates success, and a nonzero      *
//* value indicates failure.                                                   *
//*                                                                            *
//*                                                                            *
//* It is assumed that a user who wants 'rm' compatibility will probably write *
//* an alias into the .bashrc or equivalent startup file.                      *
//*                 alias rm='ctrash --rm-emulate'                             *
//* ${HOME}/.bashrc   == local user                                            *
//* /root/.bashrc     == root user                                             *
//* /etc/bashrc       == all users                                             *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  : ca   : commArgs class object (by reference)                       *
//*                 ca.argList[2] will be the first argument referenced        *
//*                                                                            *
//* Returns: 'true' if all arguments are valid                                 *
//*          'false' if invalid argument(s)                                    *
//******************************************************************************
//* Notes:                                                                     *
//*  'rm' was written by Paul Rubin, David MacKenzie, Richard M. Stallman,     *
//*  and Jim Meyering.                                                         *
//*                                                                            *
//* It's not entirely clear whether full compatibility is possible or          *
//* desirable. Still, if someone wants to alias 'ctrash' to 'rm' we would      *
//* like to be able to say it's possible.                                      *
//*                                                                            *
//* The difference between option compatibility and output compatibility,      *
//* to say nothing of functional compatibility are interesting points.         *
//* 'rm' is a member of the 'coreutils' group, so even though it was written   *
//* without foresight and is hopelessly out-of-date, we must walk softly.      *
//*                                                                            *
//* For emulation mode, we set the 'quiet' flag to prevent display of the      *
//* cTrash title. Startup errors will be still be displayed. 'rm' syntax       *
//* errors will be reported in 'rm' style.                                     *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool cTrash::ctGetRM_Args ( commArgs& ca )
{
   gString gs( ca.argList[2] ) ; // text formatting
   bool status = true ;          // return value (hope for the best)
   ca.rm_mode = true ;           // indicate emulation mode
   ca.quiet = true ;             // emulation method will control the display

   if ( ca.argCount > 2 )
   {
      //* Open a temporary file to hold the filenames to be processed.*
      ofstream ofs( this->listFile.ustr(), ofstream::out | ofstream::trunc ) ;

      short j = ZERO ;           // for multiple arguments in same token
      bool multiarg = false ;

      for ( short i = 2 ; (i < ca.argCount) || (multiarg != false) ; i++ )
      {
         if ( multiarg != false )   // if additional switches in same argument
            --i ;
         else
            j = ZERO ;

         //* If a command-line switch OR continuing previous switch argument *
         if ( ca.argList[i][j] == DASH || multiarg != false )
         {
            multiarg = false ;
            ++j ;

            if ( ca.argList[i][j] == DASH ) // (double dash)
            {  //* Long-form command switches *
               gs = ca.argList[i] ;

               //* Never ask for confirmation *
               if ( (gs.compare( L"--force" )) == ZERO )
               {
                  ca.rm_int = rmINT_NEVER ;
               }

               //* Levels of interaction. Note that the options accepted  *
               //* by 'rm' are more than those documented, AND check only *
               //* first letter: [nothing], 'n', 'o', 'a', 'y',           *
               //* PLUS: 'never' 'no' 'none' 'once' 'always' 'yes'        *
               else if ( (gs.compare( L"--interactive", true, 13 )) == ZERO )
               {
                  short nIndex ;
                  if ( (nIndex = gs.find( L'=' )) > ZERO )
                  {
                     ++nIndex ;
                     gs.shiftChars( -(nIndex) ) ;
                  }
                  else
                     gs.shiftChars( -13 ) ;
                  wchar_t warg = *gs.gstr() ;
                  switch ( warg )
                  {
                     case L'n': case L'N':   ca.rm_int = rmINT_NEVER ;  break ;
                     case L'o': case L'O':   ca.rm_int = rmINT_ONCE ;   break ;
                     case L'a': case L'A':   case NULLCHAR:
                     case L'y': case L'Y':   ca.rm_int = rmINT_EACH ;   break ;
                     default:
                     {
                        gString gsx( "rm: invalid argument '%S' for '--interactive'\n"
                                     "Valid arguments are: 'never', 'once', 'always'\n"
                                     "Try 'rm --help' for more information.",
                                     gs.gstr() ) ;
                        this->textOut ( gsx ) ;
                        this->silent = true ;   // prevent caller from saying anything
                        status = false ;        // return the bad news
                     }
                     break ;
                  }
                  if ( status == false )
                     break ;
               }

               //*  *
               else if ( ((gs.compare( L"--one-file-system" )) == ZERO) ||
                         ((gs.compare( L"--no-preserve-root" )) == ZERO) ||
                         ((gs.compare( L"--preserve-root" )) == ZERO) )
               {  /* Recognized, but ignored */ }

               //*  *
               else if ( (gs.compare( L"--recursive" )) == ZERO )
               {
                  ca.rm_rec = true ;
               }

               //*  *
               else if ( (gs.compare( L"--dir" )) == ZERO )
               {
                  ca.rm_dir = true ;
               }

               //* Produce detailed output to display *
               else if ( (gs.compare( L"--verbose" )) == ZERO )
               {
                  ca.verbose = true ;
               }

               //* Request for application version number *
               else if ( (gs.compare( L"--version" )) == ZERO )
               {
                  ca.helpFlag = ca.verFlag = true ;
                  break ;
               }

               //* Long-form request for Help *
               else if ( (gs.compare( L"--help" )) == ZERO )
               {
                  ca.helpFlag = true ;
                  break ;
               }

               else           // invalid switch
               {  //* Generate an error mesage *
                  gs.compose( "unrecognized option '%s'\n "
                              "Try 'rm --help' for more information.",
                              ca.argList[i] ) ;
                  this->textOut ( gs ) ;
                  this->silent = true ;   // prevent caller from saying anything
                  status = false ;        // return the bad news
                  break ;
               }

               continue ;     // finished with this argument
            }  // (double dash)

            char argLetter = ca.argList[i][j] ;
            switch ( argLetter )
            {
               case 'f':   ca.rm_int = rmINT_NEVER ;     break ;
               case 'i':   ca.rm_int = rmINT_EACH ;      break ;
               case 'I':   ca.rm_int = rmINT_ONCE ;      break ;
               case 'r': case 'R': ca.rm_rec = true ;    break ;
               case 'd':   ca.rm_dir = true ;            break ;
               case 'v':   ca.verbose = true ;           break ;
               default:
                  gs.compose( "unrecognized option -- '%c'\n"
                              "Try 'rm --help' for more information.",
                              &argLetter ) ;
                  this->textOut ( gs ) ;
                  this->silent = true ;   // prevent caller from saying anything
                  status = false ;        // return the bad news
                  break ;
            }
            if ( (status != false) && ca.argList[i][j + 1] != NULLCHAR )
               multiarg = true ;

            //* If request for help, we're done *
            if ( ca.helpFlag != false )
               break ;
         }
         else
         {  //* Interpret argument as a filename. *
            if ( ofs.is_open() )
            {
               ofs << ca.argList[i] << endl ;
               ++ca.itemCount ;
            }
         }
      }     // for(;;)

      //* Verify existence and configuration of default local trashcan.*
      if ( status != false )
      {
         gString rawTrash, realTrash ;
         for ( short i = ZERO ; ca.envList[i] != NULL ; ++i )
         {
            gs = ca.envList[i] ;
            if ( (gs.find( "HOME=", ZERO, true, 5 )) == ZERO )
            {
               rawTrash.compose( defaultTRASHDIR, &gs.gstr()[5] ) ;
               break ;
            }
         }

         if ( (status = this->ctfValidateTrashPath ( rawTrash, realTrash )) != false )
            realTrash.copy( ca.trashDir, gsMAXBYTES ) ; // trashcan configuration verified
         else
         {  //* Generate an error mesage.                                 *
            //* (there is a small possibility of message truncation here) *
            gs.compose( "Trashcan directory not found or improperly configured.\n"
                        "       '%S' ", rawTrash.gstr() ) ;
            ca.errMsg.append( gs.gstr() ) ;
         }
      }
   }

   if ( (status != false) && ca.itemCount == ZERO )   // no source filenames specified
   {
      this->textOut ( L"rm: missing operand\n"
                       "Try 'rm --help' for more information." ) ;
      this->silent = true ;      // prevent caller from saying anything
      status = false ;           // return the bad news
   }
   return status ;

}  //* End ctGetRM_Args() *

//*************************
//*  ctDisplayAppVersion  *
//*************************
//******************************************************************************
//* Display the application's title, version and copyright info.               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We play tricks with our title strings to make this      *
//* message look canonical. If the strings are changed, this message may get   *
//* ugly. That's why Trix are for kids....                                     *
//******************************************************************************

void cTrash::ctDisplayAppVersion ( void )
{
   #if CT_LANG == 0     // English
   const wchar_t* freeSoftware = 
    L"License GPLv3+: GNU GPL version 3 <http://gnu.org/licenses/gpl.html>\n"
     "This is free software: you are free to modify and/or redistribute it\n"
     "under the terms set out in the license.\n"
     "There is NO WARRANTY, to the extent permitted by law.\n" ;
   #elif CT_LANG == 1   // Spanish
   #elif CT_LANG == 2   // Vietnamese
   #else                // Chinese
   #endif               // CT_LANG

   gString gsOut( "\n%S%S%S\n%S", AppTitle1, AppVersion, AppTitle2, AppTitle3 ) ;
   this->textOut ( gsOut ) ;
   this->textOut ( freeSoftware ) ;
   
}  //* End ctDisplayAppVersion() *

//****************************
//* ctDisplayCommandLineHelp *
//****************************
//******************************************************************************
//* Display the brief version of command-line help.                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void cTrash::ctDisplayCommandLineHelp ( void )
{
   gString gsOut ;
   gsOut.compose( L"\n%S%S%S\n%S%S", AppTitle1, AppVersion, AppTitle2, AppTitle3, AppTitle3 ) ;
   gsOut.limitChars( 130 ) ;  // (this makes an assumption about contents of string)
   this->textOut ( gsOut ) ;
   const short HT_COUNT = 28 ;
   #if CT_LANG == 0
   const char* helpText[HT_COUNT] = 
   {
   "Move files and directory trees to the system trashcan, or restore data from\n"
   "trashcan. Also reports current trashcan contents, or empties trashcan.\n"
   "User local trashcan is referenced by default.\n"
   "Operates on: 'regular' files, 'directory' trees, 'symlink' files and 'FIFOs'.\n"
   "             Character/Block devices and Socket files are not processed.\n"
   "Returns    : the number of items processed, or (-1) if pre-processing error.\n"
   "\n",
   "Usage  : ctrash [OPTIONS] [FILENAMES]\n"
   "                Arguments may be specified in any order.\n"
   "         EXAMPLES: ctrash -t activity.log temp.txt\n"
   "                   ctrash myapp *.o\n"
   "                   ctrash -r --alt-target='./timer_OLD.cpp'\n"
   "                   ctrash -e --no-confirm\n",
   " -t    : (default, optional) specified file(s) will be moved to the trashcan\n",
   " -r    : Restore the item most recently moved to trashcan.\n"
   "         If multiple items with same deletion timestamp, all will be restored.\n",
   " -R    : Interactively restore item(s) from the trashcan.\n",
   " -e    : Empty trashcan. Delete permanently all trashcan contents.\n"
   "         Deleted items cannot be restored.\n",
   " -E    : Empty trashcan. Interactively delete specific items from the trashcan.\n"
   "         Deleted items cannot be restored.\n",
   " -s    : Statistics: summary report of trashcan contents.\n",
   " -S    : Statistics: detailed report of trashcan contents.\n",
   " -p    : Confirm, short form of '--preconfirm' option. (see below)\n",
   " -v    : Verbose, short form of '--verbose' option. (see below)\n",
   " -q    : Quiet, short form of '--quiet' option. (see below)\n",
   " -d    : Delete, short form of '--delete' option. (see below).\n",
   " -h    : Help, short form of '--help' option. (see below).\n",
//xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx|
   " --verbose   : Report detailed information on all operations performed.\n"
   "               (incompatible with '--quiet' and '--silent' options)\n",
   " --quiet     : Produce no output to display UNLESS there are errors.\n"
   "               (incompatible with '-s', '-S' and '-E' options)\n",
   " --silent    : Produce no output to display even if there are errors.\n"
   "               (incompatible with '-s', '-S' and '-E' options)\n",
   " --preconfirm: Confirm before moving specified item(s) to trashcan ('-t' option),\n"
   "               or before restoring item recently moved to trashcan ('-r' option).\n"
   "               (default for these operations is no confirmation)\n",
   " --no-confirm: Do not ask for confirmations before critical actions. Default is\n"
   "               to confirm actions that cannot be undone (deletion, overwrite).\n",
//xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx|
   " --remove    : Remove items from the Trashcan based on when they were placed in\n"
   "               the trash. Deleted items cannot be restored.\n"
   "               Arguments: --remove=[DAYS | HOURS | today]\n"
   "               DAYS: Delete items that have been in the Trashcan for AT LEAST\n"
   "                     the specified number of days.\n"
   "                     Range: 1 through 366 (1 day through 1 year, inclusive)\n"
   "               HOURS:Delete items that have been in the Trashcan NO MORE THAN\n"
   "                     the specified number of hours.\n"
   "                     Range: -1 through -24 (1 through 24 hours, inclusive)\n"
   "               today:Delete items that have been placed in the Trashcan since\n"
   "                     00:00:00 system local time (inclusive).\n"
   "               Example: ctrash --remove=30     (delete items in trash >= 30 days)\n",
   " --sort      : Specify the sort option for display of detailed statistics report\n"
   "               and for interactive operations.\n"
   "                 deletion date: d=ascending (default), D=descending\n"
   "                 item size    : s=ascending, S=descending\n"
   "                 item name    : n=ascending, N=descending\n"
   "               Example: ctrash -S --sort=D\n",
   " --file-list : Specify a plain-text (UTF-8) file containing a list of files\n"
   "               to be moved to/from trashcan (or to be deleted)\n",
   " --alt-target: Specify an alternate target for restoring data from trashcan\n"
   "               (By default, files are restored to their original position.)\n",
   " --trash-path: Specify an alternate path for the base trashcan directory.\n"
   "               Example: ctrash --trash-path=/home/sam/.local/share/Trash\n",
   " --delete    : Delete permanently the specified file(s).\n"
   "               (bypass the trashcan and 'unlink' the file(s))\n"
   "               Deleted files cannot be restored.\n",
   " --rm-emulate: 'rm' utility compatibility mode. If used, this option must be\n"
   "               the FIRST argument. All following options and filenames will be\n"
   "               interpreted as 'rm' would interpret them.\n"
   "               (See documentation for additional information.)\n",
   " --stat-cols : Specify the maximum number of columns per display line.\n"
   "               (default:132 columns, min:80, max:512)\n",
   " --version   : Display version number and copyright notice. (overrides all)\n"
   " --help, -h  : Command-line Help (overrides everything except '--version')\n" 
   "        For detailed information, see Texinfo documentation:\n"
   "              info -f ctrash.info -n 'Invoking'\n"
//xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx|
   } ;
   #elif CT_LANG == 1   // Spanish
   #elif CT_LANG == 2   // Vietnamese
   #else                // Chinese
   #endif               // CT_LANG

   for ( short i = ZERO ; i < HT_COUNT ; ++i )
   {
      gsOut = helpText[i] ;
      this->textOut ( gsOut, false ) ;
   }
   gsOut.clear() ;
   this->textOut ( gsOut ) ;

}  //* End ctDisplayCommandLineHelp() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

