//******************************************************************************
//* File       : crcPlus.cpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2017      Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located below.                   *
//* Date       : 28-Feb-2017                                                   *
//* Version    : (see AppVersion)                                              *
//*                                                                            *
//* Description:                                                               *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.00.01 09-Feb-2017                                                     *
//*   - First effort: The CRC generator is written as a C++ class definition   *
//*     called CRC_Gen. crcPlus is simply a development wrapper and            *
//*     demonstration tool for the CRC_Gen class.                              *
//*   - This application is a natural extension of our 'crcmodel' experiment,  *
//*     written in C. Its function was to develop an algorithm to generate     *
//*     CRC values for our Taggit (audio tag data editor) application, and     *
//*     while that effort was successful, the code was an embarrassing,        *
//*     crap-infested hack which we couldn't allow into a commercial-quality   *
//*     application.                                                           *
//*   - 'crcmodel' was taken directly from a reference model CRC generator     *
//*     published by Ross Williams as:                                         *
//*        "A Painless Guide To CRC Error Detection Algorithms" (1993)         *
//*            <http://www.ross.net/crc/download/crc_v3.txt>                   *
//*     Ross's article was released to the public domain, while our            *
//*     implementation is released under the GNU GPL (see above).              *
//*   - The entire crcPlus application and the CRC_Gen class, _including_ the  *
//*     user documentation were written in less than three weeks, so minor     *
//*     inconsistencies are likely to surface; however, to the best of our     *
//*     ability, both the direct-calculation algorithm and the table-driven    *
//*     algorithm are solid and mirror each other exactly for all test data    *
//*     streams and parameter combinations.                                    *
//*     -- Still, we would not be surprised if one or two bugs sneaked by :-)  *
//*        Please post us a note if insects begin crawling out of the code.    *
//*        Thank you !!                                                        *
//******************************************************************************
//* Programmer's Notes:                                                        *
//* ===================                                                        *
//* -- The I/O stream formatting manipulators are an insult to everyone who    *
//*    uses C++. In our commercial applications, we use the gString class or   *
//*    the GLib++ ustring class for output formatting. However, to minimize    *
//*    complexity in this demonstration app, we use the primitive, but         *
//*    functional 'snprintf' function for text formatting.                     *
//*                                                                            *
//* Operational Modes:                                                         *
//* ------------------                                                         *
//* -- 'file' operation:                                                       *
//*    Scan the specified file and create a CRC based on the configuration     *
//*    parameters (see below).                                                 *
//*    -- The name of the source-data file is required.                        *
//*    -- Two additional arguments may be optionally specified:                *
//*       -- The expected CRC value, if known. Default: 00000000h              *
//*       -- Generate the CRC value using a dynamically-created lookup table   *
//*          instead of using the default direct reference-model calculation.  *
//*                                                                            *
//* -- 'table' operation:                                                      *
//*    Create a lookup table (aka hash table) for efficient run-time CRC       *
//*    generation on a large scale. The table is output to a file, formatted   *
//*    as a C/C++ integer array.                                               *
//*    -- Three(3) integer widths are used:                                    *
//*       1) unsigned char table[256] : for 8-bit accumulator register.        *
//*          8-bit CRC is nearly useless since there is a 1-in-256 chance of   *
//*          undetected data corruption.                                       *
//*       2) unsigned short table[256] : for accumulator register widths       *
//*          between 9 and 16 bits, inclusive.                                 *
//*       3) unsigned int table[256] : for accumulator register widths         *
//*          between 17 and 32 bits, inclusive.                                *
//*                                                                            *
//* -- 'test' operation:                                                       *
//*    Perform an internal validation of both the direct-calculation algorithm *
//*    and the table-driven algorithm. This test uses a small, known data set  *
//*    and fixed configuration parameters. The test is run for both 16-bit     *
//*    and 32-bit CRC.                                                         *
//*                                                                            *
//*                                                                            *
//* Configuration Parameters:                                                  *
//* -------------------------                                                  *
//* -- All register widths ('regwidth' parameter) between 8 bits and 32 bits   *
//*    (inclusive) are allowed. However, all but the most esoteric hardware is *
//*    implemented with registers that are a multiple of 8 bits, and we cannot *
//*    think of a reason why anyone would use fewer bits than the actual width *
//*    of the register.                                                        *
//*    -- For instance, a 12-bit register may be specified, but it is likely   *
//*       that the actual hardware would use a 16-bit register for the         *
//*       calculations, leaving the top 4 bits of the register unused.         *
//*    -- Still, it was fun to implement the algorithm with an arbitrary       *
//*       register width.                                                      *
//*                                                                            *
//* -- All polynomial values ('poly' parameter) are allowed _provided that_    *
//*    the number of significant bits in the poly is less than or equal to the *
//*    width of the register.                                                  *
//*    -- Note that for safety, any excess bits in the specified value will    *
//*       be silently masked out to prevent register overflow.                 *
//*                                                                            *
//* -- The initial register value ('reginit' parameter) is limited to:         *
//*       a) all bits set   - Example: FFFFFFFFh, or                           *
//*       b) all bits clear - Example: 00000000h                               *
//*    While this may seem restrictive, it is necessary in order to keep the   *
//*    reference model and the lookup-table model symmetrical.                 *
//*    -- Note that for safety, any significant bits beyond the width of the   *
//*       register will be silently masked out to prevent register overflow.   *
//*                                                                            *
//* -- The final XOR value ('xorfinal' parameter) can be any hexadecimal       *
//*    value. While the algorithm smoothly handles any value, unless you have  *
//*    a very good reason to do otherwise, it is recommended that the value    *
//*    be either:                                                              *
//*       a) all bits set   - Example: FFFFFFFFh, or                           *
//*       b) all bits clear - Example: 00000000h                               *
//*    -- Note that for safety, any significant bits beyond the width of the   *
//*       register will be silently masked out to prevent register overflow.   *
//*                                                                            *
//* -- The two reflection flags ('reflectin' and 'reflectout' parameters) can  *
//*    be specified in any combination.                                        *
//*       true  / true                                                         *
//*       true  / false                                                        *
//*       false / true                                                         *
//*       false / false                                                        *
//*    Our algorithm supports any combination of these flags because we were   *
//*    curious to see if it could be done. In the real world, however, these   *
//*    flags are generally either both set, or both reset to simplify the      *
//*    processing algorithm.                                                   *
//*                                                                            *
//* -- The remaining configuration parameters are for special cases and for    *
//*    debugging the algorithm.                                                *
//*    -- 'oggvorbis' parameter: This is a special purpose parameter that sets *
//*       the configuration to generate CRC values for Ogg/Vorbis audio files. *
//*       This parameter was implemented as a convenience because most of our  *
//*       test data are derived from OGG/Vorbis audio files as a source of     *
//*       reliable (and verifiable) CRC byte streams.                          *
//*    -- 'dump' parameter: for debugging: Display all the configuration       *
//*        parameters before executing the operation.                          *
//*    -- 'pair' parameter: for debugging: For a given input file, generate    *
//*        two CRC values. For the first, use the direct reference model       *
//*        algorithm; and for the second, use the lookup-table model. Finally, *
//*        compare the generated CRC values and report whether they match.     *
//*        (If they don't match, it means that we have screwed up. :-(         *
//*    -- 'verbose' parameter: report extended diagnostics for configuration   *
//*        parameters which were out-of-range and which were silently truncated*
//*        to fit the register width.                                          *
//*                                                                            *
//* Other Options:                                                             *
//* -- Help: Display a list of command-line options.                           *
//*                                                                            *
//* -- Version: Display the application version number and copyright           *
//*    information, as well as the CRC-Gen class version.                      *
//*                                                                            *
//* The following is a table of ANSI color-code sequences for beautification   *
//* of the output.                                                             *
//*                                                                            *
//* ANSI Colors (partial list):                                                *
//* =========== Foreground  Background   ============ Foreground  Background   *
//* Black       \033[0;30m  \033[0;40m   Dark Gray    \033[1;30m  \033[1;40m   *
//* Red         \033[0;31m  \033[0;41m   Bold Red     \033[1;31m  \033[1;41m   *
//* Green       \033[0;32m  \033[0;42m   Bold Green   \033[1;32m  \033[1;42m   *
//* Yellow      \033[0;33m  \033[0;43m   Bold Yellow  \033[1;33m  \033[1;43m   *
//* Blue        \033[0;34m  \033[0;44m   Bold Blue    \033[1;34m  \033[1;44m   *
//* Purple      \033[0;35m  \033[0;45m   Bold Purple  \033[1;35m  \033[1;45m   *
//* Cyan        \033[0;36m  \033[0;46m   Bold Cyan    \033[1;36m  \033[1;46m   *
//* Light Gray  \033[0;37m  \033[0;47m   White        \033[1;37m  \033[1;47m   *
//******************************************************************************
//* To Do List:                                                                *
//* -- Possible?: Rework the GetCommlineArgs() method to make it a single      *
//*    loop. Then the mode can be at any parameter position. We can let the    *
//*    documentation stand to remind user that a mode is required.             *
//*                                                                            *
//* -- Documentation:                                                          *
//*    -- Add index markers as needed                                          *
//*    -- Add a section for CRC_Gen public methods.                            *
//*                                                                            *
//* --                                                                         *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

#include "crcPlus.hpp"     //* Application-class definition

//**************
//* Local Data *
//**************
static const char* AppVersion = "0.1.00" ;

static const char* parmError = "\n\033[0;45m\033[1;33m Warning! - Parameter out-of-range. \033[1;47m\033[0;30m" ;


//******************************************************************************
//* main:                                                                      *
//* Application entry point.                                                   *
//*                                                                            *
//* Input  : argc  : number of command-line arguments                          *
//*          argv  : an array of strings, one for each arguments               *
//*          argenv: a string containing the inherited environment             *
//*                                                                            *
//* Returns: For options that generate a checksum, the 32-bit (hex) checksum   *
//*          will be returned, otherwise zero.                                 *
//******************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   CrcPlus* cpl = new CrcPlus( argc, argv, argenv ) ;

   int exitCode = cpl->exitCode () ;

   delete cpl ;   // release the application-class object

   exit ( exitCode ) ;
}

//******************************************************************************
//* CrcPlus destructor.                                                        *
//*                                                                            *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

CrcPlus::~CrcPlus ( void )
{
   
}  //* End ~CrcPlus() *

//******************************************************************************
//* CrcPlus constructor.                                                       *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

CrcPlus::CrcPlus ( int argc, char** argv, char** argenv )
{
   //* Initialize the data members *
   //* CRC setup is the default for 32-bit CRC PkZip, Ethernet, etc. *
   this->crcp.regwidth = 32 ;             // 32-bit register
   this->crcp.poly       = crcDFLT_POLY ; // "polynomial"
   this->crcp.reginit    = 0xFFFFFFFF ;   // initial register value
   this->crcp.xorfinal   = 0xFFFFFFFF ;   // XOR value applied to final CRC value
   this->crcp.reg        = ZERO ;         // (accumulator, initial value ignored)
   this->crcp.reflectin  = true ;         // reflect (reverse) the input
   this->crcp.reflectout = true ;         // reflect (reverse) the final output value
   this->expected        = ZERO ;         // expected CRC value
   this->debugOption     = ZERO ;         // no debugging option specified
   this->verbose         = false ;        // do not report parameter errors
   *this->fName          = '\0' ;         // filename not yet specified
//   strncpy ( this->fName, Dflt_Table_File, DTF_BYTES ) ; // default output filename
   this->opStatus = ZERO ;              // operational status unknown

   //* See what the user has to say... *
   OpMode option = this->GetCommlineArgs ( argc, argv ) ;

   //* Say hello *
   if ( ((option != opVersion) && (option != opExit)) ||
        (option == opExit && this->debugOption == 1) )
      this->DisplayTitle () ;

   //* For debugging only: 
   //* 1) Display the setup parameters.               *
   //*   (Note that algorithm verification ignores    *
   //*    the setup parameters.)                      *
   //* 2) Generate a CRC for specified file,          *
   //*    a) first using the reference method,        *
   //*    b) and then using the table-lookup method.  *
   if ( (this->debugOption > ZERO) && ((option == opFile) || (option == opFileT)
       || (option == opExit && this->debugOption == 1)) )
   {
      this->DumpParms () ;
      if ( this->debugOption == 2 )
      {
         unsigned int crc1 = this->FileCRC (),
                      crc2 = this->FileCRC_T () ;
         wcout << L"            \033[0;44m\033[1;33m" 
               << (crc1 == crc2 ? " Match " : " Mismatch ") 
               << L"\033[1;47m\033[0;30m" << endl ;
         option = opExit ;    // we're done
      }
   }

   //******************************
   //* Execute the user's command *
   //******************************
   if ( option == opVerify )                 // verify the CRC algorithm
   {
      this->opStatus = VerifyAlgorithm () ;
   }
   else if ( option == opTable )             // generate a lookup table
   {
      this->GenTable () ;
   }
   else if ( option == opFile )              // calculate a CRC for specified file
   {                                         // using the reference algorithm
      this->opStatus = this->FileCRC () ;
   }
   else if ( option == opFileT )             // calculate a CRC for specified file
   {                                         // using a dynamically-created lookup table
      this->opStatus = this->FileCRC_T () ;
   }
   else if ( option == opVersion )           // report the application version
   {
      this->DisplayVersion () ;
   }
   else if ( option == opHelp )              // display command-line help
   {
      this->DisplayHelp () ;
   }
   else  // (opExit)                         // user error, exit with diagnostic message
   {
      this->opStatus = -1 ;
   }

}  //* End CrcPlus() *

//******************************************************************************
//* GetCommlineArgs:                                                           *
//* Parse the user's command-line options.                                     *
//*                                                                            *
//* Input  : argc   : number of command-line tokens                            *
//*          argv   : pointer to an array of command-line tokens               *
//*                                                                            *
//* Returns: 'false' if all parameters verified                                *
//*          'true'  if user needs help                                        *
//******************************************************************************

OpMode CrcPlus::GetCommlineArgs ( int argc, char** argv )
{
   #define DEBUG_OPTIONS (0)           // For debugging only

   const char* unrecMsg = 
      "\nUnrecognized option: '%s'\n" ;
   const char* noopMsg = 
      "\nAn operational mode must be specified.\n" ;
   const char* rangeMsg =
      "\nParameter out-of-range: '%s'\n" ; 
   const char* syntaxMsg =
      "\nSyntax Error: '%s'\n" ;
   const char* hexMsg = 
      "\nParameter requires a hexadecimal argument: '%s\n" ;
   const char* helpMsg =
      "  Please type: '\033[0;34mcrcplus --help\033[0;30m' for a list of options.\n" ;

   char outBuff[obLEN] ;               // error message formatting
   unsigned int userNum ;              // user numeric input
   OpMode status = opExit ;            // Return value

   //* Test for a short-form help request (note the early return) *
   if ( (argc > 1) && ((argv[1][0] == '?') ||
        ((argv[1][0] == '-') && ((argv[1][1] == 'h') 
                              || (argv[1][1] == 'H') || (argv[1][1] == '?')))) )
   {
      status = opHelp ;
      return status ;
   }

   //* If at least one argument AND it is a long-form switch *
   if ( (argc > 1) && (argv[1][0] == '-') && (argv[1][1] == '-') )
   {
      //********************
      //* Operational Mode *
      //********************
      short tindex = 2 ;      // index for parsing token

      //* Verify that the model is working correctly *
      if ( (strncasecmp ( &argv[1][tindex], "test", 4 )) == ZERO )
      {
         status = opVerify ;
      }

      //* Generate a CRC lookup table *
      else if ( (strncasecmp ( &argv[1][tindex], "table", 5 )) == ZERO )
      {
         status = opTable ;
         tindex += 5 ;

         //* If alternate output file specified *
         if ( (argv[1][tindex] == '=') && (argv[1][tindex + 1] != '\0') )
            strncpy ( this->fName, &argv[1][++tindex], DTF_BYTES ) ;
         else        // default output file
            strncpy ( this->fName, Dflt_Table_File, DTF_BYTES ) ;

         //* CRC setup parameter are at their default values. but *
         //* may be modified by subsequent optional parameters.   *
      }

      //* Generate a CRC checksum for the specified file *
      else if ( (strncasecmp ( &argv[1][tindex], "file", 4 )) == ZERO )
      {
         status = opFile ;
         tindex += 4 ;

         if ( (argv[1][tindex] == '=') && (argv[1][tindex + 1] != '\0') )
         {
            //* Initialize the temp data *
            userNum = 0x00000000 ;
            *this->fName = '\0' ;

            //* Separate the filename from the optional 'expected checksum' *
            short indx = ++tindex ;
            for ( ; argv[1][indx] != '\0' ; ++indx )
            {
               if ( argv[1][indx] == '/' )
               {
                  //* Isolate the filename *
                  argv[1][indx++] = '\0' ;

                  if ( (sscanf ( &argv[1][indx], "%X", &userNum )) == 1 )
                  {
                     this->expected = userNum ;    // expected checksum

                     //* Continue scanning for optional table-lookup flag *
                     for ( ; argv[1][indx] != '\0' ; ++indx )
                     {
                        if ( (argv[1][indx] == '/') && 
                             ((argv[1][indx + 1] == 't') || (argv[1][indx + 1] == 'T')) )
                        {
                           status = opFileT ;
                           break ;
                        }
                     }
                  }
                  break ;
               }
            }
            //* Save the filename *
            strncpy ( this->fName, &argv[1][tindex], DTF_BYTES ) ;
         }
         else
         {
            wcout << "\nThe '--file' option requires a valid filename as an argument.\n"
                  << helpMsg << endl ;
            status = opExit ;
         }
      }

      //* Allow user to dump the parameters without performing an operation *
      else if ( (strncasecmp ( &argv[1][tindex], "dump", 4 )) == ZERO )
      {
         this->debugOption = 1 ;
         status = opExit ;
      }

      //* Requesting application version *
      else if ( (strncasecmp ( &argv[1][tindex], "version", 7 )) == ZERO )
      {
         status = opVersion ;
      }

      //* A cry for help *
      else if ( (strncasecmp ( &argv[1][tindex], "help", 4 )) == ZERO )
      {
         status = opHelp ;
      }
      else                    // unrecognized option
      {
         #if DEBUG_OPTIONS == 0     // Production
         snprintf ( outBuff, obLEN, unrecMsg, argv[1] ) ;
         wcout << outBuff << helpMsg << endl ;
         #else    // DEBUG_OPTIONS
         wcout << L"   argv[1]: '" << argv[1]
               << L"'\n            Unrecognized option!\n"
               << L"Press Enter..." << endl ;
         getchar();
         #endif   // DEBUG_OPTIONS
      }


      //*******************************
      //* Scan for optional arguments *
      //*******************************
      if ( (status != opHelp) && (status != opVersion) && 
           (status != opExit) && (argc > 2) )
      {
         #if DEBUG_OPTIONS != 0
         snprintf ( outBuff, obLEN, "Optional arguments: argc:%d", argc ) ;
         wcout << outBuff << endl ;
         #endif   // DEBUG_OPTIONS

         for ( int avIndex = 2 ; avIndex < argc ; ++avIndex )
         {
            #if DEBUG_OPTIONS != 0
            snprintf ( outBuff, obLEN, "   argv[%d]: '%s'", avIndex, argv[avIndex] ) ;
            wcout << outBuff << endl ;
            #endif   // DEBUG_OPTIONS

            if ( (argv[avIndex][0] == '-') && (argv[avIndex][1] == '-') )
            {
               tindex = 2 ;
               if ( (strncasecmp ( &argv[avIndex][tindex], "poly", 4 )) == ZERO )
               {
                  tindex += 4 ;
                  if ( argv[avIndex][tindex++] == '=' )
                  {
                     if ( (sscanf( &argv[avIndex][tindex], "%X", &userNum )) == 1 )
                        this->crcp.poly = userNum ;
                     else           // syntax error
                     {
                        snprintf ( outBuff, obLEN, hexMsg, argv[avIndex] ) ;
                        wcout << outBuff << helpMsg << endl ;
                        status = opExit ;
                     }
                  }
                  else           // syntax error
                  {
                     snprintf ( outBuff, obLEN, hexMsg, argv[avIndex] ) ;
                     wcout << outBuff << helpMsg << endl ;
                     status = opExit ;
                  }

                  #if DEBUG_OPTIONS != 0
                  if ( status != opExit )
                  {
                     snprintf ( outBuff, obLEN, "         poly: %08Xh", this->crcp.poly ) ;
                     wcout << outBuff << endl ;
                  }
                  else
                     wcout << "         poly: ERROR!" << endl ;
                  #endif   // DEBUG_OPTIONS
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "reflectin", 9 )) == ZERO )
               {
                  tindex += 9 ;
                  if ( argv[avIndex][tindex++] == '=' )
                  {
                     if ( (argv[avIndex][tindex] == 'f') || (argv[avIndex][tindex] == 'F') )
                        this->crcp.reflectin = false ;
                     else if ( (argv[avIndex][tindex] == 't') || (argv[avIndex][tindex] == 'T') )
                        this->crcp.reflectin = true ;
                     else           // syntax error
                     {
                        snprintf ( outBuff, obLEN, syntaxMsg, argv[avIndex] ) ;
                        wcout << outBuff << helpMsg << endl ;
                        status = opExit ;
                     }
                  }
                  else           // syntax error
                  {
                     snprintf ( outBuff, obLEN, syntaxMsg, argv[avIndex] ) ;
                     wcout << outBuff << helpMsg << endl ;
                     status = opExit ;
                  }

                  #if DEBUG_OPTIONS != 0
                  if ( status != opExit )
                  {
                     snprintf ( outBuff, obLEN, "      reflectin: %s", this->crcp.reflectin ? "true" : "false" ) ;
                     wcout << outBuff << endl ;
                  }
                  else
                     wcout << "      reflectin: ERROR!" << endl ;
                  #endif   // DEBUG_OPTIONS
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "reflectout", 10 )) == ZERO )
               {
                  tindex += 10 ;
                  if ( argv[avIndex][tindex++] == '=' )
                  {
                     if ( (argv[avIndex][tindex] == 'f') || (argv[avIndex][tindex] == 'F') )
                        this->crcp.reflectout = false ;
                     else if ( (argv[avIndex][tindex] == 't') || (argv[avIndex][tindex] == 'T') )
                        this->crcp.reflectout = true ;
                     else           // syntax error
                     {
                        snprintf ( outBuff, obLEN, syntaxMsg, argv[avIndex] ) ;
                        wcout << outBuff << helpMsg << endl ;
                        status = opExit ;
                     }
                  }
                  else           // syntax error
                  {
                     snprintf ( outBuff, obLEN, syntaxMsg, argv[avIndex] ) ;
                     wcout << outBuff << helpMsg << endl ;
                     status = opExit ;
                  }

                  #if DEBUG_OPTIONS != 0
                  if ( status != opExit )
                  {
                     snprintf ( outBuff, obLEN, "    reflectout: %s", this->crcp.reflectout ? "true" : "false" ) ;
                     wcout << outBuff << endl ;
                  }
                  else
                     wcout << "    reflectout: ERROR!" << endl ;
                  #endif   // DEBUG_OPTIONS
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "regwidth", 8 )) == ZERO )
               {
                  tindex += 8 ;
                  if ( (argv[avIndex][tindex++] == '=') && (argv[avIndex][tindex] != '\0') )
                  {
                     if ( ((sscanf( &argv[avIndex][tindex], "%u", &userNum )) == 1)
                         && (userNum >= crcMIN_REGWIDTH && userNum <= crcMAX_REGWIDTH) )
//                         && (userNum == crcMID_REGWIDTH || userNum == crcMAX_REGWIDTH) )
                     {
                        this->crcp.regwidth = userNum ;
                     }
                     else        // out-of-range
                     {
                        snprintf ( outBuff, obLEN, rangeMsg, argv[avIndex] ) ; 
                        wcout << outBuff << helpMsg << endl ;
                        status = opExit ;
                     }
                  }
                  else           // syntax error
                  {
                     snprintf ( outBuff, obLEN, syntaxMsg, argv[avIndex] ) ;
                     wcout << outBuff << helpMsg << endl ;
                     status = opExit ;
                  }

                  #if DEBUG_OPTIONS != 0
                  if ( status != opExit )
                  {
                     snprintf ( outBuff, obLEN, "      regwidth: %d", this->crcp.regwidth ) ;
                     wcout << outBuff << endl ;
                  }
                  else
                     wcout << "      regwidth: ERROR!" << endl ;
                  #endif   // DEBUG_OPTIONS
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "reginit", 7 )) == ZERO )
               {
                  tindex += 7 ;
                  if ( (argv[avIndex][tindex++] == '=') && (argv[avIndex][tindex] != '\0') )
                  {
                     if ( (sscanf( &argv[avIndex][tindex], "%X", &userNum )) == 1 )
                        this->crcp.reginit = userNum ;
                     else           // syntax error
                     {
                        snprintf ( outBuff, obLEN, hexMsg, argv[avIndex] ) ;
                        wcout << outBuff << helpMsg << endl ;
                        status = opExit ;
                     }
                  }
                  else           // syntax error
                  {
                     snprintf ( outBuff, obLEN, syntaxMsg, argv[avIndex] ) ;
                     wcout << outBuff << helpMsg << endl ;
                     status = opExit ;
                  }

                  #if DEBUG_OPTIONS != 0
                  if ( status != opExit )
                  {
                     snprintf ( outBuff, obLEN, "      reginit: %08Xh", this->crcp.reginit ) ;
                     wcout << outBuff << endl ;
                  }
                  else
                     wcout << "      reginit: ERROR!" << endl ;
                  #endif   // DEBUG_OPTIONS
               }

               else if ( (strncasecmp ( &argv[avIndex][tindex], "xorfinal", 8 )) == ZERO )
               {
                  tindex += 8 ;
                  if ( (argv[avIndex][tindex++] == '=') && (argv[avIndex][tindex] != '\0') )
                  {
                     if ( (sscanf( &argv[avIndex][tindex], "%X", &userNum )) == 1 )
                        this->crcp.xorfinal = userNum ;
                     else           // syntax error
                     {
                        snprintf ( outBuff, obLEN, hexMsg, argv[avIndex] ) ;
                        wcout << outBuff << helpMsg << endl ;
                        status = opExit ;
                     }
                  }
                  else           // syntax error
                  {
                     snprintf ( outBuff, obLEN, syntaxMsg, argv[avIndex] ) ;
                     wcout << outBuff << helpMsg << endl ;
                     status = opExit ;
                  }

                  #if DEBUG_OPTIONS != 0
                  if ( status != opExit )
                  {
                     snprintf ( outBuff, obLEN, "      xorfinal: %08Xh", this->crcp.xorfinal ) ;
                     wcout << outBuff << endl ;
                  }
                  else
                     wcout << "      xorfinal: ERROR!\n" ;
                  #endif   // DEBUG_OPTIONS
               }

               //* SPECIAL CASE: Setup parameters for OGG/Vorbis CRC checksum *
               else if ( (strncasecmp ( &argv[avIndex][tindex], "oggvorbis", 9 )) == ZERO )
               {
                  //* This setup has been verified to produce the same CRC as the *
                  //* fully-functional Page 2 CRC for two Ogg/Vorbis audio files, *
                  //* 1) "None Of Us Are Free.oga" by Simon Burke                 *
                  //*    Page 2 (comments) extracted as "NoneOfUs_01.bin".        *
                  //*    Checksum generated by VLC Media Player: C274CC0Ah        *
                  //* 2) "Ogg_Sample_1.ogg" from the Xiph.org website.            *
                  //*    Page 2 (comments) extracted as "OggSamp1_01.bin".        *
                  //*    Checksum generated by the libvorbis C library: 632E7789h *
                  this->crcp.regwidth   = crcMAX_REGWIDTH ;
                  this->crcp.poly       = crcDFLT_POLY ;
                  this->crcp.reginit    = 0x00000000 ;
                  this->crcp.xorfinal   = 0x00000000 ;
                  this->crcp.reflectin  = false ;
                  this->crcp.reflectout = false ;
                  if ( this->expected == ZERO ) // insert 'expected CRC' if known
                  {
                     if ( (strncmp ( this->fName, "NoneOfUs_01.bin", 15 )) == ZERO )
                        this->expected = 0xC274CC0A ;
                     else if ( (strncmp ( this->fName, "OggSamp1_01.bin", 15 )) == ZERO )
                        this->expected = 0x632E7789 ;
                     else if ( (strncmp ( this->fName, "numbers.bin", 11 )) == ZERO )
                        this->expected = 0x89A1897F ;
                  }
               }

               //* Requesting application version *
               else if ( (strncasecmp ( &argv[avIndex][tindex], "version", 7 )) == ZERO )
                  status = opVersion ;
         
               //* A cry for help *
               else if ( (strncasecmp ( &argv[avIndex][tindex], "help", 4 )) == ZERO )
                  status = opHelp ;

               //* Undocumented: Dump the parameters *
               else if ( (strncasecmp ( &argv[avIndex][tindex], "dump", 4 )) == ZERO )
                  this->debugOption = 1 ;

               //* Undocumented: Generate a pair of CRC values *
               //* for the specified source file, one via the  *
               //* direct method and one via table lookup.     *
               else if ( (strncasecmp ( &argv[avIndex][tindex], "pair", 4 )) == ZERO )
               { this->debugOption = 2 ; }

               //* Undocumented: Detailed report of parameter errors *
               else if ( (strncasecmp ( &argv[avIndex][tindex], "verbose", 5 )) == ZERO )
               { this->verbose = true ; }

               else        // unrecognized option
               {
                  #if DEBUG_OPTIONS == 0  // Production
                  snprintf ( outBuff, obLEN, unrecMsg, argv[avIndex] ) ;
                  wcout << outBuff << helpMsg << endl ;
                  status = opExit ;
                  #else // DEBUG_OPTIONS
                  wcout << "            Unrecognized option!" << endl ;
                  #endif   // DEBUG_OPTIONS
               }
            }
            else           // syntax error
            {
               //* Test for a short-form help request *
               if ( (argv[avIndex][0] == '?') ||
                    ((argv[avIndex][0] == '-') && ((argv[avIndex][1] == 'h') 
                    || (argv[avIndex][1] == 'H') || (argv[avIndex][1] == '?'))) )
               {
                  status = opHelp ;
               }
               else
               {
                  snprintf ( outBuff, obLEN, syntaxMsg, argv[avIndex] ) ;
                  wcout << outBuff << helpMsg << endl ;
                  status = opExit ;
               }
            }

            //* If no processing will occur, no need to continue *
            if ( (status == opHelp) || (status == opVersion) || (status == opExit) )
               break ;
         }        // for(;;)
         #if DEBUG_OPTIONS != 0
         wcout << L"Press Enter..." << endl ;
         getchar();
         #endif   // DEBUG_OPTIONS
      }           // argc > 2
   }              // argc > 1
   else           // no operational mode specified (or syntax error)
   {
      if ( argc == 1 )
         wcout << noopMsg << helpMsg << endl ;
      else
      {
         snprintf ( outBuff, obLEN, syntaxMsg, argv[1] ) ;
         wcout << outBuff << helpMsg << endl ;
         status = opExit ;
      }
   }

   return status ;

}  //* End GetCommlineArgs()

//******************************************************************************
//* DisplayTitle:                                                              *
//* Clear the terminal window and display the application title.               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void CrcPlus::DisplayTitle ( void )
{
   char outBuff[obLEN] ;

   system ( "clear" ) ;
   snprintf ( outBuff, obLEN, 
       "\033[0;34m"
       "crcplus : v:%s (c)2017 The Software Samurai             \n"
       "\033[0;44m\033[1;37m"
       "------------------------------------------------------------------------------"
       "\033[1;47m\033[0;30m", AppVersion ) ;
   wcout << outBuff << endl ;

}  //* DisplayTitle() *

//******************************************************************************
//* DisplayVersion:                                                            *
//* Display the application version and a brief copyright message.             *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void CrcPlus::DisplayVersion ( void )
{
  const char* Boilerplate =
     "---------------------------------------------------------------------\n"
     "License GPLv3: GNU GPL version 3 <http://gnu.org/licenses/gpl.html> \n"
     "This is free software: you are free to modify and/or redistribute it\n"
     "under the terms set out in the license.\n"
     "There is NO WARRANTY, to the extent permitted by law.\n" ;

   wchar_t outBuff[obLEN] ;
   CRC_Gen gen ;

   swprintf ( outBuff, obLEN, 
              L"\ncrcplus v:%s (c)2017      The Software Samurai (CRC_Gen v:%s)\n",
              AppVersion, gen.version() ) ;
               
   wcout << outBuff << Boilerplate << endl ;

}  //* End DisplayVersion() *

//******************************************************************************
//* VerifyAlgorithm:                                                           *
//* Run a test on the algoritm for both 16-bit and 32-bit CRC.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: the generated CRC checkskum (32-bit)                              *
//******************************************************************************
//* Programmer's Note: The formatting is done in two steps due to a suspected  *
//* internal bug in the snprint() function which chokes when the output data   *
//* are formatted as a single string.                                          *
//******************************************************************************

unsigned int CrcPlus::VerifyAlgorithm ( void )
{
   const char *mMsg = "\033[0;32mMatch!",
              *eMsg = "\033[0;31mCRC Error!" ;
   CRC_Gen gen ;

   unsigned int exp16, exp32, act16_d, act16_t, act32_d, act32_t ;
   unsigned short obSIZE = 4096 ;
   char outBuff[obLEN] ;

   gen.validateAlgorithm ( exp16, exp32, act16_d, act16_t, act32_d, act32_t ) ;

   snprintf ( outBuff, obSIZE, 
      "\n"
      "\033[0;34mVerify the algorithm:  Expected     Actual\033[0;30m\n"
      "      \033[0;34mdirect 16-bit :\033[0;30m     %04Xh      %04Xh  %s\033[0;30m\n"
      "             \033[0;34m32-bit :\033[0;30m %08Xh  %08Xh  %s\033[0;30m\n",
      exp16, act16_d, (exp16 == act16_d ? mMsg : eMsg), 
      exp32, act32_d, (exp32 == act32_d ? mMsg : eMsg) ) ;
   wcout << outBuff ;

   snprintf ( outBuff, obSIZE, 
      "\033[0;34mtable-driven 16-bit :\033[0;30m     %04Xh      %04Xh  %s\033[0;30m\n"
      "             \033[0;34m32-bit :\033[0;30m %08Xh  %08Xh  %s\033[0;30m\n",
      exp16, act16_t, (exp16 == act16_t ? mMsg : eMsg),
      exp32, act32_t, (exp32 == act32_t ? mMsg : eMsg) ) ;
   wcout << outBuff << endl ;

   return act32_d ;

}  //* End VerifyAlgorithm() *

//******************************************************************************
//* FileCRC:                                                                   *
//* Scan the specified file and generate a CRC checksum.                       *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: the generated CRC checkskum                                       *
//******************************************************************************

unsigned int CrcPlus::FileCRC ( void )
{
   unsigned int cksum = ZERO ;   // return value

   wcout << "\n\033[0;34mProcessing:\033[0;30m '" 
         << this->fName << "' (Direct) \033[0;34m::\033[0;30m " ;

   ifstream ifs ( this->fName, ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      const int inChunk = 1024 ;    // bytes processed for each iteration
      unsigned char ibuff[inChunk] ;// input buffer
      int bytesRead,                // bytes read per iteration
          totalBytes = ZERO ;       // total bytes read
      bool final = false ;

      //* Instantiate the CRC generator *
      CRC_Gen gen( this->crcp ) ;
      if ( this->verbose && !(gen.setupSuccessful ()) )
      {
         wcout << parmError << endl ;
         gen.getSetup ( this->crcp ) ;
         this->DumpParms () ;
      }

      //* Read until end-of-file reached *
      do
      {
         ifs.read ( (char*)ibuff, inChunk ) ;
         if ( (bytesRead = ifs.gcount ()) > ZERO )
         {
            totalBytes += bytesRead ;
            if ( (ifs.eof()) || (bytesRead < inChunk) )
               final = true ;
            cksum = gen.generateCRC ( ibuff, bytesRead, final ) ;
         }
      }
      while ( bytesRead > ZERO ) ;

      ifs.close() ;                 // close the source file

      char outBuff[obLEN] ;
      snprintf ( outBuff, obLEN, "\033[0;34mBytes read:\033[0;30m %u\n"
                 "            \033[0;34mExpected:\033[0;30m \033[0;32m%08Xh\033[0;30m  "
                 "\033[0;34mChecksum:\033[0;30m \033[0;32m%08Xh\033[0;30m",
                 totalBytes, this->expected, cksum ) ;
      wcout << outBuff << (this->debugOption == 2 ? ' ' : '\n') << endl ;
   }
   else
      wcout << "\033[0;31mError! Unable to open the file.\033[0;30m\n" << endl ;

   return cksum ;

}  //* End FileCRC() *

//******************************************************************************
//* FileCRC_T:                                                                 *
//* Scan the specified file and generate a CRC checksum.                       *
//* This method produces a CRC identical to the FileCRC() method above, EXCEPT *
//* that it first creates a lookup table instead of doing the calculation      *
//* directly.                                                                  *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: the generated CRC checkskum                                       *
//******************************************************************************
//* Notes:                                                                     *
//* While the reference model algorithm (see FileCRC() above) is simple        *
//* and reliable, it is rather slow. If speed is not an important issue, then  *
//* the reference model code will do the job well.                             *
//*                                                                            *
//* In a commercial-grade product, performance is increased by embedding much  *
//* of the CRC calculation into a lookup table. This table may either be       *
//* generated dynamically at runtime or may be a pre-constructed static table. *
//*                                                                            *
//* The actual performance delta between the reference model and the lookup    *
//* table implementation is unknown; however by looking at the processing loop *
//* inside the CRC generator we see that:                                      *
//*  -- The direct method requires 22 + 2 * 8 == 38 calculations (on average)  *
//*     for each byte processed.                                               *
//*  -- The table lookup method requires only four(4) or five(5) calculations  *
//*     per byte processed.                                                    *
//*                                                                            *
//* Thus, for a non-trivial data set, the performance enhancement is likely    *
//* to be significant.                                                         *
//******************************************************************************

unsigned int CrcPlus::FileCRC_T ( void )
{
   unsigned char  *table08 = NULL ;
   unsigned short *table16 = NULL ;
   unsigned int   *table32 = NULL ;
   unsigned int cksum = ZERO ;   // return value

   wcout << "\n\033[0;34mProcessing:\033[0;30m '" 
         << this->fName << "' (Table) \033[0;34m::\033[0;30m " ;

   ifstream ifs ( this->fName, ifstream::in ) ;
   if ( ifs.is_open() )             // if input file open
   {
      const int inChunk = 1024 ;    // bytes processed for each iteration
      unsigned char ibuff[inChunk] ;// input buffer
      void* lookupTbl ;             // generic pointer to table
      int bytesRead,                // bytes read per iteration
          totalBytes = ZERO ;       // total bytes read
      bool final = false ;

      //* Instantiate the CRC generator *
      CRC_Gen gen( this->crcp ) ;
      if ( this->verbose && !(gen.setupSuccessful ()) )
      {
         wcout << parmError << endl ;
         gen.getSetup ( this->crcp ) ;
         this->DumpParms () ;
      }

      if ( this->crcp.regwidth == crcMIN_REGWIDTH )      // for 8-bit CRC
      {
         //* Generate an 8-bit lookup table *
         table08 = new unsigned char[crcTABLE_ENTRIES] ;
         gen.generateTable ( table08 ) ;
         lookupTbl = (void*)table08 ;
      }

      else if ( this->crcp.regwidth <= crcMID_REGWIDTH ) // for 9-16-bit CRC
      {
         //* Generate a 16-bit lookup table *
         table16 = new unsigned short[crcTABLE_ENTRIES] ;
         gen.generateTable ( table16 ) ;
         lookupTbl = (void*)table16 ;
      }

      else        // for 17-32-bit CRC
      {
         //* Generate a 32-bit lookup table *
         table32 = new unsigned int[crcTABLE_ENTRIES] ;
         gen.generateTable ( table32 ) ;
         lookupTbl = (void*)table32 ;
      }

      //* Read until end-of-file reached *
      do
      {
         ifs.read ( (char*)ibuff, inChunk ) ;
         if ( (bytesRead = ifs.gcount ()) > ZERO )
         {
            totalBytes += bytesRead ;
            if ( (ifs.eof()) || (bytesRead < inChunk) )
               final = true ;
            cksum = gen.generateCRC ( lookupTbl, ibuff, bytesRead, final ) ;
         }
      }
      while ( bytesRead > ZERO ) ;

      ifs.close() ;                 // close the source file

      //* Release the dynamic allocation *
      if ( table08 != NULL )
         delete [] table08 ;
      if ( table16 != NULL )
         delete [] table16 ;
      if ( table32 != NULL )
         delete [] table32 ;

      char outBuff[obLEN] ;
      snprintf ( outBuff, obLEN, 
                 "\033[0;34mBytes read:\033[0;30m %u\n"
                 "            \033[0;34mExpected:\033[0;30m \033[0;32m%08Xh\033[0;30m  "
                 "\033[0;34mChecksum:\033[0;30m \033[0;32m%08Xh\033[0;30m",
                 totalBytes, this->expected, cksum ) ;
      wcout << outBuff << (this->debugOption == 2 ? ' ' : '\n') << endl ;
   }
   else
      wcout << "\033[0;31mError! Unable to open the file.\033[0;30m\n" << endl ;

   return cksum ;

}  //* End FileCRC_T() *

//******************************************************************************
//* GenTable:                                                                  *
//* Generate a lookup table according to the setup parameters in the           *
//* data members:                                                              *
//*                                                                            *
//* fName           : name of output file                                      *
//* crcp.regwidth   : register (and table entry) width in binary bits          *
//*                   Range: 8 bits through 32 bits (inclusive)                *
//*                   Although a table may be generated for any width in this  *
//*                   range, storing the table entries requires:               *
//*                    1 byte for an 8-bit register                            *
//*                    2 bytes for 9-16-bit register                           *
//*                    4 bytes for 17-32-bit register                          *
//* crcp.poly       : "polynomial" value to use for generating the table       *
//*                   if regBits ==  8, only lower  8-bits are used            *
//*                   if regBits == 16, only lower 16-bits are used            *
//*                   if regBits == 32, 32 bits are used                       *
//* crcp.reflectin  : 'false' bits are processed high-to-low (bit 7,6...0)     *
//*                   'true'  bits are processed low-to-high                   *
//*                           bits are swapped (7->0, 6->1, 5->2, etc.)        *
//*                           then the bits are processed (bit 7, 6,...0)      *
//*                                                                            *
//* Input  : none (but see above)                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We have enhanced this method to optionally create       *
//* 8-bit lookup tables, as well as partially-populated 16-bit tables for      *
//* 9,10,11,12,13,14,15-bit registers AND partially-populated 32-bit tables for*
//* 17, 18, 19, 20,21,22,23,24,25,26,27,28,29,30,31-bit registers.             *
//*                                                                            *
//* Unfortunately, we have no way to verify the accuracy of tables for register*
//* widths other than 16-bit and 32-bit. This is not a serious problem         *
//* because it makes no sense to create a CRC that uses fewer bits than the    *
//* register width.                                                            *
//*                                                                            *
//******************************************************************************

void CrcPlus::GenTable ( void )
{
   const char* regMsg = 
      "\n\033[0;31mError! Width parameter must be 8 <= width <= 32 bits. (%d)\033[0;30m\n" ;
   const char* polyMsg = 
      "\n\033[0;31mError! Specified polynomial (%08Xh) is wider than (%d bits)\033[0;30m\n" ;

   //* Format for table entries, and number of entries per line *
   const char* eForm = "" ;
   int   perLine = ZERO ;
   if ( this->crcp.regwidth == crcMIN_REGWIDTH )
   { eForm = " 0x%02X" ; perLine = 12 ; }
   else if ( this->crcp.regwidth <= crcMID_REGWIDTH )
   { eForm = " 0x%04X" ; perLine =  8 ; }
   else
   { eForm = " 0x%08X" ; perLine =  4 ; }

   char outBuff[obLEN] ;         // formatting buffer

   //* Keep the user informed *
   wcout << "\n"
            "\033[0;34mGenerating CRC Algorithm Table\n"
            "\033[0;34m------------------------------\n" ;

   snprintf ( outBuff, obLEN, "\033[0;34mRegister    : \033[0;30m%d bits\n", this->crcp.regwidth ) ;
   wcout << outBuff << "\033[0;34mPolynomial  :\033[0;30m" ;
   snprintf ( outBuff, obLEN, eForm, this->crcp.poly ) ;
   wcout << outBuff << '\n' ;
   snprintf ( outBuff, obLEN, "\033[0;34mReflect     : \033[0;30m%s\n", this->crcp.reflectin ? "true" : "false" ) ;
   wcout << outBuff << "\033[0;34mOutput file : '\033[0;30m" << this->fName << "'" ;

   //* Open the output file and write the header info *
   ofstream ofs( this->fName, ofstream::out | ofstream::trunc ) ;
   if ( (ofs.is_open()) != false )
   {
      bool goodParms = true ;
      if ( (perLine == ZERO) || (*eForm == '\0') ) // if initialization failed
      {
         snprintf ( outBuff, obLEN, regMsg, this->crcp.regwidth ) ;
         wcout << outBuff << endl ;
         goodParms = false ;
      }
      else     // Is poly wider that register?
      {
         unsigned int maxMask = 0xFFFFFFFF ;
         for ( short bits = crcMAX_REGWIDTH ; bits >= crcMIN_REGWIDTH ; --bits )
         {
            if ( this->crcp.poly > maxMask )
            {
               snprintf ( outBuff, obLEN, polyMsg, this->crcp.poly, this->crcp.regwidth ) ;
               wcout << outBuff << endl ;
               ofs << outBuff << endl ;
               goodParms = false ;
               break ;
            }
            if ( bits <= this->crcp.regwidth )
               break ;
            maxMask >>= 1 ;
         }
      }

      //* Generate the table and write it to file *
      if ( goodParms )
      {
         ofs << 
         "//******************************************************************************\n"
         "//* CRC Lookup Table.                                                          *\n"
         "//* Table was generated by the crcPlus utility according to the following      *\n"
         "//* parameters:                                                                *\n"
         "//*   Register width: " ;

         snprintf ( outBuff, obLEN, "%2d bits", this->crcp.regwidth ) ;
         ofs << outBuff << "                                                  *\n"
                           "//*   Poly          : " ;
         if ( this->crcp.regwidth == crcMID_REGWIDTH )
            snprintf ( outBuff, obLEN, "0x%04X                        "
                                     "                           *\n", this->crcp.poly ) ;
         else
            snprintf ( outBuff, obLEN, "0x%08X                    "
                                     "                           *\n", this->crcp.poly ) ;
         ofs << outBuff << "//*   Reflect       : "
             << (this->crcp.reflectin ? "true " : "false") 
             << "                                                    *\n"
         "//*                                                                            *\n"
         "//* For more information on CRC table generation, please see the 'crcPlus'     *\n"
         "//* utility, available for download at <http://www.SoftwareSam.us/>            *\n"
         "//******************************************************************************\n"
         << endl ;

         snprintf ( outBuff, obLEN, "unsigned %s crctable[%d] =\n{\n", 
                    (this->crcp.regwidth == crcMIN_REGWIDTH ? "char" :
                    (this->crcp.regwidth <= crcMID_REGWIDTH ? "short" : "int")), 
                    crcTABLE_ENTRIES ) ;
         ofs << outBuff ;

         if ( this->crcp.regwidth == crcMIN_REGWIDTH )
         {
            unsigned char table8[crcTABLE_ENTRIES] ;
            this->GenRamTable ( table8 ) ;

            for ( short indx = ZERO ; indx < crcTABLE_ENTRIES ; ++indx )
            {
               snprintf ( outBuff, obLEN, eForm, table8[indx] ) ;
               ofs << outBuff ;
               if ( indx < (crcTABLE_ENTRIES - 1) )   // no comma after the last line
                  ofs << ',' ;
               if ( ((indx + 1) % perLine) == ZERO )
                  ofs << '\n' ;
            }
            ofs << '\n' ;
         }
         else if ( this->crcp.regwidth <= crcMID_REGWIDTH )
         {
            unsigned short table16[crcTABLE_ENTRIES] ;
            this->GenRamTable ( table16 ) ;

            for ( short indx = ZERO ; indx < crcTABLE_ENTRIES ; ++indx )
            {
               snprintf ( outBuff, obLEN, eForm, table16[indx] ) ;
               ofs << outBuff ;
               if ( indx < (crcTABLE_ENTRIES - 1) )   // no comma after the last line
                  ofs << ',' ;
               if ( ((indx + 1) % perLine) == ZERO )
                  ofs << '\n' ;
            }
         }
         else     // (this->crcp.regwidth > crcMID_REGWIDTH)
         {
            unsigned int table32[crcTABLE_ENTRIES] ;
            this->GenRamTable ( table32 ) ;

            for ( short indx = ZERO ; indx < crcTABLE_ENTRIES ; ++indx )
            {
               snprintf ( outBuff, obLEN, eForm, table32[indx] ) ;
               ofs << outBuff ;
               if ( indx < (crcTABLE_ENTRIES - 1) )   // no comma after the last line
                  ofs << ',' ;
               if ( ((indx + 1) % perLine) == ZERO )
                  ofs << '\n' ;
            }
         }

         ofs << "} ;\n\n//** END OF TABLE **\n\n" << endl ;
      }

      ofs.close() ;           // close the file

      if ( goodParms )
         wcout << "\033[0;32m   DONE\033[0;30m\n" << endl ;
   }
   else
      wcout << "\033[0;31mError! Unable to open file for writing.\033[0;30m\n" << endl ;

}  //* End GenTable() *

//******************************************************************************
//* GenRamTable:                                                               *
//* This method is the same as the GenTable() method above, except that        *
//* it writes the table to RAM memory rather that to a file.                   *
//*                                                                            *
//* Input  : tblPtr    : pointer to an allocated memory space which is large   *
//*                      enough to hold the table:                             *
//*                      256 x  8 bytes for an 8-bit table,                    *
//*                      256 x 16 bytes for a 16-bit table,                    *
//*                      256 x 32 bytes for a 32-bit table,                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void CrcPlus::GenRamTable ( void* tblPtr )
{
   CRC_Gen gen( this->crcp ) ;
   gen.generateTable ( tblPtr ) ;

}  //* End GenRamTable() *

//******************************************************************************
//* exitCode:                                                                  *
//* Return status of last operation. Typically called only from main().        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: generated CRC if available, else ZERO                             *
//******************************************************************************

int CrcPlus::exitCode ( void )
{
   return this->opStatus ;
}

//******************************************************************************
//* DumpParms:                                                                 *
//* For debugging only: Write contents of the parameter structure to stdout.   *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We do not create a specific format for display of       *
//* 8-bit CRC parameters because, frankly, we don't think anyone would ever    *
//* want to use it anyway.                                                     *
//******************************************************************************

void CrcPlus::DumpParms ( void )
{
   char outBuff[256] ;

   if ( this->crcp.regwidth <= crcMID_REGWIDTH ) // format output for 16-bit values
   {
      snprintf ( outBuff, 256, 
         "   regwidth: %d\n"
         "       poly: %04Xh\n"
         "    reginit: %04Xh\n"
         "   xorfinal: %04Xh\n"
         "        reg: %04Xh\n"
         "  reflectin: %s\n"
         " reflectout: %s\n"
         "      fName: '%s'\n"
         "   expected: %08Xh",
         this->crcp.regwidth, this->crcp.poly, 
         this->crcp.reginit, this->crcp.xorfinal, this->crcp.reg, 
         (this->crcp.reflectin ? "true" : "false"),
         (this->crcp.reflectout ? "true" : "false"),
         this->fName, this->expected ) ;
   }
   else                             // format output for 32-bit values
   {
      snprintf ( outBuff, 256, 
         "   regwidth: %d\n"
         "       poly: %08Xh\n"
         "    reginit: %08Xh\n"
         "   xorfinal: %08Xh\n"
         "        reg: %08Xh\n"
         "  reflectin: %s\n"
         " reflectout: %s\n"
         "      fName: '%s'\n"
         "   expected: %08Xh",
         this->crcp.regwidth, this->crcp.poly, 
         this->crcp.reginit, this->crcp.xorfinal, this->crcp.reg, 
         (this->crcp.reflectin ? "true" : "false"),
         (this->crcp.reflectout ? "true" : "false"),
         this->fName, this->expected ) ;
   }
   wcout << outBuff << endl ;

}  //* End DumpParms() *

//******************************************************************************
//* DisplayHelp:                                                               *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
void CrcPlus::DisplayHelp ( void )
{
   wcout <<
      "\n"
      "Usage: crcmodel --[test | file | table] OPTIONS   \n"
      "                (If arguments contain spaces, enclose in quotes.)\n"
      "                Exit value: For options that generate a checksum, the 32-bit\n"
      "                            (hex) checksum will be returned, otherwise zero.\n"
      "  --test       : Test the algorithm for both 16-bit and 32-bit checksums.\n"
      "                 Test sequence: \"123456789\" (9 ASCII bytes).\n"
      "  --file=FILENAME[/EXPECTED_CHECKSUM[/T]]\n"
      "               : Scan the specified file and create a CRC checksum for it.\n"
      "                 Optionally, specify the expected checksum.\n"
      "                 Optionally, specify that a table-based algorithm be used.\n"
      "  --table[=FILENAME] : Generate a CRC lookup table and write it to a file.\n"
      "              Optionally specify output filename.\n"
      "              Default output file will be \"crctable.out\"\n"
      "              Default setup will be regwidth==32, poly==04C11DB7h\n"
      "              and reflect==true.\n"
      "              The following options may be used to modify the format\n"
      "              of the table: '--regwidth', '--poly', and '--reflect'\n"
      "  Options:\n"
      "\n"
      "  --poly=POLY  : The so-called 'generator polynomial' or hexadecimal XOR value.\n"
      "                 Default: 04C11DB7h  This is the value used in many\n"
      "                 popular algorithms including the one of particular\n"
      "                 interest to us, which is the CRC checksum used by\n"
      "                 OGG/Vorbis audio tag data. Any hexadecimal value will work,\n"
      "                 so long as it is the same width as the register which is\n"
      "                 specified by the '--regwidth' option below.\n"
      "  --reflectin=[true|false] : Direction of bit processing for input bytes.\n"
      "                 'true'  for algorithms which shift bits right (default)\n"
      "                 'false' for algorithms which shift bits left\n"
      "  --reflectout=[true|false] : Reverse the order of bits for the final CRC\n"
      "                  checksum value before returning it.\n"
      "  --regwidth=[16|32] : Number of bits in accumulator register.\n"
      "                  16 == 16-bit calculations\n"
      "                  32 == 32-bit calculations (default)\n"
      "  --reginit=HEXVALUE : Initial register value (in hexadecimal).\n"
      "                  This will be the register value when processing begins.\n"
      "  --xorfinal=HEXVALUE : Post-calculation XOR value (in hexadecimal).\n"
      "                  Perform an XOR operation on the final CRC checksum value\n"
      "                  before returning it.\n"
      "  --oggvorbis  : Special Case: Set up processing parameters for OGG/Vorbis\n"
      "                 metadata (tag data) checksum generation.\n"
      "  --help       : Display a list of command-line options (this list)\n"
      << endl ;
}  //* End DisplayHelp() *

//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

