//******************************************************************************
//* File       : CRC_Gen.hpp                                                   *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2017      Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located below.                   *
//* Date       : 28-Feb-2017                                                   *
//* Version    : (see CRC_Gen_Version[] below)                                 *
//*                                                                            *
//* Description: This header defines the CRC_Gen class which is a generic      *
//*              CRC calculation object for both a direct-calculation model    *
//*              and a hash-table lookup model.                                *
//*              Special enhancments have been added for:                      *
//*               1) Internal algorithm verification tests.                    *
//*               2) OGG/Vorbis I metadata CRC calculation.                    *
//*                                                                            *
//* The algorithm used for the CRC_Gen class is based on a reference model     *
//* written by Ross Williams (1993) which was written in C and is in the       *
//* public domain. Ross's original model can be found at:                      *
//*        "A Painless Guide To CRC Error Detection Algorithms,"               *
//*         http://www.ross.net/crc/download/crc_v3.txt                        *
//*                                                                            *
//*         See also our implementation of Ross's reference model:             *
//*             'crcmodel', available as a separate download.                  *
//*                 Visit:  http://www.SoftwareSam.us/                         *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************
//* Version History (most recent first):                                       *
//*                                                                            *
//* v: 0.01.00 18-Feb-2017                                                     *
//*   - This class was developed under a simple console application,           *
//*     'crcPlus' which provides a user interface and practical examples of    *
//*     using the CRC_Gen class.                                               *
//*   - Our CRC algorithm is based on a reference model published by Ross      *
//*     Williams in 1993. Ross placed his model in the public domain.          *
//*              http://www.ross.net/crc/download/crc_v3.txt                   *
//*   - This is the first effort toward a C++ implementation of our C-language *
//*     'crcmodel' utility.                                                    *
//*     - Unlike the earlier implementation, the Crc_Gen class is intended as  *
//*       a production-quality release.                                        *
//*     - The direct-calculation algorithm is essentially unchanged from the   *
//*       reference model code. Certain minor changes in code layout were      *
//*       necessary for the C++ port, both to create self-documenting code and *
//*       to isolate all CRC calculations within the CRC_Gen class so that it  *
//*       run without modification in almost any C++ environment.              *
//*   - Although the original reference-model code included an option to       *
//*     generate an external hash table, it was very simplistic. This          *
//*     implementation can generate a table for all register widths between    *
//*     8 bits and 32 bits, inclusive.                                         *
//*   - The option to dynamically create a RAM-based hash table and use it to  *
//*     support a table-driven CRC algorithm are new in this release.          *
//*   - Significant error checking and automated regression testing have been  *
//*     performed to ensure that both the direct-calculation algorithm and     *
//*     table-driven algorithm produce identical results.                      *
//*   - The first commercial use of CRC_Gen is in our 'Taggit' application     *
//*     where it is used to produce CRC checksums on tag data included in audio*
//*     files. That implementation specifically targets CRC generation for     *
//*     OGG/Vorbis I metadata, but may be expanded to other targets requiring  *
//*     CRC generation.                                                        *
//*   - Because this is a new project, and because the source code is fully    *
//*     self-documenting, the first release of the external documentation is   *
//*     very simple. It is not intended as a tutorial on CRC generation.       *
//*     Instead, it simply describes how the CRC_Gen algorithm works and how   *
//*     it may be integrated into a software project.                          *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
// No headers are required for this class; HOWEVER, be sure to declare a 
// namespace in your application. Example: using namespace std ;


//* FOR DEBUGGING ONLY: Include the 'iostream' header *
//* so messages can be sent to the console (wcout).   *
#define DEBUG_CRC_Gen (0)
#if DEBUG_CRC_Gen != 0
#include <iostream>     // FOR DEBUGGING ONLY
using namespace std ;   // FOR DEBUGGING ONLY
#endif   // DEBUG_CRC_Gen


const char CRC_Gen_Version[] = { "0.1.00" } ; 

//* Macro for CRC bitshifting *
#define crcBITMASK(X) (1 << (X))

//* Number of entries in a CRC lookup table *
#define crcTABLE_ENTRIES (256)

//* Width of a byte *
#define crcBYTESIZE (8)

//* Minimum, Intermediate and Maximum register widths in bits *
#define crcMIN_REGWIDTH (crcBYTESIZE)
#define crcMID_REGWIDTH (16)
#define crcMAX_REGWIDTH (32)     // (default)

//* Default polynomial. Standard for OGG/Vorbis and others. *
const unsigned int crcDFLT_POLY = 0x04C11DB7 ;

//* Portable CRC setup parameters.Fully describes the *
//* parameters for generating a CRC checksum.         *
class Crc_Parms
{
   public:
   int            regwidth ;     // Width of register in bits [8,32]
   unsigned int   poly ;         // Algorithm's polynomial value
   unsigned int   reginit ;      // Initial CRC register value.
   unsigned int   xorfinal ;     // Final XOR vaule applied to calculated CRC
   unsigned int   reg ;          // CRC register (accumulator)
   bool           reflectin ;    // Reflect (reverse) bits of input bytes
   bool           reflectout ;   // Reflect (reverse) bits of final CRC value
} ;

//*****************************
//* CRC "checksum" generator. *
//*****************************
class CRC_Gen
{
   public:

   //************************************************************************
   //* Destructor. No cleanup is currently required.                        *
   //************************************************************************
   ~CRC_Gen ( void ) { }

   //************************************************************************
   //* Default constructor.                                                 *
   //* Members initialized for OGG/Vorbis CRC.                              *
   //*                                                                      *
   //* Input  : none                                                        *
   //* Returns: implicitly returns an instance of the class                 *
   //************************************************************************
   CRC_Gen ( void )
   {
      this->goodParms = true ;   // hope for the best
      this->oggDefault () ;      // set to default OGG/Vorbis parameters
      this->reset () ;           // initialize remaining data members
   }

   //************************************************************************
   //* Initialization constructor.                                          *
   //* Caller provides all setup parameters.                                *
   //* Use care in selecting parameter values.                              *
   //*    -- Only 'regwidth' is range checked.                              *
   //*    -- 'poly' 'reginit' and 'xorfinal' are silently truncated to the  *
   //*       width of the register.                                         *
   //*                                                                      *
   //* Input  : crcp : (by reference) contains all setup parameters         *
   //*                 .regwidth  : width of accumulator register in bits   *
   //*                              Note that 'registerWidth' must be       *
   //*                              between 8 and 32 (inclusive). If out    *
   //*                              of range, 32 will be the default.       *
   //*                 .reginit   : accumulator register value before       *
   //*                                operation begins                      *
   //*                 .xorfinal  : final calculated CRC will be XOR'd with *
   //*                                this value                            *
   //*                 .poly      : divisor for CRC (bitwise) division      *
   //*                 .reflectin : if 'true', reverse the bits of each     *
   //*                                input byte before calculation         *
   //*                              if 'false' perform calculation on each  *
   //*                                input byte as presented               *
   //*                 .reflectout: if 'true', reverse the bits of the final*
   //*                                CRC return value                      *
   //*                              if 'false', return the final CRC value  *
   //*                                as calculated                         *
   //*                                                                      *
   //* Returns: implicitly returns an instance of the class                 *
   //************************************************************************
   CRC_Gen ( const Crc_Parms& crcp )
   {
      this->goodParms = true ;   // hope for the best
      if ( (crcp.regwidth >= crcMIN_REGWIDTH ) && (crcp.regwidth <= crcMAX_REGWIDTH) )
         this->regwidth = crcp.regwidth ;
      else
      {
         this->regwidth = crcMAX_REGWIDTH ;
         this->goodParms = false ;
      }
      this->reginit    = crcp.reginit ;
      this->xorfinal   = crcp.xorfinal ;
      this->poly       = crcp.poly ;
      this->reflectin  = crcp.reflectin ;
      this->reflectout = crcp.reflectout ;
      this->reset () ;           // initialize remaining data members
   }

   //************************************************************************
   //* reset:                                                               *
   //* Return all setup parameters to their original state in preparation   *
   //* for generating a new CRC.                                            *
   //* a) Note that only the 'reg' member actually changes during           *
   //*    processing. Other parameters need to be set only during           *
   //*    instantiation, but it is convenient to isolate the setup here.    *
   //* b) To prevent register overflow due to application error, we limit   *
   //*    the width of 'poly', 'reginit' and 'xorfinal' to the actual width *
   //*    of the register.                                                  *
   //*                                                                      *
   //* Input  : none                                                        *
   //* Returns: nothing                                                     *
   //************************************************************************
   void reset ( void )
   {
      this->createMasks () ;     // create the bitmasks

      //* Do range checking *
      if ( (this->poly > this->carryMask) || (this->reginit > this->carryMask)
           || (this->xorfinal > this->carryMask) )
         this->goodParms = false ;

      //* Prevent register overflow *
      this->poly     &= this->carryMask ;
      this->reginit  &= this->carryMask ;
      this->xorfinal &= this->carryMask ;

      //* Initialize the accumulator register *
      this->reg = this->reginit ;

      this->resetFlag = true ;   // indicate that we have fresh data
   }

   //************************************************************************
   //* generateCRC:                                                         *
   //* Calculate a CRC checksum for the block of binary bytes using a       *
   //* direct calculation. This method DOES NOT use a lookup table.         *
   //*                                                                      *
   //* Input  : blkPtr : pointer to the input byte array                    *
   //*          blkLen : number of bytes in the array                       *
   //*          final  : Indicates whether the data block to be processed   *
   //*                   is the last block in the processing sequence.      *
   //*                   (see note below)                                   *
   //*                                                                      *
   //* Returns: the calculated CRC                                          *
   //*          if register width == 32 bits, all 32 bits are significant   *
   //*          if register width == 16, then only the 16 least-significant *
   //*             bits are retained.                                       *
   //*  Note that the return value is meaningful ONLY when 'final' is set.  *
   //************************************************************************
   //* Note on the 'final' parameter:                                       *
   //* Often the input stream is a file or other data source of arbitrary   *
   //* size which cannot be loaded into memory as a single block. When this *
   //* happens, the data are read into memory as a sequence of data blocks. *
   //*                                                                      *
   //* Set the 'final' parameter to 'false' for all but the last block to   *
   //* be processed. 'final must be 'true' for the last block processed.    *
   //************************************************************************
   unsigned int generateCRC ( const unsigned char* blkPtr, unsigned int blkLen, bool final )
   {
      unsigned int uch,          // input byte
                   cksum ;       // return value
      this->resetFlag = false ;  // indicate that 'reg' is no longer at initial state

      for ( unsigned int indx = 0 ; indx < blkLen ; ++indx )
      {
         // Reflected or unreflected input *
         if ( this->reflectin )
            uch = this->reflect ( blkPtr[indx], crcBYTESIZE ) ;
         else
            uch = blkPtr[indx] ;

         //* Shift LSByte into MSByte position and XOR with the register value. *
         this->reg ^= (uch << (this->regwidth - crcBYTESIZE)) ;

         for ( short i = 0 ; i < crcBYTESIZE ; ++i )
         {
            //* If MSB of the register is set, then     *
            //* shift out the MSB and XOR with the poly.*
            if ( this->reg & this->topbitMask )
               this->reg = (this->reg << 1) ^ this->poly ;

            //* Otherwise, simply shift register left by one bit.*
            else
               this->reg <<= 1 ;

            //* Apply the width mask to eliminate carry.*
            this->reg &= this->carryMask ;
         }
      }

      if ( final )      // format the final CRC value
      {
         cksum = this->formatCRC () ;
         this->reset () ;     // (in case user forgets)
      }
      else              // return an interim value (see note above)
         cksum = this->reg ;

      return cksum ;
   }

   //************************************************************************
   //* generateCRC:                                                         *
   //* Calculate a CRC checksum for the block of binary bytes using the     *
   //* specified lookup table.                                              *
   //*                                                                      *
   //* Input  : tblPtr : pointer to a table containing 256 entries          *
   //*                   (see note below)                                   *
   //*          blkPtr : pointer to the input byte array                    *
   //*          blkLen : number of bytes in the array                       *
   //*          final  : Indicates whether the data block to be processed   *
   //*                   is the last block in the processing sequence.      *
   //*                   (see note below)                                   *
   //*                                                                      *
   //* Returns: the calculated CRC                                          *
   //*          if register width == 32 bits, all 32 bits are significant   *
   //*          if register width == 16, then only the 16 least-significant *
   //*             bits are retained.                                       *
   //*  Note that the return value is meaningful ONLY when 'final' is set.  *
   //************************************************************************
   //* IMPORTANT NOTE:                                                      *
   //* ===============                                                      *
   //* The 'tblPtr' parameter is a 'void*' and will be cast to the          *
   //* appropriate table-entry width based on the current value of the      *
   //* CRC_Gen.regwidth member.                                             *
   //* It is the caller's responsibility to reference a table with entries  *
   //* of the same width as specified by the 'regwidth' parameter.          *
   //*                                                                      *
   //*  -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   *
   //* Note on reflection: There are four(4) combination of reflection:     *
   //*  1) reflectin == false && reflectout == false                        *
   //*     Neither table generation nor post-processing does reflection.    *
   //*  2) reflectin == false && reflectout == true                         *
   //*     Table is not reflected, ONLY post-processing reflection needed.  *
   //*  3) reflectin == true  && reflectout == false                        *
   //*     Reflection is done during table generation, so final CRC must    *
   //*     be re-reflected to compensate even though it is not specified.   *
   //*     IMPORTANT NOTE: This 'feature' is what most users would call a   *
   //*     'bug' except that this bug produces the correct answer.          *
   //*  4) reflectin == true  && reflectout == true                         *
   //*     Reflection is handled entirely within the table, so no           *
   //*     post-processing reflection is necessary.                         *
   //*                                                                      *
   //* Additional note on reflection:                                       *
   //*  If 'reflectin' is set, then we need to reflect the initial value    *
   //*  of the accumulator variable ('cksum') to match the reflected        *
   //*  values in the lookup table. Note that register reflection happens   *
   //*  only on the first data block in a sequence i.e. the first block     *
   //*  processed after a reset.                                            *
   //*                                                                      *
   //*  -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   *
   //* Note on the 'final' parameter:                                       *
   //* Often the input stream is a file or other data source of arbitrary   *
   //* size which cannot be loaded into memory as a single block. When this *
   //* happens, the data are read into memory as a sequence of data blocks. *
   //*                                                                      *
   //* Set the 'final' parameter to 'false' for all but the last block to   *
   //* be processed. 'final must be 'true' for the last block processed.    *
   //*                                                                      *
   //* Programmer's Note: Unlike the direct calculation version of          *
   //* generateCRC(), the table lookup version actually doesn't need the    *
   //* 'final' parameter at all. We retain it for symmetry AND because we   *
   //* use it to reset the variable data members when the CRC sequence has  *
   //* terminated.                                                          *
   //************************************************************************
   unsigned int generateCRC ( void* tblPtr, const unsigned char* blkPtr, 
                              unsigned int blkLen, bool final )
   {
      //* Byte mask (ensure that index stays within 255) *
      const unsigned char mask = 0xFF ;
      //* Right-shift count for non-reflected calculations *
      const int rtSHIFT = this->regwidth - crcBYTESIZE ;
      //* Return value - also used to load the local accumulator *
      unsigned int crcval = this->reg ;
      //* If this is the first data block in the sequence AND *
      //* if input reflection has been specified, then        *
      //* reflect the initial register value.                 *
      if ( this->reflectin && resetFlag )
      {
         crcval = this->reflect ( crcval, this->regwidth ) ;
         this->resetFlag = false ;
      }

      if ( this->regwidth == crcMIN_REGWIDTH )        // 8-bit entries
      {  // Programmer's Note: For the 8-bit calculations, we actually use a 16-bit
         // accumulator to maintain the instruction sequence accross register sizes.
         // Otherwise the left-shift and right-shift would always result in a zero
         // value in the register.
         unsigned short cksum = (unsigned short)crcval ;    // accumulator register
         unsigned char  *bPtr = (unsigned char*)tblPtr ;    // table pointer

         for ( unsigned int indx = 0 ; indx < blkLen ; ++indx )
         {
            if ( this->reflectin )
               cksum = bPtr[(cksum ^ blkPtr[indx]) & mask] ^ (cksum >> crcBYTESIZE) ;
            else
               cksum = bPtr[((cksum >> rtSHIFT) ^ blkPtr[indx]) & mask] ^ (cksum << crcBYTESIZE) ;
            cksum &= this->carryMask ;   // apply mask to eliminate register overflow
         }
         this->reg = cksum ;
      }

      else if ( this->regwidth <= crcMID_REGWIDTH )   // 9-16 bit entries
      {
         unsigned short cksum = (unsigned short)crcval ;    // accumulator register
         unsigned short *sPtr = (unsigned short*)tblPtr ;   // table pointer

         for ( unsigned int indx = 0 ; indx < blkLen ; ++indx )
         {
            if ( this->reflectin )
               cksum = sPtr[(cksum ^ blkPtr[indx]) & mask] ^ (cksum >> crcBYTESIZE) ;
            else
               cksum = sPtr[((cksum >> rtSHIFT) ^ blkPtr[indx]) & mask] ^ (cksum << crcBYTESIZE) ;
            cksum &= this->carryMask ;   // apply mask to eliminate register overflow
         }
         this->reg = cksum ;
      }

      else                                // 17-32 bit entries
      {
         unsigned int cksum = crcval ;    // accumulator register
         unsigned int *iPtr = (unsigned int*)tblPtr ; // table pointer

         for ( unsigned int indx = 0 ; indx < blkLen ; ++indx )
         {
            if ( this->reflectin )
               cksum = iPtr[(cksum ^ blkPtr[indx]) & mask] ^ (cksum >> crcBYTESIZE) ;
            else
               cksum = iPtr[((cksum >> rtSHIFT) ^ blkPtr[indx]) & mask] ^ (cksum << crcBYTESIZE) ;
            cksum &= this->carryMask ;   // apply mask to eliminate register overflow
         }
         this->reg = cksum ;
      }

      if ( final )      // format the final CRC value
      {
         //* Handle the optional final reflection. (see note above) *
         if ( (this->reflectin && !this->reflectout) ||
              (!this->reflectin && this->reflectout) )
         {
            crcval = this->xorfinal ^ this->reflect ( this->reg, this->regwidth ) ;
         }
         else
         {
            crcval = this->xorfinal ^ this->reg ;
         }
         crcval &= this->carryMask ;   // apply mask to eliminate register overflow

         this->reset () ;     // (in case user forgets)
      }
      else              // return an interim value
         crcval = this->reg ;

      return crcval ;            // interim or final CRC
   }

   //************************************************************************
   //* generateTable:                                                       *
   //* Create a CRC lookup table with 256 entries according to the          *
   //* parameters:                                                          *
   //*  1) regwidth  determines the width of each entry in binary bits      *
   //*               Range: 8 - 32 (inclusive)                              *
   //*  2) poly      determines the CRC divisor for the calculations        *
   //*               Range: All non-zero bits must fit within 'regwidth'    *
   //*  3) reflectin determines whether table entries will be reflected     *
   //*                                                                      *
   //* Input  : tblPtr    : pointer to an allocated memory space which is   *
   //*                      large enough to hold the table:                 *
   //*                      256 x 2 bytes for a 16-bit table,               *
   //*                      256 x 4 bytes for a 32-bit table,               *
   //*                                                                      *
   //* Returns: nothing                                                     *
   //************************************************************************
   void generateTable ( void* tblPtr )
   {
      if ( this->regwidth == crcMIN_REGWIDTH )
      {
         unsigned char*  bPtr = (unsigned char*)tblPtr ;
         for ( int indx = 0 ; indx < crcTABLE_ENTRIES ; ++indx )
         {
            bPtr[indx] = (unsigned char)(this->tableEntry ( indx )) ; // write an entry
//            bPtr[indx] = (unsigned short)(this->tableEntry ( indx )) ; // write an entry
         }
      }
      else if ( this->regwidth <= crcMID_REGWIDTH )
      {
         unsigned short* sPtr = (unsigned short*)tblPtr ;
         for ( int indx = 0 ; indx < crcTABLE_ENTRIES ; ++indx )
         {
            sPtr[indx] = (unsigned short)(this->tableEntry ( indx )) ; // write an entry
         }
      }
      else     // (this->regwidth > crcMID_REGWIDTH)
      {
         unsigned int* iPtr = (unsigned int*)tblPtr ;
         for ( int indx = 0 ; indx < crcTABLE_ENTRIES ; ++indx )
         {
            iPtr[indx] = (unsigned int)(this->tableEntry ( indx )) ; // write an entry
         }
      }
   }

   //************************************************************************
   //* validateAlgorithm:                                                   *
   //* Perform basic validation of the algorithm for both 16-bit and        *
   //* 32-bit CRC values.                                                   *
   //*                                                                      *
   //* Input data : The nine-byte ASCII string "123456789" not including    *
   //*              the null terminator.                                    *
   //* Setup parms: These are the setup data used in Ross Williams' original*
   //*              validation routine (see below).                         *
   //*                                                                      *
   //* Input  : expected16Bit : the CRC that _should be_ generated for the  *
   //*                          16-bit setup parameters                     *
   //*          expected32Bit : the CRC that _should be_ generated for the  *
   //*                          32-bit setup parameters                     *
   //*          actual16Bit_d : the calculated CRC for direct-calculation,  *
   //*                          with 16-bit parameters                      *
   //*          actual16Bit_t : the calculated CRC for table-driven         *
   //*                          calculation with 16-bit parameters          *
   //*          actual32Bit_d : the calculated CRC for direct-calculation,  *
   //*                          with 32-bit parameters                      *
   //*          actual32Bit_t : the calculated CRC for table-driven         *
   //*                          calculation with 32-bit parameters          *
   //*                                                                      *
   //* Returns: 'true' if calculated values match the expected values       *
   //*          'false' if one or both calculated values differs from the   *
   //*                  expected value                                      *
   //************************************************************************
   bool validateAlgorithm ( unsigned int& expected16Bit, unsigned int& expected32Bit, 
                            unsigned int& actual16Bit_d, unsigned int& actual16Bit_t, 
                            unsigned int& actual32Bit_d, unsigned int& actual32Bit_t )
   {
      const char* testData = "123456789" ;
      const int testBytes = 9 ;
      const unsigned int exp16 = 0x0000BB3D,
                         exp32 = 0xCBF43926 ;
      bool match = false ;    // return value

      expected16Bit = exp16 ;
      expected32Bit = exp32 ;

      //* 16-bit setup parameters *
      this->regwidth   = 16 ;
      this->reginit    = 0x0000 ;
      this->xorfinal   = 0x0000 ;
      this->poly       = 0x8005 ;
      this->reflectin  = true ;
      this->reflectout = true ;
      this->reset () ;
      actual16Bit_d = this->generateCRC ( (unsigned char*)testData, testBytes, true ) ;

      unsigned short table16[crcTABLE_ENTRIES] ;
      this->generateTable ( table16 ) ;
      this->reset () ;
      actual16Bit_t = this->generateCRC ( table16, (unsigned char*)testData, testBytes, true ) ;

      //* 32-bit setup parameters *
      this->regwidth   = 32 ;
      this->reginit    = 0xFFFFFFFF ;
      this->xorfinal   = 0xFFFFFFFF ;
      this->poly       = 0x04C11DB7 ;
      this->reflectin  = true ;
      this->reflectout = true ;
      this->reset () ;
      actual32Bit_d = this->generateCRC ( (unsigned char*)testData, testBytes, true ) ;

      unsigned int table32[crcTABLE_ENTRIES] ;
      this->generateTable ( table32 ) ;
      this->reset () ;
      actual32Bit_t = this->generateCRC ( table32, (unsigned char*)testData, testBytes, true ) ;

      if ( (actual16Bit_d == exp16) && (actual16Bit_t == exp16) &&
           (actual32Bit_d == exp32) && (actual32Bit_t == exp32) )
         match = true ;

      return match ;
   }

   //************************************************************************
   //* setupSuccessful:                                                     *
   //* Return the status of parameter setup.                                *
   //* Note that the CRC generator will still run successfully even if      *
   //* caller has sent invalid parameters; HOWEVER, the results may not be  *
   //* what the caller was expecting.                                       *
   //*                                                                      *
   //* Input  : none                                                        *
   //* Returns: 'true'  if all setup parameters verified                    *
   //*          'false' if one or more parameters out-of-range              *
   //************************************************************************
   bool setupSuccessful ( void )
   {
      return this->goodParms ;
   }

   //************************************************************************
   //* getSetup:                                                            *
   //* Returns a copy of the parameters which will actually be used for     *
   //* CRC generation. If the setup parameters were all verified, then      *
   //* the values returned will match caller's original values. However, if *
   //* there was an out-of-range error during instantiation, the adjusted   *
   //* parameters will be different than caller's original values.          *
   //*                                                                      *
   //* Input  : setup  : (by reference) receives a copy of the actual       *
   //*                   setup parameters                                   *
   //* Returns: nothing                                                     *
   //************************************************************************
   void getSetup ( Crc_Parms& setup )
   {
      setup.regwidth   = this->regwidth ;
      setup.poly       = this->poly ;
      setup.reginit    = this->reginit ;
      setup.xorfinal   = this->xorfinal ;
      setup.reg        = this->reg ;
      setup.reflectin  = this->reflectin ;
      setup.reflectout = this->reflectout ;
   }

   //************************************************************************
   //* version:                                                             *
   //* Return the const string indicating the class version.                *
   //*                                                                      *
   //* Input  : none                                                        *
   //*                                                                      *
   //* Returns: carry mask                                                  *
   //************************************************************************
   const char* version ( void )
   {
      return ( CRC_Gen_Version ) ;
   }

   //************************************
   //* Private methods and data members *
   //************************************
   private:

   //************************************************************************
   //* formatCRC:                                                           *
   //* Format the calculated CRC checksum based on the final XOR value      *
   //* and whether final reflection was specified.                          *
   //*                                                                      *
   //* Input  : none                                                        *
   //* Returns: the formatted CRC value, ready for return to the application*
   //************************************************************************
   unsigned int formatCRC ( void )
   {
      unsigned int crcval ;
      if ( this->reflectout )
         crcval = this->xorfinal ^ this->reflect ( this->reg, this->regwidth ) ;
      else
         crcval = this->xorfinal ^ this->reg ;

      //* Clear any overflow bits *
      crcval &= this->carryMask ;

      return crcval ;
   }

   //************************************************************************
   //* oggDefault:                                                          *
   //* Initialize all data members according to the OGG/Vorbis I            *
   //* specification for CRC generation.                                    *
   //*                                                                      *
   //* Input  : none                                                        *
   //*                                                                      *
   //* Returns: nothing                                                     *
   //************************************************************************
   void oggDefault ( void )
   {
      this->regwidth   = crcMAX_REGWIDTH ; // 32-bit register
      this->reg        =                   // clear register and setup values
      this->reginit    = 
      this->xorfinal   = 0 ;
      this->poly       = crcDFLT_POLY ;    // (from OGG/Vorbis documentation)
      this->reflectin  =                   // no reflection
      this->reflectout = false ;
   }

   //************************************************************************
   //* reflect:                                                             *
   //* Reflect (reverse) the specified number of LSB bits.                  *
   //*                                                                      *
   //* Input  : val   : value to be modified                                *
   //*          bits  : number of bits to be reflected                      *
   //* Returns: reflected value                                             *
   //************************************************************************
   unsigned int reflect ( unsigned int val, int bits )
   {
      unsigned int t = val ;           // single-bit mask

      for ( int i = 0 ; i < bits ; i++ )
      {
         if ( t & 1 )
            val |=  crcBITMASK((bits-1)-i) ;
         else
            val &= ~crcBITMASK((bits-1)-i) ;
         t >>= 1 ;
      }
      return val ;
   }

   //************************************************************************
   //* createMasks:                                                         *
   //* Create the bitmasks to be used during CRC generation.                *
   //*                                                                      *
   //* Input  : none                                                        *
   //* Returns: nothing                                                     *
   //************************************************************************
   void createMasks ( void )
   {
      this->topbitMask = crcBITMASK(this->regwidth - 1) ;
      this->carryMask  = (((this->topbitMask - 1) << 1) | 1) ;
   }

   //************************************************************************
   //* tableEntry:                                                          *
   //* Called by generateTable() to generate a table entry for the          *
   //* specified index position.                                            *
   //*                                                                      *
   //* Input  : tableIndex : offset into table                              *
   //*                                                                      *
   //* Returns: table entry value                                           *
   //************************************************************************
   unsigned int tableEntry ( int tableIndex )
   {
      unsigned int reg,
                   inbyte = (unsigned int) tableIndex ;

      //* Reflect (reverse) the bits if specified *
      if ( this->reflectin )
      {
         inbyte = this->reflect ( inbyte, crcBYTESIZE ) ;
      }

      //* Shift LSByte into MSByte position *
      reg = inbyte << (this->regwidth - crcBYTESIZE) ;

      for ( short i = 0 ; i < crcBYTESIZE ; ++i )
      {
         //* If MSB of the register is set, then     *
         //* shift out the MSB and XOR with the poly.*
         if ( reg & this->topbitMask )
            reg = (reg << 1) ^ this->poly ;

         //* Otherwise, MSB not set, simply shift register left by one bit.*
         else
            reg <<= 1 ;
      }

      //* Re-reflect if specified *
      if ( this->reflectin )
      {
         reg = this->reflect ( reg, this->regwidth ) ;
      }

      //* Apply the carry-out mask and return the value *
      return ( (reg & this->carryMask) ) ;
   }

   //****************
   //* Data Members *
   //****************
   int            regwidth ;     // Width of register in bits [8,32]
   unsigned int   reg ;          // CRC register (accumulator)
   unsigned int   reginit ;      // Initial CRC register value.
   unsigned int   xorfinal ;     // Final XOR vaule applied to calculated CRC
   unsigned int   poly ;         // Algorithm's polynomial value
   unsigned int   topbitMask ;   // Mask for isolating the MSB of the register
   unsigned int   carryMask ;    // Mask for eliminating carry-out bits
   bool           reflectin ;    // Reflect (reverse) bits of input bytes
   bool           reflectout ;   // Reflect (reverse) bits of final CRC value
   bool           resetFlag ;    // 'true' if reset has just occurred ('reg' at initial value)
   bool           goodParms ;    // 'true' if all setup parameters verified, else 'false'
} ;
