#!/usr/bin/perl
use strict ;
use warnings ;
use Term::ANSIColor ;

# **************************************** #
# Create CommExtract distribution archive. #
# **************************************** #
# Updated: 01-Oct-2024
# Author : Mahlon R. Smith
#          Copyright (c) 2022-2024 Mahlon R. Smith, The Software Samurai
#              GNU GPL copyright notice located in CommExtract.cpp
# Developed under: Perl 5.36.1 (x86_64)
# Create an archive file for distribution for the AnsiCmd class library and
# the EarthPoints test application.
# Call from base AnsiCmd/install directory: ./create_archive.pl

   # Define the application source filenames.#
   my @AppList = 
      (
       "CommExtract.hpp",        # Application-class definition
       "CommExtract.cpp",        # Application-class implementation
       "CxFile.hpp",             # File-management definitions
       "CxFile.cpp",             # File-management methods
       "gString.hpp",            # Class definition for text formatting and analysis
       "gString.cpp",            # Implementation of gString class
       "Makefile",               # make (gmake) recipe file
       "comex",                  # binary executable (x86_64)
      ) ;

   # Define the list of supplimental filenames.#
   my @SupList = 
      (
       "README",                 # Pseudo-documentation
       "create_archive.pl",      # Perl script for creating distribution archive
      ) ;


   my $AppCount = scalar @AppList ;    # number of application source files expected
   my $SupCount = scalar @SupList ;    # number of supplimental files expected
   my $TrgTotal = $AppCount + $SupCount ;
   my $TrgAcc   = 0 ;                  # total targets verified
   my $SrcOK    = 0 ;                  # number of source files found
   my $TrgOK    = 0 ;                  # number of files copied to target
   my $SrcSpec  = "" ;                 # source filespec
   my $TrgSpec  = "" ;                 # target filespec

   # Define source and target directories.#
   use Cwd qw ( getcwd ) ;
   chdir ( ".." ) ;
   my $SrcDir  = getcwd () ;
   my $TrgBase = "$SrcDir/archive" ;
   my $TrgDir  = "$TrgBase/CommExtract" ;


   system ( "clear" ) ;
   print colored ( "** Copy source to install directory. **", 'blue' ) ;
   print colored ( "\nsource: '$SrcDir'", 'blue' ) ;
   print colored ( "\ntarget: '$TrgDir'", 'blue' ) ;

   # If base target directory does not exist, create it.#
   if ( ! -d $TrgDir )
   { mkdir $TrgDir ; }

   # If all target directory exists, copy the source files.#
   if ( -d $TrgDir )
   {
      # Verify that all supplimental (non-source-code) #
      # files exist and copy them to target directory. #
      foreach my $i ( 0 .. $#SupList )
      {
         $SrcSpec = sprintf ( "%s/%s", $TrgBase, $SupList[$i] ) ;
         if ( -e $SrcSpec )
         {
            $SrcOK += 1 ;
            $TrgSpec = sprintf ( "%s/%s", $TrgDir, $SupList[$i] ) ;
            system ( "cp --preserve=all '$SrcSpec' '$TrgSpec'" ) ;
            if ( -e $TrgSpec )
            { $TrgOK += 1 ; }
            #print "\nSRC:$SrcSpec" ;    # DEBUG ONLY
            #print "\nTRG:$TrgSpec" ;    # DEBUG ONLY
         }
      }
      $TrgAcc += $TrgOK ;
      #print colored ( "\nSupCount:   $SupCount  SrcOK:   $SrcOK  TrgOK:   $TrgOK", 'blue' ) ;  # DEBUG ONLY

      if ( $TrgOK != $SupCount )
      {
         print "\nError(s) copying supplimental files to target!\n\n" ;
         exit ;
      }
      $SrcOK = 0 ;      # reset the counters
      $TrgOK = 0 ;

      # Verify that all base directory source files exist #
      # and copy them to target directory.                #
      foreach my $i ( 0 .. $#AppList )
      {
         $SrcSpec = sprintf ( "%s/%s", $SrcDir, $AppList[$i] ) ;
         if ( -e $SrcSpec )
         {
            $SrcOK += 1 ;
            $TrgSpec = sprintf ( "%s/%s", $TrgDir, $AppList[$i] ) ;
            system ( "cp --preserve=all '$SrcSpec' '$TrgSpec'" ) ;
            if ( -e $TrgSpec )
            { $TrgOK += 1 ; }
            else
            { print colored ( "\nTarget Error! $TrgSpec", 'red' ) ; }
            #print "\n$SrcSpec" ;    # DEBUG ONLY
            #print "\n$TrgSpec" ;    # DEBUG ONLY
         }
         else
         { print colored ( "\nSource Error! $SrcSpec", 'red' ) ; }
      }
      $TrgAcc += $TrgOK ;
      #print colored ( "\nAppCount:  $AppCount  SrcOK:  $SrcOK  TrgOK:  $TrgOK", 'blue' ) ; #TEMP

      print colored ( "\nTotals  :  $TrgAcc of $TrgTotal\n", 'blue' ) ;
   }
   else
   {
      print "\nError! Target directory not found.\n\n" ;
      exit ;
   }

   if ( $TrgAcc == $TrgTotal )
   {
      # Go to target base directory. Archive created here. #
      chdir ( "$TrgBase" ) ;
      my $WD=getcwd() ;

      if ( $WD eq $TrgBase )
      {
         # Verify that the conditional-compilations are disabled.#
         print colored ( "Debuging Code Disabled?\n", 'blue' ) ;
         my $SearchString = "#define DEBUG_" ;
         $TrgSpec = sprintf ( "%s/%s", $TrgDir, $AppList[1] ) ;
         my $DebugString = qx ( grep -h '$SearchString' $TrgSpec 2>&1 ) ;
         if ( ! (src_scan ( $DebugString )) )
         {
            print colored ( "    Disable Debugging Code, Dummy!\n", 'red' ) ;
            exit ;
         }

         # Create the distribution archive file.            #
         # The name of the target archive is based on the   #
         # application version number which is located in   #
         # the main source file. Forman: "n.n.nn"           #
         $SearchString = "static const wchar_t[*] const appVersion = L" ;
         $TrgSpec = sprintf ( "%s/%s", $TrgDir, $AppList[1] ) ;
         my $VersionString = qx ( grep -hm 1 '$SearchString' $TrgSpec 2>&1 ) ;
         chomp $VersionString ;
         my $vi = 0 ;
         if ( ($vi = index ( "$VersionString", "\"" )) > 0 )
         {
            ++$vi ;
            my $AppVersion = substr ( $VersionString, $vi, 6 ) ;
            print colored ( "Distro Version:'$AppVersion'\n", 'blue' ) ;

            my $targetName = "commextract" ;
            my $PACK_TARGET = "$targetName-$AppVersion.tar.bz2" ;
            print colored ( "Create Archive: ", 'blue' ) ;
            print "'$WD/$PACK_TARGET'\n" ;

            unlink $PACK_TARGET ;      # delete existing target (if any)

            # Create the archive #
            system ( "tar --numeric-owner -cjf '$PACK_TARGET' 'CommExtract'" ) ;

            # DEBUG ONLY: visually verify the results #
            #print "\n" ; system ( "tar -tvf $PACK_TARGET" ) ;  # DEBUG ONLY

            # Verify archive creation #
            if ( -e $PACK_TARGET )
            {
               print colored ( "Created: $PACK_TARGET\t", 'blue' ) ;
               print colored ( "OK", 'green' ) ;

               # Delete the temporary files. #
               print colored ( "\nDelete temporary files:\t\t", 'blue' ) ;
               system ( "rm $TrgDir/*" ) ;
               system ( "rmdir $TrgDir" ) ;
               if ( ! -e $TrgDir )
               { print colored ( "OK", 'green' ) ; }
               else
               { print colored ( "ERROR!", 'red' ) ; }
            }
            else
            { print "\nError creating archive!" ; }
         }
         else
         { print "\nError! Version string not found in $AppList[1]." ; }
      }
      else
      { print "\nError! Unable to reach archive directory." ; }
   }

   print "\n\n" ;


   # Scan the source code for conditional-compile  #
   # declarations used during development.         #
   # Isolate each declaration by deleting comments #
   # and leading/trailing whitespace.              #
   # Test whether enabled "(1)" or disabled "(0)". #
   sub src_scan
   {
      my $src = $_[0] ;
      my $bi = 0 ;
      my $ei = 0 ;
      my $dbFound = 0 ;
      my $dbGood  = 0 ;
      my $allGood = 0 ;
      my $s  = "" ;

      while ( ($ei = index ( $src, ')', $bi )) > 0 )
      {
         ++$dbFound ;
         ++$ei ;
         $s = substr ( $src, $bi, ($ei-$bi) ) ;
         $s =~ s/^\s+|\s+$//g ;
         print colored ( "   '$s'\t", 'blue' ) ;
         if ( ((index ( "$s", "#define DEBUG_" )) >= 0) &&
              ((index ( "$s", "(0)" )) > 0) )
         { print colored ( "OK\n", 'green' ) ; ++$dbGood ; }
         else
         { print colored ( "ERROR!\n\n", 'red' ) ; }
         $bi = $ei ;
      }
      if ( ($dbFound > 0) && ($dbFound == $dbGood) )
      { $allGood = 1 ; }
      return $allGood ;
   }
