//********************************************************************************
//* BubbleSort.cpp                                                               *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2025 Mahlon R. Smith, The Software Samurai        *
//*                  GNU GPL copyright notice located in FileMangler.hpp         *
//* Date       : 20-Feb-2025                                                     *
//* Version    : (see appVersion in BubbleSort.hpp)                              *
//*                                                                              *
//* Description:                                                                 *
//* Perform a bidirectional bubble sort on a randomized array of strings or      *
//* a randomized array of integer values.                                        *
//*                                                                              *
//* Notes:                                                                       *
//* 1) The "bubble sort" is the simplest (and slowest) of the common sorting     *
//*    algorithms. It's great flexibility is its strength, but it is recommend   *
//*    that it be used only if the total number of records is "relatively small".*
//* 2) This application uses a bi-directional bubble sort, also known as the     *
//*    "cocktail sort", presumably because its inventor was sipping a martini    *
//*    at the time.                                                              *
//* 3) Sorting record _pointers_ is much faster than sorting the actual records, *
//*    regardless of record content, and therefore, except for pure numeric data,*
//*    pointer sorting is recommended whenever practical.                        *
//* 4) When performing string comparisons, there is a choice to be made between  *
//*    a simple numeric byte comparison: 'strncmp' 'strncasecmp' versus          *
//*    a locale-aware byte comparison: 'strcoll'.                                *
//*    -- The so-called POSIX or C/C++ locale is very similar to the English     *
//*       language (ASCII) locale, so these byte comparisons are equivalent.     *
//*    -- If a language-specific comparison of UTF-8 data (all of the same       *
//*       language) is desired, the locale-aware 'strcoll' function should be    *
//*       used. See the SortCollated() function.                                 *
//*    -- Note that this application sets the active locale during startup, so   *
//*       that either the numeric sort or locale-aware sort may be used.         *
//*    -- Because your author typically works simultaneously in multiple         *
//*       languages, the UTF-8 comparison functions are bypassed in favor of a   *
//*       generalized approach.                                                  *
//*       By comparing UTF-32 (wchar_t) characters rather that UTF-8 bytes,      *
//*       the data are sorted numerically, but by character. This provides a     *
//*       consistent result whether the data are English, Italian, Chinese,      *
//*       Arabic, or some combination of all human and non-human languages.      *
//*       Refer to the C-language functions, 'wcsncmp' and 'wcsncasecmp'.        *
//* 5) When sorting complex records such as class instances, a method must be    *
//*    devised to determine which record is greater or lesser than the other.    *
//*    -- The example of a complex object used in this application is the        *
//*       WinPos class which has built-in "operator==" and "operator!=" methods, *
//*       as well as an assignment operator.                                     *
//*    -- If the object has no comparison operators, one of the data members     *
//*       can be used for the comparison; for example sort numerically by the    *
//*       'row' member. if ( wpA.row > wpB.row ) ...                             *
//*    -- Again, sorting pointers to objects is MUCH faster that sorting the     *
//*       objects themselves. Hint: Pointers _are_ numeric data.                 *
//*       This application demonstrates both methods; however, pointer sorting   *
//*       is the recommended method.                                             *
//* 6) Command-line Arguments:                                                   *
//*    --dir=[a | d]                                                             *
//*         'a' == ascending (default)                                           *
//*         'd' == descending                                                    *
//*    --type=[t | T | a | A | r | R | n] (optional)                             *
//*         't' == text data pointers (default)                                  *
//*         'T' == text data direct                                              *
//*         'a' == alternate text data pointers                                  *
//*         'A' == alternate text data direct                                    *
//*         'r' == complex data record pointers (see note above)                 *
//*         'R' == complex data records direct                                   *
//*         'n' == integer numeric data                                          *
//*    --src=filename (optional) file containing TEXT data to be sorted          *
//*         - If specified, read each record (row) of the file, sort the         *
//*           records and write them to stdout. Output may be redirected         *
//*           (piped) to an output file.                                         *
//*           Example: bubsort --dir=a --src='source_file' 1>>'target_file'      *
//*         - If not specified, then internal sample data are used.              *
//* 7) String comparison and copying are performed by the author's gString class *
//*    so smelly (and dangerous) C-language string manipulation does not foul    *
//*    our pristine coding style.                                                *
//*    --Note that string comparisons performed here are case-sensitive by       *
//*      default. Case-insensitive sort may be specified via the '--case=n' flag.*
//*      This resets the 'casesen' parameter for the call to the gString         *
//*      comparison function:                                                    *
//*        Case Sensitive   : if ( (gsI.compare( gsL, true )) < ZERO )           *
//*        Case Insensitive : if ( (gsI.compare( gsL, false )) < ZERO )          *
//*    --To perform locale-aware comparisons on text data, the 'compare' call    *
//*      may be replaced with a call to 'compcoll':                              *
//*        if ( (gsI.compcoll( gsL )) < ZERO )                                   *
//*      This will invoke the underlying 'wcscoll' library function.             *
//*      Note that case-sensitivity is automatic for the 'compcoll' and          *
//*      'wcscoll' functions.                                                    *
//*      Please refer to the SortCollated() method for more information.         *
//*                                                                              *
//*------------------------------------------------------------------------------*
//* Copyright Notice:                                                            *
//* This program is free software: you can redistribute it and/or modify it      *
//* under the terms of the GNU General Public License as published by the Free   *
//* Software Foundation, either version 3 of the License, or (at your option)    *
//* any later version.                                                           *
//*                                                                              *
//* This program is distributed in the hope that it will be useful, but          *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
//* for more details.                                                            *
//*                                                                              *
//* You should have received a copy of the GNU General Public License along      *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.              *
//*------------------------------------------------------------------------------*
//* Version History (most recent first):                                         *
//*                                                                              *
//* v: 0.00.03 18-Feb-2025                                                       *
//*   -- Create an alternate text dataset to better exercise case-sensitive      *
//*      vs. case-insensitive sorting.                                           *
//*   -- Add debugging output to the SortCollated method for exploring the       *
//*      standard library wcscoll() function.                                    *
//*                                                                              *
//* v: 0.00.02 02-Feb-2025                                                       *
//*   -- Clean up comments, move support code to seperate module.                *
//*   -- Add support for reading external source data.                           *
//*   -- Add option for case-insensitive sort.                                   *
//*   -- Update the SortCollated() method to use more demonstrative source data. *
//*   -- Create a README file by copying and enhancing source code notes.        *
//*      Create an HTML version of the README file for use on the website.       *
//*   -- Integrate the full beta-release of the gString class, a pre-built       *
//*      instance of gString.lib. It is planned that for the general release,    *
//*      the mainline source code for the gString class will be included in the  *
//*      package.                                                                *
//*                                                                              *
//* v: 0.00.01 30-Jan-2025                                                       *
//*   -- First effort: This utility was created to assist in debugging of        *
//*      bubble sorts used in other applications.                                *
//*   -- Template for the algorithm was taken from the author's stable           *
//*      FileMangler application, available as a seperate download.              *
//*      (See the FMgrSort.cpp module of that package.)                          *
//*   -- Note that this release uses the _temporary_ bString class which         *
//*      as of this release, is still in beta-test. It is hoped that the         *
//*      production release of the gString class will soon be available.         *
//*   -- All sorting algorithms have been verified.                              *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************
#include "BubbleSort.hpp"

//***************
//* Definitions *
//***************

//**************
//* Local data *
//**************

//********************
//* Test data - Text *
//********************
//* Source text data for standard numeric sort.*
static wchar_t stdText[srcRECORDS][recLENGTH] = 
{
   { L"Cranberries are sour.\n" },
   { L"Grapes grow in California.\n" },
   { L"Hummus tastes awful.\n" },
   { L"Blueberries are messy.\n" },
   { L"Dates are grown in Egypt.\n" },
   { L"Jalapeños are spicy.\n" },
   { L"Apples are tasty\n" },
   { L"Ice cream grows on ice plants.\n" },
   { L"Flour is made from wheat.\n" },
   { L"Eggs come from chickens.\n" },
} ;
static wchar_t* stdPtr[srcRECORDS] = 
{
   stdText[0], stdText[1], stdText[2], stdText[3], stdText[4], 
   stdText[5], stdText[6], stdText[7], stdText[8], stdText[9] 
} ;

//* Mixed uppercase/lowercase dataset *
static wchar_t altText[srcRECORDS][recLENGTH] = 
{
   { L"bears eat bugs.\n" },
   { L"Dogs are furry.\n" },
   { L"Eagles fly north.\n" },
   { L"apples are red.\n" },
   { L"Bears eat honey.\n" },
   { L"dogs are friendly.\n" },
   { L"Cartoons are animated.\n" },
   { L"Apples are green.\n" },
   { L"eagles fly south.\n" },
   { L"cartoons are funny.\n" },
} ;
static wchar_t* altPtr[srcRECORDS] = 
{
   altText[0], altText[1], altText[2], altText[3], altText[4], 
   altText[5], altText[6], altText[7], altText[8], altText[9] 
} ;

//***********************
//* Test data - Numeric *
//***********************
static int srcValues[srcRECORDS] = 
{ 350, 150, 151, 214, 976, 500, 475, 101, 115 } ;

//*******************************
//* Test data - Complex records *
//*******************************
static WinPos winposData[srcRECORDS] =
{
   { 16, 16 }, { 13, 13 }, { 18, 18 }, { 11, 11 }, 
   { 19, 19 }, { 14, 14 }, { 17, 17 }, { 12, 12 }, { 15, 15 }
} ;
static WinPos* winposPtr[srcRECORDS] = 
{
   &winposData[0], &winposData[1], &winposData[2], &winposData[3], 
   &winposData[4], &winposData[5], &winposData[6], &winposData[7], &winposData[8] 
} ;

//********************
//* Local prototypes *
//********************


//********************************************************************************
//* main:                                                                        *
//* Program entry point.                                                         *
//* In a C++ program, 'main' should contain as little code as possible.          *
//*                                                                              *
//********************************************************************************

int main ( int argc, char* argv[], char* argenv[] )
{
   BubbSort* bsptr = new BubbSort ( argc, argv, argenv ) ;

   if ( bsptr != NULL )
      delete bsptr ;

   return EXIT_SUCCESS;

}  //* End main() *

//********************************************************************************
//* BubbSort initialization constructor:                                         *
//*  1) initialize all data members                                              *
//*  2) initialize the display area and write app title                          *
//*  3) set the locale from the environment (if available)                       *
//*  4) parse command-line arguments                                             *
//*  5) Execute:                                                                 *
//*     a) if valid arguments, perform the specified sort                        *
//*     b) else display version number if requested, else display help           *
//*                                                                              *
//* Input  : argc    : number of arguments                                       *
//*          argv    : text of each argument (argv[0] is the application name)   *
//*          argenv  : array of environment variables (currently unused)         *
//*                                                                              *
//* Returns: implicitly returns a pointer to the object                          *
//********************************************************************************

BubbSort::BubbSort ( int argc, char* argv[], char* argenv[] )
{
   const char* unsortTemplate = 
         "Unsorted Data (%d records)\n-------------------------------------\n" ;
   const char* sortedTemplate = 
         "Sorted Data (%s, %s)\n-------------------------------------\n" ;
   gString gsOut, gs ;              // text formatting

   this->low2high   = true ;        // initialize data members
   this->records    = srcRECORDS ;
   this->stype      = tpSptr ;
   this->casesen    = true ;
   this->version    = false ;
   this->srcFile[0] = L'\0' ;
   this->extData    = NULL ;
   this->extDPtr    = NULL ;
   this->extSize    = ZERO ;

   system ( "clear" ) ;
   gsOut.compose( titleTemplate, appVersion, appYears ) ;
   gsOut.padCols( gsOut.gscols() * 2, L'=' ) ;
   wcout << gsOut << endl ;

   //* Set the "locale" for character encoding *
   //* and text-format handling.               *
   this->SetLocale () ;

   //* Decode command-line arguments *
   if ( (this->GetCommlineArgs ( argc, argv, argenv )) )
   {
      if ( (this->stype == tpSptr) || (this->stype == tpStxt) ||
           (this->stype == tpAptr) || (this->stype == tpAtxt) )
      {
         wchar_t** sourcePtr = ((this->stype == tpAptr) || (this->stype == tpAtxt)) ? 
                                altPtr : stdPtr ;

         gsOut.compose( unsortTemplate, &this->records ) ;
         wcout << gsOut ;
         this->report ( sourcePtr, this->records, ZERO ) ;

         if ( (this->stype == tpSptr) || (this->stype == tpAptr) )
         {
            if ( this->low2high )
               this->SortLow2HighTextPointer ( sourcePtr, this->records ) ;
            else
               this->SortHigh2LowTextPointer ( sourcePtr, this->records ) ;
         }
         else
         {
            if ( this->low2high )
               this->SortLow2HighTextDirect ( (this->stype == tpAtxt ? altText : stdText), 
                                              this->records ) ;
            else
               this->SortHigh2LowTextDirect ( (this->stype == tpAtxt ? altText : stdText), 
                                              this->records ) ;
         }

         gsOut.compose( sortedTemplate, 
                        (this->low2high ? "ascending" : "descending"),
                        (this->casesen  ? "sensitive" : "insensitive") ) ;
         wcout << gsOut ;
         this->report ( sourcePtr, this->records, ZERO ) ;
      }
      else if ( (this->stype == tpRptr) || (this->stype == tpRrec) )
      {
         gsOut.compose( unsortTemplate, &this->records ) ;
         wcout << gsOut ;
         this->report ( winposPtr, this->records, ZERO ) ;

         if ( this->stype == tpRptr )
         {
            if ( this->low2high )
               SortLow2HighRecordPointer ( winposPtr, this->records ) ;
            else
               SortHigh2LowRecordPointer ( winposPtr, this->records ) ;
         }
         else  // tbRrec
         {
            if ( this->low2high )
               SortLow2HighRecords ( winposData, this->records ) ;
            else
               SortHigh2LowRecords ( winposData, this->records ) ;
         }

         gsOut.compose( sortedTemplate, (this->low2high ? "ascending" : "descending") ) ;
         wcout << gsOut ;
         this->report ( winposPtr, this->records, ZERO ) ;
      }
      else if ( this->stype == tpInt )    // integer numeric data
      {
         gsOut.compose( unsortTemplate, &this->records ) ;
         wcout << gsOut ;
         this->report ( srcValues, this->records, ZERO ) ;

         if ( this->low2high )
            SortLow2HighInteger ( srcValues, this->records ) ;
         else
            SortHigh2LowInteger ( srcValues, this->records ) ;

         gsOut.compose( sortedTemplate, (this->low2high ? "ascending" : "descending") ) ;
         wcout << gsOut ;
         this->report ( srcValues, this->records, ZERO ) ;
      }
      else if ( this->stype == tpColl )   // collate text (pointers only)
      {
         gsOut.compose( unsortTemplate, &this->records ) ;
         gsOut.insert( L"   " ) ;
         int indx = gsOut.after( L'\n' ) ;
         gsOut.insert( L"   ", indx ) ;
         wcout << gsOut ;
         this->report ( altPtr, this->records, 3 ) ;
         this->SortCollated () ;
      }
      else  // (stype == stFile)
      {
         if ( (this->SortExternalRecords ()) )
         {
            gsOut.compose( sortedTemplate, (this->low2high ? "ascending" : "descending") ) ;
            gs.compose( "(%d records)", &this->records ) ;
            gsOut.insert( gs.gstr(), (gsOut.after( L"Data " )) ) ;
            gsOut.insert( L"--", (gsOut.findlast( L'\n' )) ) ;
            wcout << gsOut ;
            this->report ( this->extDPtr, this->records, ZERO ) ;
         }
      }
      wcout.flush() ;
   }

   //* Argument error, or cry for help *
   else
   {
      if ( this->version )
         this->Version () ;
      else
         this->Help () ;
   }

   //* If a locale object has been allocated, delete it now.*
   if ( this->ioLocale != NULL )
      delete ( ioLocale ) ;

}  //* End BubbSort() *

//********************************************************************************
//* SortLow2HighTextPointer:                                                     *
//* Sort an array of POINTERS to text data.                                      *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortLow2HighTextPointer ( wchar_t** src, int recs )
{
   gString gsI, gsL, gsG ;       // temp buffers
   wchar_t *saveLow, *saveHigh ; // temp pointers
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         gsI = src[ini] ;              // load records for comparison
         gsL = src[lti] ;

         //* Compare the records.                  *
         //* If current record is less than        *
         //* lower-limit record, swap the records. *
         if ( (gsI.compare( gsL, this->casesen )) < ZERO )
         {
            saveLow  = src[lti] ;
            src[lti] = src[ini] ;
            src[ini] = saveLow ;
            gsL = gsI ;                   // refresh the comparison data
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            gsG = src[gti] ;

            //* Compare the records.                     *
            //* If record at upper limit is less than    *
            //* target record, swap the records.         *
            if ( (gsG.compare( gsI, this->casesen )) < ZERO )
            {
               saveHigh = src[gti] ;
               src[gti] = src[ini] ;
               src[ini] = saveHigh ;
               gsI = gsG ;                // refresh the comparison data
            }

            //* Compare the records.               *
            //* If current record less than target *
            //* record, swap the records.          *
            if ( (gsI.compare( gsL, this->casesen )) < ZERO )
            {
               saveLow = src[lti] ;
               src[lti] = src[ini] ;
               src[ini] = saveLow ;
            }
         }
      }
   }

}  //* End SortLow2HighTextPointer() *

//********************************************************************************
//* SortHigh2LowTextPointer:                                                     *
//* Sort an array of POINTERS to text data.                                      *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortHigh2LowTextPointer ( wchar_t** src, int recs )
{
   gString gsI, gsL, gsG ;       // temp buffers
   wchar_t *saveLow, *saveHigh ; // temp pointers
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index
   gString gsOut ;               // output buffer

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         gsI = src[ini] ;              // load records for comparison
         gsL = src[lti] ;

         //* Compare the records.                  *
         //* If current record is greater than     *
         //* lower-limit record, swap the records. *
         if ( (gsI.compare( gsL, this->casesen )) > ZERO )
         {
            saveLow  = src[lti] ;
            src[lti] = src[ini] ;
            src[ini] = saveLow ;
            gsL = gsI ;                   // refresh the comparison data
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            gsG = src[gti] ;

            //* Compare the records.                     *
            //* If record at upper limit is greater than *
            //* target record, swap the records.         *
            if ( (gsG.compare( gsI, this->casesen )) > ZERO )
            {
               saveHigh = src[gti] ;
               src[gti] = src[ini] ;
               src[ini] = saveHigh ;
               gsI = gsG ;                // refresh the comparison data
            }

            //* Compare the records.                  *
            //* If current record greater than target *
            //* record, swap the records.             *
            if ( (gsI.compare( gsL, this->casesen )) > ZERO )
            {
               saveLow = src[lti] ;
               src[lti] = src[ini] ;
               src[ini] = saveLow ;
            }
         }
      }
   }

}  //* End SortHigh2LowTextPointer() *

//********************************************************************************
//* SortLow2HighTextDirect:                                                      *
//* Sort an array of text strings by moving the string data directly.            *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortLow2HighTextDirect ( wchar_t src[][recLENGTH], int recs )
{
   gString gsI, gsL, gsG ;       // temp buffers
   gString saveLow, saveHigh ;   // temp buffers
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         gsI = src[ini] ;              // load records for comparison
         gsL = src[lti] ;

         //* Compare the records.                  *
         //* If current record is less than        *
         //* lower-limit record, swap the records. *
         if ( (gsI.compare( gsL, this->casesen )) < ZERO )
         {
            gsI.copy( src[lti], recLENGTH ) ;
            gsL.copy( src[ini], recLENGTH ) ;
            saveLow = gsL ;            // refresh the comparison data
            gsL = gsI ;
            gsI = saveLow ;
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            gsG = src[gti] ;

            //* Compare the records.                     *
            //* If record at upper limit is less than    *
            //* target record, swap the records.         *
            if ( (gsG.compare( gsI, this->casesen )) < ZERO )
            {
               gsI.copy( src[gti], recLENGTH ) ;
               gsG.copy( src[ini], recLENGTH ) ;
               saveHigh = gsG ;        // refresh the comparison data
               gsG = gsI ;
               gsI = saveHigh ;
            }

            //* Compare the records.               *
            //* If current record less than target *
            //* record, swap the records.          *
            if ( (gsI.compare( gsL, this->casesen )) < ZERO )
            {
               gsI.copy( src[lti], recLENGTH ) ;
               gsL.copy( src[ini], recLENGTH ) ;
            }
         }
      }
   }

}  //* End sortLow2HighTextDirect() *

//********************************************************************************
//* SortHigh2LowTextDirect:                                                      *
//* Sort an array of text strings by moving the string data directly.            *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortHigh2LowTextDirect ( wchar_t src[][recLENGTH], int recs )
{
   gString gsI, gsL, gsG ;       // temp buffers
   gString saveLow, saveHigh ;   // temp buffers
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         gsI = src[ini] ;              // load records for comparison
         gsL = src[lti] ;

         //* Compare the records.                  *
         //* If current record is greater than     *
         //* lower-limit record, swap the records. *
         if ( (gsI.compare( gsL, this->casesen )) > ZERO )
         {
            gsI.copy( src[lti], recLENGTH ) ;
            gsL.copy( src[ini], recLENGTH ) ;
            saveLow = gsL ;            // refresh the comparison data
            gsL = gsI ;
            gsI = saveLow ;
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            gsG = src[gti] ;

            //* Compare the records.                     *
            //* If record at upper limit is greater than *
            //* target record, swap the records.         *
            if ( (gsG.compare( gsI, this->casesen )) > ZERO )
            {
               gsI.copy( src[gti], recLENGTH ) ;
               gsG.copy( src[ini], recLENGTH ) ;
               saveHigh = gsG ;        // refresh the comparison data
               gsG = gsI ;
               gsI = saveHigh ;
            }

            //* Compare the records.            *
            //* If current record greater than  *
            //* target record, swap the records.*
            if ( (gsI.compare( gsL, this->casesen )) > ZERO )
            {
               gsI.copy( src[lti], recLENGTH ) ;
               gsL.copy( src[ini], recLENGTH ) ;
            }
         }
      }
   }

}  //* End sortHigh2LowTextDirect() *

//********************************************************************************
//* SortLow2HighInteger: Sort an array of integer values.                        *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortLow2HighInteger ( int* src, int recs )
{
   int saveLow, saveHigh ;       // temp storage
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         //* Compare the records.                  *
         //* If current record is less than        *
         //* lower-limit record, swap the records. *
         if ( src[ini] < src[lti] )
         {
            saveLow  = src[lti] ;
            src[lti] = src[ini] ;
            src[ini] = saveLow ;
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            //* Compare the records.                     *
            //* If record at upper limit is less than    *
            //* target record, swap the records.         *
            if ( src[gti] < src[ini] )
            {
               saveHigh = src[gti] ;
               src[gti] = src[ini] ;
               src[ini] = saveHigh ;
            }

            //* Compare the records.               *
            //* If current record less than target *
            //* record, swap the records.          *
            if ( src[ini] < src[lti] )
            {
               saveLow = src[lti] ;
               src[lti] = src[ini] ;
               src[ini] = saveLow ;
            }
         }
      }
   }

}  //* End SortLow2HighInteger() *

//********************************************************************************
//* SortHigh2LowInteger: Sort an array of integer values.                        *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortHigh2LowInteger ( int* src, int recs )
{
   int saveLow, saveHigh ;       // temp storage
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         //* Compare the records.                  *
         //* If current record is less than        *
         //* lower-limit record, swap the records. *
         if ( src[ini] > src[lti] )
         {
            saveLow  = src[lti] ;
            src[lti] = src[ini] ;
            src[ini] = saveLow ;
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            //* Compare the records.                     *
            //* If record at upper limit is less than    *
            //* target record, swap the records.         *
            if ( src[gti] > src[ini] )
            {
               saveHigh = src[gti] ;
               src[gti] = src[ini] ;
               src[ini] = saveHigh ;
            }

            //* Compare the records.               *
            //* If current record less than target *
            //* record, swap the records.          *
            if ( src[ini] > src[lti] )
            {
               saveLow = src[lti] ;
               src[lti] = src[ini] ;
               src[ini] = saveLow ;
            }
         }
      }
   }

}  //* End SortHigh2LowInteger() *

//********************************************************************************
//* SortLow2HighRecordPointer: Sort an array of WinPos-class objects.            *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortLow2HighRecordPointer ( WinPos** src, int recs )
{
   WinPos *saveLow, *saveHigh ;  // temp storage
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         //* Compare the records.                  *
         //* If current record is less than        *
         //* lower-limit record, swap the records. *
         if ( src[ini]->row < src[lti]->row )
         {
            saveLow  = src[lti] ;
            src[lti] = src[ini] ;
            src[ini] = saveLow ;
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            //* Compare the records.                  *
            //* If record at upper limit is less than *
            //* target record, swap the records.      *
            if ( src[gti]->row < src[ini]->row )
            {
               saveHigh = src[gti] ;
               src[gti] = src[ini] ;
               src[ini] = saveHigh ;
            }

            //* Compare the records.               *
            //* If current record less than target *
            //* record, swap the records.          *
            if ( src[ini]->row < src[lti]->row )
            {
               saveLow = src[lti] ;
               src[lti] = src[ini] ;
               src[ini] = saveLow ;
            }
         }
      }
   }

}  //* End SortLow2HighRecordPointer() *

//********************************************************************************
//* SortHigh2LowRecordPointer: Sort an array of WinPos-class objects.            *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortHigh2LowRecordPointer ( WinPos** src, int recs )
{
   WinPos *saveLow, *saveHigh ;  // temp storage
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         //* Compare the records.                  *
         //* If current record is greater than     *
         //* lower-limit record, swap the records. *
         if ( src[ini]->row > src[lti]->row )
         {
            saveLow  = src[lti] ;
            src[lti] = src[ini] ;
            src[ini] = saveLow ;
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            //* Compare the records.                  *
            //* If record at upper limit is greater   *
            //* than target record, swap the records. *
            if ( src[gti]->row > src[ini]->row )
            {
               saveHigh = src[gti] ;
               src[gti] = src[ini] ;
               src[ini] = saveHigh ;
            }

            //* Compare the records.            *
            //* If current record greater than  *
            //* target record, swap the records.*
            if ( src[ini]->row > src[lti]->row )
            {
               saveLow = src[lti] ;
               src[lti] = src[ini] ;
               src[ini] = saveLow ;
            }
         }
      }
   }

}  //* End SortLow2HighRecordPointer() *

//********************************************************************************
//* SortLow2HighRecords: Sort an array of WinPos-class objects.                  *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortLow2HighRecords ( WinPos* src, int recs )
{
   WinPos saveLow, saveHigh ;    // temp storage
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         //* Compare the records.                  *
         //* If current record is less than        *
         //* lower-limit record, swap the records. *
         if ( src[ini].row < src[lti].row )
         {
            saveLow  = src[lti] ;
            src[lti] = src[ini] ;
            src[ini] = saveLow ;
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            //* Compare the records.                  *
            //* If record at upper limit is less than *
            //* target record, swap the records.      *
            if ( src[gti].row < src[ini].row )
            {
               saveHigh = src[gti] ;
               src[gti] = src[ini] ;
               src[ini] = saveHigh ;
            }

            //* Compare the records.               *
            //* If current record less than target *
            //* record, swap the records.          *
            if ( src[ini].row < src[lti].row )
            {
               saveLow = src[lti] ;
               src[lti] = src[ini] ;
               src[ini] = saveLow ;
            }
         }
      }
   }

}  //* End SortLow2HighRecords() *

//********************************************************************************
//* SortHigh2LowRecords: Sort an array of WinPos-class objects.                  *
//*                                                                              *
//* Input  : src  : pointer to source array                                      *
//*          recs : number of records                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::SortHigh2LowRecords ( WinPos* src, int recs )
{
   WinPos saveLow, saveHigh ;    // temp storage
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini ;                     // inner-loop index

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         //* Compare the records.                  *
         //* If current record is greater than     *
         //* lower-limit record, swap the records. *
         if ( src[ini].row > src[lti].row )
         {
            saveLow  = src[lti] ;
            src[lti] = src[ini] ;
            src[ini] = saveLow ;
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            //* Compare the records.                  *
            //* If record at upper limit is greater   *
            //* than target record, swap the records. *
            if ( src[gti].row > src[ini].row )
            {
               saveHigh = src[gti] ;
               src[gti] = src[ini] ;
               src[ini] = saveHigh ;
            }

            //* Compare the records.            *
            //* If current record greater than  *
            //* target record, swap the records.*
            if ( src[ini].row > src[lti].row )
            {
               saveLow = src[lti] ;
               src[lti] = src[ini] ;
               src[ini] = saveLow ;
            }
         }
      }
   }

}  //* End SortHigh2LowRecords() *

//********************************************************************************
//* SortCollated: debugging function: exercise C-library 'wcscoll' function.     *
//* This function is implemented in the gString 'compcoll' method.               *
//*    (The equivalent narrow function is 'strcoll' defined in string.h.)        *
//*    (The narrow (UTF-8) function is not used here.                   )        *
//*                                                                              *
//* -- Note that case-sensitivity is built into the called function.             *
//* -- The specified sort direction is supported by this method.                 *
//*                                                                              *
//* Technical Note:                                                              *
//* ---------------                                                              *
//* In the ASCII world, the letter 'A' is numerically less than the letter 'a';  *
//* however, in the world of locale-specific comparison, English lowercase is    *
//* "less than" uppercase, so 'a' is less than 'A'.                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* ======                                                                       *
//* 1) For the English locale, at least, the comparison performed by 'wcscoll'   *
//*    seems to be case-insensitive and returns a value based on what appears    *
//*    to be a sum of compared characters. For example:                          *
//*       comparing "abcdef" with "abcdeg" returns  1.                           *
//*       comparing "abcdef" with "abcdee" returns -1.                           *
//*       comparing "Apples are green." with "apples are red."   returns -142    *
//*       comparing "Eagles fly north." with "eagles fly south." returns -91     *
//*       comparing "Bears eat honey." with "bears eat bugs."    returns 76      *
//*       comparing "Eagles fly north." "Dogs are furry."        returns 12      *
//*       comparing "Eagles fly north." "dogs are friendly."     returns 12      *
//*    The answer to the library function's algorithm is certainly within this   *
//*    data, but frankly, we don't care enough about it to do the math.- Sam     *
//*                                                                              *
//* 2) Note that the gString 'compcoll' method normalizes                        *
//*    the return value as: [-1 | 0 | 1], but for the curious among us, the      *
//*    debugging code used here reports the raw return value.                    *
//*                                                                              *
//* 3) The data would suggest that the collation algorithm is essentially useless*
//*    unless you are writing a dictionary; and it may be useless even then.     *
//********************************************************************************

void BubbSort::SortCollated ( void )
{
   #define DEBUG_COMPCOLL (0)    // for debugging only
   #if DEBUG_COMPCOLL != 0
   const char* dbFile = "./bsdbg.txt" ;
   const char* dbTemplate = "%C) %S (%2d) %C %S" ; 
   gString gsdbg ;
   const wchar_t lt = L'<', gt = L'>', eq = L'=' ;
   wchar_t p, t ;
   int padto = 61 ;
   ofstream ofsdbg ( dbFile, ofstream::out | ofstream::trunc ) ;
   #endif   // DEBUG_COMPCOLL

   gString gsI, gsL, gsG ;       // temp buffers
   wchar_t** src = altPtr ;      // pointer to pointers to source array
   int recs = srcRECORDS ;       // number of records
   wchar_t *saveLow, *saveHigh ; // temp pointers
   int lti = ZERO,               // less-than index
       gti = recs - 1,           // greater-than index
       ini,                      // inner-loop index
       cmp ;                     // result of comparison

   gString gs( "   Collation Algorithm Test: (%s)\n   ",
               (this->low2high ? "ascending" : "descending") ) ;
   gs.padCols( gs.gscols() * 2 - 6, L'-' ) ;
   wcout << gs << endl ;

   for ( lti = 0, gti = recs-1 ; (gti-lti) > ZERO ; ++lti, --gti )
   {
      for ( ini = lti+1 ; ini <= gti ; ++ini )
      {
         gsI = src[ini] ;              // load records for comparison
         gsL = src[lti] ;

         //* Compare the records.                  *
         //* If current record is less than        *
         //* lower-limit record, swap the records. *
         cmp = gsI.compcoll( gsL ) ;

         #if DEBUG_COMPCOLL != 0
         if ( (ofsdbg.is_open()) )
         {
            cmp = wcscoll ( gsI.gstr(), gsL.gstr() ) ;
            t = L'A' ;
            p = (cmp < ZERO ? lt : cmp > ZERO ? gt : eq) ;
            gsdbg.compose( dbTemplate, &t, gsI.gstr(), &cmp, &p, gsL.gstr() ) ;
            gsdbg.erase( L'\n', ZERO, false, true ) ;
            gsdbg.padCols( padto ) ;
            ofsdbg << gsdbg << endl ;
         }
         #endif   // DEBUG_COMPCOLL

         if ( (this->low2high && (cmp < ZERO)) ||
              (!this->low2high && (cmp > ZERO)) )
         {
            saveLow  = src[lti] ;
            src[lti] = src[ini] ;
            src[ini] = saveLow ;
            gsL = gsI ;                   // refresh the comparison data
         }

         //* If current index less that index of limit *
         //*   (avoid comparing a record to itself)    *
         if ( ini < gti )
         {
            gsG = src[gti] ;

            //* Compare the records.                     *
            //* If record at upper limit is less than    *
            //* target record, swap the records.         *
            cmp = gsG.compcoll( gsI ) ;

            #if DEBUG_COMPCOLL != 0
            if ( (ofsdbg.is_open()) )
            {
               cmp = wcscoll ( gsG.gstr(), gsI.gstr() ) ;
               t = L'B' ;
               p = (cmp < ZERO ? lt : cmp > ZERO ? gt : eq) ;
               gsdbg.compose( dbTemplate, &t, gsG.gstr(), &cmp, &p, gsI.gstr() ) ;
               gsdbg.erase( L'\n', ZERO, false, true ) ;
               gsdbg.padCols( padto ) ;
               ofsdbg << gsdbg << endl ;
            }
            #endif   // DEBUG_COMPCOLL

            if ( (this->low2high && (cmp < ZERO)) ||
                 (!this->low2high && (cmp > ZERO)) )
            {
               saveHigh = src[gti] ;
               src[gti] = src[ini] ;
               src[ini] = saveHigh ;
               gsI = gsG ;                // refresh the comparison data
            }

            //* Compare the records.               *
            //* If current record less than target *
            //* record, swap the records.          *
            cmp = gsI.compcoll( gsL ) ;

            #if DEBUG_COMPCOLL != 0
            if ( (ofsdbg.is_open()) )
            {
               cmp = wcscoll ( gsI.gstr(), gsL.gstr() ) ;
               t = L'C' ;
               p = (cmp < ZERO ? lt : cmp > ZERO ? gt : eq) ;
               gsdbg.compose( dbTemplate, &t, gsI.gstr(), &cmp, &p, gsL.gstr() ) ;
               gsdbg.erase( L'\n', ZERO, false, true ) ;
               gsdbg.padCols( padto ) ;
               ofsdbg << gsdbg << endl ;
            }
            #endif   // DEBUG_COMPCOLL

            if ( (this->low2high && (cmp < ZERO)) ||
                 (!this->low2high && (cmp > ZERO)) )
            {
               saveLow = src[lti] ;
               src[lti] = src[ini] ;
               src[ini] = saveLow ;
            }
         }
      }
   }

   this->report ( altPtr, srcRECORDS, 3 ) ;  // report the results

   #if DEBUG_COMPCOLL != 0
   if ( (ofsdbg.is_open()) ) { ofsdbg.close() ; }
   #endif   // DEBUG_COMPCOLL
   #undef DEBUG_COMPCOLL

}  //* End SortCollated() *

//********************************************************************************
//* SortRecordFile:                                                              *
//* Read an array of records from a file.                                        *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: 'true'  if file records captured and saved to '?' member            *
//*          'false' if file does not exist, or is not a "regular" file,         *
//*                  or is not a plain-text file, or if no records were          *
//*                  extracted from the file.                                    *
//********************************************************************************
//* Note for future enhancement: 'colOffset' and 'colWidth' data members:        *
//* -- This is the character offset at which to begin data comparison.           *
//*    This value may be used to sort data which is organized in columns, by     *
//*    indicating which column is to be referenced for the sort.                 *
//* -- Tab characters (ASCII 09h) _ARE NOT_ recognized as column delimiters.     *
//*    A CSV (comma-seperated-values) file which has been formatted for output   *
//*    by passing it through the author's CsvView utility, may be used as input  *
//*    for this option.                                                          *
//* -- This variable is ZERO by default, and a non-zero value is recognized      *
//*    ONLY if the source data are from an external file.                        *
//* -- Note that this option is NOT CURRENTLY IMPLEMENTED.                       *
//********************************************************************************

bool BubbSort::SortExternalRecords ( void )
{
   bool status = false ;

   if ( (this->extData != NULL) && (this->extDPtr != NULL) && (this->records > ZERO ) )
   {
      if ( this->low2high )
         this->SortLow2HighTextPointer ( this->extDPtr, this->records ) ;
      else
         this->SortHigh2LowTextPointer ( this->extDPtr, this->records ) ;
      status = true ;
   }
   return status ;

}  //* End SortExternalRecords() *

//********************************************************************************
//* SetLocale:                                                                   *
//*                                                                              *
//* Set the "locale" for the application.                                        *
//* By default, the application locale is taken from terminal environment.       *
//* (Type: echo $LANG   at the command line to get the terminal's locale.)       *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note:                                                           *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

void BubbSort::SetLocale ( void )
{

   this->ioLocale = new locale("") ;// get default locale from environment
   //* Give the locale "global" (application/process) scope.*
   this->ioLocale->global( *this->ioLocale ) ;

}  //* End SetLocale() *

//********************************************************************************
//* Display the list of text records.                                            *
//*                                                                              *
//* Input  : src    : pointer to source array                                    *
//*          recs   : number of records                                          *
//*          indent : number of columns to indent                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::report ( wchar_t** src, int recs, int indent )
{
   gString gsOut, gsIndent ;
   gsIndent.padCols( indent ) ;

   for ( int r = ZERO ; r < recs ; ++r )
   {
      gsOut.compose( "%S%02d) %S", gsIndent.gstr(), &r, src[r] ) ;
      wcout << gsOut ; wcout.flush() ;
   }
   wcout << endl ;

}  //* End report() *

//********************************************************************************
//* Display the list of numeric records.                                         *
//*                                                                              *
//* Input  : src    : pointer to source array                                    *
//*          recs   : number of records                                          *
//*          indent : number of columns to indent                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::report ( int* src, int recs, int indent )
{
   gString gsOut, gsIndent ;
   gsIndent.padCols( indent ) ;

   for ( int r = ZERO ; r < recs ; ++r )
   {
      gsOut.compose( "%S%02d) %3d\n", gsIndent.gstr(), &r, &src[r] ) ;
      wcout << gsOut ; wcout.flush() ;
   }
   wcout << endl ;

}  //* End report() *

//********************************************************************************
//* Display the list of WinPos records.                                          *
//*                                                                              *
//* Input  : src    : pointer to source array                                    *
//*          recs   : number of records                                          *
//*          indent : number of columns to indent                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::report ( WinPos** src, int recs, int indent )
{
   gString gsOut, gsIndent ;
   gsIndent.padCols( indent ) ;

   for ( int r = ZERO ; r < recs ; ++r )
   {
      gsOut.compose( "%S%02d) %2hd:%2hd\n", 
                     gsIndent.gstr(), &r, &src[r]->row, &src[r]->col ) ;
      wcout << gsOut ; wcout.flush() ;
   }
   wcout << endl ;

}  //* End report() *

//********************************************************************************
//* shuffle: Debugging function: re-order the source data.                       *
//*                                                                              *
//* Input  : shuf : reshuffle seed value                                         *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void BubbSort::shuffle ( int shuf )
{
   gString gsSrc, gsTrg ;
   if ( shuf < 1 )
      shuf = 1 ;
   else if ( shuf >= srcRECORDS )
      shuf = srcRECORDS - 1 ;

   int src = shuf, trg = ZERO ;

   //* Standard-text Dataset *
   if ( (this->stype == tpSptr) || (this->stype == tpStxt) )
   {
      do
      {
         gsSrc = stdText[src] ;
         gsTrg = stdText[trg] ;
         gsTrg.copy( stdText[src], recLENGTH ) ;
         gsSrc.copy( stdText[trg], recLENGTH ) ;
         ++trg ;
      }
      while ( ++src < srcRECORDS ) ;
   }

   //* Alternate-text Dataset *
   else
   {
      do
      {
         gsSrc = altText[src] ;
         gsTrg = altText[trg] ;
         gsTrg.copy( altText[src], recLENGTH ) ;
         gsSrc.copy( altText[trg], recLENGTH ) ;
         ++trg ;
      }
      while ( ++src < srcRECORDS ) ;
   }

}  //* End shuffle() *
