//********************************************************************************
//* File       : AnsiCmdWin.hpp                                                  *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2022-2023 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 09-Aug-2023                                                     *
//* Version    : (see AnsiCmdVersion string in AnsiCmd.cpp)                      *
//*                                                                              *
//* Description: AnsiCmd class definition, supplementary.                        *
//*              Window-oriented objects are defined here.                       *
//*                                                                              *
//*                                                                              *
//********************************************************************************

//***********************************************************************
//* acWin class                                                         *
//* -----------                                                         *
//* This is a derived class whose parent is the AnsiCmd class.          *
//*                                                                     *
//* The public methods of AnsiCmd are inherited from the parent.        *
//* These include the methods which provide the ANSI escape             *
//* sequences and general housekeeping methods.                         *
//*                                                                     *
//* The private data members of the parent class are not directly       *
//* available.                                                          *
//***********************************************************************

class ACWin : public AnsiCmd
{
   public:

   //* Destructor: Return all allocated resources to the system before  *
   //* exit.                                                            *
   //*                                                                  *
   //* Input  : none                                                    *
   //*                                                                  *
   //* Returns: nothing                                                 *
   virtual ~ACWin () ;

   //* Default Constructor:                                             *
   //* Create a default window:                                         *
   //*    posRow  : offset one row from top of terminal window          *
   //*    posCol  : centered in the terminal window                     *
   //*    winHgt  : height == eight(8) rows                             *
   //*    winWid  : width  == thirty-six(36) columns                    *
   //*    All other parameters use the initialization defaults.         *
   //*                                                                  *
   //* Input  : none                                                    *
   //*                                                                  *
   //* Returns: implicitly returns a pointer to object                  *
   ACWin ( void ) ;

   //* Initialization Constructor:                                      *
   //* Four parameters are required: upper-left row, upper-left column  *
   //* window height (rows) and window width (columns).                 *
   //* Optional parameters, if not specified, are set to default values.*
   //*                                                                  *
   //* Input  : posRow : Row - position of window within terminal window*
   //*          posCol    : Column - position of window within terminal *
   //*                      window                                      *
   //*          winHeight : Height of window in rows                    *
   //*          winWidth  : Width of window in columns                  *
   //*          bdrLine   : (optional, ltSINGLE by default, member      *
   //*                      enum LineType) border style                 *
   //*          bdrAttrib : (optional, acaATTR_DFLT by default,         *
   //*                      terminal defaults) border color attributes  *
   //*                      OR'd value of acAttr bitfields              *
   //*          txtAttrib : (optional, acaATTR_DFLT by default,         *
   //*                      terminal defaults)                          *
   //*                      interior color attributes and text          *
   //*                      modifiers OR'd value of acAttr bitfields    *
   //*          winTitle  : (optional, null pointer by default)         *
   //*                      if specified, text for window title         *
   //*                      displayed in top border                     *
   //*          formDef   : (optional, null pointer by default)         *
   //*                      an 'skForm' object which specifies the      *
   //*                      position and dimensions of the input fields *
   //*                                                                  *
   //* Returns: implicitly returns a pointer to object                  *
   ACWin ( short posRow, short posCol, short winHeight, short winWidth,
           LineType bdrLine = ltSINGLE, acAttr bdrAttrib  = acaATTR_DFLT,
           acAttr txtAttrib  = acaATTR_DFLT, const char* winTitle = NULL,
           skForm *formDef = NULL ) ;

   //* Complete window setup and display it within the terminal window. *
   //*                                                                  *
   //* Input  : initText : (optional, null pointer by default)          *
   //*                     initial text to be written to window         *
   //*          initField: (optional, -1 by default) [CURRENTLY UNUSED] *
   //*                     index of target field for storing 'initText' *
   //*                     data. Ignored if 'initText' not specified.   *
   //* Returns: cursor position (offset) i.e. text insertion point      *
   WinPos OpenWindow ( const char* initText = NULL, short initField = -1 ) ;

   //* Close the window: Erase the window display data.                 *
   //*                                                                  *
   //* Note that the window object still exists in memory.              *
   //* To release the window's resources, invoke the destructor i.e.    *
   //* delete the ACWin object. Ex: ACWin *winPtr; ... delete winPtr;   *
   //*                                                                  *
   //* Input  : clear    : (optional, 'true' by default)                *
   //*                     if 'true',  clear (erase) the display area   *
   //*                     if 'false', do not clear the display area    *
   //*          fillColor: (optional, terminal dflt attr by default)    *
   //*                     color attribute used to erase data from      *
   //*                     window.                                      *
   //* Returns: nothing                                                 *
   void CloseWindow ( bool clear = true, acAttr fillAttr = acaUSEDFLT ) ;

   //* Redraw the window to display any changes that have been made,    *
   //* or to recover when other data have obscured the the window.      *
   //* For instance, call RefreshWin() after a call to HideWin().       *
   //*                                                                  *
   //* Input  : none                                                    *
   //* Returns: nothing                                                 *
   void RefreshWin ( void ) ;

   //* Temporarily erase the window from the terminal display.          *
   //* Data in the skForm object (if any) will be retained; however,    *
   //* any text previously written directly into the window interior    *
   //* will be lost.                                                    *
   //*                                                                  *
   //* Input  : fillAttr : (optional, default: terminal dflt attribute) *
   //*          color attribute used to erase data from window.         *
   //* Returns: nothing                                                 *
   void HideWin ( acAttr fillAttr = acaUSEDFLT ) ;

   //* Move the window to another position within the terminal window.  *
   //* This is simply erasing the window, updating the base position    *
   //* and redrawing the window at the new position.                    *
   //* Data in the skForm object (if any) will be redrawn as part of    *
   //* the move; however, any text previously written directly into the *
   //* window interior will be lost. (see 'newText')                    *
   //*                                                                  *
   //* Input  : newPos   : (by reference) new row/column position for   *
   //*                     window origin (upper-left corner of window)  *
   //*          newText  : (optional, null pointer by default)          *
   //*                     text to be written to interior of window,    *
   //*                     beginning at the upper left corner of the    *
   //*                     window interior.                             *
   //* Returns: 'true'  if success                                      *
   //*          'false' if specified position would place any part      *
   //*                  of the window outside the visible area of the   *
   //*                  terminal display.                               *
   bool MoveWin ( const WinPos& newPos, const char *newText = NULL ) ;

   //* Erase all text in the interior (text area) of the window.        *
   //* The text in the skForm object (if any) is not redrawn.           *
   //* See also RefreshWin().                                           *
   //*                                                                  *
   //* Input  : newAttr : (optional, acaUSEDFLT by default)             *
   //*                    new color attributes for text area of window  *
   //*                    (logical OR of forground and background color)*
   //*                    (attributes and attribute modifiers          )*
   //* Returns: Cursor position (text insertion point)                  *
   //*          This is the upper-left corner of the text area (0/0).   *
   WinPos ClearWin ( acAttr newAttr = acaUSEDFLT ) ;

   //* Erase text from the specified row of the text area.              *
   //*                                                                  *
   //* To clear a single text field within the window, use the          *
   //* ClearField() method instead.                                     *
   //*                                                                  *
   //* Input  : trgRow  : target row (0-based offset into text area)    *
   //*          txtAttr : (optional, acaUSEDFLT by default)             *
   //*                    new color attributes for text area of window  *
   //*                    (logical OR of forground and background color)*
   //*                    (attributes and attribute modifiers          )*
   //* Returns: Cursor position (text insertion point)                  *
   //*          This is the left edge of the specified row (trgRow/0).  *
   //*          Note: If 'trgRow' is out-of-range, window will not be   *
   //*                modified.                                         *
   WinPos ClearRow ( short trgRow, acAttr txtAttr = acaUSEDFLT ) ;

   //* Replace the current contents of the field with new data.         *
   //* The 'ip' (insertion point) of the field is set to zero, and the  *
   //* cursor is set to the field origin.                               *
   //*                                                                  *
   //* Input  : fIndx : index of target field                           *
   //*          gsTxt : (by referene) new text data for field           *
   //*                                                                  *
   //* Returns: 'true'  if successful                                   *
   //*          'false' if invalid field index specified                *
   bool  SetFieldText ( short fIndx, const gString& gsTxt ) ;

   //* Get a copy of the current contents of the field.                 *
   //*                                                                  *
   //* Input  : fIndx : index of target field                           *
   //*          gsTxt : (by reference) receives a copy of field text    *
   //*                                                                  *
   //* Returns: 'true'  if successful                                   *
   //*                  gsTxt contains field text data                  *
   //*          'false' if invalid field index specified                *
   //*                  gsTxt cleared to empty string                   *
   bool  GetFieldText ( short fIndx, gString& gsTxt ) const ;

   //* Erase text from the specified text field. The stored text is     *
   //* deleted, and the display area of the field is cleared.           *
   //*                                                                  *
   //* Input  : fIndx   : index of target field                         *
   //*          txtAttr : (optional, acaUSEDFLT by default)             *
   //*                    new color attributes for target field         *
   //*                    (logical OR of forground and background color)*
   //*                    (attributes and attribute modifiers          )*
   //*                                                                  *
   //* Returns: 'true'  if target field cleared                         *
   //*          'false; if fldIndex out-of-range (data not modified)    *
   bool ClearField ( short fIndx, acAttr txtAttr = acaUSEDFLT ) ;

   //* Set the insertion point (IP) for the field. This will set the    *
   //* cursor position over the character at the specified offset.      *
   //*                                                                  *
   //* If 'fIndx' references the field which has input focus, the v     *
   //* isible cursor will also be updated.                              *
   //*                                                                  *
   //* Special case: If 'newIp' is greater that the number of characters*
   //* in the field, IP will be set to end-of-text (over the null       *
   //* terminator).                                                     *
   //*                                                                  *
   //* Input  : fIndx   : index of target field                         *
   //*          newIp   : character offset into the text of the field   *
   //*                    ZERO <= newIp <= text characters              *
   //*                    ZERO == field origin and any value at or      *
   //*                    beyond the number of characters in the array  *
   //*                    signals that IP should be set to end-of-text. *
   //*                    (A safe end-of-text value is: gsMAXCHARS.)    *
   //*                                                                  *
   //* Returns: 'true'  if successful                                   *
   //*          'false' if invalid field index specified                *
   bool SetFieldIP ( short fIndx, short newIp ) ;

   //* Refresh (erase and redraw) the contents of the specified field,  *
   //* or optionally, refresh all fields defined by the skForm object.  *
   //* Input focus remains on the currently active field, and cursor    *
   //* is returned to insertion point within that field.                *
   //*                                                                  *
   //* Input  : fIndx: index of target text field                       *
   //*                 If fldIndex >= total of fields defined (fCnt),   *
   //*                 all fields will be refreshed.                    *
   //*                                                                  *
   //* Returns: nothing                                                 *
   void RefreshField ( short fIndx ) ;

   //* Refresh (erase and redraw) the contents of the field which has   *
   //* input focus.                                                     *
   //*                                                                  *
   //* Input  : none                                                    *
   //*                                                                  *
   //* Returns: nothing                                                 *
   void RefreshField ( void ) ;

   //* Set the color attributes and modifiers for the specified text    *
   //* field. Refresh the field display using the modified attributes.  *
   //*                                                                  *
   //* On return, the cursor is positioned in field with input focus.   *
   //*                                                                  *
   //* Input  : fIndx   : index of target field within the skForm       *
   //*                    object                                        *
   //*          txtAttr : (acAttr bitfield) a logical OR of forground   *
   //*                    and background color attributes and attribute *
   //*                    modifiers                                     *
   //*                                                                  *
   //* Returns: 'true'  if attribute set successfully                   *
   //*          'false' if invalid field index specified                *
   bool SetFieldAttr ( short fldIndx, acAttr txtAttr ) ;

   //* Set input focus to the specified field, and refresh field        *
   //* contents. The cursor is placed on the text insertion point.      *
   //* Note: Read-only fields cannot receive the input focus.           *
   //*                                                                  *
   //* Input  : fldIndex: index of target text field                    *
   //*                                                                  *
   //* Returns: index of field with focus                               *
   short SetInputFocus ( short fldIndex ) ;

   //* Set input focus to the next field, and refresh field contents.   *
   //* If already on the last field (index==fCnt-1), wrap around to     *
   //* the first field of the array.                                    *
   //* The cursor is placed on the text insertion point.                *
   //* Note: Read-only fields cannot receive the input focus.           *
   //*                                                                  *
   //* Input  : none                                                    *
   //*                                                                  *
   //* Returns: index of field with focus                               *
   short Focus2NextField ( void ) ;

   //* Set input focus to the previous field, and refresh field         *
   //* contents.                                                        *
   //* If already on the first field (index==zero), wrap around to      *
   //* the last field of the array.                                     *
   //* The cursor is placed on the text insertion point.                *
   //* Note: Read-only fields cannot receive the input focus.           *
   //*                                                                  *
   //* Input  : none                                                    *
   //*                                                                  *
   //* Returns: index of field with focus                               *
   short Focus2PrevField ( void ) ;

   //* Enable or disable user access to the Insert/Overstrike input     *
   //* mode when user is editing a field, and specify the initial mode. *
   //*                                                                  *
   //* Insert/Overstrike toggle is active by default, and the initial   *
   //* mode is Insert Mode. For some fields, or some kinds of data,     *
   //* however, it is useful to set one mode or the other and lock out  *
   //* user access.                                                     *
   //*                                                                  *
   //* Optionally enable or disable cursor style change when            *
   //* Insert/Overstrike mode is toggled. This provides a visual cue    *
   //* to the user whether Insert mode or Overstrike mode is active.    *
   //* Optionally, the cursor style for each mode may be specified as   *
   //* a member of enum CurStyle (excluding csShow and csHide).         *
   //*                                                                  *
   //* Input  : enable  : 'true'  == enable user access to Insert key   *
   //*                               (this is the default)              *
   //*                    'false' == disable user access to Insert key  *
   //*                               (the mode is locked by application)*
   //*          initMode: initial input mode                            *
   //*                    'true'  == Insert Mode                        *
   //*                    'false' == Overstrike Mode                    *
   //*          styles  : (optional 'false' by default)                 *
   //*                    Enable or disable automatic change of cursor  *
   //*                    style when Insert/Overstrike is toggled.      *
   //*                    'true'  == enable cursor style change         *
   //*                    'false' == disable cursor style change        *
   //*          insStyle: (optional, istyleDFLT by default)             *
   //*                     cursor style for insert mode                 *
   //*                    referenced only when the 'styles' flag is set *
   //*          ovrStyle: (optional, ostyleDFLT by default)             *
   //*                     cursor style for overstrike mode             *
   //*                    referenced only when the 'styles' flag is set *
   //*                                                                  *
   //* Returns: 'true'  if Ins/Ovr toggle enabled, else 'false'         *
   bool InsertKey ( bool enable, bool initMode, bool styles = false, 
                    CurStyle insStyle = istyleDFLT, CurStyle ovrStyle = ostyleDFLT ) ;

   //* Interactively edit the fields defined within the skForm object.  *
   //*                                                                  *
   //* Input  : exitChar : when this character is received, return to   *
   //*                     caller                                       *
   //*          focusFld : (by reference) receives the index of the     *
   //*                     field with input focus on return.            *
   //*                                                                  *
   //* Returns: keycode of last key pressed                             *
   wchar_t EditForm ( wchar_t exitChar, short& focusFld ) ;

   //* Capture user input to the specified input field.                 *
   //*                                                                  *
   //* Input  : fIndx   : (by reference) index of target field          *
   //*                    On entry, specifies the field to be edited.   *
   //*                    On return, index of field with input focus.   *
   //*          exitChar: (optional, NEWLINE '\n' by default)           *
   //*                    This character terminates the input loop.     *
   //*          allChar : (optional, 'false' by default)                *
   //*                    if 'false', 'exitChar' controls the loop      *
   //*                    if 'true',  return to caller after each       *
   //*                                iteration of the loop             *
   //* Returns: last keycode captured                                   *
   wchar_t EditField ( short& fIndx, wchar_t exitChar = L'\n', bool allChar = false ) ;

   //* Set the window title.                                            *
   //* Draw the title centered in the top border of the window.         *
   //*                                                                  *
   //* Input  : newTitle : text of window title                         *
   //*                     (empty string or null pointer erases title)  *
   //*          titAttr  : (optional, existing attributes by default)   *
   //*                     If specified, this is a logical OR of acAttr *
   //*                     values specifying the foreground/background  *
   //*                     color attributes.                            *
   //*                     See the 'acAttr' enumerated type for details.*
   //*          bdrLine  : (optional, ltHORIZ i.e. invalid line type    *
   //*                      by default) If specified, new border style: *
   //*                     ltSINGLE or ltDUAL                           *
   //*                                                                  *
   //* Returns: 'true'  if success                                      *
   //*          'false' if title truncated to fit                       *
   bool SetTitle ( const char* newTitle, acAttr titAttr = acaUSEDFLT, 
                   LineType bdrLine = ltHORIZ ) ;

   //* Write text to the interior of the window.                        *
   //* This group of methods writes directly into the window's text     *
   //* area, bypassing any defined text fields.                         *
   //*                                                                  *
   //* Input  : txtPos : WinPos-class object indicates the zero-based   *
   //*                  OFFSET from upper-left corner of window interior*
   //*          txt    : text to write in one of the following formats: *
   //*                   -- gString object by reference                 *
   //*                   -- wchar_t pointer                             *
   //*                   -- char pointer                                *
   //*          txtAttr: (optional, existing text attributes by default)*
   //*                   text attributes and text modifiers             *
   //*          clear  : (optional, 'false' by default)                 *
   //*                   if 'false', existing displayed text unmodified *
   //*                   if 'true',  erase window interior before write *
   //*                                                                  *
   //* Returns: current cursor position (window offset)                 *
   WinPos Write ( WinPos txtPos, const gString& txtData, 
                  acAttr txtAttr = acaUSEDFLT, bool clear = false ) ;
   WinPos Write ( WinPos txtPos, const wchar_t* txtData, 
                  acAttr txtAttr = acaUSEDFLT, bool clear = false ) ;
   WinPos Write ( WinPos txtPos, const char* txtData, 
                  acAttr txtAttr = acaUSEDFLT, bool clear = false ) ;

   //* Get the current cursor offset within the window's text area.     *
   //*                                                                  *
   //* Input  : none                                                    *
   //* Returns: current cursor position (window offset)                 *
   WinPos GetCursorPos ( void ) ;

   //* Set the cursor offset within the window's text area.             *
   //*                                                                  *
   //*          ** Cursor position may be specified in either of 2 ways:*
   //* Input  : wPos : (by reference) offset from upper-left corner of  *
   //*                 window's text area (0-based offset)              *
   //*               OR                                                 *
   //*          rOff : row offset                                       *
   //*          cOff : column offset                                    *
   //*                                                                  *
   //* Returns: current cursor position (window offset)                 *
   WinPos SetCursorPos ( const WinPos& wPos ) ;
   WinPos SetCursorPos ( short rOff, short cOff ) ;

   //* Write text to the specified field within the window.             *
   //*                                                                  *
   //* This group of methods inserts the specified text into the        *
   //* existing text of the target field at the specified offset ('ip');*
   //* or if the 'clear' flag is set, replaces the existing text.       *
   //* The field is then redrawn to display the updated data.           *
   //*                                                                  *
   //* Input  : fIndx  : index of target field                          *
   //*          txt    : text to write in one of the following formats: *
   //*                   -- gString object by reference                 *
   //*                   -- wchar_t pointer                             *
   //*                   -- char pointer                                *
   //*          offset : (optional, -1 by default, indicationg that the *
   //*                    current insertion point ('ip') should be used)*
   //*                   character offset into existing text at which   *
   //*                   to insert the new text                         *
   //*          clear  : (optional, 'false' by default)                 *
   //*                   if 'false', new text will be added to the      *
   //*                               existing text as described         *
   //*                   if 'true',  new text will replace the existing *
   //*                               text                               *
   //*          txtAttr: (optional, acaUSEDFLT by default)              *
   //*                   (existing text attributes will be used)        *
   //*                   If specified, color attributes and text mods   *
   //*                                                                  *
   //* Returns: current insertion point ('ip'), that is, the character  *
   //*          offset at which the cursor is positioned. This will be  *
   //*          at the point following the inserted text, or at the     *
   //*          last character cell of the field.                       *
   //*          (or zero if invalid field index specified)              *
   short Write ( short fIndx, const gString& txtData, short offset = -1, 
                 bool clear = false, acAttr txtAttr = acaUSEDFLT ) ;
   short Write ( short fIndx, const wchar_t* txtData, short offset = -1, 
                 bool clear = false, acAttr txtAttr = acaUSEDFLT ) ;
   short Write ( short fIndx, const char* txtData, short offset = -1, 
                 bool clear = false, acAttr txtAttr = acaUSEDFLT ) ;

   //* Set the color attributes and modifiers for the window text area. *
   //*                                                                  *
   //* Input  : txtAttr : (acAttr bitfield) a logical OR of forground   *
   //*                    and background color attributes and attribute *
   //*                    modifiers                                     *
   //* Returns: nothing                                                 *
   void SetTextAttr ( acAttr txtAttr ) ;

   //* Set the color attributes and modifiers for the window border.    *
   //*                                                                  *
   //* Input  : bdrAttr : (acAttr bitfield) a logical OR of forground   *
   //*                    and background color attributes and attribute *
   //*                    modifiers                                     *
   //*          bdrStyle: (optional, ltHORIZ i.e. an invalid value by   *
   //*                    default) specify the border line style        *
   //*                    either ltSINGLE or ltDUAL                     *
   //* Returns: nothing                                                 *
   void SetBorderAttr ( acAttr bdrAttr, LineType bdrStyle = ltHORIZ ) ;

   //* Draw a box within the window's text area.                        *
   //*                                                                  *
   //* Input  : pos   : offset from upper-left corner of text area to   *
   //*                  to upper-left corner of box (0-based)           *
   //*          height: number of rows for box                          *
   //*          width : number of columns for box                       *
   //*          lType : (optional, ltSINGLE by default) line style      *
   //*                  member of enum LineType: (ltSINGLE or ltDUAL)   *
   //*          bxAttr: (optional, acaUSEDFLT by default)               *
   //*                  If specified, these are box color attributes.   *
   //*                  If not specified, existing text attributes used.*
   //* Returns: 'true'  if box fits within the window text area         *
   //*          'false' if invalid offset or dimensions specified       *
   bool DrawBox ( WinPos wpOrig, short height, short width, 
                  LineType lType = ltSINGLE, acAttr bxAttr = acaUSEDFLT ) ;

   //* Draw a horizontal or vertical line in the window.                *
   //*                                                                  *
   //* Note: If the line intersects the window border, the endpoint     *
   //*       character will be automatically selected to match the      *
   //*       intersection of the drawn line and the border line style.  *
   //*                                                                  *
   //* Input  : pos     : offset from upper-left corner of window       *
   //*                    (0-based)                                     *
   //*          length  : length of the line: number of number of rows  *
   //*                    (vertical), or number of columns (horizontal).*
   //*          vertical: 'true'  : vertical line                       *
   //*                    'false' : horizontal line                     *
   //*          lType   : (optional, ltSINGLE by default) line style    *
   //*                     member of enum LineType: ltSINGLE or ltDUAL  *
   //*          begChar : (optional, null character by default)         *
   //*                     specify the beginning endpoint character     *
   //*          endChar : (optional, null character by default)         *
   //*                     specify the final endpoint character         *
   //*          lnAttr  : (optional, acaUSEDFLT by default)             *
   //*                    If specified, color attributes to be used.   *
   //*                    If not specified, stored text attributes used.*
   //* Returns: cursor position at the end of the line:                 *
   //*          For horizontal lines, the column after last character.  *
   //*          For vertical lines, the column below the last character.*
   WinPos DrawLine ( const WinPos& wpOrig, short length, bool vertical, 
                     LineType lType = ltSINGLE, 
                     wchar_t begChar = L'\0', wchar_t endChar = L'\0', 
                     acAttr lnAttr = acaUSEDFLT ) ;

   //* Draw an outline around the outside of the specified field within *
   //* the window border.                                               *
   //* The target field must have at least one space open on all sides. *
   //* That is, it must not be positioned against the window border,    *
   //* and must not be adjacent to any other field.                     *
   //*                                                                  *
   //* Input  : fIndx   : index of target field                         *
   //*                    if 'fIndx' == number of fields defined, then  *
   //*                    draw an outline around each field.            *
   //*          lType   : (optional, ltSINGLE by default) line style    *
   //*                     member of enum LineType: ltSINGLE or ltDUAL  *
   //*          lnAttr  : (optional, acaUSEDFLT by default)             *
   //*                    If specified, color attributes to be used.    *
   //*                    If not specified, stored text attributes used.*
   //*                                                                  *
   //* Returns: 'true'  if successful                                   *
   //*          'false' if invalid field index or insufficient space    *
   //*                  around the field                                *
   bool FieldOutline ( short fIndx, LineType lType = ltSINGLE, 
                       acAttr lnAttr = acaUSEDFLT ) ;

   #if DEBUG_ANSICMD != 0     // For Debugging Only
   //* Debugging method: Get a copy of the window attributes.           *
   //* Input  : txtExp : receives decoded window interior attributes    *
   //*          bdrExp : receives decoded window border attributes      *
   //* Returns: nothing                                                 *
   void GetWindowAttributes ( acaExpand& txtExp, acaExpand& bdrExp )
   { txtExp = this->txtAttr ; bdrExp = this->bdrAttr ; }
   //* Debugging method: Get window dimensions and position.            *
   WinPos GetWinConfig ( short& height, short& width )
   { height = this->height ; width = this->width ; return ( this->winBase ) ; }
   //* Debugging method: Get a pointer to the window's skForm object.   *
   //* Note: This is a 'const' pointer. Contents may not be modified.   *
   //* Input  : nothing                                                 *
   //* Returns: pointer to the window's skForm object.                  *
   const skForm* GetForm ( void )
   { return ( this->skf ) ; }
   #endif   // DEBUG_ANSICMD


   //****************************
   //* Private methods and data *
   //****************************
   private:
   //* Initialize the window data members.*
   void initialize ( short posRow, short posCol, short winHeight, short winWidth ) ;
   //* Initialize the skForm data member *
   void init_skForm ( const skForm *formDef = NULL ) ;
   //* Range-check the window fields' data, and adjust if necessary.*
   bool init_skFields ( void ) ;
   //* Scan the text data, and if necessary, reformat it to fit the field.*
   void formatFieldText ( skField *fptr ) ;
   //* Clear (erase) the area defined for the window.*
   void clearWin ( const acaExpand& acaEx, bool all = false ) ;
   //* Draw the window border.*
   void drawBorder ( acAttr titAttr = acaUSEDFLT ) ;
   //* Set the color attributes and modifiers to stored terminal-window values.*
   void restoreAttributes ( void ) ;
   //* Determine whether the specified keycode is one of the "special" keys.*
   bool specialkeyTest ( wchar_t& wkey ) const ;
   //* Determine whether 'special' keycode requires conversion.*
   wchar_t specialkeyConvert ( wchar_t wkey ) ;
   //* Execute the operation associated with the specified "special" key.*
   void specialkeyProc ( wchar_t wkey ) ;
   //* Calculate the 'ip' and 'cur' values for the vertical cursor movement.*
   void skpInterRow ( skField *fptr, wchar_t wkey ) ;
   //* Erase the character reference by 'ip'.*
   void skpDeleteChar ( skField *fptr ) ;
   //* Add a normal (printing, "graphing") character to the target field.*
   void normalkeyProc ( wchar_t wkey ) ;

   //**************************
   //* Protected Data Members *
   //**************************
   protected:
   WinPos winBase ;              //* Terminal window offset (1-based)
   WinPos txtBase ;              //* Upper left corner of interior (text area) (1-based)
   WinPos txtOff ;               //* Current cursor position (0-based offset)
   short height ;                //* Height of window object (rows)
   short width ;                 //* Width of window object (columns)
   short txtRows ;               //* Number of text (interior) rows
   short txtCols ;               //* Number of text (interior) columns
   LineType bdrStyle ;           //* Line type for border
   acaExpand bdrAttr ;           //* Color attributes for border
   acaExpand txtAttr ;           //* Color attributes for interior (text & background)
   acaExpand termAttr ;          //* Color attributes for parent terminal window
   wchar_t title[gsMAXCHARS] ;   //* Title to be displayed in top row of object
   skForm *skf ;                 //* Definition of text fields within the window
   // Doubly-linked list of ACWin objects. Not yet implemented.
   ACWin *prevWin ;              //* Pointer to "Previous" window (or null pointer)
   ACWin *nextWin ;              //* Pointer to "Next" window (or null pointer)

} ;   // ACWin class

