//********************************************************************************
//* File       : AnsiCmdWin.cpp                                                  *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2022-2023 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 20-Aug-2023                                                     *
//* Version    : (see AnsiCmdVersion string in AnsiCmd.cpp)                      *
//*                                                                              *
//* Description: ACWin class implementation. ACWin is derived from the           *
//* AnsiCmd class and defines a high-level "window" object.                      *
//*                                                                              *
//*               ┌───────┤ ACWin-class Window ├───────┐                         *
//*               │                                    │                         *
//*               │            Hello World!            │                         *
//*               │                                    │                         *
//*               │                                    │                         *
//*               └────────────────────────────────────┘                         *
//*                                                                              *
//* This window is not as flexible as an ncurses window, but it is implemented   *
//* in about 4,000 lines of code, while ncurses is over 200,000 lines of code.   *
//* See the NcDialog API library for the the author's full, robust, all          *
//* bells-and-whistles implementation of ncurses support.                        *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************
#include "AnsiCmd.hpp"     //* AnsiCmd-class definition
#include "AnsiCmdWin.hpp"  //* AnsiCmd-class window-object definitions

//***************
//* Definitions *
//***************
#if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
extern ofstream ofsdbg ;
extern gString  gsdbg ;
#endif   // DEBUG_ANSICMD && DEBUG_LOG


//**************
//* Local data *
//**************
extern AnsiCmd *acObjPtr ;    // non-member variable (see AnsiCmd.cpp)


//********************
//* Local prototypes *
//********************


//*************************
//*        ~ACWin         *
//*************************
//********************************************************************************
//* Destructor: Return all allocated resources to the system before exit.        *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

ACWin::~ACWin ( void )
{
   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
   if ( (ofsdbg.is_open()) )
   { ofsdbg << "~ACWin()" << endl ; }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG

   //* If 'insc' is set, return to default cursor style.*
   if ( this->skf->insc )
      this->acSetCursorStyle ( csDflt ) ;

   if ( this->skf != NULL )
   { delete ( this->skf ) ; this->skf = NULL ; }

   //* Update the linked list of window objects.*
   // NOT YET IMPLEMENTED

}  //* End ~ACWin *

//*************************
//*        ACWin          *
//*************************
//********************************************************************************
//* Default Constructor:                                                         *
//* Create a default window:                                                     *
//*    posRow  : offset one row from top of terminal window                      *
//*    posCol  : centered in the terminal window                                 *
//*    winHgt  : height == eight(8) rows                                         *
//*    winWid  : width  == thirty-six(36) columns                                *
//*    All other parameters use the initialization defaults.                     *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: implicitly returns a pointer to object                              *
//********************************************************************************
//* Note on visibility of inherited elements for a derived class.                *
//* This is related to the methods and data of the parent class (AnsiCmd)        *
//* which are accessible to the ACWin derived class.                             *
//*                                                                              *
//* Derived Classes:                                                             *
//* Inheritance Type  Base Class    Derived Class                                *
//* ----------------  -----------   --------------                               *
//*   public           public        public                                      *
//*                    protected     protected                                   *
//*                    private       no direct access                            *
//*   protected        public        protected                                   *
//*                    protected     protected                                   *
//*                    private       no direct access                            *
//*   private          public        private                                     *
//*                    protected     private                                     *
//*                    private       no direct access                            *
//*                                                                              *
//********************************************************************************

ACWin::ACWin ( void )
{
   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
   if ( (ofsdbg.is_open()) )
   { ofsdbg << "ACWin(dflt)" << endl ; }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG

   const short posRow = 2 ;
   const short posCol = 1 ;
   const short winHgt = 8 ;
   const short winWid = 36 ;

   //* Initialize the terminal setup data and allocate text storage.*
   this->initialize ( posRow, posCol, winHgt, winWid ) ;

   //* Adjust horizontal position of window.*
   short hOffset = this->termCols / 2 - winWid / 2 ;
   this->winBase.col = hOffset ;
   this->txtBase.col = this->winBase.col + 1 ;

   //* Create the default set of input fields.*
   this->init_skForm () ;
   
}  //* End ACWin() *

//*************************
//*        ACWin          *
//*************************
//********************************************************************************
//* Initialization Constructor:                                                  *
//* Four parameters are required: upper-left row, upper-left column window       *
//* height (rows) and window width (columns).                                    *
//* Other parameters, if not specified, are set to default values.               *
//*                                                                              *
//* Input  : posRow    : Row    - position of window within the terminal window  *
//*          posCol    : Column - position of window within the terminal window  *
//*          winHeight : Height of window in rows                                *
//*          winWidth  : Width of window in columns                              *
//*          bdrLine   : (optional, member enum LineType, ltSINGLE by default)   *
//*                      border style i.e. type of line for border               *
//*                      Note: specifying either ltHORIZ OR ltVERT will cause    *
//*                      the border to be drawn with spaces only (no line).      *
//*          bdrAttrib : (optional, acaATTR_DFLT by default, terminal defaults)  *
//*                      border color attributes OR'd value of acAttr bitfields  *
//*          txtAttrib : (optional, acaATTR_DFLT by default, terminal defaults)  *
//*                      interior color attributes and text modifiers            *
//*                      OR'd value of acAttr bitfields                          *
//*          winTitle  : (optional, null pointer by default)                     *
//*                      if specified, text for window title displayed in        *
//*                      top border                                              *
//*          formDef   : (optional, null pointer by default)                     *
//*                      an 'skForm' object which specifies the                  *
//*                      position and dimensions of the input fields             *
//*                      (see example below)                                     *
//*                                                                              *
//* Returns: implicitly returns a pointer to object                              *
//********************************************************************************
//* Sample initialization of an skForm object:                                   *
//* ------------------------------------------                                   *
//* Some developers will want to write directly into the window, while others    *
//* will want to design an array of text-input fields for user interactions      *
//* which are defined by some-kind-of-form i.e. the skForm class.                *
//*                                                                              *
//* If the ACWin constructor is called without specifying an skForm object, a    *
//* default skForm object is created based on the size of the window. If the     *
//* window is small (less than 1,000 character cells), then the skForm object    *
//* is created with one(1) skField. Else, the skForm will generate an array of   *
//* skField objects, one per row of the window interior.                         *
//*                                                                              *
//* To achieve control over the way the skForm object is configured, a pointer   *
//* to an skForm template object ('formDef')is passed to the ACWin constructor.  *
//* The position and dimensions of each field must fall entirely within the      *
//* text area of the window, and the fields may not overlap. Although range      *
//* checking and automatic adjustments of these fields is performed, an          *
//* over-caffinated developer could still cause chaos, so it is important to     *
//* carefully verify the parameters provided.                                    *
//*                                                                              *
//* There are several ACWin objects created in the test code included with this  *
//* library. Some tests write text directly into the window, while others write  *
//* into the defined fields (or some combination of the two). Please refer to    *
//* those examples, as well as the discussion of skForm and skField objects in   *
//* the documentation for this package.                                          *
//*                                                                              *
//* To create an skForm template, allocate a new object specifying the number    *
//* of fields and the row/colum offset from the upper left corner of the window  *
//* interior to the upper left corner of the target area for the form.           *
//* Here we define an skForm object with two(2) fields with the form origin at   *
//* offset 0,0. This is the upper left corner of the window interior space.      *
//*                                                                              *
//*   skForm *skfPtr = new skForm( 2, 0, 0 ) ;                                   *
//*                                                                              *
//*               ┌───────┤ ACWin-class Window ├───────┐                         *
//*               │                                    │                         *
//*               │ ██████████████████████████████████ │                         *
//*               │                                    │                         *
//*               │ ██████████████████████████████████ │                         *
//*               │                                    │                         *
//*               └────────────────────────────────────┘                         *
//*                                                                              *
//* Initialize the FORM-LEVEL members. Note that 'fCnt' is initialized during    *
//* instantiation, and must not be changed. Also, the 'base' value (offset) is   *
//* overridden by the ACWin initialization.                                      *
//*                                                                              *
//*   skfPtr->ins  = false ;      // insert/overstrike flag                      *
//*   skfPtr->togg = true ;       // insert/overstrike lock                      *
//*   skfPtr->beep = true ;       // make noise if invalid user input            *
//*   skfPtr->fi   = ZERO ;       // field index                                 *
//*                                                                              *
//* Initialize the FIELD-LEVEL members:                                          *
//* WinPos wpul(1,1) ;                                                           *
//* short f = ZERO ;                                                             *
//* skfPtr->fld[f].orig = wpul ;  // origin (row/column offset) for field        *
//* skfPtr->fld[f].hgt = 1 ;      // field height                                *
//* skfPtr->fld[f].wid = winWIDTH - 4 ; // width of field                        *
//* skfPtr->fld[f].ip = ZERO ;    // text insertion point (offset into field)    *
//* skfPtr->fld[f].txt = "Some text for field 0." ;                              *
//*                                                                              *
//* wpul.row += 2 ;               // position of next field==two rows below      *
//* ++f ;                                                                        *
//*                                                                              *
//* skfPtr->fld[f].orig = wpul ;                                                 *
//* skfPtr->fld[f].hgt = 1 ;                                                     *
//* skfPtr->fld[f].wid = winWIDTH - 4 ;                                          *
//* skfPtr->fld[f].ip = 5 ;                                                      *
//* skfPtr->fld[f].txt = "Some text for field 1." ;                              *
//*                                                                              *
//* -- Note that the 'cur' (physical cursor position) is dynamically calculated  *
//*    from the 'ip' (insertion point).                                          *
//* -- The 'aca' (color attributes) member is the logical OR of enum acAttr      *
//*    values. If not specified, the terminal default attributes are used.       *
//* -- The 'ro' (read-only) flag retains its default value (false).              *
//* -- The 'off' (text positioning) member is defined for future development     *
//*    and is currently ignored.                                                 *
//*                                                                              *
//* The initialized skForm template is then passed as a parameter for the        *
//* ACWin constructor.                                                           *
//*                                                                              *
//* - - - - -                                                                    *
//* Programmer's Note: The AnsiCmd constructor is called as part of the ACWin    *
//* constructor. It is important to monitor the interaction between the two      *
//* constructors.                                                                *
//*                                                                              *
//********************************************************************************

ACWin::ACWin ( short posRow, short posCol, short winHeight, short winWidth,
               LineType bdrLine, acAttr bdrAttrib, acAttr txtAttrib, 
               const char* winTitle, skForm *formDef )
{
   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
   if ( (ofsdbg.is_open()) )
   { ofsdbg << "ACWin(init)" << endl ; }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG

   //* Initialize the terminal setup data and allocate text storage.*
   this->initialize ( posRow, posCol, winHeight, winWidth ) ;

   //* Note: ltHORIZ or ltVERT specifies no line drawn in border.*
   if ( bdrLine == ltVERT )
      bdrLine = ltHORIZ ;
   this->bdrStyle = bdrLine ;

   //* For borders, some text-modifier attributes     *
   //* are meaningless or just plain ugly. Clear them.*
   bdrAttrib = acAttr(bdrAttrib & 
                      ~(acaITALIC | acaUNDERLINE | acaOVERLINE | 
                        acaXOUT | acaCONCEAL)) ;

   //* Decode the border and interior attributes.*
   this->bdrAttr.decode( bdrAttrib ) ;
   this->txtAttr.decode( txtAttrib ) ;

   //* If window title specified, save title text.*
   if ( winTitle != NULL )
   {
      gString gs( winTitle ) ;
      gs.limitCols( this->width - 4 ) ;      // if necessary, truncate to fit
      gs.copy( this->title, gsMAXCHARS ) ;
   }

   //* If an skForm object was specified, use it as a template.  *
   //* Else, (formDef==NULL), create default set of input fields.*
   this->init_skForm ( formDef ) ;

}  //* End ACWin() *

//*************************
//*      initialize       *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Called during instantiation to initialize the data members, terminal         *
//* environment and text-management structure.                                   *
//*                                                                              *
//* Instead of setting the terminal environment, directly, request a copy of     *
//* the settings from the existing AnsiCmd, parent-class object and use them     *
//* to initialize our member variables.                                          *
//*                                                                              *
//* Input  : posRow   : Row    - position of window within the terminal window   *
//*          posCol   : Column - position of window within the terminal window   *
//*          winHeight: Height of window in rows                                 *
//*          winWidth : Width of window in columns                               *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::initialize ( short posRow, short posCol, short winHeight, short winWidth )
{
   const short minDIM = 3 ;   // a window requires at least three rows/columns

   this->winBase  = { 1, 1 } ;         // set all data members to defaults
   this->txtBase  = { 2, 2 } ;
   this->txtOff   = { 0, 0 } ;
   this->height   = 3 ;
   this->width    = 3 ;
   this->txtRows  = 1 ;
   this->txtCols  = 1 ;
   this->ioLocale = NULL ;
   this->bdrStyle = ltSINGLE ;
   *this->title   = NULLCHAR ;
   this->prevWin  = NULL ;
   this->nextWin  = NULL ;
   this->bdrAttr.reset() ;
   this->txtAttr.reset() ;

   this->tset = new TermSet ;          // allocate and initialize TermSet object

   this->ioLocale = new locale("") ;   // get the terminal default locale

   //* Get a copy of the data members from the parent AnsiCmd object.*
   acObjPtr->getDataMembers ( &this->termAttr,
                              this->ioLocale,
                              this->termRows,
                              this->termCols,
                              this->fontNum,
                              this->fullReset,
                              this->wideStream
                            ) ;

   //* Range check: window dimensions >= minimum
   if ( winHeight < minDIM )           winHeight = minDIM ;
   if ( winWidth < minDIM )            winWidth = minDIM ;
   //* Range check: window dimensions <= terminal dimensions.*
   if ( winHeight > this->termRows )   winHeight = this->termRows ;
   if ( winWidth > this->termCols )    winWidth = this->termCols ;
   //* Range check: window origin is within terminal window.*
   if ( posRow <= ZERO )               posRow = 1 ;
   if ( posCol <= ZERO )               posCol = 1 ;
   //* Range check: window does not extend beyond terminal window.*
   if ( (posRow + winHeight - 1) > this->termRows )
      posRow = this->termRows - winHeight + 1 ;
   if ( (posCol + winWidth - 1) > this->termCols )
      posCol = this->termCols - winWidth + 1 ;

   //* Window position and dimensions now within the terminal window. (1-based) *
   this->winBase = { posRow, posCol } ;   // window position
   this->height = winHeight ;             // window height (incl. border)
   this->width  = winWidth ;              // window width (incl. border)
   this->txtRows = this->height - 2 ;     // text (interior) rows
   this->txtCols = this->width - 2 ;      // text (interior) columns

   //* Upper left corner of window interior i.e. text area (1-based) *
   this->txtBase = { short(this->winBase.row + 1), short(this->winBase.col + 1) } ;
   this->txtOff = { 0, 0 } ;     // OFFSET from upper-left corner of interior

}  //* End initialize() *

//*************************
//*      init_skForm      *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Initialize the 'skForm' data member. (see notes below)                       *
//*                                                                              *
//* Input  : (optional, null pointer by default)                                 *
//*          If specified, initialize the custom skForm object.                  *
//*          Else create the default skForm object as described below.           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes On Default Forms:                                                      *
//* -- If a custom form is specified, it is copied to the internal form object.  *
//*    the data are range checked and needed adjustments are applied.            *
//* -- If no custom form specified:                                              *
//*    -- If the window dimensions allow, define a single text field using       *
//*       the position and dimensions of the window interior.                    *
//*    -- Otherwise, define a field for each text row of the window.             *
//*    -- The text for each defined field is initially an empty string.          *
//*    -- The 'ip' and 'cur' values for each field are set to zero i.e. they     *
//*       reference the field origin by default.                                 *
//*    -- The 'ins', 'insc', 'togg' and 'beep' flags and the 'icur'/'ocur' values*
//*       of the skForm object use skForm defaults.                              *
//*                                                                              *
//* The maximum size of a text field is determined by the storage capacity of    *
//* the gString object for the field (gsMAXCHARS), and is defined as the         *
//* number of characters corresponding to the number of physical row/column      *
//* cells in the window PLUS a newline character for each row of the field,      *
//* PLUS the null terminator. This maximum will come into play only for very     *
//* large, multi-row fields.                                                     *
//********************************************************************************

void ACWin::init_skForm ( const skForm *formDef )
{
   #define DEBUG_skForm (0)      // for debugging only, report contents of 'skf'

   //* Maximum character cells supported by a single field. (see note above) *
   const short FIELD_MAX = gsMAXCHARS - this->txtRows - 1 ;

   if ( this->skf != NULL )            // prevent memory leaks
      delete this->skf ;

   if ( formDef != NULL )              // custom form definition
   {
      this->skf = new skForm( formDef->fCnt, this->txtBase.row, this->txtBase.col ) ;
      *this->skf = formDef ;
      //* Note the odd syntax needed to make the "operator=" work *

      //* Set terminal-relative 'base' position.   *
      //* Note: This overrides the caller's value. *
      this->skf->base = this->txtBase ;

      //* Range-check the window fields' data.*
      this->init_skFields () ;
   }
   else                                // default form definition
   {
      WinPos wp( 0, 0 ) ;              // offset into the text area
      short fieldCnt = 1,              // number of fields defined
            height   = this->txtRows ; // height (rows) for each field

      if ( (this->txtRows * this->txtCols) >= FIELD_MAX )
      { fieldCnt = this->txtRows ; height = 1 ; }

      this->skf = new skForm ( fieldCnt, this->txtBase.row, this->txtBase.col ) ;

      //* Initialize the object's data members.*
      for ( short f = ZERO ; f < fieldCnt ; ++f, ++wp.row )
      {
         this->skf->fld[f].orig = wp ;
         this->skf->fld[f].hgt  = height ;
         this->skf->fld[f].wid  = this->txtCols ;
      }
      //* Note: Although we "know" that the parameters are in range, we *
      //* call init_skFields() here to initialize the color attributes. *
      this->init_skFields () ;
   }

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_skForm != 0
   if ( (ofsdbg.is_open()) )
   {
      if ( this->skf != NULL )
      {
         gsdbg.compose( "  skf: fCnt:%hd fi:%hd base:%hd/%hd ins:%hhd insc:%hhd "
                        "togg:%hhd icur:%hd, ocur:%hd beep:%hhd ",
                        &this->skf->fCnt, &this->skf->fi,
                        &this->skf->base.row, &this->skf->base.col,
                        &this->skf->ins, &this->skf->insc, &this->skf->togg, 
                        &this->skf->icur, &this->skf->ocur, &this->skf->beep ) ;
         ofsdbg << gsdbg.ustr() << endl ;
   
         for ( short f = ZERO ; f < this->skf->fCnt ; ++f )
         {
            gsdbg.compose( "  %hd) orig:%-2hd/%-2hd hgt:%-2hd wid:%-2hd ip:%-2hd\n"
                           "   cur :%-2hd/%-2hd aca:%04Xh ro:%hhd\n"
                           "   txt:",
                           &f, &this->skf->fld[f].orig.row, 
                           &this->skf->fld[f].orig.col,
                           &this->skf->fld[f].hgt,
                           &this->skf->fld[f].wid,
                           &this->skf->fld[f].ip,
                           &this->skf->fld[f].cur.row,
                           &this->skf->fld[f].cur.col, 
                           &this->skf->fld[f].aca.acaVal,
                           &this->skf->fld[f].ro ) ;

            if ( (this->skf->fld[f].txt.gschars()) > 1 )
               gsdbg.append( "'%S'", this->skf->fld[f].txt.gstr() ) ;
            else
               gsdbg.append( "(no text specified)" ) ;
            ofsdbg << gsdbg.ustr() << endl ;
         }
      }
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_skForm

   #undef DEBUG_skForm
}  //* End init_skForm() *

//*************************
//*     init_skFields     *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Range-check the window fields' data, and adjust if necessary to fit          *
//* within the target fields.                                                    *
//*                                                                              *
//* Programmer's Note: It is possible that the caller COULD fool our simple      *
//* range checking with some truly stupid initialization value, but in that      *
//* case the caller would deserve the consequent overlapping of fields, field    *
//* extending beyond the text area, cursor referencing open space, etc.          *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: 'true'  if field(s) correct as specified                            *
//*          'false' if parameters modified to fit within the window             *
//********************************************************************************
//* Notes On Initialization of 'skForm' object:                                  *
//* -------------------------------------------                                  *
//* 1) 'fCnt' member must be initialized by the skForm constructor and not       *
//*     modified.                                                                *
//* 2) 'fi' member must be: 0 <= fi < fCnt    (0 by default)                     *
//* 3) 'base' member: initial value is ignored. 'base' is set to the             *
//*     coordinates specified by the window's 'textBase' member.                 *
//*     These coordinates are the one-based offset from the terminal window      *
//*     origin (upper-left corner).                                              *
//* 4) 'pKey' member is ignored               ('\0' by default)                  *
//* 5) 'ins' and 'beep' flags may be either set or reset. (defaults are          *
//*    controlled by the skForm constructor.)                                    *
//* 6) For each skField object in the 'fld' array:                               *
//*    a) 'orig.row' member should be the zero-based OFFSET into the text area   *
//*                  of the window, and must be less than height of text area.   *
//*    b) 'orig.col' member should be the zero-based OFFSET into the text area   *
//*                  of the window, and must be less than width of text area.    *
//*    c) 'hgt' member: 1 <= hgt <= text_area_rows                               *
//*    d) 'wid' member: 1 <= wid <= text_area_columns                            *
//*    e) 'ip' member: 0 <= ip <= text_characters in 'txt' member.               *
//*        (default vaule is zero)                                               *
//*    f) 'cur' member: initial value is ignored and is initialized to           *
//*        correspond to the value of the 'ip' member.                           *
//*    g) 'ro' member: may be set, indicating that the field is read-only and    *
//*        cannot receive input focus; or reset i.e. field text may be edited.   *
//*    h) Fields may not overlap and must lie completely within the text area    *
//*       of the window.                                                         *
//*    i) The number of character cells is limited by the capacity of the        *
//*       gString object. If the field is defined with more character cells      *
//*       i.e. > gsMAXCHARS, (rather unlikely), then that excess display area    *
//*       will not be used.                                                      *
//*    j) 'txt' member may be initialized to any sequence of characters allowed  *
//*       by the gString-class; however, the number of characters actually       *
//*       displayed is limited by the number of text rows and column defined     *
//*       for the field.                                                         *
//*       [In a future release, we may implement horizontal scrolling ]          *
//*       [within a field ('off' member). See the NcDialog API for    ]          *
//*       [examples of this.                                          ]          *
//*    k) The 'aca' member is set to acaATTR_DFLT when the skField object is     *
//*       created. If the aca.acaVal value is acaATTR_DFLT, this method          *
//*       initializes the member to the window's text attributes ('txtAttr').    *
//*       If the aca.acaVal value is not acaAttr_DFLT, we simply decode the      *
//*       attribute value provided for that field.                               *
//*                                                                              *
//* Notes On Initialization of 'skField' array:                                  *
//* -------------------------------------------                                  *
//* -- By default, the window interior is a single field which includes the      *
//*    entire interior area of the window.                                       *
//*    Example: skForm f( 1 );                                                   *
//*             f.fld[0].hgt = ROWS_IN_TEXT_AREA ;                               *
//*             f.fld[0].wid = COLS_IN_TEXT_AREA ;                               *
//*             f.fld[0].orig = ( 0, 0 ) ;                                       *
//*             (Field text is initially an empty string, "".)                   *
//* -- Define one field for each text row of the window (text rows == n).        *
//*    Optionally, initialize text for each field.                               *
//*    Example: skForm f( n );                                                   *
//*             f.fld[0].hgt = 1 ;                                               *
//*             f.fld[0].wid = COLS_IN_TEXT_AREA ;                               *
//*             f.fld[0].orig = ( 0, 0 ) ;                                       *
//*             f.fld[0].txt = "Text for first field." ;                         *
//*             . . . .                                                          *
//*             f.fld[n - 1].hgt = 1 ;                                           *
//*             f.fld[n - 1].wid = COLS_IN_TEXT_AREA ;                           *
//*             f.fld[n - 1].orig = ( n - 1, 0 ) ;                               *
//*             f.fld[0].txt = "Text for last field." ;                          *
//* -- Define a custom number of fields and field position/dimensions for each.  *
//*    Example: skForm f( 3 );                                                   *
//*             f.fld[0].hgt = 1 ;                                               *
//*             f.fld[0].wid = COLS_IN_TEXT_AREA - 2 ;                           *
//*             f.fld[0].orig = ( 1, 1 ) ;                                       *
//*             f.fld[0].txt = "Text for field A." ;                             *
//*             f.fld[0].ip = 5 ;                                                *
//*             f.fld[1].hgt = 2 ;                                               *
//*             f.fld[1].wid = COLS_IN_TEXT_AREA - 6 ;                           *
//*             f.fld[1].orig = ( 3, 4 ) ;                                       *
//*             f.fld[1].txt = "Text for field B.\nAdditional text." ;           *
//*             f.fld[0].ip = 17 ;                                               *
//*             f.fld[2].hgt = 1 ;                                               *
//*             f.fld[2].wid = COLS_IN_TEXT_AREA - 2 ;                           *
//*             f.fld[2].orig = ( 6, 1 ) ;                                       *
//*             f.fld[2].txt = "Text for field C." ;                             *
//*                                                                              *
//* See skForm-class documentation for more information.                         *
//*                                                                              *
//********************************************************************************

bool ACWin::init_skFields ( void )
{
   skField *fptr ;                        // pointer to field under test
   bool status = true ;                   // return value

   //* Range check offsets, dimensions and text width.*
   for ( short f = ZERO ; f < this->skf->fCnt ; ++f )
   {
      fptr = &this->skf->fld[f] ;
      if ( (fptr->orig.row < ZERO) || (fptr->orig.row >= this->txtRows) )
      { fptr->orig.row = f ; status = false ; }
      if ( (fptr->orig.col < ZERO) || (fptr->orig.col >= this->txtCols) )
      { fptr->orig.col = ZERO ; status = false ; }
      if ( (fptr->hgt < 1) || ((fptr->orig.row + fptr->hgt) > this->txtRows) )
      { fptr->hgt = 1 ; status = false ; }
      if ( (fptr->wid < 1) || ((fptr->orig.col + fptr->wid) > this->txtCols) ) 
      { fptr->wid = 1 ; status = false ; }

      //* If the text data are not properly formatted to *
      //* fit within the field, reformat the data.       *
      //* Range check the index into the text data, and  *
      //* initialize the cursor offset.                  *
      this->formatFieldText ( fptr ) ;

//      //* Range check the index into the text data, *
//      //* and initialize the cursor offset.         *
//OBSOLETE      fptr->cur = this->ipOffset ( fptr ) ;

      //* Set the color attribute for each field. (see note above) *
      if ( fptr->aca.acaVal == acaATTR_DFLT )
         fptr->aca.acaVal = this->txtAttr.acaVal ;
      fptr->aca.decode( fptr->aca.acaVal ) ;
   }
   return status ;

}  //* End init_skFields() *

//*************************
//*    formatFieldText    *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Scan the text data, and if necessary, reformat it to fit the field.          *
//*                                                                              *
//* The cursor offset within the field is updated to match the insertion point   *
//* (IP). If the IP is not in the desired position, the cursor will also be      *
//* incorrect. Beware!                                                           *
//*                                                                              *
//* Note that the data are not truncated; however, newline characters are        *
//* inserted if necessary to be sure the data do not overrun the edge of         *
//* the field.                                                                   *
//*                                                                              *
//* Input  : fptr : pointer to the field containing text to be formatted.        *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::formatFieldText ( skField *fptr )
{
   gString gsParse ;             // receives 
   short si = ZERO ;             // source text index

   //* Newlines are incompatible with single-row *
   //* fields. Silently delete them.             *
   if ( fptr->hgt == 1 )
   {
      while ( (si = fptr->txt.find( NEWLINE, si )) >= ZERO )
         fptr->txt.erase( NEWLINE, si ) ;
   }

   //* Validate formatting for multi-row fields, *
   //* and update format if necessary.           *
   //* (Text is not truncated.)                  *
   else
   {
      this->parseFieldText ( fptr, gsParse, false ) ;
      fptr->txt = gsParse ;
   }

   //* Set cursor offset to match current IP, *
   fptr->cur = this->ipOffset ( fptr ) ;

}  //* End formatFieldText() *

//*************************
//*      OpenWindow       *
//*************************
//********************************************************************************
//* Complete window setup and display it within the terminal window.             *
//*                                                                              *
//* Input  : initText : (optional, null pointer by default)                      *
//*                     initial text to be written to window                     *
//*          initField: (optional, -1 by default) [CURRENTLY UNUSED]             *
//*                     index of target field for storing 'initText' data        *
//*                     Ignored if 'initText' not specified.                     *
//*                                                                              *
//* Returns: cursor position (offset) i.e. text insertion point                  *
//********************************************************************************

WinPos ACWin::OpenWindow ( const char* initText, short initField )
{
   //* Set cursor to invisible while drawing window.     *
   //* This make for a less-visible write to the display.*
   this->acSetCursorStyle ( csHide ) ;

   //* Clear the area of the terminal window *
   //* where the window will be drawn.       *
   this->clearWin ( this->txtAttr, true ) ;

   //* Draw the window border *
   this->drawBorder () ;

   //* Draw the initial window text, if provided. *
   //* Cursor position will be at end-of-text.    *
   //* Else, set cursor to window origin.         *
   if ( initText != NULL )
   {
      this->setAttributes ( this->txtAttr ) ;
      this->txtOff = this->Write ( this->txtOff, initText ) ;
   }
   else
      this->acSetCursor ( this->txtBase ) ;

   this->acSetCursorStyle ( csShow ) ;    // make the cursor visible

   //* If cursor style changes when Insert key is pressed  *
   //* is enabled, set the initial cursor style.           *
   //* Otherwise, current cursor style is not modified.    *
   if ( this->skf->insc != false )
      this->acSetCursorStyle ( (this->skf->ins ? this->skf->icur : this->skf->ocur) ) ;

   //* For defined fields which contain text, display field contents *
   for ( short f = ZERO ; f < this->skf->fCnt ; ++f )
   {
      if ( (this->skf->fld[f].txt.gschars()) > 1 )
         this->RefreshField ( f ) ;
   }

   //* Place the cursor within the active field *
   this->setCursor2IP ( &this->skf->fld[this->skf->fi], this->txtBase ) ;

   //* Restore fgnd/bkgnd attributes to current terminal attributes.*
   this->restoreAttributes () ;

   //* Disable input buffering and set "soft-echo" option.*
   // Programmer's Note: Blocking read of the input stream is enabled in case 
   // application uses stdin OUTSIDE the window.
   this->acBufferedInput ( false, softEchoA, true ) ;

   return ( this->txtOff ) ;     // return window insertion point (relative offset)

}  //* End OpenWindow() *

//*************************
//*      CloseWindow      *
//*************************
//********************************************************************************
//* Close the window: Erase the window display data.                             *
//* As a side-effect, input buffering is re-enabled.                             *
//*                                                                              *
//* Note that the window object still exists in memory. To release the window's  *
//* resources, invoke the destructor i.e. delete the ACWin object.               *
//* Example: ACWin *winPtr = new ACWin ;  ...  delete winPtr ;                   *
//*                                                                              *
//* Input  : clear    : (optional, 'true' by default)                            *
//*                     if 'true',  clear (erase) the display area               *
//*                     if 'false', do not clear the display area                *
//*          fillColor: (optional, terminal dflt attr by default)                *
//*                     color attribute used to erase data from window.          *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::CloseWindow ( bool clear, acAttr fillColor )
{
   //* Erase the window object from the terminal display. *
   if ( clear )
   {
      if ( fillColor == acaUSEDFLT )   // use saved terminal display attributes
         this->clearWin ( this->termAttr, true ) ;
      else                             // decode caller's background attribute
      {
         acaExpand acatmp( fillColor ) ;
         this->clearWin ( acatmp, true ) ;
      }
   }
   this->restoreAttributes () ;        // Return background to terminal default

   this->acBufferedInput ( true, termEcho ) ;   // Re-enable buffered input

}  //* End CloseWindow() *

//*************************
//*      RefreshWin       *
//*************************
//********************************************************************************
//* Redraw the window to display any changes that have been made, or to recover  *
//* when other data have obscured the the window.                                *
//* For instance, call RefreshWin() after a previoius call to HideWin().         *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::RefreshWin ( void )
{
   //* Set cursor to invisible while drawing window.     *
   //* This make for a less-visible write to the display.*
   this->acSetCursorStyle ( csHide ) ;

   //* Clear the area of the terminal window *
   //* where the window will be drawn.       *
   this->clearWin ( this->txtAttr, true ) ;

   //* Draw the window border *
   this->drawBorder () ;

   //* For defined fields which contain text, display field contents *
   for ( short f = ZERO ; f < this->skf->fCnt ; ++f )
   {
      if ( (this->skf->fld[f].txt.gschars()) > 1 )
         this->RefreshField ( f ) ;
   }

   this->acSetCursorStyle ( csShow ) ;    // make the cursor visible

   //* Restore fgnd/bkgnd attributes to current terminal attributes.*
   this->restoreAttributes () ;

}  //* End RefreshWin() *

//*************************
//*        HideWin        *
//*************************
//********************************************************************************
//* Temporarily erase the window from the terminal display.                      *
//*                                                                              *
//* Data in the skForm object (if any) will be retained; however, any text       *
//* previously written directly into the window interior will be lost.           *
//*                                                                              *
//* Input  : fillAttr : (optional, default: terminal default attributes)         *
//*                     color attribute used to erase data from window.          *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::HideWin ( acAttr fillAttr )
{
   if ( fillAttr == acaUSEDFLT )    // use saved terminal display attributes
      this->clearWin ( this->termAttr, true ) ;
   else                             // decode caller's background attribute
   {
      acaExpand acatmp( fillAttr ) ;
      this->clearWin ( acatmp, true ) ;
      this->setAttributes ( this->termAttr ) ;  // restore terminal attributes
   }

}  //* End HideWin() *

//*************************
//*        MoveWin        *
//*************************
//********************************************************************************
//* Move the window to another position within the terminal window.              *
//* This is simply erasing the window, updating the base position and redrawing  *
//* the window at the new position.                                              *
//*                                                                              *
//* Data in the skForm object (if any) will be redrawn as part of the move;      *
//* however, any text previously written directly into the window interior will  *
//* be lost. (see 'newText')                                                     *
//*                                                                              *
//* Input  : newPos   : (by reference) new row/column position for window origin *
//*                     (upper-left corner of window)                            *
//*          newText  : (optional, null pointer by default)                      *
//*                     text to be written to interior of window, beginning at   *
//*                     the upper left corner of the window interior.            *
//*                                                                              *
//* Returns: 'true'  if success                                                  *
//*          'false' if specified position would place any part of the window    *
//*                  outside the visible area of the terminal display.           *
//********************************************************************************

bool ACWin::MoveWin ( const WinPos& newPos, const char *newText )
{
   WinPos wpnew = newPos ;                // working copy of new position
   bool iLikeToMoveIt = false ;           // return value

   //* Range check the new position *
   if ( wpnew.row < 1 )    wpnew.row = 1 ; // silently convert zero to 1-based.
   if ( wpnew.col < 1 )    wpnew.col = 1 ;
   if ( ((wpnew.row + this->height) < this->termRows) &&
        ((wpnew.col + this->width) < this->termCols) )
   {
      this->HideWin () ;
      this->winBase = wpnew ;
      this->txtBase = { short(this->winBase.row + 1), 
                        short(this->winBase.col + 1) } ;
      this->skf->base = this->txtBase ;
      this->RefreshWin () ;
      iLikeToMoveIt = true ;
   }

   return iLikeToMoveIt ;

}  //* End MoveWin() *

//*************************
//*       ClearWin        *
//*************************
//********************************************************************************
//* Erase all text from the interior (text area) of the window.                  *
//*                                                                              *
//* The text in the skForm object (if any) is not redrawn. See also RefreshWin().*
//*                                                                              *
//* Input  : newAttr : (optional, acaUSEDFLT by default)                         *
//*                    new color attributes for text area of window              *
//*                    (logical OR of forground and background color)            *
//*                    (attributes and attribute modifiers          )            *
//*                                                                              *
//* Returns: Cursor position (text insertion point)                              *
//*          This is the upper-left corner of the text area (0/0).               *
//********************************************************************************

WinPos ACWin::ClearWin ( acAttr newAttr )
{
   if ( newAttr != acaUSEDFLT )     // if new interior attribute specified
      this->txtAttr.decode( newAttr ) ;

   //* Set the interior background color attribute *
   this->setAttributes ( this->txtAttr, false, true ) ;

   //* Clear the display area *
   this->acClearArea ( this->txtBase.row, txtBase.col, 
                       this->txtRows, this->txtCols ) ;

   //* Set cursor offset to upper-left corner, *
   //* and set physical cursor to txtBase.     *
   this->txtOff = { 0, 0 } ;
   this->acSetCursor ( this->txtBase ) ;

   //* Restore current terminal attributes *
   this->restoreAttributes () ;

   return ( this->txtOff ) ;     // return new cursor position

}  //* End ClearWin() *

//*************************
//*       ClearRow        *
//*************************
//********************************************************************************
//* Erase text from the specified row of the text area.                          *
//*                                                                              *
//* Note: ClearRow() does not clear the stored text-field data associated with   *
//*       the target row. To clear a single text field within the window, use    *
//*       the ClearField() method instead.                                       *
//*                                                                              *
//* Input  : trgRow  : target row (0-based offset into text area)                *
//*          txtAttr : (optional, acaUSEDFLT by default)                         *
//*                    new color attributes for text area of window              *
//*                    (logical OR of forground and background color)            *
//*                    (attributes and attribute modifiers          )            *
//*                                                                              *
//* Returns: Cursor position (text insertion point)                              *
//*          This is the left edge of the specified row (trgRow/0).              *
//*          Note: If 'trgRow' is out-of-range, window will not be modified.     *
//********************************************************************************

WinPos ACWin::ClearRow ( short trgRow, acAttr txtAttr )
{
   if ( (trgRow >= ZERO) && (trgRow < this->txtRows) )
   {
      if ( txtAttr != acaUSEDFLT )     // if new interior attribute specified
         this->txtAttr.decode( txtAttr ) ;

      //* Set the background color attribute *
      this->acSetBg ( this->txtAttr.aesBgnd ) ;

      //* Clear the target row *
      gString gs ;
      gs.padCols( this->txtCols ) ;
      this->acSetCursor ( this->txtBase.row + trgRow, this->txtBase.col ) ;
      this->ttyWrite ( gs ) ;

      //* Set cursor position to left edge of target row *
      this->txtOff = { trgRow, 0 } ;

      //* Restore terminal-window background attribute *
      this->acSetBg ( this->termAttr.aesBgnd ) ;

      //* Restore terminal-window background attribute *
      this->acSetBg ( this->termAttr.aesBgnd ) ;
   }
   return ( this->txtOff ) ;

}  //* End ClearRow() *

//*************************
//*     SetFieldText      *
//*************************
//********************************************************************************
//* Replace the current contents of the field with new data.                     *
//* The 'ip' (insertion point) of the field is set to zero, and the              *
//* cursor is set to the field origin.                                           *
//*                                                                              *
//* Input  : fIndx   : index of target field within the skForm object            *
//*          gsTxt : (by referene) new text data for field                       *
//*                                                                              *
//* Returns: 'true'  if successful                                               *
//*          'false' if invalid field index specified                            *
//********************************************************************************

bool ACWin::SetFieldText ( short fIndx, const gString& gsTxt )
{
   bool status = false ;               // return value

   if ( (fIndx >= ZERO) && (fIndx < this->skf->fCnt) )
   {
      skField *fptr = &this->skf->fld[fIndx] ;
      fptr->txt = gsTxt ;
      fptr->ip = ZERO ;
      fptr->cur = { 0, 0 } ;
      this->formatFieldText ( fptr ) ;
      this->RefreshField ( fIndx ) ;
   }
   return status ;

}  //* End SetFieldText() *

//*************************
//*     GetFieldText      *
//*************************
//********************************************************************************
//* Get a copy of the current contents of the field.                             *
//*                                                                              *
//* Input  : fIndx   : index of target field within the skForm object            *
//*          gsTxt : (by reference) receives a copy of field text                *
//*                                                                              *
//* Returns: 'true'  if successful                                               *
//*                  gsTxt contains field text data                              *
//*          'false' if invalid field index specified                            *
//*                  gsTxt cleared to empty string                               *
//********************************************************************************

bool ACWin::GetFieldText ( short fIndx, gString& gsTxt ) const
{
   bool status = false ;               // return value

   if ( (fIndx >= ZERO) && (fIndx < this->skf->fCnt) )
   {
      gsTxt = this->skf->fld[fIndx].txt ;
      status = true ;
   }
   return status ;

}  //* End GetFieldText() *

//*************************
//*      ClearField       *
//*************************
//********************************************************************************
//* Erase text from the specified text field. The stored text is deleted, and    *
//* the display area of the field is cleared.                                    *
//*                                                                              *
//* Input  : fIndx   : index of target field                                     *
//*          txtAttr : (optional, acaUSEDFLT by default)                         *
//*                    new color attributes for the target field                 *
//*                    (logical OR of forground and background color)            *
//*                    (attributes and attribute modifiers          )            *
//*                                                                              *
//* Returns: 'true'  if target field cleared                                     *
//*          'false; if fldIndex out-of-range (data not modified)                *
//********************************************************************************

bool ACWin::ClearField ( short fIndx, acAttr txtAttr )
{
   bool status = false ;            // return value

   if ( (fIndx >= ZERO) && (fIndx < this->skf->fCnt) )
   {
      skField *fptr = &this->skf->fld[fIndx] ; // point to the target field

      fptr->txt.clear() ;                    // clear the stored text
      fptr->ip = ZERO ;                      // set IP to field origin
      fptr->cur = this->ipOffset ( fptr ) ;  // update cursor position
      if ( txtAttr != acaUSEDFLT )           // if specified, set new attributes
         fptr->aca.decode( txtAttr ) ;
      this->RefreshField ( this->skf->fi ) ; // update the display
      this->restoreAttributes () ;           // restore terminal attributes
      status = true ;      // success
   }
   return status ;

}  //* End ClearField() *

//*************************
//*      SetFieldIP       *
//*************************
//********************************************************************************
//* Set the insertion point (IP) for the field. This will set the cursor         *
//* position over the character at the specified offset.                         *
//*                                                                              *
//* If 'fIndx' references the field which has input focus, the visible cursor    *
//* will also be updated.                                                        *
//*                                                                              *
//* Special case: If 'newIp' is greater that the number of characters in the     *
//* field, IP will be set to end-of-text (over the null terminator).             *
//*                                                                              *
//* Input  : fIndx   : index of target field                                     *
//*          newIp   : character offset into the text of the field               *
//*                    ZERO <= newIp <= text characters                          *
//*                    where ZERO == field origin and any value at or beyond     *
//*                    the number of characters in the array signals that IP     *
//*                    should be set to end-of-text.                             *
//*                    (A safe end-of-text value is: gsMAXCHARS.)                *
//*                                                                              *
//* Returns: 'true'  if successful                                               *
//*          'false' if invalid field index specified                            *
//********************************************************************************

bool ACWin::SetFieldIP ( short fIndx, short newIp )
{
   bool status = false ;               // return value

   if ( (fIndx >= ZERO) && (fIndx < this->skf->fCnt) )
   {
      skField *fptr = &this->skf->fld[fIndx] ;
      short maxIp = fptr->txt.gschars() - 1 ;
      if ( newIp < ZERO )        newIp = ZERO ;    // range check
      else if ( newIp > maxIp )  newIp = maxIp ;

      //* Save the new IP and update the cursor offset within the field.*
      fptr->ip = newIp ;
      fptr->cur = this->ipOffset ( fptr ) ;

      //* If target field currently has the input focus, *
      //* update the physical cursor position.           *
      if ( fIndx == this->skf->fi )
         this->setCursor2IP ( &this->skf->fld[fIndx], this->txtBase ) ;

      status = true ;
   }
   return status ;

}  //* End SetFieldIP() *

//*************************
//*     RefreshField      *
//*************************
//********************************************************************************
//* Refresh (erase and redraw) the contents of the specified field, or           *
//* optionally, refresh all fields defined by the skForm object.                 *
//* Input focus remains on the currently active field, and cursor is returned    *
//* to insertion point within that field.                                        *
//*                                                                              *
//* Input  : fIndx: index of target text field                                   *
//*                 If fIndx >= total of fields defined (fCnt),                  *
//*                 all fields will be refreshed.                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::RefreshField ( short fIndx )
{
   WinPos wp ;                      // cursor position (terminal relative)
   skField *fptr ;                  // pointer to field

   //* Range check and if out-of-range,   *
   //* silently set to refresh all fields.*
   if ( (fIndx < ZERO) || (fIndx > skf->fCnt) )
      fIndx = skf->fCnt ;

   this->acSetCursorStyle ( csHide ) ;             // make the cursor invisible

   short firstField = (fIndx == skf->fCnt ? ZERO : fIndx),
         lastField  = fIndx ;

   for ( short f = firstField ; f <= lastField ; ++f )
   {
      fptr = &skf->fld[f] ;         // point to target field
      this->setAttributes ( fptr->aca ) ; // set field attributes
      //* Clear the field. Terminal-relative origin of field is returned.*
      wp = this->clearField ( skf, f ) ;

      //* Write the field text, if any.*
      if ( (fptr->txt.gschars()) > 1 )
         this->refreshText ( fptr, wp ) ;
   }

   //* Set background attribute for field with focus.*
   //* It may be this field or another field.        *
   fptr = &skf->fld[skf->fi] ;
   this->setAttributes ( fptr->aca ) ;

   //* Set cursor to insertion point of field with focus.*
   this->setCursor2IP ( fptr, skf->base ) ;

   this->acSetCursorStyle ( csShow ) ;       // make the cursor visible

}  //* End RefreshField() *

//*************************
//*     RefreshField      *
//*************************
//********************************************************************************
//* Refresh (erase and redraw) the contents of the field which has input focus.  *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::RefreshField ( void )
{

   this->RefreshField ( this->skf->fi ) ;

}  //* End RefreshField() *

//*************************
//*     SetFieldAttr      *
//*************************
//********************************************************************************
//* Set the color attributes and modifiers for the specified text field.         *
//* Refresh the field display using the modified attributes.                     *
//*                                                                              *
//* On return, the cursor is positioned in the field with input focus.           *
//*                                                                              *
//* Input  : fIndx   : index of target field within the skForm object            *
//*          txtAttr : (acAttr bitfield) a logical OR of forground and           *
//*                    background color attributes and attribute modifiers       *
//*                                                                              *
//* Returns: 'true'  if attribute set successfully                               *
//*          'false' if invalid field index specified                            *
//********************************************************************************

bool ACWin::SetFieldAttr ( short fIndx, acAttr txtAttr )
{
   bool status = false ;

   if ( (fIndx >= ZERO) && (fIndx < this->skf->fCnt) )
   {
      //* Decode the field attribute *
      this->skf->fld[fIndx].aca.decode( txtAttr ) ;

      //* Redraw the text in the field *
      this->RefreshField ( fIndx ) ;

      //* Set the cursor to its position in the field with focus.*
      if ( fIndx != this->skf->fi )
         this->RefreshField ( this->skf->fi ) ;

      //* Restore current terminal attributes *
      this->restoreAttributes () ;

      status = true ;
   }
   return status ;

}  //* End SetFieldAttr() *

//*************************
//*     SetInputFocus     *
//*************************
//********************************************************************************
//* Set input focus to the specified field, and refresh field contents.          *
//* The cursor is placed on the text insertion point.                            *
//* Note: Read-only fields cannot receive the input focus.                       *
//*                                                                              *
//* Input  : fldIndex: index of target text field                                *
//*                    The specified field becomes the active field.             *
//*                                                                              *
//* Returns: index of field with focus                                           *
//********************************************************************************

short ACWin::SetInputFocus ( short fldIndex )
{
   if ( (fldIndex >= ZERO) && (fldIndex < this->skf->fCnt) &&
        (this->skf->fld[fldIndex].ro == false) )
   {
      this->skf->fi = fldIndex ;                // set the index
      this->RefreshField ( this->skf->fi ) ;    // update the display
   }
   return this->skf->fi ;

}  //* End SetInputFocus() *

//*************************
//*    Focus2NextField    *
//*************************
//********************************************************************************
//* Set input focus to the next field, and refresh field contents.               *
//* If already on the last field (index==fCnt-1), wrap around to the first       *
//* field of the array.                                                          *
//* The cursor is placed on the text insertion point.                            *
//* Note: Read-only fields cannot receive the input focus.                       *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: index of field with focus                                           *
//********************************************************************************

short ACWin::Focus2NextField ( void )
{
   if ( this->skf->fCnt > 1 )             // must be multiple fields
   {
      short fdef = this->skf->fCnt,       // number of fields defined
            fcur = this->skf->fi,         // index of field with focus
            fnew = fcur + 1 ;             // index of next field
      while ( fnew != fcur )              // scan for the next accessible field
      {
         if ( fnew >= fdef )              // wrap-around to first field
            fnew = ZERO ;  // (yes, it's possible that fnew==fdef, but that's OK)
         if ( this->skf->fld[fnew].ro == false ) // field is accessible
         {
            this->skf->fi = fnew ;        // set index of active field
            this->RefreshField ( fnew ) ; // redraw field and position the cursor
            break ;
         }
         ++fnew ;                         // reference next field
      }
   }
   return this->skf->fi ;

}  //* End Focus2NextField() *

//*************************
//*    Focus2PrevField    *
//*************************
//********************************************************************************
//* Set input focus to the previous field, and refresh field contents.           *
//* If already on the first field (index==zero), wrap around to the last         *
//* field of the array.                                                          *
//* The cursor is placed on the text insertion point.                            *
//* Note: Read-only fields cannot receive the input focus.                       *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: index of field with focus                                           *
//********************************************************************************

short ACWin::Focus2PrevField ( void )
{
   if ( this->skf->fCnt > 1 )             // must be multiple fields
   {
      short fdef = this->skf->fCnt,       // number of fields defined
            fcur = this->skf->fi,         // index of field with focus
            fnew = fcur - 1 ;             // index of next field

      while ( fnew != fcur )              // scan for the prev accessible field
      {
         if ( fnew < ZERO )               // wrap-around to last field
            fnew = fdef - 1 ; // (yes, it's possible that fnew==fdef, but that's OK)
         if ( this->skf->fld[fnew].ro == false ) // field is accessible
         {
            this->skf->fi = fnew ;        // set index of active field
            this->RefreshField ( fnew ) ; // redraw field and position the cursor
            break ;
         }
         --fnew ;                         // reference previous field
      }
   }
   return this->skf->fi ;

}  //* End Focus2PrevField() *

//*************************
//*       InsertKey       *
//*************************
//********************************************************************************
//* Enable or disable user access to the Insert/Overstrike input mode when user  *
//* is editing a field, and specify the initial mode.                            *
//*                                                                              *
//* Insert/Overstrike toggle is active by default, and the initial mode is       *
//* Insert Mode. For some fields, or some kinds of data, however, it is useful   *
//* to set one mode or the other and lock out user access.                       *
//*                                                                              *
//* Optionally enable or disable cursor style change when Insert/Overstrike mode *
//* is toggled. This provides a visual cue to the user whether Insert mode or    *
//* Overstrike mode is active. This is common in word-processing programs and    *
//* text editors. Optionally, the cursor style for each mode may be specified    *
//* as a member of enum CurStyle (excluding csShow and csHide).                  *
//*                                                                              *
//* Input  : enable  : 'true'  == enable user access to Insert key               *
//*                               (this is the default)                          *
//*                    'false' == disable user access to Insert key              *
//*                               (the mode is locked by application)            *
//*          initMode: initial input mode                                        *
//*                    'true'  == Insert Mode                                    *
//*                    'false' == Overstrike Mode                                *
//*          styles  : (optional 'false' by default)                             *
//*                    Enable or disable automatic change of cursor style when   *
//*                    Insert/Overstrike is toggled.                             *
//*                    'true'  == enable cursor style change                     *
//*                    'false' == disable cursor style change                    *
//*          insStyle: (optional, istyleDFLT by default)                         *
//*                     cursor style for insert mode                             *
//*                     referenced only when the 'styles' flag is set            *
//*          ovrStyle: (optional, ostyleDFLT by default)                         *
//*                     cursor style for overstrike mode                         *
//*                     referenced only when the 'styles' flag is set            *
//*                                                                              *
//* Returns: 'true'  if Insert/Overstrike toggle enabled, else 'false'           *
//********************************************************************************

bool ACWin::InsertKey ( bool enable, bool initMode, bool styles, 
                        CurStyle insStyle, CurStyle ovrStyle )
{
   this->skf->togg = enable ;             // initialize the lock/unlock flag
   this->skf->ins  = initMode ;           // set the specified mode
   this->skf->insc = styles ;             // initialize the cursor-style flag

   switch ( insStyle )                    // set the Insert mode cursor style
   {
      case csBblock:
      case csDflt:
      case csSblock:
      case csBuline:
      case csSuline:
      case csBvbar:
      case csSvbar:
         this->skf->icur = insStyle ;
         break ;
      default:
         this->skf->icur = istyleDFLT ;
         break ;
   }
   switch ( ovrStyle )                    // set the Overstrike mode cursor style
   {
      case csBblock:
      case csDflt:
      case csSblock:
      case csBuline:
      case csSuline:
      case csBvbar:
      case csSvbar:
         this->skf->ocur = ovrStyle ;
         break ;
      default:
         this->skf->ocur = ostyleDFLT ;
         break ;
   }

   //* If enabled, set the current cursor style *
   if ( this->skf->insc != false )
      this->acSetCursorStyle ( (this->skf->ins ? this->skf->icur : this->skf->ocur) ) ;

   return this->skf->togg ;

}  //* End InsertKey() *

//*************************
//*       EditForm        *
//*************************
//********************************************************************************
//* Interactively edit the fields defined within the skForm object.              *
//*                                                                              *
//* Input  : exitChar : when this character is received, return to caller        *
//*          focusFld : (by reference) receives the index of the field with      *
//*                     input focus on return to caller.                         *
//*                                                                              *
//* Returns: keycode of last key pressed                                         *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* 1) Tab and Shift+Tab change fields (if multiple accessible fields defined).  *
//* 2) Left, Right, Home, End maneuver within the field with input focus.        *
//* 3) For multi-row fields, UpArrow and DownArrow keys maneuver within field.   *
//*    For single-row fields UpArrow and DownArrow keys are translated to        *
//*    Shift+Tab and Tab, respectively.                                          *
//* 4) Until we implement support for multiple ACWin objects within the          *
//*    terminal windows, PageUp, PageDn keys are translated to Shift+Tab and     *
//*    Tab, respectively.                                                        *
//* 5) Insert key toggles Insert/Overstrike input mode.                          *
//*    For some types of data entry it is advantageous to initialize the         *
//*    Insert/Overstrike flag, and then ignore the Insert keycode. This is       *
//*    handled by the 'togg' member which enables or disables recognition of     *
//*    of the Insert keycode.                                                    *
//* 6) Delete:                                                                   *
//*    Overwrite the character under the cursor by shifting all trailing         *
//*    characters left by the width of that character.                           *
//* 7) Backspace: The cursor is positioned one character to the left, and then   *
//*    that character is overwritten by shifting all trailing characters left    *
//*    by the width of that character.                                           *
//*                                                                              *
//* -- There is a design decision to be made whether to allow the input method   *
//*    (acRead()) to automatically echo printing characters to the display as    *
//*    soon as they are received.                                                *
//*    -- Display updates within the field-edit methods is simplified if         *
//*       automatic display update is disabled because it give greater control   *
//*       over when (or if) the display is updated.                              *
//*    -- Conversely, the application may want to interact with the user         *
//*       _outside_ of the edit methods, in which case the automatic echo of     *
//*       input would need to be enabled.                                        *
//*    Toggling between echo and non-echo modes is a non-trivial system          *
//*    configuration task which should not happen every few milliseconds because *
//*    the system will eventually become confused. (poor, brain-dead system :-)  *
//*    Currently, all soft-echo options except 'noEcho' perform automatic        *
//*    display updates. We may revisit this in a later release.                  *
//*                                                                              *
//* -- The cursor must stay within the display area of the field with            *
//*    input focus.                                                              *
//*                                                                              *
//* -- The cursor is positioned on a displayed character OR is positioned on     *
//*    the free column immediately following the last displayed character of     *
//*    the current row (if any). This position will generally reference either   *
//*    a newline character or the null terminator. The cursor may not be         *
//*    positioned in empty space beyond that point.                              *
//*                                                                              *
//* -- Doing the two-step:                                                       *
//*    For multi-row fields the wcsHOME and wcsEND special keycodes indicate     *
//*    a two-step operation.                                                     *
//*    -- wcsHOME:                                                               *
//*       1) If cursor is not already at the left edge of the current row,       *
//*          move cursor to left edge of current row.                            *
//*       2) If cursor is already at the left edge of the current row, but       *
//*          not at the field origin, move cursor to field origin.               *
//*    -- wcsEND:                                                                *
//*       1) If cursor is not already at the right end of text for the current   *
//*          row, move cursor to the column after the last character displayed   *
//*          on the row, or if the row is fully occupied, the right edge of      *
//*          the row.                                                            *
//*       2) If cursor is already at the end of the current row, but not at      *
//*          the end of the displayed text, move cursor to end of displayed      *
//*          text.                                                               *
//*                                                                              *
//* -- Scrolling text within the field.                                          *
//*    For the current release, we have not implemented scrolling of data        *
//*    within the field. This is a non-trivial task which we may address in      *
//*    a future release.                                                         *
//*    Note that the author's NcDialog API _does_ implement scrolling within     *
//*    text fields, as well as providing full support for RTL (Right-To-Left)    *
//*    languages. Please refer to the NcDialog API documentation for further     *
//*    information.                                                              *
//*                                                                              *
//********************************************************************************

wchar_t ACWin::EditForm ( wchar_t exitChar, short& focusFld )
{
   wchar_t wkey ;                   // user input (and return value)

   //* Refresh the field with input focus. This sets the *
   //* color attributes and positions the visible cursor.*
   this->RefreshField ( this->skf->fi ) ;

   while ( (wkey = this->acRead ()) != exitChar )
   {
      if ( (this->specialkeyTest ( wkey )) && (wkey != exitChar) )
      {
         //*************************************************
         //* Convert certain keycodes dependent upon the   *
         //* field definitions within the skForm object.   *
         //*************************************************
          if ( (wkey = this->specialkeyConvert ( wkey )) == NULLCHAR )
             continue ;

         //**************************************
         //* Perform the operation associated   *
         //* with the special keycode.          *
         //**************************************
         this->specialkeyProc ( wkey ) ;

      }     // special key

      //* If a a special key, but not a member of the special-key *
      //* group, the null character was returned. Alert user.     *
      else if ( (wkey == NULLCHAR) && this->skf->beep )
         this->acBeep () ;

      //***********************************************
      //* Else, if a 'normal' (printing) character,   *
      //* insert the character into the text array    *
      //* and update the display.                     *
      //***********************************************
      else if ( (wkey != NEWLINE) && (wkey != NULLCHAR) && (wkey != exitChar) )
         this->normalkeyProc ( wkey ) ;

   }  // while()

   this->restoreAttributes () ;           // return to terminal attributes
   focusFld = this->skf->fi ;             // report the field which has focus
   return wkey ;                          // return the last key pressed

}  //* End EditForm() *

//*************************
//*       EditField       *
//*************************
//********************************************************************************
//* Capture user input to the specified input field.                             *
//*                                                                              *
//* Important Note: The purpose of the Tab and Shift+Tab keys is to move among   *
//* fields defined by the skForm object. Therefore, in addition to the keycode   *
//* specified for loop termination by the 'exitChar' argument, the Tab and       *
//* Shift+Tab keys (if active as "Special" keycodes) will always terminate       *
//* the input loop, so caller can decide whether to continue.                    *
//*                                                                              *
//*                                                                              *
//* Input  : fIndx   : (by reference) index of target field                      *
//*                    On entry, specifies the field to be edited.               *
//*                    On return, index of field with input focus.               *
//*          exitChar: (optional, NEWLINE '\n' by default)                       *
//*                    This character terminates the input loop.                 *
//*          allChar : (optional, 'false' by default)                            *
//*                    if 'false', 'exitChar' controls the loop                  *
//*                    if 'true',  return to caller after each iteration         *
//*                                of the loop                                   *
//*                                                                              *
//* Returns: last keycode captured                                               *
//********************************************************************************

wchar_t ACWin::EditField ( short& fIndx, wchar_t exitChar, bool allChar )
{
   wchar_t wkey ;                   // return value

   //* Range check the target-field index, and if *
   //* in range, set the input focus to that field
   if ( (fIndx >= ZERO) && (fIndx < this->skf->fCnt) )
      this->skf->fi = fIndx ;

   //* Refresh the field with input focus. This sets the *
   //* color attributes and positions the visible cursor.*
   this->RefreshField ( this->skf->fi ) ;

   while ( (wkey = this->acRead ()) != exitChar )
   {
      if ( (this->specialkeyTest ( wkey )) && (wkey != exitChar) )
      {
         //*************************************************
         //* Convert certain keycodes dependent upon the   *
         //* field definitions within the skForm object.   *
         //*************************************************
          if ( (wkey = this->specialkeyConvert ( wkey )) == NULLCHAR )
             continue ;

         //**************************************
         //* Perform the operation associated   *
         //* with the special keycode.          *
         //**************************************
         this->specialkeyProc ( wkey ) ;

          //* If user is moving out of the field, we're done.*
          if ( (wkey == wcsTAB) || (wkey == wcsSTAB) )
             allChar = true ;

      }     // special key

      //* If a a special key, but not a member of the special-key *
      //* group, the null character was returned. Alert user.     *
      else if ( (wkey == NULLCHAR) && this->skf->beep )
         this->acBeep () ;

      //***********************************************
      //* Else, if a 'normal' (printing) character,   *
      //* insert the character into the text array    *
      //* and update the display.                     *
      //***********************************************
      else if ( (wkey != NEWLINE) && (wkey != NULLCHAR) && (wkey != exitChar) )
         this->normalkeyProc ( wkey ) ;

      if ( allChar != false )       // if return on all input characters
         break ;
   }  // while()

   this->restoreAttributes () ;     // return to terminal attributes
   fIndx = this->skf->fi ;          // update caller's copy of field with focus
   return wkey ;                    // return the last key pressed

}  //* End EditField() *

//*************************
//*    specialkeyTest     *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Determine whether the specified keycode is one of the "special" keys.        *
//* This is a stub for calling the AnsiCmd-class method.                         *
//* a) The 'alert' parameter for the call is always 'false'.                     *
//* b) The 'eopt' parameter for the call is always 'softEchoA' i.e. test         *
//*    for all keycodes.                                                         *
//*                                                                              *
//* Input  : wkey : (by reference) keycode to be tested                          *
//*                                                                              *
//* Returns: 'true'  if keycode is one of the special keycodes,                  *
//*                  otherwise 'false'                                           *
//********************************************************************************
//* Programmer's Note:                                                           *
//* -- In a production build, the soft-echo option used for this test is         *
//*    always softEchoA (all special keycodes recognized), overriding the        *
//*    value specified by the 'tset-inpEch' member variable.                     *
//* -- However, when debugging code is enabled i.e. when the test methods are    *
//*    included in the build, 'tset->inpEch' specifies the soft-echo option      *
//*    used for this test. In this way, the test methods are allowed to modify   *
//*    the soft-echo option for the specific test being performed.               *
//********************************************************************************

bool ACWin::specialkeyTest ( wchar_t& wkey ) const
{

   #if DEBUG_ANSICMD == 0
   EchoOpt echoAll = softEchoA ;    // all 'special' keys are recognized
   return ( (this->acSpecialKeyTest ( wkey, false, &echoAll )) ) ;
   #else    // for debugging only (see note above)
   return ( (this->acSpecialKeyTest ( wkey, false )) ) ;
   #endif   // DEBUG_ANSICMD

}  //* End specialkeyTest() *

//*************************
//*   specialkeyConvert   *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Called by EditForm() and EditField() to determine whether the specified      *
//* 'special' keycode is to be converted based on the following criteria:        *
//*                                                                              *
//* A) For skForm objects with only one field, discard inter-field keycodes.     *
//*      wcsTAB    --> '\0' (nullchar)                                           *
//*      wcsSTAB   --> '\0' (nullchar)                                           *
//*      wcsPGUP   --> '\0' (nullchar)                                           *
//*      wcsPGDN   --> '\0' (nullchar)                                           *
//*      wcsUARROW --> '\0' (nullchar)                                           *
//*      wcsDARROW --> '\0' (nullchar)                                           *
//* B) For skForm objects with multiple fields, convert certain inter-field      *
//*    keycodes.                                                                 *
//*      wcsPGDN --> wcsTAB                                                      *
//*      wcsPGUP --> wcsSTAB                                                     *
//*    In a future release, wcsPGDN and wcsPGUP will indicate a move among       *
//*    ACWin objects within the terminal window.                                 *
//* C) For single-row fields, convert certain keycodes which make no sense       *
//*    within a single-row field.                                                *
//*      wcsDARROW --> wcsTAB                                                    *
//*      wcsPGDN   --> wcsTAB                                                    *
//*      wcsAENTER --> wcsTAB                                                    *
//*      wcsUARROW --> wcsSTAB                                                   *
//*      wcsPGUP   --> wcsSTAB                                                   *
//* D) While within a multi-row field, wcsDARROW and wcsUARROW have variable     *
//*    meanings depending on the row referenced by the cursor position.          *
//*    If cur.row == zero, then:                                                 *
//*      wcsUARROW --> wcsSTAB                                                   *
//*    If cur.row == (hgt - 1), then:                                            *
//*      wcsDARROW --> wcsTAB                                                    *
//*    Otherwise wcsDARROW and wcsUARROW are returned unmodified.                *
//*                                                                              *
//* Input  : wkey : 'special' key to test for conversion.                        *
//*                                                                              *
//* Returns: If keycode converted, returns converted keycode                     *
//*          If keycode NOT converted, keycode is returned unmodified            *
//********************************************************************************

wchar_t ACWin::specialkeyConvert ( wchar_t wkey )
{
   //* If only one field defined, then inter-field *
   //* keycods discarded.                          *
   if ( this->skf->fCnt == 1 )
   {
      if ( (wkey == wcsTAB)  || (wkey == wcsSTAB) ||
           (wkey == wcsPGUP) || (wkey == wcsPGDN) )
      {
         wkey = NULLCHAR ;
         //* If keycode is invalid in current context, *
         //* and if audible alert is enabled.          *
         if ( this->skf->beep )
            this->acBeep () ;
         return wkey ;           // Note the early return
      }
   }

   //* Multiple fields defined. Convert PgUp and PgDown.*
   else
   {
      if ( wkey == wcsPGDN )      { wkey = wcsTAB ; }
      else if ( wkey == wcsPGUP ) { wkey = wcsSTAB ; }
   }

   //* For single-row fields, convert UpArrow, DownArrow and SoftEnter.*
   if ( this->skf->fld[this->skf->fi].hgt == 1 )
   {
      if ( wkey == wcsDARROW )      { wkey = wcsTAB ; }
      else if ( wkey == wcsAENTER ) { wkey = wcsTAB ; }
      else if ( wkey == wcsUARROW ) { wkey = wcsSTAB ; }
   }

   //* For multi-row fields, UpArrow and DownArrow are a special case.*
   else if ( (wkey == wcsDARROW) || (wkey == wcsUARROW) )
   {
      skField *fptr = &this->skf->fld[this->skf->fi] ;
      if ( (wkey == wcsDARROW) && fptr->cur.row == (fptr->hgt - 1) )
         wkey = wcsTAB ;
      else if ( (wkey == wcsUARROW) && fptr->cur.row == ZERO )
         wkey = wcsSTAB ;
   }

   return wkey ;

}  //* End specialkeyConvert() *

//*************************
//*    specialkeyProc     *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Execute the operation associated with the specified "special" key.           *
//*                                                                              *
//* The target field is defined as the skField object referenced by the          *
//* 'fi' (index) into the skForm's fld[] array.                                  *
//*      Example: skField *fptr = &this->skf->fld[this->skf->fi] ;               *
//*                                                                              *
//* 1) For cursor movements, update the skForm 'ip' (insertion point) and        *
//*    'cur' (cursor offset) members. Then update the visible cursor position.   *
//*    Note on DownArrow and UpArrow keys: see Inter-field Movement, below.      *
//*                                                                              *
//* 2) For text-edit keys:                                                       *
//*    Delete:                                                                   *
//*    Overwrite the character referenced by 'ip' by shifting all trailing       *
//*    characters left by the width of that character.                           *
//*                                                                              *
//*    Backspace:                                                                *
//*    Shift the 'ip' one character to the left, then overwrite that             *
//*    character by shifting all trailing characters left by the width of        *
//*    that character.                                                           *
//*                                                                              *
//*    Insert:                                                                   *
//*    Toggle the 'ins' Insert/Overstrike flag, and if enabled, toggle the       *
//*    cursor style.                                                             *
//*                                                                              *
//*    Soft Enter:                                                               *
//*    In multi-row fields only, reformat the data by inserting a newline at     *
//*    'ip' and advancing 'ip' to the beginning of the next row.                 *
//*                                                                              *
//* 3) Inter-field Movement:                                                     *
//*    Tab and Shift+Tab:                                                        *
//*    If multiple fields are defined (fCnt > 1), then move to the next or       *
//*    previous field, respectively.                                             *
//*                                                                              *
//*    PageDown and PageUp:                                                      *
//*    Currently, these keycodes perform the same operation as Tab and           *
//*    Shift+Tab, respectively.                                                  *
//*                                                                              *
//*    DownArrow and UpArrow:                                                    *
//*    -- For single-row fields, these keycodes perform the same operation       *
//*       as Tab and Shift+Tab, respectively.                                    *
//*    -- For multi-row fields, these keycodes move one row down or up,          *
//*       respectively, UNLESS the field contains no additional rows(s) in       *
//*       the indicated direction; so in that case these keycodes perform the    *
//*       same operation as Tab and Shift+Tab, respectively.                     *
//*                                                                              *
//*                                                                              *
//* Input  : wkey : keycode to be processed                                      *
//*          skf  : pointer to the target skForm-class object                    *
//*                 skf.fi indexes the field with input focus.                   *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::specialkeyProc ( wchar_t wkey )
{
   skField *fptr = &this->skf->fld[this->skf->fi] ; // pointer to field with focus
   gString gsParse ;                // received parsed display text
   WinPos eorOff,                   // offset to end-of-row
          eotOff ;                  // offset to end-of-text
   short   ipEor,                   // insertion point corresponding to end-of-row
           ipEot ;                  // insertion point corresponding to end-of-text

   this->skf->pKey = wkey ;         // remember the keycode

   //********************************************
   //* Form-level, inter-field special keycodes *
   //********************************************
   if ( wkey == wcsTAB )                  //** TAB key - move to next field
   {
      this->Focus2NextField () ;
   }
   else if ( wkey == wcsSTAB )            //** SHIFT+TAB key - move to previous field
   {
      this->Focus2PrevField () ;
   }

   //***********************************
   //* Field-specific special keycodes *
   //***********************************
   else if ( wkey == wcsHOME )            //** HOME Key **
   {
      // Programmer's Note: Use a logical shortcut (as much as possible) for 
      // setting IP to the Home position based on the fact that "Home" is always 
      // offset 0:0. This saves CPU cycles.
      if ( fptr->ip > ZERO )
      {
         if ( fptr->cur.col > ZERO )
         {
            if ( (fptr->hgt == 1) || (fptr->cur.row == ZERO) )
            { fptr->ip = ZERO ; fptr->cur.row = fptr->cur.col = ZERO ; }
            else
            {
               fptr->cur = this->ipOffset ( fptr, ipoToR, &fptr->ip ) ;
            }
         }
         else
         { fptr->ip = ZERO ; fptr->cur.row = fptr->cur.col = ZERO ; }

         //* Update terminal-relative cursor position.*
         this->setCursor2IP ( fptr, this->txtBase ) ;
      }
      else if ( this->skf->beep )
         this->acBeep() ;
   }  // (wkey==wcsHOME)
   else if ( wkey == wcsEND )             //** END Key **
   {
      //* Get the display limit for text by parsing the source text.*
      ipEot = this->parseFieldText ( fptr, gsParse ) ;

      //* If field contains printable text, AND if  *
      //* insertion point is not yet at end-of-text.*
      if ( ((fptr->txt.gscols()) > ZERO) && (fptr->ip < ipEot) )
      {
         //* For single-row fields, go directly to end-of-text.*
         if ( fptr->hgt == 1 )      // single-row field
            fptr->cur = this->ipOffset ( fptr, ipoEoT, &fptr->ip ) ;

         //* For multi-row fields, first get IP for end of current *
         //* row, (this may or may not also be the end-of-text).   *
         //* If not already at end of current row, set IP to       *
         //* end-of-row. Else set IP to end-of-text.               *
         else                       // multi-row field
         {
            eorOff = this->ipOffset ( fptr, ipoEoR, &ipEor ) ;
            //* If not currently at E-O-R, reference E-O-R. *
            if ( fptr->ip < ipEor )
            { fptr->ip = ipEor ; fptr->cur = eorOff ; }
            //* Otherwise reference E-O-T. (see note above) *
            else
            {
               fptr->cur = this->ipOffset ( fptr, ipoEoT, &fptr->ip ) ;
            }
         }
         //* Update terminal-relative cursor position.*
         this->setCursor2IP ( fptr, this->txtBase ) ;
      }
      else if ( this->skf->beep )
         this->acBeep() ;
   }  // (wkey==wcsEND)
   else if ( wkey == wcsLARROW )          //** LEFT-ARROW Key **
   {
      if ( fptr->ip > ZERO )
      {
         //* Move the IP one character toward the top of text and calculate   *
         //* the cursor position. If IP returned in 'ipEor' != current IP, it *
         //* indicates that IP references a character which is outside the    *
         //* visible area. If the referenced character is a newline (likely), *
         //* adjust the IP to reference the last visible character of the row.*
         //* (Note that calculated cursor position will be correct.)          *
         --fptr->ip ;
         fptr->cur = this->ipOffset ( fptr, ipoCur, &ipEor ) ;
         if ( (ipEor != fptr->ip) && (fptr->txt.gstr()[fptr->ip] == NEWLINE) )
            --fptr->ip ;
         this->setCursor2IP ( fptr, this->txtBase ) ;
      }
      else if ( this->skf->beep )   // already at field origin
         this->acBeep() ;
   }  // (wkey==wcsLARROW)
   else if ( wkey == wcsRARROW )          //** RIGHT-ARROW Key **
   {
      //* Get the display limit for text by parsing the source text.*
      ipEot = this->parseFieldText ( fptr, gsParse ) ;

      if ( fptr->ip < ipEot )
      {
         //* Move the IP forward by one character and calculate the   *
         //* cursor position. If IP returned in 'ipEor' != current IP,*
         //* it indicates that IP references a character which is     *
         //* outside the visible area. If the referenced character is *
         //* a newline, move to beginning of next row.                *
         ++fptr->ip ;               // move IP forward by one character
         fptr->cur = this->ipOffset ( fptr, ipoCur, &ipEor ) ;
         if ( (ipEor != fptr->ip) && (fptr->txt.gstr()[fptr->ip] == NEWLINE) )
         {
            ++fptr->ip ;
            fptr->cur = this->ipOffset ( fptr ) ;
         }
         this->setCursor2IP ( fptr, this->txtBase ) ;
      }
      else if ( this->skf->beep )   // already at field origin
         this->acBeep() ;
   }  // (wkey==wcsRARROW)
   else if ( (wkey == wcsUARROW) || (wkey == wcsDARROW) ) //** UP-ARROW and DOWN-ARROW Keys **
   {
      //* Note: We see these keycodes ONLY when a multi-row field *
      //* has focus, AND when the IP references neither the top   *
      //* nor the bottom row of the field; however, we test for   *
      //* it anyway to avoid future problems.                     *
      if ( ((wkey == wcsUARROW) && (fptr->cur.row > ZERO)) ||
           ((wkey == wcsDARROW) && (fptr->cur.row < (fptr->hgt - 1))) )
      { this->skpInterRow ( fptr, wkey ) ; }
      else if ( this->skf->beep )
         this->acBeep() ;
   }
   else if ( wkey == wcsINSERT )          //** INSERT Key **
   {
      //* Toggle the Insert/Overstrike flag.*
      if ( this->skf->togg )     // if toggle enabled
      {
         this->skf->ins = this->skf->ins ? false : true ;

         //* If enabled, also toggle the style of the visible cursor.*
         if ( this->skf->insc )
            this->acSetCursorStyle ( (this->skf->ins ? this->skf->icur : this->skf->ocur) ) ;
      }
   }
   else if ( (wkey == wcsBKSP) ||         //** BKSP || DEL Key **
             (wkey == wcsDELETE) )
   {
      //* Prevents an attempt to either delete empty air or *
      //* to backspace beyond top-of-text.                  *
      if ( ((fptr->txt.gschars()) > 1) && 
           (((wkey == wcsBKSP) && (fptr->ip > ZERO)) ||
            ((wkey == wcsDELETE) && (fptr->txt.gstr()[fptr->ip] != NULLCHAR))) )
      {
         //* For Backspace, move IP one character toward head of text.*
         if ( wkey == wcsBKSP )
         {
            --fptr->ip ;
            fptr->cur = this->ipOffset ( fptr ) ;
         }

         //* Delete the character referenced by 'ip.         *
         //* This shifts all trailing characters left by one,*
         //* and removes any trailing newlines.              *
         this->skpDeleteChar ( fptr ) ;

         //* Update the display *
         this->RefreshField ( this->skf->fi ) ;
      }
      else if ( this->skf->beep )
         this->acBeep() ;
   }  // (wkey == wcsBKSP) || (wkey == wcsDELETE)
   else if ( (wkey == wcsAENTER) )
   {
      //* If target is a multi-row field _and_ not at end-of-text, *
      //* insert a soft-Enter i.e. a newline and reformat the data.*
      ipEot = this->parseFieldText ( fptr, gsParse ) ;
      if ( (fptr->hgt > 1) && (fptr->ip < ipEot) )
      {
         fptr->txt.insert( NEWLINE, fptr->ip++ ) ;
         //* Reformat the data and update the cursor offset.*
         this->formatFieldText ( fptr ) ;
         //* Update the display *
         this->RefreshField ( this->skf->fi ) ;
      }
      //* Not a valid key in single-row fields, nor *
      //* at the end-of-text in a multi-row field.  *
      else if ( this->skf->beep )
         this->acBeep() ;
   }  // (wkey == wcsAENTER)

   //* Unknown "special" keycode. This should never happen.*
   else if ( this->skf->beep )
      this->acBeep() ;

}  //* End specialkeyProc() *

//*************************
//*      skpInterRow      *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Called by specialkeyProc() when:                                             *
//* 1) Target field is a multi-row field, AND                                    *
//* 2) Either wcsDARROW or wcsUARROW is received, AND                            *
//* 3) IP is located within one of the interior rows of the field, i.e. neither  *
//*    the top row nor the bottom row of the field.                              *
//*                                                                              *
//* Calculate the 'ip' and 'cur' values for the vertical cursor movement         *
//* within the target field and position the physical cursor.                    *
//*                                                                              *
//* Input  : fptr  : pointer to target field                                     *
//*          wkey  : received keycode                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* Target is one row upward or one row downward and in the same column as in    *
//* the current row. The problems is that the target row may have no text in     *
//* that column, OR that column may not be the base column for the character     *
//* displayed at that position. For these reasons, the calculation is not        *
//* straightforward.                                                             *
//*                                                                              *
//********************************************************************************

void ACWin::skpInterRow ( skField *fptr, wchar_t wkey )
{
   #define DEBUG_skpInterRow (0)             // for debugging only

   wchar_t wbuff[gsMAXCHARS] ;               // work buffer
   const wchar_t *wptr = fptr->txt.gstr() ;  // pointer to data displayed in the field
   gString gsParse ;                         // text analysis
   short wIp = fptr->ip,                     // copy of current IP
         ipTor,                              // IP for top of row
         ipEor,                              // IP for end of row
         rowChars,                           // number of characters in target row
         rowCols,                            // number of columns in target row
         colAcc = ZERO ;                     // column-count accumulator
   const short *colArray ;                   // pointer to array of character widths

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_skpInterRow != 0
   if ( (ofsdbg.is_open()) )
   {
      gsdbg.compose( "%S FROM IP: '%S'", 
                     (wkey == wcsUARROW ? L"UP" : L"DN"), &wptr[fptr->ip] ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_skpInterRow

   if ( wkey == wcsUARROW )               // move one row upward
   {
      //* If IP reference a newline character, step back by one.*
      if ( wptr[wIp] == NEWLINE ) { --wIp ; }

      //* Scan backward to newline which indicates the end of the row above.*
      for ( ; (wptr[wIp] != NEWLINE) && (wIp > ZERO) ; --wIp ) ;

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_skpInterRow != 0
      if ( (ofsdbg.is_open()) )
      {
         gsdbg.compose( "FROM Eor: '%S'", &wptr[wIp] ) ;
         ofsdbg << gsdbg.ustr() << endl ;
      }
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_skpInterRow

      //* If referencing the newline, step back one more character.*
      if ( fptr->txt.gstr()[wIp] == NEWLINE ) --wIp ;
      ipEor = wIp ;                       // remember end-of-row

      //* Scan backward to beginning of target row.*
      for ( ; (wptr[wIp] != NEWLINE) && wIp > ZERO ; --wIp ) ;
      if ( wptr[wIp] == NEWLINE )   ++wIp ;

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_skpInterRow != 0
      gsdbg.compose( "FROM Tor: '%S'", &fptr->txt.gstr()[wIp] ) ;
      ofsdbg << gsdbg.ustr() << endl ;
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_skpInterRow

      //* Copy text of target row to work buffer *
      ipTor = wIp ;                       // source-data index
      for ( short i = ZERO ; i < gsMAXCHARS ; ++i )
      {
         wbuff[i] = wptr[ipTor] ;
         if ( ipTor > ipEor )
         { wbuff[i] = NULLCHAR ; break ; }
         ++ipTor ;
      }
      gsParse = wbuff ;                   // load text for target row
   }  // wkey==wcsUARROW

   else                                   // move one row downward
   {
      //* If not already at end of row, scan foreward to newline *
      //* which indicates the end of the current row.            *
      if ( wptr[wIp] != NEWLINE )
      { for ( ; (wptr[wIp] != NEWLINE) && (wptr[wIp] != NULLCHAR) ; ++wIp ) ; }

      //* If referencing the newline, step forward one more character.*
      if ( wptr[wIp] == NEWLINE ) ++wIp ;

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_skpInterRow != 0
      if ( (ofsdbg.is_open()) )
      {
         gsdbg.compose( "FROM Tor: '%S'", &wptr[wIp] ) ;
         ofsdbg << gsdbg.ustr() << endl ;
      }
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_skpInterRow

      //* Copy text of target row to work buffer *
      ipTor = wIp ;                       // source-data index
      for ( short i = ZERO ; i < gsMAXCHARS ; ++i )
      {
         wbuff[i] = wptr[ipTor] ;
         if ( (wbuff[i] == NEWLINE) || (wbuff[i] == NULLCHAR) )
         { wbuff[i] = NULLCHAR ; break ; }
         ++ipTor ;
      }
      gsParse = wbuff ;                   // load text for target row
   }  // wkey==wcsDARROW

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_skpInterRow != 0
   if ( (ofsdbg.is_open()) )
   {
      gsdbg.compose( "gsParse: '%S'", gsParse.gstr() ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_skpInterRow

   //* Scan forward on target row by the  *
   //* number of columns on original row. *
   if ( fptr->cur.col > ZERO )
   {
      rowCols = gsParse.gscols() ;
      colArray = gsParse.gscols( rowChars ) ;
      for ( short i = ZERO ; i < rowChars ; ++i )
      {
         ++wIp ;
         if ( ((colAcc += colArray[i]) >= fptr->cur.col) || (colAcc >= rowCols) )
            break ;
      }
   }
   fptr->ip = wIp ;                             // set the new IP
   fptr->cur = this->ipOffset ( fptr ) ;        // calculate the cursor offset
   this->setCursor2IP ( fptr, this->txtBase ) ; // position the physical cursor

   #undef DEBUG_skpInterRow
}  //* End skpInterRow() *

//*************************
//*     skpDeleteChar     *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Called by specialkeyProc() when erasing the character referenced by IP.      *
//*                                                                              *
//* If target field contains newline characters at or beyond the IP, remove      *
//* them from the data before reformatting the text. This prevents unintended    *
//* line breaks in the trailing data.                                            *
//*                                                                              *
//* Input  : fptr  : pointer to target field                                     *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::skpDeleteChar ( skField *fptr )
{
   #define DEBUG_skpDeleteChar (0)     // for debugging only

   short off ;                         // character offset

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_skpDeleteChar != 0
   if ( ofsdbg.is_open() )
   {
      gsdbg.compose( "skpDeleteChar( ip:%hd cur:%hd,%hd ", 
                     &fptr->ip, &fptr->cur.row, &fptr->cur.col ) ;
      if ( fptr->txt.gstr()[fptr->ip] == NEWLINE )
         gsdbg.append( "ipChar:nl )" ) ;
      else if ( fptr->txt.gstr()[fptr->ip] == NULLCHAR )
         gsdbg.append( "ipChar:nc )" ) ;
      else
         gsdbg.append( "ipChar:%C )", &fptr->txt.gstr()[fptr->ip] ) ;
      ofsdbg << gsdbg.ustr() << endl ;
      gsdbg = fptr->txt ;
      gsdbg.replace( NEWLINE, L'^', ZERO, false, true ) ;
      gsdbg.insert( "   a: " ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_skpDeleteChar

   //* Do not attempt to delete the null terminator.     *
   //* gString won't let you do it, but it's dumb to try.*
   if ( fptr->txt.gstr()[fptr->ip] == NULLCHAR )
      return ; // (note the early return)

   //* If no trailing newlines, simply delete the character referenced by IP.*
   if ( (off = fptr->txt.find( NEWLINE, fptr->ip )) < ZERO )
   {
      fptr->txt.erase( fptr->ip, 1 ) ;

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_skpDeleteChar != 0
      if ( ofsdbg.is_open() )
      {
         gsdbg = fptr->txt ;
         gsdbg.replace( NEWLINE, L'^', ZERO, false, true ) ;
         gsdbg.insert( "   b: " ) ;
         ofsdbg << gsdbg.ustr() << endl ;
      }
      #endif // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_skpDeleteChar
   }

   //* Delete the character referenced by IP, then *
   //* scan the positions of all trailing newlines.*
   else
   {
      //* Delete the character referenced by IP.                               *
      //* Special Case: If IP references a newline, step back by one character.*
      if ( (fptr->txt.gstr()[fptr->ip] == NEWLINE) && (fptr->ip > ZERO) )
         --fptr->ip ;
      fptr->txt.erase( fptr->ip, 1 ) ;
      off = fptr->ip ;                 // index start of newline scan

      //* Delete the trailing newline characters.*
      while ( (off = fptr->txt.find( NEWLINE, off)) > ZERO )
         fptr->txt.erase( off, 1 ) ;

      //* Re-parse the data into individual lines and update the cursor offset.*
      this->formatFieldText ( fptr ) ;

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_skpDeleteChar != 0
      if ( ofsdbg.is_open() )
      {
         gsdbg = fptr->txt ;
         gsdbg.replace( NEWLINE, L'^', ZERO, false, true ) ;
         gsdbg.insert( "   c: " ) ;
         ofsdbg << gsdbg.ustr() << endl ;
      }
      #endif // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_skpDeleteChar
   }

   #undef DEBUG_skpDeleteChar
}  //* End skpDeleteChar() *

//*************************
//*     normalkeyProc     *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Add a normal (printing, "graphing") character to the field with the          *
//* input focus (skf.fi member)                                                  *
//*                                                                              *
//* -- If Insert Mode, insert the new character into the text field, adjust      *
//*    the cursor position and refresh the text in the field.                    *
//* -- If Overstrike Mode, replace the character under the cursor with the       *
//*    new character, adjust the cursor position and refresh the text in the     *
//*    field.                                                                    *
//*                                                                              *
//* Note: The keycodes NEWLINE and NULLCHAR are control codes and should not     *
//*       arrive here, but if they do, they will be silently discarded.          *
//*                                                                              *
//* Input  : wkey : keycode to be processed                                      *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: This method will be updated after the sophisticated       *
//* auto-line-wrap algorithm is implemented.                                     *
//*                                                                              *
//********************************************************************************

void ACWin::normalkeyProc ( wchar_t wkey )
{
   #define DEBUG_normalkeyProc (0)     // for debugging only

   //* The following keycodes are not printing characters.*
   //* Note the early return. See note above.             *
   if ( (wkey == NEWLINE) || (wkey == NULLCHAR) )
      return ;

   skField *fptr = &this->skf->fld[this->skf->fi] ; // pointer to active field
   wchar_t refChar = fptr->txt.gstr()[fptr->ip] ;   // character referenced by 'ip'
   bool  delete_newlines = true ;                   // 'true' if deleting trailing newlines

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_normalkeyProc != 0
   wchar_t rc = ((refChar == NEWLINE) || (refChar == NULLCHAR)) ? L'^' : refChar,
           rc2 = rc == NULLCHAR ? NULLCHAR : fptr->txt.gstr()[fptr->ip + 1] ;
   if ( (rc2 == NEWLINE) || (rc2 == NULLCHAR) ) rc2 = L'^' ;
   if ( ofsdbg.is_open() )
   {
      gsdbg.compose( "normalkeyProc( wkey:%C ip:%2hd refChar:%C,%C cur:%2hd,%2hd )", 
                     &wkey, &fptr->ip, &rc, &rc2, &fptr->cur.row, &fptr->cur.col ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_normalkeyProc

   //* If IP references the null terminator, *
   //* insert the new character before it    *
   //* and advance the IP.                   *
   //* (data reformatting not necessary)     *
   if ( refChar == NULLCHAR )
   {
      fptr->txt.insert( wkey, fptr->ip++ ) ;
      delete_newlines = false ;
   }

   //* If IP references a newline character, *
   //* i.e. last character on current row.   *
   else if ( refChar == NEWLINE )
   {
      //* If cursor is at the right edge of the row, *
      //* replace the newline with the new character *
      //* and advance the IP past where the newline  *
      //* _will be_ after reformatting the data.     *
      if ( fptr->cur.col >= (fptr->wid - 1) )   // if at right edge of field
      {
         fptr->txt.replace( refChar, wkey, fptr->ip++ ) ;
         ++fptr->ip ;
      }
      //* If the cursor is not at the right edge of  *
      //* the row, insert the new character before   *
      //* the newline, which will push it forward.   *
      //* Advance IP to reference the newline.       *
      //* (data reformatting will not be necessary)  *
      else
      {
         fptr->txt.insert( wkey, fptr->ip++ ) ;
         delete_newlines = false ;
      }
   }

   //* Referenced character is presumed to be a printing character.        *
   //* Integrate the new character according to insert/overstrike criteria.*
   else if ( this->skf->ins ) // insert mode
   {
      fptr->txt.insert( wkey, fptr->ip++ ) ;
   }
   else                       // overstrike mode
   {
      //* If at the right edge of the row, the following *
      //* character will be newline or null terminator.  *
      if ( fptr->cur.col >= (fptr->wid - 1) )   // if at right edge of field
      {
         //* If the following character is the null terminator, *
         //* insert the character before the nullchar.          *
         if ( fptr->txt.gstr()[fptr->ip + 1] == NULLCHAR )
            fptr->txt.insert( wkey, fptr->ip++ ) ;
         //* Replace the newline with the new character *
         //* and advance the IP past where the newline  *
         //* _will be_ after reformatting the data.     *
         else
         {
            fptr->txt.replace( refChar, wkey, fptr->ip++ ) ;
            ++fptr->ip ;
         }
      }
      else                                      // not at right edge
      {
         //* Replace the character referenced by IP with the new character.*
         fptr->txt.replace( refChar, wkey, fptr->ip++ ) ;
         //* If the width of new character == width of character *
         //* being replaced, no formatting update required.      *
         if ( (wcwidth ( refChar )) == (wcwidth ( wkey )) )
            delete_newlines = false ;
      }
   }
   
   //* Delete trailing newlines *
   if ( delete_newlines )
   {
      short off = fptr->ip ;
      while ( (off = fptr->txt.find( NEWLINE, off)) > ZERO )
         fptr->txt.erase( off, 1 ) ;
   }

   //* Reformat the data and update the cursor offset.*
   this->formatFieldText ( fptr ) ;

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_normalkeyProc != 0
   rc  = fptr->txt.gstr()[fptr->ip] ;
   rc2 = fptr->txt.gstr()[fptr->ip + 1] ;
   if ( (rc  == NEWLINE) || (rc  == NULLCHAR) ) rc  = L'^' ;
   if ( (rc2 == NEWLINE) || (rc2 == NULLCHAR) ) rc2 = L'^' ;
   if ( ofsdbg.is_open() )
   {
      gsdbg.compose( "             (        ip:%2hd refChar:%C,%C cur:%2hd,%2hd )", 
                     &fptr->ip, &rc, &rc2, &fptr->cur.row, &fptr->cur.col ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_normalkeyProc

   //* Refresh the display and set visible cursor to new position.*
   this->RefreshField ( this->skf->fi ) ;

}  //* End normalkeyProc() *

#if 0    // CURRENTLY UNUSED
//*************************
//*       wpos2tpos       *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Convert field-relative positions to terminal-relative positions.             *
//* Called by EditForm().                                                        *
//*                                                                              *
//* Input  : wpul : (by reference) origin (upper-left) of field                  *
//*          wplr : (by reference) lower-right of field                          *
//*          wpet : (by reference) end-of-text (position of null terminator)     *
//*          wpcu : (by reference) current cursor position                       *
//*                                                                              *
//* Returns: pointer to field with input focus                                   *
//********************************************************************************

   //* Convert field-relative positions to terminal-relative positions.*
   skField* wpos2tpos ( WinPos& wpul, WinPos& wplr, WinPos& wpet, WinPos& wpcu ) ;
skField* ACWin::wpos2tpos ( WinPos& wpul, WinPos& wplr, WinPos& wpet, WinPos& wpcu )
{
   #define DEBUG_wpos2tpos (0)
   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_wpos2tpos != 0
   if ( (ofsdbg.is_open()) )
   {
      gsdbg.compose( "fi:%hd base:%02hd/%02hd fCnt:%hd\n"
                     "  hgt:%02hd wid:%02hd orig:%02hd/%02hd cur:%02hd/%02hd", 
                     &this->skf->fi,
                     &this->skf->base.row, &this->skf->base.col,
                     &this->skf->fCnt,
                     &this->skf->fld[this->skf->fi].hgt,
                     &this->skf->fld[this->skf->fi].wid,
                     &this->skf->fld[this->skf->fi].orig.row,
                     &this->skf->fld[this->skf->fi].orig.col,
                     &this->skf->fld[this->skf->fi].cur.row,
                     &this->skf->fld[this->skf->fi].cur.col ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_wpos2tpos

   skField *fptr = &this->skf->fld[this->skf->fi] ;   // pointer to active field

   wpul = { short(this->txtBase.row + fptr->orig.row),
            short(this->txtBase.col + fptr->orig.col) },
   wplr = { short(wpul.row + fptr->hgt - 1),
            short(wpul.col + fptr->wid - 1) },
   wpcu = { short(wpul.row + fptr->cur.row),
            short(wpul.col + fptr->cur.col) } ;

   //* Calculate position for end-of-text.*
   if ( fptr->hgt == 1 )               // single-row field
   {
      wpet.row = wpul.row ;
      wpet.col = wpul.col + fptr->txt.gscols() ;
      if ( wpet.col > wplr.col )       // prevent field overrun
         wpet.col = wplr.col ;
   }
   else                                // multi-row field
   {
      //* Determine how many rows contain text data.*
      short txtRows  = 1,
            nlindx   = ZERO ;
      while ( (nlindx = fptr->txt.after( NEWLINE, nlindx )) >= ZERO )
         ++txtRows ;
   }
   return fptr ;

}  //* End wpos2tpos() *

//*************************
//*       tpos2wpos       *
//*************************
//********************************************************************************
//* Non-member Method:                                                           *
//* ------------------                                                           *
//* Convert terminal-relative positions to field-relative positions.             *
//* Called by EditForm().                                                        *
//*                                                                              *
//* Input  : fptr : pointer to field with input focus                            *
//*          wpul : (by reference) field origin (terminal-relative)              *
//*          wpcu : (by reference) current cursor position (terminal-relative)   *
//*                 Converted value stored in 'cur' member of field with focus   *
//*                 as a 0-based offset from field origin.                       *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

   //* Convert terminal-relative positions to field-relative positions.*
   void tpos2wpos ( skField *fptr, const WinPos& wpul, const WinPos& wpcu ) ;
void ACWin::tpos2wpos ( skField *fptr, const WinPos& wpul, const WinPos& wpcu )
{
   if ( fptr->hgt == 1 )               // single-row field
   {
      fptr->cur.col = wpcu.col - this->txtBase.col - fptr->orig.col ; 
   }
   else                                // multi-row field
   {
   }

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
   if ( (ofsdbg.is_open()) )
   {
      gsdbg.compose( "tpos2wpos( %02hd/%02hd --> %02hd/%02hd",
                     &wpcu.row, &wpcu.col, &fptr->cur.row, &fptr->cur.col ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG
}  //* End tpos2wpos() *
#endif   // CURRENTLY UNUSED

//*************************
//*       clearWin        *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Clear the window area using the specified background color attribute.        *
//*                                                                              *
//* Input  : acaEx : specifies the background color attribute to be used when    *
//*                  clearing the area. (this is an initialized acaExpand object)*
//*          all   : (optional, 'false' by default) specify target area          *
//*                  if 'false'; erase the interior area only                    *
//*                  if 'true',  erase the entire window including border        *
//*                                                                              *
//* Returns: nothing                                                             *
//*          Important Note: On return, the background attribute remains set to  *
//*                          the specified attribute, and 'txtOff' is set to     *
//*                          origin (upper-left corner of text area).            *
//********************************************************************************

void ACWin::clearWin ( const acaExpand& acaEx, bool all )
{
   short ulr = this->txtBase.row,
         ulc = this->txtBase.col,
         hgt = this->txtRows,
         wid = this->txtCols ;

   //* If border is also to be erased *
   if ( all != false )
   {
      ulr = this->winBase.row,
      ulc = this->winBase.col,
      hgt = this->height,
      wid = this->width ;
   }

   //* Set the background color attribute *
   this->setAttributes ( acaEx, false, true ) ;

   //* Clear the area *
   this->acClearArea ( ulr, ulc, hgt, wid ) ;

   //* Set cursor position *
   this->txtOff = { 0, 0 } ;

}  //* End clearWin() *

//*************************
//*       SetTitle        *
//*************************
//********************************************************************************
//* Set the window title.                                                        *
//* Draw the title centered in the top border of the window.                     *
//* Example:      ┌───────┤ ACWin-class Window ├───────┐                         *
//* -- It is recommended that the first and last characters of the text be       *
//*    space characters as shown in the example.                                 *
//* -- The border will occupy the remainder of the top row, with terminator      *
//*    characters on each end.                                                   *
//* -- If necessary, the specified text will be truncated to fit the available   *
//*    space. In this case, the returned value will be 'false'.                  *
//* -- Optionally, the text can be rendered in an alternate color.               *
//* -- Optionally, a new border style can be specified. This parameter is not    *
//*    very useful, but it was implemented for testing purposes and remains      *
//*    available.                                                                *
//*                                                                              *
//* Input  : newTitle : text of window title                                     *
//*                     (empty string or null pointer erases title)              *
//*          titAttr  : (optional, existing border attribute by default)         *
//*                     If specified, this is a logical OR of acAttr values      *
//*                     specifying the foreground/background color attributes.   *
//*                     See the 'acAttr' enumerated type for details.            *
//*          bdrLine  : (optional, ltHORIZ i.e. invalid line type by default)    *
//*                     if specified, new border style: ltSINGLE or ltDUAL       *
//*                                                                              *
//* Returns: 'true'  if title set as specified                                   *
//*          'false' if title truncated to fit window width                      *
//********************************************************************************

bool ACWin::SetTitle ( const char* newTitle, acAttr titAttr, LineType bdrLine )
{
   gString gs( newTitle ) ;               // text formatting
   short maxTit = this->width - 4 ;       // max columns available for title text
   bool  status = true ;                  // return value

   if ( (gs.gscols()) > maxTit )          // if necessary, truncate to fit
   {
      gs.limitCols( maxTit ) ;
      status = false ;
   }
   gs.copy( this->title, gsMAXCHARS ) ;   // save the new title

   //* Optionally update the border line style *
   if ( (bdrLine == ltSINGLE) || (bdrLine == ltDUAL) )
      this->bdrStyle = bdrLine ;

   //* Redraw the border with the new title.*
   this->drawBorder ( titAttr ) ;

   //* Restore current terminal attributes *
   this->restoreAttributes () ;

   return status ;

}  //* End SetTitle() *

//*************************
//*      SetTextAttr      *
//*************************
//********************************************************************************
//* Set the color attributes and modifiers for the window text area.             *
//* This method decodes and stores the new attributes for the text area of       *
//* the window. New text written to the text area will be displayed using        *
//* the specified attributes.                                                    *
//*                                                                              *
//* Important Note: The terminal attributes are not updated here. The specified  *
//*                 attributes will be used for _subsequent_ writes.             *
//*                                                                              *
//* Input  : txtAttr : (acAttr bitfield) a logical OR of forground and           *
//*                    background color attributes and attribute modifiers       *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::SetTextAttr ( acAttr txtAttr )
{

   //* Decode the text attribute.*
   this->txtAttr.decode( txtAttr ) ;

}  //* End SetTextAttr() *

//*************************
//*     SetBorderAttr     *
//*************************
//********************************************************************************
//* Set the color attributes and modifiers for the window border and redraw      *
//* the border. Optionally, sets the border style.                               *
//*                                                                              *
//* On return, the cursor is positioned in the field with input focus.           *
//*                                                                              *
//* This method strips the text modifiers that make no sense for the border.     *
//* Only the 'bold text' modifier makes any sense in this context.               *
//*                                                                              *
//* Input  : bdrAttr : (acAttr bitfield) a logical OR of forground and           *
//*                    background color attributes and attribute modifiers       *
//*          bdrStyle: (optional, ltHORIZ i.e. an invalid value by default)      *
//*                    specify the border line style, either ltSINGLE or ltDUAL  *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::SetBorderAttr ( acAttr bdrAttr, LineType bdrStyle )
{
   //* For borders, some text-modifier attributes     *
   //* are meaningless or just plain ugly. Clear them.*
   bdrAttr = acAttr(bdrAttr & 
                    ~(acaITALIC | acaUNDERLINE | acaOVERLINE | 
                      acaXOUT | acaCONCEAL)) ;

   //* Decode the border attribute.*
   this->bdrAttr.decode( bdrAttr ) ;

   //* Redraw the border *
   this->drawBorder () ;

   //* Restore fgnd/bkgnd attributes to current terminal attributes.*
   this->restoreAttributes () ;

}  //* End SetBorderAttr() *

//*************************
//*         Write         *
//*************************
//********************************************************************************
//* Write text to the interior of the window.                                    *
//*                                                                              *
//* This group of methods writes directly into the window's text area,           *
//* bypassing any defined text fields. The text data are not saved, so if the    *
//* window is moved or hidden/restored, the displayed text data will be lost.    *
//*                                                                              *
//* Important Note:                                                              *
//* ---------------                                                              *
//* This method employs an extremely simplistic automatic line break algorithm.  *
//* -- This algorithm breaks lines at the right edge and bottom edge of the      *
//*    text area.                                                                *
//* -- The break algorithm is character-based rather than word-based or          *
//*    syntax-based as a more sophisticated algorithm would employ.              *
//* For this reason, it is STRONGLY RECOMMENDED that the caller pre-format       *
//* any necessary line breaks to ensure the quality of the text formatting,      *
//* and that the text does not violate the boundaries of the text area.          *
//*                       -----------------                                      *
//*                                                                              *
//* Input  : txtPos : WinPos-class object indicates the zero-based               *
//*                   OFFSET from upper-left corner of window interior           *
//*          txtData: text to write in one of the following formats:             *
//*                   -- gString object by reference                             *
//*                   -- wchar_t pointer                                         *
//*                   -- char pointer                                            *
//*          txtAttr: (optional, default: existing text attribute (acaUSEDFLT))  *
//*                   text attributes and text modifiers for this write ONLY.    *
//*                   Window attributes unchanged.                               *
//*          clear  : (optional, 'false' by default)                             *
//*                   if 'false', existing displayed text unmodified             *
//*                   if 'true',  erase window interior before write             *
//*                               (erased using existing bgnd color)             *
//*                                                                              *
//* Returns: current cursor position (window offset)                             *
//********************************************************************************
//* Programmer's Note: We _could_ use the parseFieldText() method for output     *
//* formatting; however, that would limit the text length to gsMAXCHARS.         *
//* That is why we do the output formatting locally.                             *
//********************************************************************************

WinPos ACWin::Write ( WinPos txtPos, const wchar_t* txtData, acAttr txtAttr, bool clear )
{
   const wchar_t errCHAR = L'?' ;         // indicator for non-printing characters
   const short wbLEN = this->txtRows * this->txtCols + this->txtRows * 2 ;
   const short wbOVR = wbLEN - 2 ;        // loop control

   wchar_t *wbuff = new wchar_t[wbLEN] ;  // work buffer (dynamic allocation)
   gString gs ;                           // text formatting and analysis
   wchar_t wchar ;                        // character under test
   short srcIndx = ZERO,                  // index into 'txtData' array
         trgIndx = ZERO,                  // index into 'wbuff' array
         rowCols = ZERO,                  // column accumulator
         charCols ;                       // column width of character under test

   //* Range check the provided offsets *
   if ( (txtPos.row < ZERO) || (txtPos.row >= this->txtRows) )
      txtPos.row = ZERO ;
   if ( (txtPos.col < ZERO) || (txtPos.col >= this->txtCols) )
      txtPos.col = ZERO ;

   //* Set local offset-tracking member to initial position.  *
   //* Note that the final 'txtOff' value is the return value.*
   this->txtOff = txtPos ;
   short baseRow = this->txtOff.row,      // base row offset
         baseCol = this->txtOff.col ;     // base column offset
   short grpRows = ZERO,                  // number of data rows in output group
         capRows = ZERO ;                 // number of data rows currently in wbuff[]

   //* Calculate number of data rows in an output group *
   for ( ; grpRows < this->txtRows ; ++grpRows )
   {
      if ( (grpRows * this->txtCols) >= (gsMAXCHARS - grpRows) )
      { --grpRows ; break ;}
   }

   //* Convert window-relative zero-based offset to *
   //* terminal-window one-based offset.            *
   WinPos wp( this->txtBase.row + this->txtOff.row,
              this->txtBase.col + this->txtOff.col ) ;

   //* If specified, clear window interior (use stored attributes).*
   if ( clear != false )
      this->clearWin ( this->txtAttr ) ;

   //* Set the text attributes, either previously established interior  *
   //* attribute, or alternate attributes specified for this write only.*
   if ( txtAttr == acaUSEDFLT )
      this->setAttributes ( this->txtAttr ) ;
   else
   {
      acaExpand altAttr( txtAttr ) ;
      this->setAttributes ( altAttr ) ;
   }

   //* Perform any necessary automagic line-wrapping *
   //* to keep text within the window's text area.   *
   while ( txtData[srcIndx] != NULLCHAR )
   {
      //* Read and analyze a source character *
      if ( (wchar = txtData[srcIndx++]) == NEWLINE )
      {
         wbuff[trgIndx++] = wchar ;    // add the newline
         rowCols = ZERO ;              // reset the accumulator
         if ( ++baseRow >= this->txtRows ) // reference next row
            break ;
      }

      //* All other characters (not '\n' or '\0') *
      else
      {
         //* If a control character, substitute    *
         //* a '?' indicating an invalid character.*
         if ( (iswcntrl ( wchar )) != ZERO )
            wchar = errCHAR ;

         //* Get the character width. If the column sum fits within *
         //* the display space, add the character to the row.       *
         // Programmer's Note: For the wcwidth() function, non-printing 
         // characters are reported as having a negative width. The null 
         // character has a zero width. Printing characters have a width >= zero.
         if ( (charCols = wcwidth ( wchar )) < ZERO ) charCols = ZERO ;

         if ( (baseCol + rowCols + charCols) <= this->txtCols )
         {
            wbuff[trgIndx++] = wchar ;
            rowCols += charCols ;
         }
         //* Otherwise the character will be the *
         //* first character of the next row.    *
         else
         {
            --srcIndx ;                // return source index to current character
            wbuff[trgIndx++] = L'\n' ; // add a newline
            rowCols = ZERO ;           // reset the accumulator

            //* If max output buffer size, output currently captured *
            //* data and reset the counters.                         *
            if ( ++capRows == grpRows )
            {
               wbuff[trgIndx] = NULLCHAR ;
               wp = this->acWrite ( wp, wbuff ) ;
               trgIndx = ZERO ;
               capRows = ZERO ;
            }

            if ( ++baseRow >= this->txtRows ) // reference next row
               break ;
         }
      }
      if ( trgIndx >= wbOVR ) break ;  // prevent buffer overflow
   }  // while()
   wbuff[trgIndx] = NULLCHAR ;         // terminate the string

   //* Write the text data into the window *
   if ( trgIndx > ZERO )
      wp = this->acWrite ( wp, wbuff ) ;

   delete [] wbuff ;                      // release the dynamic alocation

   //* Convert terminal-oriented cursor position to window offset.*
   this->txtOff = { short(wp.row - this->txtBase.row), 
                    short(wp.col - this->txtBase.col) } ;

   //* Restore fgnd/bkgnd attributes to current terminal attributes.*
   this->acSetFgBg ( this->termAttr.aesFgnd, this->termAttr.aesBgnd ) ;

   return ( this->txtOff ) ;              // return the current cursor offset

}  //* End Write() *

WinPos ACWin::Write ( WinPos txtPos, const gString& txtData, acAttr txtAttr, bool clear )
{

   return ( this->Write ( txtPos, txtData.gstr(), txtAttr, clear )  ) ;

}  //* End Write() *

WinPos ACWin::Write ( WinPos txtPos, const char* txtData, acAttr txtAttr, bool clear )
{

   gString gs( txtData ) ;
   return ( this->Write ( txtPos, gs.gstr(), txtAttr, clear )  ) ;

}  //* End Write() *

//*************************
//*     GetCursorPos      *
//*************************
//********************************************************************************
//* Get the current cursor offset within the window's text area.                 *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: current cursor position (window offset)                             *
//********************************************************************************

WinPos ACWin::GetCursorPos ( void )
{

   return ( this->txtOff ) ;

}  //* End GetCursorPos() *

#if 0    // CURRENTLY UNUSED
//*************************
//*     GetCursorPos      *
//*************************
//********************************************************************************
//* Get the current cursor offset within the specified field.                    *
//*                                                                              *
//* Input  : fldIndx : index of target field                                     *
//*                                                                              *
//* Returns: current cursor position (field offset)                              *
//*          (returns 0,0 if 'fldIndx' is out-of-range)                          *
//********************************************************************************

   //* Get the current cursor offset within the specified field.        *
   //*                                                                  *
   //* Input  : fldIndx : index of target field                         *
   //* Returns: current cursor position (field offset)                  *
   WinPos GetCursorPos ( short fldIndx ) ;

WinPos ACWin::GetCursorPos ( short fldIndx )
{
// This method is useless. It's useful to return IP, but not cursor position.
   WinPos wp( 0, 0 ) ;
   if ( (fldIndx > ZERO) && (fldIndx < this->skf->fCnt) )
      wp = this->skf->fld[fldIndx].cur ;
   return ( wp ) ;

}  //* End GetCursorPos() *
#endif   // CURRENTLY UNUSED

//*************************
//*     SetCursorPos      *
//*************************
//********************************************************************************
//* Set the cursor offset within the window's text area.                         *
//*                                                                              *
//* Input  : wPos : (by reference) offset into the text area of the window       *
//*                 (0-based offset)                                             *
//*                                                                              *
//* Returns: current cursor position (window offset)                             *
//********************************************************************************

WinPos ACWin::SetCursorPos ( const WinPos& wPos )
{
   this->txtOff = { ZERO, ZERO } ;           // initialize the offset

   if ( (wPos.row >= ZERO) && (wPos.row < this->txtRows) )  // Range check
      this->txtOff.row = wPos.row ;
   if ( (wPos.col >= ZERO) && (wPos.col < this->txtCols) )
      this->txtOff.col = wPos.col ;

   //* Set "absolute" i.e. terminal- relative cursor position.*
   this->acSetCursor ( (this->txtBase.row + this->txtOff.row), 
                       (this->txtBase.col + this->txtOff.col) ) ;

   return ( this->txtOff ) ;

}  //* End SetCursorPos() *

//* Overload using individual row/column arguments.*
WinPos ACWin::SetCursorPos ( short rOff, short cOff )
{

   WinPos wp( rOff, cOff ) ;
   return ( (this->SetCursorPos ( wp )) ) ;

}  //* End SetCursorPos() *

//*************************
//*         Write         *
//*************************
//********************************************************************************
//* Write text to the specified field within the window.                         *
//*                                                                              *
//* This group of methods inserts the specified text into the existing text      *
//* of the target field at the specified offset ('ip'); or if the 'clear' flag   *
//* is set, replaces the existing text. The field is then redrawn to display     *
//* the updated data.                                                            *
//*                                                                              *
//* Input  : fIndx  : index of target field                                      *
//*          txt    : text to write in one of the following formats:             *
//*                   -- gString object by reference                             *
//*                   -- wchar_t pointer                                         *
//*                   -- char pointer                                            *
//*          offset : (optional, -1 by default, indicationg that the             *
//*                    current insertion point ('ip') should be used)            *
//*                   character offset into existing text at which to insert     *
//*                   the new text                                               *
//*          offset : (optional, current IP (insertion point) by default)        *
//*                   character offset into existing text at which to insert     *
//*                   the new text                                               *
//*          clear  : (optional, 'false' by default)                             *
//*                   if 'false', new text will be added to the existing text    *
//*                               as described                                   *
//*                   if 'true',  new text will replace the existing text        *
//*          txtAttr: (optional, acaUSEDFLT by default)                          *
//*                   (existing text attributes will be used)                    *
//*                   If specified, color attributes and text modifiers          *
//*                                                                              *
//* Returns: current insertion point ('ip'), that is, the character offset       *
//*          at which the cursor is positioned. This will be at the point        *
//*          following the inserted text, or at the last character cell of the   *
//*          field. (or zero if invalid field index specified)                   *
//********************************************************************************
//* Note: If 'offset' < zero, use current 'ip'.                                  *
//*       If 'offset' > length of existing text (beyond null terminator),        *
//*                     append new text to existing text                         *
//*       In the unlikely event that the combined length of the existing text    *
//*       and new text > gsMAXCHARS, the gString object will automatically       *
//*       truncate the excess.                                                   *
//********************************************************************************

short ACWin::Write ( short fIndx, const wchar_t* txtData, short offset, 
                     bool clear, acAttr txtAttr )
{
   short newIp = ZERO ;                   // return value

   if ( (fIndx >= ZERO) && (fIndx < this->skf->fCnt) )
   {
      gString gsOut,                      // work buffer
              gsNew( txtData ) ;          // new text data

      //* Point to the target skField object *
      skField *fptr = &this->skf->fld[fIndx] ;

      //* If specified, replace the current text with the new text, *
      //* and set the 'ip' to end of text.                          *
      if ( clear )
      {
         gsOut = txtData ;
         fptr->ip = gsOut.gschars() - 1 ; // index of null terminator
      }
      //* Insert the new text into the existing text.*
      else
      {
         gsOut = fptr->txt ;              // copy of current data
         if ( offset < ZERO )             // use current IP
            offset = fptr->ip ;
         else if ( offset > ((fptr->txt.gschars()) - 1) ) // append to existing text
            offset = fptr->txt.gschars() - 1 ;
         gsOut.insert( gsNew.gstr(), offset ) ;
      }
      fptr->ip = offset + gsNew.gschars() - 1 ; // reference end of inserted text
      if ( fptr->ip >= gsOut.gschars() )  // limit to index of null terminator
         fptr->ip = gsOut.gschars() - 1 ;

      //* Verify new IP and calculate cursor position.  *
      //* If new IP is beyond the physical limit of     *
      //* the field, returned value will be normalized. *
      fptr->cur = this->ipOffset ( fptr, ipoCur, &newIp ) ;
      if ( newIp == fptr->ip )
      {
         fptr->txt = gsOut ;              // update the text data

         //* If specified, set new color attributes *
         if ( txtAttr != acaUSEDFLT )
            fptr->aca.decode( txtAttr ) ;

         //* Refresh the display of the field *
         this->RefreshField ( this->skf->fi ) ;

         //* Reset attributes to terminal values *
         this->restoreAttributes () ;
      }
      else     // Holy calculation error, Batman!
         newIp = ZERO ;
   }
   return newIp ;

}  //* End Write() *

short ACWin::Write ( short fIndx, const gString& txtData, short offset, 
                     bool clear, acAttr txtAttr )
{

   return ( this->Write ( fIndx, txtData.gstr(), offset, clear, txtAttr  )  ) ;

}  //* End Write() *

short ACWin::Write ( short fIndx, const char* txtData, short offset, 
                     bool clear, acAttr txtAttr )
{

   gString gs( txtData ) ;
   return ( this->Write ( fIndx, gs.gstr(), offset, clear, txtAttr )  ) ;

}  //* End Write() *

//*************************
//*      drawBorder       *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Draw the window border using the border style and color attributes stored    *
//* in the 'bdrStyle' and'bdrAttr' members (includes title (if defined)).        *
//*                                                                              *
//* Important Note: This methods sets, but DOES NOT RESTORE color attributes.    *
//*                 It is the caller's responsibility to adjust color attributes *
//*                 as desired after returning from this call.                   *
//*                                                                              *
//* Input  : titAttr: (optional, existing border attribute by default)           *
//*                   If specified, this is the alternate color for title text.  *
//*                   The border itself is drawn in the existing colors.         *
//*                                                                              *
//* Returns: nothing                                                             *
//*          Note: On return the fg/bg attributes are either the border colors   *
//*                _OR_ the specified alternate color attribute.                 *
//********************************************************************************

void ACWin::drawBorder ( acAttr titAttr )
{
   gString gsOut ;                        // text formatting
   wchar_t ul, ur, ll, lr, h, v, lt, rt ; // characters for drawing border

   //* Initialize the drawing characters according to the line type *
   switch ( this->bdrStyle )
   {
      case ltDUAL:                     // dual line
         ul = wcsULd ;
         ur = wcsURd ;
         ll = wcsLLd ;
         lr = wcsLRd ;
         h  = wcsHORIZd ;
         v  = wcsVERTd ;
         lt = wcsLTEEd ;
         rt = wcsRTEEd ;
         break ;
      case ltSINGLEBOLD:               // single-line bold
      case ltDASH2BOLD:                // two-dash bold
      case ltDASH3BOLD:                // three-dash bold
      case ltDASH4BOLD:                // four-dash bold
         ul = wcsULb ;
         ur = wcsURb ;
         ll = wcsLLb ;
         lr = wcsLRb ;
         lt = wcsLTEEb ;
         rt = wcsRTEEb ;
         switch ( this->bdrStyle )
         {
            case ltDASH2BOLD: h = wcsHDASH2b ; v = wcsVDASH2b ; break ;
            case ltDASH3BOLD: h = wcsHDASH3b ; v = wcsVDASH3b ; break ;
            case ltDASH4BOLD: h = wcsHDASH4b ; v = wcsVDASH4b ; break ;
            default:          h = wcsHORIZb ;  v = wcsVERTb ;   break ;
         } ;
         break ;
         break ;
      case ltHORIZ:                    // draw border with spaces only
      case ltVERT:                     // i.e. no border
         ul = ur = ll = lr = h = v = lt = rt = SPACE ;
         break ;
      case ltSINGLE:                   // single-line normal
      case ltDASH2:                    // two-dash
      case ltDASH3:                    // three-dash
      case ltDASH4:                    // four-dash
      default:
         ul = wcsULs ;
         ur = wcsURs ;
         ll = wcsLLs ;
         lr = wcsLRs ;
         lt = wcsLTEEs ;
         rt = wcsRTEEs ;
         switch ( this->bdrStyle )
         {
            case ltDASH2BOLD: h = wcsHDASH2s ; v = wcsVDASH2s ; break ;
            case ltDASH3BOLD: h = wcsHDASH3s ; v = wcsVDASH3s ; break ;
            case ltDASH4BOLD: h = wcsHDASH4s ; v = wcsVDASH4s ; break ;
            default:          h = wcsHORIZs ; v = wcsVERTs ;    break ;
         } ;
         break ;
   } ;

   //* Set the border color attributes and optional text modifiers.*
   this->setAttributes ( this->bdrAttr ) ;

   //* Draw top border *
   gsOut.compose( "%C", &ul ) ;
   gsOut.padCols( this->width - 1, h ) ;
   gsOut.append( "%C", &ur ) ;
   this->acSetCursor ( this->winBase ) ;
   this->ttyWrite ( gsOut ) ;

   //* Create the vertical bar for left and right borders.*
   //* Draw the right border, then the left border.       *
   gsOut.clear() ;
   for ( short irows = 2 ; irows < this->height ; ++irows )
      gsOut.append( "%C\n", &v ) ;
   this->acWrite ( (this->winBase.row + 1), (this->winBase.col + this->width - 1), gsOut, false ) ;
   this->acWrite ( (this->winBase.row + 1), this->winBase.col, gsOut, false ) ;

   //* Draw bottom border *
   gsOut.compose( "%C", &ll ) ;
   gsOut.padCols( this->width - 1, h ) ;
   gsOut.append( "%C", &lr ) ;
   this->acSetCursor ( (this->winBase.row + this->height - 1), this->winBase.col ) ;
   this->ttyWrite ( gsOut ) ;

   //* If a title was specified, position it in the top border *
   if ( this->title[0] != '\0' )
   {  //* Center of top border row *
      WinPos wpb( this->winBase.row, (this->winBase.col + (this->width / 2)) ) ;

      gsOut = this->title ;                     // get title text
      gsOut.limitCols( this->width - 4 ) ;      // truncate title text if necessary
      gsOut.insert( rt ) ;                      // title delimeters
      gsOut.append( lt ) ;
      wpb.col -= gsOut.gscols() / 2 ;           // left column of title text
      if ( wpb.col <= this->winBase.col )       // (may be off-by-one)
         wpb.col = this->winBase.col + 1 ;
      this->acWrite ( wpb, gsOut, false ) ;     // write using border colors
      if ( titAttr != acaUSEDFLT )              // if using border colors
      {
         acaExpand tmpAttr ;
         tmpAttr.decode( titAttr ) ;
         this->setAttributes ( titAttr ) ;
         gsOut.shiftChars( -1 ) ;               // remove the delimiters
         gsOut.limitChars( gsOut.gschars() - 2 ) ;
         ++wpb.col ;
         this->acWrite ( wpb, gsOut, false ) ;  // write text using alt colors
      }
   }
}  //* End drawBorder()*

//*************************
//*   restoreAttributes   *
//*************************
//********************************************************************************
//* Private Method:                                                              *
//* ---------------                                                              *
//* Set the color attributes and modifiers to stored terminal-window values.     *
//*                                                                              *
//* The setting stored in the 'termAttr' member are assumed to be correct.       *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void ACWin::restoreAttributes ( void )
{

   this->setAttributes ( this->termAttr ) ;

}  //* End restoreAttributes() *

//*************************
//*        DrawBox        *
//*************************
//********************************************************************************
//* Draw a rectangle within the window's text area.                              *
//*                                                                              *
//* Input  : pos   : offset from upper-left corner of text area to               *
//*                  upper-left corner of box (0-based)                          *
//*          height: number of rows for box                                      *
//*          width : number of columns for box                                   *
//*          lType : (optional, ltSINGLE by default) line style                  *
//*                  member of enum LineType: (ltSINGLE or ltDUAL)               *
//*          bxAttr: (optional, acaUSEDFLT by default)                           *
//*                  If specified, these are the color attributes used.          *
//*                  If not specified, stored text attributes are used.          *
//*                                                                              *
//* Returns: 'true'  if box fits within the window text area                     *
//*          'false' if invalid offset or dimensions specified                   *
//********************************************************************************

bool ACWin::DrawBox ( WinPos wpOrig, short height, short width, 
                      LineType lType, acAttr bxAttr )
{
   bool status = false ;               // return value

   //* Range check the parameters *
   if ( (wpOrig.row >= ZERO) && (wpOrig.row < this->txtRows) &&
        (wpOrig.col >= ZERO) && (wpOrig.col < this->txtCols) &&
        ((wpOrig.row + height) <= this->txtRows) &&
        ((wpOrig.col + width) <= this->txtCols)
      )
   {
      status = true ;               // declare success

      //* Convert offset (0-based) to terminal relative offset (1-based).*
      WinPos wp( short(this->txtBase.row + wpOrig.row), 
                 short(this->txtBase.col + wpOrig.col) ) ;

      //* Set the color attributes *
      if ( bxAttr == acaUSEDFLT )
      { this->setAttributes ( this->txtAttr ) ; }
      else
      { acaExpand acax( bxAttr ) ; this->setAttributes ( acax ) ; }

      //* Draw the box *
      this->acDrawBox ( wp, height, width, lType ) ;

      this->restoreAttributes () ;  // restore terminal attributes
   }
   return status ;

}  //* End DrawBox() *

//*************************
//*       DrawLine        *
//*************************
//********************************************************************************
//* Draw a horizontal or vertical line within the window's text area.            *
//* Horizontal lines are drawn left-to-right.                                    *
//* Vertical lines are drawn top-to-bottom.                                      *
//*                                                                              *
//* This is a specialized instance of the AnsiCmd acDrawLine() method.           *
//* The only difference is that the parameters are range checked to fit entirely *
//* within the window.                                                           *
//* Note that if an endpoint of the line intersects the border of the window,    *
//* the endpoint will be drawn using the color attributes of the border, while   *
//* the portion of the line within the text area of the window will be drawn     *
//* using the color attributes of the window interior.                           *
//*                                                                              *
//* Note: If the line intersects the window border, the endpoint                 *
//*       character will be automatically selected to match the                  *
//*       intersection of the drawn line and the border line style.              *
//*       As a corollary to this automatic character selection,                  *
//*       avoid positioning the start of a line such that it would               *
//*       obscure the window title (if any).                                     *
//*                                                                              *
//* Input  : pos     : offset from upper-left corner of window (0-based)         *
//*          length  : length of the line: number of number of rows (vertical),  *
//*                    or number of columns (horizontal).                        *
//*          vertical: 'true'  : vertical line                                   *
//*                    'false' : horizontal line                                 *
//*          lType   : (optional, ltSINGLE by default) line style                *
//*                     member of enum LineType: either ltSINGLE or ltDUAL       *
//*          begChar : (optional, null character by default)                     *
//*                     specify the beginning endpoint character                 *
//*          endChar : (optional, null character by default)                     *
//*                     specify the final endpoint character                     *
//*          lnAttr  : (optional, acaUSEDFLT by default)                         *
//*                    If specified, these are the color attributes used.        *
//*                    If not specified, stored text attributes are used.        *
//*                                                                              *
//* Returns: cursor position at the end of the line:                             *
//*          For horizontal lines, the column following the last character       *
//*          (limited at right edge of window)                                   *
//*          For vertical lines, the column below the last character             *
//*          (limited at bottom edge of window)                                  *
//********************************************************************************

WinPos ACWin::DrawLine ( const WinPos& wpOrig, short length, bool vertical, 
                         LineType lType, wchar_t begChar, wchar_t endChar, acAttr lnAttr )
{
   gString gsOut ;                  // text formatting
   WinPos wp = wpOrig ;             // working copy of origin (and return value)
   WinPos wpTerm ;                  // terminal-relative cursor position
   bool bdrTerm = false ;           // 'true' if line terminates in the border

   if ( lType != ltDUAL )     lType = ltSINGLE ;   // limit line-type options

   //* Range check position of origin and line length *
   if ( length < 2 )       length = 2 ;      // minimum length == 2 endpoints
   if ( wp.row < ZERO )    wp.row = ZERO ;   // no negative offsets
   if ( wp.col < ZERO )    wp.col = ZERO ;
   if ( vertical )      //** vertical line **
   {
      if ( length > this->height )  length = this->height ;
      if ( (wp.row + length) > this->height )   wp.row = ZERO ;

      //* If line intersects the window border, set endpoint character(s),*
      //* and set border attributes and write character(s).               *
      if ( (wp.row == ZERO) || ((wp.row + length) == this->height) )
         this->setAttributes ( this->bdrAttr ) ;
      if ( wp.row == ZERO )                     // set begChar
      {
         if ( this->bdrStyle == ltDUAL )
         { begChar = (lType == ltDUAL) ? wcsTTEEd : wcsTTEEdh ; }
         else
         { begChar = (lType == ltDUAL) ? wcsTTEEdv : wcsTTEE ; }
         gsOut.compose( "%C\n", &begChar ) ;
         wpTerm = this->acWrite ( this->winBase.row, 
                                  short(this->winBase.col + wp.col), gsOut ) ;
         begChar = L'\0' ;
         ++wp.row ;
         if ( --length == 1 )    // write the final character
         {
            // Programmer's Note: This final character cannot intersect 
            // bottom border because minimum window height == 3.
            this->setAttributes ( this->txtAttr ) ;
            if ( endChar == L'\0' )   endChar = lType ;
            gsOut.compose( "%C\n", &endChar ) ;
            wpTerm = this->acWrite ( wpTerm, gsOut ) ;
            endChar = L'\0' ;
            ++wp.row ;
            length = ZERO ;      // line complete
         }
      }
      if ( (wp.row + length) == this->height )  // set endChar
      {
         if ( this->bdrStyle == ltDUAL )
         { endChar = (lType == ltDUAL) ? wcsBTEEd : wcsBTEEdh ; }
         else
         { endChar = (lType == ltDUAL) ? wcsBTEEdv : wcsBTEE ; }
         gsOut.compose( "%C", &endChar ) ;
         wpTerm = { short(this->winBase.row + this->height - 1),
                    short(this->winBase.col + wp.col) } ;
         this->acWrite ( wpTerm, gsOut ) ;   // write end char into bottom border
         endChar = L'\0' ;
         if ( --length == 1 )
         {
            // Programmer's Note: The initial character cannot intersect
            // the top border because minimum window height == 3.
            this->setAttributes ( this->txtAttr ) ;
            --wpTerm.row ;
            if ( begChar == L'\0' )   begChar = lType ;
            gsOut.compose( "%C\n", &begChar ) ;
            wpTerm = this->acWrite ( wpTerm, gsOut ) ;
            wp.row = this->height - 1 ;
            length = ZERO ;      // line complete
         }
         bdrTerm = true ;        // line terminates in the window border
      }
   }        //** vertical line **
   else     //** horizontal line **
   {
      if ( length > this->width )  length = this->width ;
      if ( (wp.col + length) > this->width )    wp.col = ZERO ;
      if ( (wp.col == ZERO) || ((wp.col + length) == this->width) )
         this->setAttributes ( this->bdrAttr ) ;
      if ( wp.col == ZERO )                     // set begChar
      {
         if ( this->bdrStyle == ltDUAL )
         { begChar = (lType == ltDUAL) ? wcsLTEEd : wcsLTEEdv ; }
         else
         { begChar = (lType == ltDUAL) ? wcsLTEEdh : wcsLTEE ; }
         gsOut.compose( "%C", &begChar ) ;
         wpTerm = this->acWrite ( short(this->winBase.row + wp.row), 
                                  this->winBase.col, gsOut ) ;
         begChar = L'\0' ;
         ++wp.col ;
         if ( --length == 1 )    // write the final character
         {
            // Programmer's Note: This final character cannot intersect 
            // right border because minimum window width == 3.
            this->setAttributes ( this->txtAttr ) ;
            if ( endChar == L'\0' )   endChar = lType ;
            gsOut.compose( "%C", &endChar ) ;
            wpTerm = this->acWrite ( wpTerm, gsOut ) ;
            endChar = L'\0' ;
            ++wp.col ;
            length = ZERO ;      // line complete
         }
      }
      if ( (wp.col + length) == this->width )   // set endChar
      {
         if ( this->bdrStyle == ltDUAL )
         { endChar = (lType == ltDUAL) ? wcsRTEEd : wcsRTEEdv ; }
         else
         { endChar = (lType == ltDUAL) ? wcsRTEEdh : wcsRTEE ; }
         gsOut.compose( "%C", &endChar ) ;
         wpTerm = { short(this->winBase.row + wp.row),
                    short(this->winBase.col + this->width - 1) } ;
         this->acWrite ( wpTerm, gsOut ) ;
         endChar = L'\0' ;
         if ( --length  == 1 )
         {
            // Programmer's Note: The initial character cannot intersect
            // the Left border because minimum window width == 3.
            this->setAttributes ( this->txtAttr ) ;
            --wpTerm.col ;
            if ( begChar == L'\0' )   begChar = lType ;
            gsOut.compose( "%C", &begChar ) ;
            wpTerm = this->acWrite ( wpTerm, gsOut ) ;
            wp.col = this->width - 1 ;
            length = ZERO ;      // line complete
         }
         bdrTerm = true ;        // line terminates in the window border
      }
   }        //** horizontal line **

   //* Convert window-relative origin to terminal relative *
   wpTerm.row = this->winBase.row + wp.row ;
   wpTerm.col = this->winBase.col + wp.col ;

   //* If line length is STILL at or above minimum *
   if ( length >= 2 )
   {
      //* Draw the portion of the line that lies in the interior of the window.*
      if ( lnAttr == acaUSEDFLT )
         this->setAttributes ( this->txtAttr ) ;
      else
      {
         acaExpand acax( lnAttr ) ;
         this->setAttributes ( acax ) ;
      }
      wpTerm = this->acDrawLine ( wpTerm, length, vertical, lType, begChar, endChar ) ;
   }

   //* Convert terminal-relative position to window-relative   *
   //* and ensure that final position stays within the window. *
   wp.row = wpTerm.row - this->winBase.row ;
   wp.col = wpTerm.col - this->winBase.col ;
   if ( bdrTerm )       // if line terminates in the window border
   {
      if ( vertical )   wp.row = this->height - 1 ;
      else              wp.col = this->width - 1 ;
   }
   if ( wp.row >= this->height )    wp.row = this->height - 1 ;
   if ( wp.col >= this->width )     wp.col = this->width - 1 ;

   this->restoreAttributes () ;     // restore terminal attributes
   return wp ;

}

//*********************
//*   FieldOutline    *
//*********************
//********************************************************************************
//* Draw an outline around the outside of the specified field within the window  *
//* border.                                                                      *
//*                                                                              *
//* The target field must have at least one space open on all sides. That is,    *
//* it must not be positioned against the window border, and must not be         *
//* adjacent to any other field.                                                 *
//* Programmer's Note: Potential incursion into the border is range checked;     *
//* however, potential incursion into other fields _is not_ tested. Beware!      *
//*                                                                              *
//* Input  : fIndx   : index of target field                                     *
//*                    if 'fIndx' == number of fields defined, then draw an      *
//*                    outline around each field.                                *
//*          lType   : (optional, ltSINGLE by default) line style                *
//*                     member of enum LineType: ltSINGLE or ltDUAL              *
//*          lnAttr  : (optional, acaUSEDFLT by default)                         *
//*                    If specified, color attributes to be used.                *
//*                    If not specified, stored text attributes used.            *
//*                                                                              *
//* Returns: 'true'  if successful                                               *
//*          'false' if invalid field index or insufficient space                *
//*                  around the field                                            *
//********************************************************************************

bool ACWin::FieldOutline ( short fIndx, LineType lType, acAttr lnAttr )
{
   bool status = false ;                     // return value

   if ( (fIndx >= ZERO) && (fIndx < this->skf->fCnt) )
   {
      const skField *fptr = &this->skf->fld[fIndx] ; // pointer to target field
      WinPos bpos( short(fptr->orig.row - 1),   // box origin (zero-based offset)
                   short(fptr->orig.col - 1) ) ;
      short bhgt = fptr->hgt + 2,               // box height
            bwid = fptr->wid + 2 ;              // box width

      //* Range check position and dimensions *
      if ( (bpos.row >= ZERO) && (bpos.col >= ZERO) &&
           ((bpos.row + bhgt) <= this->txtRows) &&
           ((bpos.col + bwid) <= this->txtCols) )
      {
         this->DrawBox ( bpos, bhgt, bwid, lType, lnAttr ) ;
         status = true ;
      }
   }
   else if ( fIndx == this->skf->fCnt )
   {
      //* Recursively call for each field *
      for ( short f = ZERO ; f < this->skf->fCnt ; ++f )
      {
         if ( !(status = this->FieldOutline ( f, lType, lnAttr )) )
            break ;
      }
   }
   return status ;

}  //* End FieldOutline() *

//* ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- *
//* ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- *
//* ----                  accExpand-class implementation                  ---- *
//* ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- *
//* ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- ---- *
//*********************
//*     acaExpand     *
//*********************
//********************************************************************************
//* Default Constructor:                                                         *
//* Initialize all data members to default values.                               *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

acaExpand::acaExpand ( void )
{
   //* Initialize with terminal default foreground/background *
   acAttr acaBits = acaATTR_DFLT ;

   this->decode ( acaBits ) ;

} ;   //* End acaExpand() *

//*********************
//*     acaExpand     *
//*********************
//********************************************************************************
//* Initialization Constructor:                                                  *
//* Decode the specified bitfield value and store the decoded data in the        *
//* data members.                                                                *
//*                                                                              *
//* Input  : acaValue : an initialized acAttr bitfield value                     *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

acaExpand::acaExpand ( acAttr acaValue )
{

   this->decode ( acaValue ) ;

} ;   //* End acaExpand() *

//*********************
//*      decode       *
//*********************
//********************************************************************************
//* Decode the specified bitfield value and store the decoded data in the        *
//* data members.                                                                *
//*                                                                              *
//* The 'acaValue' argument is a bitfield containing the OR'd fields             *
//* describing the foreground and background color attributes and text           *
//* modifiers to be set before writing text to the terminal window.              *
//*                                                                              *
//* Input  : acaValue : an initialized acAttr bitfield value                     *
//*                     (This becomes the 'acaVal' member.)                      *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* ------                                                                       *
//* Caller provides the OR'd bitfield values for the color attributes,           *
//* text-modifier flags and miscellaneous flags needed to define foregroud       *
//* and background attributes for writing data to the terminal window.           *
//* This method decodes all bitfields into their individual fields, providing    *
//* uniform access to the information without the mainline code having to        *
//* perform error-prone bitfield twiddling.                                      *
//*                                                                              *
//* AnsiCmd (enum acAttr) color-attribute bit fields:                            *
//* 0x000000FF  index for foreground color                                       *
//*             ......00-......0F  16  4-bit color attributes                    *
//*             ......10-......E7  216 8-bit color attributes                    *
//*             ......E8-......FF  24  greyscale color attributes                *
//*             ......00-......FC  232 RGB color attributes                      *
//* 0x0000FF00  index for background color                                       *
//*             ....00..-....0F..  16  4-bit color attributes                    *
//*             ....10..-....E7..  216 8-bit color attributes                    *
//*             ....E8..-....FF..  24  greyscale color attributes                *
//*             ....00..-....FC..  232 RGB color attributes                      *
//* 0x00FF0000  text modifier flags                                              *
//*             ..00....   plain text (no modifiers)                             *
//*             ..01....   bold (intense)                                        *
//*             ..02....   italic                                                *
//*             ..04....   underline                                             *
//*             ..08....   overline                                              *
//*             ..10....   x-out (strike-through)                                *
//*             ..20....   blink                                                 *
//*             ..40....   concealed (invisible)                                 *
//*             ..80....   reversed fg/bg                                        *
//* 0xFF000000  additional flags                                                 *
//*             01......   clear existing modifiers before setting the new ones  *
//*             02......   use terminal default foreground attribute             *
//*                        (foreground index ignored)                            *
//*             04......   if set, it indicates that foreground index is the     *
//*                        index of a "web-safe" RGB foreground color attribute  *
//*                        with a range of: minRGB <= value <= maxRGB            *
//*             08......   use terminal default background attribute             *
//*                        (background index ignored)                            *
//*             10......   if set, it indicates that background index is the     *
//*                        index of a "web-safe" RGB background color attribute  *
//*                        with a range of: minRGB <= value <= maxRGB            *
//*             20......   (reserved)                                            *
//*             40......   (reserved)                                            *
//*             80......   (reserved as internal placeholder value)              *
//*                                                                              *
//* To specify the terminal default foreground attribute, set the acaFG_DFLT     *
//* flag. The foreground index value will be ignored.                            *
//*               Fgnd:  xxxx-x01x-xxxx-xxxx-xxxx-xxxx-....-....                 *
//*                                                                              *
//* To specify the terminal default background attribute, set the acaBG_DFLT     *
//* flag. The background index value will be ignored.                            *
//*               Bgnd:  xxx0-1xxx-xxxx-xxxx-....-....-xxxx-xxxx                 *
//*                                                                              *
//* To specify a 4-bit foreground attribute, _reset_ the acaFG_DFLT and          *
//* acaFG_RGB flags and insert the desired member of enum acAttr in the range:   *
//* acaFG_BLACK through acaFGb_GREY inclusive.                                   *
//*                                                                              *
//* To specify a 4-bit background attribute, _reset_ the acaBG_DFLT and          *
//* acaBG_RGB flags and insert the desired member of enum acAttr in the range:   *
//* acaBG_BLACK through acaBGb_GREY inclusive.                                   *
//*                                                                              *
//* To specify an 8-bit indexed color foreground attribute, initialize the       *
//* foreground index and _reset_ both the acaFG_DFLT and acaFG_RGB flags.        *
//*               Fgnd:  xxxx-x00x-xxxx-xxxx-xxxx-xxxx-nnnn-nnnn                 *
//*                                                                              *
//* To specify an 8-bit indexed color background attribute, initialize the       *
//* background index and _reset_ both the acaBG_DFLT and acaBG_RGB flags.        *
//*               Bgnd:  xxx0-0xxx-xxxx-xxxx-nnnn-nnnn-xxxx-xxxx                 *
//*                                                                              *
//* To specify one of the "web-safe" RGB register combinations, initialize the   *
//* foreground or background index and set the acaFG_RGB or acaBG_RGB flag,      *
//* respectively. Fgnd:  xxxx-x10x-xxxx-xxxx-xxxx-xxxx-nnnn-nnnn                 *
//*               Bgnd:  xxx1-0xxx-xxxx-xxxx-nnnn-nnnn-xxxx-xxxx                 *
//*                                                                              *
//* To set one or more text modifiers, _reset_ the acaCLEAR_MODS flag and        *
//* create a logical OR of the desired attributes. (include existing fg/bg bits.)*
//* Ex: acAttr((fg | bg |acaBOLD | acaITALIC | acaUNDERLINE) & ~acaCLEAR_MODS)   *
//*               Mods:  xxxx-xxx0-xxxx-x111-xxxx-xxxx-xxxx-xxxx                 *
//*                                                                              *
//* To reset one or more text modifier, set the acaCLEAR_MODS flag and _set_     *
//* the flag for the modfier to be reset. (include existing fg/bg bits.)         *
//* Example: acAttr(fg | bg | acaITALIC | acaCLEAR_MODS)                         *
//*               Mods:  xxxx-xxx1-0000-0010-xxxx-xxxx-xxxx-xxxx                 *
//*                                                                              *
//********************************************************************************
//* Notes: The functionality can be tested by enabling:                          *
//*        DEBUG_ANSICMD && DEBUG_LOG && DEBUG_decode, and then use the          *
//*        Test_Sandbox method to repeatedly call decode() to verify that        *
//*        the bitmap is updated correctly.                                      *
//*                                                                              *
//* // This example may be used to test one of the RGB "hue" groups.             *
//* acaExpand acaEx(acaPLAINTXT);                                                *
//* uint8_t rgbReg ;                                                             *
//* for ( uint8_t shade = wsrgbSHADE_MIN ; shade <= wsrgbSHADE_MAX ; ++shade )   *
//* {                                                                            *
//*    rgbReg = wsrgbRED + shade ;                                               *
//*    gsdbg.compose( "acaEx (fgnd wsrgbRED x%02hhX)", &rgbReg );                *
//*    ofsdbg << gsdbg.ustr() << endl ;                                          *
//*    acaEx.decode( acAttr(acaFG_RGB | acaBG_DFLT | rgbReg) );                 *
//* }                                                                            *
//*                                                                              *
//********************************************************************************

void acaExpand::decode ( acAttr acaValue )
{
   #define DEBUG_decode (0)   // for debugging only

   this->acaVal    = acaValue ;                       // full, encoded value
   this->fgBits    = this->acaVal & acaFG_MASK ;      // all foreground bits
   this->bgBits    = this->acaVal & acaBG_MASK ;      // all background bits
   this->allFlags  = this->acaVal & acaFLAG_MASK ;    // all binary flags
   this->modFlags  = this->acaVal & acaMOD_MASK ;     // text-modifier flags
   this->acaFgnd   = acAttr(this->acaVal & acaFG_INDEX) ; // fgnd acAttr member
   this->acaBgnd   = acAttr(this->acaVal & acaBG_INDEX) ; // bgnd acAttr member
   this->fgIndex   = this->acaFgnd ;                  // fgnd byte index into lookup table
   this->bgIndex   = this->acaBgnd >> 8 ;             // bgnd byte index into lookup table

   //* Text-modifier flags *
   this->boldMod   = bool(this->acaVal & acaBOLD) ;   // bold-text flag
   this->italicMod = bool(this->acaVal & acaITALIC) ; // italic-text flag
   this->ulineMod  = bool(this->acaVal & acaUNDERLINE) ; // underlined-text flag
   this->olineMod  = bool(this->acaVal & acaOVERLINE) ;  // overlined-text flag
   this->xoutMod   = bool(this->acaVal & acaXOUT) ;   // strikethrough-text flag
   this->blinkMod  = bool(this->acaVal & acaBLINK) ;  // blinking-text flag
   this->invisMod  = bool(this->acaVal & acaCONCEAL) ;// invisible-text (concealed) flag
   this->revMod    = bool(this->acaVal & acaREVERSE) ;// reversed fgnd/bgnd flag
   //* Miscellaneous flags *
   this->fgDflt    = bool(this->acaVal & acaFG_DFLT) ;// foreground-default flag
   this->bgDflt    = bool(this->acaVal & acaBG_DFLT) ;// background-default flag
   this->clrMods   = bool(this->acaVal & acaCLEAR_MODS) ;// clear-mods flag

   //* If RGB background specified, call the low-level conversion method   *
   //* to set the 'rgbBgR', 'rgbBgG' and 'rgbBgB' register values.         *
   //* If 'bgIndex' is out-of-range, registers set for wsrgbMIN (black)    *
   //* Note that the following data members are also initialized:          *
   //*    bgRgb, bgIndex, 'aesBgnd', 'aesBgType' (bgDflt is reset)         *
   if ( (this->bgRgb = bool(this->acaVal & acaBG_RGB)) != false )
   {
      this->web2regs ( this->bgIndex, false ) ;
   }
   //* If not RGB, initialize the background RGB register values to zero.  *
   //* 'bgIndex' contains the 4-bit/8-bit index value.                     *
   else
   {
      this->bgHue   = wsrgbMIN ;
      this->bgShade = wsrgbSHADE_MIN ;
      this->rgbBgR = this->rgbBgG = this->rgbBgB = ZERO ;
      this->aesBgnd = this->bits2aeseq ( acAttr(this->bgBits), this->bgIndex, false ) ;
      //* Indicate which Bg color-attribute type is active.*
      if ( (this->aesBgnd >= aesBG_BLACK) && (this->aesBgnd <= aesBG_DFLT) )
         this->aesBgType = aesBG_DFLT ;
      else
         this->aesBgType = aesBG_INDEX ;
   }

   //* If RGB foreground specified, call the low-level conversion method   *
   //* to set the 'rgbFgR', 'rgbFgG' and 'rgbFgB' register values.         *
   //* If 'fgIndex' is out-of-range, registers set for wsrgbMIN (black)    *
   //* Note that the following data members are also initialized:          *
   //*    fgRgb, fgIndex, 'aesFgnd', 'aesFgType' (fgDflt is reset)         *
   if ( (this->fgRgb = bool(this->acaVal & acaFG_RGB)) != false )
   {
      this->web2regs ( this->fgIndex, true ) ;
   }

   //* If not RGB, initialize the foreground RGB register values to zero.  *
   //* 'fgIndex' contains the 4-bit/8-bit index value.                     *
   else
   {
      this->fgHue   = wsrgbMIN ;
      this->fgShade = wsrgbSHADE_MIN ;
      this->rgbFgR = this->rgbFgG = this->rgbFgB = ZERO ;
      this->aesFgnd = this->bits2aeseq ( acAttr(this->fgBits), this->fgIndex, true ) ;
      //* Indicate which Fg color-attribute type is active.*
      if ( (this->aesFgnd >= aesFG_BLACK) && (this->aesFgnd <= aesFG_DFLT) )
         this->aesFgType = aesFG_DFLT ;
      else
         this->aesFgType = aesFG_INDEX ;
   }

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_decode != 0
   if ( ofsdbg.is_open() )
   {
      gsdbg.compose( "decode( fgIndex:%02hhX hue:%02hhX shade:%02hhX r:%hhu g:%hhu b:%hhu )\n"
                     "      ( bgIndex:%02hhX hue:%02hhX shade:%02hhX r:%hhu g:%hhu b:%hhu )\n"
                     "        ---r brfc --mods--- --bgnd--- --fgnd---\n"
                     "       %#-b",
                     &this->fgIndex, &this->fgHue, &this->fgShade, 
                     &this->rgbFgR,  &this->rgbFgG, &this->rgbFgB, 
                     &this->bgIndex, &this->bgHue, &this->bgShade, 
                     &this->rgbBgR,  &this->rgbBgG, &this->rgbBgB, 
                     &this->acaVal) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_decode

}  //* End decode() *

//*************************
//*        update         *
//*************************
//********************************************************************************
//* Update attribute tracking data by converting the provided member of          *
//* enum aeSeq to the equivalent acAttr bitfield value.                          *
//*                                                                              *
//* Called by the methods which directly issue ANSI escape sequences:            *
//* acSetFg(), acSetBg(), acSetMod(), acSet8bit() acSetGreyscale(), setRgbWeb(). *
//*                                                                              *
//* Input  : aesValue : member of enum aeSeq indicating the color attribute or   *
//*                     text modifier which caller has just changed.             *
//*          colorIndx: (optional)                                               *
//*                  -- lookup-table index for 8-bit color attribute             *
//*                     when: aesValue == aesFG_INDEX or                         *
//*                           aesValue == aesBG_INDEX                            *
//*                           Range: min8BIT <= colorIndx <= max8BIT             *
//*                  -- lookup-table index for "web-safe" RGB color attribute    *
//*                     when: aesValue == aesFG_RGB or                           *
//*                           aesValue == aesBG_RGB                              *
//*                           Range: minRGB <= colorIndx <= maxRGB               *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void acaExpand::update ( aeSeq aesValue, uint8_t colorIndx )
{
   #define DEBUG_update (0)         // for debugging only

   bool setFlag = true ;            // for binary flags, indicate set vs. reset

   //* Convert the aeSeq value to acAttr value.  *
   //* For text modifiers, 'setFlag' reports     *
   //* whether the modifer flag s/b set or reset.*
   acAttr acaBits = this->aeseq2bits ( aesValue, setFlag ) ;

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_update != 0
   if ( ofsdbg.is_open() )
   {
      gsdbg = "          update(        ---r brfc --mods--- --bgnd--- --fgnd---\n" ;
      if ( acaBits != acaPLAINTXT )
         gsdbg.append( "                  init :%#-b \n", &acaBits ) ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_update

   //* If adding or removing one of the text modifiers  *
   if ( (aesValue >= aesBOLD) && (aesValue <=aesOVERLINE_OFF) )
   {
      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_update != 0
      wchar_t t = L't', f = L'f' ;
      if ( ofsdbg.is_open() )
      { gsdbg.append( "                mods(%C):%#-b \n", (setFlag ? &t : &f), &acaBits ) ; }
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_update

      if ( setFlag )       // set the specified modifier bit
         acaBits = acAttr(this->modFlags | acaBits) ;
      else                 // reset the specified modifier bit
         setFlag = false ;

      //* Insert both foreground and background bits.*
      acaBits = acAttr(acaBits | (this->fgBits | this->bgBits)) ;

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_update != 0
      if ( ofsdbg.is_open() )
      { gsdbg.append( "                mods   :%#-b \n", &acaBits ) ; }
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_update
   }

   //* If updating the foreground attribute, insert the    *
   //* current background attribute, OR if updating the    *
   //* background attribute, insert the current foreground *
   //* attribute. This prevents accidental modification of *
   //* the attribute which was not specified.              *
   else
   {
      //* If a new 4-bit background attribute specified,  *
      //* insert the current foreground into the bitfield.*
      if ( (aesValue >= aesBG_BLACK) && (aesValue <= aesBG_DFLT) )
         acaBits = acAttr(acaBits | this->fgBits) ;
      //* If a new 4-bit foreground attribute specified,  *
      //* insert the current background into the bitfield.*
      if ( (aesValue >= aesFG_BLACK) && (aesValue <= aesFG_DFLT) )
         acaBits = acAttr(acaBits | this->bgBits) ;

      //* If a new 8-bit background index specified *
      if ( (aesValue == aesBG_INDEX) && 
           (((colorIndx >= min8BIT) && (colorIndx <= max8BIT)) ||
            ((colorIndx >= minGSCALE) && (colorIndx <= maxGSCALE))) )
      {
         acaBits = acAttr(acaBits | this->fgBits | (colorIndx << 8)) ;

         #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_update != 0
         if ( ofsdbg.is_open() )
         { gsdbg.append( "                 indBG :%#-b \n", &acaBits ) ; }
         #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_update
      }

      //* If a new 8-bit foreground index specified *
      else if ( (aesValue == aesFG_INDEX) && 
                (((colorIndx >= min8BIT) && (colorIndx <= max8BIT)) ||
                 ((colorIndx >= minGSCALE) && (colorIndx <= maxGSCALE))) )
      {
         acaBits = acAttr(acaBits | this->bgBits | colorIndx) ;

         #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_update != 0
         if ( ofsdbg.is_open() )
         { gsdbg.append( "                 indFG :%#-b \n", &acaBits ) ; }
         #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_update
      }

      //* If a new RGB "web-safe" background specified *
      else if ( (aesValue == aesBG_RGB) &&
                ((colorIndx >= wsrgbMIN) && (colorIndx <= wsrgbMAX)) )
      {
         acaBits = acAttr(acaBits | this->fgBits | (colorIndx << 8) | acaBG_RGB) ;

         #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_update != 0
         if ( ofsdbg.is_open() )
         { gsdbg.append( "                 rgbBG :%#-b \n", &acaBits ) ; }
         #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_update
      }

      //* If a new RGB "web-safe" foreground specified *
      else if ( (aesValue == aesFG_RGB) &&
                ((colorIndx >= wsrgbMIN) && (colorIndx <= wsrgbMAX)) )
      {
         acaBits = acAttr(acaBits | this->bgBits | colorIndx | acaFG_RGB) ;

         #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_update != 0
         if ( ofsdbg.is_open() )
         { gsdbg.append( "                 rgbFG :%#-b \n", &acaBits ) ; }
         #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_update
      }
   }

   //* Modify the target field *
   this->modify ( acaBits, setFlag ) ;

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_update != 0
   if ( ofsdbg.is_open() )
   {
      gsdbg.append( "                 newval:%#-b )", &this->acaVal ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_update

}  //* End update() *

//*********************
//*      modify       *
//*********************
//********************************************************************************
//* Update one or more data-member bitfields which track color attributes and    *
//* text-modification options.                                                   *
//* The caller is responsible for actually issuing the command to the terminal.  *
//* This method updates the attribute tracking data to reflect that change.      *
//*                                                                              *
//* 1) Specify a foreground color attribute, OR                                  *
//* 2) Specify a background color attribute, OR                                  *
//* 3) Specify both a foreground and background color attributes, OR             *
//* 4) Specify setting one or more text modifiers OR'd together, OR              *
//* 5) Specify resetting one or more text modifiers OR'd together.               *
//* 6) Specify both attribute and text modifiers.                                *
//*                                                                              *
//* Input  : acaValue : one or more members of enum acAttr OR'd together.        *
//*                     -- if 'add' is set, OR the value with the existing       *
//*                        data and recalculate all data members.                *
//*                     -- if 'add' is reset, then XOR the inverted value with   *
//*                        the existing data and recalculate all data members.   *
//*          add      : for text-modifier flags only (ignored for color attr.).  *
//*                   : if 'true'  set the specified modifier flags              *
//*                     if 'false' reset the specified modifier flags            *
//*                                existing data.                                *
//*                                                                              *
//* Returns: the modified acAttr value                                           *
//********************************************************************************
//* Notes: The functionality can be tested by enabling:                          *
//*        DEBUG_ANSICMD && DEBUG_LOG && DEBUG_modify, and then use the          *
//*        Test_Sandbox method to repeatedly call modify() to verify that        *
//*        the bitmap is updated correctly. Example:                             *
//*                                                                              *
//* acaExpand acaEx( acaPLAINTXT ) ;                                             *
//* ofsdbg << "Test( set acaXOUT | acaBLINK | acaCONCEAL | acaREVERSE )" << endl;*
//* acaEx.modify( acAttr(acaXOUT | acaBLINK | acaCONCEAL | acaREVERSE), true );  *
//*                                                                              *
//* ofsdbg << "\nSandbox( set acaBOLD )" << endl ;                               *
//* acaEx.modify( acaBOLD, true ) ;                                              *
//*                                                                              *
//* ofsdbg << "\nSandbox( reset acaBOLD )" << endl;                              *
//* acaEx.modify( acaBOLD, false );                                              *
//*                                                                              *
//* ofsdbg << "\nSandbox( set 4-bit fg/bg )" << endl ;                           *
//* acaEx.modify( acaFGb_RED | acaBG_BLUE );                                     *
//*                                                                              *
//* ofsdbg << "\nSandbox( set 8-bit fg/bg )" << endl ;                           *
//* acAttr acattr =                                                              *
//*             acComposeAttributes( (min8BIT + 4), (max8BIT-2), false, false ); *
//* acaEx.modify( acattr ) ;                                                     *
//*                                                                              *
//* ofsdbg << "\nSandbox( set RGB fg/bg )" << endl ;                             *
//* acAttr acattr =                                                              *
//*         acComposeAttributes( (wsrgbGREEN + 12), (wsrgbRED + 2), true, true );*
//* acaEx.modify( acattr ) ;                                                     *
//*                                                                              *
//* ofsdbg << "\nSandbox( set RGB fg and 8-bit bg )" << endl ;                   *
//* acAttr acattr = acComposeAttributes( (wsrgbGREEN + 2), (42), true, false );  *
//* acaEx.modify( acattr ) ;                                                     *
//*                                                                              *
//*                                                                              *
//********************************************************************************

acAttr acaExpand::modify ( acAttr acaValue, bool add )
{
   #define DEBUG_modify (0)            // for debugging only
   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_modify != 0
   if ( ofsdbg.is_open() )
   {
      gsdbg.compose( "modify( acaValue:%08X add:%hhd\n"
                     "                         ---r brfc --mods--- --bgnd--- --fgnd---\n"
                     "                 acaVal:%#-b ",
                     &acaValue, &add, &this->acaVal ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_modify

   acAttr acamods = acAttr(acaValue & acaMOD_MASK) ;  // isolate modifier flags
   bool modsUpdate = false,            // set if modifier update
        fgndUpdate = false,            // set if foreground update
        bgndUpdate = false ;           // set if background update

   //*******************************
   //* Toggle a text modifier flag *
   //*******************************
   if ( acaValue & acaCLEAR_MODS )
   {
      acamods = acaMOD_MASK ;
      add = false ;
      modsUpdate = true ;
   }
   else if ( acamods & acaMOD_MASK )
      modsUpdate = true ;

   if ( modsUpdate )
   {
      this->acaVal = acAttr(add ? (this->acaVal | acamods) : (this->acaVal & ~acamods)) ;

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_modify != 0
      if ( ofsdbg.is_open() )
      {
         gsdbg.compose( " modify( mods: acaValue:%#-b\n"
                        "                 acaVal:%#-b", &acaValue, &this->acaVal ) ;
         ofsdbg << gsdbg.ustr() << endl ;
      }
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_modify
   }

   //****************************
   //* Modify a color attribute *
   //****************************

   //* Seperate the fgnd and bgnd bitfields *
   acAttr newFg = acAttr(acaValue & acaFG_MASK),
          newBg = acAttr(acaValue & acaBG_MASK) ;

   //* RGB foreground and/or background attribute specified *
   if ( (acaValue & acaFG_RGB) || (acaValue & acaBG_RGB) )
   {
      if ( (acaValue & acaFG_RGB) && (!this->fgRgb || (this->fgIndex != newFg)) )
      {
         //* Clear the current fgnd bits and integrate the new fgnd bits.*
         this->acaVal = acAttr(this->acaVal & ~acaFG_MASK) ;
         this->acaVal = acAttr(this->acaVal | newFg) ;
         fgndUpdate = true ;
      }
      if ( (acaValue & acaBG_RGB) && (!this->bgRgb || (this->bgIndex != newBg)) )
      {
         //* Clear the current bgnd bits and integrate the new bgnd bits.*
         this->acaVal = acAttr(this->acaVal & ~acaBG_MASK) ;
         this->acaVal = acAttr(this->acaVal | newBg) ;
         bgndUpdate = true ;
      }

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_modify != 0
      if ( (ofsdbg.is_open()) && (fgndUpdate || bgndUpdate) )
      {
         gsdbg.compose( " modify(  rgb: acaValue:%#-b\n"
                        "                 acaVal:%#-b", &acaValue, &this->acaVal ) ;
         ofsdbg << gsdbg.ustr() << endl ;
      }
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_modify
   }

   //* Indexed foreground and/or background attribute *
   if ( !fgndUpdate || !bgndUpdate )
   {
      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_modify != 0
      bool ftmp = fgndUpdate, btmp = bgndUpdate ;
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_modify

      //* If new fgnd bits != existing fgnd bits, replace fgnd *
      if ( !fgndUpdate && (newFg != this->fgBits) )
      {
         //* Clear the current fgnd bits *
         this->acaVal = acAttr(this->acaVal & ~acaFG_MASK ) ;

         //* Set the new value for fgnd bitfield *
         this->acaVal = acAttr(this->acaVal | newFg) ;
         fgndUpdate = true ;
      }

      //* If new bgnd bits != existing bgnd bits, replace bgnd *
      if ( !bgndUpdate && (newBg != this->bgBits) )
      {
         //* Clear the current bgnd bits *
         this->acaVal = acAttr(this->acaVal & ~acaBG_MASK) ;

         //* Set the new value for bgnd bitfield *
         this->acaVal = acAttr(this->acaVal | newBg) ;
         bgndUpdate = true ;
      }

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_modify != 0
      if ( (ofsdbg.is_open()) && ((fgndUpdate != ftmp) || (bgndUpdate != btmp)) )
      {
         gsdbg.compose( " modify( indx: acaValue:%#-b\n"
                        "                 acaVal:%#-b", &acaValue, &this->acaVal ) ;
         ofsdbg << gsdbg.ustr() << endl ;
      }
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_modify
   }

   //* If bitfields modified, decode them *
   if ( modsUpdate || fgndUpdate || bgndUpdate )
      this->decode ( this->acaVal ) ;

   return ( this->acaVal ) ;

}  //* End modify() *

//*************************
//*        compose        *
//*************************
//********************************************************************************
//* Construct an acAttr attribute bitmap from component values:                  *
//* This method is used when specifying an 8-bit fgnd/bgnd or an RGB fgnd/bgnd   *
//* attribute.                                                                   *
//* See also the API-level method acComposeAttributes() which calls this method  *
//* without the application having to know anything about bit fields.            *
//*                                                                              *
//* Note on 4-bit values:                                                        *
//* ---------------------                                                        *
//* The 4-bit color attributes are individually named, so the attributes may be  *
//* specified directly.                                                          *
//*    Examples: acSetAttributes( acAttr(acaFG_BLUE | acaBGb_BROWN) );           *
//*              acSetAttributes( acAttr(acaFG_DFLT | acaBGb_BLUE) );            *
//* In any case, 4-bit data are handled here to avoid problems.                  *
//*                                                                              *
//* Interpretation of 'fgIndex' field is based on 'fgRgb' flag:                  *
//*    If fgRgb set, then fgIndex contains web-safe RGB index.                   *
//*    If fgRgb reset, then fgIndex contains 8-bit color index                   *
//*    Special Case: fgRgb reset AND fgIndex < min8BIT, then 4-bit color assumed *
//* Interpretation of 'bgIndex' field is based on 'bgRgb' flag:                  *
//*    If bgRgb set, then bgIndex contains web-safe RGB index.                   *
//*    If bgRgb reset, then bgIndex contains 8-bit color index                   *
//*    Special Case: bgRgb reset AND bgIndex < min8BIT, then 4-bit color assumed *
//*                                                                              *
//* Text-modifier flags may be set if desired.                                   *
//* Modifiers may be specified either by placing the logical OR of the flags in  *
//* the 'modFlags' member, or by setting the individual modifier flags.          *
//* Note that if modFlags != acaPLAINTXT, it overrides the individual flags.     *
//*                                                                              *
//* Input  : none (data members have been initialized by caller):                *
//*            fgIndex, bgIndex, fgRgb, bgRgb, and                               *
//*            either modFlags or individual modifier flags                      *
//*                                                                              *
//* Returns: the constructed acAttr value                                        *
//********************************************************************************
//* Notes: The functionality can be tested by enabling:                          *
//*        DEBUG_ANSICMD && DEBUG_LOG && DEBUG_compose, and then use the         *
//*        Test_Sandbox method to repeatedly call compose() via the public       *
//*        acComposeAttributes() method to verify that the bitmap is updated     *
//*        correctly. Example:                                                   *
//*                                                                              *
//* // 4-bit fgnd, 8-bit bgnd, with modifiers                                    *
//* this->acComposeAttributes ( 0x03, 0xC0, false, false,                        *
//*                  acAttr(acaBOLD | acaITALIC | acaUNDERLINE | acaOVERLINE) ); *
//* // Intentional out-of-range RGB fgnd/bgnd, with modifiers                    *
//* this->acComposeAttributes ( 0xFF, 0xFF, true, true,                          *
//*                      acAttr(acaXOUT | acaBLINK | acaCONCEAL | acaREVERSE) ); *
//* // 8-bit fgnd/bgnd                                                           *
//* this->acComposeAttributes ( 0x22, 0x33, false, false ) ;                     *
//* // RGB fgnd/bgnd                                                             *
//* this->acComposeAttributes ( 0x44, 0x11, true, true ) ;                       *
//********************************************************************************

acAttr acaExpand::compose ( void )
{
   #define DEBUG_compose (0)     // for debugging only

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_compose != 0
   if ( ofsdbg.is_open() )
   {
      gsdbg.compose( "compose( acaVal:%08X fgIndex:%02hhX bgIndex:%02hhX "
                     "fgRgb:%hhd bgRgb:%hhd modFlags:%08X )\n" 
                     "          ---r brfc --mods--- --bgnd--- --fgnd---\n"
                     "  acaVal:%#-b ",
                     &this->acaVal, &this->fgIndex, &this->bgIndex, 
                     &this->fgRgb, &this->bgRgb, &this->modFlags, &this->acaVal ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif    // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_compose

   //* Concatenate foreground and background indices. *
   acAttr newVal = acAttr(this->fgIndex | (this->bgIndex << 8)) ;

   if ( this->fgRgb )                  // if RGB foreground
      newVal = acAttr(newVal | acaFG_RGB) ;
   else if ( this->fgDflt )            // default (4-bit) foreground
   {
      newVal = acAttr(newVal | acaFG_DFLT) ;
      newVal = acAttr(newVal & ~acaFG_INDEX) ; // (force to zero index for default)
   }
   if ( this->bgRgb )                  // if RGB background
      newVal = acAttr(newVal | acaBG_RGB) ;
   else if ( this->bgDflt )            // default (4-bit) background
   {
      newVal = acAttr(newVal | acaBG_DFLT) ;
      newVal = acAttr(newVal & ~acaBG_INDEX) ; // (force to zero index for default)
   }

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_compose != 0
   if ( ofsdbg.is_open() )
   {
      gsdbg.compose( "  fg/bg :%#-b ", &newVal ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif    // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_compose

   //* If modifier flag(s) set, concatenate them into the new value.  *
   //* Modifiers may be specified using either the 'modFlags' member, *
   //* or by setting the individual modifier flags.                   *
   if ( this->modFlags != acaPLAINTXT)
   { newVal = acAttr(newVal | this->modFlags) ; }
   else
   {
      if ( this->boldMod )    newVal = acAttr(newVal | acaBOLD) ;
      if ( this->italicMod )  newVal = acAttr(newVal | acaITALIC) ;
      if ( this->ulineMod )   newVal = acAttr(newVal | acaUNDERLINE) ;
      if ( this->olineMod )   newVal = acAttr(newVal | acaOVERLINE) ;
      if ( this->xoutMod )    newVal = acAttr(newVal | acaXOUT) ;
      if ( this->blinkMod )   newVal = acAttr(newVal | acaBLINK) ;
      if ( this->invisMod )   newVal = acAttr(newVal | acaCONCEAL) ;
      if ( this->revMod )     newVal = acAttr(newVal | acaREVERSE) ;
   }

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_compose != 0
   if ( ofsdbg.is_open() )
   {
      gsdbg.compose( "   mods :%#-b ", &newVal ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif    // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_compose

   //* Re-decode to initialize remaining data members.*
   this->decode ( newVal ) ;

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_compose != 0
   if ( ofsdbg.is_open() )
   {
      gsdbg.compose( "  acaVal:%#-b ", &this->acaVal ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif    // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_compose

   return this->acaVal ;

}  //* End compose() *

//*************************
//*      bits2aeseq       *
//*************************
//********************************************************************************
//* Convert the color index information into a member of enum aeSeq              *
//* corresponding to the ANSI escape sequence for that color attribute.          *
//*                                                                              *
//* Note: If the aeSeq value is one of the 4-bit "bold" foreground attributes,   *
//* the 'boldMod' flag is set. Otherwise the flag is not modified.               *
//*                                                                              *
//* Input  : acaBits  : member of enum acAttr                                    *
//*          byteIndx : byte index value (fg or bg)                              *
//*          isFgnd   : 'true'  if foreground target                             *
//*                     'false' if background target                             *
//*          isRgb    : (optional, 'false' by default)                           *
//*                     'true'  if 'acaBits' references one of the web-safe      *
//*                             RGB register indices                             *
//*                     'false' if 'acaBits' is a lookup table index:            *
//*                                0-15 : 4-bit color index                      *
//*                              16-231 : 8-bit color index                      *
//*                             232-255 : greyscale index                        *
//*                                                                              *
//* Returns: member of enum aeSeq                                                *
//********************************************************************************

aeSeq acaExpand::bits2aeseq ( acAttr acaBits, uint8_t byteIndx, bool isFgnd, bool isRgb )
{
   aeSeq aesCmd = aesFG_DFLT ;            // return value

   //* If value specifies one of the web-safe RGB color attributes.*
   if ( isRgb )
   {
      if ( isFgnd )  aesCmd = aesFG_RGB ;    // RGB foreground
      else           aesCmd = aesBG_RGB ;    // RGB background
   }

   //* Else, is a lookup table index *
   else
   {
      //* If 4-bit foreground code *
      if ( isFgnd && ((acaBits == acaFG_DFLT) ||
           ((acaBits >= acaFG_BLACK) && (acaBits <= acaFGb_GREY))) )
      {
         switch ( acaBits )
         {
            case acaFG_BLACK:    aesCmd = aesFG_BLACK ;        break ;
            case acaFG_RED:      aesCmd = aesFG_RED ;          break ;
            case acaFG_GREEN:    aesCmd = aesFG_GREEN ;        break ;
            case acaFG_BROWN:    aesCmd = aesFG_BROWN ;        break ;
            case acaFG_BLUE:     aesCmd = aesFG_BLUE ;         break ;
            case acaFG_MAGENTA:  aesCmd = aesFG_MAGENTA ;      break ;
            case acaFG_CYAN:     aesCmd = aesFG_CYAN ;         break ;
            case acaFG_GREY:     aesCmd = aesFG_GREY ;         break ;

            case acaFGb_BLACK:   aesCmd = aesFGb_BLACK ;       break ;
            case acaFGb_RED:     aesCmd = aesFGb_RED ;         break ;
            case acaFGb_GREEN:   aesCmd = aesFGb_GREEN ;       break ;
            case acaFGb_BROWN:   aesCmd = aesFGb_BROWN ;       break ;
            case acaFGb_BLUE:    aesCmd = aesFGb_BLUE ;        break ;
            case acaFGb_MAGENTA: aesCmd = aesFGb_MAGENTA ;     break ;
            case acaFGb_CYAN:    aesCmd = aesFGb_CYAN ;        break ;
            case acaFGb_GREY:    aesCmd = aesFGb_GREY ;        break ;
            default:             aesCmd = aesFG_DFLT ;         break ;
         } ;
         if ( (aesCmd >= aesFGb_BLACK) && (aesCmd <= aesFGb_GREY) )
            this->boldMod = true ;
      }

      //* If 4-bit background code *
      else if ( !isFgnd && ((acaBits == acaBG_DFLT) ||
                ((acaBits >= acaBG_BLACK) && (acaBits <= acaBGb_GREY))) )
      {
         switch ( acaBits )
         {
            case acaBG_BLACK:    aesCmd = aesBG_BLACK ;        break ;
            case acaBG_RED:      aesCmd = aesBG_RED ;          break ;
            case acaBG_GREEN:    aesCmd = aesBG_GREEN ;        break ;
            case acaBG_BROWN:    aesCmd = aesBG_BROWN ;        break ;
            case acaBG_BLUE:     aesCmd = aesBG_BLUE ;         break ;
            case acaBG_MAGENTA:  aesCmd = aesBG_MAGENTA ;      break ;
            case acaBG_CYAN:     aesCmd = aesBG_CYAN ;         break ;
            case acaBG_GREY:     aesCmd = aesBG_GREY ;         break ;

            case acaBGb_BLACK:   aesCmd = aesBGb_BLACK ;       break ;
            case acaBGb_RED:     aesCmd = aesBGb_RED ;         break ;
            case acaBGb_GREEN:   aesCmd = aesBGb_GREEN ;       break ;
            case acaBGb_BROWN:   aesCmd = aesBGb_BROWN ;       break ;
            case acaBGb_BLUE:    aesCmd = aesBGb_BLUE ;        break ;
            case acaBGb_MAGENTA: aesCmd = aesBGb_MAGENTA ;     break ;
            case acaBGb_CYAN:    aesCmd = aesBGb_CYAN ;        break ;
            case acaBGb_GREY:    aesCmd = aesBGb_GREY ;        break ;
            default:             aesCmd = aesBG_DFLT ;         break ;
         } ;
      }

      //* Else, is a non-RGB 8-bit index code (8-bit color or greyscale) *
      else if ( isFgnd )
         aesCmd = aesFG_INDEX ;         // 8-bit foreground
      else
         aesCmd = aesBG_INDEX ;         // 8-bit background
   }
   return aesCmd ;

}  //* End bits2aeseq() *

//*************************
//*      aeseq2bits       *
//*************************
//********************************************************************************
//* Convert between aeSeq (ANSI escape sequence name) and acAttr (bitfield).     *
//* For named escape sequences only; specifically:                               *
//*   a) 4-bit foreground                                                        *
//*   b) 4-bit background                                                        *
//*   c) text modifier flags                                                     *
//* All other aeSeq values return acaPLAINTXT (i.e. zero).                       *
//*                                                                              *
//* Note: If the aeSeq value is one of the 4-bit "bold" foreground or            *
//*       background attributes, the 'boldMod' flag is set. Otherwise the        *
//*       flag is not modified.                                                  *
//*                                                                              *
//* Input  : aesCmd  : member of enum aeSeq                                      *
//*          setFlag : (by reference) for binary tracking flags, receives        *
//*                    the state for setting/resetting the flag                  *
//*                    'true'  set the target flag                               *
//*                    'false' reset the target flag                             *
//*                    (For non-flag commands, this is always 'false' and )      *
//*                    (should be ignored.                                )      *
//*                                                                              *
//* Returns: member of enum acAttr                                               *
//********************************************************************************

acAttr acaExpand::aeseq2bits ( aeSeq aesCmd, bool& setFlag )
{
   acAttr newBits = acaPLAINTXT ;   // return value
   bool handled = true ;            // test control

   setFlag = false ;                // initialize caller's flag

   //* Test for text-modifier commands and indicate *
   //* whether the flag is to be set or reset.      *
   switch ( aesCmd )
   {
      case aesBOLD:
      case aesFAINT:
      case aesBOLD_OFF:
         newBits = acaBOLD ;
         setFlag = aesCmd == aesBOLD_OFF ? false : true ;
         break ;

      case aesITALIC:
      case aesFRACTUR:           // unsupported "fractur" is redirected as italic
      case aesITALIC_OFF:
         newBits = acaITALIC ;
         setFlag = aesCmd == aesITALIC_OFF ? false : true ;
         break ;

      case aesUNDERLINE:
      case aesDBL_UNDERLINE:
      case aesUNDERLINE_OFF:
         newBits = acaUNDERLINE ;
         setFlag = aesCmd == aesUNDERLINE_OFF ? false : true ;
         break ;

      case aesBLINK_SLOW:
      case aesBLINK_FAST:
      case aesBLINK_OFF:
         newBits = acaBLINK ;
         setFlag = aesCmd == aesBLINK_OFF ? false : true ;
         break ;

      case aesREVERSE:
      case aesREVERSE_OFF:
         newBits = acaREVERSE ;
         setFlag = aesCmd == aesREVERSE ? true : false ;
         break ;

      case aesXOUT:
      case aesXOUT_OFF:
         newBits = acaXOUT ;
         setFlag = aesCmd == aesXOUT ? true : false ;
         break ;

      case aesOVERLINE:
      case aesOVERLINE_OFF:
         newBits = acaOVERLINE ;
         setFlag = aesCmd == aesOVERLINE ? true : false ;
         break ;

      case aesCONCEAL:
      case aesCONCEAL_OFF:
         newBits = acaCONCEAL ;
         setFlag = aesCmd == aesCONCEAL ? true : false ;
         break ;

      case aesFRAMED:
      case aesENCIRCLE:
      case aesFRM_ENC_OFF:
         setFlag = aesCmd == aesFRM_ENC_OFF ? false : true ;
         break ;

      default:
         handled = false ;
         break ;
   } ;

   //* If command was not a text modifier, test for  *
   //* 4-bit foreground/background attribute command.*
   if ( ! handled )
   {
      switch ( aesCmd )
      {
         case aesFG_DFLT:     newBits = acaFG_DFLT ;                       break ;
         case aesFG_BLACK:    newBits = acaFG_BLACK ;                      break ;
         case aesFG_RED:      newBits = acaFG_RED ;                        break ;
         case aesFG_GREEN:    newBits = acaFG_GREEN ;                      break ;
         case aesFG_BROWN:    newBits = acaFG_BROWN ;                      break ;
         case aesFG_BLUE:     newBits = acaFG_BLUE ;                       break ;
         case aesFG_MAGENTA:  newBits = acaFG_MAGENTA ;                    break ;
         case aesFG_CYAN:     newBits = acaFG_CYAN ;                       break ;
         case aesFG_GREY:     newBits = acaFG_GREY ;                       break ;
         case aesFGb_BLACK:   newBits = acaFGb_BLACK ;                     break ;
         case aesFGb_RED:     newBits = acaFGb_RED ;                       break ;
         case aesFGb_GREEN:   newBits = acaFGb_GREEN ;                     break ;
         case aesFGb_BROWN:   newBits = acaFGb_BROWN ;                     break ;
         case aesFGb_BLUE:    newBits = acaFGb_BLUE ;                      break ;
         case aesFGb_MAGENTA: newBits = acaFGb_MAGENTA ;                   break ;
         case aesFGb_CYAN:    newBits = acaFGb_CYAN ;                      break ;
         case aesFGb_GREY:    newBits = acaFGb_GREY ;                      break ;

         case aesBG_DFLT:     newBits = acaBG_DFLT ;                       break ;
         case aesBG_BLACK:    newBits = acaBG_BLACK ;                      break ;
         case aesBG_RED:      newBits = acaBG_RED ;                        break ;
         case aesBG_GREEN:    newBits = acaBG_GREEN ;                      break ;
         case aesBG_BROWN:    newBits = acaBG_BROWN ;                      break ;
         case aesBG_BLUE:     newBits = acaBG_BLUE ;                       break ;
         case aesBG_MAGENTA:  newBits = acaBG_MAGENTA ;                    break ;
         case aesBG_CYAN:     newBits = acaBG_CYAN ;                       break ;
         case aesBG_GREY:     newBits = acaBG_GREY ;                       break ;
         case aesBGb_BLACK:   newBits = acaBGb_BLACK ;                     break ;
         case aesBGb_RED:     newBits = acaBGb_RED ;                       break ;
         case aesBGb_GREEN:   newBits = acaBGb_GREEN ;                     break ;
         case aesBGb_BROWN:   newBits = acaBGb_BROWN ;                     break ;
         case aesBGb_BLUE:    newBits = acaBGb_BLUE ;                      break ;
         case aesBGb_MAGENTA: newBits = acaBGb_MAGENTA ;                   break ;
         case aesBGb_CYAN:    newBits = acaBGb_CYAN ;                      break ;
         case aesBGb_GREY:    newBits = acaBGb_GREY ;                      break ;
         default:       // un-handled aeSeq member
            break ;
      } ;
   }

   return newBits ;

}  //* End aeseq2bits() *

//*********************
//*     web2regs      *
//*********************
//********************************************************************************
//* Convert web-safe RGB index into RGB component values.                        *
//* For foreground value, the following are initialized:                         *
//*  fgRgb, fgHue, fgShade, rgbFgR, rgbFgG, rgbFgB, aesFgnd, aesFgType, fgDflt.  *
//* For background value, the following are initialized:                         *
//*  bgRgb, bgHue, bgShade, rgbBgR, rgbBgG, rgbBgB, aesBgnd, aesBgType, bgDflt.  *
//*                                                                              *
//* Note: If webrgb out-of-range, then target members                            *
//*       will default to settings for wsrgbBLACK.                               *
//*                                                                              *
//* Input  : webrgb  : value indicating one of the web-safe RGB register groups  *
//*                    in the range: wsrgbMIN <= webrgb <= wsrgbMAX.             *
//*                    See chart below.                                          *
//*          fgnd    : if true,  'wsrgb' contains foreground data                *
//*                    if false, 'wsrgb' contains background data                *
//*                                                                              *
//* Returns: 'true'  if valid parameters,                                        *
//*          'false' if invalid parameters (defaults used)                       *
//********************************************************************************
//* Specify the value component values representing one of the 216 "web-safe"    *
//* RGB attributes. These include:                                               *
//*  -- "hue" This is one of the eight)8) color groups: black, red, green, blue  *
//*     brown, magenta, cyan and grey.                                           *
//*  -- "shade" This is the gradient within the specified color group. Smaller   *
//*     values of shade result in darker or less intense colors, while larger    *
//*     values result in lighter or more intense colors.                         *
//*  -- R/G/B register values are the indices into the system's 256-element      *
//*     color lookup table.                                                      *
//*                                                                              *
//* The "hue" and "shade" are parameters used to call the ANSI methods:          *
//* acSetWebFg() and acSetWebBg(). The range for these values is define in       *
//* enum WebSafeRGB.                                                             *
//* Because the API-level methods use only the "web-safe" RGB colors, the R/G/B  *
//* register values corresponding to the hue and shade are not actually needed   *
//* within the acaExpand class; however, we record them here in case we decide   *
//* to support the full R/G/B range in a later release.                          *
//*                                                                              *
//*                DARK >>>> LIGHT   CORRESPONDING RGB REGISTER INDICES          *
//* Black        :    0               16;16;16                                   *
//* Red group    :    1      36       22;16;16  >>>>  231;16;16                  *
//* Green group  :   37      72       16;22;16  >>>>  16;231;16                  *
//* Blue group   :   73     108       16;16;22  >>>>  16;16;231                  *
//* Brown group  :  109     144       22;22;16  >>>>  231;231;16                 *
//* Magenta group:  145     180       22;16;22  >>>>  231;16;231                 *
//* Cyan group   :  181     216       16;22;22  >>>>  16;231;231                 *
//* Grey group   :  217     231       22;22;22  >>>>  231;231;231                *
//*                                                                              *
//********************************************************************************

bool acaExpand::web2regs ( uint8_t webrgb, bool fgnd )
{
   bool status = true ;                // return value

   if ( (webrgb < wsrgbMIN) || (webrgb > wsrgbMAX) )  // range check
      webrgb = wsrgbMIN ;
   else
      status = false ;

   //* Initialize the target values.*
   if ( fgnd )    // foreground
   {
      this->fgIndex = webrgb ;
      this->fgRgb = true ;
      this->fgDflt = false ;
      this->aesFgnd = this->aesFgType = aesFG_RGB ;
   }
   else           // background
   {
      this->bgIndex = webrgb ;
      this->bgRgb = true ;
      this->bgDflt = false ;
      this->aesBgnd = this->aesBgType = aesBG_RGB ;
   }

   if ( webrgb == wsrgbBLACK )
   {
      if ( fgnd )    // foreground
      {
         this->rgbFgR = this->rgbFgG = this->rgbFgB = minRGB ;
         this->fgHue = wsrgbBLACK ;
         this->fgShade = wsrgbSHADE_MIN ;
      }
      else           // background
      {
         this->rgbBgR = this->rgbBgG = this->rgbBgB = minRGB ;
         this->bgHue = wsrgbBLACK ;
         this->bgShade = wsrgbSHADE_MIN ;
      }
   }
   else if ( webrgb < wsrgbGREEN )     // RED shades
   {
      if ( fgnd )    // foreground
      {
         this->rgbFgG = this->rgbFgB = minRGB ;
         this->rgbFgR = minRGB + webrgb * wsrgbSTEP ;
         if ( this->rgbFgR > maxRGB ) this->rgbFgR = maxRGB ;
         this->fgHue = wsrgbRED ;
         //* "shade" is the specified index - base color index * 
         this->fgShade = webrgb - wsrgbRED ; 
      }
      else           // background
      {
         this->rgbBgG = this->rgbBgB = minRGB ;
         this->rgbBgR = minRGB + webrgb * wsrgbSTEP ;
         if ( this->rgbBgR > maxRGB ) this->rgbBgR = maxRGB ;
         this->bgHue = wsrgbRED ;
         this->bgShade = webrgb - wsrgbRED ; 
      }
   }
   else if ( webrgb < wsrgbBLUE )      // GREEN shades
   {
      if ( fgnd )    // foreground
      {
         this->rgbFgR = this->rgbFgB = minRGB ;
         this->rgbFgG = minRGB + ((webrgb - wsrgbGREEN + 1) * wsrgbSTEP) ;
         if ( this->rgbFgG > maxRGB ) this->rgbFgG = maxRGB ;
         this->fgHue = wsrgbGREEN ;
         this->fgShade = webrgb - wsrgbGREEN ; 
      }
      else           // background
      {
         this->rgbBgR = this->rgbBgB = minRGB ;
         this->rgbBgG = minRGB + ((webrgb - wsrgbGREEN + 1) * wsrgbSTEP) ;
         if ( this->rgbBgG > maxRGB ) this->rgbBgG = maxRGB ;
         this->bgHue = wsrgbGREEN ;
         this->bgShade = webrgb - wsrgbGREEN ; 
      }
   }
   else if ( webrgb < wsrgbBROWN )     // BLUE shades
   {
      if ( fgnd )    // foreground
      {
         this->rgbFgR = this->rgbFgG = minRGB ;
         this->rgbFgB = minRGB + ((webrgb - wsrgbBLUE + 1) * wsrgbSTEP) ;
         if ( this->rgbFgB > maxRGB ) this->rgbFgB = maxRGB ;
         this->fgHue = wsrgbBLUE ;
         this->fgShade = webrgb - wsrgbBLUE ; 
      }
      else           // background
      {
         this->rgbBgR = this->rgbBgG = minRGB ;
         this->rgbBgB = minRGB + ((webrgb - wsrgbBLUE + 1) * wsrgbSTEP) ;
         if ( this->rgbBgB > maxRGB ) this->rgbBgB = maxRGB ;
         this->bgHue = wsrgbBLUE ;
         this->bgShade = webrgb - wsrgbBLUE ; 
      }
   }
   else if ( webrgb < wsrgbMAGENTA )   // BROWN shades
   {
      if ( fgnd )    // foreground
      {
         this->rgbFgB = minRGB ;
         this->rgbFgR = this->rgbFgG = 
                              minRGB + ((webrgb - wsrgbBROWN + 1) * wsrgbSTEP) ;
         if ( this->rgbFgR > maxRGB )
            this->rgbFgR = this->rgbFgG = maxRGB ;
         this->fgHue = wsrgbBROWN ;
         this->fgShade = webrgb - wsrgbBROWN ;
      }
      else           // background
      {
         this->rgbBgB = minRGB ;
         this->rgbBgR = this->rgbBgG = 
                              minRGB + ((webrgb - wsrgbBROWN + 1) * wsrgbSTEP) ;
         if ( this->rgbBgR > maxRGB )
            this->rgbBgR = this->rgbBgG = maxRGB ;
         this->bgHue = wsrgbBROWN ;
         this->bgShade = webrgb - wsrgbBROWN ; 
      }
   }
   else if ( webrgb < wsrgbCYAN )      // MAGENTA shades
   {
      if ( fgnd )    // foreground
      {
         this->rgbFgG = minRGB ;
         this->rgbFgR = this->rgbFgB = 
                              minRGB + ((webrgb - wsrgbMAGENTA + 1) * wsrgbSTEP) ;
         if ( this->rgbFgR > maxRGB )
            this->rgbFgR = this->rgbFgB = maxRGB ;
         this->fgHue = wsrgbMAGENTA ;
         this->fgShade = webrgb - wsrgbMAGENTA ; 
      }
      else           // background
      {
         this->rgbBgG = minRGB ;
         this->rgbBgR = this->rgbBgB = 
                              minRGB + ((webrgb - wsrgbMAGENTA + 1) * wsrgbSTEP) ;
         if ( this->rgbBgR > maxRGB )
            this->rgbBgR = this->rgbBgB = maxRGB ;
         this->bgHue = wsrgbMAGENTA ;
         this->bgShade = webrgb - wsrgbMAGENTA ; 
      }
   }
   else if ( webrgb < wsrgbGREY )      // CYAN shades
   {
      if ( fgnd )    // foreground
      {
         this->rgbFgR = minRGB ;
         this->rgbFgG = this->rgbFgB = 
                              minRGB + ((webrgb - wsrgbCYAN + 1) * wsrgbSTEP) ;
         if ( this->rgbFgG > maxRGB )
            this->rgbFgG = this->rgbFgB = maxRGB ;
         this->fgHue = wsrgbCYAN ;
         this->fgShade = webrgb - wsrgbCYAN ; 
      }
      else           // background
      {
         this->rgbBgR = minRGB ;
         this->rgbBgG = this->rgbBgB = 
                              minRGB + ((webrgb - wsrgbCYAN + 1) * wsrgbSTEP) ;
         if ( this->rgbBgG > maxRGB )
            this->rgbBgG = this->rgbBgB = maxRGB ;
         this->bgHue = wsrgbCYAN ;
         this->bgShade = webrgb - wsrgbCYAN ; 
      }
   }
   else if ( webrgb <= wsrgbMAX )      // GREYSCALE group
   {
      if ( fgnd )    // foreground
      {
         this->rgbFgR = this->rgbFgG = this->rgbFgB = 
                              minRGB + ((webrgb - wsrgbGREY + 1) * wsrgbSTEP) ;
         if ( this->rgbFgR > maxRGB )
            this->rgbFgR = this->rgbFgG = this->rgbFgB = maxRGB ;
         this->fgHue = wsrgbGREY ;
         this->fgShade = webrgb - wsrgbGREY ; 
      }
      else           // background
      {
         this->rgbBgR = this->rgbBgG = this->rgbBgB = 
                              minRGB + ((webrgb - wsrgbGREY + 1) * wsrgbSTEP) ;
         if ( this->rgbBgR > maxRGB )
            this->rgbBgR = this->rgbBgG = this->rgbBgB = maxRGB ;
         this->bgHue = wsrgbGREY ;
         this->bgShade = webrgb - wsrgbGREY ; 
      }
   }
   return status ;

}  //* End web2regs() *

//*********************
//*       reset       *
//*********************
//********************************************************************************
//* Reset (clear) all data members (or text-modifier flags only).                *
//*                                                                              *
//* 1) All data members:                                                         *
//*    The new settings will reflect terminal-default foreground and background  *
//*    attributes with no text modifiers.                                        *
//* 2) Text modifiers only:                                                      *
//*    All text-modifier flags will be reset, and other data will be unchanged.  *
//*                                                                              *
//* Input  : modsOnly : (optional, 'false' by default)                           *
//*                     if 'false', reset all data members                       *
//*                     if 'true',  reset text-modifier flags only               *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void acaExpand::reset ( bool modsOnly )
{
   acAttr newBits ;

   if ( modsOnly )
      newBits = acAttr(this->acaVal & ~(acaMOD_MASK)) ;
   
   //* Reset all members to default fgnd/bgnd with no modifiers *
   else
      newBits = acaATTR_DFLT ;

   this->decode ( newBits ) ;

}  //* End reset() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

