//********************************************************************************
//* File       : AnsiCmdTest.cpp                                                 *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2022-2023 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 21-Aug-2023                                                     *
//* Version    : (see AnsiCmdVersion string in AnsiCmd.cpp)                      *
//*                                                                              *
//* Description: AnsiCmd class test methods. All these methods are under         *
//*              conditional-compilation statements and are not required by      *
//*              the AnsiCmd mainline functionality.                             *
//*              The primary purpose of the test methods is to determine         *
//*              which ANSI commands are supported by the host terminal          *
//*              program.                                                        *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************
#include "AnsiCmd.hpp"     //* AnsiCmd-class definition
#include "AnsiCmdWin.hpp"  //* AnsiCmd-class window-object definitions

//** Compilation of the entire module is controlled by this statement. **
#if DEBUG_ANSICMD != 0

//***************
//* Definitions *
//***************
#if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
extern ofstream ofsdbg ;
extern gString  gsdbg ;
#endif   // DEBUG_ANSICMD && DEBUG_LOG

//**************
//* Local data *
//**************

//** ANSI commands (escape sequences)             **
//** These are indexed via members of enum aeSeq. **
extern const wchar_t *ansiSeq[aesCOUNT] ;

//******************
//* Static Methods *
//******************
static void tuiReport ( gString& gs, const void *buff, short cnt, bool wStream ) ;
static void formatRgbString ( gString& gsOut, short colWidth, uint8_t wsrgb ) ;
static bool twDecodeFgBg ( gString& gsIn, short& fgVal, short& bgVal, 
                           bool& isRgbFg, bool& isRgbBg ) ;


//*************************
//*     ansiTest_Menu     *
//*************************
//********************************************************************************
//* Perform the specified test of terminal functionality. This functionality     *
//* is divided into different groups (see below). There will of course be some   *
//* overlap in the tests, but we try to minimize it to produce clean tests.      *
//*                                                                              *
//* Major  Minor   Description                                                   *
//* -----  -----   ------------------------------------------------------------- *
//*  'C'           Test color attributes                                         *
//*         'f'    Test the 16 basic foreground (text) and background colors     *
//*                as defined by the SGR (Select Graphics Rendition) table       *
//*                for 3-bit and 4-bit color engines.                            *
//*                These are the 8 base colors and 8 bold (bright/intense)       *
//*                colors.                                                       *
//*                These are designated by color pair, 30-37 and 39 (default).   *
//*         'b'    Test the 16 basic background colors as defined by             *
//*                SGR (Select Graphics Rendition) table for 3-bit and 4-bit     *
//*                color engines.                                                *
//*                These are the 8 base colors and 8 bold (bright/intense)       *
//*                colors. These are designated by color pair, 40-47 and 49      *
//*                (default). These bold attributes are compared with the        *
//*                AnsiCmd synthesized bold background colors and the AIX        *
//*                (non-standard) bold background attributes.                    *
//*         't'    Test the 216 predefined, 8-bit colors, (three(3) registers    *
//*                per group). The color combinations accessed through these     *
//*                groups are pre-defined by the terminal to provide             *
//*                predictable incremental contrast across the color spectrum.   *
//*                  Test the foreground colors designated by the                *
//*                  sequences: \x1B[38;5;nm  where 'n' is the color index       *
//*                             \x1B[38;2,r;g;b  where 'r' 'g' 'b' are the       *
//*                                       real (or virtual) RGB registers        *
//*                  Test the background colors designated by the                *
//*                  sequences: \x1B[48;5;nm  where 'n' is the color index       *
//*                             \x1B[48;2,r;g;b  where 'r' 'g' 'b' are the       *
//*                                       real (or virtual) RGB registers        *
//*         'g'    Test the greyscale colors defined as index 232-255 in the     *
//*                256 color lookup table (24 greyscale steps).                  *
//*                Note that this test uses the same \x1B[38;5;nm and            *
//*                \x1B[48;5;nm indexing method as the 8-bit table lookup        *
//*                described for the previous test.                              *
//*         'r'    Test any combination of the RGB (Red/Green/Blue) registers    *
//*                available in the 8-bit color spectrum by selecting the        *
//*                individual RGB registers.                                     *
//*                Supplimentary Command indicates the sub-test to perform:      *
//*                'c' step through the RGB colors in a compressed format        *
//*                    This is the default option for the RGB test group.        *
//*                'r' step through the Red spectrum                             *
//*                 'g' step through the Green spectrum                          *
//*                 'b' step through the Blue spectrum                           *
//*                'a' step through all colors simultaneously (diagonal)         *
//*                                                                              *
//*                While the terminal will define the actual values used to      *
//*                initialize the individual RGB registers (real or virtual),    *
//*                depending on the platform, these preset values can often      *
//*                be edited manually. That operation requires detailed          *
//*                knowledge of the terminal definition file and is not          *
//*                described here.                                               *
//*                For more information, please refer to the NcDialog API        *
//*                documentation by the same author, and specifically to the     *
//*                Dialog2 test application, Test #4.                            *
//*         'a'    Test the Aixterm extensions for setting intense foreground    *
//*                and intense background. These extensions (if implemented)     *
//*                do not use the intense (bold) modifier flag. The internal     *
//*                implementation is unclear. Presumably they use an algorithm   *
//*                similar to the way AnsiCmd implements synthesized             *
//*                intense background color attributes.                          *
//*                -- Note: These are ANSI codes 90-97 (bright foreground)       *
//*                   and 100-107 (bright background). They were defined for     *
//*                   the IBM AIX operating system for partial compatibility     *
//*                   with UNIX/Linux systems.                                   *
//*         'd'    Test decoding of acaExpand class. Ask the user to modify      *
//*                various color-attributes and text modifiers, and verify       *
//*                that the tracking fields are accurately updated.              *
//*                'a' use API-level methods (default)                           *
//*                'b' use Basic low-level ANSI escape sequence methods directly *
//*                                                                              *
//*  'P'           Test cursor-positioning commands. The commands may indicate   *
//*                absolute row or column positions or movement relative to      *
//*                the current cursor position.                                  *
//*                                                                              *
//*  'E'           Test erasure commands. These commands erase existing text     *
//*                specific areas of the terminal window.                        *
//*                                                                              *
//*  'U'           Test the unbuffered input stream. The wide stream is          *
//*                assumed EXCEPT when the 'minor' argument (color attribute)    *
//*                is set to 'n' (Brown).                                        *
//*                                                                              *
//*  'A'           Test the ASCII control codes. These one-byte ASCII values     *
//*                perform the same function as some of the ANSI escape          *
//*                sequences. Many, if not all of these are captured by the      *
//*                terminal, so this test may not be very useful.                *
//*                                                                              *
//*  'F'           Test the alternate-font commands. In addition to the          *
//*                primary terminal font, up to nine(9) alternate fonts may      *
//*                be supported.                                                 *
//*                                                                              *
//*  'I'           Test the Ideogram group of commands. It is likely that        *
//*                these commands are not supported by any mainstream            *
//*                terminal emulation software.                                  *
//*                -- The "ideogram" group appears to have been an abandoned     *
//*                   attempt to define text borders or highlighting.            *
//*                                                                              *
//*  'T'           Test terminal setup options and validate the member           *
//*                variable values that track terminal settings.                 *
//*         'i'    Test input stream capture methods under both buffered and     *
//*                unbuffered input data.                                        *
//*         'e'    Test the "soft-echo" functionality which captures input       *
//*                data when terminal-controlled echo is off and echoes the      *
//*                data after applying certain filters to prevent garbage        *
//*                being written to the terminal window.                         *
//*                'a' All soft-echo special keys translated:                    *
//*                'c' Cursor movement keys translated:                          *
//*                'e' Edit keys (bksp/del/ins) translated:                      *
//*                'd' Disable translation of special keycodes:                  *
//*                'r' Report special keycode translations:                      *
//*                't' Terminal echo of key input:                               *
//*                'n' No echo of key input:                                     *
//*                                                                              *
//*         't'    Test terminal-modification options and tracking-members for   *
//*                buffering, input echo, cursor type and panic-key capture.     *
//*         'b'    Test non-Blocking read of the input stream (stdin).           *
//*                This exercises the AnsiCmd methods for enabling and           *
//*                disabling non-blocking reads. A non-blocking read is useful   *
//*                for capturing escape sequences and the Escape key itself.     *
//*         'm'    Test text modifiers such as Bold, Italic, Underline,          *
//*                Reverse, Strikethrough, Overline, etc. This also validates    *
//*                the modifier-tracking flags.                                  *
//*                                                                              *
//*  'W'           Window-oriented tests (application-level functionality).      *
//*         'w'    Window. Basic window functionality: instantiate, open,        *
//*                close, clear text, write unformatted text, auto line break,   *
//*                adjust color and text attributes.                             *
//*         'f'    Form. Exercise formatted-text fields within a window.         *
//*                This is the skForm class functionality.                       *
//*         'd'    Default constructor for the ACWin class. Exercise various     *
//*                window-configuration methods.                                 *
//*         'b'    Box. Demonstrate drawing of boxes (AC_Box class).             *
//*         'l'    Line drawing with the acDrawLine() method.                    *
//*                                                                              *
//*  'S'           Sandbox. This command executes user-defined functionality     *
//*                created for ad hoc testing. Code in the sandbox will be       *
//*                affected by the AnsiCmd class initialization. Be aware.       *
//*                For details, see the TermConfig class which is passed to      *
//*                the AnsiCmd constructor.                                      *
//*                                                                              *
//*  '?'           If a parameter contains the '?' character, it indicates       *
//*                that user did not specify that argument.                      *
//*                                                                              *
//*                                                                              *
//* Input  : args :                                                              *
//*            major   : specifies the test to perform (see table above)         *
//*            minor   : specifies test sub-section    (see table above)         *
//*            supp    : specifies sub-test for RGB _or_ Echo (see table above)  *
//*            attr    : specifies alternate foreground color attribute          *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Notes:                                                          *
//* 1) These tests have been performed on our favorite terminal programs:        *
//*      Gnometerm v:3.40.3                                                      *
//*      Xterm v:366                                                             *
//*      Konsole v:21.08.3                                                       *
//*    Any notes about available functionality refer to execution on these       *
//*    terminal emulators.                                                       *
//*                                                                              *
//* 2) This method makes some intimate assumptions about the way ANSI escape     *
//*    sequences are constructed and what data they contain. We feel fairly      *
//*    comfortable taking these liberties because the ANSI definitions are       *
//*    well-established and because this is "only" a test method.                *
//*    (People familiar with our work, know there is no such thing as "only".)   *
//*    (The author is a nerd with OCD; but then, aren't we all?!             )   *
//*                                                                              *
//* 3) On Gnometerm v:3.40.3 the "bold-off/dbl-underline" (21) produces a        *
//*    double underline. Also on this terminal the "neither-bold-nor-dim"        *
//*    (aesBOLDE_OFF==22) DOES work to disable bold text.                        *
//*    For this reason, we _optionally_ invoke the "Reset" (aesRESET==0)         *
//*    to switch off bold/dim text.                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

void AnsiCmd::ansiTest_Menu ( dbgArgs& args )
{
   const wchar_t *testOptions = 
           L"     ANSI Command - Test Options\n"
            "-----------------------------------------------\n"
            "'C'  Color: color-attribute tests.\n"
            "'P'  Positioning: cursor-positioning tests.\n"
            "'E'  Erasure: text-erasure (clear-area) tests.\n"
            "'A'  ASCII: control code tests.\n"
            "'F'  Font: alternate-font selection tests.\n"
            "'I'  Ideogram: ideogram (logogram) group tests.\n"
            "'T'  Terminal: Setup options, data validation.\n"
            "'W'  Window: Application-level functionality.\n"
            "'S'  Sandbox: Design temporary tests.\n" ;
   const wchar_t *attrOptions = 
           L"     Color Attribute Testing Options:\n"
            "-----------------------------------------------\n"
            "'f'  Four-bit Fgnd/Bkgnd color attributes.\n"
            "'b'  Background color attribute options.\n"
            "'t'  Table lookup, 8-bit color attributes.\n"
            "'r'  R/G/B color-attribute matrix (6x6x6).\n"
            "'g'  Greyscale color attributes.\n"
            "'a'  Aixterm Fg/Bg extensions (non-standard).\n"
            "'d'  Decode tracking data for attribute mods.\n" ;
   const wchar_t *termOptions = 
           L"'i'  Input Stream capture methods.\n"
            "'e'  Echo options for key input.\n"
            "'t'  Terminal-setup flags verification.\n"
            "'b'  Non-blocking read from 'stdin'.\n"
            "'m'  Modifiers, text-mods verification.\n" ;
   const wchar_t *echoOptions =
           L"'a' All soft-echo special keys translated.\n"
            "'c' Cursor-movement special keys translated.\n"
            "'e' Edit keys: Backspace/Del/Insert translated.\n"
            "'d' Disable soft-echo special-key translation.\n"
            "'r' Report special key translations.\n"
            "'t' Terminal echo of key input.\n"
            "'n' No echo of key input.\n" ;
   const wchar_t *windowOptions =
           L"'w'  ACWin-class (basic test)\n"
            "'f'  skForm-class (text fields)\n"
            "'d'  ACWin-class (default window)\n"
            "'b'  AC_Box-class (draw rectangles)\n"
            "'l'  Line drawing\n" ;

   //* Normalize upper/lower case *
   args.major = toupper ( args.major ) ;
   args.minor = tolower ( args.minor ) ;
   args.supp  = tolower ( args.supp ) ;
   args.attr  = tolower ( args.attr ) ;

   //* If alternate foreground attribute specified, *
   //* move it to the 'attr' field.                 *
   //* Note that color attribute is ignored for     *
   //* 'C' (color) and 'S' (sandbox) tests.         *
   aeSeq fg ;           // selected foreground color for test
   if ( args.major != L'?' && (args.major != L'C') && (args.major != L'S') )
   {
      //* If 'attr' not initialized, initialize it  *
      //* to the color selection character (if any).*
      if ( args.attr == L'?' )         // if 'attr' not initialized
      {
         //* 'W'indow Tests *
         if ( args.major == L'W' )
         {
            switch ( args.minor )
            {
               case L'w':     // supplementary argument will be color attribute
               case L'f':
               case L'd':
               case L'b':
               case L'l':
                  args.attr = args.supp ;
                  break ;
               default:
                  break ;
            }
         }
         //* 'T'erminal Tests *
         else if ( args.major == L'T' )
         {
            switch ( args.minor )
            {
               case L'i':     // supplementary argument will be color attribute
               case L't':
               case L'b':
               case L'm':
                  args.attr = args.supp ;
                  break ;
               case L'e':     // 'attr' is already set or is unused
               default:
                  break ;
            }
         }
         //* Other tests have no secondary arguments *
         else
            args.attr = args.minor ;
      }
      //* Decode the color selection *
      switch ( args.attr )
      {
         case L'k':  fg = aesFG_BLACK ;      break ;
         case L'r':  fg = aesFG_RED ;        break ;
         case L'g':  fg = aesFG_GREEN ;      break ;
         case L'n':  fg = aesFG_BROWN ;      break ;
         case L'b':  fg = aesFG_BLUE ;       break ;
         case L'm':  fg = aesFG_MAGENTA ;    break ;
         case L'c':  fg = aesFG_CYAN ;       break ;
         case L'y':  fg = aesFG_GREY ;       break ;
         default:    fg = aesFG_DFLT ;       break ;
      } ;
   }
   //* For decoding test, if unspecified (or invalid) argument, set default.*
   if ( (args.major == L'C') && (args.minor == 'd') && (args.supp != L'b') )
      args.supp = L'a' ;

   //* Set all ANSI attributes and attribute-tracking     *
   //* members to their default values.                   *
   this->acReset () ;

   //* Call the system to clear the terminal window, *
   //* then write the title and test parameters.     *
   // Programmer's Note: We call the system to clear the window because 
   // it is not known at this point whether the terminal supports the 
   // ANSI clear-window command.
   system ( "clear" ) ;
   gString gsOut ;
   //* For tests that use the 'supp' parameter, report it. *
   if ( (args.major == 'C' && ((args.minor == 'r') || (args.minor == 'd'))) || 
        (args.major == 'T' && args.minor == 'e') ||
        (args.major == 'S') )
   {
      gsOut.compose( "Test ANSI Commands: '%C' '%C' '%C'\n" // text formatting
                     "-------------------------------\n\n", 
                     &args.major, &args.minor, &args.supp ) ;
   }
   else
   {
      gsOut.compose( "Test ANSI Commands: '%C' '%C'\n" // text formatting
                     "---------------------------\n\n", 
                     &args.major, &args.minor ) ;
   }
   this->ttyWrite ( gsOut ) ;

   if ( args.major != L'?' )
   {
      if ( args.major == L'C' )                          // Color tests
      {
         if ( args.minor == L'f' )                       // Basic 4-bit Fg/Bg
            this->Test_4Bit () ;

         else if ( args.minor == L'b' )                  // Background
            this->Test_Bg () ;

         else if ( args.minor == L't' )                  // Table lookup
            this->Test_8bitColor () ;

         else if ( args.minor == L'g' )                  // Greyscale
            this->Test_Greyscale () ;

         else if ( args.minor == L'r' )                  // RGB
            this->Test_RGB ( args.supp ) ;

         else if ( args.minor == L'a' )                  // Aixterm
            this->Test_AixFgBg () ;

         else if ( args.minor == 'd' )                   // Decode tracking data
            this->Test_acaExpand ( args.supp ) ;

         //* Sub-test not specified, or invalid option, display test options.*
         else
         {
            this->ttyWrite ( attrOptions, true ) ;
         }
      }
      else if ( args.major == L'S' )      // Sandbox
      {
         this->Test_Sandbox ( args.minor, args.supp ) ;
      }
      else if ( args.major == L'P' )      // Positioning test
         this->Test_CursorPos ( fg ) ;
      else if ( args.major == L'E' )      // Erasure test
         this->Test_Erasures ( fg ) ;
      else if ( args.major == L'A' )      // ASCII control-code test
         this->Test_AsciiCodes ( fg ) ;
      else if ( args.major == L'F' )      // Alternate-font test
         this->Test_AltFonts ( fg ) ;
      else if ( args.major == L'I' )      // Ideogram-group test
         this->Test_Ideogram ( fg ) ;

      else if ( args.major == L'T' )      // Terminal Setup
      {
         if ( args.minor == L'i' )        // input stream test
            this->Test_UnbufferedInput ( fg ) ;
         else if ( args.minor == L'e' )   // input-echo test
         {
            if ( (args.supp == 'a') || (args.supp == 'c') || (args.supp == 'e') || 
                 (args.supp == 'd') || (args.supp == 'r') || (args.supp == 't') || 
                 (args.supp == 'n') )
            { this->Test_SoftEcho ( args.supp, fg ) ; }
            else
               this->ttyWrite ( echoOptions ) ;
         }
         else if ( args.minor == L'b' )   // non-blocking read test
            this->Test_NonBlock ( fg ) ;
         else if ( args.minor == L't' )   // terminal-flags test
            this->Test_TermFlags ( fg ) ;
         else if ( args.minor == L'm' )   // mods-flags test
            this->Test_ModFlags ( fg ) ;
         else
            this->ttyWrite ( termOptions ) ;
      }

      else if ( args.major == L'W' )      // Window-definition tests
      {
         if ( args.minor == L'w' )
            this->Test_Window ( fg ) ;
         else if ( args.minor == L'f' )
            this->Test_Form ( fg ) ;
         else if ( args.minor == L'd' )
            this->Test_WinDefault ( fg ) ;
         else if ( args.minor == L'b' )
            this->Test_Box ( fg ) ;
         else if ( args.minor == L'l' )
            this->Test_Line ( fg ) ;
         else
            this->ttyWrite ( windowOptions ) ;
      }
      else
      { this->ttyWrite ( testOptions ) ; }
   }
   else
   { this->ttyWrite ( testOptions ) ; }

}  //* End ansiTest_Menu() *

//*************************
//*       Test_4Bit       *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test 4-bit foreground and background color sequences.                        *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: See also the Test_Bg() method which displays the Intense  *
//* (Bold) background attributes in basic, synthesized and AIX configurations.   *
//********************************************************************************

void AnsiCmd::Test_4Bit ( void )
{
   //* Loop control: two groups of 8 color sequences + the default sequence *
   const short loopCNT = 17 ;
   const wchar_t *ColorName[loopCNT] = 
   { 
      L"00) Black", L"01) Red",     L"02) Green", L"03) Brown", 
      L"04) Blue",  L"05) Magenta", L"06) Cyan",  L"07) Grey",
      L"08) Intense Black", L"09) Intense Red", L"10) Intense Green", 
      L"11) Intense Brown", L"12) Intense Blue", L"13) Intense Magenta", 
      L"14) Intense Cyan",  L"15) Intense Grey", L"--) Default"
   } ;

   const wchar_t *DescFg = 
         L"Foreground Colors Test:\n"
          "-----------------------------------------------------------\n"
          "The terminal defines sixteen(16) basic foreground colors:\n"
          "   Eight(8) standard colors, plus\n"
          "   Eight(8) intense (bold) colors. (std + \"intense\" bit).\n"
          "The actual color for each of these is defined in the\n"
          "terminal's setup.\n"
          "For Gnometerm and Konsole this is the \"Profile\".\n"
          "For Xterm, these colors are defined in the \".Xdefaults\"\n"
          "or \".Xresources\" file, but note that under Wayland the\n"
          "file will not be read.\n"
          "\n"
          "Please note that if the foreground and background are the\n"
          "same color, the text will be invisible.\n"
          "See the Intense Grey example which causes the text to be\n"
          "rendered as white. We display it on a grey background to\n"
          "distinguish the text from the (presumed) white background." ;
   const wchar_t *DescBg = 
         L"Background Colors Test:\n"
          "------------------------------------------------------------\n"
          "The terminal defines sixteen(16) basic background colors:\n"
          "   Eight(8) standard colors, plus\n"
          "   Eight(8) intense (bold) colors. (std + \"intense\" bit).\n"
          "The actual color for each of these is defined in the\n"
          "terminal's setup.\n"
          "For Gnometerm and Konsole this is the \"Profile\".\n"
          "For Xterm, these colors are defined in the \".Xdefaults\"\n"
          "or \".Xresources\" file, but note that under Wayland the\n"
          "the file will not be read.\n"
          "\n"
          "Unfortunately, the default intense background colors are not\n"
          "rendered as expected because the \"intense\" bit affects\n"
          "only the foreground attribute. For this reason, we provide\n"
          "synthesized versions of the intense background attributes.\n"
          "See the background-specific test for additional information." ;

   const short baseROW = 3 ;     // base display row
   const short exitROW = this->termRows - 1 ; // cursor row when preparing to exit
   const short descCOL = 60 ;    // offset for description text
   const short dumpCOL = 22 ;    // offset for settings dump

   gString gsOut ;               // text formatting
   short row = baseROW - 1,      // output row
         bgndRow,                // first row for display of background attributes
         maxindx = aesFG_BLACK + loopCNT, // loop termination
         cnindx  = ZERO ;        // 'ColorName' index
   //* Warn about minimum terminal window dimensions.*
   this->acWrite ( 1, 28, "  (requires terminal dimensions of 39 rows X 155 columns)" ) ;

   //* Clear the heading row and display the column headings.*
   this->acSetCursor ( aesCUR_ABSPOS, row, 1 ) ;
   this->acEraseArea ( aesERASE_LINE ) ;
   this->acCaptureAttr ( gsOut, 1, true, true ) ;
   this->acWrite ( row, dumpCOL, gsOut ) ;
   this->acSetCursor ( aesCUR_ABSPOS, ++row, 1 ) ;

   //** Test aesFG_xxxx group **
   for ( short regindx = aesFG_BLACK ; regindx < maxindx ; ++regindx, ++row )
   {
      if ( regindx == aesFGb_GREY )
         this->acSetBg ( aesBG_BLACK ) ;
      this->acSetFg ( aeSeq(regindx) ) ;        // set the foreground color
      this->ttyWrite ( ColorName[cnindx++] ) ; // write the text data

      //* Capture and format the tracking data *
      this->acCaptureAttr ( gsOut, 1, true ) ;

      this->acSetFg ( aesFG_DFLT ) ;            // return to default foreground
      if ( regindx == aesFGb_GREY )
         this->acSetBg ( aesBG_DFLT ) ;
      this->acFlushOut () ;                     // flush the output stream

      //* Report the tracking data *
      this->acWrite ( row, dumpCOL, gsOut, true, false ) ;
   }
   bgndRow = ++row ;                            // leave some space
   cnindx = ZERO ;                              // reset the array index
   maxindx = aesBG_BLACK + loopCNT ;            // loop termination
   this->acSetCursor ( aesCUR_ABSPOS, row, 1 ) ;

   //** Test aesBG_xxxx group **
   for ( short regindx = aesBG_BLACK ; regindx < maxindx ; ++regindx, ++row )
   {
      //* Make foreground visible for black backgrounds.       *
      //* (This assumes that the default foreground is black.) *
      if ( (regindx == aesBG_BLACK) || (regindx == aesBGb_BLACK) )
         this->acSetFg ( aesFG_GREY ) ;
      this->acSetBg ( aeSeq(regindx), false ) ; // set the background color
      this->ttyWrite ( ColorName[cnindx++] ) ; // write the text data

      //* Capture and format the tracking data *
      this->acCaptureAttr ( gsOut, 1, true ) ;

      this->acSetBg ( aesBG_DFLT ) ;            // return to default foreground
      if ( (regindx == aesBG_BLACK) || (regindx == aesBGb_BLACK) )
         this->acSetFg ( aesFG_DFLT ) ;
      this->acFlushOut () ;                     // flush the output stream

      //* Report the tracking data *
      this->acWrite ( row, dumpCOL, gsOut, true, false ) ;
   }

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   this->acWrite ( (baseROW - 1), descCOL, DescFg, false ) ;
   this->acWrite ( bgndRow, descCOL, DescBg, true ) ;

   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;

}  //* End Test_4Bit() *

//*************************
//*        Test_Bg        *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test background color sequences. The Intense (Bold) sequences are rendered   *
//* in three flavors:                                                            *
//*   1) Basic 4-bit i.e. table indices 8-15.                                    *
//*   2) Synthesized 4-bit, based on the brightest of the R/G/B combinations     *
//*   3) AIX implementation of Intense background colors.                        *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_Bg ( void )
{
   //* Loop control: two groups of 8 color sequences + the default sequence *
   const short loopCNT = 17 ;
   const wchar_t *ColorName[loopCNT] = 
   { L" 00) Black   \n",
     L" 01) Red     \n",
     L" 02) Green   \n",
     L" 03) Brown   \n", 
     L" 04) Blue    \n",
     L" 05) Magenta \n",
     L" 06) Cyan    \n",
     L" 07) Grey    \n",
     L" 08) Intense Black   \n",
     L" 09) Intense Red     \n",
     L" 10) Intense Green   \n", 
     L" 11) Intense Brown   \n",
     L" 12) Intense Blue    \n",
     L" 13) Intense Magenta \n", 
     L" 14) Intense Cyan    \n",
     L" 15) Intense Grey    \n",
     L" --) Default         \n"
   } ;
   const wchar_t *AIX_Colors[] = 
   {
     L" 100) Intense Black Background   \n", L" 101) Intense Red Background     \n", 
     L" 102) Intense Green Background   \n", L" 103) Intense Brown Background   \n", 
     L" 104) Intense Blue Background    \n", L" 105) Intense Magenta Background \n", 
     L" 106) Intense Cyan Background    \n", L" 107) Intense Grey Background    \n",
   } ;
   const wchar_t *Desc_a = 
         L"Background Colors Test:\n"
          "------------------------------------------------------------\n"
          "The terminal defines sixteen(16) basic background colors:\n"
          "   Eight(8) standard colors, plus\n"
          "   Eight(8) intense (bold) colors. (std + \"intense\" bit).\n"
          "The actual color for each of these is defined in the\n"
          "terminal's setup.\n"
          "For Gnometerm and Konsole this is the \"Profile\".\n"
          "For Xterm, these colors are defined in the \".Xdefaults\"\n"
          "or \".Xresources\" file, but note that under Wayland the\n"
          "the file will not be read.\n"
          "\n" ;
   const wchar_t *Desc_b = 
         L"Unfortunately, the default intense background colors are not\n"
          "rendered as expected because the \"intense\" bit affects\n"
          "only the foreground attribute. For this reason, we provide\n"
          "synthesized versions of the intense background attributes\n"
          "which are displayed in the second columm.\n"
          "\n"
          "Displayed in the third column, are the intense background\n"
          "attributes defined as part of the IBM AIX operating system\n"
          "as a means of terminal compatibility between AIXterm and\n"
          "Linux. Although the Aixterm foreground and background color\n"
          "attribute commands are not part of the ANSI standard, they\n"
          "are supported by many Linux-based terminal emulators." ;

   const short baseROW  = 4 ;    // base display row
   const short exitROW  = 28 ;   // cursor row when preparing to exit
   const short descCOL  = 85 ;   // offset for description text
   const short synthCOL = 25 ;   // column for display of synthesized values
   const short aixCOL   = 49 ;   // column for display of AIX values

   gString gsOut ;               // text formatting
   WinPos  wp( baseROW, 1 ) ;    // text position
   short maxindx = aesBG_BLACK + loopCNT, // loop termination
         cnindx  = ZERO ;        // 'ColorName' index

   //** Test aesBG_xxxx group **
   for ( short regindx = aesBG_BLACK ; regindx < maxindx ; ++regindx )
   {
      //* Make foreground visible for black backgrounds.       *
      //* (This assumes that the default foreground is black.) *
      if ( (regindx == aesBG_BLACK) || (regindx == aesBGb_BLACK) )
         this->acSetFg ( aesFG_GREY ) ;
      this->acSetBg ( aeSeq(regindx), false ) ; // set the background color
      wp = this->acWrite ( wp, ColorName[cnindx++], false ) ; // write the text data 
      if ( (regindx == aesBG_BLACK) || (regindx == aesBGb_BLACK) )
         this->acSetFg ( aesFG_DFLT ) ;         // return to default foreground
      this->acFlushOut () ;                     // flush the output stream
   }

   //* Display the synthesized background color attributes *
   short synthRow = baseROW + 7 ;
   cnindx = 8 ;
   wp = { synthRow, synthCOL } ;
   wp = this->acWrite ( wp, L"Synthesized Attributes\n" ) ;
   for ( short regindx = aesBGb_BLACK ; regindx <= aesBGb_GREY ; ++regindx )
   {
      if ( regindx == aesBGb_BLACK )
         this->acSetFg ( aesFG_GREY ) ;
      this->acSetBg ( aeSeq(regindx), true ) ;  // set the background color
      //* Write the text data *
      wp = this->acWrite ( wp, ColorName[cnindx++], false ) ;
      this->acSetBg ( aesFG_DFLT ) ;            // return to default foreground
      if ( regindx == aesBGb_BLACK )
         this->acSetFg ( aesFG_DFLT ) ;
      this->acFlushOut () ;                     // flush the output stream
   }
   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;

   //* Display the AIX intense background color attributes *
   synthRow = baseROW + 7 ;
   cnindx = ZERO ;
   wp = { synthRow, aixCOL } ;
   wp = this->acWrite ( wp, L"AIXterm Background Attributes\n" ) ;

   for ( short regindx = aesAIXBGb_BLACK ; regindx <= aesAIXBGb_GREY ; ++regindx )
   {
      this->acSetAixBg ( aeSeq(regindx) ) ;     // set the foreground color
      wp = this->acWrite ( wp, AIX_Colors[cnindx++], false ) ;
      this->acSetFg ( aesBG_DFLT ) ;            // return to default foreground
      this->acFlushOut () ;                     // flush the output stream
   }

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   this->acWrite ( baseROW, descCOL, Desc_a ) ;
   this->acWrite ( (baseROW + 12), descCOL, Desc_b ) ;

   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;

}  //* End Test_Bg() *

//*************************
//*    Test_8bitColor     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test fg/bg color selection via the 256-entry color lookup table.             *
//*                                                                              *
//* The target entry is accessed by inserting the index into the ANSI sequence   *
//* templates: ansiSeq[aesFG_INDEX] for foreground and                           *
//*            ansiSeq[aesBG_INDEX] for background.                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_8bitColor ( void )
{
   const wchar_t *Desc = 
         L"Eight-bit Color Attributes Test:\n"
         L"------------------------------------------------------\n"
         L"Console color attributes are defined by a standardized\n"
         L"256-element array of attribute registers.\n"
         L"-- The first sixteen(16) registers are reserved for\n"
         L"   the 3-bit and 4-bit color attributes.\n"
         L"-- The next 216 registers define the 8-bit color\n"
         L"   attributes available to console applications.\n"
         L"   These attributes are displayed here for both\n"
         L"   foreground and background.\n"
         L"-- The remaining 24 registers define the available\n"
         L"   greyscale attributes. (see the greyscale test)\n" ;
   const wchar_t *wTemplate = L" %3hhu " ; // formatting template

   const short baseROW = 4 ;
   const short colOFF  = 93 ;

   gString gsOut ;                        // text formattring
   short cnt = ZERO,                      // loop counter
         exitRow = baseROW ;              // cursor row for preparing to exit

   //** Test aesFG_INDEX **
   for ( uint8_t ebIndx = min8BIT ; ebIndx <= max8BIT ; ++ebIndx )
   {
      this->acSet8bitFg ( ebIndx ) ;
      gsOut.compose( wTemplate, &ebIndx ) ;
      this->ttyWrite ( gsOut, false ) ;
      if ( (++cnt % 18) == ZERO )
      {
         this->ttyWrite ( L"\n" ) ;
         ++exitRow ;
      }
   }

   //* Reset to defaut foreground and reset counter *
   this->acSetFg ( aesFG_DFLT ) ;
   this->ttyWrite ( L"\n\n" ) ;
   exitRow += 2 ;
   cnt = ZERO ;

   //** Test aesBG_INDEX **
   for ( short ebIndx = min8BIT ; ebIndx <= max8BIT ; ++ebIndx )
   {
      this->acSet8bitBg ( ebIndx ) ;
      gsOut.compose( wTemplate, &ebIndx ) ;
      this->ttyWrite ( gsOut, false ) ;
      if ( (++cnt % 18) == ZERO )
      {
         this->ttyWrite ( L"\n" ) ;
         ++exitRow ;
      }
   }

   //* Set fg/bg defaults  and flush the output buffer *
   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   //* Write a description of the test *
   this->acWrite ( baseROW, colOFF, Desc ) ;
   this->acSetCursor ( aesCUR_ABSPOS, exitRow, 1 ) ;
   this->acFlushOut () ;            // flush the output stream

}  //* End Test8BitCColor() *

//*************************
//*    Test_Greyscale     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test the terminal's greyscale color attribute support.                       *
//*                                                                              *
//* The target entry is accessed by inserting the index into the ANSI sequence   *
//* templates: ansiSeq[aesFG_INDEX] for foreground and                           *
//*            ansiSeq[aesBG_INDEX] for background.                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_Greyscale ( void )
{
   const wchar_t *Desc = 
         L"Greyscale Gradient Test:\n"
         L"-----------------------------------------------------------\n"
         L"The last 24 elements of the system color attribute array\n"
         L"(indices 232-255) define the range of greyscale attributes.\n"
         L"These values may be applied to either the foreground text\n"
         L"or to the background.\n"
         L"\n"
         L"Examples:\n"
         L"  acSetGreyscaleFg ( shade )\n"
         L"  acSetGreyscaleBg ( shade )\n"
         L"where: minGSCALE <= shade <= maxGSCALE\n" ;

   const short baseROW = 4 ;
   const short colOFF  = 58 ;
   const short exitROW = 28 ;

   gString gsOut ;                  // text formattring

   //** Test aesFG_INDEX **
   //** Test aesBG_INDEX **
   uint8_t gscIndx = minGSCALE ;
   do
   {
      this->acSetBg ( aesBG_BLUE ) ;            // set contrasting blue background
      this->acSetGreyscaleFg ( gscIndx ) ;      // set greyscale foreground
      gsOut = L"  Greyscale Foreground  " ;     // write some text
      this->ttyWrite ( gsOut, false ) ;

      //* Set blue foreground and default background *
      this->acSetFgBg ( aesFG_BLUE, aesBG_DFLT ) ;

      gsOut.compose( L"  %3hhu  ", &gscIndx ) ; // display the greyscale index
      this->ttyWrite ( gsOut, false ) ;

      this->acSetGreyscaleBg ( gscIndx ) ;      // set greyscale background
      gsOut = L"  Greyscale Background  " ;     // write some text
      this->ttyWrite ( gsOut, false ) ;

      this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ; // set fg/bg defaults
      gsOut = L"\n" ;                           // write the newline
      this->ttyWrite ( L"\n", false ) ;
   }
   while ( gscIndx++ < maxGSCALE ) ;
   // Programmer's Note: The weird loop test is due to the fact that 'gscIndx' 
   // is an 8-bit unsigned integer and could wrap around to zero if we aren't careful.

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ; // set fg/bg defaults
   this->acWrite ( baseROW, colOFF, Desc ) ;    // write the test description

   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;
   this->acFlushOut () ;                        // flush the output stream

}  //* End TestGreyscale() *

//*************************
//*       Test_RGB        *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test color selection via the Red/Green/Blue (RGB) registers.                 *
//*                                                                              *
//* Range for R, G, and B indices:                                               *
//* What the bleep does this mean?                                               *
//* 16-231:  6 × 6 × 6 cube (216 colors): 16 + 36 × r + 6 × g + b                *
//*          (0 ≤ r, g, b ≤ 5)                                                   *
//*   With a base of 16: 16 <= green <= max.green < min.blue                     *
//*                      max.green < blue <= max.blue < min.red                  *
//*                      max.blue < red <= max.red==231                          *
//* 216 "RGB" registers in a "6x6x6 cube" means 72 registers for each base       *
//* color. If that is true then each 72-register block should be dedicated to    *
//* a color                                                                      *
//*                                                                              *
//* - There appears to be little or no red in 16 through 51.                     *
//* - Red becomes apparent in the purples at index 52-57.                        *
//* - Red tends to dominate in the first 12 indices of each sequential           *
//*   36-register group.                                                         *
//* - The greens seem to be in the central 72 block                              *
//* - This would mean that the 3rd 72-register block s/b dominated by red.       *
//* - With increasing index, the blue transitions to green.                      *
//* - With increasing index, the green transitions to red.                       *
//* - Going above index 231 introduces grey/white into the mix. Most websites    *
//*   that talk about the "6x6x6 cube" explicitly exclue the grey group.         *
//* - What kind of range checking does the system do on the /r/g/b parameters?   *
//*   What substitutions (if any)) are made for out-of-range values?             *
//*   - For "pure" color spectrum, the other two colors may be zero(0).          *
//*     This actually makes no sense because zero is outside the RGB register    *
//*     range of 16-231. While 16 is the logical minimum                         *
//*                                                                              *
//*                                                                              *
//* - This is what we did for the "web-save color chart": hold one value         *
//*   static while exercising the other two values:                              *
//*   - Try holding the green and blue at 16 and cycling the red element         *
//*     through the entire 216-register range.                                   *
//*     Results:                                                                 *
//*   - Try holding the red and blue at 16 and cycling the green element         *
//*     through the entire 216-register range.                                   *
//*     Results:                                                                 *
//*   - Try holding the red and green at 16 and cycling the blue element         *
//*     through the entire 216-register range.                                   *
//*     Results:                                                                 *
//* - Hold two of the three colors static and cycle the 3rd through 216 colors   *
//*   - 16 <= red <= 231   g==16  blue==16                                       *
//*                                                                              *
//*   - r==16  16 <= green <= 231   b==16,                                       *
//*     Approximately the first 20 are saturated black, with green nocicable     *
//*     starting at approximately 36.                                            *
//*     Groups are relatively indistinguishable, so it makes sense to skip       *
//*     x of every y. What are x and y? Groups of 6 are most logical/likely.     *
//*     This would leave 36 shades of green, with the first 3 or 4 tending       *
//*     to be saturated to black.                                                *
//*     - Try setting the static colors to 231 instead of 16 to see what         *
//*       happens.                                                               *
//*     - Try setting the static colors to 255 instead of 16 to see what         * 
//*       happens.                                                               *
//*                                                                              *
//*   - r==16     g==16    16 <= blue <= 231                                     *
//*     Approximately the first 36 are saturated black, with blue nocicable      *
//*     starting at approximately 52.                                            *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  : rgbTest : indicates sub-test:                                       *
//*                    'c' == Step through the 6x6x6 matric of color             *
//*                           combinations. (default)                            *
//*                    'r' == Step Red, Green and Blue static at min/med/max     *
//*                    'g' == Step Green, Red and Blue static at min/med/max     *
//*                    'b' == Step Blue, Red and Green static at min/med/max     *
//*                    'd' == output diagonal (all three colors step forward)    *
//*                                                                              *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_RGB ( wchar_t rgbTest )
{
   const wchar_t *rgbOptions = 
           L"     RGB Test Options:\n"
           L"---------------------------------------------------\n"
           L"'r'  Red  : Display full range of red colors.\n"
           L"'g'  Green: Display full range of green colors.\n"
           L"'b'  Blue : Display full range of blue colors.\n"
           L"'y'  greY : Display full range of greyscale colors.\n"
           L"'w'  Web  : Display \"web-safe\" colors.\n" ;

   const short baseROW = 1 ;
   const short colOFF  = 32 ;

   gString gsOut ;                  // text formattring
   bool    menu = false ;           // if 'true', display sub-menu

   //* Warn about minimum terminal window dimensions.*
   this->acWrite ( baseROW, colOFF, 
                  "  (requires terminal dimensions of 39 rows X 155 columns)\n"
                  "---------------------------------------------------------" ) ; 

   //* Set white foreground so text in the tests will be visible.*
   this->acSetFg ( aesFGb_GREY ) ;

   if ( rgbTest == 'r' )
      this->trgbStepRed () ;
   else if ( rgbTest == 'g' )
      this->trgbStepGreen () ;
   else if ( rgbTest == 'b' )
      this->trgbStepBlue () ;
   else if ( rgbTest == 'y' )
      this->trgbStepGrey () ;
   else if ( rgbTest == 'w' )
   {
      this->acEraseArea ( aesERASE_LINE ) ;
      this->trgbStepWeb () ;
   }
   else
      menu = true ;

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ; // set fg/bg defaults
   if ( menu )                            // if invalid sub-option specified
   {
      this->acSetCursor ( aesCUR_NEXTROW ) ;
      this->ttyWrite ( rgbOptions ) ;
   }
   this->acSetCursor ( aesCUR_ABSPOS, this->termRows - 1, 1 ) ;
   this->acFlushOut () ;                  // flush the output stream

}  //* End Test_RGB() *

//*************************
//*      trgbStepRed      *
//*************************
void AnsiCmd::trgbStepRed ( void )
{
   const wchar_t *Desc = 
         L"Full-spectrum,\n"
         L"8-bit Red Test:\n"
         L"-------------------------\n"
         L"There are 216 values in\n"
         L"the console lookup table\n"
         L"which are reserved for\n"
         L"8-bit color attributes.\n"
         L"This test displays the\n"
         L"full range of Red\n"
         L"color attributes using\n"
         L"minimum values of both\n"
         L"minRGB(16) and ZERO(0).\n" ;

   const short baseROW = 3 ;        // base row for data display
   const short baseCOL = 2 ;        // base column for data display
   const short colOFF  = 130 ;      // column for description text
   const short exitROW = this->termRows - 1 ; // row for exit

   gString gsOut ;
   WinPos wp( baseROW, baseCOL ) ;
   uint8_t r = minRGB,
           g = minRGB,
           b = minRGB ;
   this->acSetCursor ( wp ) ;       // set initial cursor position

   for ( short tst = ZERO ; tst < 2 ; ++tst )
   {
      for ( short i = minRGB ; i <= maxRGB ; ++i )
      {
         #if 1    // Production: Write as background attributes.
         this->acSetRgbBg ( r, g, b ) ;
         #else    // Debugging only: Write as foreground attributes.
         this->acSetRgbFg ( r, g, b ) ;
         #endif   // Bg vs. Fg
         gsOut.compose( " %hhu;%hhu;%hhu ", &r, &g, &b ) ;
         if ( r >= 88 )
            gsOut.padCols( g == minRGB ? 11 : 9 ) ;
         gsOut.append( L'\n' ) ;          // cursor ends at column 1 of next row

         // Programmer's Note: We use the ttyWrite() + acSetCursor() sequence 
         // for speed because it avoids the multiple set-cursor and get-cursor 
         // operations used within the acWrite() method group.
         this->ttyWrite ( gsOut ) ;
         this->acSetCursor ( aesCUR_ABSCOL, wp.col ) ; // move cursor to target column

         ++r ;
         if ( ++wp.row > 38 )             // move to top of next column
         {
            wp = { baseROW, short(wp.col + gsOut.gscols() + 1) } ;
            this->acSetCursor ( wp ) ;
         }
      }

      r = minRGB ;
      //g = b = minRGB ;     // experimental - static colors set to base RGB
      //g = b = maxRGB ;     // experimental - static colors set to maximum (brightest)
      g = b = ZERO ;          // static colors set to minimum
   }

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   this->acWrite ( baseROW, colOFF, Desc ) ;
   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;

}  //* End trgbStepRed() *

//*************************
//*     trgbStepGreen     *
//*************************
void AnsiCmd::trgbStepGreen ( void )
{
   const wchar_t *Desc = 
         L"Full-spectrum,\n"
         L"8-bit Green Test:\n"
         L"-------------------------\n"
         L"There are 216 values in\n"
         L"the console lookup table\n"
         L"which are reserved for\n"
         L"8-bit color attributes.\n"
         L"This test displays the\n"
         L"full range of Green\n"
         L"color attributes using\n"
         L"minimum values of both\n"
         L"minRGB(16) and ZERO(0).\n" ;

   const short baseROW = 3 ;        // base row for data display
   const short baseCOL = 2 ;        // base column for data display
   const short colOFF  = 130 ;      // column for description text
   const short exitROW = this->termRows - 1 ; // row for exit

   gString gsOut ;
   WinPos wp( baseROW, baseCOL ) ;
   uint8_t r = minRGB,
           g = minRGB,
           b = minRGB ;
   this->acSetCursor ( wp ) ;       // set initial cursor position

   for ( short tst = ZERO ; tst < 2 ; ++tst )
   {
      for ( short i = minRGB ; i <= maxRGB ; ++i )
      {
         #if 1    // Production: Write as background attributes.
         this->acSetRgbBg ( r, g, b ) ;
         #else    // Debugging only: Write as foreground attributes.
         this->acSetRgbFg ( r, g, b ) ;
         #endif   // Bg vs. Fg
         gsOut.compose( " %hhu;%hhu;%hhu ", &r, &g, &b ) ;
         if ( g >= 88 )
            gsOut.padCols( r == minRGB ? 11 : 9 ) ;
         gsOut.append( L'\n' ) ;          // cursor ends at column 1 of next row

         // Programmer's Note: We use the ttyWrite() + acSetCursor() sequence 
         // for speed because it avoids the multiple set-cursor and get-cursor 
         // operations used within the acWrite() method group.
         this->ttyWrite ( gsOut ) ;
         this->acSetCursor ( aesCUR_ABSCOL, wp.col ) ; // move cursor to target column

         ++g ;
         if ( ++wp.row > 38 )             // move to top of next column
         {
            wp = { baseROW, short(wp.col + gsOut.gscols() + 1) } ;
            this->acSetCursor ( wp ) ;
         }
      }

      g = minRGB ;
      //r = b = minRGB ;     // experimental - static colorse set to base RGB
      //r = b = maxRGB ;     // experimental - static colors set to maximum (brightest)
      r = b = ZERO ;          // static colors set to minimum
   }

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   this->acWrite ( baseROW, colOFF, Desc ) ;
   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;

}  //* End trgbStepGreen() *

//*************************
//*     trgbStepBlue      *
//*************************
void AnsiCmd::trgbStepBlue ( void )
{
   const wchar_t *Desc = 
         L"Full-spectrum,\n"
         L"8-bit Blue Test:\n"
         L"-------------------------\n"
         L"There are 216 values in\n"
         L"the console lookup table\n"
         L"which are reserved for\n"
         L"8-bit color attributes.\n"
         L"This test displays the\n"
         L"full range of Blue\n"
         L"color attributes using\n"
         L"minimum values of both\n"
         L"minRGB(16) and ZERO(0).\n" ;

   const short baseROW = 3 ;        // base row for data display
   const short baseCOL = 2 ;        // base column for data display
   const short colOFF  = 130 ;      // column for description text
   const short exitROW = this->termRows - 1 ; // row for exit

   gString gsOut ;
   WinPos wp( baseROW, baseCOL ) ;
   uint8_t r = minRGB,
           g = minRGB,
           b = minRGB ;
   this->acSetCursor ( wp ) ;       // set initial cursor position

   for ( short tst = ZERO ; tst < 2 ; ++tst )
   {
      for ( short i = minRGB ; i <= maxRGB ; ++i )
      {
         #if 1    // Production: Write as background attributes.
         this->acSetRgbBg ( r, g, b ) ;
         #else    // Debugging : Write as foreground attributes.
         this->acSetRgbFg ( r, g, b ) ;
         #endif   // Bg vs. Fg
         gsOut.compose( " %hhu;%hhu;%hhu ", &r, &g, &b ) ;
         if ( b >= 88 )
            gsOut.padCols( r == minRGB ? 11 : 9 ) ;
         gsOut.append( L'\n' ) ;          // cursor ends at column 1 of next row

         // Programmer's Note: We use the ttyWrite() + acSetCursor() sequence 
         // for speed because it avoids the multiple set-cursor and get-cursor 
         // operations used within the acWrite() method group.
         this->ttyWrite ( gsOut ) ;
         this->acSetCursor ( aesCUR_ABSCOL, wp.col ) ; // move cursor to target column

         ++b ;
         if ( ++wp.row > 38 )             // move to top of next column
         {
            wp = { baseROW, short(wp.col + gsOut.gscols() + 1) } ;
            this->acSetCursor ( wp ) ;
         }
      }

      b = minRGB ;
      //r = g = minRGB ;     // experimental - static colors set to base RGB
      //r = g = maxRGB ;     // experimental - static colors set to maximum (brightest)
      r = g = ZERO ;       // static colors set to minimum
   }

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   this->acWrite ( baseROW, colOFF, Desc ) ;
   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;

}  //* End trgbStepBlue() *

//*************************
//*     trgbStepGrey     *
//*************************
void AnsiCmd::trgbStepGrey ( void )
{
   #define REPORT_2_MAX (0)   // continue output to top of 256-element array

   const wchar_t *Desc = 
         L"R/G/B Greyscale Test:\n"
         L"--------------------------------------------------\n"
         L"Each of the primary colors, Red/Green/Blue may be \n"
         L"displayed in 216 shades (see other RGB tests).\n"
         L"If the Red, Green and Blue indices step through\n"
         L"the range of elements in unison, a map of 216\n"
         L"greyscale shades is generated.\n"
         L"Note: This is separate from the 24 table elements\n"
         L"reserved specifically for greyscale attributes.\n" ;

   const short baseROW = 3 ;        // base row for data display
   const short baseCOL = 2 ;        // base column for data display
   const short colOFF  = 93 ;       // column for description text
   const short exitROW = this->termRows - 1 ; // row for exit

   gString gsOut ;
   WinPos wp( baseROW, baseCOL ) ;
//   short row = baseROW,
//         col = baseCOL ;
   uint8_t r = minRGB,
           g = minRGB,
           b = minRGB ;
   this->acSetCursor ( wp ) ;       // set initial cursor position

   #if REPORT_2_MAX != 0   // Terminate at top of register array
   for ( short i = minRGB ; i <= maxLOOKUP ; ++i )
   #else    // Terminate at top of RGB range
   for ( short i = minRGB ; i <= maxRGB ; ++i )
   #endif   // REPORT_2_MAX
   {
      #if 1    // Production: Write as background attributes.
      this->acSetRgbBg ( r, g, b ) ;
      #else    // Debugging only: Write as foreground attributes.
      this->acSetRgbFg ( r, g, b ) ;
      #endif   // Bg vs. Fg
      gsOut.compose( " %hhu;%hhu;%hhu", &r, &g, &b ) ;
      gsOut.padCols( 13 ) ;
      if ( r > maxRGB )
         gsOut.insert( L'|' ) ;
      gsOut.append( L'\n' ) ;          // cursor ends at column 1 of next row

      // Programmer's Note: We use the ttyWrite() + acSetCursor() sequence 
      // for speed because it avoids the multiple set-cursor and get-cursor 
      // operations used within the acWrite() method group.
      this->ttyWrite ( gsOut ) ;
      this->acSetCursor ( aesCUR_ABSCOL, wp.col ) ; // move cursor to target column

      ++r ;
      ++g ;
      ++b ;
      if ( ++wp.row > 38 )
      {
         wp = { baseROW, short(wp.col + gsOut.gscols() + 1) } ;
         this->acSetCursor ( wp ) ;
         if ( r == 160 )
            this->acSetFg ( aesFG_BLACK ) ;
         #if REPORT_2_MAX != 0
         else if ( r == (maxRGB + 1) )
         { this->acWrite ( (baseROW + 11), col, L"Greyscale Registers\n" ) ; }
         #endif   // REPORT_2_MAX
      }
   }

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   //* Write a description of the test *
   this->acWrite ( baseROW, colOFF, Desc ) ;
   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;

}  //* End trgbStepGrey() *

//*************************
//*      trgbStepWeb      *
//*************************
//********************************************************************************
//* Display the RGB color block (minRGB through maxRGB).                         *
//*                                                                              *
//* Display the table of R/G/B color combinations referencing the "web-safe"     *
//* color attributes (one of every six values). This corresponds to the          *
//* "web-safe" HTML/CSS color attributes.                                        *
//* The HTML/CSS values use eight(8) bits for each color to create a 24-bit      *
//* color value, but only one sixth (1/6) of them are considered as "web safe".  *
//* The theory is that all systems and web browsers can accurately render        *
//* these colors. Shades in between may need to be "dithered".                   *
//*                                                                              *
//*                ANSI                                                          *
//*  HTML/CSS   HEX    DEC                                                       *
//*  --------  ------  ---                                                       *
//*   000000   0x0010   16                                                       *
//*   000033   0x0020   22                                                       *
//*   000066   0x0030   28                                                       *
//*   000099   0x0040   34                                                       *
//*   0000CC   0x0050   40                                                       *
//*   0000FF   0x0060   46                                                       *
//*    ...      ...                                                              *
//*   FFFFFF   0x00E7  231                                                       *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::trgbStepWeb ( void )
{
   //* For debugging only. Set to non-zero to  *
   //* render foreground instead of background.*
   #define DEBUG_RGB_WEB (0)

   const wchar_t *Desc = 
         L"R/G/B Web-Safe-Colors Test:\n"
         L"Although each of the primary colors, Red/Green/Blue\n"
         L"may be displayed in 216 shades (see other RGB tests),\n"
         L"many shades are indistinguishable to the eye or are not\n"
         L"rendered uniquely by the software/hardware.\n"
         L"For this reason, the so-called \"web-safe\" color pallete\n"
         L"has been defined to display every sixth shade in a uniform\n"
         L"way across a wide variety of software systems and video\n"
         L"hardware.\n"
         L"\n"
         L"The RGB register indices displayed here represent\n"
         L"respectively the Red, Green and Blue indices for each\n"
         L"of these \"web-safe\" colors.\n" ;
   const short baseROW   = 2 ;      // base row for data display
   const short baseCOL   = 2 ;      // base column for data display
   const short totHUES   = 7 ;      // total number of hues to display

   gString gsOut ;                  // text formatting
   WinPos  wp( baseROW, baseCOL ) ; // text position
   short colWidth = 11 ;            // width of display column
   WebSafeRGB hue ;                 // target color group
   uint8_t shade,                   // color intensity
           webIndex = wsrgbRED ;    // RGB color index

   //* For each color column to be displayed *
   for ( short ihues = ZERO ; ihues < totHUES ; ++ihues )
   {
      //* Initialize values for this iteration *
      wp.row = baseROW ;      // top row of display
      shade = ZERO ;          // minimum intensity
      if ( ihues > ZERO )     // advance start column for display
         wp.col += gsOut.gscols() + 1 ;
      this->acSetCursor ( wp ) ;

      switch ( ihues )
      {
         case 0: hue = wsrgbRED ;      break ;  // Red
         case 1: hue = wsrgbGREEN ;    break ;  // Green
         case 2: hue = wsrgbBLUE ;     break ;  // Blue
         case 3: hue = wsrgbBROWN ; ++colWidth ; break ; // Brown
         case 4: hue = wsrgbMAGENTA ;  break ;  // Magenta
         case 5: hue = wsrgbCYAN ;     break ;  // Cyan
         case 6: hue = wsrgbGREY ;  ++colWidth ; break ;  // Grey
      } ;

      //* Special Case: Black attribute *
      //* Production: Set background attributes.*
      #if DEBUG_ANSICMD == 0 || DEBUG_RGB_WEB == 0
      this->acSetWebBg ( hue, shade ) ;   // set background attribute
      #else    // Set foreground attributes (for debugging only)
      this->acSetWebFg ( hue, shade ) ;
      #endif   // DEBUG_RGB_WEB
      formatRgbString ( gsOut, colWidth, wsrgbBLACK ) ;
      this->ttyWrite ( gsOut ) ;
      this->acSetCursor ( aesCUR_ABSCOL, wp.col ) ; // move cursor to target column

      //* For each shade of the current hue *
      for ( ; shade <= wsrgbSHADE_MAX ; ++shade )
      {
         //* Production: Set background attributes.*
         #if DEBUG_ANSICMD == 0 || DEBUG_RGB_WEB == 0
         this->acSetWebBg ( hue, shade ) ;   // set background attribute
         #else    // Set foreground attributes (for debugging only)
         this->acSetWebFg ( hue, shade ) ;   // set foreground attribute
         #endif   // DEBUG_RGB_WEB
         //* Format the display text *
         formatRgbString ( gsOut, colWidth, webIndex++ ) ;

         // Programmer's Note: We use the ttyWrite() + acSetCursor() sequence 
         // for speed because it avoids the multiple set-cursor and get-cursor 
         // operations used within the acWrite() method group.
         this->ttyWrite ( gsOut ) ;
         this->acSetCursor ( aesCUR_ABSCOL, wp.col ) ; // move cursor to target column
      }
   }

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   wp = { baseROW, short(wp.col + gsOut.gscols() + 3) } ;
   this->acWrite ( wp, Desc ) ;

   #undef DEBUG_RGB_WEB
}  //* End trgbStepWeb() *

//*************************
//*     Test_AltFonts      *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test the terminal's support for alternate fonts.                             *
//*                                                                              *
//* Most terminal programs support few if any alternate fonts; however,          *
//* see acSetMod(aesITALIC) to set an italic font.                               *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_AltFonts ( aeSeq fg )
{
   const wchar_t *Desc = 
         L"Alternate Fonts Test:\n"
          "-------------------------------------------------------\n"
          "The ANSI standard (ANSI X3.64 (ISO/IEC6429)) provides\n"
          "commands to select among ten(10) fonts for use by the\n"
          "terminal. These are the \"Primary\" font and nine\n"
          "\"Alternate\" fonts.\n"
          "For Gnometerm and Konsole the Primary font is specified\n"
          "in the \"Profile\". For Xterm, the Primary font is\n"
          "defined in the \".Xdefaults\" or \".Xresources\" file,\n"
          "but note that under Wayland the file will not be read.\n"
          "\n"
          "When dinosaurs and UNIX geeks roamed the earth and\n"
          "physical terminals were still common, on-the-fly font\n"
          "selection was useful. However, terminal emulation\n"
          "software may not support selection of alternate fonts.\n" ;
   const short baseROW = 4 ;
   const short baseCOL = 1 ;
   const short colOFF  = 48 ;

   gString gsOut ;                  // text formatting

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* Write a description of the test *
   this->acWrite ( baseROW, colOFF, Desc ) ;
   this->acSetCursor ( aesCUR_ABSPOS, baseROW, baseCOL ) ;

   //** Test aesPRIMARY_FONT **
   this->acSetFont ( aesPRIMARY_FONT ) ;     // Set primary font
   gsOut = L" Primary Terminal Font. (#00) \n\n" ;
   this->ttyWrite ( gsOut ) ;

   //** Test aesALTERNATE_FONT **
   for ( uint8_t indx = 1 ; indx <= 9 ; ++indx )
   {
      this->acSetFont ( aesALTERNATE_FONT, indx ) ;
      gsOut.compose( " Alternate Font #%hhu\n\n", &indx ) ;
      this->ttyWrite ( gsOut ) ;
   }

   this->acSetFont ( aesPRIMARY_FONT ) ;     // Reset to primary font
   this->acSetFg ( aesFG_DFLT ) ;            // Reset to terminal default foreground color
   this->acFlushOut () ;                     // flush the output stream

}  //* Test_AltFonts() *

//*************************
//*    Test_CursorPos     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test the ANSI escape sequences used for cursor-positioning.                  *
//*                                                                              *
//* We also call acGetTermDimensions() method and report the number of rows      *
//* and columns in the terminal window.                                          *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* -- See note in module header regarding 1-based vs. 0-based cursor            *
//*    positioning.                                                              *
//* -- Using the 'ZERO' argument with aesCUR_ABSCOL works correctly even         *
//*    though the position of the left edge may be interpreted as one-based      *
//*    by some or all of the cursor commands.                                    *
//* -- Using the 'ZERO' arguments with aesCUR_ABSPOS is accepted although the    *
//*    the base row/column position is interpreted as one-based by some or all   *
//*    of the cursor commands AND the 0th row and 1st row are interpreted as     *
//*    THE SAME ROW.                                                             *
//* -- Save and Restore commands come in two flavors, DEC and SCO.               *
//*    For Gnometerm (v:3.40.3), Konsole (v:21.08.3) and Xterm (v:366),          *
//*    the DEC commands appear to be non-functional, while the SCO commands      *
//*    work as expected.                                                         *
//*                                                                              *
//*                                                                              *
//*                                                                              *
//********************************************************************************

void AnsiCmd::Test_CursorPos ( aeSeq fg )
{
   const uint16_t tsec = 7 ;        // pause time 0.7 second
   const short msgRow = 4,          // Test-message position
               msgCol = 1,
               cpROW  = 1,          // report cursor position here
               cpCOL  = this->termCols - 20 ;

   gString gsOut,                   // text formatting
           gsErase,                 // for erasing text from message line
           gsCur ;                  // special character for display of cursor position
   WinPos  wpMsg( msgRow, msgCol ),  // message position
           wp ;                     // receives/sets cursor position
   short targetRow,                 // target row
         targetCol ;                // target column

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* Construct the special character sequence for     *
   //* character written to the current cursor position.*
   gsCur.compose( "%S$\b%S", 
                  ansiSeq[fg == aesFG_GREEN ? aesFG_RED : aesFG_GREEN],
                  ansiSeq[fg] ) ;

   gsErase.padCols( this->termCols ) ; // construct erasure string

   //* Set terminal configuration for unbuffered input (no echo) *
   this->acBufferedInput ( false, noEcho ) ;

   //* Report the current terminal dimensions *
   WinSize ws = this->acGetTermSize () ;
   gsOut.compose( "term rows: %hu\n"
                  "term cols: %hu\n",
          &ws.ws_row, &ws.ws_col, &ws.ws_xpixel, &ws.ws_ypixel ) ;
   this->acWrite ( (cpROW + 1), cpCOL, gsOut ) ;

   //* Display instructions *
   gsOut = L"Cursor should be at \"home\" position (upper left corner). "
           L"Press any key to continue..." ;
   this->acWrite ( msgRow, msgCol, gsOut ) ;

   //** Test aesCUR_HOME
   //** Test aesCUR_REPORT - (see Test_ReportCurPos())
   this->acSetCursor ( aesCUR_HOME ) ;             // set cursor at home (0,0 or 1,1)
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->acRead () ;                               // get user response
   gsOut.compose( L"%ST%S", ansiSeq[aesFG_DFLT], ansiSeq[fg] ) ;
   this->ttyWrite ( gsOut ) ;                      // restore the title

   //** Test aesCUR_ABSPOS
   targetRow = 11 ; targetCol = 40 ;
   this->acWrite ( wpMsg, gsErase ) ;              // clear the message area
   gsOut.compose( L"Set cursor to an absolute position %hd,%hd. "
                  L"Press any key to continue...",
                  &targetRow, &targetCol ) ;
   this->acWrite ( wpMsg, gsOut ) ;
   //* Set cursor to target position *
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, targetCol ) ; // set absolute row,col
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->acRead () ;                               // get user input
   //* Erase the message line and write a new message *
   this->acWrite ( wpMsg, gsErase ) ;              // clear the message area
   gsOut = L"Moving relative to base position, right, left, down and up." ;
   this->acWrite ( wpMsg, gsOut ) ;
   //* Return to target position *
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, targetCol ) ; // set absolute row,col
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position

   //** Test aesCUR_COLSRT
   //** Test aesCUR_COLSLT
   //** Test aesCUR_ROWSDN
   //** Test aesCUR_ROWSUP
   this->acSetCursor ( aesCUR_COLSRT, 3 ) ;        // move right relative
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec ) ;                         // wait a moment
   this->acSetCursor ( aesCUR_ROWSDN, 2 ) ;        // move down relative
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec ) ;                         // wait a moment
   this->acSetCursor ( aesCUR_COLSLT, 3 ) ;        // move left relative
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec ) ;                         // wait a moment
   this->acSetCursor ( aesCUR_COLSLT, 3 ) ;        // move left relative
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec ) ;                         // wait a moment
   this->acSetCursor ( aesCUR_ROWSUP, 2 ) ;        // move up relative
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec ) ;                         // wait a moment
   this->acSetCursor ( aesCUR_ROWSUP, 2 ) ;        // move up relative
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec ) ;                         // wait a moment
   this->acSetCursor ( aesCUR_COLSRT, 3 ) ;        // move right relative
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec ) ;                         // wait a moment
   this->acSetCursor ( aesCUR_COLSRT, 3 ) ;        // move right relative
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec ) ;                         // wait a moment

   //* Prompt for user response *
   this->acSetCursor ( aesCUR_ABSPOS, msgRow, msgCol ) ; // set absolute row,col
   gsOut.append( L" DONE - Press any key to continue..." ) ;
   this->ttyWrite ( gsOut ) ;
   this->acRead () ;                               // get user input

   //** Test aesCUR_ROWSDN
   //** Test aesCUR_ABSCOL
   //** Test aesCUR_NEXTROW
   //** Test aesCUR_PREVROW
   //** Test aesCUR_ROWSUP
   //* Erase the message line and write a new message *
   this->acWrite ( wpMsg, gsErase ) ;
   gsOut = L"Moving relative to base position, down one row, then up one row." ;
   this->acWrite ( wpMsg, gsOut ) ;

   //* Return to target position, then to message position, *
   //* write the message and move back to target position,  *
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, targetCol ) ; // return to target position
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec * 2 ) ;                     // wait a moment
   this->acSetCursor ( aesCUR_ROWSDN, 1 ) ;        // move one row downward
   this->acSetCursor ( aesCUR_ABSCOL, 1 ) ;        // left edge of current row
   gsOut = L"  Move to left edge of row below." ;
   this->ttyWrite ( gsOut ) ;
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, targetCol ) ; // return to target position

   //* Go to left edge of next row and write the cursor marker *
   this->acSetCursor ( aesCUR_NEXTROW ) ;          // left edge of next row
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec ) ;                         // wait a moment

   //* Return to target position, then to message position, *
   //* write the message and move back to target position,  *
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, targetCol ) ; // set absolute row,col
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec * 2 ) ;                     // wait a moment
   this->acSetCursor ( aesCUR_ROWSUP, 1 ) ;        // move to next row up
   this->acSetCursor ( aesCUR_ABSCOL, 1 ) ;        // left edge of current row
   gsOut = L"  Move to left edge of row above." ;
   this->ttyWrite ( gsOut ) ;
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, targetCol ) ; // set absolute row,col

   //* Go to left edge of previous row and write the cursor marker *
   this->acSetCursor ( aesCUR_PREVROW ) ;          // left edge of previous row
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   this->nsleep ( tsec * 3 ) ;                     // wait a moment

   //** Test aesCUR_ROWDEL
   //** Test aesCUR_ABSCOL
   this->acWrite ( wpMsg, gsErase ) ;              // clear the message area
   gsOut = L"Press any key to delete this row. All lower rows will move upward." ;
   this->acWrite ( wpMsg, gsOut ) ;
   this->acSetCursor ( aesCUR_ABSCOL, 1 ) ;        // left edge of current row
   this->acRead () ;                               // get user input
   //* Delete the message row *
   this->acSetCursor ( aesCUR_ROWDEL ) ;           // remove prompt row
   --targetRow ;                                   // subtract the deleted row
   this->acWrite ( msgRow, msgCol, L"Deleted..." ) ;
   this->nsleep ( tsec * 2 ) ;                     // wait a moment

   //* Clear the test-data area *
   targetRow = 10 ; targetCol = 25 ;
   this->acClearArea ( (targetRow - 3), 1, 7, 50, L'.' ) ;

   //** Test aesCUR_REPORT - see acGetCursor() method **
   //* Return to message line, erase it and write new message *
   this->acWrite ( wpMsg, gsErase ) ;
   gsOut.compose( "Capture the current cursor position. Target row,col: %hd,%hd",
                  &targetRow, &targetCol ) ;
   this->acWrite ( wpMsg, gsOut ) ;
   //* Go to target position and write the cursor marker *
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, targetCol ) ;
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
   //* Get the position report and display position *
   wp = this->acGetCursor () ;
   gsOut.append( " - Press any key to continue...\n"
                 "                                   Reported row,col: %hd,%hd",
                 &wp.row, &wp.col ) ;
   this->acWrite ( wpMsg, gsOut ) ;
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, targetCol ) ;
   this->ttyWrite ( gsCur ) ;                      // write the cursor marker
   this->acRead () ;                               // get user input
   this->acWrite ( wpMsg, gsErase ) ;              // clear the message area
   this->acWrite ( (wpMsg.row + 1), wpMsg.col, gsErase ) ;

   //* Test the cursor position save/restore commands.                     *
   //* aesCUR_SAVE/aesCUR_RESTORE, then aesCUR_SAVE_SCO/aesCUR_RESTORE_SCO *
   bool decMsg = true ;
   for ( short loopy = 2 ; loopy > ZERO ; --loopy, decMsg = false )
   {
      //* Erase the message line, and write a new message *
      gsOut.compose( "Saving the current cursor position (%S): row,col: %hd,%hd",
                     (decMsg ? L"DEC" : L"SCO"), &targetRow, &targetCol ) ;
      this->acWrite ( wpMsg, gsErase ) ;
      this->acWrite ( wpMsg, gsOut ) ;

      //** Test aesCUR_SAVE_DEC
      //** Test aesCUR_RESTORE_DEC
      //** Test aesCUR_SAVE_SCO
      //** Test aesCUR_RESTORE_SCO
      //* Set cursor to target coordinates and request *
      //* that terminal save those coordinates.        *
      this->acSetCursor ( aesCUR_ABSPOS, targetRow, targetCol ) ;
      this->acSetCursor ( decMsg ? aesCUR_SAVE_DEC : aesCUR_SAVE_SCO ) ;
      this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
      this->nsleep ( tsec * 4 ) ;                     // wait a moment

      //* Update the message *
      gsOut.compose( "Move to another position, then restore to previous position (%S).",
                     (decMsg ? L"DEC" : L"SCO") ) ;
      this->acWrite ( wpMsg, gsErase ) ;              // clear the message area
      this->acWrite ( wpMsg, gsOut ) ;

      //* Move to another position, write the cursor marker,       *
      //* wait a moment, then restore the previous cursor position.*
      if ( decMsg )
         this->acSetCursor ( aesCUR_ABSPOS, (targetRow - 2), (targetCol - 14) ) ;
      else
         this->acSetCursor ( aesCUR_ABSPOS, (targetRow - 2), (targetCol + 14) ) ;
      this->ttyWrite ( gsCur ) ;                      // write the cursor marker
      this->Test_ReportCurPos ( cpROW, cpCOL ) ;      // report the cursor position
      this->nsleep ( tsec * 4 ) ;                     // wait a moment
      this->acSetCursor ( decMsg ? aesCUR_RESTORE_DEC : aesCUR_RESTORE_SCO ) ;
      this->nsleep ( tsec * 2 ) ;                     // wait a moment
      //* Test whether restore was successful *
      wp = this->acGetCursor () ;
      gsOut.append( (wp.row == targetRow && wp.col == targetCol ? L" Success!" : L" Failed.") ) ;
      this->acWrite ( wpMsg, gsOut ) ;
      this->nsleep ( tsec * 3 ) ;                     // wait a moment

      //* After first loop iteration, erase old cursor marker.*
      if ( decMsg )
         this->acWrite ( (targetRow - 2), (targetCol - 14), L".", false ) ;
   }
   this->nsleep ( tsec * 2 ) ;                        // wait a moment

   this->acSetCursor ( aesCUR_ABSPOS, msgRow, 1 ) ;
   this->acEraseArea ( aesERASE_LINE ) ;
   this->acWrite ( msgRow, msgCol, 
                  L"\n       Cursor positioning tests complete. "
                   "Press any key to exit..." ) ;
   this->acRead () ;                                  // get user input

   //* Prepare to exit *
   this->acSetCursor ( aesCUR_ROWSDN, 11 ) ;
   this->acSetCursor ( aesCUR_ABSCOL, 1 ) ;
   this->acSetMod ( aesBLINK_OFF ) ;   // Disable cursor blinking
   this->acSetFg ( aesFG_DFLT ) ;      // Reset to terminal default foreground color
   this->acRestoreTermInfo () ;        // Restore original terminal configuration
   this->acFlushOut () ;               // flush the output stream

}  //* End Test_CursorPos() *

//*************************
//* Test_UnbufferedInput  *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Enable unbuffered terminal input. Test user responses to various prompts,    *
//* then disable unbuffered input before returning to caller.                    *
//*                                                                              *
//* Test Unbuffered Input:                                                       *
//*  1) wcin.read( wchar_t*, short )  and  cin.read( char*, short )              *
//*  2) wcin.get( void )  and  cin.get( void )                                   *
//*  3) wcin.get( wchar_t& )  and  cin.get( char& )                              *
//*  4) wchar_t acRead ( void ) ;                 handles wide and narrow input  *
//*  5) short acRead ( gString& gsIn, short ) ;   handles wide and narrow input  *
//*  6) wcin >> wchar_t[]  and  cin >> char[]                                    *
//*  7) wchar_t getwchar()  and char getchar()                                   *
//*                                                                              *
//*                                                                              *
//* Input  : fg   : foreground color attribute (member of aeSeq)                 *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_UnbufferedInput ( aeSeq fg )
{
   const wchar_t *Desc = 
         L"Unbuffered Input Test:\n"
          "-----------------------------------------------------------\n"
          "The 'stdin' stream ('wcin' or 'cin' in the C++ environment)\n"
          "is buffered by default so that an application must select\n"
          "the standard library input methods with this in mind.\n"
          "By contrast, terminal applications often benefit by using\n"
          "unbuffered data from the input stream, that is, reading a\n"
          "single keycode or a specific number of keycodes from the\n"
          "stream, particularly when the application is sending and\n"
          "receiving ANSI escape sequences, as we are doing here.\n"
          "\n"
          "This test uses several different ways to access the\n"
          "input stream.\n"
          "- The first and last input prompts use the standard\n"
          "  buffered input stream and therefore require the Enter key\n"
          "  to terminate the input.\n"
          "- The other eight(8) prompts use various C++ and C library\n"
          "  calls to capture key input.\n"
          "The data captured by each test are displayed at the right\n"
          "edge of the screen.\n" ;

   const short colOFF  = 54 ; // offset for display of description
   const short exitROW = 34 ; // cursor row when exiting the test

   TermIos tios ;             // structured terminal I/O settings
   tios.c_iflag = ZERO ;      // initialize the target data structure
   tios.c_oflag = ZERO ;
   tios.c_cflag = ZERO ;
   tios.c_lflag = ZERO ;

   gString gsIn, gsOut ;      // text input capture and output formatting
   wchar_t winbuff[64] ;      // receives raw input stream
   wchar_t winchar ;          // receives raw input stream
   char    cinbuff[64] ;      // receives raw input stream
   char    cinchar ;          // receives raw input stream
   short gcnt ;               // input bytes (chars)

   //* Warn about minimum terminal window dimensions.*
   this->acWrite ( 1, 28, "  (requires terminal rows/cols of 35/110)\n"
                        "-----------------------------------------" ) ;

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* Create a landing zone *
   const short lzROWS = 10, lzCOLS = 40, lzCOLOFF = (lzCOLS + 37) ;
   short lzr = 4, lzc = 1 ;
   this->acClearArea ( lzr, lzc, lzROWS, lzCOLS, L'.' ) ;
   lzc += 4 ;

   //* Limit width of captured-data display *
   const short outMAXWIDTH = this->termCols - lzCOLOFF + 1 ;


   //* Display the prompt and the test criteria *
   this->acSetMod ( aesBOLD ) ;
   this->acSetMod ( aesUNDERLINE ) ;
   this->acWrite ( (lzr - 1), (lzCOLS + 2), "Type your response to each test prompt:" ) ;

   this->acResetMods () ;           // reset all text modifiers

   gsOut = L"buffered wcin.read(wchar_t*,1)   :\n"
            "unbuffered wcin.read(wchar_t*,1) :\n"
            "unbuffered wcin.get(void)        :\n"
            "unbuffered wcin.get(char_type*)  :\n"
            "unbuffered acRead(void)          :\n"
            "unbuffered acRead(gString&, 1)   :\n"
            "unbuffered acRead(gString&, 3)   :\n"
            "unbuffered wcin >> (cin >>)      :\n"
            "unbuffered getwchar() (getchar()):\n"
            "buffered, Press Enter to exit    :\n" ;
   this->acWrite ( lzr, (lzCOLS + 2), gsOut ) ;

   //* Get the existing terminal settings and display the data *
   gString dtiTitle( "Original Term Settings:" ) ;
   short dumpRow = (lzr + lzROWS + 1), dumpCol = 1 ;
   this->acGetTermInfo ( stdIn, tios ) ;     // get original term settings
   this->acDumpTermInfo ( tios, dumpRow, dumpCol, 2, &dtiTitle ) ;

   //* Display the test description/instructions *
   this->acWrite ( dumpRow, colOFF, Desc ) ;


   //*****************************************
   //* Input buffering enabled:              *
   //* Test wcin.read( char_type*, 1 )       *
   //*   or cin.read( char_type*, 1 )        *
   //*****************************************
   //* Results: Echoes chars. Enter required to terminate. Count=1  *
   //* After Enter detected, returns first char in stdIn buffer,    *
   //* leaving any additional characters in the stdin buffer.       *
   //* Afterward, buffer is cleared to avoid tainting the next test.*
   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   if ( this->wideStream )          // get user response (wide)
   { wcin.read( winbuff, 1 ) ; gcnt = wcin.gcount() ; }
   else                             // get user response (narrow)
   { cin.read( cinbuff, 1 ) ; gcnt = cin.gcount() ; }
   this->acFlushIn () ;             // discard abandoned data (incl. newline)
   tuiReport ( gsOut, (this->wideStream ? (void*)winbuff : 
               (void*)cinbuff), gcnt, this->wideStream ) ;
   this->acWrite ( lzr++, lzCOLOFF, gsOut ) ;   // display captured input data

   //* Set terminal configuration for unbuffered input *
   this->acBufferedInput ( false, noEcho ) ;

   //* Retrieve and display the modified settings *
   this->acGetTermInfo ( stdIn, tios ) ;
   dtiTitle = "Retrieved _new_ settings:" ;
   this->acDumpTermInfo ( tios, dumpRow, dumpCol, 2, &dtiTitle ) ;

   //*****************************************
   //* Input buffering disabled:             *
   //* Test wcin.read( char_type*, 1 )       *
   //*   or cin.read( char_type*, 1 )        *
   //*****************************************
   //* Results: Works correctly. count==1, returns keycode *
   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   if ( this->wideStream )                      // get user response (wide)
   { wcin.read( winbuff, 1 ) ; gcnt = wcin.gcount() ; }
   else                                         // get user response (narrow)
   { cin.read( cinbuff, 1 ) ;  gcnt = cin.gcount() ;  }
   tuiReport ( gsOut, (this->wideStream ? (void*)winbuff : (void*)cinbuff), gcnt, this->wideStream ) ;
   this->acWrite ( lzr++, lzCOLOFF, gsOut ) ;   // display captured input data

   //*****************************************
   //* Input buffering disabled:             *
   //* Test wcin.get( void )                 *
   //*   or cin.get( void )                  *
   //*****************************************
   //* Results: Works correctly. count==1, returns keycode *
   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   if ( this->wideStream )                      // get user response (wide)
   { winchar = wcin.get() ; gcnt = wcin.gcount() ; }
   else                                         // get user response (narrow)
   { winchar = cin.get() ;  gcnt = cin.gcount() ;  }
   winbuff[0] = winchar ; winbuff[1] = NULLCHAR ;
   tuiReport ( gsOut, (void*)winbuff, gcnt, this->wideStream ) ;
   this->acWrite ( lzr++, lzCOLOFF, gsOut ) ;   // display captured input data

   //*****************************************
   //* Input buffering disabled:             *
   //* Test wcin.get( char_type* )           *
   //*   or cin.get( char_type* )            *
   //*****************************************
   //* Results: Works correctly. count==1, returns keycode *
   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   if ( this->wideStream )                      // get user response (wide)
   { wcin.get( winchar ) ; gcnt = wcin.gcount() ; winbuff[0] = winchar ; winbuff[1] = NULLCHAR ; }
   else                                         // get user response (narrow)
   { cin.get( cinchar ) ;  gcnt = cin.gcount() ; cinbuff[0] = cinchar ; cinbuff[1] = NULLCHAR ; }
   tuiReport ( gsOut, (this->wideStream ? (void*)winbuff : (void*)cinbuff), gcnt, this->wideStream ) ;
   this->acWrite ( lzr++, lzCOLOFF, gsOut ) ;   // display captured input data

   //*****************************************
   //* Input buffering disabled:             *
   //* Test acRead( void )                   *
   //*****************************************
   //* Results: Works correctly. returns keycode *
   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   winchar = this->acRead () ;                  // get user response
   gcnt = 1 ;                                   // number of bytes captured (assumed)
   winbuff[0] = winchar ; winbuff[1] = NULLCHAR ;
   tuiReport ( gsOut, (void*)winbuff, gcnt, this->wideStream ) ;
   this->acWrite ( lzr++, lzCOLOFF, gsOut ) ;   // display captured input data

   //*****************************************
   //* Input buffering disabled:             *
   //* Test acRead( gString&, 1 )            *
   //*****************************************
   //* Results: Works correctly, returns one character *
   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   gcnt = this->acRead ( gsIn, 1 ) ;
   tuiReport ( gsOut, (this->wideStream ? (void*)(gsIn.gstr()) : (void*)(gsIn.ustr())), gcnt, this->wideStream ) ;
   this->acWrite ( lzr++, lzCOLOFF, gsOut ) ;   // display captured input data

   //*****************************************
   //* Input buffering disabled:             *
   //* Test acRead( gString&, 3 )            *
   //*****************************************
   //* Results: Works correctly, returns up to three characters. *
   //* The Enter keycode, if encountered, is discarded.          *
   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   gcnt = this->acRead ( gsIn, 3 ) ;
   tuiReport ( gsOut, (this->wideStream ? (void*)(gsIn.gstr()) : (void*)(gsIn.ustr())), gcnt, this->wideStream ) ;
   this->acWrite ( lzr++, lzCOLOFF, gsOut ) ;   // display captured input data

   //**************************
   //* Test wcin >> winbuff   *
   //* Test cin  >> cinbuff   *
   //**************************
   //* Results: Works as designed, but the design is grossly inefficient.
   //* (requires character PLUS Enter. Enter is not counted nor returned) *
   // Programmer's Note: For the stream redirection input, wcin.gcount() and
   // cin.gcount() always returns 1, regardless of the number of characters/bytes
   // captured. ALSO, these stream redirectors may not know the size of the target
   // buffer, potentially causing a buffer overflow. For these reasons, we
   // STRONGLY RECOMMEND that stream redirection not be used except with 
   // std::wstring or std::string which use dynamic allocations.
   // Note also that the terminating Enter key is not extracted from the input 
   // buffer and is not counted in wcin.size(). For this reason, we call
   // wcin.getline() or cin.getline() to extract and discard the Enter keycode.
   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   if ( this->wideStream )
   {
      std::wstring wstr ;                          // dynamic-allocation input buffer
      wcin >> wstr ;                               // get user response
      gcnt = wstr.size() ;                         // number of characters captured
      gsIn.loadChars( wstr.c_str(), gsMAXCHARS ) ; // copy to main buffer
      wcin.getline( winbuff, gsMAXCHARS ) ;        // clear newline from input stream
   }
   else
   {
      std::string cstr ;                           // dynamic-allocation input buffer
      cin >> cstr ;                                // get user response
      gcnt = cstr.size() ;                         // number of bytes captured
      gsIn.loadChars( cstr.c_str(), gsMAXCHARS ) ; // copy to main buffer
      cin.getline( cinbuff, gsMAXBYTES ) ;         // clear newline from input stream
   }
   tuiReport ( gsOut, (this->wideStream ? (void*)(gsIn.gstr()) : (void*)(gsIn.ustr())), gcnt, this->wideStream ) ;
   if ( (gsOut.gscols()) > outMAXWIDTH )           // limit output width
   {
      gsOut.limitCols( outMAXWIDTH - 3 ) ;
      gsOut.append( L"..." ) ;
   }
   this->acWrite ( lzr++, lzCOLOFF, gsOut ) ;   // display captured input data

   //*****************************************
   //* Input buffering disabled:             *
   //* Test C-style getwchar() -- wide       *
   //*   or C-style getchar()  -- narrow     *
   //*****************************************
   //* Results: Works correctly, returns a single character *
   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   if ( this->wideStream )                      // wide stream
   { winchar = getwchar () ; winbuff[0] = winchar ; winbuff[1] = NULLCHAR ; }
   else                                         // narrow stream
   { cinchar = getchar () ; cinbuff[0] = cinchar ; cinbuff[1] = NULLCHAR ; }
   gcnt = 1 ;                                   // number of bytes captured (assumed)
   tuiReport ( gsOut, (this->wideStream ? (void*)(winbuff) : (void*)(cinbuff)), gcnt, this->wideStream ) ;
   this->acWrite ( lzr++, lzCOLOFF, gsOut ) ;   // display captured input data

   //* Restore original terminal settings *
   this->acRestoreTermInfo () ;

   //* Retrieve a copy of the data we just set *
   this->acGetTermInfo ( stdIn, tios ) ;
   dtiTitle = "Original settings restored:" ;
   this->acDumpTermInfo ( tios, dumpRow, dumpCol, 2, &dtiTitle ) ;

   this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ; // go to landing zone
   this->acRead ( gsIn ) ;                      // get user response

   this->acSetFg ( aesFG_DFLT ) ;      // Reset to terminal default foreground color
   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;
   this->acFlushStreams () ;           // Clear input and output buffers

}  //* End Test_UnbufferedInput() *

//********************************************************************************
//* tuiReport:                                                                   *
//* Non-member support method for Test_UnbufferedInput().                        *
//* Format the raw data from the input stream showing keycodes and printing      *
//* characters.                                                                  *
//*                                                                              *
//* Input  : gs      : (by reference) receives formatted report text             *
//*          wbuff   : raw input                                                 *
//*          cnt     : number of characters in wbuff[]                           *
//*          wStream : 'true' if wide input stream, 'false' if narrow stream     *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

static void tuiReport ( gString& gs, const void *buff, short cnt, bool wStream )
{
   wchar_t wbuff[gsMAXCHARS] ;            // Working copy of raw data
   wchar_t wchar ;                        // character being processed

   //* Copy the raw wide-stream data to the work buffer.*
   if ( wStream )
   {
      gs = (const wchar_t *)(buff) ;
      gs.copy( wbuff, gsMAXCHARS ) ;
   }
   //* Convert narrow-stream data to wide stream for parsing *
   else
   {
      gs = (const char *)(buff) ;
      cnt = gs.gschars() ;
      gs.copy( wbuff, gsMAXCHARS ) ;
   }

   gs.compose( "%hd chars: [", &cnt ) ;
   for ( short i = ZERO ; i < cnt ; ++ i )
   {
      wchar = wbuff[i] ;

      gs.append( "%04X", &wchar ) ;
      if ( (iswprint ( wchar )) )         // printing characters reported directly
         gs.append( "'%C' ", &wchar ) ;

      //* Control characters are converted to plain text.*
      else if ( (iswcntrl ( wchar )) )
      {
         switch ( wchar )                 // ASCII control characters
         {
            case 0x0000:                  // ^0 NUL
               gs.append( "'\\0'" ) ;
               break ;
            case 0x000A:                  // ^J LF (linefeed)
               gs.append( "'\\n'" ) ;
               break ; 
            case 0x000D:                  // ^M CR (carriage return)
               gs.append( "'\\r'" ) ;
               break ; 
            case 0x0009:                  // ^I HT
               gs.append( "'\\t'" ) ;
               break ; 
            case 0x001B:                  // (n/a) ESC
               gs.append( "'esc'" ) ;
               break ;
            case 0x0001:                  // ^A SOH
            case 0x0002:                  // ^B STX
            case 0x0003:                  // ^C ETX
            case 0x0004:                  // ^D EOT
            case 0x0005:                  // ^E ENQ
            case 0x0006:                  // ^F ACK
            case 0x0007:                  // ^G BEL
            case 0x0008:                  // ^H BS
            case 0x000B:                  // ^K VT
            case 0x000C:                  // ^L FF
            case 0x000E:                  // ^N SO
            case 0x000F:                  // ^O SI
            case 0x0010:                  // ^P DLE
            case 0x0011:                  // ^Q DC1
            case 0x0012:                  // ^R DC2
            case 0x0013:                  // ^S DC3
            case 0x0014:                  // ^T DC4
            case 0x0015:                  // ^U NAK
            case 0x0016:                  // ^V SYN
            case 0x0017:                  // ^W ETB
            case 0x0018:                  // ^X CAN
            case 0x0019:                  // ^Y EM
            case 0x001A:                  // ^Z SUB
               wchar += 0x0040 ;
               gs.append( "'^%C'", &wchar ) ;
               break ;
            case 0x001C:                  // ^4 FS
            case 0x001D:                  // ^5 GS
            case 0x001E:                  // ^6 RS
            case 0x001F:                  // ^7 US
               wchar += 0x0018 ;
               gs.append( "'^%C'", &wchar ) ;
               break ;
            default:
               gs.append( "'^?'" ) ;
               break ;
         } ;
      }
      //* Neither a printing character nor a recognized control character.*
      else
      {
         wchar = L'?' ;
         gs.append( "'%C' ", &wchar ) ;
      }
   }
   gs.strip() ;
   gs.append( L"]" ) ;

}  //* End tuiReport() *

//*************************
//*     Test_Erasures     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test the ANSI commands of the aesERASE_xxx group.                            *
//*                                                                              *
//* Input  : fg   : foreground color attribute (member of aeSeq)                 *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_Erasures ( aeSeq fg )
{
   const wchar_t ARROW_UP    = L'↑',         // arrow up
                 ARROW_DOWN  = L'↓' ;        // arrow down

   gString gsOut ;                        // text formatting
   //* Construct the special character sequence for     *
   //* character written to the current cursor position.*
   gString gsCur( "%S$\b%S", 
                  ansiSeq[fg == aesFG_GREEN ? aesFG_RED : aesFG_GREEN],
                  ansiSeq[fg] ) ;
   short targetRow = 2,
         targetCol = this->termCols / 2 ;

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* Set terminal configuration for unbuffered input (echo disabled) *
   this->acBufferedInput ( false, noEcho ) ;

   //* Set the entire window (except first two rows) to a series of characters *
   this->acClearArea ( 3, 1, (this->termRows - 3), this->termCols, L'.' ) ;

   //** Test aesERASE_EOL
   //* Write a message telling user the test being performed *
   //* Set cursor on third line, write the cursor marker.    *
   gsOut = "Erase from cursor position to end of line. Press a key..." ;
   this->acWrite ( targetRow++, (targetCol - ((gsOut.gscols() / 2) - 1)), gsOut ) ;
   //* Set cursor on third line, write the cursor marker *
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, (this->termCols / 2 + 1) ) ;
   this->ttyWrite ( gsCur ) ;
   this->acRead () ;                         // get user response
   this->acEraseArea ( aesERASE_EOL ) ;      // erase to end of row
   targetRow += 2 ;

   //** Test aesERASE_BOL
   //* Write a message telling user the test being performed *
   //* Set cursor on sixth line, write the cursor marker.    *
   gsOut = "Erase from cursor position to start of line. Press a key..." ;
   this->acWrite ( targetRow++, (targetCol - ((gsOut.gscols() / 2) - 1)), gsOut ) ;
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, (this->termCols / 2 + 1) ) ;
   this->ttyWrite ( gsCur ) ;
   this->acRead () ;                         // get user response
   this->acEraseArea ( aesERASE_BOL ) ;      // erase to beginning of row
   targetRow += 2 ;

   //** Test aesERASE_LINE
   //* Write a message telling user the test being performed *
   //* Set cursor on ninth line, write the cursor marker.    *
   gsOut = "Erase the entire line where cursor sits. Press a key..." ;
   this->acWrite ( targetRow++, (targetCol - ((gsOut.gscols() / 2) - 1)), gsOut ) ;
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, (this->termCols / 2 + 1) ) ;
   this->ttyWrite ( gsCur ) ;
   this->acRead () ;                         // get user response
   this->acEraseArea ( aesERASE_LINE ) ;     // erase entire row
   targetRow += 2 ;

   //** Test aesERASE_TOW **
   //* Write a message telling user the test being performed *
   //* Set cursor on eleventh line, write the cursor marker. *
   gsOut = "Erase all data from the cursor to the top of the window. Press a key..." ;
   this->acWrite ( targetRow++, (targetCol - ((gsOut.gscols() / 2) - 1)), gsOut ) ;
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, (this->termCols / 2 + 1) ) ;
   this->ttyWrite ( gsCur ) ;
   this->acRead () ;                         // get user response
   this->acEraseArea ( aesERASE_TOW ) ;      // erase to top of window
   targetRow += 2 ;

   //** Test aesERASE_BOW **
   gsOut = "Erase all data from the cursor to the bottom of the window. Press a key..." ;
   this->acWrite ( targetRow++, (targetCol - ((gsOut.gscols() / 2) - 1)), gsOut ) ;
   this->acSetCursor ( aesCUR_ABSPOS, targetRow, (this->termCols / 2 + 1) ) ;
   this->acRead () ;                         // get user response
   this->acEraseArea ( aesERASE_BOW ) ;      // erase to bottom of window
   this->acClearArea ( (targetRow - 1), ZERO, 1, this->termCols, L'.' ) ;
   targetRow += 2 ;

   //** Test aesERASE_SAVED **
   //* (This erases the history i.e. the lines stored above the    *
   //* window for scroll-back).                                    *
   //* 1) Insert some data into some rows which will be scrolled   *
   //*    off the top of the window and wait for user input.       *
   //* 2) Walk the cursor downward to push the top rows off-screen *
   //* 3) Walk the cursor back to the top of the window.           *
   //* 4) Prompt the user to scroll downward to reveal the hidden  *
   //*    text, then scroll it back out-of-sight.                  *
   //* 5) Erase the text that is _logically_ above the window      *
   //*    (although user may have left it actually visible).       *
   //* This test is made more difficult due to the fact that none  *
   //* of the ANSI sequences cause data in the window to scroll.   *
   const wchar_t *esaveMSG =
     L"  These rows of text will be scrolled upward and out of the window.\n"
     L"  They will then be scrolled back into view to prove they are still present.\n"
     L"  Afterward they will be scrolled out again and will be deleted. %S" ;
   const wchar_t *esavePAK = L"Press a key..." ;

   WinPos savedPos = this->acGetCursor () ; // save the target position
   gsOut.compose( esaveMSG, esavePAK ) ;
   this->acWrite ( (targetRow - 8), 1, gsOut ) ;
   this->acRead () ;                      // get user response
   //* Erase the prompt.                                   *
   // *(Yes, this is much more complicated than necessary, *
   //* but we're exercising the functionality.)            *
   gsOut = esavePAK ;                     // erase the prompt
   this->acSetCursor ( aesCUR_COLSLT, (gsOut.gscols()) ) ;
   this->acEraseArea ( aesERASE_EOL ) ;
   this->acSetCursor ( savedPos ) ;       // return to the target position

   //* This is fancy: Scroll to the bottom of the terminal window *
   //* using a visual cue to show what's happening. Then enhance  *
   //* the visual cue and scroll the window upward, moving the    *
   //* first few rows out the top. These rows are the "saved"     *
   //* data which will be deleted while they are off-screen.      *
   WinPos wp ;
   const uint16_t tsec = 0 ;              // delay (1/10 sec)
   const uint16_t msec = 35 ;             // delay (millisec)
   targetCol = this->termCols / 3 + 1 ;
   this->acSetCursor ( aesCUR_ABSCOL, targetCol ) ;
   short stepRows = this->termRows - targetRow + 1,
         scrollRows = this->termRows - stepRows ;
   gsOut.compose( "%C", &ARROW_DOWN ) ;
   for ( short i = ZERO ; i < stepRows ; ++i )
   {
      this->acSetCursor ( aesCUR_NEXTROW, 1 ) ;
      this->ttyWrite ( gsOut ) ;
      this->nsleep ( tsec, msec ) ;
   }

   //* Add a newline so writing to stdout will scroll.*
   gsOut.append( "%C\n", &ARROW_DOWN ) ;
   this->acSetCursor ( aesCUR_COLSLT, 1 ) ;
   for ( short i = ZERO ; i < scrollRows ; ++i )
   {
      this->ttyWrite ( gsOut ) ;
      this->nsleep ( tsec, msec ) ;
   }
   //* Point upward, and then move upward *
   //* to top of terminal window.         *
   gsOut.compose( "%C%C", &ARROW_UP, &ARROW_UP ) ;
   this->acSetCursor ( aesCUR_COLSRT, 4 ) ;
   for ( short i = scrollRows ; i > ZERO ; --i )
   {
      this->acSetCursor ( aesCUR_ROWSUP, 1 ) ;
      this->acSetCursor ( aesCUR_COLSLT, 2 ) ;
      this->ttyWrite ( gsOut ) ;
      this->nsleep ( tsec, msec ) ;
   }
   gsOut.compose( "%C", &ARROW_UP ) ;
   do
   {
      this->acSetCursor ( aesCUR_ROWSUP, 1 ) ;
      this->acSetCursor ( aesCUR_COLSLT, 1 ) ;
      this->ttyWrite ( gsOut ) ;
      wp = this->acGetCursor () ;
      this->nsleep ( tsec, msec ) ;
   }
   while ( wp.row > 1 ) ;
   this->acSetCursor ( aesCUR_COLSRT, 2 ) ;

   //* This message is written to the firs column of the       *
   //* current row which _should be_ the top row of the window.*
   gsOut = "Use the mouse scroll wheel to scroll the messages back into view,\n"
           "then scroll the lines back out of the window, then press a key.\n" ;
   this->acSetCursor ( aesCUR_ABSCOL, 1 ) ;
   this->ttyWrite ( gsOut ) ;
   this->acRead () ;

   //* Erase the off-screen data *
   this->acEraseArea ( aesERASE_SAVED ) ;

   // Programmer's Note: This message is written at the _current_ cursor position.
   gsOut = "Hidden lines deleted.                                            \n"
           "Use the mouse scroll wheel to scroll the messages back into view.\n"
           "They should no longer be available -- then press a key." ;
   this->acWrite ( 1, 1, gsOut ) ;
   this->acRead () ;                         // get user response

   //** Test aesERASE_WIN **
   //* Write new data to the window, and prompt for a user response  *
   //* Note that the prompt is written in an API-level window in the *
   //* center of the terminal window as a test of that functionality.*
   this->acClearScreen ( L'.' ) ;

   const wchar_t* eraseMsg = L" Erase all data from the window. \n"
                             L"          Press a key...         " ;
   gsOut = eraseMsg ;
   targetRow = this->termRows / 4 + 1 ;
   targetCol = (this->termCols / 2) - ((gsOut.gscols()) / 4) ;

   ACWin acWin( targetRow, targetCol, 4, (((gsOut.gscols()) / 2) + 2),
                ltDUAL,                            // border style
                acAttr(acaFG_GREY | acaBG_BLUE),   // border attributes
                acAttr(acaFG_DFLT | acaBG_DFLT)    // interior attributes
      ) ;
   acWin.OpenWindow ( gsOut.ustr() ) ;
   this->acSetCursor ( aesCUR_COLSLT, 9 ) ;

   this->acRead () ;                      // get user response

   //* Close the window *
   acWin.CloseWindow ()  ;

   this->acBufferedInput ( true, termEcho ) ;// Restore buffered input
   this->acSetFg ( aesFG_DFLT ) ;            // Reset foreground to terminal default
   this->acEraseArea ( aesERASE_WIN ) ;      // erase entire window

   //* Prepare to exit *
   this->acWrite ( 1, 1, L"AnsiCmd - Erasure Tests Complete\n" ) ;
   this->acFlushStreams () ;           // Clear input and output buffers

}  //* End Test_Erasures() *

//*************************
//*     Test_AixFgBg      *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test the ANSI commands of the aesAIXFG_xxx and aesAIXBG_xxx groups.          *
//* Note that the AIXTERM commands are not part of the ANSI specification,       *
//* but the intense (bold) foreground and background commands may be supported.  *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: We default the foreground at the end of each record,      *
//* not because it is necessary, but because one record (probably ecsWHITE)      *
//* will be invisible due to having the same foreground and background colors.   *
//********************************************************************************

void AnsiCmd::Test_AixFgBg ( void )
{
   //* Loop control: two groups of 8 color sequences + the default sequence *
   const short recCNT = 16 ;
   const wchar_t *ColorName[recCNT] = 
   {
     L"  90) Intense Black Foreground   \n", L"  91) Intense Red Foreground     \n", 
     L"  92) Intense Green Foreground   \n", L"  93) Intense Brown Foreground   \n", 
     L"  94) Intense Blue Foreground    \n", L"  95) Intense Magenta Foreground \n", 
     L"  96) Intense Cyan Foreground    \n", L"  97) Intense Grey Foreground    \n\n",
     L" 100) Intense Black Background   \n", L" 101) Intense Red Background     \n", 
     L" 102) Intense Green Background   \n", L" 103) Intense Brown Background   \n", 
     L" 104) Intense Blue Background    \n", L" 105) Intense Magenta Background \n", 
     L" 106) Intense Cyan Background    \n", L" 107) Intense Grey Background    \n",
   } ;
   const wchar_t *Desc = 
         L"AIXTERM Extensions for Foreground/Background Colors:\n"
          "--------------------------------------------------------------\n"
          "As part of the IBM AIX operating system, Aixterm includes a\n"
          "semi-proprietary implementation of the ANSI standard escape\n"
          "sequences used for terminal control (ANSI X3.64 (ISO/IEC6429).\n"
          "\n"
          "Although the Aixterm foreground and background color\n"
          "attribute commands are not part of the ANSI standard, they\n"
          "are supported by many Linux-based terminal emulators.\n"
          "\n"
          "Please note that if the foreground and background are the\n"
          "same color, the text will be invisible.\n"
          "See the Intense Grey foreground example which causes the text\n"
          "to be rendered as white. We display it on a grey background to\n"
          "distinguish the text from the (presumed) white background." ;
   const short baseROW = 4 ;     // base display row
   const short exitROW = recCNT + 6 ; // cursor row when preparing to exit
   const short descCOL = 44 ;    // offset for description text

   gString gsOut ;               // text formatting
   short cnindx  = ZERO ;        // 'ColorName' index

   //** Test aesAIXFG_xxxx group **
   for ( short regindx = aesAIXFGb_BLACK ; regindx <= aesAIXFGb_GREY ; ++regindx )
   {
      if ( regindx == aesAIXFGb_GREY )          // set special-case background
         this->acSetBg ( aesBG_GREY ) ;
      this->acSetAixFg ( aeSeq(regindx) ) ;     // set the foreground color
      this->ttyWrite ( ColorName[cnindx++] ) ;  // write the text data
      this->acSetFg ( aesFG_DFLT ) ;            // return to default foreground
      if ( regindx == aesAIXFGb_GREY )          // reset special-case background
         this->acSetBg ( aesBG_DFLT ) ;
      this->acFlushOut () ;                     // flush the output stream
   }

   //** Test aesAIXBG_xxxx group **
   for ( short regindx = aesAIXBGb_BLACK ; regindx <= aesAIXBGb_GREY ; ++regindx )
   {
      this->acSetAixBg ( aeSeq(regindx) ) ;     // set the background color
      this->ttyWrite ( ColorName[cnindx++] ) ;  // write the text data
      this->acSetFg ( aesBG_DFLT ) ;            // return to default background
      this->acFlushOut () ;                     // flush the output stream
   }

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   this->acWrite ( baseROW, descCOL, Desc ) ;

   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;

}  //* End Test_AixFgBg() *

//*************************
//*    Test_acaExpand     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Exercise the color-attribute tracking data.                                  *
//* The tracking data are updated whenever the color attributes or text          *
//* modifiers are changed.                                                       *
//* Tracking data are stored in the "attrBits" member which is an acaExpand      *
//* class object.                                                                *
//*                                                                              *
//* There are two major groups of attribute-modification methods:                *
//*  1) Processing the application-level commands, (see enum acAttr).            *
//*     Decoding of data for these commands is integrated into the commands      *
//*     themselves.                                                              *
//*  2) Processing the basic ANSI commands, (see enum aeSeq).                    *
//*     Decoding of data for these commands is in addition to issuing the        *
//*     actual ANSI escape sequences. For this reason, the opportunity for       *
//*     errors is higher, so be aware.                                           *
//*                                                                              *
//* Input  : call : if 'b', use low-level ANSI calls, else use API calls.        *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_acaExpand ( wchar_t call )
{
   //* This method tests the acaExpand data member updates. *
   //* These data members are updated either through the    *
   //* low-level ANSI-escape-sequence calls, OR through the *
   //* higher-level application-level calls.                *
   //* This flag indicates whether the tests are executed   *
   //* through the API-level methods, or the Basic          *
   //* low-level, direct ANSI methods.                      *
   const bool apiCalls = bool(call == L'b' ? false : true) ;

   const short dumpROW = 4 ;        // position of data dump
   const short dumpCOL = 60 ;
   const short exitROW = 36 ;       // exit position
   const short sampCOL = 4 ;        // position for sample text
   const short sampROW = dumpROW + 2 ;
   const short sampHGT = 3 ;        // dimensions of sample-text area
   const short sampWID = 40 ;

   const wchar_t *attrMenu = 
      L"a) Fg Black    j) Bg Black     r) Bold\n"
       "b) Fg Red      k) Bg Red       s) Italic\n"
       "c) Fg Green    l) Bg Green     t) Underline\n"
       "d) Fg Brown    m) Bg Brown     u) Overline\n"
       "e) Fg Blue     n) Bg Blue      v) X-out\n"
       "f) Fg Magenta  o) Bg Magenta   w) Blink\n"
       "g) Fg Cyan     p) Bg Cyan      x) Reversed\n"
       "h) Fg Grey     q) Bg Grey      y) Hidden\n"
       "\n"
      L"A) Fg Black    J) Bg Black     R) Bold Off\n"
       "B) Fg Red      K) Bg Red       S) Italic Off\n"
       "C) Fg Green    L) Bg Green     T) Uline Off\n"
       "D) Fg Brown    M) Bg Brown     U) Oline Off\n"
       "E) Fg Blue     N) Bg Blue      V) X-out Off\n"
       "F) Fg Magenta  O) Bg Magenta   W) Blink Off\n"
       "G) Fg Cyan     P) Bg Cyan      X) Reverse Off\n"
       "H) Fg Grey     Q) Bg Grey      Y) Hidden Off\n"
       "\n"
      L"1) 8-bit Fgnd  4) 8-bit Bgnd   Z) All Mods Off\n"
       "2) RGB Fgnd    5) RGB Bgnd     z) Terminal\n"
       "3) Fg Default  6) Bg Default      Defaults\n"
       
       ;
   const wchar_t *Headers = 
      L"   4-Bit Fg       4-Bit Bg     Text Mods      \n"
       "\n\n\n\n\n\n\n\n"
       "   Bold Fg        Bold Bg      Text Mod Off   \n"
       "\n\n\n\n\n\n\n\n"
       "  8Bit,RGB Fg    8Bit,RGB Bg   Resets         \n" ;
   const wchar_t *Instructions = 
      L"   Sample Text Using Current Attributes\n"
      "\n\n\n\n\n\n"
      "Select an attribute to apply to sample text.\n"
      "           Press ENTER to Exit." ;
   const wchar_t *Description = 
      L"This set of tests exercises the attribute-tracking data configured\n"
       "as bitfields in an acaExpand object which is a member of the AnsiCmd\n"
       "class dataset. Each of the 4-bit foreground and background attributes\n"
       "may be specified, and all text modifiers may be set or reset individually.\n" ;
   const wchar_t *Desc2 = 
      L"The 8-bit attributes are implemented as a pair of tests using\n"
       "representative values: fgnd:%hhu(%02hhXh) lime green, and bgnd:%hhu(%02hhXh) purple.\n" ;
   const wchar_t *Desc3 = 
      L"The RGB attributes are implemented as a pair of tests using\n"
       "representative values: fgnd:%hhu(%02hhXh) cyan and bgnd:%hhu(%02hhXh) green." ;
   const wchar_t *sampText =
      L" Sample text rendered using the current \n"
       "  color attributes and text modifiers.  \n"
       " Where in the World is Carmen SanDiego? " ;

   //* Example 8-bit and web-safe RGB attribute indices *
   //* representing these attribute groups.             *
   const uint8_t fg8BIT = 154 ;                             // lime green
   const uint8_t bg8BIT = 91 ;                              // purple
   const uint8_t fgRGB = wsrgbCYAN + (wsrgbSHADES - 2) ;    // light cyan
   const uint8_t bgRGB = wsrgbGREEN + (wsrgbSHADES / 3) ;   // medium green

   acaExpand dfltBits,              // copy of default settings
             currBits ;             // copy of current settings
   acAttr newAttr ;                 // construct bitfields
   bool   newfgnd, newbgnd, newmods ; // type of change specified
   gString gsOut, gsTitle ;         // text formatting
   wchar_t wch ;                    // user input
   WinPos wpBox( short(sampROW - 1), short(sampCOL - 1) ),
          wpTxt( sampROW, sampCOL ),
          wpLZone ;
   bool   goodUser = true ;         // 'true' if user selects a valid option

   //* Set terminal configuration for unbuffered input (no echo) *
   this->acBufferedInput ( false, noEcho ) ;

   //* Set all attributes to default values *
   //* and display the default settings.    *
   this->acReset () ;
   this->acGetAttributes ( dfltBits ) ; // save a copy of the default settings
   gsTitle = L"Default acaExp-object Settings" ;
   this->acaExpandDump ( dfltBits, dumpROW, (dumpCOL + 40), &gsTitle ) ;
   gsTitle = L"  Current acaExp-object Settings  " ;

   //* Visually define the sample-text area and display user instructions.*
   wpLZone = this->acWrite ( dumpROW, wpBox.col, Instructions ) ;
   this->acDrawBox ( wpBox, (sampHGT + 2), (sampWID + 2) ) ;

   //* Display description of this test *
   gsOut.compose( "%S Testing Specified.\n",
                  (apiCalls ? L"API-level (acSetAttributes)" : 
                              L"ANSI-level (acSetFg/acSetBg/acSetMod)") ) ;
   this->acSetMod ( aesITALIC ) ;
   WinPos wp = this->acWrite ( (dumpROW + 13), dumpCOL, gsOut ) ;
   this->acSetMod ( aesITALIC_OFF ) ;
   wp = this->acWrite ( wp, Description ) ;
   gsOut.compose( Desc2, &fg8BIT, &fg8BIT, &bg8BIT, &bg8BIT ) ;
   wp = this->acWrite ( wp, gsOut ) ;
   gsOut.compose( Desc3, &fgRGB, &fgRGB, &bgRGB, &bgRGB ) ;
   wp = this->acWrite ( wp, gsOut ) ;

   //* Display the user menu *
   if ( apiCalls )   //** API-level testing **
   {
      this->acSetAttributes ( acAttr(dfltBits.acaVal | acaUNDERLINE | acaOVERLINE) ) ;
      this->acWrite ( wpLZone.row + 2, wpBox.col, Headers ) ;
      this->acSetAttributes ( acAttr(dfltBits.acaVal | acaCLEAR_MODS | acaUNDERLINE | acaOVERLINE) ) ;
      this->acWrite ( wpLZone.row + 3, wpBox.col, attrMenu ) ;
   }
   else     //** ANSI-level testing **
   {
      this->acSetMod ( aesUNDERLINE ) ;
      this->acSetMod ( aesOVERLINE ) ;
      this->acWrite ( wpLZone.row + 2, wpBox.col, Headers ) ;
      this->acSetMod ( aesUNDERLINE_OFF ) ;
      this->acSetMod ( aesOVERLINE_OFF ) ;
      this->acWrite ( wpLZone.row + 3, wpBox.col, attrMenu ) ;
   }        //** ANSI-level testing **

   //* Get user's selections *
   do
   {
      //* If valid user selection *
      if ( goodUser )
      {
         //* Display some text using the current settings *
         this->acWrite ( wpTxt, sampText ) ;

         //* Display the current attribute settings *
         this->acGetAttributes ( currBits ) ;

         if ( apiCalls )   //** API-level testing **
         {
            this->acSetAttributes ( acaATTR_DFLT, true ) ;  // set default attributes
            this->acaExpandDump ( currBits, dumpROW, dumpCOL, &gsTitle ) ; // display data
            this->acSetAttributes ( currBits.acaVal ) ;     // restore target attributes
         }        //** API-level testing

         else     //** ANSI-level testing **
         {
            this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;    // set default attributes
            this->acResetMods () ;                          // reset all modifiers
            this->acaExpandDump ( currBits, dumpROW, dumpCOL, &gsTitle ) ; // display data
            if ( currBits.bgBits != acaBG_DFLT )            // restore target bgnd
               this->acSetBg ( currBits.aesBgnd ) ;
            if ( currBits.acaFgnd != acaFG_DFLT )           // restore target fgnd
               this->acSetFg ( currBits.aesFgnd ) ;
            if ( currBits.boldMod )       this->acSetMod ( aesBOLD ) ; // restore mods
            if ( currBits.italicMod )     this->acSetMod ( aesITALIC ) ;
            if ( currBits.ulineMod )      this->acSetMod ( aesUNDERLINE ) ;
            if ( currBits.olineMod )      this->acSetMod ( aesOVERLINE ) ;
            if ( currBits.xoutMod )       this->acSetMod ( aesXOUT ) ;
            if ( currBits.blinkMod )      this->acSetMod ( aesBLINK_FAST ) ;
            if ( currBits.revMod )        this->acSetMod ( aesREVERSE ) ;
            if ( currBits.invisMod )      this->acSetMod ( aesCONCEAL ) ;
         }     //** ANSI-level testing **

         this->acSetCursor ( wpLZone ) ;
      }
      goodUser = true ;             // assume user can read (always a risky assumption :-)

      wch = this->acRead () ;       // get user input

      //***********************
      //** API-level testing **
      //***********************
      if ( apiCalls )
      {
         newfgnd = newbgnd = newmods = false ;
         newAttr = acaPLAINTXT ;
         switch ( wch )
         {
            //** 4-Bit Foreground **
            case L'a':  newAttr = acaFG_BLACK ;    newfgnd = true ;  break ;
            case L'b':  newAttr = acaFG_RED ;      newfgnd = true ;  break ;
            case L'c':  newAttr = acaFG_GREEN ;    newfgnd = true ;  break ;
            case L'd':  newAttr = acaFG_BROWN ;    newfgnd = true ;  break ;
            case L'e':  newAttr = acaFG_BLUE ;     newfgnd = true ;  break ;
            case L'f':  newAttr = acaFG_MAGENTA ;  newfgnd = true ;  break ;
            case L'g':  newAttr = acaFG_CYAN ;     newfgnd = true ;  break ;
            case L'h':  newAttr = acaFG_GREY ;     newfgnd = true ;  break ;
            case L'A':  newAttr = acaFGb_BLACK ;   newfgnd = true ;  break ;
            case L'B':  newAttr = acaFGb_RED ;     newfgnd = true ;  break ;
            case L'C':  newAttr = acaFGb_GREEN ;   newfgnd = true ;  break ;
            case L'D':  newAttr = acaFGb_BROWN ;   newfgnd = true ;  break ;
            case L'E':  newAttr = acaFGb_BLUE ;    newfgnd = true ;  break ;
            case L'F':  newAttr = acaFGb_MAGENTA ; newfgnd = true ;  break ;
            case L'G':  newAttr = acaFGb_CYAN ;    newfgnd = true ;  break ;
            case L'H':  newAttr = acaFGb_GREY ;    newfgnd = true ;  break ;
            case L'3':  newAttr = acaFG_DFLT ;     newfgnd = true ;  break ;

            //** 8-Bit Foreground **
            case L'1':
               {
               uint8_t currBgnd = ZERO ;  // placeholder (integration performed below)
               newAttr = this->acComposeAttributes ( fg8BIT, currBgnd, false, false ) ;
               newfgnd = true ;
               }
               break ;

            //** RGB Foreground **
            case L'2':
               {
               uint8_t currBgnd = ZERO ;  // placeholder (integration performed below)
               newAttr = this->acComposeAttributes ( fgRGB, currBgnd, true, false ) ;
               newfgnd = true ;
               }
               break ;

            //** 4-Bit Background **
            case L'j':  newAttr = acaBG_BLACK ;    newbgnd = true ;  break ;
            case L'k':  newAttr = acaBG_RED ;      newbgnd = true ;  break ;
            case L'l':  newAttr = acaBG_GREEN ;    newbgnd = true ;  break ;
            case L'm':  newAttr = acaBG_BROWN ;    newbgnd = true ;  break ;
            case L'n':  newAttr = acaBG_BLUE ;     newbgnd = true ;  break ;
            case L'o':  newAttr = acaBG_MAGENTA ;  newbgnd = true ;  break ;
            case L'p':  newAttr = acaBG_CYAN ;     newbgnd = true ;  break ;
            case L'q':  newAttr = acaBG_GREY ;     newbgnd = true ;  break ;
            case L'J':  newAttr = acaBGb_BLACK ;   newbgnd = true ;  break ;
            case L'K':  newAttr = acaBGb_RED ;     newbgnd = true ;  break ;
            case L'L':  newAttr = acaBGb_GREEN ;   newbgnd = true ;  break ;
            case L'M':  newAttr = acaBGb_BROWN ;   newbgnd = true ;  break ;
            case L'N':  newAttr = acaBGb_BLUE ;    newbgnd = true ;  break ;
            case L'O':  newAttr = acaBGb_MAGENTA ; newbgnd = true ;  break ;
            case L'P':  newAttr = acaBGb_CYAN ;    newbgnd = true ;  break ;
            case L'Q':  newAttr = acaBGb_GREY ;    newbgnd = true ;  break ;
            case L'6':  newAttr = acaBG_DFLT ;     newbgnd = true ;  break ;

            //** 8-Bit Background **
            case L'4':
               {
               uint8_t currFgnd = ZERO ;  // placeholder (integration performed below)
               newAttr = this->acComposeAttributes ( currFgnd, bg8BIT, false, false ) ;
               newbgnd = true ;
               }
               break ;

            //** RGB Background **
            case L'5':
               {
               uint8_t currFgnd = ZERO ;  // placeholder (integration performed below)
               newAttr = this->acComposeAttributes ( currFgnd, bgRGB, false, true ) ;
               newbgnd = true ;
               }
               break ;

            //** Text Modifiers **
            case L'r':  newAttr = acaBOLD ;              break ;
            case L's':  newAttr = acaITALIC ;            break ;
            case L't':  newAttr = acaUNDERLINE ;         break ;
            case L'u':  newAttr = acaOVERLINE ;          break ;
            case L'v':  newAttr = acaXOUT ;              break ;
            case L'w':  newAttr = acaBLINK ;             break ;
            case L'x':  newAttr = acaREVERSE ;           break ;
            case L'y':  newAttr = acaCONCEAL ;           break ;

            case L'R':  newAttr = acAttr(acaCLEAR_MODS | acaBOLD) ;        break ;
            case L'S':  newAttr = acAttr(acaCLEAR_MODS | acaITALIC) ;      break ;
            case L'T':  newAttr = acAttr(acaCLEAR_MODS | acaUNDERLINE) ;   break ;
            case L'U':  newAttr = acAttr(acaCLEAR_MODS | acaOVERLINE) ;    break ;
            case L'V':  newAttr = acAttr(acaCLEAR_MODS | acaXOUT) ;        break ;
            case L'W':  newAttr = acAttr(acaCLEAR_MODS | acaBLINK) ;       break ;
            case L'X':  newAttr = acAttr(acaCLEAR_MODS | acaREVERSE) ;     break ;
            case L'Y':  newAttr = acAttr(acaCLEAR_MODS | acaCONCEAL) ;     break ;
            case L'Z':  newAttr = acaCLEAR_MODS ;                          break ;

            //** Terminal Defaults **
            case L'z':
               newAttr = acAttr(acaCLEAR_MODS | acaATTR_DFLT) ;
               newfgnd = newbgnd = true ;
               break ;

            //** Invalid Selection **
            default:
               if ( wch != NEWLINE )
               { this->acBeep () ; goodUser = false ; }
               break ;
         } ;
         if ( goodUser )
         {
            //* If no change to fgnd/bgnd specified, integrate existing values.*
            if ( ! newfgnd )  newAttr = acAttr(newAttr | currBits.fgBits) ;
            if ( ! newbgnd )  newAttr = acAttr(newAttr | currBits.bgBits) ;
            this->acSetAttributes ( newAttr ) ;
         }
      }        //** API-level testing **

      //************************
      //** ANSI-level testing **
      //************************
      else
      {
         switch ( wch )
         {
            //** 4-Bit Foreground **
            case L'a':  this->acSetFg ( aesFG_BLACK ) ;        break ;
            case L'b':  this->acSetFg ( aesFG_RED ) ;          break ;
            case L'c':  this->acSetFg ( aesFG_GREEN ) ;        break ;
            case L'd':  this->acSetFg ( aesFG_BROWN ) ;        break ;
            case L'e':  this->acSetFg ( aesFG_BLUE ) ;         break ;
            case L'f':  this->acSetFg ( aesFG_MAGENTA ) ;      break ;
            case L'g':  this->acSetFg ( aesFG_CYAN ) ;         break ;
            case L'h':  this->acSetFg ( aesFG_GREY ) ;         break ;
            case L'A':  this->acSetFg ( aesFGb_BLACK ) ;       break ;
            case L'B':  this->acSetFg ( aesFGb_RED ) ;         break ;
            case L'C':  this->acSetFg ( aesFGb_GREEN ) ;       break ;
            case L'D':  this->acSetFg ( aesFGb_BROWN ) ;       break ;
            case L'E':  this->acSetFg ( aesFGb_BLUE ) ;        break ;
            case L'F':  this->acSetFg ( aesFGb_MAGENTA ) ;     break ;
            case L'G':  this->acSetFg ( aesFGb_CYAN ) ;        break ;
            case L'H':  this->acSetFg ( aesFGb_GREY ) ;        break ;
            case L'3':  this->acSetFg ( aesFG_DFLT ) ;         break ;

            //** 8-Bit Foreground **
            case L'1':  this->acBeep ( 2, 2 ) ; goodUser = false ;   break ;

            //** RGB Foreground **
            case L'2':  this->acBeep ( 2, 2 ) ; goodUser = false ;   break ;

            //** 4-Bit Background **
            case L'j':  this->acSetBg ( aesBG_BLACK ) ;        break ;
            case L'k':  this->acSetBg ( aesBG_RED ) ;          break ;
            case L'l':  this->acSetBg ( aesBG_GREEN ) ;        break ;
            case L'm':  this->acSetBg ( aesBG_BROWN ) ;        break ;
            case L'n':  this->acSetBg ( aesBG_BLUE ) ;         break ;
            case L'o':  this->acSetBg ( aesBG_MAGENTA ) ;      break ;
            case L'p':  this->acSetBg ( aesBG_CYAN ) ;         break ;
            case L'q':  this->acSetBg ( aesBG_GREY ) ;         break ;
            case L'J':  this->acSetBg ( aesBGb_BLACK ) ;       break ;
            case L'K':  this->acSetBg ( aesBGb_RED ) ;         break ;
            case L'L':  this->acSetBg ( aesBGb_GREEN ) ;       break ;
            case L'M':  this->acSetBg ( aesBGb_BROWN ) ;       break ;
            case L'N':  this->acSetBg ( aesBGb_BLUE ) ;        break ;
            case L'O':  this->acSetBg ( aesBGb_MAGENTA ) ;     break ;
            case L'P':  this->acSetBg ( aesBGb_CYAN ) ;        break ;
            case L'Q':  this->acSetBg ( aesBGb_GREY ) ;        break ;
            case L'6':  this->acSetBg ( aesBG_DFLT ) ;         break ;

            //** 8-Bit Background **
            case L'4':  this->acBeep ( 2, 2 ) ; goodUser = false ;   break ;

            //** RGB Background **
            case L'5':  this->acBeep ( 2, 2 ) ; goodUser = false ;   break ;

            //** Text Modifiersd **
            case L'r':  this->acSetMod ( aesBOLD ) ;           break ;
            case L's':  this->acSetMod ( aesITALIC ) ;         break ;
            case L't':  this->acSetMod ( aesUNDERLINE ) ;      break ;
            case L'u':  this->acSetMod ( aesOVERLINE ) ;       break ;
            case L'v':  this->acSetMod ( aesXOUT ) ;           break ;
            case L'w':  this->acSetMod ( aesBLINK_FAST ) ;     break ;
            case L'x':  this->acSetMod ( aesREVERSE ) ;        break ;
            case L'y':  this->acSetMod ( aesCONCEAL ) ;        break ;

            case L'R':  this->acSetMod ( aesBOLD_OFF ) ;       break ;
            case L'S':  this->acSetMod ( aesITALIC_OFF ) ;     break ;
            case L'T':  this->acSetMod ( aesUNDERLINE_OFF ) ;  break ;
            case L'U':  this->acSetMod ( aesOVERLINE_OFF ) ;   break ;
            case L'V':  this->acSetMod ( aesXOUT_OFF ) ;       break ;
            case L'W':  this->acSetMod ( aesBLINK_OFF ) ;      break ;
            case L'X':  this->acSetMod ( aesREVERSE_OFF ) ;    break ;
            case L'Y':  this->acSetMod ( aesCONCEAL_OFF ) ;    break ;
            case L'Z':  this->acResetMods () ;                 break ;

            //** Terminal Defaults **
            case L'z':  this->acReset () ;                     break ;

            //** Invalid Selection **
            default:
               if ( wch != NEWLINE )
               { this->acBeep () ; goodUser = false ; }
         } ;
      }     //** ANSI-level testing **
   }
   while ( wch != NEWLINE ) ;

   //* Prettify the sample area *
   this->acReset () ;      // set all color attributes and mods to defaults
   this->acClearArea ( sampROW, sampCOL, sampHGT, sampWID, L'-' ) ;

   //* Prepare to exit *
   this->acBufferedInput ( true, termEcho ) ;         // restore buffered input
   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;  // set cursor to exit position

}  //* End Test_acaExpand() *

//*************************
//*    Test_AsciiCodes    *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Exercise the ASCII control codes.                                            *
//* Note that many of these will be captured by the system, and others will      *
//* simply be ignored. We have moved a long way beyond DOS.                      *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_AsciiCodes ( aeSeq fg )
{
   const short testCNT = 32 ;    // control-code group plus DEL (7Fh)
   char CtrlCodes[testCNT] =                    // list of control characters
   {  //* Keep synchronized with labelText[] *
      //* Common codes which might actually be recognized *
      0x07,    // Bell
      0x08,    // Backspace
      0x09,    // H Tab
      0x0A,    // Linefeed
      0x0B,    // V Tab
      0x0D,    // Carriage Return
      0x0C,    // Formfeed
      0x7F,    // Delete

      //* Group 2: unlikely to be recognized *
      0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x0E, 0x0F,
      0x10, 0x11, 0x12, 0x13, 

      //* Group 3: unlikely to be recognized *
      0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
      0x1C, 0x1D, 0x1E, 0x1F,
   } ;

   const wchar_t *labelText[testCNT] = 
   {  //* Keep synchronized with CtrlCodes[] *
      L"07h \\a  Bell",
      L"08h \\b  Backspace",
      L"09h \\t  Horizontal Tab",
      L"0Ah \\n  Linefeed",
      L"0Bh \\v  Vertical Tab",
      L"0Dh \\r  Carriage Return",
      L"0Ch \\f  Form Feed",
      L"7Fh ^?  Delete",

      L"01h ^A  Start of-Heading",
      L"02h ^B  Start-of-Text",
      L"03h ^C  End-of-Text",
      L"04h ^D  End-of-Transmission",
      L"05h ^E  Enquiry",
      L"06h ^F  Acknowledgement",
      L"0Eh ^N  Shift Out",
      L"0Fh ^O  Shift In",
      L"10h ^P  Data Link Escape",
      L"11h ^Q  Dev Ctrl 1 (XOFF)",
      L"12h ^R  Dev Ctrl 2",
      L"13h ^S  Dev Ctrl 3 (XON)",

      L"14h ^T  Dev Ctrl 4",
      L"15h ^U  Neg. Ack. (NAK)",
      L"16h ^V  Synchronous Idle",
      L"17h ^W  End Transmission Block",
      L"18h ^X  Cancel",
      L"19h ^Y  End-of-Medium",
      L"1Ah ^Z  Substitute",
      L"1Bh ^[  Escape",
      L"1Ch ^\\  File Separator",
      L"1Dh ^]  Group Seperator",
      L"1Eh ^^  Record Seperator",
      L"1Fh ^_  Unit Seperator",
      //L"00h \\0  Null",
   } ;
   const wchar_t *Desc = 
   L"ASCII Control Codes Test:\n"
    "-------------------------------------------------------------------------------\n"
    "The ASCII (American Standard Code for Information) Exchange) control codes are\n"
    "the first 32 characters of the ASCII table. These are non-printing characters,\n"
    "that is, they will not be visible on the screen.\n"
    "Many of these codes were designed to control printers, teletype machines,\n"
    "paper-tape punches, and so on. For this reason, most of them have fallen into\n"
    "disuse since the end of the last Ice Age.\n"
    "\n"
    "To prevent chaos, the terminal software will capture and discard most of these\n"
    "codes; however, a few codes are still in common use, especially in console\n"
    "applications which directly control cursor positioning.\n"
    "The most common are bell(07h \\a), backspace(08h \\b), delete(7Fh),\n"
    "horizontal tab(09h \\t), vertical tab(0Bh \\v), linefeed/newline(0Ah \\n),\n"
    "carriage return(0Dh \\r), and of course escape(1Bh \\e).\n" ;
   const short baseROW = 4 ;                    // base display row
   const short baseCOL = 2 ;                    // base display column
   const short lzROWS = 12 ;                    // height of landing zone
   const short lzCOLS = 48 ;                    // width of landing zone
   const short labROWS = lzROWS ;               // label rows
   const short labCOLS = 30 ;                   // label columns
   const short labOFF  = baseCOL + lzCOLS + 2 ; // label start column
   const short descROW = baseROW + lzROWS + 1 ; // row for description text
   const short descCOL = baseCOL ;              // column for description text
   const short promptROW = baseROW - 1 ;        // row of user prompt
   const short promptCOL = baseCOL + 11 ;       // column of user prompt
   const short reportCOL = promptCOL ;          // cursor-report column
   const short exitROW = descROW + 15 ;         // cursor row when exiting the test
   const short commonCODES = 8 ;                // items in "common" group
   const short grp3INDEX   = 20 ;               // index of group three

   gString gsOut,                               // text formatting
           gsPrompt( "Press a key...  (Q to quit)" ), // user prompt
           gsPClear,                            // clear prompt
           gsRClear( "...................." ) ; // clear position report
   const short promptWIDTH = gsPrompt.gscols() ; // number of columns in prompt
   short indx ;                                 // loop control
   gsPClear.padCols( promptWIDTH ) ;               // prompt erase text
   wchar_t userIn ;                             // user response to prompt
   short   tsec = 15 ;                          // delay in tenths of a second

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* Write a description of the test *
   this->acWrite ( descROW, descCOL, Desc ) ;

   //* Create a landing zone *
   short lzr = baseROW, lzc = baseCOL, lzcShift = 8 ;
   #if 1    // Production: single-column character
   this->acClearArea ( lzr, lzc, lzROWS, lzCOLS, L'.' ) ;
   #else    // For Testing Only: test handling of multi-column characters
   this->acClearArea ( lzr, lzc, lzROWS, lzCOLS, L'马' ) ;
   #endif   // End production
   lzc += lzCOLS / 2 - 1 ;

   //* Set terminal configuration for unbuffered input *
   this->acBufferedInput ( false, noEcho ) ;

   for ( indx = ZERO ; indx < commonCODES ; ++indx )
   {
      this->acWrite ( lzr, labOFF, labelText[indx], false ) ;
      this->acWrite ( promptROW, promptCOL, gsPrompt, false ) ;
      if ( (CtrlCodes[indx] == 0x0B) || (CtrlCodes[indx] == 0x0C) )
         lzc -= lzcShift ;
      this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ;  // position cursor for test
      if ( CtrlCodes[indx] != 0x07 )                   // report cursor position
         this->Test_ReportCurPos ( (lzr - 1), reportCOL ) ;
      userIn = this->acRead () ;                       // get user response
      this->acWrite ( promptROW, promptCOL, gsPClear, false ) ;
      this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ;  // reposition cursor
      if ( (CtrlCodes[indx] == 0x0B) || (CtrlCodes[indx] == 0x0C) )
         lzc += lzcShift ;

      if ( (userIn = toupper ( userIn )) == L'Q' )     // get user response
         break ;
      this->ttyWrite ( CtrlCodes[indx] ) ;

      if ( CtrlCodes[indx] != 0x07 )                   // report cursor position
         this->Test_ReportCurPos ( (lzr - 1), reportCOL ) ;
      this->nsleep ( tsec ) ;       // wait a moment so user can see the result
      if ( CtrlCodes[indx] != 0x07 )                   // clear cursor report
         this->acWrite ( (lzr - 1), reportCOL, gsRClear, false ) ;
      ++lzr ;                                          // move to next row
   }

   if ( userIn != L'Q' )      // if no user abort
   {
      //* Clear the label area *
      this->acClearArea ( baseROW, labOFF, labROWS, labCOLS ) ;
      lzr = baseROW ;                        // return to top of landing zone
      tsec /= 2 ;    // shorter delay (because nothing is likely to happen)

      for ( ; indx < grp3INDEX ; ++indx )    // process group 2
      {
         this->acWrite ( lzr, labOFF, labelText[indx], false ) ;
         this->acWrite ( promptROW, promptCOL, gsPrompt, false ) ;
         this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ;  // position cursor for test
         userIn = this->acRead () ;                       // get user response
         this->acWrite ( promptROW, promptCOL, gsPClear, false ) ;
         this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ;  // reposition cursor

         if ( (userIn = toupper ( userIn )) == L'Q' )     // get user response
            break ;
         this->ttyWrite ( CtrlCodes[indx] ) ;
         this->nsleep ( tsec ) ;    // wait a moment so user can see the result
         ++lzr ;                                          // move to next row
      }
      lzr = baseROW ;                     // return to top of landing zone
   }

   if ( userIn != L'Q' )      // if no user abort
   {
      //* Clear the label area *
      this->acClearArea ( baseROW, labOFF, labROWS, labCOLS ) ;
      lzr = baseROW ;                        // return to top of landing zone

      for ( ; indx < testCNT ; ++indx )      // process group 3
      {
         this->acWrite ( lzr, labOFF, labelText[indx], false ) ;
         this->acWrite ( promptROW, promptCOL, gsPrompt, false ) ;
         this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ;  // position cursor for test
         userIn = this->acRead () ;                       // get user response
         this->acWrite ( promptROW, promptCOL, gsPClear, false ) ;
         this->acSetCursor ( aesCUR_ABSPOS, lzr, lzc ) ;  // reposition cursor

         if ( (userIn = toupper ( userIn )) == L'Q' )     // get user response
            break ;
         this->ttyWrite ( CtrlCodes[indx] ) ;
         this->nsleep ( tsec ) ;    // wait a moment so user can see the result
         ++lzr ;                                          // move to next row
      }
   }

   if ( userIn != L'Q' )
   {
      gsOut = L"All tests complete. Press any key to exit..." ;
      this->acWrite ( promptROW, promptCOL, gsOut ) ;
      this->acRead () ;
   }
   else
   {
      gsOut = L"Test sequence terminated." ;
      this->acWrite ( promptROW, promptCOL, gsOut ) ;
   }

   //* Prepare to exit *
   this->acRestoreTermInfo () ;        // Restore original terminal settings
   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ; // set cursor
   this->acSetFg ( aesFG_DFLT ) ;      // Reset to terminal default foreground color
   this->acFlushStreams () ;           // Clear input and output buffers

}  //* End Test_AsciiCodes() *

//*************************
//*     Test_Ideogram     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test for system support for the Ideogram group of ANSI escape sequences.     *
//*                                                                              *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_Ideogram ( aeSeq fg )
{
   const short grpSIZE = 6 ;
   const short loopCNT = grpSIZE + 2 ;
   const wchar_t *TestName[loopCNT] = 
   {
      L"Plain Text",
      L"aesIDEO_UNDER   - Underline",
      L"aesIDEO_UDOUBLE - Double Underline",
      L"aesIDEO_OVER    - Overline",
      L"aesIDEO_ODOUBLE - Double Overline",
      L"aesIDEO_STRESS  - Stress Marking",
      L"aesIDEO_OFF     - Ideogram Attributes Off",
      L"Plain Text",
   } ;
   const wchar_t *Desc = 
         L"Ideogram Test:\n"
          "-------------------------------------------------\n"
          "The ANSI standard (ANSI X3.64 (ISO/IEC6429))\n"
          "includes six(6) escape sequences for \"ideograms\".\n"
          "These were apparently intended to provide\n"
          "highlighting or separation of text blocks.\n"
          "\n"
          "These commands are a holdover from early versions\n"
          "of the standard, and it is unlikely that they\n"
          "are supported by terminal emulation software.\n" ;
   aeSeq aesGroup[loopCNT - 2] =
   {
      aesIDEO_UNDER, aesIDEO_UDOUBLE, aesIDEO_OVER, aesIDEO_ODOUBLE, 
      aesIDEO_STRESS, aesIDEO_OFF
   } ;
   const short baseROW = 4 ;     // base display row
   const short baseCOL = 2 ;     // base display column
   const short colOFF  = 50 ;    // offset for description text

   gString gsOut ;               // text formatting
   short   row = baseROW,        // output row
           idea = ZERO ;         // index into aesGroup

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* Write a description of the test *
   this->acWrite ( baseROW, colOFF, Desc ) ;
   this->acSetCursor ( aesCUR_ABSPOS, baseROW, baseCOL ) ;

   for ( short i = ZERO ; i < loopCNT ; ++i )
   {
      if ( (i >= 1) && (i < grpSIZE) )
         this->acSetIdeogram ( aesGroup[idea++] ) ;

      this->acWrite ( row, baseCOL, TestName[i] ) ;
      row += 2 ;
   }

   this->acSetCursor ( aesCUR_ABSPOS, row, 1 ) ; // prepare to exit
   this->acSetFg ( aesFG_DFLT ) ;      // Reset to terminal default foreground color
   this->acFlushStreams () ;           // Clear input and output buffers

}  //* End Test_Ideogram() *

//*************************
//*    Test_TermFlags     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest_Menu().                             *
//* Test tracking flags for:                                                     *
//*  1) tset->inpBuf  ---------------------- input buffering flag                *
//*  2) tset->inpEch  ---------------------- input echo option                   *
//*  3) tset->inpBlk  ---------------------- input blocking-read flag            *
//*  4) tset->cchType ---------------------- break key capture                   *
//*  5) tset->curType ---------------------- cursor type                         *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_TermFlags ( aeSeq fg )
{
   const short baseROW  = 7 ;       // base display row for flags tests
   const short baseCOL1 = 1 ;       // base column for display column 1
   const short baseCOL2 = 80 ;      // base column for display column 2
   const short baseROW2 = 16 ;      // base display row for CTRL+C test
   const short prOFF1   = 43 ;      // user-input prompt position 1 (column)
   const short prOFF2   = 43 ;      // user-input prompt position 2 (column)
   gString gsOut,                         // text formatting
           gsIn ;                         // capture user input
   TermIos tios ;                         // dummy object
   short row = baseROW, col = baseCOL1 ;  // output position

   //* Warn about minimum terminal window dimensions.*
   this->acWrite ( 1, 28, 
         L"  (requires terminal dimensions of 39 rows X 155 columns)\n"
          "---------------------------------------------------------" ) ; 

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* If break-signal captured, disable it *
   if ( this->tset->cchType != cchtTerm )
      this->acCaptureBreakSignal ( false ) ;
   //* Set cursor style to default *
   this->acSetCursorStyle ( csDflt ) ;

   //* Display title and column headings *
   gsOut = L" Validate Terminal Setup Flags \n"
            "-------------------------------" ;
   col = (this->termCols / 2) - (gsOut.gscols() / 4) ;
   this->acWrite ( (baseROW - 4), col, gsOut ) ;
   col = baseCOL1 ;

   this->acWrite ( (baseROW - 2), baseCOL1,
         L"           Input Buffering and Echo Options (dflt cursor shape)\n"
          "----------------------------------------------------------------------------\n" ) ;
   this->acWrite ( (baseROW - 2), baseCOL2,
         L"                            Cursor Shape Tests\n"
          "--------------------------------------------------------------------------\n" ) ;

   //********************************
   //********************************
   //**  Buffering and Echo Tests  **
   //********************************
   //********************************

   //* Display initial flag settings *
   gsOut = "Initial Settings:" ;
   this->acDumpTermInfo ( tios, row, col, 3, &gsOut ) ;

   this->tfPrompt ( row++, prOFF1, 1 ) ;     // user prompt

   //************************************************
   //* Set unbuffered input (terminal echo enabled) *
   //************************************************
   this->acBufferedInput ( false, termEcho ) ;

   //* Display the modified flags *
   gsOut = "Unbuffered, with term echo:" ;
   this->acDumpTermInfo ( tios, row += 4, col, 3, &gsOut ) ;

   this->tfPrompt ( row++, prOFF1, 1 ) ;     // user prompt

   //**********************************************
   //* Set unbuffered input (with soft echo, All) *
   //**********************************************
   this->acBufferedInput ( false, softEchoA ) ;

   //* Display the modified flags *
   gsOut = "Unbuffered, with soft echo (all):" ;
   this->acDumpTermInfo ( tios, row += 4, col, 3, &gsOut ) ;

   this->tfPrompt ( row++, prOFF1, 1 ) ;     // user prompt

   //*************************************************************
   //* Set unbuffered input, non-blocking, (with soft echo, All) *
   //*************************************************************
   this->acBufferedInput ( false, softEchoA, false ) ;

   //* Display the modified flags *
   gsOut = "Nonblocking, with soft echo (all):" ;
   this->acDumpTermInfo ( tios, row += 4, col, 3, &gsOut ) ;

   this->tfPrompt ( row++, prOFF1, 4 ) ;     // user prompt

   //***************************************
   //* Set unbuffered input (with no echo) *
   //***************************************
   this->acBufferedInput ( false, noEcho ) ;

   //* Display the modified flags *
   gsOut = "Unbuffered, with no echo:" ;
   this->acDumpTermInfo ( tios, row += 4, col, 3, &gsOut ) ;

   this->tfPrompt ( row++, prOFF1, 1 ) ;     // user prompt

   //*********************************************
   //* Restore buffered input with blocking read *
   //*********************************************
   this->acRestoreTermInfo () ;

   //* Display the modified flags *
   gsOut = "Restore Original Settings:" ;
   this->acDumpTermInfo ( tios, row += 4, col, 3, &gsOut ) ;

   //* Prompt user to test the restored settings *
   //* presumably buffered with terminal echo.   *
   this->tfPrompt ( row++, prOFF1, 1 ) ;

   //***************************
   //***************************
   //**  Cursor -Shape Tests  **
   //***************************
   //***************************

   //* Set unbuffered input with terminal echo *
   this->acBufferedInput ( false, termEcho ) ;

   row = baseROW ;                     // establish the second display column
   col = baseCOL2 ;

   //* Cursor - Blinking Block *
   this->acSetCursorStyle ( csBblock ) ;
   gsOut = "Cursor Shape: Blinking Block" ;
   this->acDumpTermInfo ( tios, row, col, 3, &gsOut ) ;
   this->tfPrompt ( row, (col + prOFF2), 2, L"Blinking Block       " ) ;

   //* Cursor - Default Cursor Shape *
   this->acSetCursorStyle ( csDflt ) ;
   gsOut = "Cursor Shape: Default" ;
   this->acDumpTermInfo ( tios, row, col, 3, &gsOut ) ;
   this->tfPrompt ( row, (col + prOFF2), 2, L"Default Cursor Shape " ) ;

   //* Cursor - Steady Block *
   this->acSetCursorStyle ( csSblock ) ;
   gsOut = "Cursor Shape: Steady Block" ;
   this->acDumpTermInfo ( tios, row, col, 3, &gsOut ) ;
   this->tfPrompt ( row, (col + prOFF2), 2, L"Steady Block         " ) ;

   //* Cursor - Blinking Underline *
   this->acSetCursorStyle ( csBuline ) ;
   gsOut = "Cursor Shape: Blinking Underline" ;
   this->acDumpTermInfo ( tios, row, col, 3, &gsOut ) ;
   this->tfPrompt ( row, (col + prOFF2), 2, L"Blinking Underline   " ) ;

   //* Cursor - Steady Underline *
   this->acSetCursorStyle ( csSuline ) ;
   gsOut = "Cursor Shape: Steady Underline" ;
   this->acDumpTermInfo ( tios, row, col, 3, &gsOut ) ;
   this->tfPrompt ( row, (col + prOFF2), 2, L"Steady Underline     " ) ;

   //* Cursor - Blinking Vertical Bar *
   this->acSetCursorStyle ( csBvbar ) ;
   gsOut = "Cursor Shape: Blinking Vertical" ;
   this->acDumpTermInfo ( tios, row, col, 3, &gsOut ) ;
   this->tfPrompt ( row, (col + prOFF2), 2, L"Blinking Vertical Bar" ) ;

   //* Cursor - Steady Vertical Bar *
   this->acSetCursorStyle ( csSvbar ) ;
   gsOut = "Cursor Shape: Steady Vertical" ;
   this->acDumpTermInfo ( tios, row, col, 3, &gsOut ) ;
   this->tfPrompt ( row, (col + prOFF2), 2, L"Steady Vertical Bar  " ) ;

   //* Return to default cursor style *
   this->acSetCursorStyle ( csDflt ) ;


   //***********************************
   //***********************************
   //**  CTRL+C Signal Capture Tests  **
   //***********************************
   //***********************************

   //* Display the column heading *
   row = baseROW2 ;
   col = baseCOL2 ;
   this->acWrite ( (baseROW2 - 2), baseCOL2,
    L"                       CTRL+C (break signal) Capture\n"
     "--------------------------------------------------------------------------\n" ) ;

   //* This test uses unbuffered input, with  *
   //* soft-echo and "special" keys disabled. *
   this->acBufferedInput ( false, softEchoD ) ;

   //* Capture break signal using default *
   //* handler breakSignalHandler(), and  *
   //* default handler type: Ignore ('i'),*
   //* (with audible alert).              *
   //* Display the terminal settings, then*
   //* prompt user to press CTRL+C.       *
   this->acCaptureBreakSignal ( true, cchtIgnore, true ) ;

   gsOut = "Ctrl+C 'Ignore'" ;
   this->acDumpTermInfo ( tios, row, col, 3, &gsOut ) ;
   this->tfPrompt ( row++, col + prOFF2, 3, L"Signal Handler Active\n"
                                           "Ctrl+C : \"Ignored\"," ) ;

   //* Change handler type to User ('u'). *
   //* Display the terminal settings, then*
   //* prompt user to press CTRL+C.       *
   this->acCaptureBreakSignal ( true, cchtUser ) ;
   gsOut = "Ctrl+C 'User Prompt'" ;
   this->acDumpTermInfo ( tios, row += 4, col, 3, &gsOut ) ;
   this->tfPrompt ( row++, col + prOFF2, 3, L"Signal Handler Active\n"
                                           "Ctrl+C : \"User Prompt\"," ) ;

   //* Change handler type to Exit ('e'). *
   //* Display the terminal settings, then*
   //* prompt user to press CTRL+C.       *
   this->acCaptureBreakSignal ( true, cchtExit ) ;
   gsOut = "Ctrl+C 'Exit'" ;
   this->acDumpTermInfo ( tios, row += 4, col, 3, &gsOut ) ;
   this->tfPrompt ( row++, col + prOFF2, 3, L"Signal Handler Active\n"
                                           "Ctrl+C : \"Exit\"," ) ;

   //* If user did not exit the application via CTRL+C, *
   //* or (gasp!), there was a program bug, say bye-bye.*
   this->acWrite ( (row + 4), baseCOL2, 
                  L"Ctrl+C Exit failed, OR Ctrl+C not seen.\n"
                   "Press Any Key To Exit..." ) ;
   this->acRead () ;

   //* In case the user did not exit the application *
   //* using the CTRL+C signal, prepare to exit.     *
   this->acBufferedInput ( true, termEcho ) ;   // Re-enable buffered input
   this->acSetCursor ( aesCUR_ABSPOS, (this->termRows - 2), 1 ) ; // Set cursor position
   this->acCaptureBreakSignal ( false ) ; // disable break-signal capture
   this->acSetFg ( aesFG_DFLT ) ;      // Reset to terminal default foreground color
   this->acFlushStreams () ;           // Clear input and output buffers

}  //* End Test_TermFlags() *

//*************************
//*       tfPrompt        *
//*************************
//********************************************************************************
//* Called by Test_Flags() method.                                               *
//* Prompt the user for some key input, capture it and display the captured      *
//* data.                                                                        *
//*                                                                              *
//* Note: If one of the soft-echo options is active, only printing characters    *
//*       (and a select few control characters) are returned from acRead()       *
//*       unmodified.                                                            *
//*                                                                              *
//* Input  : baseRow : base row for prompt                                       *
//*          baseCol : base column for prompt                                    *
//*          prompt  : specifies which prompt to use                             *
//*          p2      : (optional, null pointer by default)                       *
//*                    When 'prompt' arg == 2, this is the supplimentary         *
//*                    text for the user prompt                                  *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::tfPrompt ( short baseRow, short baseCol, short prompt, const wchar_t *p2 )
{
   const wchar_t *userPrompt1 = L"Write some text, then press Enter:\n"
                                "->" ;
   const wchar_t *userPrompt2 = L"%S\n"
                                 "Write some text,\n"
                                 "then press Enter..." ;
   const wchar_t *userPrompt3 = L"%S\n"
                                 "Press Ctrl+C: " ;
   const short promptRESP1 = 2 ;       // position for echo of captured input
   gString gsOut,                      // text formatting
           gsIn ;                      // capture user responses

   //* First series of tests: buffering states and echo options.  *
   //* Note: Break key is under terminal control for these tests, *
   //*       so Ctrl+C will kill the application.                 *
   if ( prompt == 1 )
   {
      //* Display the prompt *
      this->acWrite ( baseRow++, baseCol, userPrompt1 ) ;

      //* Get user response *
      this->acRead ( gsIn )  ;

      //* Dispay the captured data *
      this->acWrite ( ++baseRow, (baseCol + promptRESP1), gsIn ) ;
      this->acSetCursor ( aesCUR_NEXTROW ) ;
   }  // (prompt==1)

   //* Second set of tests: Cursor shape *
   else if ( prompt == 2 )
   {
      const short prOFF2   = 19 ;      // user-input prompt position
      gString gsIn ;                   // receives user input
      gsOut.compose( userPrompt2, p2 ) ;
      this->acWrite ( baseRow, baseCol, gsOut ) ;
      this->acRead ( gsIn, 15 ) ;
      this->acSetCursor ( aesCUR_ABSPOS, (baseRow + 2), (baseCol + prOFF2) ) ;
      this->acEraseArea ( aesERASE_EOL ) ;
   }  // (prompt==2)

   //* Third set of tests: Ctrl+C signal capture.*
   else if ( prompt == 3 )
   {
      gsOut.compose ( userPrompt3, p2 ) ;
      this->acWrite ( baseRow++, baseCol, gsOut ) ;

      //* Get user response *
      this->acRead ( gsIn )  ;

   }  // (prompt==3)
   //* Same as prompt #1, except with non-blocking read *
   else if ( prompt == 4 )
   {
      //* Display the prompt *
      this->acWrite ( baseRow++, baseCol, userPrompt1 ) ;

      this->acFlushIn () ;          // clear the input buffer

      //* Get user response: (Yes, this is complicated, but it's fun.)  *
      //* Call acReadNB() in a loop, until Enter keycode is received.   *
      //* The printing characters are echoed to the terminal by the     *
      //* soft-echo algorithm, while the encoded "special" keycodes are *
      //* returned to us for local processing.                          *
      //* -- This loop actually prints the special keys as verification *
      //*    that they were properly captured and encoded.              *
      //* -- When the non-blocking read returns without capturing a     *
      //*    keycode, we print a full-stop ('.') to indicate that we    *
      //*    are waiting for user input. Then, when user key input is   *
      //*    received, we backspace over the '.' characters and echo    *
      //*    the captured character to the terminal.                    *
      //*                          Fun!!                                *
      gString gstmp ;
      wchar_t wchar ;
      short   points = ZERO ;
      gsIn.clear() ;
      gsOut.clear() ;

      do
      {
         //* Read and echo printing characters (special keys not echoed).*
         if ( (this->acReadNB ( wchar, 10, 100, false )) )
         {
            while ( points-- > ZERO )
               this->ttyWrite ( "\b \b", false ) ;
            points = ZERO ;
            if ( (isprint ( wchar, *this->ioLocale )) )
            {
               if ( !(this->acSpecialKeyTest ( wchar )) )
                  this->ttyWrite ( "\b \b", false ) ; // erase the echoed character
               this->ttyWrite ( wchar ) ;
               gsIn.append( "%C", &wchar ) ;
            }
            else
            { /* Do Nothing */ }
         }
         else
         {
            this->ttyWrite ( L'.' ) ;
            ++points ;
         }
      }
      while ( wchar != NEWLINE ) ;

      //* Dispay the captured data *
      this->acSetCursor ( ++baseRow, (baseCol + promptRESP1) ) ;
      this->ttyWrite ( gsIn ) ;
      this->acSetCursor ( aesCUR_NEXTROW ) ;
   }  // (prompt==4)

}  //* End tfPrompt() *

//*************************
//*     Test_ModFlags     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Test text modifiers and the associated tracking flags:                       *
//*  1) intenseMod   : Bold and Faint                                            *
//*  2) italicMod    : Italic and Fractur                                        *
//*  3) ulineMod     : Underline                                                 *
//*  4) blinkMod     : Blink-slow and Blink-fast                                 *
//*  5) reverseMod   : Reversed fg/bg                                            *
//*  6) concealMod   : Concealed (invisible) text                                *
//*  7) xoutMod      : Xout (crossed-out) text                                   *
//*  8) frameMod     : Framed text and Encircled text                            *
//*  9) olineMod     : Overlined text                                            *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_ModFlags ( aeSeq fg )
{
  const wchar_t *Desc = 
  L"Text-attribute Modifiers Test:\n"
   "---------------------------------------------------------------------------\n"
   "In addition to setting the foreground/background color attributes for text,\n"
   "several other text modifiers are available to draw the user's attention.\n"
   "This test exercises each of the text modifiers defined by the ANSI standard\n"
   "(ANSI X3.64 (ISO/IEC6429)). Support for these modifiers is system dependent,\n"
   "but Gnometerm, Konsole and XTerm all support the common options.\n"
   "The internal tracking flags for these modifiers are also displayed.\n"
   "\n"
   "A few of the options in the original ANSI specification, such as \"Framed\"\n"
   "and \"Encircled\" are seldom if ever supported by modern terminal programs.\n" ;

   const short baseROW = 1 ;
   const short colOFF  = 28 ;
   const short descROW = 27 ;
   const short descCOL = 78 ;
   short row = 4,                         // display row
         col = 1,                         // display column
         drow = 3,                        // flag-report row
         dcol = 30 ;                      // flag-report column

   //* Warn about minimum terminal window dimensions.*
   this->acWrite ( baseROW, colOFF, 
                  "  (requires terminal dimensions of 39 rows X 155 columns)\n"
                  "---------------------------------------------------------" ) ;

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* Write a description of the test *
   this->acWrite ( descROW, descCOL, Desc ) ;
   this->acSetCursor ( aesCUR_ABSPOS, baseROW, 1 ) ;

   //* Reset all mods to default values *
   this->acResetMods () ;
   this->acWrite ( row++, col, L"0) Plain Text" ) ;
   this->acDumpMods ( drow, dcol, L"0)Reset All" ) ; dcol += 16 ;

   //* Test Bold *
   this->acSetMod ( aesBOLD ) ;
   this->acWrite ( row++, col, L"1) Bold Text" ) ;
   this->acDumpMods ( drow, dcol, L"1)Bold On" ) ; dcol += 16 ;

   //* Add Underline to the Bold *
   this->acSetMod ( aesUNDERLINE ) ;
   this->acWrite ( row, col, L"2) Bold, Underlined Text" ) ;   row += 2 ;
   this->acDumpMods ( drow, dcol, L"2)UnderlineOn" ) ; dcol += 16 ;

   //* Clear Bold and Underline *
   this->acSetMod ( aesBOLD_OFF ) ;
   this->acSetMod ( aesUNDERLINE_OFF ) ;
   this->acWrite ( row++, col, L"3) Plain Text" ) ;
   this->acDumpMods ( drow, dcol, L"3)Plain Text" ) ; dcol += 16 ;

   //* Faint text *
   this->acSetMod ( aesFAINT ) ;
   this->acWrite ( row++, col, L"4) Faint Text" ) ;
   this->acDumpMods ( drow, dcol, L"4)Faint Text" ) ; dcol += 16 ;

   //* Back to plain text *
   this->acSetMod ( aesBOLD_OFF ) ;
   this->acWrite ( row++, col, L"5) Plain Text" ) ;
   this->acDumpMods ( drow, dcol, L"5)Plain Text" ) ; dcol += 16 ;

   //* Overline on *
   this->acSetMod ( aesOVERLINE ) ;
   this->acWrite ( row, col, L"6) Overline On" ) ;    row += 2 ;
   this->acDumpMods ( drow, dcol, L"6)Overline On" ) ; dcol += 16 ;

   //* Add slow-blinking to overline
   this->acSetMod ( aesBLINK_SLOW ) ;
   this->acWrite ( row, col, L"7) Slow Blink" ) ;     row += 2 ;
   this->acDumpMods ( drow, dcol, L"7)Slow Blink" ) ; dcol += 16 ;

   drow += 12 ; dcol = 30 ;               // next report row

   //* Fast blink *
   this->acSetMod ( aesBLINK_FAST ) ;
   this->acWrite ( row, col, L"8) Fast Blink" ) ;     row += 2 ;
   this->acDumpMods ( drow, dcol, L"8)Fast Blink" ) ; dcol += 16 ;

   //* Reset blink and Overline *
   this->acSetMod ( aesBLINK_OFF ) ;
   this->acSetMod ( aesOVERLINE_OFF ) ;
   this->acDumpMods ( drow, dcol, L"9)Plain Text" ) ; dcol += 16 ;
   this->acWrite ( row, col, L"9) Plain Text" ) ;     row += 2 ;

   //* Set Italic text *
   this->acSetMod ( aesITALIC ) ;
   this->acDumpMods ( drow, dcol, L"10)Italic" ) ; dcol += 16 ;
   this->acWrite ( row, col, L"10)Italic Text" ) ;    row += 2 ;

   //* Add Reverse fg/bg to the Italic *
   this->acSetMod ( aesREVERSE ) ;
   this->acDumpMods ( drow, dcol, L"11)Reverse" ) ; dcol += 16 ;
   this->acWrite ( row, col, L"11) Italic/Reversed Text" ) ;   row += 2 ;

   //* Reset Italic and Reversed fg/bg, set X-out (strikethrough) *
   this->acSetMod ( aesITALIC_OFF ) ;
   this->acSetMod ( aesREVERSE_OFF ) ;
   this->acSetMod ( aesXOUT ) ;
   this->acDumpMods ( drow, dcol, L"12)Xout" ) ; dcol += 16 ;
   this->acWrite ( row++, col, L"12) Xout Text" ) ;

   //* Point to concealed text, then conceal the text *
   this->acSetMod ( aesXOUT_OFF ) ;
   this->acWrite ( row++, col, L"| Concealed Text Below |" ) ;
   this->acSetMod ( aesCONCEAL ) ;
   this->acDumpMods ( drow, dcol, L"13)Conceal" ) ; dcol += 16 ;
   this->acWrite ( row++, col, L"13) Concealed Text" ) ;

   //* Reset concealed text *
   this->acSetMod ( aesCONCEAL_OFF ) ;
   this->acDumpMods ( drow, dcol, L"14)Plain Text" ) ; dcol += 16 ;
   this->acWrite ( row++, col, L"14) Plain Text" ) ;

   //* Test "Fractur" (italicMod). (In practice, this is rendered as italic.)*
   this->acSetMod ( aesFRACTUR ) ;
   this->acDumpMods ( drow, dcol, L"15)Fractur" ) ; dcol += 16 ;
   this->acWrite ( row++, col, L"15) Fractur Text" ) ;

   drow += 12 ; dcol = 30 ;               // next report row

   //* Reset Fractur (italicMod), then: test "Framed". (frameMod) *
   this->acSetMod ( aesITALIC_OFF ) ;
   this->acSetMod ( aesFRAMED ) ;
   this->acDumpMods ( drow, dcol, L"16)Framed" ) ; dcol += 16 ;
   this->acWrite ( row++, col, L"16) Framed Text" ) ;

   //* Test "Encircle". (frameMod flag) *
   this->acSetMod ( aesFRM_ENC_OFF ) ;
   this->acSetMod ( aesENCIRCLE ) ;
   this->acWrite ( row, col, L"17) Encircled Text" ) ;  row += 2 ;
   this->acDumpMods ( drow, dcol, L"17)Encircle" ) ; dcol += 16 ;

   //* Return all mods to default values *
   this->acResetMods () ;
   this->acWrite ( row, col, L"18) Plain Text - Exit" ) ;
   this->acDumpMods ( drow, dcol, L"18)Reset All" ) ;

   //* Prepare to exit *
   this->acSetCursor ( aesCUR_ABSPOS, (this->termRows - 1), 1 ) ;
   this->acSetFg ( aesFG_DFLT ) ;

}  //* End Test_ModFlags() *

//*************************
//*     Test_NonBlock     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Non-breaking read from stdin:                                                *
//* =============================                                                *
//*  Our testing is done with input buffering disabled via terminal settings     *
//*  and non-blocking input stream set through the (virtual) file descriptor.    *
//*                                                                              *
//* 1) Library calls cannot directly be made non-blocking, so first, the         *
//*    file descriptor (actually virtual file descriptor) must be set as         *
//*    non-blocking.                                                             *
//*                                                                              *
//* 2) When the input stream is set for non-blocking read, the std::wcin.peek()  *
//*    method is able to query the input stream.                                 *
//*    The documentation says that the return value is:                          *
//*       If good() == true, peek() returns the next character in the stream.    *
//*       Otherwise, it returns Traits::eof(). (WEOF==-1)                        *
//*                                                                              *
//* 3) Although a low-level C function, poll() can be used to query the stream,  *
//*    we prefer to remain (as much as possible) in the land of C++ because      *
//*    the dinosaurs and UNIX programmers in C-land make us nervous.             *
//*    Unfortunately, we need to dip our toe in the C-esspool for the            *
//*    fcntl() function in order to set or reset the non-blocking read flag.     *
//*                                                                              *
//* 4) fcntl() function:                                                         *
//*    int fd = 0 ;     // zero (0) identifies the 'stdin' input stream          *
//*    fcntl(fd, F_GETFL, 0)                                                     *
//*    F_GETFL  This macro requests the current flag settings: The file          *
//*             access mode and the status flags are returned.                   *
//*             (Note that the third argument of the call is    )                *
//*             (ignored, but we set it to zero just to be safe.)                *
//*                                                                              *
//*    F_SETFL  This macro sets the flags for the specified stream:              *
//*       From the man page:                                                     *
//*       Set  the file status flags to the value specified by arg.              *
//*       File access mode (O_RDONLY, O_WRONLY, O_RDWR) and file creation        *
//*       flags (i.e., O_CREAT, O_EXCL, O_NOCTTY, O_TRUNC) in arg are ignored.   *
//*       On Linux, this command can change only the O_APPEND, O_ASYNC,          *
//*       O_DIRECT, O_NOAT‐IME, and O_NONBLOCK flags.                            *
//*       It is not possible to change the O_DSYNC and O_SYNC flags.             *
//*                                                                              *
//*       The O_NONBLOCK flag is the flag of interest here.                      *
//*       #include <fcntl.h>                                                     *
//*       int fd = 0 ;                            // 0 == stdin stream           *
//*       int flags = fcntl(fd, F_GETFL, 0);      // get current flags           *
//*       fcntl(fd, F_SETFL, flags | O_NONBLOCK); // set the O_NONBLOCK flag     *
//*                                                                              *
//********************************************************************************

void AnsiCmd::Test_NonBlock ( aeSeq fg )
{
   #define CALL_NONBLK (1) // Select between direct access and call to AnsiCmd method
   #if CALL_NONBLK != 0
   uint16_t msDelay = 100,                // delay in mSec for acReadNB(): 0.1 sec
            rptCnt = 8 ;                  // polling repeat count for acReadNB()
   bool     status ;                      // return value from acReadNB()
   #else // (CALL_NONBLK==0)
   short    tsec = 5 ;                    // sleep interval (tenths of a second)
   wchar_t  pv2 ;                         // peek-value 2
   #endif   // CALL_NONBLK

   const short baseROW = 4 ;              // first output row
   gString gsOut ;                        // text formatting
   WinPos wp( baseROW, 1 ) ;               // initial output position
   uint32_t pval ;                        // returned value from peek()
   wchar_t wch ;                          // returned value from wcin()
   uint32_t iost ;                       // ios_base::iostate (istream flags)
   bool good, fail, bad, eof ;             // input-stream status flags

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* User Prompt *
   wp = this->acWrite ( wp, 
         L"Test non-blocking read of input stream, 'stdin'.\n"
          "Type a few characters, then wait a moment and type a few more.\n"
          "The value returned by peek() reports whether data are available\n"
          "and if so, get() captures the data.  " ) ;
   this->acSetMod ( aesUNDERLINE ) ;
   this->acSetMod ( aesBOLD ) ;
   this->ttyWrite ( L"Press ENTER to exit:\n" ) ;
   this->acSetMod ( aesBOLD_OFF ) ;
   this->acSetMod ( aesUNDERLINE_OFF ) ;

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
   if ( (ofsdbg.is_open()) )
   {
      gsdbg.compose( "Test_Nonblock1 ( inpBuf:%hhd inpEch:%hd inpBlk:%hhd cchType:%hd )", 
                     &this->tset->inpBuf, &this->tset->inpEch, 
                     &this->tset->inpBlk, &this->tset->cchType ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG

   //* Set stdin stream for non-blocking read  *
   //* (As a side-effect, input buffering is ) *
   //* (disabled, and echo is set to 'noEcho') *
   this->acBlockingInput ( false ) ;

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
   if ( (ofsdbg.is_open()) )
   {
      gsdbg.compose( "Test_Nonblock2 ( inpBuf:%hhd inpEch:%hd inpBlk:%hhd cchType:%hd )", 
                     &this->tset->inpBuf, &this->tset->inpEch, 
                     &this->tset->inpBlk, &this->tset->cchType ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG

   do
   {
      #if CALL_NONBLK != 0              // Call AnsiCmd method
      //**************************
      //* Query the input stream *
      //**************************
      //* Note: 'silent' parameter for this call is set to 'false'. *
      //* This is done so that any "special" keys will be decoded.  *
      //* However, the 'noEcho' option set above overrides, so the  *
      //* call will not write to display.                           *
      status = this->acReadNB ( wch, rptCnt, msDelay, false, &iost ) ;
      good = bool(iost & std::wistream::goodbit) ;
      fail = bool(iost & std::wistream::failbit) ;
      bad  = bool(iost & std::wistream::badbit) ;
      eof  = bool(iost & std::wistream::eofbit) ;

      //* Don't print non-printing characters *
      if ( (isprint ( wch, *this->ioLocale )) ) pval = wch ;
      else                                      pval = L'-' ; 
      gsOut.compose( "[%04X '%C' g:%hhd,f:%hhd,b:%hhd,e:%hhd]\n", 
                    &wch, &pval, &good, &fail, &bad, &eof ) ;
      this->ttyWrite ( gsOut ) ;

      if ( status != false )               // if valid character data
      {
         if ( (isprint ( wch, *this->ioLocale )) )
            gsOut.compose( "[%04X] '%C'\n", &wch, &wch ) ;
         else
            gsOut.compose( "[%04X] '-'\n", &wch ) ;
         this->ttyWrite ( gsOut ) ;
         this->nsleep ( 1 ) ;             // pause for 1/10 second
      }

      // ------------------------------------------------------------------
      // ------------------------------------------------------------------
      #else                             // Call library methods directly
      this->nsleep ( tsec ) ;             // wait a moment for input

      //**************************
      //* Query the input stream *
      //**************************
      if ( this->wideStream )             // wide-stream input
      {
         pval = std::wcin.peek () ;
         iost = std::wcin.rdstate() ;
         good = bool(iost & std::wistream::goodbit) ;
         fail = bool(iost & std::wistream::failbit) ;
         bad  = bool(iost & std::wistream::badbit) ;
         eof  = bool(iost & std::wistream::eofbit) ;
      }
      else                              // narrow-stream input
      {
         pval = std::cin.peek () ;
         good = bool(iost & std::istream::goodbit) ;
         fail = bool(iost & std::istream::failbit) ;
         bad  = bool(iost & std::istream::badbit) ;
         eof  = bool(iost & std::istream::eofbit) ;
      }

      //* Do not echo the WEOF or control characters *
      // Programmer's Note: The "special" keys are not decoded in this 
      // direct-access test, so partial escape sequences will appear as 
      // garbage in the output. See how the asReadNB() method is used in the 
      // production-worthy test above to eliminate this garbage data.
      if ( (isprint ( wch, *this->ioLocale )) && (pval != L'\n') ) pv2 = pval ;
      else                                      pv2 = L'-' ;
      gsOut.compose( "[%04X '%C' g:%hhd,f:%hhd,b:%hhd,e:%hhd]\n", 
                    &pval,  &pv2, &good, &fail, &bad, &eof ) ;
      this->ttyWrite ( gsOut ) ;
      tsec = eof ? 5 : 1 ;                // adjust pause interval

      //*****************************************
      //* If data available, get next character *
      //* from the input stream.                *
      //*****************************************
      if ( pval != WEOF )
      {
         if ( this->wideStream )             // wide-stream input
            wch = std::wcin.get() ;
         else                              // narrow-stream input
            wch = wchar_t(std::cin.get()) ;
         if ( (isprint ( wch, *this->ioLocale )) )
            gsOut.compose( "[%04X] '%C'\n", &wch, &wch ) ;
         else
            gsOut.compose( "[%04X] '-'\n", &wch, &wch ) ;
         this->ttyWrite ( gsOut ) ;
         this->nsleep ( 1 ) ;             // pause for 1/10 second
      }
      //* If peek() returned an error i.e. no data available, *
      //* clear the error code and continue.                  *
      else
      {
         if ( this->wideStream )           // wide-stream input
            std::wcin.clear() ;
         else                              // narrow-stream input
            std::cin.clear() ;
      }
      #endif   // CALL_NONBLK
   }
   while ( wch != NEWLINE ) ;

   //* Say Good Night, Gracie *
   this->ttyWrite ( L"----- Bye-bye! -----\n\n" ) ;

   //* Return terminal to buffered input with   *
   //* terminal echo AND blocking read of input.*
   this->acBufferedInput ( true, termEcho ) ;

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
   if ( (ofsdbg.is_open()) )
   {
      gsdbg.compose( "Test_Nonblock3 ( inpBuf:%hhd inpEch:%hd inpBlk:%hhd cchType:%hd )", 
                     &this->tset->inpBuf, &this->tset->inpEch, 
                     &this->tset->inpBlk, &this->tset->cchType ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG

   //* Restore foreground to default attribute *
   this->acSetFg ( aesFG_DFLT ) ;

}  //* End Test_NonBlock() *

//*************************
//*     Test_SoftEcho     *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//*                                                                              *
//* Input  : eOpt: Input-echo option for unbuffered input.                       *
//*                'a' All special keys translations active.                     *
//*                'b' Backspace and other edit keys translation active.         *
//*                'c' Cursor-movement keys translation active.                  *
//*                'd' Disable special-key translation.                          *
//*                't' Terminal echo                                             *
//*                'n' No echo                                                   *
//*                'r' Report                                                    *
//*          fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note: For tests which use direct read of the input stream, the  *
//* stream is explicitly set to unbuffered input for the specified test.         *
//* For tests which gather input through the ACWin user interface, buffering     *
//* configuration is handled automatically.                                      *
//********************************************************************************

void AnsiCmd::Test_SoftEcho ( wchar_t eOpt, aeSeq fg )
{
   const short baseROW = 7 ;              // base output row
   const short baseCOL = 3 ;              // base output column
   const short exitROW = baseROW + 19 ;   // target row for exit
   const short instROW = 4 ;              // position for instructions text
   const short instCOL = 3 ;
   const short dumpCOL = 78 ;
   const short dumpOPT = 4 ;              // verbose term-setup dump
   const short colOFF  = 32 ;             // offset for title extension
   gString gsOut ;                        // text formatting
   WinPos wp,                             // cursor position
          wpDump ;                        // Position for term-info dump
   wchar_t wch ;

   //* Warn about minimum terminal window dimensions.*
   this->acWrite ( 1, colOFF, 
                  L"  (requires terminal dimensions of 36 rows X 155 columns)\n"
                   "---------------------------------------------------------" ) ; 

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   //* Set terminal configuration for unbuffered input and *
   //* specified soft echo, term-echo or no-echo option.   *
   EchoOpt ech ;           // echo option from caller
   switch ( eOpt )
   {
      case 't': ech = termEcho ;  break ; // Terminal echo of key input
      case 'n': ech = noEcho ;    break ; // No echo of key input
      case 'c': ech = softEchoC ; break ; // Cursor group of special keys
      case 'e': ech = softEchoE ; break ; // Edit group of special keys: Bksp/Del/Ins
      case 'd': ech = softEchoD ; break ; // Disable (discard) special keys
      case 'a':                           // All special keys exercised
      case 'r':                           // Report all special keys
         ech = softEchoA ; break ;
   } ;

   //* Display the names of the "special" keys *
   gsOut = L"Special Keys: LeftArrow, RightArrow, UpArrow, DownArrow, Home, End,\n"
            "PageUp, PageDown, Tab, Shift+Tab, Alt+Enter, Backspace, Delete, Insert.\n"
            "-----------------------------------------------------------------------\n" ;
   wpDump = this->acWrite ( instROW, dumpCOL, gsOut ) ;
   
   //* Get terminal settings and report them *
   gsOut = L"Term Setup for Input Echo" ;
   TermIos tios ;
   this->acGetTermInfo ( stdIn, tios ) ;
   this->acDumpTermInfo ( tios, wpDump.row, wpDump.col, dumpOPT, &gsOut ) ;

   gsOut.compose( "Key Input Echo Option: %S\n",
                 (ech == softEchoA) ? L"softEchoA (All Special keys translated)" :
                 (ech == softEchoC) ? L"softEchoC (Cursor keys translated)" :
                 (ech == softEchoD) ? L"softEchoD (Special Key translation disabled)" :
                 (ech == softEchoE) ? L"softEchoE (Text Edit keys translated)" : 
                 (ech == termEcho)  ? L"Terminal Echo of Key Input\n"
                                       "(\"special\" keys will display as escape sequences)" : 
                 (ech == noEcho)    ? L"No Echo of Key Input\n"
                                       "(a beep indicates that keycode was received)"
                                    : L"Error!" ) ;
   gsOut.append( L"       Press Enter key to exit." ) ;
   this->acWrite ( instROW, instCOL, gsOut ) ;

   //********************************************************
   //* softEchoA: This test uses an "invisible" ACWin window*
   //* with an embedded skForm object for these test.       *
   //* define the skForm used for testing.                  *
   //* ACWin objects use the softEchoA option, but certain  *
   //* keycodes are interpreted according to the specific   *
   //* type of field under edit.                            *
   //********************************************************
   if ( (ech == softEchoA) && (eOpt != L'r') )
   {
      const short winROW = 3 ;
      const short winCOL = 2 ;
      const short winHEIGHT  = 15 ;       // window dimensions
      const short winWIDTH   = 71 ;
      const short instCOL = 3 ;
      const short rptROW = 18 ;           // position for stats report
      const short rptCOL = instCOL ;

      short f = ZERO ;                    // field index for initialization
      WinPos wpul( 4, 1 ),                // initialize field origin
             wpRpt( rptROW, rptCOL ),     // position stats report
             wpIns( 2, 63 ) ;             // offset of ins/ovr report
       bool insFlag = true ;              // track current state of ins/ovr flag

      //* Construct an skForm-class object and initialize it.     *
      //* The initialized object is passed as a parameter to the  *
      //* ACWin-class constructor.                                *
      //* -- The window border and interior both use the terminal *
      //*    default color attributes.                            *
      //* -- The window border is invisible to give the impression*
      //*    that the fields stand alone in the terminal window,  *
      //*    BUT they do not.                                     *
      //* -- Note that one of the fields is initialized with      *
      //*    multi-column chars to stress test cursor movements.  *
      skForm *skfPtr = new skForm( 4, 0, 0 ) ;

      //* Initialize the form-level members. Note that 'fCnt' is  *
      //* initialized during instantiation, and that 'base' value *
      //* is overridden by the ACWin initialization.              *
      skfPtr->ins  = insFlag ;
      skfPtr->togg = true ;
      skfPtr->beep = true ;
      skfPtr->fi   = ZERO ;

      //* Initialize the field-level members. Note that the 'ip'  *
      //* determines the value of the 'cur' member'. 'aca', 'off' *
      //* and 'ro' all use their default values.                  *
      skfPtr->fld[f].orig = wpul ;           // field index 0
      skfPtr->fld[f].hgt  = 1 ;
      skfPtr->fld[f].wid  = winWIDTH - 4 ;
      skfPtr->fld[f].ip = ZERO ;
      skfPtr->fld[f++].txt = 
               "Use the Arrow keys and Home/End keys to navigate within a field." ;
      wpul.row += 2 ;            // offset of next field

      skfPtr->fld[f].orig = wpul ;           // field index 1
      skfPtr->fld[f].hgt  = 1 ;
      skfPtr->fld[f].wid  = winWIDTH - 4 ;
      skfPtr->fld[f].ip = 24 ;
      skfPtr->fld[f++].txt = 
               "Use the Backspace, .....x..... Delete and Insert Keys to edit text." ;
      wpul.row += 2 ;            // offset of next field

      skfPtr->fld[f].orig = wpul ;           // field index 2
      skfPtr->fld[f].hgt  = 3 ;
      skfPtr->fld[f].wid  = winWIDTH - 4 ;
      skfPtr->fld[f].ip = 36 ;
      skfPtr->fld[f++].txt = 
               "Use the Tab and Shift+Tab keys .....x..... to move among fields.\n"
               "Note: Within the edit method, PgDown/DownArrow and PgUp/UpArrow\n"
               "are automatically converted to Tab and Shift+Tab, respectively." ;
      wpul.row += 4 ;            // offset of next field

      skfPtr->fld[f].orig = wpul ;           // field index 3
      skfPtr->fld[f].hgt  = 1 ;
      skfPtr->fld[f].wid  = winWIDTH - 4 ;
      skfPtr->fld[f].ip = 13 ;
      skfPtr->fld[f].txt = "道是生命的本质　。。。。。三。。。。。　还是巧克力？" ;

      //* For testing only: optionally make border visible and change     *
      //* colors. For production, border is invisible with default colors.*
      wchar_t border = L'i', color = L'd' ;

      ACWin *winPtr = new ACWin
            ( winROW, winCOL,                      // position (upper-left corner) 
              winHEIGHT, winWIDTH,                 // dimensions (height, width)
              (border=='d' ? ltDUAL : border=='s' ? ltSINGLE : ltHORIZ),
              (color=='d' ? acAttr(acaFG_DFLT | acaBG_DFLT) : acAttr(acaFGb_BROWN | acaBG_BLUE)),
              acAttr(acaFG_DFLT | acaBG_DFLT),     // interior attributes
              NULL,
              skfPtr
            ) ;

      //* Note: Opening the window disables input buffering AND *
      //*       sets the echo option to 'softEchoA'.            *
      winPtr->OpenWindow(
         "   Key Input Echo Option: softEchoA - (within an skForm object)\n"
         "   (special keys interpreted according to the field under edit)\n"
         "       Press Enter To Exit                                     <INS>\n\n"
         "[                                                                   ]\n\n"
         "[                                                                   ]\n"
         "┌                                                                   ┐\n"
         "│                                                                   │\n"
         "│                                                                   │\n"
         "│                                                                   │\n"
         "└                                                                   ┘\n"
         "[                                                                   ]" ) ;

      //* Delete the local skForm object and reference   *
      //* the fully-initialized object within the window.*
      delete skfPtr ;                  // delete the local skForm object
      wchar_t wkey ;                   // user input
      short fIndx = ZERO ;             // index of field with focus
      const skForm *skfp = winPtr->GetForm () ;

      //* Update Insert/Overstrike report *
      winPtr->Write ( wpIns, (skfp->ins ? L"<INS>" : L"<OVR>"), 
                      (skfp->ins ? acAttr(acaFG_GREEN | acaBG_DFLT) : 
                                   acAttr(acaFG_BLUE  | acaBG_DFLT)) ) ;

      //* Report the data members of each field *
      this->tseReport ( *skfp, wpRpt ) ;

      //* Capture user input until a newline ('\n') character is received.*
      //* Note that the call includes the 'allChar' flag set so we can    *
      //* report the skForm/skField stats in real-time.                   *
      while ( (wkey = winPtr->EditField ( fIndx, NEWLINE, true )) != NEWLINE )
      {
         //* Update Insert/Overstrike report *
         if ( skfp->ins != insFlag )
         {
            winPtr->Write ( wpIns, (skfp->ins ? L"<INS>" : L"<OVR>"), 
                            (skfp->ins ? acAttr(acaFG_GREEN | acaBG_DFLT) : 
                                         acAttr(acaFG_BLUE  | acaBG_DFLT)) ) ;
            insFlag = skfp->ins ;
         }

         //* Report the data members of each field *
         this->tseReport ( *skfp, wpRpt ) ;
         winPtr->RefreshField ( skfPtr->fi ) ;
      }

      //* Note: Closing the window re-enables input buffering *
      //*       AND resets the echo option to 'termEcho'.     *
      //* Note: For this test, window display is not erased.  *
      winPtr->CloseWindow ( false ) ;

      delete ( winPtr ) ;     // release the allocated resourses

   }  // (ech==softEchoA && eOpt != L'r')

   //************************************************************
   //* softEchoA: Capture and report individual keypress events.*
   //* All special keys are active, but are reported only,      *
   //* not processed.                                           *
   //* -------------------------------------------------------- *
   //* softEchoD: Exercise the disable option.                  *
   //* Special keys should be discarded (set to NULLCHAR by the *
   //* call to acSpecialKeyTest()), and should not be available *
   //* to the application.                                      *
   //************************************************************
   else if ( (ech == softEchoA) || (ech == softEchoD) )
   {
      bool prow = true ;            // indicates first write to display

      //* Set unbuffered input using the specified echo option.*
      this->acBufferedInput ( false, ech, true ) ;

      wp = { baseROW, baseCOL } ;
      this->acSetCursor ( wp ) ;
      this->ttyWrite ( L">>>" ) ;
      this->acSetCursor ( wp ) ;
      do
      {
         if ( wp.row >= (exitROW - 2) )
         {
            wp = { baseROW, short(wp.col + 10) } ;
            this->acSetCursor ( wp ) ;
         }
         wch = this->acRead () ;
         if ( (this->acSpecialKeyTest ( wch )) )
         {
            gsOut.compose( "%C 0x%04X\n", &wch, &wch ) ;
            wp = this->acWrite ( wp, gsOut ) ;
         }
         else
         {
            gsOut.compose( " 0x%04X\n", &wch ) ;

            //* If on prompt row and if nullchar (rejected special key), or  *
            //* newline, then character was not written, so erase the prompt.*
            if ( prow )
            {
               if ( (wch == NULLCHAR) || (wch == NEWLINE) )
                  this->ttyWrite ( L" \b" ) ;
               prow = false ;
            }

            wp = this->acWrite ( wp.row, ++wp.col, gsOut ) ;
            --wp.col ;
            this->acSetCursor ( wp ) ;
         }
      }
      while ( wch != NEWLINE ) ;
   }  // (ech==softEchoA || ech==softEchoD)

   //********************************************************
   //* softEchoC: This test uses the special keys defined   *
   //* for cursor movement.                                 *
   //* Note that an "invisible" ACWin window is used to     *
   //* define the skForm used for testing. By default, the  *
   //* ACWin object uses a customized set of special keys   *
   //* with certain keycodes converted to accomodate the    *
   //* specific type of field under edit.                   *
   //* This test overrides that customized set of keycodes. *
   //********************************************************
   else if ( ech == softEchoC )
   {
      const short winROW = 7 ;            // window position
      const short winCOL = 8 ;
      const short winHEIGHT  = 12 ;       // window dimensions
      const short winWIDTH   = 48 ;
      const short rptROW = 20 ;           // position for stats report
      const short rptCOL = winCOL ; 

      WinPos wpul( 3, 1 ),                // initialize field origin
             wpRpt( rptROW, rptCOL ) ;    // position stats report
      short f = ZERO ;                    // index for initialization

      //* Construct an skForm-class object and initialize it.     *
      //* The initialized object is passed as a parameter to the  *
      //* ACWin-class constructor.                                *
      //* -- The window border and interior both use the terminal *
      //*    default color attributes.                            *
      //* -- The window border is invisible to give the impression*
      //*    that the field stands alone in the terminal window,  *
      //*    BUT it does not.                                     *
      skForm *skfPtr = new skForm( 2, 1, 1 ) ;

      //* Initialize the form-level members. Note that 'fCnt' is  *
      //* initialized during instantiation, and that 'base' value *
      //* is overridden by the ACWin initialization.              *
      skfPtr->ins  = false ;  // Overstrike mode for printing characters
      skfPtr->togg = false ;  // Lock overstrike mode so used cannot modify it.
      skfPtr->beep = true ;   // Noisy mode
      skfPtr->fi   = ZERO ;   // Index the only field

      //* Initialize the field-level members. Note that the 'ip'  *
      //* determines the value of the 'cur' member'. 'aca', 'off' *
      //* and 'ro'' all use their default values.                 *
      skfPtr->fld[f].orig = { 0, 0 } ;       // field index 0
      skfPtr->fld[f].hgt  = winHEIGHT - 4 ;
      skfPtr->fld[f].wid  = winWIDTH - 2 ;
      skfPtr->fld[f].ip = ZERO ;
      skfPtr->fld[f++].txt =
         "..............................................\n"
         "............Exercise the cursor keys:.........\n"
         "..LeftArrow, RightArrow, UpArrow, DownArrow,..\n"
         "..........Home, End, Tab, Shift+Tab,..........\n"
         "...............PageUp, PageDown...............\n"
         ".............Press Enter to Exit..............\n"
         "..............................................\n"
         ".............................................." ;
      skfPtr->fld[f].orig = { 9, 8 } ;       // field index 1
      skfPtr->fld[f].hgt  = 1 ;
      skfPtr->fld[f].wid  = winWIDTH - 18 ;
      skfPtr->fld[f].ip = 8 ;
      skfPtr->fld[f++].txt = "         Hello World!" ;

      //* For testing only: optionally make border visible and change     *
      //* colors. For production, border is invisible with default colors.*
      wchar_t border = L'i', color = L'd' ;

      ACWin *winPtr = new ACWin
            ( winROW, winCOL,                      // position (upper-left corner) 
              winHEIGHT, winWIDTH,                 // dimensions (height, width)
              (border=='d' ? ltDUAL : border=='s' ? ltSINGLE : ltHORIZ),
              (color=='d' ? acAttr(acaFG_DFLT | acaBG_DFLT) : acAttr(acaFGb_BROWN | acaBG_BLUE)),
              acAttr(acaFG_DFLT | acaBG_DFLT),      // interior attributes
              NULL,
              skfPtr
            ) ;

      //* Note: Opening the window disables input buffering AND *
      //*       set the echo option to 'softEchoA'.             *
      winPtr->OpenWindow( "\n\n\n\n\n\n\n\n\n"
                          "       [                              ]" ) ;

      //* Delete the local skForm object and reference   *
      //* the fully-initialized object within the window.*
      delete skfPtr ;                  // delete the local skForm object
      wchar_t wkey ;                   // user input
      short fIndx = ZERO ;             // index of field with focus
      const skForm *skfp = winPtr->GetForm () ;

      //* For this test only: Override the window's soft-echo option.*
      //*          Don't try this at home! See note above.           *
      winPtr->acEchoOption ( softEchoC ) ;

      //* Set cursor to most noticeable style. *
      this->acSetCursorStyle ( csBblock ) ;

      //* Report the data members of each field *
      this->tseReport ( *skfp, wpRpt ) ;

      //* Capture user input until a newline ('\n') character is received.*
      //* Note that the call includes the 'allChar' flag set so we can    *
      //* report the skForm/skField stats in real-time.                   *
      while ( (wkey = winPtr->EditField ( fIndx, NEWLINE, true )) != NEWLINE )
      {
         //* Report the data members of each field *
         this->tseReport ( *skfp, wpRpt ) ;
         winPtr->RefreshField ( skfPtr->fi ) ;
      }

      //* Return cursor to default shape.*
      this->acSetCursorStyle ( csDflt ) ;

      //* Note: Closing the window re-enables input buffering *
      //*       AND resets the echo option to 'termEcho'.     *
      //* Note: For this test, window display is not erased.  *
      winPtr->CloseWindow ( false ) ;

      delete ( winPtr ) ;     // release the allocated resourses

   }  // (ech==softEchoC)

   //****************************************************************
   //* softEchoE: Exercise the edit keys.                           *
   //* This test uses the Backspace/Delete/Insert/Tab/ShiftTab keys.*
   //* Note that an "invisible" ACWin window is used to define the  *
   //* skForm object used for testing.                              *
   //****************************************************************
   else if ( ech == softEchoE ) 
   {
      const short winROW = 3 ;
      const short winCOL = 2 ;
      const short winHEIGHT  = 11 ;       // window dimensions
      const short winWIDTH   = 71 ;
      const short instCOL = 3 ;
      const short rptROW = 14 ;           // position for stats report
      const short rptCOL = instCOL ;

      short f = ZERO ;                    // field index for initialization
      WinPos wpul( 3, 1 ),                // initialize field origin
             wpRpt( rptROW, rptCOL ),     // position stats report
             wpIns( 1, 63 ) ;             // offset of ins/ovr report
       bool insFlag = true ;              // track current state of ins/ovr flag

      //* Construct an skForm-class object and initialize it.     *
      //* The initialized object is passed as a parameter to the  *
      //* ACWin-class constructor.                                *
      //* -- The window border and interior both use the terminal *
      //*    default color attributes.                            *
      //* -- The window border is invisible to give the impression*
      //*    that the fields stand alone in the terminal window,  *
      //*    BUT they do not.                                     *
      //* -- Note that one of the fields is initialized with      *
      //*    multi-column chars to stress test cursor movements.  *
      skForm *skfPtr = new skForm( 3, 1, 1 ) ;

      //* Initialize the form-level members. Note that 'fCnt' is  *
      //* initialized during instantiation, and that 'base' value *
      //* is overridden by the ACWin initialization.              *
      skfPtr->ins  = insFlag ;
      skfPtr->togg = true ;
      skfPtr->beep = true ;
      skfPtr->fi   = ZERO ;

      //* Initialize the field-level members. Note that the 'ip'  *
      //* determines the value of the 'cur' member'.              *
      //* 'off' and 'ro' use their default values.                *
      skfPtr->fld[f].orig = wpul ;           // field index 0
      skfPtr->fld[f].hgt  = 1 ;
      skfPtr->fld[f].wid  = winWIDTH - 4 ;
      skfPtr->fld[f].ip = winWIDTH ; // note: this is an invalid value and s/b auto-corrected
      skfPtr->fld[f++].txt = L"For the \"softEchoE\" option, the cursor keys are disabled." ;
      wpul.row += 2 ;            // offset of next field

      skfPtr->fld[f].orig = wpul ;           // field index 1
      skfPtr->fld[f].hgt  = 1 ;
      skfPtr->fld[f].wid  = winWIDTH - 4 ;
      skfPtr->fld[f].ip = 24 ;
      skfPtr->fld[f++].txt = 
               "Use the Backspace, .....x..... Delete and Insert Keys to edit text." ;
      wpul.row += 2 ;            // offset of next field

      skfPtr->fld[f].orig = wpul ;           // field index 2
      skfPtr->fld[f].hgt  = 1 ;
      skfPtr->fld[f].wid  = winWIDTH - 4 ;
      skfPtr->fld[f].ip = 36 ;
      skfPtr->fld[f++].txt = 
               "Use the Tab and Shift+Tab keys .....x..... to move among fields.\n" ;
      wpul.row += 4 ;            // offset of next field

      //* For testing only: optionally make border visible and change     *
      //* colors. For production, border is invisible with default colors.*
      wchar_t border = L'i', color = L'd' ;

      ACWin *winPtr = new ACWin
            ( winROW, winCOL,                      // position (upper-left corner) 
              winHEIGHT, winWIDTH,                 // dimensions (height, width)
              (border=='d' ? ltDUAL : border=='s' ? ltSINGLE : ltHORIZ),
              (color=='d' ? acAttr(acaFG_DFLT | acaBG_DFLT) : 
                            acAttr(acaFGb_BROWN | acaBG_BLUE)),
              acAttr(acaFG_DFLT | acaBG_DFLT),      // interior attributes
              NULL,
              skfPtr
            ) ;

      //* Note: Opening the window disables input buffering AND *
      //*       sets the echo option to 'softEchoA'.            *
      winPtr->OpenWindow(
         "   Key Input Echo Option: softEchoE (Text Edit keys translated)\n"
         "       Press Enter To Exit\n\n"
         "[                                                                   ]\n\n"
         "[                                                                   ]\n\n"
         "[                                                                   ]" ) ;

      //* Delete the local skForm object and reference   *
      //* the fully-initialized object within the window.*
      delete skfPtr ;                  // delete the local skForm object
      wchar_t wkey ;                   // user input
      short fIndx = ZERO ;             // index of field with focus
      const skForm *skfp = winPtr->GetForm () ;

      //* For this test only: Override the window's soft-echo option.*
      //*          Don't try this at home! See note above.           *
      winPtr->acEchoOption ( softEchoE ) ;

      //* Update Insert/Overstrike report *
      winPtr->Write ( wpIns, (skfp->ins ? L"<INS>" : L"<OVR>"), 
                      (skfp->ins ? acAttr(acaFG_GREEN | acaBG_DFLT) : 
                                   acAttr(acaFG_BLUE  | acaBG_DFLT)) ) ;

      //* Report the data members of each field *
      this->tseReport ( *skfp, wpRpt ) ;

      //* Capture user input until a newline ('\n') character is received.*
      //* Note that the call includes the 'allChar' flag set so we can    *
      //* report the skForm/skField stats in real-time.                   *
      while ( (wkey = winPtr->EditField ( fIndx, NEWLINE, true )) != NEWLINE )
      {
         //* Update Insert/Overstrike report *
         if ( skfp->ins != insFlag )
         {
            winPtr->Write ( wpIns, (skfp->ins ? L"<INS>" : L"<OVR>"), 
                            (skfp->ins ? acAttr(acaFG_GREEN | acaBG_DFLT) : 
                                         acAttr(acaFG_BLUE  | acaBG_DFLT)) ) ;
            insFlag = skfp->ins ;
         }

         //* Report the data members of each field *
         this->tseReport ( *skfp, wpRpt ) ;
         winPtr->RefreshField ( skfPtr->fi ) ;
      }

      //* Note: Closing the window re-enables input buffering *
      //*       AND resets the echo option to 'termEcho'.     *
      //* Note: For this test, window display is not erased.  *
      winPtr->CloseWindow ( false ) ;

      delete ( winPtr ) ;     // release the allocated resourses

   }  // (ech==softEchoE)

   else if ( ech == termEcho )
   {
      short curCol = baseCOL ;   // base display column

      //* Set unbuffered input using the specified echo option.*
      this->acBufferedInput ( false, ech, true ) ;

      wp = { baseROW + 1, curCol } ;
      this->acSetCursor ( wp ) ;
      this->ttyWrite ( L">>>" ) ;

      do
      {
         if ( wp.row >= (exitROW - 2) )
         {
            wp = { short(baseROW + 1), short(curCol += 17) } ;
            this->acSetCursor ( wp ) ;
         }

         wch = this->acRead () ;

         if ( wch == NEWLINE )
            this->acSetCursor ( wp.row, (wp.col + 1) ) ;

         gsOut.compose( "  (0x%04X)\n", &wch ) ;
         this->ttyWrite ( gsOut ) ;
         if ( (wch == NEWLINE) && (curCol > baseCOL) )
            wp = { short(exitROW - 2), 1 } ;
         else
            wp = { short(wp.row + 1), curCol } ;
         this->acSetCursor ( wp ) ;
      }
      while ( wch != NEWLINE ) ;
   }  // (ech==termEcho)

   else  // (ech==noEcho)
   {
      //* Set unbuffered input using the specified echo option.*
      this->acBufferedInput ( false, ech, true ) ;

      wp = { short(baseROW + 1), baseCOL } ;
      this->acSetCursor ( wp ) ;
      this->ttyWrite ( L">>>" ) ;
      while ( (wch = this->acRead ()) != NEWLINE )
         this->acBeep () ;
   }  // (ech==noEcho)

   //* Restore foreground to default attribute *
   this->acSetFg ( aesFG_DFLT ) ;

   //* Restore original terminal settings and prepare to exit.*
   this->acRestoreTermInfo () ;
   this->acSetCursor ( aesCUR_ABSPOS, exitROW, 1 ) ;  // set cursor position

}  //* End Test_SoftEcho() *

//*************************
//*       tseReport       *
//*************************
//********************************************************************************
//* PRIVATE METHOD                                                               *
//* Called by Test_SoftEcho()to report user input and keycode processing.        *
//*                                                                              *
//* Input  : skf  : (by reference) skForm object containing data to be reported  *
//*          wpRpt: (by reference) position for report                           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
void AnsiCmd::tseReport ( const skForm& skf, const WinPos& wpRpt )
{
   WinPos wp ;
   gString gsOut( "skf: base:%02hd/%02hd fi:%02hd ins:%hhd pKey:%C %04X\n", 
                  &skf.base.row, &skf.base.col, &skf.fi, &skf.ins, 
                  &skf.pKey, &skf.pKey, &skf.togg, &skf.beep ) ;
   wp = this->acWrite ( wpRpt, gsOut ) ;

   for ( short f = ZERO ; f < skf.fCnt ; ++f )
   {
      gsOut.compose( "f:%-2hd orig:%02hd/%02hd cur:%02hd/%02hd ip:%-3hd\n", &f, 
                     &skf.fld[f].orig.row, &skf.fld[f].orig.col, 
                     &skf.fld[f].cur.row,  &skf.fld[f].cur.col,
                     &skf.fld[f].ip ) ;
      wp = this->acWrite ( wp, gsOut ) ;
   }

}  //* End tseReport() *

//*************************
//*      Test_Window      *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Create an ACWin object and ask the user to make adjustments to the           *
//* parameters.                                                                  *
//*                                                                              *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                [currently unused]                                            *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_Window ( aeSeq fg )
{
   #define DEBUG_TestWindow (0)    // FOR DEBUGGING ONLY

   const wchar_t* promptText = 
      L"Specify Window Configuration and Press Enter:\n\n\n" ;
   const char* bdTitle = "Border Attributes" ;
   const char* tdTitle = "Interior Attributes" ;
   const char* fldText = "------------Window refreshed.-------------" ;

   const short winROW     = 4 ;           // window position
   const short winCOL     = 6 ;
   const short altROW     = 3 ;
   const short altCOL     = 1 ;
   const short winHEIGHT  = 10 ;          // window dimensions
   const short winWIDTH   = 48 ;
   const short dumpROW    = winROW - 1 ;            // attribute dump
   const short tdumpCOL   = winCOL + winWIDTH + 4 ; // interior attribute dump
   const short bdumpCOL   = tdumpCOL + 40 ;         // border attribute dump
   const short promptRow  = 16 ;          // first row of prompt/menu area
   const short promptCol  = 6 ;           // left column of prompt/menu area
   const short promptHeight = 20 ;        // rows of prompt/menu area
   const short promptWidth  = 48 ;        // columns of prompt/menu area
   const short menuRow = promptRow + 2;   // row position for menu display
   const short menuCol = promptCol ;      // column position for menu display
   const short ccRow   = promptRow + 1 ;  // row position for color chart
   const short ccCol   = bdumpCOL - 7 ;   // column position for color chart
   const short rptRows = 20 ;             // operation-report rows
   const short rptCols = 26 ;             // operation-report columns

   gString gsOut,                         // text formatting
           gsIn ;                         // user input
   acaExpand txtExp, bdrExp ;             // local copy of window attributes
   WinPos winBase( winROW, winCOL ) ;     // window origin
   WinPos wpos ;                          // cursor offset into window object
   WinPos ppos( promptRow, promptCol ) ;  // prompt position
   WinPos mpos( menuRow, menuCol ) ;      // menu position
   WinPos cpos( ccRow, ccCol ) ;          // color-chart position
   WinPos wpDbg( ccRow, tdumpCOL ) ;      // position for debugging output

   //* Warn about minimum terminal window dimensions.*
   this->acWrite ( 1, 28, "  (requires terminal rows/cols of 37/142)\n"
                        "-----------------------------------------" ) ; 

   //* Set terminal configuration for unbuffered input *
   this->acBufferedInput ( false, softEchoA ) ;

   //* Display the color chart *
   this->twColorChart ( cpos ) ;

   //* Set attributes to specified terminal foreground value *
   this->acSetFgBg ( fg, aesBG_DFLT ) ;

   //* Define a single, one-row field with a read-only message. *
   //* Note that user is not offered a chance to edit the field.*
   //* Note also that the field is obscured by text written to  *
   //* the window EXCEPT for when the window is refreshed.      *
   skForm *skfPtr = new skForm( 1, ZERO, ZERO ) ;
   wpos = { 2, 2 } ;
   skfPtr->fld[0].orig = wpos ;
   skfPtr->fld[0].hgt  = 1 ;
   skfPtr->fld[0].wid  = winWIDTH - 6 ;
   skfPtr->fld[0].txt  = fldText ;
   skfPtr->fld[0].ip   = ZERO ;
   skfPtr->fld[0].ro   = true ;

   //**************************
   //* Define an ACWin object *
   //**************************
   ACWin *winPtr = new ACWin
         ( winROW, winCOL,                      // position (upper-left corner) 
           winHEIGHT, winWIDTH,                 // dimensions (height, width)
           ltSINGLE,                            // border style
           #if 1     // Initial Colors
           acAttr(acaFGb_BROWN | acaBG_BLUE),   // border attributes
           acAttr(acaFG_RED | acaBG_GREY),      // interior attributes
           #else     // Alternate Colors
           acAttr(acaFGb_GREEN | acaBG_BLUE),   // border attributes
           acAttr(acaFG_RED | acaBG_GREY),      // interior attributes
           #endif   // Initial Colors
           " Where In the World Is Carmen San Diego ", // title text
           skfPtr
         ) ;
   delete skfPtr ;                  // release the dynamic allocation

   //* Get a copy of the window attributes and display them.*
   winPtr->GetWindowAttributes ( txtExp, bdrExp ) ;

   gsOut = tdTitle ;
   this->acaExpandDump ( txtExp, dumpROW, tdumpCOL, &gsOut ) ;
   gsOut = bdTitle ;
   this->acaExpandDump ( bdrExp, dumpROW, bdumpCOL, &gsOut ) ;

   //*************************************
   //* Open the window (make it visible) *
   //*************************************
   wpos = winPtr->OpenWindow ( "Master thief Carmen San Diego is meeting with\n"
                               "the hench-persons of her gang, V.I.L.E.\n"
                             ) ;
   //wpos = winPtr->OpenWindow ( "----------1---------2---------3---------4----z\n"
   //                            "012345678901234567890123456789012345678\n"
   //                          ) ;
   short rows = winHEIGHT - 2, cols = winWIDTH  - 2 ;
   gString gsDbg( "rows/cols: %02hd / %02hd\n"
                  "wpos     : %2hd / %2hd\n",
                  &rows, &cols, &wpos.row, &wpos.col ) ;

   //* Enter some additional text at the current offset to verify *
   //* cursor position and box's interior color attributes.       *
   //* (Multi-column text characters test position parsing.)      *
   wpos = winPtr->Write ( wpos, L">(i.e. Villains International League of Evil)<\n" ) ;
   gsDbg.append( "wpos     : %02hd / %02hd\n", &wpos.row, &wpos.col ) ;

   wpos.col += 10 ;
   gsDbg.append( "wpos     : %02hd /*%02hd\n", &wpos.row, &wpos.col ) ;

   //* Chinese translation of "Villains International League of Evil" *
   wpos = winPtr->Write ( wpos, L">（即反派国际邪恶联盟）<" ) ;
   gsDbg.append( "wpos     : %02hd / %02hd\n", &wpos.row, &wpos.col ) ;

   //* Add text at an arbitrary offset and in a different color.*
   wpos.row += 2 ;
   wpos.col = 11 ;
   wpos = winPtr->Write ( wpos, "Who is Carmen, really?!\n"
                            "Stay tuned to find out.",
                            acAttr(acaFGb_BLUE | acaBG_GREY) ) ;

   gsDbg.append( "wpos     : %02hd / %02hd\n", &wpos.row, &wpos.col ) ;
   wpDbg = this->acWrite ( wpDbg, gsDbg ) ;

   //**-----------------------------------**
   //** Initial window setup is complete. **
   //**-----------------------------------**

   //* Clear the prompt/menu area and display main option menu.*
   this->acClearArea ( promptRow, promptCol, promptHeight, promptWidth ) ;
   this->acSetFgBg ( fg, aesBG_DFLT ) ;
   this->acSetMod ( aesUNDERLINE ) ;
   this->acWrite ( mpos.row, (mpos.col + 4),
                   L"(enclose arguments in single quotes)\n" ) ;
   this->acSetMod ( aesUNDERLINE_OFF ) ;
   this->acWrite ( (mpos.row + 1), mpos.col,
                   L"'c' (clear)   : erase window text\n"
                    "    Optional 'n' indicates row, else clear all\n"
                    "    Example: clear all: c  clear line: c='4'\n"
                    "'p' (position): specify text insertion\n"
                    "    point. Example: p='0,14'\n"
                    "'t' (addtext) : add text to window\n"
                    "    (caret: '^'==newline) (special case:'verb')\n"
                    "    Example: t='Hello World!^How are You?'\n"
                    "'T' (Title)[a]: specify new window title\n"
                    "    Example: T='ANSI World' or Ta='ANSI World'\n"
                    "'a' (txtattr) : set text fg/bg and update text\n"
                    "    Example: a='4,11' (a-h for RGB: a='c,12')\n"
                    "'b' (bdrattr) : set border fg/bg, update border\n"
                    "    Example: b='8,3' (a-h for RGB: b='b,e')\n"
                    "'d' (draw)[d] : lines and boxes\n"
                    "'h' (hide)    : hide (erase) the window\n"
                    "'r' (refresh) : redraw the window\n"
                    "'m' (move)    : move window to alt position\n"
                    "'x' (exit)    : exit the test\n"
                 ) ;

   #if 0    // Verify Attribute "Type"
   short frgbType  = aesFG_RGB,   brgbType   = aesBG_RGB,
         f4bitType = aesFG_DFLT,  b4bitType  = aesBG_DFLT,
         f8bitType = aesFG_INDEX, b8bitType = aesBG_INDEX ;
   gsOut.compose( "-----aes Types-----\n"
                  "4bitFg:%hX 4bitBg:%hX\n"
                  "8bitFg:%hX 8bitBg:%hX\n"
                  " rgbFg:%hX  rgbBg:%hX\n",
                  &f4bitType, &b4bitType, &f8bitType, 
                  &b8bitType, &frgbType, &brgbType ) ;
   this->acWrite ( 3, 135, gsOut ) ;
   #endif   // Verify Attribute "Type"

   //**********************
   //* Display the prompt *
   //**********************
   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ;
   this->acSetMod ( aesBOLD ) ;
   gsOut.clear() ;
   gsOut.padCols( promptWidth, wcsHORIZs ) ;
   this->acWrite ( (promptRow + 1), promptCol, gsOut ) ;
   this->acWrite ( (promptRow - 1), promptCol, promptText ) ;
   this->acSetMod ( aesBOLD_OFF ) ;

   //********************
   //* User interaction *
   //********************
   WinPos   userIP( 0, 0 ) ;              // text insertion point (offset)
   short    userRow,                      // row offset
            cindx,                        // character index
            newAttrSteps = 3 ;            // for setting text/border attributes
   bool     done = false ;                // loop control

   while ( ! done )
   {
      //* Clear the prompt area *
      this->acClearArea ( promptRow, promptCol, 1, promptWidth ) ;
      this->acSetCursor ( aesCUR_ABSPOS, promptRow, promptCol ) ;

      //******************
      //* Get user input *
      //******************
      this->acRead ( gsIn, promptWidth ) ;

      #if 1    // Debugging Only
      //* Used for repeated use of the same input *
      if ( *gsIn.gstr() == L'z' )
      {
         if ( newAttrSteps == 3 )         gsIn = L"a='c,b'" ;
         else if ( newAttrSteps == 2 )    gsIn = L"a='h,f'" ;
         else if ( newAttrSteps == 1 )    gsIn = L"a='f,c'" ;
      }
      #endif   // Debugging Only

      //* If the operation-report area is full, clear the area *
      if ( wpDbg.row >= (ccRow + rptRows) )
      {
         this->acClearArea (ccRow, tdumpCOL, rptRows, rptCols ) ;
         wpDbg = {ccRow, tdumpCOL} ;
      }

      //*---------------------------*
      //* Clear one row or all rows *
      //*---------------------------*
      if ( ((gsIn.find( L'c' )) == ZERO) || ((gsIn.find( L"clear" )) == ZERO) )
      {
         //* Clear a single row *
         if ( (cindx = gsIn.after( L"='" )) > ZERO )
         {
            gsIn.shiftChars( -(cindx) ) ;
            if ( (gsIn.gscanf( "%hd", &userRow )) != 1 ) // syntax error
               userRow = -1 ;
            if ( userRow >= (winHEIGHT - 2) )            // range check
               userRow = -1 ;
            if ( userRow >= ZERO )
               userIP = winPtr->ClearRow ( userRow ) ;
            gsDbg.compose( "clear    : %hd\n", &userRow ) ;
            wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
         }
         //* Clear entire window interior *
         else
         {
            userIP = winPtr->ClearWin () ;
            gsDbg = "clear    : win\n" ;
            wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
         }
      }

      //*------------------------------------------------*
      //* Specify text insertion point (cursor position) *
      //*------------------------------------------------*
      else if ( ((gsIn.find( L'p' )) == ZERO) || ((gsIn.find( L"position" )) == ZERO) )
      {
         //* Note: This position will be used for the subsequent write *
         //*       of text into the window.                            *
         if ( (cindx = gsIn.after( L"='" )) > ZERO )
         {
            gsIn.shiftChars( -(cindx) ) ;
            if ( ((gsIn.gscanf( "%hd,%hd", &userIP.row, &userIP.col )) != 2) ||
                 (userIP.row < ZERO) || (userIP.row >= (winHEIGHT - 2)) ||
                 (userIP.col < ZERO) || (userIP.col >= (winWIDTH - 2)) )
            {
               userIP.row = userIP.col = -1 ;
               gsDbg.compose( "position : %02hd / %02hd (error)\n", &userIP.row, &userIP.col ) ;
            }
            else
               gsDbg.compose( "position : %02hd / %02hd (saved)\n", &userIP.row, &userIP.col ) ;
         }
         else
            gsDbg = "position error\n" ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }

      //*-----------------------------------------------*
      //* Add text to window at current insertion point *
      //*-----------------------------------------------*
      else if ( ((gsIn.find( L't', ZERO, true )) == ZERO) || 
                ((gsIn.find( L"addtext" )) == ZERO) )
      {
         //* This text DOES NOT include line breaks and therefore may be used *
         //* to test automatic line-wrap functionality of the Write() method. *
         const wchar_t *verboseText = 
            L"When in the Course of human events it becomes necessary for one "
             "people to dissolve the political bands which have connected them "
             "with another, and to assume among the Powers of the earth, the "
             "separate and equal station to which the Laws of Nature and of "
             "Nature's God entitle them, a decent respect to the opinions of "
             "mankind requires that they should declare the causes which impel "
             "them to the separation." ;

         if ( (cindx = gsIn.after( L"='" )) > ZERO )
         {
            gsIn.shiftChars( -(cindx) ) ;
            if ( (gsIn.findlast( L'\'' )) == (gsIn.gschars() - 2) )
               gsIn.limitChars( (gsIn.gschars() - 2) ) ;
            gsIn.replace( '^', '\n', ZERO, false, true ) ;
            gsDbg.compose( "add text : %C%C...\n", 
                           &gsIn.gstr()[0], &gsIn.gstr()[1] ) ;

            //* Special Case: Use pre-defined verbose text.*
            if ( (gsIn.compare( L"verb", true, 4 )) == ZERO )
               gsIn = verboseText ;

            //* Write the text and save the new position *
            userIP = winPtr->Write ( userIP, gsIn ) ;
         }
         else
            gsDbg = "error\n" ;
         if ( (gsIn.findlast( L'\n' )) != (gsIn.gschars() - 2) )
            gsDbg.append( "\n" ) ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }

      //*---------------------------*
      //* Set text fg/bg attributes *
      //*---------------------------*
      else if ( ((gsIn.find( L'a' )) == ZERO) || ((gsIn.find( L"txtattr" )) == ZERO) )
      {
         const wchar_t *NewAttrText = L"New text attributes have been specified.\n" ;

         acAttr newAttr ;           // encoded attributes
         short   fgVal,  bgVal ;    // for color index
         bool isRgbFg, isRgbBg ;    // RGB indicators

         gsDbg = gsIn ;             // copy source data
         if ( (twDecodeFgBg ( gsDbg, fgVal, bgVal, isRgbFg, isRgbBg )) )
         {
            //* Construct the attribute bitmap *
            newAttr = winPtr->acComposeAttributes ( fgVal, bgVal, isRgbFg, isRgbBg ) ;
            wpos = { 2, 3 } ;       // test text output position

            //* Set the new attributes according to specified method.*
            //* These methods exercise different parts of the code.  *
            switch ( newAttrSteps )
            {
               //* Perform the update in three discrete steps *
               case 3:
                  winPtr->SetTextAttr ( newAttr ) ;            // set new attributes
                  winPtr->ClearWin () ;                        // erase window interior
                  userIP = winPtr->Write ( wpos, NewAttrText ) ; // write the text
                  break ;

               //* Perform the update in two steps *
               case 2:
                  winPtr->ClearWin ( newAttr ) ; // set new attributes and clear window
                  userIP = winPtr->Write ( wpos, NewAttrText ) ; // write the text
                  break ;

               //* Perform the update in a single step.  *
               //* Note that setting attributes during   *
               //* a Write() call is for that write only.*
               case 1:
                  //* Set attributes, clear window interior and write text *
                  gsOut.compose( "%S         (for this write only)          \n", 
                                 NewAttrText ) ;
                  userIP = winPtr->Write ( wpos, gsOut, newAttr, true ) ;
                  break ;
            }

            //* Display new attribute configuration *
            winPtr->GetWindowAttributes ( txtExp, bdrExp ) ;
            gsOut = tdTitle ;
            this->acaExpandDump ( txtExp, dumpROW, tdumpCOL, &gsOut ) ;
         }
         else
            gsDbg = "txtattr  : error\n" ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }  // text attributes

      //*-----------------------------*
      //* Set border fg/bg attributes *
      //*-----------------------------*
      else if ( ((gsIn.find( L'b' )) == ZERO) || ((gsIn.find( L"bdrattr" )) == ZERO) )
      {
         acAttr newAttr ;           // encoded attributes
         short   fgVal,  bgVal ;    // for color index
         bool isRgbFg, isRgbBg,     // RGB indicators
              goodUser = false ;    // 'true' if good user input

         gsDbg = gsIn ;             // copy source data
         if ( (goodUser = twDecodeFgBg ( gsDbg, fgVal, bgVal, isRgbFg, isRgbBg )) )
         {
            //* Construct the attribute bitmap *
            newAttr = winPtr->acComposeAttributes ( fgVal, bgVal, isRgbFg, isRgbBg ) ;

            winPtr->SetBorderAttr ( newAttr ) ;

            //* Display new attribute configuration *
            winPtr->GetWindowAttributes ( txtExp, bdrExp ) ;
            gsOut = bdTitle ;
            this->acaExpandDump ( bdrExp, dumpROW, bdumpCOL, &gsOut ) ;
         }
         else
            gsDbg = "bdrattr  : error\n" ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }     // border attributes

      //*------------------*
      //* Set window title *
      //*------------------*
      else if ( ((gsIn.find( L'T', ZERO, true )) == ZERO) || ((gsIn.find( L"Title" )) == ZERO) )
      {
         if ( (cindx = gsIn.after( L"='" )) > ZERO )
         {
            bool altAttr = false ;              // assume border color attributes
            if ( (gsIn.find( L'a' )) == cindx - 3 )
               altAttr = true ;                 // alternate text colors
            gsIn.shiftChars( -(cindx) ) ;
            if ( (gsIn.findlast( L'\'' )) == (gsIn.gschars() - 2) )
               gsIn.limitChars( (gsIn.gschars() - 2) ) ;
            gsDbg.compose( "Title    : %S\n", gsIn.gstr() ) ;
            if ( ! altAttr )
               winPtr->SetTitle ( gsIn.ustr() ) ;
            else
            {
               gsDbg.replace( L"Title   ", L"TitleAlt" ) ;
               winPtr->SetTitle ( gsIn.ustr(), acAttr(acaFG_BLUE | acaBGb_GREEN) ) ;
            }
         }
         else
            gsDbg = "Title error\n" ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }

      //*----------------------*
      //* Draw lines and boxes *
      //*----------------------*
      else if ( ((gsIn.find( L'd' )) == ZERO) || ((gsIn.find( L"draw" )) == ZERO) )
      {
         const acAttr boxCOLORS = acAttr(0x0000340A) ;
         const acAttr altATTR   = acAttr(0x0000340E) ;
         const short boxHEIGHT = 4 ;
         const short boxWIDTH = 26 ;
         LineType bdrStyle = ((gsIn.gstr()[1] == L'd') ? ltDUAL : ltSINGLE) ;
         //* Set the window title *
         winPtr->SetTitle ( " Line-drawing Tests ", acaUSEDFLT, bdrStyle ) ;
         //* Clear window and set color attributes *
         winPtr->ClearWin ( boxCOLORS ) ;

         //* Draw a box *
         wpos = { 4, 10 } ;
         winPtr->DrawBox ( wpos, boxHEIGHT, boxWIDTH, ltDUAL, altATTR ) ;
         //* Draw horizontal interior lines (no endpoints) *
         wpos.row += 2 ;
         wpos.col += 5 ;
         winPtr->DrawLine ( wpos, boxWIDTH - 8, false, ltSINGLE, L'\0', L'\0', altATTR ) ;
         ++wpos.row ;
         winPtr->DrawLine ( wpos, boxWIDTH - 8, false, ltDUAL, L'\0', L'\0', altATTR ) ;

         //* Draw vertical interior lines (no endpoints) *
         --wpos.row ;
         wpos.col -= 2 ;
         winPtr->DrawLine ( wpos, boxHEIGHT - 2, true, ltDUAL, L'\0', L'\0', altATTR ) ;
         wpos.col += boxWIDTH - 5 ;
         winPtr->DrawLine ( wpos, boxHEIGHT - 2, true, ltSINGLE, L'\0', L'\0', altATTR ) ;

         //* Draw horizontal lines the full width of window.*
         wpos = { 1, 0 } ;
         winPtr->DrawLine ( wpos, winWIDTH, false, ltSINGLE ) ;
         wpos.row += 3 ;
         winPtr->DrawLine (wpos, winWIDTH, false, ltDUAL ) ;

         //* Draw vertical lines the full height of window.*
         //* These intersect not only with the border, but *
         //* also with the full-width horizontal lines.    *
         wpos = { 0, 4 } ;
         wpos = winPtr->DrawLine ( wpos, 2, true, ltSINGLE, L'\0', wcsINSECT ) ;
         wpos = winPtr->DrawLine ( wpos, 3, true, ltSINGLE, L'\0', wcsINSECTdh ) ;
         winPtr->DrawLine ( wpos, 5, true, ltSINGLE ) ;
         wpos = { 0, short(winWIDTH - 5) } ;
         wpos = winPtr->DrawLine ( wpos, 2, true, ltDUAL, L'\0', wcsINSECTdv ) ;
         wpos = winPtr->DrawLine ( wpos, 3, true, ltDUAL, L'\0', wcsINSECTd ) ;
         winPtr->DrawLine ( wpos, 5, true, ltDUAL ) ;

         #if 1    // Production
         //* Draw interior lines with appropriate endpoints.*
         wpos = { 2, 12 } ;
         winPtr->DrawLine ( wpos, 24, false, ltSINGLE, wcsLTEE, wcsRTEE ) ;
         ++wpos.row ;
         --wpos.col ;
         winPtr->DrawLine ( wpos, 26, false, ltDUAL, wcsLTEEd, wcsRTEEd ) ;
         wpos = { 5, 7 } ;
         winPtr->DrawLine ( wpos, 4, true, ltSINGLE, wcsTTEE, wcsBTEE ) ;
         wpos.col = winWIDTH - 8 ;
         winPtr->DrawLine ( wpos, 4, true, ltDUAL, wcsTTEEd, wcsBTEEd ) ;
         #else    // Debugging only: Test length==2 (this is a special case)
         WinPos wpt = { 2, 0 } ;
         wpt = winPtr->DrawLine ( wpt, 2, false, ltSINGLE, L'\0', L'x' ) ;
         wpt = winPtr->DrawLine ( wpt, 44, false, ltSINGLE, L'y' ) ;
         wpt = winPtr->DrawLine ( wpt, 2, false, ltSINGLE, L'z' ) ;
         wpt = { 0, 6 } ;
         wpt = winPtr->DrawLine ( wpt, 2, true, ltSINGLE, L'\0', L'x' ) ;
         wpt = winPtr->DrawLine ( wpt, 6, true, ltSINGLE, L'y' ) ;
         wpt = winPtr->DrawLine ( wpt, 2, true, ltSINGLE, L'z' ) ;
         #endif   // Debugging only

         gsDbg = "draw     : OK\n" ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }

      //*-----------------*
      //* Hide the window *
      //*-----------------*
      else if ( ((gsIn.find( L'h' )) == ZERO) || ((gsIn.find( L"hide" )) == ZERO) )
      {
         acAttr hideAttr = acaUSEDFLT ;
         if ( gsIn.gstr()[1] != NULLCHAR )   // test alternate clear attribute
            hideAttr = acAttr(acaFG_BLUE | acaBG_CYAN) ;
         winPtr->HideWin ( hideAttr ) ;
         gsDbg = "hide     : OK\n" ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }

      //*------------------------------*
      //* Refresh (re-draw) the window *
      //*------------------------------*
      else if ( ((gsIn.find( L'r' )) == ZERO) || ((gsIn.find( L"refresh" )) == ZERO) )
      {
         winPtr->RefreshWin () ;
         gsDbg = "refresh  : OK\n" ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }

      //*-------------------------------------------*
      //* Move the window to the alternate position *
      //*-------------------------------------------*
      else if ( ((gsIn.find( L'm' )) == ZERO) || ((gsIn.find( L"move" )) == ZERO) )
      {
         if ( winBase.row == winROW )                 // alternate window position
            winBase = {altROW, altCOL} ;
         else                                         // original window position
            winBase = {winROW, winCOL} ;
         winPtr->MoveWin ( winBase ) ;
         gsDbg = "move     : OK\n" ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }

      //*------------------------------------*
      //* Exit the test and return to caller *
      //*------------------------------------*
      else if ( ((gsIn.find( L'x' )) == ZERO) || ((gsIn.find( L"exit" )) == ZERO) )
      {
         done = true ;
      }

      //* Options to specify which text attribute modification test *
      //* to perform. Not included in the user menu. (see above)    *
      else if ( ((gsIn.find( L'1' )) == ZERO) ||
                ((gsIn.find( L'2' )) == ZERO) ||
                ((gsIn.find( L'3' )) == ZERO) )
      {
         gsIn.gscanf( "%hd", &newAttrSteps ) ;
         gsDbg.compose( "attrSteps: %hd\n", &newAttrSteps ) ;
         wpDbg = this->acWrite ( wpDbg, gsDbg ) ;     // report
      }

   }  // while()

   //* Close the window (but do not erase it) *
   winPtr->CloseWindow ( false ) ;

   //* Delete the window object *
   delete winPtr ;

   //* Restore original terminal settings *
   this->acRestoreTermInfo () ;

   //* Prepare to Exit *
   this->acSetCursor ( aesCUR_ABSPOS, this->termRows - 1, 1 ) ;

}  //* End Test_Window() *

//*************************
//*     twDecodeFgBg      *
//*************************
//********************************************************************************
//* Non-member method: Called by Test_Window() to decode fg/bg user input.       *
//* ------------------                                                           *
//* Raw Input Format: a='x,y'  or  b='x,y'                                       *
//*  a) Indexed values are integers between zero and max8BIT.                    *
//*  b) RGB colors are indicated by alpha characters between 'a' and 'h'.        *
//*                                                                              *
//* Input  : gsIn   : (by reference) user input string                           *
//*          fgVal  : (by reference) receives decoded foreground index           *
//*          bgVal  : (by reference) receives decoded background index           *
//*          isRgbFg: (by reference) 'true' if fgnd is RGB index                 *
//*                                  'false' if fgnd is table index              *
//*          isRgbBg: (by reference) 'true' if bgnd is RGB index                 *
//*                                  'false' if bgnd is table index              *
//*                                                                              *
//* Returns: 'true' if data validated, else 'false'                              *
//*          On return, gsIn contains the report message.                        *
//********************************************************************************

static bool twDecodeFgBg ( gString& gsIn, short& fgVal, short& bgVal, 
                           bool& isRgbFg, bool& isRgbBg )
{
   //* Reporting formats *
   const char *rr = "txtattr  :  %C /  %C\n" ;
   const char *ii = "txtattr  : %02hd / %02hd\n" ;
   const char *ir = "txtattr  : %02hd /  %C\n" ;
   const char *ri = "txtattr  :  %C / %02hd\n" ;

   wchar_t fgChar = L'\0',                // alpha characters
           bgChar = L'\0' ;
   short indexed = ZERO,                  // number of indexed values scanned
         rgbeed  = ZERO,                  // number of RGB alpha characters scanned
         cindx ;                          // source index
   bool border   = false,                 // true if border, false if text
        goodUser = false ;                // return value

   fgVal = bgVal = 256 ;                  // initialize indices
   isRgbFg = isRgbBg = false ;            // initialize flags

   //* Determine whether target area is text (interior) or border *
   if ( gsIn.gstr()[0] == L'b' )
      border = true ;

   //* Strip the enclosure leaving only the parameters seperated by a comma.*
   if ( (cindx = gsIn.after( L"='" )) > ZERO )
   {
      gsIn.shiftChars( -(cindx) ) ;
      if ( (gsIn.findlast( L'\'' )) == (gsIn.gschars() - 2) )
         gsIn.limitChars( (gsIn.gschars() - 2) ) ;

      if ( (indexed = gsIn.gscanf( "%hd,%hd", &fgVal, &bgVal )) == ZERO )
      {
         if ( (rgbeed  = gsIn.gscanf( "%C,%C", &fgChar, &bgChar )) > ZERO )
         {
            if ( (bgChar >= L'0') && (bgChar <= L'9') )
            { bgChar = L'\0' ; --rgbeed ; }
         }
      }

      //* If at exactly one parameter scanned, *
      //* scan for the remaining parameter.    *
      if ( ((indexed + rgbeed) == 1) && ((cindx = gsIn.find( L',' )) > ZERO) )
      {
         if ( indexed == 1 )                 // indexed fg found, scan for rgb bg
            rgbeed += gsIn.gscanf( cindx, ",%C", &bgChar ) ;
         else                                // rgb fg found, scan for indexed bg
            indexed += gsIn.gscanf( cindx, ",%hd", &bgVal ) ;
      }

      //* If two parameters were captured *
      if ( (indexed + rgbeed) == 2 )
      {
         bool goodFg = false, goodBg = false ;
         if ( (fgChar >= L'a') && (fgChar <= L'h') )  // convert rgb fg to index
         {
            //* Convert alpha user input to web-safe RGB indices.*
            switch ( fgChar )
            {
               case L'a':  fgVal = 0 ;    break ;
               case L'b':  fgVal = 36 ;   break ;
               case L'c':  fgVal = 72 ;   break ;
               case L'd':  fgVal = 108 ;  break ;
               case L'e':  fgVal = 144 ;  break ;
               case L'f':  fgVal = 180 ;  break ;
               case L'g':  fgVal = 216 ;  break ;
               case L'h':  fgVal = 252 ;  break ;
            }
            goodFg = isRgbFg = true ;
         }
         if ( (bgChar >= L'a') && (bgChar <= L'h') )  // convert rgb bg to index
         {
            switch ( bgChar )
            {
               case L'a':  bgVal = 0 ;    break ;
               case L'b':  bgVal = 36 ;   break ;
               case L'c':  bgVal = 72 ;   break ;
               case L'd':  bgVal = 108 ;  break ;
               case L'e':  bgVal = 144 ;  break ;
               case L'f':  bgVal = 180 ;  break ;
               case L'g':  bgVal = 216 ;  break ;
               case L'h':  bgVal = 252 ;  break ;
            }
            goodBg = isRgbBg = true ;
         }

         //* If fg != valid rgb *
         if ( ! goodFg && ((fgVal >= ZERO) && (fgVal <= max8BIT)) )
         {
              goodFg = true ;
         }

         //* If bg != valid rgb *
         if ( ! goodBg && ((bgVal >= ZERO) && (bgVal <= max8BIT)) )
         {
            goodBg = true ;
         }

         if ( goodFg && goodBg )       // fg and bg captured and range checked
         {
            goodUser = true ;
   
            //* Format the caller's report *
            if ( !isRgbFg && !isRgbBg )
               gsIn.compose( ii, &fgVal, &bgVal) ;
            else if ( isRgbFg  && isRgbBg )
               gsIn.compose( rr, &fgChar, &bgChar) ;
            else if ( !isRgbFg && isRgbBg )
               gsIn.compose( ir, &fgVal, &bgChar) ;
            else if ( isRgbFg && !isRgbBg )
               gsIn.compose( ri, &fgChar, &bgVal) ;
            if ( border )
               gsIn.replace( L"txt", L"bdr" ) ;
         }
      }
   }     // strip
   return goodUser ;

}  //* End twDecodeFgBg() *

//*************************
//*       Test_Form       *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Create an ACWin object and exercise access to the functionality of the       *
//* skForm object within an ACWin window.                                        *
//*                                                                              *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_Form ( aeSeq fg )
{
   const char* bdTitle = "Border Attributes" ;
   const char* tdTitle = "Interior Attributes" ;
   const char* skTitle = "Settings: skForm Object" ;
   const short winROW     = 4 ;           // window position
   const short winCOL     = 6 ;
   const short winHEIGHT  = 17 ;          // window dimensions
   const short winWIDTH   = 48 ;
   const short dumpROW    = winROW - 1 ;            // attribute dump
   const short bdumpCOL   = winCOL + winWIDTH + 4 ; // border attribute dump
   const short tdumpCOL   = bdumpCOL + 40 ;         // interior attribute dump
   const short skfROW     = dumpROW + 13 ;          // skForm dump
   const short skfCOL     = bdumpCOL ;

   gString gsOut,                         // text formatting
           gsIn ;                         // user input
   acaExpand txtExp, bdrExp ;             // local copy of window attributes
   WinPos wpos ;                          // cursor offset into window object

   //* Warn about minimum terminal window dimensions.*
   this->acWrite ( 1, 28, "  (requires terminal rows/cols of 37/140)\n"
                        "-----------------------------------------" ) ; 

   //* -------------------------------------------------------- *
   //* Important Note: The call to OpenWindow() sets unbuffered *
   //* input with blocking read and echo option 'softEchoA'.    *
   //* -------------------------------------------------------- *

   //* Set attributes to specified terminal foreground value *
   this->acSetFgBg ( fg, aesBG_DFLT ) ;

   //* -------------------------------------------------------- *
   //* Define the input fields (skForm object).                *
   //* -One field is three(3) rows in height, and the other    *
   //*  fields have a height of one row.                       *
   //* -Field width varies: 2 are wide, 3 are narrow.          *
   //* - Field color attributes set to terminal defaults during*
   //*   instantiation, and then automagically set to window   *
   //*   interior attributes by ACWin constructor UNLESS       *
   //*   explicitly specified in the skForm initialization.    *
   //* -Position of field 'orig' is relative to window origin. *
   //* -'ip' indicates initial cursor position (insertion      *
   //*   point. 'cur' value is calculated based on 'ip'.       *
   //*                                                         *
   //* -------------------------------------------------------- *
   const short widA = winWIDTH - 6,             // width of field 0 and 3
               widB = widA / 2 - 1 ;            // width of field 1, 2, 4
   short fldIndx,                               // index of field with focus
         f = ZERO ;                             // field index
   WinPos wpul( 2, 2 ) ;                        // field origin
   //* Create an object with five(5) fields    *
   //* and initialize each field.              *
   //* See also Insert key configuration below.*
   skForm *skfPtr = new skForm( 5, wpul.row, wpul.col ) ;
   skfPtr->fld[f].orig = wpul ;                 // field index 0
   skfPtr->fld[f].hgt  = 3 ;
   skfPtr->fld[f].wid  = widA ;
   #if 1    // Production
   skfPtr->fld[f].txt = "Edit keys are: Left, Right, Home, End, \n"
                        "Tab, Shift+Tab, PageUp, PageDown, plus \n"
                        "Backspace, Delete and Insert." ;
   #else    // TEST: mixed-width-characters
   // (Although our wife is far away, she is always with us.)
   skfPtr->fld[f].txt = L"We are happy to present 我们的妻子苏菲。"
                         "Although 她在远方， we feel that "
                         "她总是和我们在一起。Happy feeling!" ;
   #endif   // TEST: mixed-width-character test
   skfPtr->fld[f].ip = 56 ;   // (this is approximately the center of the field)
   skfPtr->fld[f++].aca.acaVal = acAttr(acaFG_BLUE | acaBG_GREY) ;
   wpul.row += 5 ;            // position of next field

   skfPtr->fld[f].orig = wpul ;                 // field index 1
   skfPtr->fld[f].hgt  = 1 ;
   skfPtr->fld[f].wid  = widB ;
   skfPtr->fld[f].ip   = ZERO ;
   skfPtr->fld[f++].aca.acaVal = acAttr(acaFG_GREEN | acaBG_GREY) ;
   wpul.col += widB + 2 ;     // position of next field

   skfPtr->fld[f].orig = wpul ;                 // field index 2
   skfPtr->fld[f].hgt  = 1 ;
   skfPtr->fld[f].wid  = widB ;
   skfPtr->fld[f].txt  = "Gặp tôi ở Sai Gon." ;  // "Meet me in Saigon."
   skfPtr->fld[f].ip   = 10 ;                   // position IP in mid-text
   skfPtr->fld[f++].aca.acaVal = acAttr(acaFG_RED | acaBG_GREY) ;
   wpul.row += 3 ;            // position of next field
   wpul.col -= widB + 2 ;

   skfPtr->fld[f].orig = wpul ;                 // field index 3
   skfPtr->fld[f].hgt  = 1 ;
   skfPtr->fld[f].wid  = widA ;
   skfPtr->fld[f].txt  = "这他妈的并不像看起来那么容易。" ; // "It's not as fucking easy as it looks."
   skfPtr->fld[f].ip   = ZERO ;
   skfPtr->fld[f++].aca.acaVal = acAttr(acaFG_MAGENTA | acaBG_GREY) ;
   wpul.row += 3 ;            // position of next field
   wpul.col = (winWIDTH - 2) / 2 - (widB + 1) / 2 ;

   skfPtr->fld[f].orig = wpul ;                 // field index 4
   skfPtr->fld[f].hgt  = 1 ;
   skfPtr->fld[f].wid  = widB + 1 ;
   skfPtr->fld[f].txt  = L" Press Enter To Exit " ;
   skfPtr->fld[f].ip   = ZERO ;
   skfPtr->fld[f].ro   = true ;  // field is read-only (cannot receive input focus)
   skfPtr->fld[f].aca.acaVal = acAttr(acaFGb_BROWN | acaBG_BLUE) ;
   fldIndx = skfPtr->fi = ZERO ; // input focus on field zero (0)

   //**************************
   //* Define an ACWin object *
   //**************************
   ACWin *winPtr = new ACWin
         ( winROW, winCOL,                        // position (upper-left corner) 
           winHEIGHT, winWIDTH,                   // dimensions (height, width)
           ltSINGLE,                              // border style
           acAttr(acaFGb_BROWN | acaBG_BLUE),     // border attributes
           acAttr(acaFG_RED | acaBG_GREY),        // interior attributes
           "  Window Formatted With \"skForm\" ", // title text
           skfPtr                                 // partially-initialized skForm object
         ) ;

   //* Get a copy of the window attributes and display them.*
   winPtr->GetWindowAttributes ( txtExp, bdrExp ) ;
   gsOut = bdTitle ;
   this->acaExpandDump ( bdrExp, dumpROW, bdumpCOL, &gsOut ) ;
   gsOut = tdTitle ;
   this->acaExpandDump ( txtExp, dumpROW, tdumpCOL, &gsOut ) ;

   //*************************************
   //* Open the window (make it visible) *
   //*************************************
   //* Note that we create borders around the defined fields.   *
   //* Important Note: The call to OpenWindow() sets unbuffered *
   //* input with blocking read and echo option 'softEchoA'.    *
   wpos = winPtr->OpenWindow ( " Tab among fields. Edit the fields as desired." ) ;

   //* Configure the functionality of the "special" Insert key *
   winPtr->InsertKey ( true, true, true ) ;

   //* Delete the local skForm object and reference   *
   //* the fully-initialized object within the window.*
   delete skfPtr ;
   gsOut = skTitle ;
   const skForm *skfp = winPtr->GetForm () ;
   this->skFormDump ( *skfp, skfROW, skfCOL, &gsOut ) ;

   //* Draw boxes around four of the five fields.*
   winPtr->FieldOutline ( 0, ltSINGLE, acAttr(acaFG_BLUE | acaBG_GREY) ) ;
   winPtr->FieldOutline ( 1, ltSINGLE, acAttr(acaFG_GREEN | acaBG_GREY) ) ;
   winPtr->FieldOutline ( 2, ltSINGLE, acAttr(acaFG_CYAN | acaBG_BLUE) ) ;
   winPtr->FieldOutline ( 3, ltSINGLE, acAttr(acaFG_MAGENTA | acaBG_GREY) ) ;

   //* Refresh the fields obscured by the boxes.*
   //* Focus is initially on Field Zero.        *
   winPtr->RefreshField ( skfp->fCnt ) ;

   //******************
   //* Get User Input *
   //******************
   winPtr->EditForm ( NEWLINE, fldIndx ) ;

   //* Close the window and delete the object.*
   //* Important Note: The call to CloseWindow() sets buffered *
   //* input with blocking read and echo option 'termEcho'.    *
   winPtr->CloseWindow () ;
   delete winPtr ;
   this->acWrite ( winROW, winCOL, L"Window closed and ACWin object deleted." ) ;

   //* Restore foreground to default attribute *
   this->acSetFg ( aesFG_DFLT ) ;

   //* Restore original terminal settings *
   this->acRestoreTermInfo () ;

   //* Prepare to Exit *
   this->acSetCursor ( aesCUR_ABSPOS, this->termRows - 1, 1 ) ;

}  //* End Test_Form() *

//*************************
//*    Test_WinDefault    *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Create an ACWin object using the default constructor.                        *
//* Exercise various window-configuration methods.                               *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_WinDefault ( aeSeq fg )
{
   const acAttr Text1Attr   = acAttr(acaFG_BLUE | acaBG_CYAN) ;
   const acAttr Text2Attr   = acAttr(acaFG_DFLT | acaBG_DFLT) ;
   const acAttr Title1Attr  = acAttr(acaFGb_GREEN | acaBG_BROWN) ;
   const acAttr Border1Attr = acAttr(acaFG_GREEN | acaBG_BROWN) ;
   const acAttr Border2Attr = acAttr(acaFG_DFLT | acaBG_DFLT) ;
   const acAttr Field1Attr  = acAttr(acaFGb_GREEN | acaBG_BLACK) ;
   const acAttr Field2Attr  = acAttr(acaFG_DFLT | acaBG_DFLT) ;
   const acAttr CloseAttr   = acAttr(acaFG_DFLT | acaBG_GREY) ;
   const char  *Title1Text  = "  Default ACWin Constructor  " ;
   const char  *Title2Text  = "" ;

   gString gsOut, gsIns, gsTxt ;       // text formatting
   WinPos  wpos ;                      // window offset in terminal window
   WinPos  wpTxt( 0, 0 ) ;             // offset into window's text area
   const skForm *skfp ;                // pointer to window's skForm object
   acAttr  txtAttr = Text2Attr ;       // current text (window interior) attributes
   acAttr  bdrAttr = Border2Attr ;     // current border attributes
   acAttr  fldAttr = Field2Attr ;      // input field color attributes
   wchar_t wkey ;                      // user input
   short   whgt,                       // window height (rows)
           wwid,                       // window width (columns)
           fIndx = ZERO ;              // receives field index on return from EditForm()
   bool    title = false ;             // set if title currently displayed

   //* Set unbuffered read for initial interaction.            *
   //* Window will take control of buffering when it is opened.*
   this->acBufferedInput ( false, noEcho ) ;

   //* Instantiate the ACWin object using the default constructor.       *
   //* All parameters are set to the default values for that constructor.*
   ACWin *winPtr = new ACWin ;

   //* Get window dimensions and position, *
   //* and compose user instructions.      *
   wpos  = winPtr->GetWinConfig ( whgt, wwid ) ;
   skfp  = winPtr->GetForm () ;
   gsIns.compose( "pos:%hd:%hd rows:%hd cols:%hd fields:%hd\n"
                  "focus:%hd ins:%hhd togg:%hhd beep:%hhd\n"
                  "'t' text color   'b' border color\n"
                  "'f' field color  'e' edit field\n"
                  "'T' set title    'c' cursor move\n"
                  "                 'q' to quit.", 
                  &wpos.row, &wpos.col, &whgt, &wwid, &skfp->fCnt,
                  &skfp->fi, &skfp->ins, &skfp->togg, &skfp->beep ) ;
   gsTxt = " Hello World! " ;
   gsTxt.padCols( wwid - 4, L'-', true ) ;
   gsTxt.insert( "\n\n<" ) ;
   gsTxt.append( ">\n   (press any key to continue)" ) ;

   //* Open the window and write some text into the text area.*
   winPtr->OpenWindow ( gsTxt.ustr() ) ;
   this->acRead () ;                // wait for user response

   //* Initialize the (only) field with instructions *
   //* on how to use the test.                       *
   winPtr->SetFieldText ( fIndx, gsIns ) ;

   while ( (wkey = this->acRead ()) != L'q' )
   {
      switch ( wkey )
      {
         case L't':              // Text color (window interior)
            txtAttr = txtAttr == Text2Attr ? Text1Attr : Text2Attr ;
            winPtr->SetTextAttr ( txtAttr ) ;
            winPtr->Write ( wpTxt, gsTxt, acaUSEDFLT, true ) ;
            this->acRead () ;    // wait for user response
            winPtr->RefreshField ( fIndx ) ;
            break ;
         case L'b':              // Border color
            bdrAttr = bdrAttr == Border2Attr ? Border1Attr : Border2Attr ;
            winPtr->SetBorderAttr ( bdrAttr ) ;
            if ( title )
               winPtr->SetTitle ( Title1Text, Title1Attr ) ;
            break ;
         case L'f':              // Field text color
            fldAttr = fldAttr == Field2Attr ? Field1Attr : Field2Attr ;
            winPtr->SetFieldAttr ( fIndx, fldAttr ) ;
            break ;
         case L'e':              // Edit text
            fldAttr = fldAttr == Field2Attr ? Field1Attr : Field2Attr ;
            winPtr->SetFieldAttr ( fIndx, fldAttr ) ;
            winPtr->EditForm ( L'q', fIndx ) ;
            #if 0    // Test GetFieldText()
            {
            winPtr->GetFieldText ( fIndx, gsOut ) ;
            WinPos wp( short(wpos.row + whgt + 1), short(wpos.col + 1) ) ;
            this->acSetFg ( fg ) ;
            this->acSetMod ( aesUNDERLINE ) ;
            wp = this->acWrite ( wp, L"Copy of Field Text Data           \n" ) ;
            this->acSetMod ( aesUNDERLINE_OFF ) ;
            this->acWrite ( wp, gsOut ) ;
            this->acSetFg ( aesFG_DFLT ) ;
            }
            #endif   // Test GetFieldText()
            winPtr->SetFieldText ( fIndx, gsIns ) ;   // restore instructions
            fldAttr = fldAttr == Field2Attr ? Field1Attr : Field2Attr ;
            winPtr->SetFieldAttr ( fIndx, fldAttr ) ;
            break ;
         case L'T':              // Title
            if ( ! title )
            { winPtr->SetTitle ( Title1Text, Title1Attr ) ; title = true ; }
            else
            { winPtr->SetTitle ( Title2Text, bdrAttr ) ; title = false ; }
            break ;
         case L'c':              // Cursor movement within field
            winPtr->SetFieldIP ( fIndx, 9 ) ;
            this->nsleep ( 16 ) ;
            winPtr->SetFieldIP ( fIndx, 79 ) ;
            this->nsleep ( 16 ) ;
            winPtr->SetFieldIP ( fIndx, gsMAXCHARS ) ;
            this->nsleep ( 16 ) ;
            winPtr->SetFieldIP ( fIndx, ZERO ) ;
            break ;
         default:
            this->acBeep () ;
      }
   }

   //* Close the window and erase the display area using   *
   //* a grey background.                                  *
   //* Note: Closing the window re-enables input buffering *
   //*       AND resets the echo option to 'termEcho'.     *
   winPtr->CloseWindow ( true, CloseAttr ) ;

   delete winPtr ;                     // delete the ACWin object

   //* Say bye-bye and prepare to exit.               *
   //* (Note the newlines used to set exit position.) *
   this->acSetFg ( fg ) ;
   this->acSetCursor ( wpos.row + 2, wpos.col + 1 ) ;
   this->ttyWrite ( "Default-constructor test complete.\n\n\n\n\n\n" ) ;
   this->acSetFg ( aesFG_DFLT ) ;

}  //* End Test_WinDefault() *

//*************************
//*       Test_Box        *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Create an AC_Box object with various configuration options.                  *
//*                                                                              *
//* The AC_Box class and associated AnsiCmd support methods are intentionally    *
//* simplistic in order to achieve maximum speed using minimum resources.        *
//* To get additional bells and whistles, see the ACWin class.                   *
//*                                                                              *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* The examples given in this test use simple, ANSI escape sequences            *
//* implemented in the basic AnsiCmd-class methods. Give particular attention    *
//* to the SEQUENCE in which the commands are executed.                          *
//* Anything even _moderately_ fancy here is done simply to fascilitate a        *
//* smooth user interface for the test.                                          *
//*                                                                              *
//* Programmer's Note: It is recommended to reset all text modifiers before      *
//* calling acBoxDraw(). Modifiers can cause a severe case of the Uglies when    *
//* drawing the border. Use caution.                                             *
//********************************************************************************

void AnsiCmd::Test_Box ( aeSeq fg )
{
   const char *Hello = "\n\n\n\n                 Hello World! " ;
   const char *boxInstructions = 
         "'a' Add text to the box.\n"
         "'w' Write new text in the box.\n"
         "'c' Clear the text area of the box.\n"
         "'e' Erase the box.\n"
         "'r' Redraw the box.\n"
         "'s' Set cursor position\n"
         "'t' Text color attributes.\n"
         "'b' Border color attributes.\n"
         "'l' Line type for border.\n"
         "'i' Italic text modifier (toggle)\n"
         "'u' Underline text modifier (toggle)\n"
         "'q' Quit." ;
   // This paragraph is an excerpt from a short story written by the author.
   // (c) Copyright 2014 The Software Samurai (all rights reserved)
   const char *Chuck = 
         "Yes, Chuck was a nerd, but he had a burning\n"
         "desire to express himself and an overwhelming\n"
         "need to create something that would be of\n"
         "lasting value to someone, though his peers\n"
         "would probably still steal his lunch, and he\n"
         "was sure that his family would just continue\n"
         "to stare at him in slack-jawed\n"
         "incomprehension." ;
   

   const short boxHEIGHT = 11 ;              // boxB height
   const short boxWIDTH  = 48 ;              // boxB width
   const aeSeq bdrFG1  = aesFG_GREEN ;       // boxB border attributes
   const aeSeq bdrBG1  = aesBG_BLACK ;
   const aeSeq bdrFG2  = aesFGb_RED ;
   const aeSeq bdrBG2  = aesBG_GREY ;
   const aeSeq txtFG1  = aesFGb_CYAN ;       // boxB text attributes
   const aeSeq txtBG1  = aesBG_BLUE ;
   const aeSeq txtFG2  = aesFG_BLUE ;
   const aeSeq txtBG2  = aesBG_CYAN ;

   gString gsOut ;                  // text formatting
   short termRows, termCols ;       // terminal dimensions
   WinPos boxOrig( 4, 0 ) ;         // position - upper left corner of box
   WinPos txtOrig ;                 // coordinates of text area within the box
   WinPos txtCurs ;                 // cursor position
   WinPos wpur, wplr, wpll ;        // corners of text area
   WinPos wpExit( short(boxOrig.row + boxHEIGHT * 2 + 4), 1 ) ; // exit position
   WinPos wp ;                      // text offset
   wchar_t wkey ;                   // user input
   const char *txtPtr = Hello ;     // pointer to active text string
   bool italic = false ;            // if true, write text with italic modifier
   bool underline = false ;         // if true, write text with underline modifier

   //* Warn about minimum terminal window dimensions.*
   this->acWrite ( 1, 28, "  (requires terminal dimensions of 36 rows X 140 columns)\n"
                          "---------------------------------------------------------" ) ;

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   this->acBufferedInput ( false, noEcho ) ;    // set unbuffered input

   this->acGetTermDimensions ( termRows, termCols ) ; // get terminal window dimensions
   WinPos bdbOrig( short(boxOrig.row + 2), 4 ) ; // brain-dead-box origin
   wp = { short(bdbOrig.row + 7), bdbOrig.col } ;

   //* Say hello (yes, it's too complex :-) *
   this->acSetMod ( aesUNDERLINE ) ;
   this->acSetMod ( aesBOLD ) ;
   this->acSetMod ( aesREVERSE ) ;
   this->acWrite ( boxOrig.row, bdbOrig.col, 
         "          Welcome To Boxing Day!          " ) ;
   this->acSetMod ( aesREVERSE_OFF ) ;
   this->acSetMod ( aesBOLD_OFF ) ;

   //* Display user instructions *
   wp = this->acWrite ( wp, "             Testing Options             \n" ) ;
   this->acSetMod ( aesUNDERLINE_OFF ) ;
   this->acWrite ( wp, boxInstructions ) ;

   //************************************************
   //* Draw a primitive box, and write text into it.*
   //************************************************
   this->acDrawBox ( bdbOrig, 6, 42, ltDUAL, 
                     "Hello. This is a simple, rectangle drawn\n"
                     "in the terminal window. It has no color\n"
                     "attributes or data storage capabilities.\n"
                     "Please see REAL boxes to the right.--->" ) ;

   //*******************************************
   //* AC_Box basic-initialization constructor *
   //*******************************************
   boxOrig.col = termCols / 2 - boxWIDTH / 2 ;  // calculate horizontal offset
   AC_Box boxA( boxOrig, boxHEIGHT, boxWIDTH ) ;
   txtOrig = this->acBoxDraw ( boxA ) ;
   wp = this->acWrite ( txtOrig, "This is the basic AC_Box structure.\n\n" ) ;
   wp = this->acWrite ( wp, "Terminal default color attributes are used\n"
                            "to draw both foreground and background.\n\n" ) ;
   wp = this->acWrite ( wp, "Use the acWrite() or ttyWrite() method to\n"
                            "write text to the interior of the box.\n"
                            "Range checking of position and width ARE NOT\n"
                            "performed." ) ;
   boxOrig.row += 12 ;

   //******************************************
   //* AC_Box full-initialization constructor *
   //******************************************
   AC_Box boxB( boxOrig, boxHEIGHT, boxWIDTH, ltDUAL, bdrFG1, bdrBG1, txtFG1, txtBG1,
                " Full-initialization AC_Box Class " ) ;
   txtOrig = this->acBoxDraw ( boxB, Hello ) ;
   txtCurs = this->acGetCursor () ;    // get current cursor position

   //* Calculate position of interior corners (for cursor positioning test) *
   wpur = { txtOrig.row, short(txtOrig.col + boxWIDTH - 3) } ;
   wplr = { short(txtOrig.row + boxHEIGHT - 3), wpur.col } ;
   wpll = { wplr.row, txtOrig.col } ;

   while ( (wkey = this->acRead ()) != L'q' )
   {
      switch ( wkey )
      {
         case L'a':           // Add text
            if ( txtCurs.row < (boxOrig.row + boxHEIGHT - 2) )
               ++txtCurs.row ;
            else
               txtCurs = txtOrig ;
            txtCurs.col = txtOrig.col + 7 ;
            this->acSetFgBg ( boxB.tFgnd, boxB.tBgnd ) ; // box interior attributes
            if ( underline ) { this->acSetMod ( aesUNDERLINE ) ; }
            if ( italic )    { this->acSetMod ( aesITALIC ) ; }
            txtCurs = this->acWrite ( txtCurs, "(Here is some additional text.)" ) ;
            if ( underline ) { this->acSetMod ( aesUNDERLINE_OFF ) ; }
            if ( italic )    { this->acSetMod ( aesITALIC_OFF ) ; }
            this->acSetFgBg ( fg, aesBG_DFLT ) ;         // restore terminal attributes
            break ;
         case L'w':           // Write alternate text
            txtPtr = txtPtr == Hello ? Chuck : Hello ; // select the text to write

            this->acBoxClear ( boxB ) ;       // clear text area AND set attributes
            if ( underline ) { this->acSetMod ( aesUNDERLINE ) ; }
            if ( italic )    { this->acSetMod ( aesITALIC ) ; }
            txtCurs = this->acWrite ( txtOrig, txtPtr ) ; // write the text
            if ( underline ) { this->acSetMod ( aesUNDERLINE_OFF ) ; }
            if ( italic )    { this->acSetMod ( aesITALIC_OFF ) ; }
            this->acSetFgBg ( fg, aesBG_DFLT ) ;       // restore terminal attributes
            break ;
         case L'c':           // Clear text area
            txtOrig = this->acBoxClear ( boxB ) ;
            txtPtr = Hello ;
            break ;
         case L'e':           // Erase the box
            this->acBoxErase ( boxB ) ;
            txtPtr = Hello ;
            break ;
         case L'r':           // Redraw the box
            this->acBoxDraw ( boxB, txtPtr ) ;
            break ;
         case L's':           // Set (visible) cursor position
            txtCurs = this->acGetCursor () ;
            if ( txtCurs == txtOrig )     txtCurs = wpur ;
            else if ( txtCurs == wpur )   txtCurs = wplr ;
            else if ( txtCurs == wplr )   txtCurs = wpll ;
            else                          txtCurs = txtOrig ;
            this->acSetCursor ( txtCurs ) ;
            break ;
         case L't':           // Text color
            if ( boxB.tFgnd == txtFG1 )   // set the alternate colors
            { boxB.tFgnd = txtFG2 ; boxB.tBgnd = txtBG2 ; }
            else
            { boxB.tFgnd = txtFG1 ; boxB.tBgnd = txtBG1 ; }
            this->acBoxDraw ( boxB, txtPtr ) ;
            break ;
         case L'b':           // Border color
            if ( boxB.bFgnd == bdrFG1 )   // set the alternate colors
            { boxB.bFgnd = bdrFG2 ; boxB.bBgnd = bdrBG2 ; }
            else
            { boxB.bFgnd = bdrFG1 ; boxB.bBgnd = bdrBG1 ; }
            this->acBoxDraw ( boxB, txtPtr ) ;
            break ;
         case L'l':           // Line type for border
            boxB.lnType = boxB.lnType == ltDUAL ? ltSINGLE : ltDUAL ;
            this->acBoxDraw ( boxB, txtPtr ) ;
            break ;
         case L'i':           // Italic modifier
         case L'u':           // Underline modifier
            if ( wkey == L'i' )      { italic = italic ? false : true ; }
            else if ( wkey == L'u' ) { underline = underline ? false : true ; }

            this->acBoxClear ( boxB ) ;       // clear text area AND set attributes
            if ( underline ) { this->acSetMod ( aesUNDERLINE ) ; }
            if ( italic )    { this->acSetMod ( aesITALIC ) ; }
            txtCurs = this->acWrite ( txtOrig, txtPtr ) ;
            if ( underline ) { this->acSetMod ( aesUNDERLINE_OFF ) ; }
            if ( italic )    { this->acSetMod ( aesITALIC_OFF ) ; }
            this->acSetFgBg ( fg, aesBG_DFLT ) ;   // restore terminal attributes
            break ;
         default:
            this->acBeep () ;
            break ;
      }
   }

   //* Restore buffered input and prepare to exit.*
   this->acBufferedInput ( true, termEcho ) ;
   this->acSetFg ( aesFG_DFLT ) ;
   this->acSetCursor ( wpExit ) ;

}  //* End Test_Box() *

//*************************
//*       Test_Line       *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY ansiTest().                                  *
//* Use the acDrawLine() method to draw shapes.                                  *
//*                                                                              *
//* Input  : fg  : foreground color attribute to be used for the tests           *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_Line (  aeSeq fg )
{
   const short boxHEIGHT = 9 ;      // box height
   const short boxWIDTH  = 41 ;     // box width
   const short boxULr = 7 ;         // box origin
   const short boxULc = 4 ;
   gString gsOut ;                  // text formatting
   WinPos wpMsg( 4, 4 ) ;           // message position
   WinPos wpBox( boxULr, boxULc ) ; // position - upper left corner of box
   WinPos wpExit( 16, 1 ) ;         // exit position
   acaExpand acaex ;                // convert aeSeq to acAttr
   bool setFlag ;                   // (dummy)
   acAttr boxBg = (fg == aesFG_DFLT) ? acAttr(acaBGb_GREY | acaREVERSE) :
                  acAttr((acaex.aeseq2bits ( fg, setFlag )) | acaREVERSE) ;

   //* Say Hello *
   this->acSetMod ( aesUNDERLINE ) ;
   this->acSetMod ( aesBOLD ) ;
   this->acSetMod ( aesREVERSE ) ;
   this->acWrite ( wpMsg, "        OK, Recruits, Get In Line!        " ) ;
   this->acSetMod ( aesREVERSE_OFF ) ;
   this->acSetMod ( aesBOLD_OFF ) ;
   this->acSetMod ( aesUNDERLINE_OFF ) ;
   wpMsg.row += 2 ;

   //* Set the foreground color *
   this->acSetFg ( fg ) ;

   this->acBufferedInput ( false, noEcho ) ;    // set unbuffered input

   //* Draw a box using individual lines *
   this->acWrite ( wpMsg, L"Draw a box using individual lines. Press a key..." ) ;
   this->acRead () ;
   wpBox = this->acDrawLine ( wpBox, boxWIDTH - 1, false, ltSINGLE, wcsUL ) ;
   this->acDrawLine ( wpBox, boxHEIGHT, true, ltSINGLE, wcsUR, wcsLR ) ;
   wpBox = { short(boxULr + 1), boxULc } ;
   wpBox = this->acDrawLine ( wpBox, boxHEIGHT - 1, true, ltSINGLE, L'\0', wcsLL ) ;
   --wpBox.row ;
   this->acDrawLine ( wpBox, boxWIDTH, false, ltSINGLE, wcsLL, wcsLR ) ;

   //* Draw intersecting lines within the box *
   this->acSetCursor ( wpMsg ) ;
   this->acEraseArea ( aesERASE_LINE ) ;
   this->acWrite ( wpMsg, L"Draw horizontal and vertical intersecting lines. Press a key..." ) ;
   this->acRead () ;
   wpBox = { short(boxULr + boxHEIGHT / 2), boxULc } ;
   this->acDrawLine ( wpBox, boxWIDTH, false, ltSINGLE, wcsLTEE, wcsRTEE ) ;
   wpBox = { boxULr, short(boxULc + boxWIDTH / 2) } ;
   wpBox = this->acDrawLine ( wpBox, (boxHEIGHT / 2 + 1), true, 
                              ltSINGLE, wcsTTEE, wcsINSECT ) ;
   this->acDrawLine ( wpBox, (boxHEIGHT / 2), true, ltSINGLE, L'\0', wcsBTEE ) ;

   //* Draw the same object using dual-line characters *
   this->acSetCursor ( wpMsg ) ;
   this->acEraseArea ( aesERASE_LINE ) ;
   this->acWrite ( wpMsg, L"And now, the same drawing using dual line characters. Press a key..." ) ;
   this->acRead () ;
   wpBox = { boxULr, boxULc } ;
   this->acClearArea ( wpBox, boxHEIGHT, boxWIDTH, L' ', boxBg ) ;
   wpBox = this->acDrawLine ( wpBox, boxWIDTH - 1, false, ltDUAL, wcsULd ) ;
   this->acDrawLine ( wpBox, boxHEIGHT, true, ltDUAL, wcsURd, wcsLRd ) ;
   wpBox = { short(boxULr + 1), boxULc } ;
   wpBox = this->acDrawLine ( wpBox, boxHEIGHT - 1, true, ltDUAL, L'\0', wcsLLd ) ;
   --wpBox.row ;
   this->acDrawLine ( wpBox, boxWIDTH, false, ltDUAL, wcsLLd, wcsLRd ) ;
   wpBox = { short(boxULr + boxHEIGHT / 2), boxULc } ;
   this->acDrawLine ( wpBox, boxWIDTH, false, ltDUAL, wcsLTEEd, wcsRTEEd ) ;
   wpBox = { boxULr, short(boxULc + boxWIDTH / 2) } ;
   wpBox = this->acDrawLine ( wpBox, (boxHEIGHT / 2 + 1), true, 
                              ltDUAL, wcsTTEEd, wcsINSECTd ) ;
   this->acDrawLine ( wpBox, (boxHEIGHT / 2), true, ltDUAL, L'\0', wcsBTEEd ) ;

   //* Reset attributes to terminal defaults and say "Goodnight, Gracie". *
   this->acSetAttributes ( acAttr(acaFG_DFLT | acaBG_DFLT | acaCLEAR_MODS) );
   this->acSetCursor ( wpMsg ) ;
   this->acEraseArea ( aesERASE_LINE ) ;
   this->acWrite ( wpMsg, L" Well done, Recruits. Fast on the draw!" ) ;

   //* Restore buffered input and prepare to exit.*
   this->acBufferedInput ( true, termEcho ) ;
   this->acSetFg ( aesFG_DFLT ) ;
   this->acSetCursor ( wpExit ) ;

}  //* End Test_Line() *

//*************************
//*     twColorChart      *
//*************************
//********************************************************************************
//* PRIVATE METHOD - CALLED ONLY BY Test_Window().                               *
//* Display a chart of color attributes available for testing the window         *
//* attribute settings.                                                          *
//*                                                                              *
//* Input  : basePos : base position for chart                                   *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::twColorChart ( const WinPos& basePos )
{
   const wchar_t *spaceBar = L"            \n" ;
   const short loopCNT = 17 ;                // iterations
   gString gsOut( "   4-Bit Color\n" ) ;     // text formatting
   WinPos wp = basePos ;                     // cursor position

   //** 4-Bit Color **
   short startIndx = aesBG_BLACK ;           // loop initial
   short endIndx = startIndx + loopCNT ;     // loop termination
   for ( short regnum = ZERO ; regnum < min8BIT ; ++regnum )
      gsOut.append( "%2hd\n", &regnum ) ;
   this->acWrite ( wp, gsOut ) ;
   wp = { short(wp.row + 1), short(wp.col + 3) } ;
   this->acSetCursor ( wp ) ;
   for ( short regindx = aesBG_BLACK ; regindx < endIndx ; ++regindx )
   {
      this->acSetBg ( aeSeq(regindx), true ) ; // set the background color
      this->ttyWrite ( spaceBar ) ;
      this->acSetCursor ( aesCUR_ABSCOL, wp.col ) ; // move cursor to target column
   }
   this->acSetBg ( aesFG_DFLT ) ;            // return to default background

   //** 8-Bit Color **
   const short ABEX = 19 ;
   const uint8_t AteBits[ABEX] = { 16, 21, 27, 28, 33, 52, 57, 64, 
                                   69, 82, 87, 124, 129, 141, 196, 201, 202, 213,
                                   226//, 231
                                   } ;
   wp = { short(basePos.row - 1), short(basePos.col + 16) } ;
   gsOut.compose( "   8-bit Color\n"
                  "   (examples)\n" ) ;
   WinPos wpLegend = wp = this->acWrite ( wp, gsOut ) ;
   gsOut.clear() ;
   wp.col += 3 ;
   this->acSetCursor ( wp ) ;
   for ( short atebit = ZERO ; atebit < ABEX ; ++atebit )
   {
      gsOut.append( "%3hhu\n", &AteBits[atebit] ) ;
      this->acSet8bitBg ( AteBits[atebit] ) ;   // set the background color
      this->ttyWrite ( spaceBar ) ;
      this->acSetCursor ( aesCUR_ABSCOL, wp.col ) ; // move cursor to target column
   }
   this->acSetBg ( aesBG_DFLT ) ;            // return to default background
   gsOut.append( "   Range:16-231" ) ;
   this->acWrite ( wpLegend, gsOut ) ;

   //** RGB Color **
   const uint8_t  wsrgbREDD_MAX  = wsrgbGREEN - 1 ;
   const uint8_t  wsrgbGREE_MAX  = wsrgbBLUE - 1 ;
   const uint8_t  wsrgbBLUE_MAX  = wsrgbBROWN - 1 ;
   const uint8_t  wsrgbBROW_MAX  = wsrgbMAGENTA - 1 ;
   const uint8_t  wsrgbMAGE_MAX  = wsrgbCYAN - 1 ;
   const uint8_t  wsrgbCYAN_MAX  = wsrgbGREY - 1 ;
   const uint8_t  wsrgbGREY_MAX  = wsrgbMAX ;
   const uint8_t  wsrgbBlack = wsrgbBLACK ;
   const uint8_t  wsrgbMax   = wsrgbMAX ;

   gString gs ;               // selector formatting
   acaExpand acaEx ;          // calculate RGB register values
   WebSafeRGB hue ;           // RGB color group
   uint8_t    shade = ZERO ;  // "shade" within the specified RGB color group (hue)
   wchar_t    userSel ;       // character for user selection of RGB attribute

   wp = { short(basePos.row - 1), short(wp.col + 13) } ;
   gsOut.compose( "   Web-safe RGB\n"
                  "   (examples)\n"
                  "%3hhu\n"
                  "%3hhu\n"
                  "%3hhu\n"
                  "%3hhu\n"
                  "%3hhu\n"
                  "%3hhu\n"
                  "%3hhu\n"
                  "%3hhu\n"
                  "   Range:%hhu-%hhu\n",
                  &wsrgbBlack,    &wsrgbREDD_MAX,  &wsrgbGREE_MAX, &wsrgbBLUE_MAX,
                  &wsrgbBROW_MAX, &wsrgbMAGE_MAX, &wsrgbCYAN_MAX, &wsrgbGREY_MAX,
                  &wsrgbBlack, &wsrgbMax ) ;
   this->acWrite ( wp, gsOut ) ;
   wp = { short(wp.row + 2), short(wp.col + 3) } ;
   this->acSetCursor ( wp ) ;
   this->acSetFg ( aesFGb_GREY ) ;

   for ( short r = ZERO ; r <= 7 ; ++r )
   {
      switch ( r )
      {
         case 0:  formatRgbString ( gsOut, 13, wsrgbBLACK ) ;
                  hue = wsrgbRED ;     userSel = L'a' ;     break ;
         case 1:  formatRgbString ( gsOut, 13, wsrgbREDD_MAX ) ;
                  hue = wsrgbRED ;     userSel = L'b' ;     break ;
         case 2:  formatRgbString ( gsOut, 13, wsrgbGREE_MAX ) ;
                  hue = wsrgbGREEN ;   userSel = L'c' ;     break ;
         case 3:  formatRgbString ( gsOut, 13, wsrgbBLUE_MAX ) ;
                  hue = wsrgbBLUE ;    userSel = L'd' ;     break ;
         case 4:  formatRgbString ( gsOut, 13, wsrgbBROW_MAX ) ;
                  this->acSetFg ( aesFG_DFLT ) ; userSel = L'e' ; hue = wsrgbBROWN ;   break ;
         case 5:  formatRgbString ( gsOut, 13, wsrgbMAGE_MAX ) ;
                  hue = wsrgbMAGENTA ; userSel = L'f' ;     break ;
         case 6:  formatRgbString ( gsOut, 13, wsrgbCYAN_MAX ) ;
                  hue = wsrgbCYAN ;    userSel = L'g' ;     break ;
         case 7:  formatRgbString ( gsOut, 13, wsrgbGREY_MAX ) ;
                  hue = wsrgbGREY ;    userSel = L'h' ;     break ;
      }
      gs.compose( L"%C: ", &userSel ) ;
      gsOut.insert( gs.gstr(), 1 ) ;
      this->acSetWebBg ( hue, shade ) ;         // set background attribute
      this->ttyWrite ( gsOut ) ;
      this->acSetCursor ( aesCUR_ABSCOL, wp.col ) ; // move cursor to target column
      shade = wsrgbSHADE_MAX ;   // maximum shade for target color group (hue)
   }

   this->acSetFgBg ( aesFG_DFLT, aesBG_DFLT ) ; // return to default fgnd/bkgnd

}  //* End twColorChart() *

//*************************
//*     Test_Sandbox      *
//*************************
//********************************************************************************
//* Create ad hoc tests for the AnsiCmd class.                                   *
//* For those who were too lazy to take Latin in high school, "ad hoc" means     *
//* "created or done for a particular purpose as necessary."                     *
//*                                                                              *
//* Input  : parmA : caller defined                                              *
//*          parmB : caller defined                                              *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_Sandbox ( wchar_t parmA, wchar_t parmB )
{
   gString gsOut( "Welcome to the Sandbox!\n" ) ;
   this->ttyWrite ( gsOut ) ;

   //* Create Tests Here... *









   //* --- End of Sandbox Area - Do Not Modify This Line --- **
   this->acSetCursor ( aesCUR_ABSPOS, (this->termRows - 1), 1 ) ;

}  //* End Test_Sandbox() *

//*************************
//*   Test_ReportCurPos   *
//*************************
//********************************************************************************
//* FOR DEBUGGING ONLY:                                                          *
//* Called only by the debugging methods.                                        *
//* Get the current cursor position and report the positon.                      *
//* Then return the cursor position to the REPORTED position.                    *
//*                                                                              *
//* Input  : row  : row for position report                                      *
//*          col  : column for position report                                   *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::Test_ReportCurPos ( short row, short col, WinPos *wpos )
{
   //* Get the current cursor position and report it *
   WinPos wp = this->acGetCursor () ;
   gString gsOut( "Cursor Pos: %hd,%hd", &wp.row, &wp.col ) ;
   gsOut.padCols( 20 ) ;
   this->acWrite ( row, col, gsOut ) ;

   //* If caller provided a WinPos object, return cursor coordinates *
   if ( wpos != NULL )
   {
      wpos->row = wp.row ;
      wpos->col = wp.col ;
   }

   this->acSetCursor ( wp ) ;             // return cursor to reported position

}  //* End Test_ReportCurPos() *

//*************************
//*    acDumpTermInfo     *
//*************************
//********************************************************************************
//* FOR DEBUGGING ONLY:                                                          *
//* Display debugging information for terminal I/O settings                      *
//*                                                                              *
//* Input  : tios : (by reference) data to be formatted and displayed            *
//*          row  : row to begin display                                         *
//*          col  : column to begin display                                      *
//*          verb : (optional, 0 by default) level of verbosity                  *
//*                 0 == report decoded bit fields only (default)                *
//*                      This includes the NCCS macro (c_cc array elements),     *
//*                      c_cc[VTIME] and c_cc[VMIN]                              *
//*                 1 == report binary flags only                                *
//*                 2 == report decoded bit fields AND the binary flags          *
//*                 3 == report only class members that track term settings      *
//*                 4 == report class members, binary flags and decoded fields   *
//*          gstit: (optional, null pointer by default)                          *
//*                 if specified, this is a pointer to a gString object          *
//*                 containing a title to be displayed for the data dump         *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Notes:                                                                       *
//* 1) 'csize' bits: 0x00==5 bits, 0x01==6 bits, 0x02==7 bits, 0x03==8 bits.     *
//*    Obviously, they never heard of wchar_t characters i.e. byte stream only.  *
//*                                                                              *
//********************************************************************************

void AnsiCmd::acDumpTermInfo ( const TermIos& tios, short row, short col, 
                               short verb, const gString *gstit )
{
   gString gsOut, gsBin ;     // text formatting
   WinPos wp( row, col ) ;     // text position
   short clrRows,             // rows in cleared area
         clrCols ;            // columns in cleared area

   this->acSetCursorStyle ( csHide ) ;    // make cursor invisible

   //* Clear the target area *
   switch ( verb )
   {
      case 0:  clrRows = 12 ; clrCols = 48 ; break ;
      case 1:  clrRows =  4 ; clrCols = 48 ; break ;
      case 2:  clrRows = 16 ; clrCols = 48 ; break ;
      case 3:  clrRows =  3 ; clrCols = 41 ; break ;
      case 4:
      default: clrRows = 19 ; clrCols = 48 ; break ;
   } ;
   if ( gstit != NULL )       // add a row for the title
      ++clrRows ;
   this->acClearArea ( row, col, clrRows, clrCols ) ;

   //* If specified, display the title for the dump *
   if ( gstit != NULL )
      this->acWrite ( wp.row++, wp.col, *gstit, false ) ;

   //* If specified, format and display the bit fields.*
   if ( (verb == 1) || (verb == 2) || (verb == 4) )
   {
      gsBin.compose( L"-----------------%hB", &tios.c_iflag ) ;
      gsOut.compose( "c_iflag: %S\n", gsBin.gstr() ) ;
      wp = this->acWrite ( wp, gsOut, false ) ;

      gsBin.compose( L"-----------------%hB", &tios.c_oflag ) ;
      gsOut.compose( "c_oflag: %S\n", gsBin.gstr() ) ;
      wp = this->acWrite ( wp, gsOut, false ) ;

      gsBin.compose( L"-----------------%hB", &tios.c_cflag ) ;
      gsOut.compose( "c_cflag: %S\n", gsBin.gstr() ) ;
      wp = this->acWrite ( wp, gsOut, false ) ;

      gsBin.compose( L"-----------------%hB", &tios.c_lflag ) ;
      gsOut.compose( "c_lflag: %S\n", gsBin.gstr() ) ;
      wp = this->acWrite ( wp, gsOut, false ) ;
   }

   //* Display the interesting flags in human-readable format.*
   if ( (verb == 0) || (verb == 2) || (verb == 4) )
   {
      const wchar_t * en = L"enabled " ;
      const wchar_t * di = L"disabled" ;
      short csize  = short(((tios.c_cflag & CSIZE) >> 4) + 5) ; // (see note above)
      bool istrip  = (tios.c_iflag & ISTRIP),
           opost   = (tios.c_oflag & OPOST),
           onlcr   = (tios.c_oflag & ONLCR),
           xtabs   = (tios.c_oflag & XTABS),
           cread   = (tios.c_cflag & CREAD),
           icanon  = (tios.c_lflag & ICANON),
           echo    = (tios.c_lflag & ECHO),
           echonl  = (tios.c_lflag & ECHONL),
           echoctl = (tios.c_lflag & ECHOCTL),
           iexten  = (tios.c_lflag & IEXTEN) ;

      gsOut.compose( "istrip : 7-bit input                 %S\n"
                     "opost  : output flags                %S\n"
                     "onlcr  : convert newlines to cr/lf   %S\n"
                     "xtabs  : convert tabs to spaces      %S\n"
                     "cread  : read terminal input         %S\n"
                     "csize  : character width             %hd bits\n"
                     "icanon : enable canonical input      %S\n"
                     "echo   : echo character input        %S\n"
                     "echonl : echo newlines               %S\n"
                     "echoctl: echo control chars as (^x)  %S\n"
                     "iexten : posix extensions            %S\n",
                     (istrip ? en : di),
                     (opost ? en : di),
                     (onlcr ? en : di),
                     (xtabs ? en : di),
                     (cread ? en : di),
                     &csize,
                     (icanon ? en : di),
                     (echo ? en : di),
                     ((echonl && icanon) ? en : di),
                     (echoctl ? en : di),
                     (iexten ? en : di) ) ;
      this->acWrite ( wp, gsOut ) ;
      wp.row += (gstit == NULL ? 10 : 11) ;

      //* Array of control characters *
      short nccs  = NCCS,
            vtime = VTIME,
            vmin  = VMIN ;
      gsOut.compose( "c_cc[%hd] c_cc[VTIME==%hd]:%hhXh c_cc[VMIN==%hd]:%hhXh\n",
                     &nccs, &vtime, &tios.c_cc[VTIME], &vmin, &tios.c_cc[VMIN] ) ;
      wp = this->acWrite ( wp, gsOut ) ;
   }

   if ( (verb == 3) || (verb == 4) )
   {
      char seOpt[16] ;
      wchar_t seoptChar = (this->tset->inpEch == softEchoC ? L'c' :
                           this->tset->inpEch == softEchoD ? L'd' :
                           this->tset->inpEch == softEchoE ? L'e' : L'a' ) ;
      gsOut.compose( "s%C", &seoptChar ) ;
      gsOut.copy( seOpt, 16 ) ;
      gsOut.compose( "tset->inpBuf  : %hhd   tset->inpEch  : %hd(%s)\n"
                     "tset->inpBlk  : %hhd   tset->cchType : %hd(%s)\n"
                     "this->curType : %hd(%s)\n",
                     &this->tset->inpBuf,
                     &this->tset->inpEch, 
                     (this->tset->inpEch == termEcho ? "t" : 
                      this->tset->inpEch == noEcho ? "n" : seOpt),
                     &this->tset->inpBlk,
                     &this->tset->cchType,
                     (this->tset->cchType == cchtExit   ? "e" : 
                      this->tset->cchType == cchtUser   ? "u" : 
                      this->tset->cchType == cchtIgnore ? "i" : "t"),
                     &this->tset->curStyle,
                     (this->tset->curStyle == csBblock ? "bb" : 
                      this->tset->curStyle == csSblock ? "sb" : 
                      this->tset->curStyle == csBuline ? "bu" :
                      this->tset->curStyle == csSuline ? "su" : 
                      this->tset->curStyle == csBvbar  ? "bv" : 
                      this->tset->curStyle == csSvbar  ? "sv" : "d")
                   ) ;
      this->acWrite ( wp, gsOut ) ;
   }

this->acSetCursorStyle ( csShow ) ;    // make cursor visible

}  //* End acDumpTermInfo() *

//*************************
//*     acaExpandDump     *
//*************************
//********************************************************************************
//* FOR DEBUGGING ONLY:                                                          *
//* Dump contents of a acaExpand object -- for debugging only.                   *
//*                                                                              *
//* Input  : da   : (by reference) acaExpand object to be displayed              *
//*          row  : row to begin display                                         *
//*          col  : column to begin display                                      *
//*          gstit: (optional, null pointer by default)                          *
//*                 if specified, this is a pointer to a gString object          *
//*                 containing a title to be displayed for the data dump         *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::acaExpandDump ( const acaExpand& da, short row, short col, 
                              const gString *gstit )
{
   const short DUMP_ROWS = 12,
               DUMP_COLS = 35 ;
   gString gsOut ;                     // text formatting
   WinPos  wp( row, col ) ;            // text position

   //* Clear the target area *
   this->acClearArea ( row, col, DUMP_ROWS, DUMP_COLS ) ;

   if ( gstit != NULL )                // if title specified
   {
      gsOut = gstit->gstr() ;          // working copy of title
      gsOut.limitCols( DUMP_COLS ) ;   // truncate if necessary
      gsOut.padCols( DUMP_COLS, SPACE, true ) ;
      // Programmer's Note: We write these ANSI escape sequences directly, so 
      // we don't interfere with the tests in progress.
      this->ttyWrite ( ansiSeq[aesUNDERLINE] ) ;
      this->acWrite ( wp.row++, wp.col, gsOut ) ;
      this->ttyWrite ( ansiSeq[aesUNDERLINE_OFF] ) ;
   }

   gsOut.compose(
         "acaVal  :%08X  fgBits :%08X\n"
         "allFlags:%08X      fgIndex:  %02hhX\n"
         "modFlags:%08X  bgBits :%08X\n"
         "fgDflt   :%hhd bgDflt  :%hhd bgIndex:%02hhX\n"
         "boldMod  :%hhd ulineMod:%hhd invisMod:%hhd\n"
         "italicMod:%hhd olineMod:%hhd revMod  :%hhd\n"
         "xoutMod  :%hhd blinkMod:%hhd clrMods :%hhd\n",
         &da.acaVal,    &da.fgBits,
         &da.allFlags,  &da.fgIndex,
         &da.modFlags,  &da.bgBits,
         &da.fgDflt,    &da.bgDflt,   &da.bgIndex,
         &da.boldMod,   &da.ulineMod, &da.invisMod,
         &da.italicMod, &da.olineMod, &da.revMod,
         &da.xoutMod,   &da.blinkMod, &da.clrMods ) ;
   wp = this->acWrite ( wp, gsOut ) ;

   gsOut.compose(
         "fgRgb    :%hhd:%02hhX,%02hhX    bgRgb:%hhd:%02hhX,%02hhX\n"
         "FgType   :%3hd(%02hXh)  BgType:%3hd(%02hXh)\n"
         "aesFgnd  :%3hd(%02hXh) aesBgnd:%3hd(%02hXh)\n"
         "rgbRegs  :%02hhX,%02hhX,%02hhX rgbRegs:%02hhX,%02hhX,%02hhX\n",
         &da.fgRgb, &da.fgHue, &da.fgShade, &da.bgRgb, &da.bgHue, &da.bgShade, 
         &da.aesFgType, &da.aesFgType, &da.aesBgType, &da.aesBgType,
         &da.aesFgnd,   &da.aesFgnd,   &da.aesBgnd,   &da.aesBgnd,
         &da.rgbFgR,    &da.rgbFgG,    &da.rgbFgB,
         &da.rgbBgR,    &da.rgbBgG,    &da.rgbBgB ) ;
   wp = this->acWrite ( wp, gsOut ) ;

}  //* End acaExpandDump() *

//*************************
//*      skFormDump       *
//*************************
//********************************************************************************
//* Private Method: called by Test_Form.                                         *
//*                                                                              *
//*                                                                              *
//* Input  : skf  : (by reference) skForm object to be displayed                 *
//*          row  : row to begin display                                         *
//*          col  : column to begin display                                      *
//*          gstit: (optional, null pointer by default)                          *
//*                 if specified, this is a pointer to a gString object          *
//*                 containing a title to be displayed for the data dump         *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::skFormDump ( const skForm& skf, short row, short col, const gString* gstit )
{
   gString gsOut( "%S\n"
                  "fCnt:%hd fi:%hd ins:%hhd insc:%hhd togg:%hhd beep:%hhd \n",
                  (gstit != NULL ? gstit->gstr() : L"skForm Contents"),
                  &skf.fCnt, &skf.fi, &skf.ins, &skf.insc, &skf.togg, &skf.beep 
                ) ;
   for ( short f = ZERO ; f < skf.fCnt ; ++f )
   {
      gsOut.append( "  %hd) orig:%-2hd/%-2hd hgt:%-2hd wid:%-2hd ip:%-2hd\n"
                    "     cur :%-2hd/%-2hd aca:%04Xh ro:%hhd\n",
                    &f, 
                    &skf.fld[f].orig.row, 
                    &skf.fld[f].orig.col,
                    &skf.fld[f].hgt,
                    &skf.fld[f].wid,
                    &skf.fld[f].ip,
                    &skf.fld[f].cur.row,
                    &skf.fld[f].cur.col, 
                    &skf.fld[f].aca.acaVal,
                    &skf.fld[f].ro ) ;
      if ( (skf.fld[f].txt.gschars()) > 1 )
      {
         gsOut.append( L"     \"" ) ;
         gsOut.loadChars( skf.fld[f].txt.gstr(), 26, true ) ;
         gsOut.append( L"\"\n" ) ;
      }
   }

   this->acWrite ( row, col, gsOut ) ;

}  //* End skFormDump() *

//*************************
//*     acDumpMods        *
//*************************
//********************************************************************************
//* Private Method: called by acSetMod() or acResetMods().                       *
//* Display the text-modifier tracking flags.                                    *
//*                                                                              *
//* Input  : row  : row for position of display                                  *
//*          col  : column for position of display                               *
//*          title: (optional, null pointer by default)                          *
//*                 if specified, display title on first line of display         *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************
//* Programmer's Note:                                                           *
//* We make some kludgy moves in this method by scanning the provided title      *
//* text to determine the intended interpretation of: 'Bold' (bold vs. faint)    *
//* and 'Blink' (slow vs. fast) flags.                                           *
//*                                                                              *
//* Note that the 'Framed/Encircled' options are seldom if ever supported by     *
//* terminal emulators, so the attrBits object makes no attempt to support       *
//* those options. Instead, we identify them indirectly via the text of the      *
//* title.                                                                       *
//*                                                                              *
//********************************************************************************

void AnsiCmd::acDumpMods ( short row, short col, const wchar_t *title )
{
   gString gsTit( "Modifiers:" ) ;        // Set title
   if ( title != NULL )
   { gsTit = title ; gsTit.limitCols( 13 ) ; }
   bool fast = false, frame = false, circle = false ;

   //* Save the current modifiers *
   bool tmpBoldMod    = this->attrBits.boldMod,
        tmpItalicMod  = this->attrBits.italicMod,
        tmpUlineMod   = this->attrBits.ulineMod,
        tmpOlineMod   = this->attrBits.olineMod,
        tmpSlineMod   = this->attrBits.xoutMod,
        tmpBlinkMod   = this->attrBits.blinkMod,
        tmpInvisMod   = this->attrBits.invisMod,
        tmpRevMod     = this->attrBits.revMod,
        tmpFrameMod   = false ; 
        //* Note: The acaExpand class does not support the obsolete "framed" *
        //* or "encircled" mods, so we query the title. (see note above)     *
        if ( (gsTit.find( L"Framed" )) >= ZERO )
           frame = true ;
        else if ( (gsTit.find( L"Encircle" )) >= ZERO )
           circle = true ;
        if ( frame || circle )
           tmpFrameMod = true ;

   //* Reset all text modifiers for output *
   this->acResetMods () ;

   gString gsOut( "%S\n"
                  "-------------\n"
                  "boldMod   : %hhd\n"
                  "italicMod : %hhd\n"
                  "ulineMod  : %hhd\n"
                  "olineMod  : %hhd\n"
                  "xoutMod   : %hhd\n"
                  "blinkMod  : %hhd\n"
                  "concealMod: %hhd\n"
                  "reverseMod: %hhd\n"
                  "frameMod  : %hhd\n",
                  gsTit.gstr(),
                  &tmpBoldMod, &tmpItalicMod, 
                  &tmpUlineMod, &tmpOlineMod, &tmpSlineMod, 
                  &tmpBlinkMod, &tmpInvisMod, &tmpRevMod, 
                  &tmpFrameMod ) ;

   this->acWrite ( row, col, gsOut ) ;

   //*********************************
   //* Restore caller's mod settings *
   //*********************************
   //*The 'boldMod' flag can indicate either Bold or Faint, but the Faint mod *
   //* is not implemented in any of our test platforms. (see note above)      *
   if ( tmpBoldMod )
   {
      if ( (gsTit.find( L"faint" )) < ZERO )
         this->acSetMod ( aesBOLD ) ;
   }
   //* The 'blinkMod' flag can indicate either Fast Blink or Slow Blink, but *
   //* our test platforms have only one blink speed. (see note above)        *
   if ( tmpBlinkMod )
   {
      fast = bool((gsTit.find( L"fast" )) >= ZERO) ;
      this->acSetMod ( fast ? aesBLINK_FAST : aesBLINK_SLOW ) ;
   }
   //* The 'Framed' and 'Encircled' options share the tmpFrameMod flag. *
   //* Determine which was used.                                        *
   if ( tmpFrameMod )
   {
      this->acSetMod ( frame ? aesFRAMED : aesENCIRCLE ) ;
   }

   if ( tmpItalicMod )     this->acSetMod ( aesITALIC ) ;
   if ( tmpUlineMod )      this->acSetMod ( aesUNDERLINE ) ;
   if ( tmpOlineMod )      this->acSetMod ( aesOVERLINE ) ;
   if ( tmpSlineMod )      this->acSetMod ( aesXOUT ) ;
   if ( tmpInvisMod )      this->acSetMod ( aesCONCEAL ) ;
   if ( tmpRevMod )        this->acSetMod ( aesREVERSE ) ;
   if ( tmpFrameMod )      this->acSetMod ( aesFRAMED ) ;

}  //* End acDumpMods() *

//*************************
//*     acCaptureAttr     *
//*************************
//********************************************************************************
//* Private Method: called by Test_4Bit()                                        *
//* Capture the current state of the color-attribute settings.                   *
//*                                                                              *
//* Input  : gs   : (by reference) receives the formatted attribute data         *
//*          data : specify which data to report                                 *
//*                 1 : 4-bit foreground and background only                     *
//*                 2 : 8-bit foreground and background indices only             *
//*                 3 : RGB foreground and background only                       *
//*                 4 : All options: 1 + 2 + 3                                   *
//*          nl   : 'true'  append a newline character to output                 *
//*                 'false' no newline character                                 *
//*          title: (optional, false by default) if specified format the         *
//*                 title text corresponding to 'data' value                     *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::acCaptureAttr ( gString& gs, short data, bool nl, bool title )
{
   const short NAME_ITEMS = 17 ;
   const wchar_t *FgNames[NAME_ITEMS] = 
   {
      L"aesFG_BLACK    ",
      L"aesFG_RED      ",
      L"aesFG_GREEN    ",
      L"aesFG_BROWN    ",
      L"aesFG_BLUE     ",
      L"aesFG_MAGENTA  ",
      L"aesFG_CYAN     ",
      L"aesFG_GREY     ",
      L"aesFGb_BLACK   ",
      L"aesFGb_RED     ",
      L"aesFGb_GREEN   ",
      L"aesFGb_BROWN   ",
      L"aesFGb_BLUE    ",
      L"aesFGb_MAGENTA ",
      L"aesFGb_CYAN    ",
      L"aesFGb_GREY    ",
      L"aesFG_DFLT     ",
   } ;
   const wchar_t *BgNames[NAME_ITEMS] = 
   {
      L"aesBG_BLACK    ",
      L"aesBG_RED      ",
      L"aesBG_GREEN    ",
      L"aesBG_BROWN    ",
      L"aesBG_BLUE     ",
      L"aesBG_MAGENTA  ",
      L"aesBG_CYAN     ",
      L"aesBG_GREY     ",
      L"aesBGb_BLACK   ",
      L"aesBGb_RED     ",
      L"aesBGb_GREEN   ",
      L"aesBGb_BROWN   ",
      L"aesBGb_BLUE    ",
      L"aesBGb_MAGENTA ",
      L"aesBGb_CYAN    ",
      L"aesBGb_GREY    ",
      L"aesBG_DFLT     ",
   } ;
   const wchar_t *Title1 = L"%S4-Bit Foregnd  %S  %S4-Bit Backgnd  %S" ;
   const wchar_t *Title2 = L"%SFg Index%S  %SBg Index%S" ;
   const wchar_t *Title3 = L"%SR-- G-- B--%S  %SR-- G-- B--%S" ;
   const char    *Title4 = "%S  %S  %S" ;
   const char    *data1Template = "%S  %S" ;
   const char    *data2Template = "  %3hd       %3hd   " ;
   const char    *data3Template = "%-3hd %-3hd %-3hd  %-3hd %-3hd %-3hd" ;
   const char    *data4Template = "%s  %s  %s" ;
   gString gsTemplate ;                // concatenation template
   short fgIndex = (NAME_ITEMS - 1),   // reference "Default"
         bgIndex = (NAME_ITEMS - 1) ;  // reference "Default"

   //* Return formatted column titles *
   if ( title != false )
   {
      if ( data == 1 )
         gs.compose( Title1, ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF],
                             ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF] ) ;
      else if ( data == 2 )
         gs.compose( Title2, ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF],
                             ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF] ) ;
      else if ( data == 3 )
         gs.compose( Title3, ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF],
                             ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF] ) ;
      else
      {
         gsTemplate.compose( Title4, Title1, Title2, Title3 ) ;
         gs.compose( gsTemplate.gstr(), 
                     ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF],
                     ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF],
                     ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF],
                     ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF],
                     ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF],
                     ansiSeq[aesUNDERLINE], ansiSeq[aesUNDERLINE_OFF] ) ;
      }
   }

   //* Return formatted report of attribute flags *
   else
   {
      //* Set the index into the description arrays *
      if ( this->attrBits.aesFgnd == aesFG_DFLT )
         fgIndex = NAME_ITEMS - 1 ;

      else if ( (this->attrBits.aesFgnd >= aesFG_BLACK) &&
              (this->attrBits.aesFgnd < aesFG_DFLT) )
      { fgIndex = this->attrBits.aesFgnd - aesFG_BLACK ; }

      if ( this->attrBits.aesBgnd == aesBG_DFLT )
         bgIndex = NAME_ITEMS - 1 ;
      else if ( (this->attrBits.aesBgnd >= aesBG_BLACK) &&
              (this->attrBits.aesBgnd < aesBG_DFLT) )
      { bgIndex = this->attrBits.aesBgnd - aesBG_BLACK ; }

      if ( data == 1 )           // display 4-bit foreground/background data
      {
         gs.compose( data1Template, FgNames[fgIndex], BgNames[bgIndex] ) ;
      }
      else if ( data == 2 )      // display 8-bit table lookup for fg/bg and/or greyscale
      {
         #if 0    // acCaptureAttr(2) - Currently Unused
         gs.compose( data2Template, &this->fgnd.attrIndex, &this->bgnd.attrIndex ) ;
         #endif   // U/C
      }
      else if ( data == 3 )      // display R/G/B register settings
      {
         #if 0    // acCaptureAttr(3) - Currently Unused
         gs.compose( data3Template,
                     &this->fgnd.rgbR, &this->fgnd.rgbG, &this->fgnd.rgbB, 
                     &this->bgnd.rgbR, &this->bgnd.rgbG, &this->bgnd.rgbB ) ;
         #endif   // U/C
      }
      else                       // display all data
      {
         gsTemplate.compose( data4Template, data1Template, 
                             data2Template, data3Template ) ;
         #if 0    // acCaptureAttr(4) - Currently Unused
         gs.compose( gsTemplate.gstr(), 
                     FgNames[fgIndex], BgNames[bgIndex], 
                     &this->fgnd.attrIndex, &this->bgnd.attrIndex, 
                     &this->fgnd.rgbR, &this->fgnd.rgbG, &this->fgnd.rgbB, 
                     &this->bgnd.rgbR, &this->bgnd.rgbG, &this->bgnd.rgbB ) ;
         #endif   // U/C
      }
   }

   if ( nl )
      gs.append( L'\n' ) ;

}  //* End acCaptureAttr() *

//*************************
//*    formatRgbString    *
//*************************
//********************************************************************************
//* Non-member Method:                                                           *
//*                                                                              *
//* Format display text containing RGB register indices. Called by test          *
//* methods which display RGB indices.                                           *
//*                                                                              *
//* Input  : gsOut   : (by reference)                                            *
//*          colWidth: column width of constructed string                        *
//*          webIndex: index indicating R/G/B register combination used for      *
//*                    call to web2regs().                                       *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

static void formatRgbString ( gString& gsOut, short colWidth, uint8_t webIndex )
{
   acaExpand acax ;                    // instance of class for bitfield parsing

   acax.web2regs( webIndex, false ) ;  // get RGB register combination
   gsOut.compose( " %hhu;%hhu;%hhu",   // format the data string
                  &acax.rgbBgR, &acax.rgbBgG, &acax.rgbBgB ) ;
   gsOut.padCols( colWidth ) ;         // pad to column width
   gsOut.append( L'\n' ) ;             // cursor ends at column 1 of next row

}  //* End formatRgbString() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//* Input  : none                                                                *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************


//** Compilation of the entire module is controlled by this statement. **
#endif   // DEBUG_ANSICMD
