//********************************************************************************
//* File       : AnsiCmdSkf.cpp                                                  *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2022-2023 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 15-Aug-2023                                                     *
//* Version    : (see AnsiCmdVersion string in AnsiCmd.cpp)                      *
//*                                                                              *
//* Description: The methods in this module manage the functionality and data    *
//*              within the skForm class. The skForm class and the associated    *
//*              skField class are defined in AnsiCmdDef.hpp.                    *
//*                                                                              *
//* The skForm class defines a user interface consisting of an array of          *
//* interlocking text input fields. Each field within the skForm object is       *
//* defined as an skField-class object.                                          *
//*                                                                              *
//* Editing of text within the fields is handled by the ACWin-class methods.     *
//*    EditForm() or                                                             *
//*    EditField()                                                               *
//* It is possible (though not recommended) to write an application-level method *
//* to edit the fields outside an ACWin object. See the Test_SoftEcho() method   *
//* in AnsiCmdTest.cpp for some examples of using the skForm object within an    *
//* "invisible" ACWin object.                                                    *
//*                                                                              *
//* Certain "special" keys are defined for editing within a field:               *
//*    wcsUARROW  Up-Arrow key                                                   *
//*    wcsDARROW  Down-Arrow key                                                 *
//*    wcsLARROW  Left-Arrow key                                                 *
//*    wcsRARROW  Right-Arrow key                                                *
//*    wcsHOME    Home key                                                       *
//*    wcsEND     End key                                                        *
//*    wcsBKSP    Backspace key                                                  *
//*    wcsDELETE  Delete key                                                     *
//*    wcsINSERT  Insert/Overstrike key                                          *
//*                                                                              *
//*    [not yet implemented]                                                     *
//*    wcsCOPY    Copy field text to the clipboard.                              *
//*    wcsPASTE   Paste text from the clipboard into the field.                  *
//*                                                                              *
//* Moving among the fields of the skForm is implemented by additional           *
//* "special" keycodes:                                                          *
//*    wcsTAB     Tab key                                                        *
//*    wcsSTAB    Shift+Tab key                                                  *
//*    wcsPGUP    Page-Up key                                                    *
//*    wcsPGDN    Page-Up key                                                    *
//*                                                                              *
//* Other user-interface widgets may be added in future releases.                *
//*                                                                              *
//********************************************************************************

//*****************
//* Include Files *
//*****************
#include "AnsiCmd.hpp"     //* AnsiCmd-class definition

//***************
//* Definitions *
//***************
#if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0
extern ofstream ofsdbg ;
extern gString  gsdbg ;
#endif   // DEBUG_ANSICMD && DEBUG_LOG

//**************
//* Local data *
//**************

//********************
//* Local prototypes *
//********************


//*************************
//*   acSpecialKeyTest    *
//*************************
//********************************************************************************
//* Determine whether the specified keycode is one of the "special" keys, AND    *
//* whether that keycode is a member of the active soft-echo sub-group.          *
//*                                                                              *
//* The test is performed against the soft-echo option specified by the          *
//* previous terminal setup, (but see the 'eopt' argument).                      *
//*                                                                              *
//* The available character-echo options are members of enum EchoOpt. Please     *
//* refer to the documentation for the TermConfig class, and specifically the    *
//* 'echoOption' member variable for terminal setup information.                 *
//*                                                                              *
//* NOTE: This method should not be called when the echo option is not one       *
//*       of the soft-echo options, but if it is, it will only result in a       *
//*       harmess waste of CPU cycles.                                           *
//*                                                                              *
//* Each of the soft-echo options encompasses a subset of the supported          *
//* "special" keycodes. Please see the documentation for a list of keycodes      *
//* included in each of the soft-echo options.                                   *
//*                                                                              *
//* Input  : wkey : (by reference) keycode to be tested                          *
//*                                                                              *
//*          alert: (optional, 'false' by default) audible alert                 *
//*                 'true'  execute a terminal beep if keycode under test        *
//*                         IS one of the supported soft-echo keycodes, but      *
//*                         IS NOT not included in the active soft-echo group.   *
//*                 'false' do not execute an audible alert                      *
//*          eopt : (optional, null pointer by default)                          *
//*                 If specified, this is a pointer to an alternate soft-echo    *
//*                 option to be used for the test.                              *
//*                    (used internally for special-key verification)            *
//*                                                                              *
//* Returns: 'true'  if key is one of the ACTIVE special keys                    *
//*          'false' if key is not one of the active special keys                *
//*                  If the caller's keycode is a Special key that IS NOT a      *
//*                  member of the active group. the keycode will be set to      *
//*                  the null character (NULLCHAR) on return.                    *
//********************************************************************************
//* Programmer's Note: In a future release, we could enable a subset of the      *
//* the "Special" keycodes for the 'noEcho' options.                             *
//*                                                                              *
//********************************************************************************

bool AnsiCmd::acSpecialKeyTest ( wchar_t& wkey, bool alert, EchoOpt *eopt ) const
{
   //* Local copy of echo option *
   EchoOpt localOpt = eopt != NULL ? *eopt : this->tset->inpEch ;
   bool isSpecKey = false ;               // return value

   switch ( localOpt )
   {
      case softEchoA:      // all soft-echo keys
         switch ( wkey )
         {
            case wcsUARROW:   case wcsDARROW:   case wcsRARROW:   case wcsLARROW:
            case wcsHOME:     case wcsEND:      case wcsTAB:      case wcsSTAB:
            case wcsPGUP:     case wcsPGDN:
            case wcsBKSP:     case wcsDELETE:   case wcsINSERT:   case wcsAENTER:
            isSpecKey = true ;  break ;
            default:            break ;
         } ;
         break ;
      case softEchoC:      // soft-echo cursor keys
         switch ( wkey )
         {
            case wcsUARROW:   case wcsDARROW:   case wcsRARROW:   case wcsLARROW:
            case wcsHOME:     case wcsEND:      case wcsTAB:      case wcsSTAB:
            case wcsPGUP:     case wcsPGDN:
               isSpecKey = true ;  break ;
            case wcsBKSP:     case wcsDELETE:   case wcsINSERT:
               wkey = NULLCHAR ;   break ;
            default:               break ;
         } ;
         break ;
      case softEchoD:      // all soft-echo keys disabled
         switch ( wkey )
         {
            case wcsUARROW:   case wcsDARROW:   case wcsRARROW:   case wcsLARROW:
            case wcsHOME:     case wcsEND:      case wcsPGUP:     case wcsPGDN:
            case wcsBKSP:     case wcsDELETE:   case wcsTAB:      case wcsSTAB:
            case wcsINSERT:
               wkey = NULLCHAR ;    break ;
            default:                break ;
         }
      case softEchoE:      // soft-echo edit keys
                          // (this group is useful mostly for testing)
         switch ( wkey )
         {
            case wcsBKSP:     case wcsDELETE:   case wcsTAB:      case wcsSTAB:
            case wcsINSERT:   case wcsPGUP:     case wcsPGDN:     case wcsAENTER:
               isSpecKey = true ;  break ;
            case wcsUARROW:   case wcsDARROW:   case wcsRARROW:   case wcsLARROW:
            case wcsHOME:     case wcsEND:
               wkey = NULLCHAR ;   break ;
            default:                break ;
         } ;
         break ;

      //* Echo option is not a member of the soft-echo group.*
      //* See note above.                                    *
      case termEcho:
      case noEcho:
      default:
         break ;
   } ;

   //* PageUp and PageDown keys are currently       *
   //* converted to Shift+Tab and Tab, respectively.*
   if ( isSpecKey )
   {
      if ( wkey == wcsPGUP )        wkey = wcsSTAB ;
      else if ( wkey == wcsPGDN )   wkey = wcsTAB ;
   }

   //* Make noise if keycode IS a special key, but *
   //* IS NOT a member of the active group.        *
   if ( alert && ! isSpecKey && (wkey == NULLCHAR) )
      this->acBeep () ;

   return isSpecKey ;

}  //* End acSpecialKeyTest() *

//*************************
//*       eotOffset       *
//*************************
//********************************************************************************
//* Protected Method:                                                            *
//* -----------------                                                            *
//* Calculate cursor offset for end-of-text within a field, or optionally,       *
//* the end-of-text on current row.                                              *
//*                                                                              *
//* Input  : fptr : pointer to target field                                      *
//*          wpptr: (optional, null pointer by default)                          *
//*                 For multi-row fields only, if 'wpptr' != NULL, then target   *
//*                 receives end position for current row.                       *
//*                                                                              *
//* Returns: zero-based offset from field origin                                 *
//********************************************************************************
//* Notes:                                                                       *
//* -- End-of-Text (EOT): This is defined as the position (character cell) in    *
//*    the field which represents the null character ('\0') which terminates     *
//*    the text string. This may or may not fall within the display area of      *
//*    the field because an skField object may contain more text than can be     *
//*    simultaneously displayed.                                                 *
//*    -- Scrolling of the data within a field may be implemented in a future    *
//*       release.                                                               *
//* -- Single-row fields: End-Of-Text is a fairly simple calculation.            *
//*    -- If the text is less than the width of the field, EOT is the position   *
//*       following the last displayed character.                                *
//*    -- If the text exactly fills the field, or if not all of the text can     *
//*       be displayed in the field, then the actual EOT is not available.       *
//*       In that case the position returned is over the last character          *
//*       displayed. For single-column characters, this is the last character    *
//*       cell (lower-left corner) in the field. For multi-column characters,    *
//*       this will be the cell containing the left edge of the character.       *
//*                                                                              *
//* -- Multi-row fields: The End-Of-Text calculation is based on several         *
//*    factors such as whether the text contains newline characters ('\n')       *
//*    and where the newlines are placed relative to the edges of the field.     *
//*    -- If the data are properly formatted to fit within the field, then a     *
//*       newline character will signal the line break for each line             *
//*       (except the line which contains the null terminator).                  *
//*    -- If the text contains no newlines, (or poorly-positioned newlines),     *
//*       then the character at which to break each row of text is determined    *
//*       by the number of display columns available for that row.               *
//*       -- For the current release, the data wrap to the next row occurs       *
//*          when all columns of that row have been filled, regardless of        *
//*          whether a word will be broken.                                      *
//*          -- An exception would be if the number of free columns at the end   *
//*             of a row are fewer than the columns needed to display the next   *
//*             character. In this case, the final column(s) of the row may      *
//*             be left empty.                                                   *
//*       -- In a future release, we may implement line breaks at whitespece     *
//*          and punctuation. See the author's NcDialog API algorithm for a      *
//*          line wrap algorithm that breaks the line at whitespace and          *
//*          punctuation.                                                        *
//*                                                                              *
//********************************************************************************

WinPos AnsiCmd::eotOffset ( skField *fptr, WinPos *wpptr ) const
{
   #define DEBUG_EOTOFF (0)               // for debugging only

   gString gs ;                           // text analysis
   WinPos wpOff( 0, 0 ) ;                 // return value
   short  txtCols = fptr->txt.gscols(),   // columns required by text data
          indx  = ZERO ;

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_EOTOFF != 0
   if ( (ofsdbg.is_open()) )
   {
      gsdbg.compose( "eotOffset(txtCols:%hd hgt:%hd wid:%hd)\n\"%S\"", 
                     &txtCols, &fptr->hgt, &fptr->wid, fptr->txt.gstr() ) ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_EOTOFF

   //* Single-row fields *
   if ( fptr->hgt == 1 )
   {
      if ( (wpOff.col = txtCols) >= fptr->wid )
         wpOff.col = fptr->wid - 1 ;
   }

   //* Multi-row fields *
   else
   {
      short chCnt,                        // character count
            colAcc = ZERO ;               // column-width accumulator
      const wchar_t *wptr = fptr->txt.gstr() ; // array of wide characters
      const short *colArray = fptr->txt.gscols( chCnt ) ; // array of character widths

      //* If end-of-row position requested *
      if ( wpptr != NULL )
      {
         gString gsParse ;
         this->parseFieldText ( fptr, gsParse, false ) ;

         //* Isolate the row which contains the IP.*
         for ( short r = ZERO ; r < fptr->hgt ; ++r )
         {
            if ( r < fptr->cur.row )         // discard data for leading rows
            {
               if ( (indx = gsParse.after( NEWLINE )) >= ZERO )
               {
                  gsParse.shiftChars( -(indx) ) ;
               }
            }
            else
            {
               //* Truncate data for trailing rows *
               if ( (indx = gsParse.find( NEWLINE )) >= ZERO )
                  gsParse.limitChars( indx ) ;
               wpptr->row = fptr->cur.row ;  // current row
               if ( (wpptr->col = (gsParse.gscols())) >= fptr->wid )
                  wpptr->col = fptr->wid - 1 ;
               break ;
            }
         }
      }

      indx = ZERO ;
      while ( wptr[indx] != NULLCHAR )
      {
         if ( wptr[indx] == NEWLINE )     // newline signals end of row
         {
            //* If text extends beyond the last row of the field, or if *
            //* text ends with the newline, reference the newline as    *
            //* the end-of-text.                                        *
            if ( (wpOff.row == (fptr->hgt - 1)) || (wptr[indx + 1] == NULLCHAR) )
            {
               wpOff.col = colAcc ;
               break ;
            }

            //* If not on last row, and if additional text on next row, *
            //* reset the accumulator and continue scan.                *
            if ( wpOff.row < (fptr->hgt - 1) )
            {
               ++wpOff.row ;
               colAcc = ZERO ;
            }

         }
         else                             // non-newline character
         {
            colAcc += colArray[indx] ;    // count text columns
            //* If current row is filled with text, the currently       *
            //* indexed character is the first character displayed on   *
            //* the next row (if any).                                  *
            if ( colAcc >= fptr->wid )
            {
               //* If already on last display row, previous character   *
               //* is referenced as end-of-text.                        *
               if ( (wpOff.row == (fptr->hgt - 1)) || (wptr[indx + 1] == NULLCHAR) )
               {
                  wpOff.col = colAcc - colArray[indx] ;
                  break ;
               }
               //* If NOT on last display row, reset  *
               //* the accumulator and continue scan. *
               else
               {
                  ++wpOff.row ;
                  colAcc = ZERO ;
                  continue ;
               }
            }
            //* If next character is the null terminator,   *
            //* end-of-text has been found on current row.  *
            //* wpOff.col references the null terminator.   *
            else if ( wptr[indx + 1] == NULLCHAR )
            {
               wpOff.col = colAcc ;
               break ;
            }
         }
         ++indx ;
      }     // while()
   }

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_EOTOFF != 0
   if ( (ofsdbg.is_open()) )
   {
      gsdbg.append( "\n  wpOff:%hd/%hd", &wpOff.row, &wpOff.col ) ;
      ofsdbg << gsdbg.ustr() << endl ;
   }
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_EOTOFF

   return wpOff ;

}  //* End eotOffset() *

//*************************
//*       ipOffset        *
//*************************
//********************************************************************************
//* Protected Method:                                                            *
//* -----------------                                                            *
//* Calculate cursor offset from the field origin corresponding to the           *
//* insertion point 'ip' member (or the specified alternate ip value).           *
//*                                                                              *
//* Input  : fptr : pointer to target field                                      *
//*          iptrg: (optional, ipoCur by default) indicates the target index     *
//*                 for calculating the cursor offset                            *
//*          ipval: (optional, null pointer by default)                          *
//*                 If specified, this is a pointer to the 16-bit variable to    *
//*                 receive the calculated IP value (or a copy of fptr->ip).     *
//*                                                                              *
//* Returns: zero-based row/column cursor offset from field origin               *
//********************************************************************************
//* Programmer's Note:                                                           *
//* The calculation is a bit kludgy and esoteric, but it is a functional         *
//* first effort. The algorithm may become smoother in subsequent releases.      *
//*                                                                              *
//* Note that because we have not implemented scrolling text within a field,     *
//* the ipoEoT parameter indexes the end of the VISIBLE text within the          *
//* window. It references the actual end of text only if all text is visible     *
//* within the window.                                                           *
//*                                                                              *
//********************************************************************************

WinPos AnsiCmd::ipOffset ( skField *fptr, ipoTarget iptrg, short *ipval )
{
   #define DEBUG_ipOffset (0)       // for debugging only

   #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_ipOffset != 0
   gString gsC, gsW, gsI ;          // format output
   WinPos wpTmp ;                   // holds final value during debug output
   const wchar_t nln = wcsLARROW ;  // replace newlines in output
   const wchar_t nul = wcsDARROW ;  // replace nullchar in output
   #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_ipOffset

   WinPos wpOff( 0, 0 ) ;           // return value (initially at field origin)
   short txtLen = fptr->txt.gschars() - 1 ; // index of source data null terminator
   if ( fptr->ip < ZERO )           // minimum value
      fptr->ip = ZERO ;
   else if ( fptr->ip > txtLen )    // maximum value
      fptr->ip = txtLen ;

   //* Target insertion point (return value).      *
   //* Default target is current IP (fptr->ip).    *
   //* If ipoCur or ipoToT, we set the target here.*
   //* Else, target will be calculated below.      *
   short trgIp  = (iptrg == ipoToT) ? ZERO : fptr->ip ;

   //* If target IP is field origin, then 'wpOff' and 'trgIp'  *
   //* already contain the return data. Otherwise, target IP   *
   //* is not field origin, so calculate offset to target IP.  *
   if ( (iptrg != ipoToT) && (txtLen > ZERO) ) 
   {
      gString gsParse ;             // analysis of parsed text
      const wchar_t *wptr ;         // pointer to source data
      const short *colArray ;       // pointer to array of column width
      short chCnt,                  // total source characters (loop control)
            colAcc = ZERO,          // column accumulator
            si = ZERO ;             // index into source buffer

      //* For single-row fields, use the source directly.           *
      //* For multi-row fields, parse the data into individual rows.*
      if ( fptr->hgt == 1 )         // single-row field
         gsParse = fptr->txt ;
      else                          // multi-row field
         this->parseFieldText ( fptr, gsParse ) ;
      wptr = gsParse.gstr() ;
      colArray = gsParse.gscols( chCnt ) ;

      //* If request for end-of-text:                  *
      //* a) If EOT is in visible space, reference EOT *
      //*    (usually nullchar)                        *
      //* b) If eot is not visible, reference last     *
      //*    character in visible space.               *
      if ( iptrg == ipoEoT )
      { trgIp = gsParse.gschars() - 1 ; }

      //* For muti-row fields only,         *
      //* caller may have requested either: *
      //* ipoToR  : Top-Of-Row              *
      //* ipoEoR  : End-Of-Row              *
      else if ( (fptr->hgt > 1) && ((iptrg == ipoToR) || (iptrg == ipoEoR)) )
      {
         //* Scan to the end of the current row.*
         if ( iptrg == ipoEoR )
         {
            trgIp = fptr->ip ;
            for ( ; (wptr[trgIp] != NEWLINE) && (wptr[trgIp] != NULLCHAR) ; ++trgIp ) ;
         }
         else  // (iptrg == ipoToR)
         {
            for ( ; (wptr[trgIp] != NEWLINE) && (trgIp > ZERO) ; --trgIp ) ;
            if ( wptr[trgIp] == NEWLINE ) // if at end of previous line, step forward
               ++trgIp ;
         }
      }

      //* If specified, copy target IP to caller's variable.*
      if ( ipval != NULL )
         *ipval = trgIp ;

      for ( si = ZERO ; si < chCnt ; ++si )
      {
         colAcc += colArray[si] ;

         //* If a newline but not yet at IP *
         if ( (wptr[si] == NEWLINE) && (si < trgIp) )
         {
            // Programmer's Note: We rely here on the fact that the parsing 
            // routine will only include a newline in the parsed data IF 
            // additional data follow it.
            ++wpOff.row ;
            colAcc = ZERO ;
         }

         //* If IP reached, or if end-of-text *
         if ( (si == trgIp) || (wptr[si] == NULLCHAR) )
         {
            wpOff.col = colAcc - colArray[si] ; // save the column offset

            //* If overrun of right edge of field, move left *
            //* by width of last character displayed on row. *
            if ( wpOff.col >= fptr->wid )
            {
               wpOff.col = fptr->wid - colArray[si - 1] ;
               if ( ipval != NULL )
                  *ipval = --trgIp ;
            }

            #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_ipOffset != 0
            gsI.append( "%02hd", &wpOff.col ) ;
            gsC.append( " %C", (wptr[si] == NEWLINE ? 
                        &nln : wptr[si] == NULLCHAR ? &nul : &wptr[si]) ) ;
            if ( colArray[si] == 2 )   // two-column character (Chinese)
               gsW.append( "%3hd", &colArray[si] ) ;
            else                       // assume 1-column character
               gsW.append( "%2hd", &colArray[si] ) ;
            wpTmp = wpOff ;
            #else    // Production
            break ;     // done
            #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_ipOffset
         }

         #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_ipOffset != 0
         else
         {
            gsC.append( " %C", 
               (wptr[si] == NEWLINE ? &nln : wptr[si] == NULLCHAR ? &nul : &wptr[si]) ) ;
            if ( colArray[si] == 2 )   // two-column character (Chinese)
            { gsW.append( "%3hd", &colArray[si] ) ; gsI.append( "   " ) ; }
            else                       // assume 1-column character
            { gsW.append( "%2hd", &colArray[si] ) ; gsI.append( "  " ) ; }
         }
         if ( si >= trgIp ) break ;
         #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_ipOffset
      }     // for(;;)

      #if DEBUG_ANSICMD != 0 && DEBUG_LOG != 0 && DEBUG_ipOffset != 0
      wpOff = wpTmp ;
      gsdbg.compose( " wpOff: %hd:%hd", &wpOff.row, &wpOff.col ) ;
      ofsdbg << gsC.ustr() << endl ;
      ofsdbg << gsW.ustr() << endl ;
      ofsdbg << gsI.ustr() << endl ;
      ofsdbg << gsdbg.ustr() << endl ;
      #endif   // DEBUG_ANSICMD && DEBUG_LOG && DEBUG_ipOffset
   }
   return wpOff ;

   #undef DEBUG_ipOffset
}  //* End ipOffset() *

//*************************
//*      clearField       *
//*************************
//********************************************************************************
//* Protected Method:                                                            *
//* -----------------                                                            *
//* Erase the displayed text from the target field.                              *
//* Caller is responsible for setting color attributes.                          *
//* Caller is responsible for final cursor position.                             *
//*                                                                              *
//* Input  : skf   : pointer to skForm object containing target field            *
//*          findx : index of target field )(range check NOT performed)          *
//*                                                                              *
//* Returns: terminal relative cursor position of field origin                   *
//*          (It is assumed that caller will want to write to the field.)        *
//********************************************************************************

WinPos AnsiCmd::clearField ( const skForm *skf, short findx )
{
   gString gs ;                              // text formatting
   gs.padCols( skf->fld[findx].wid ) ; // pad to width of field

   //* Cursor position (terminal relative) *
   WinPos wpOrig( short(skf->base.row + skf->fld[findx].orig.row), 
                  short(skf->base.col + skf->fld[findx].orig.col) ),
          wp = wpOrig ;

   //* Clear each row of the field *
   for ( short r = ZERO ; r < skf->fld[findx].hgt ; ++r )
   {
      this->acSetCursor ( wp ) ;    // set cursor
      this->ttyWrite ( gs ) ;       // write the padding string
      ++wp.row ;                    // advance to next row
   }
   return wpOrig ;

}  //* End clearField() *

//*************************
//*      refreshText      *
//*************************
//********************************************************************************
//* Protected Method:                                                            *
//* -----------------                                                            *
//* Format and display the text for the specified field object.                  *
//* Caller has verified that the 'txt' member contains text data.                *
//* Caller is responsible for setting color attributes.                          *
//* Caller is responsible for setting cursor offset to IP.                       *
//*                                                                              *
//* Input  : fptr : pointer to field to be updated.                              *
//*          wp   : (by reference) terminal-relative cursor position of          *
//*                 field origin (upper-left corner)                             *
//*                                                                              *
//* Returns: nothing                                                             *
//********************************************************************************

void AnsiCmd::refreshText ( skField *fptr, const WinPos& wp ) const
{
   gString gsOut ;                        // output buffer

   //* Single-row field *
   if ( fptr->hgt == 1 )
   {
      gsOut = fptr->txt ;
      gsOut.limitCols( fptr->wid ) ;      // limit text width to width of field
      this->acSetCursor ( wp ) ;          // set cursor to field origin
      this->ttyWrite ( gsOut ) ;          // display the text
   }

   //* Multi-row field *
   else
   {
      //* Parse the field text into its component display rows,*
      //* concatenate the results and display the text.        *
      this->parseFieldText ( fptr, gsOut ) ;
      this->acWrite ( wp, gsOut ) ;
   }        // multi-row

}  //* End refreshText()

//*************************
//*    parseFieldText     *
//*************************
//********************************************************************************
//* Protected Method:                                                            *
//* -----------------                                                            *
//* Parse the text for the field into one sub-string for each row of the field.  *
//* The parsed data are then concatenated into a single string with each row     *
//* (except the last) defined by a newline character ('\n').                     *
//*                                                                              *
//* Input  : fptr   : pointer to field to be parsed.                             *
//*          gsParse: (by reference) receives the parsed text data               *
//*          trunc  : (optional, 'true' by default)                              *
//*                   if 'true',  truncate the row data after the last           *
//*                               display row is processed.                      *
//*                   if 'false', process all source data, even if some of the   *
//*                               data rows do not fit within the display area.  *
//*                                                                              *
//* Returns: number of character in parsed data not including null terminator.   *
//*          (this is the index of the null character)                           *
//********************************************************************************

short AnsiCmd::parseFieldText ( const skField *fptr, gString& gsParse, bool trunc ) const
{
   gString gsScn = fptr->txt,          // text scan
           gsFmt,                      // text formatting
           gsTmp ;                     // temp buffer
   const wchar_t *wptr = gsScn.gstr() ;// pointer to source text
   wchar_t wbuff[gsMAXCHARS] ;         // work buffer
   short   row = ZERO,                 // index of target row
           si = ZERO, ti = ZERO ;      // indices for source and target data
   short txtRows = ZERO ;              // count rows of formatted text
   gsParse.clear() ;                   // initialize caller's buffer

   while ( *wptr != NULLCHAR )
   {
      //* Copy characters to work buffer.* 
      if ( (wptr[si] != NEWLINE) && (wptr[si] != NULLCHAR) )
         wbuff[ti++] = wptr[si++] ;

      //* Newline characters indicate end-of-row.*
      //* Null character indicates end-of-text.  *
      else
      {
         wbuff[ti] = NULLCHAR ;           // terminate the sub-string
         gsFmt = wbuff ;                  // test column count

         //* If isolated row text fits within the field, *
         //* copy it to caller's buffer.                 *
         if ( (gsFmt.gscols()) <= fptr->wid )
         {
            gsParse.append( "%S\n", gsFmt.gstr() ) ;
            gsScn.shiftChars( -((gsFmt.gschars())) ) ;
            wptr = gsScn.gstr() ;
            si = ti = ZERO ;
         }

         //* Text for this row is wider than the field.*
         //* Capture the text that fits and wrap the   *
         //* excess to the next row.                   *
         else
         {
            gsTmp = gsFmt ;
            gsFmt.limitCols( fptr->wid ) ;
            gsParse.append( "%S\n", gsFmt.gstr() ) ;
            gsScn.shiftChars( -((gsFmt.gschars()) - 1) ) ;
            wptr = gsScn.gstr() ;
            si = ti = ZERO ;
         }

         //* If not at end-of-text AND         *
         //* if field contains additional rows *
         if ( (*wptr != NULLCHAR) && (++row < fptr->hgt) )
            ++txtRows ;

         //* Either end-of-text reached, OR there is more *
         //* text than can be displayed in the field.     *
         else if ( (*wptr == NULLCHAR) || trunc )
            break ;

      }  // E-O-R _or_ E-O-T
   }     // while()

   //* Strip the trailing newline.*
   short eoti = (gsParse.gschars()) - 2 ;
   ti = gsParse.findlast( NEWLINE ) ;
   if ( ti == eoti )
      gsParse.limitChars( eoti ) ;

   return ( (gsParse.gschars()) - 1 ) ;

}  //* End parseFieldText() *

//*************************
//*     setCursor2IP      *
//*************************
//********************************************************************************
//* Protected Method:                                                            *
//* -----------------                                                            *
//* Set the cursor to the insertion point for specified field.                   *
//*                                                                              *
//* Input  : fptr : pointer to target field                                      *
//*          wpBase: (by reference) terminal-relative position of field origin   *
//*                                                                              *
//* Returns: insertion point as a terminal-relative (1-based) offset             *
//********************************************************************************

WinPos AnsiCmd::setCursor2IP ( const skField *fptr, const WinPos& wpBase )
{
   WinPos wp( short(wpBase.row + fptr->orig.row + fptr->cur.row),
              short(wpBase.col + fptr->orig.col + fptr->cur.col) ) ;
   this->acSetCursor ( wp ) ;
   return wp ;

}  //* End setCursor2IP() *

//*************************
//*                       *
//*************************
//********************************************************************************
//*                                                                              *
//*                                                                              *
//*                                                                              *
//* Input  :                                                                     *
//*                                                                              *
//* Returns:                                                                     *
//********************************************************************************

