//********************************************************************************
//* File       : AnsiCmdDef.hpp                                                  *
//* Author     : Mahlon R. Smith                                                 *
//*              Copyright (c) 2022-2023 Mahlon R. Smith, The Software Samurai   *
//*                 GNU GPL copyright notice below                               *
//* Date       : 20-Aug-2023                                                     *
//* Version    : (see AnsiCmdVersion string in AnsiCmd.cpp)                      *
//*                                                                              *
//* Description: General definitions for the AnsiCmd class.                      *
//*              - ANSI escape sequences defined in the standard, listed         *
//*                as the "aeSeq" (Ansi Escape SEQuence) enumerated type.        *
//*              - Color-attribute bitfields listed as the "acAttr"              *
//*                (AnsiCmd ATTRibutes) enumerated type.                         *
//*                type                                                          *
//*              - The acaExpand class handles manipulation of the color-        *
//*                attribute bitfield data.                                      *
//*              - Structured parameters for setting RGB (Red/Green/Blue)        *
//*                color attributes defined as the "WebSafeRGB" enumerated       *
//*                type.                                                         *
//*              - The skField and skForm classes which define text input        *
//*                fields, primarily within the ACWin window class, but may      *
//*                also be used to define stand-alone input fields.              *
//*              - The WinPos class defines the mechanism for cursor             *
//*                positioning within the terminal window.                       *
//*              - Special character definitions (e.g. line drawing characters)  *
//*                and keycode definitions (e.g. cursor control keys).           *
//*              - The AC_Box class and AC_Line class define structured access   *
//*                to line-drawing operations in the terminal window.            *
//*              - Miscellaneous defined constants.                              *
//*                                                                              *
//********************************************************************************

using namespace std ;      //* C++ Scope quailfier

//*****************
//* Include Files *
//*****************
#include <unistd.h>           //* UNIX system interface
#include <cstdlib>            //* Misc. functionality
#include <sys/ioctl.h>        //* Get low-level terminal info (e.g. struct winsize)
#include <termios.h>          //* Terminal attributes (i.e. /usr/include/asm-generic/termios.h)
#include <signal.h>           //* Signal capture (CTRL+C capture)
#include <limits>             //* Define max size of input stream
#include <fcntl.h>            //* For fcntl() function (non-blocking read)
#include <fstream>            //* C++ file I/O
#include "gString.hpp"        //* Text analysis and formatting


//***************
//* Definitions *
//***************
//* Convert declaration of C structure for terminal dimensions to C++ format.  *
typedef struct winsize WinSize ;
//* Convert declaration of C structure for terminal I/O settings to C++ format.*
typedef struct termios TermIos ;
//* Convert declaration of C structure for clock access to C++ format.*
typedef struct timespec TimeSpec ;

//* General constant values *
const wchar_t NEWLINE   = L'\n' ;
const wchar_t NULLCHAR  = L'\0' ;
const wchar_t SPACE     = L' ' ;
const wchar_t CR        = L'\r' ;
const wchar_t BACKSPACE = L'\b' ;
const short   ZERO      = (0) ;
const char    fSLASH    = '/' ;
const char    DASH      = '-' ;

//* Maximum number of ACWin objects supported within a terminal window.*
const short maxACWIN = 12 ;

//* Cursor Shapes (style)                           *
//* These are VT520 i.e. XTerm definitions.         *
//* Additional styles and colors may be added later.*
enum CurStyle: short
{
   csBblock = 0,  // Full-character cell, blinking block
   csDflt   = 1,  // Cursor type defined in the terminal settings (usually csBblock)
   csSblock = 2,  // Full-character cell, steady (non-blinking) block
   csBuline = 3,  // Blinking underline
   csSuline = 4,  // Steady underline
   csBvbar  = 5,  // Blinking vertical bar
   csSvbar  = 6,  // Steady vertical bar
   csHide   = 7,  // Hide cursor (make it invisible)
   csShow   = 8,  // Show cursor (make it visible)

} ;

//* Cursor positioning and reporting. (Adapted from author's NcDialog API) *
class WinPos
{
   public:
   //* Default constructor. (terminal window positions are 1-based offsets) *
   WinPos ( void ) { row = 1 ; col = 1 ; }
   //* Initialization constructor *
   WinPos( const short& rval, const short& cval ) : row(rval), col(cval) {}
   bool operator == ( const WinPos& wp )
   {
      bool eq = false ;
      if ( (wp.row == this->row) && (wp.col == this->col) ) { eq = true ; }
      return eq ;
   }
   bool operator != ( const WinPos& wp )
   {
      bool eq = false ;
      if ( (wp.row != this->row) || (wp.col != this->col) ) { eq = true ; }
      return eq ;
   }
   //* Public data members *
   short row ;                // Row number    (1-based offset)
   short col ;                // Column number (1-based offset)
} ;   // WinPos

//* Line-drawing codes for drawing lines and window borders *
enum LineType : short 
{ 
   ltHORIZ,  ltVERT, 
   ltSINGLE, ltSINGLEBOLD, ltDUAL,
   ltDASH2,  ltDASH2BOLD, 
   ltDASH3,  ltDASH3BOLD,
   ltDASH4,  ltDASH4BOLD
} ;

//* Argument for call to private method, ipOffset() *
//* indicating the target of the calculation.       *
enum ipoTarget : short
{
   ipoCur,        // calculate offset to current insertion point
   ipoToT,        // calculate offset to Top-Of-Text
   ipoEoT,        // calculate offset to End-Of-Text (actually end of _visible_ text)
   //ipoEoV,        // calculate offset to end of _visible_ text [CURRENTLY UNUSED]
   ipoToR,        // calculate offset to top of current text row
   ipoEoR,        // calculate offset to end of current text row
} ;

//******************************************************
//** All ANSI escape sequences ("aesXXX") and control **
//** codes defined in ECMA-48, ISO/IEC 6429 standards.**
//** Note that not all of these options are supported **
//** by all operating systems and environments.       **
//******************************************************
enum aeSeq : short
{
   //** Text Display Modification Options. **
   //** (supported by most Linux systems   **
   //**  and terminal emulator software)   **
   aesRESET = 0,              // reset all attributes
   aesBOLD,                   // bold (increase intensity)
   aesFAINT,                  // faint/dim (decrease intensity)
   aesITALIC,                 // italic (or inverse)
   aesUNDERLINE,              // underline
   aesBLINK_SLOW,             // slow blink
   aesBLINK_FAST,             // fast blink
   aesREVERSE,                // reverse foreground and background colors
   aesCONCEAL,                // hide text
   aesXOUT,                   // crossed out text (marked for deletion)
   aesFRACTUR,                // a variety of German calligraphy
   aesDBL_UNDERLINE,          // double-underline (or bold off)
   aesFRAMED,                 // "frame" the text
   aesENCIRCLE,               // "encircle" the text
   aesOVERLINE,               // overlined text
   aesBOLD_OFF,               // normal intensity (neither bold nor faint)
   aesITALIC_OFF,             // italic off (and Fractur off)
   aesUNDERLINE_OFF,          // underline off
   aesBLINK_OFF,              // blinking off
   aesREVERSE_OFF,            // reverse off
   aesCONCEAL_OFF,            // conceal off
   aesXOUT_OFF,               // not crossed out
   aesFRM_ENC_OFF,            // not framed, not encircled
   aesOVERLINE_OFF,           // overline off

   //** Foreground Color Attributes (3-bit and 4-bit color selection) **
   aesFG_BLACK,               // set foreground color to black (i==24)
   aesFG_RED,                 // set foreground color to red
   aesFG_GREEN,               // set foreground color to green
   aesFG_BROWN,               // set foreground color to brown
   aesFG_BLUE,                // set foreground color to blue
   aesFG_MAGENTA,             // set foreground color to magenta
   aesFG_CYAN,                // set foreground color to cyan
   aesFG_GREY,                // set foreground color to grey (dull white)
   aesFGb_BLACK,              // set foreground color to bright black (grey)
   aesFGb_RED,                // set foreground color to bright red
   aesFGb_GREEN,              // set foreground color to bright green
   aesFGb_BROWN,              // set foreground color to yellow (bright brown)
   aesFGb_BLUE,               // set foreground color to bright blue
   aesFGb_MAGENTA,            // set foreground color to bright magenta
   aesFGb_CYAN,               // set foreground color to bright cyan
   aesFGb_GREY,               // set foreground color to white (bright grey)
   aesFG_DFLT,                // set foreground color to terminal default

   //** Background Color Attributes (3-bit and 4-bit color selection) **
   aesBG_BLACK,               // set background color to black (i==41)
   aesBG_RED,                 // set background color to red
   aesBG_GREEN,               // set background color to green
   aesBG_BROWN,               // set background color to brown
   aesBG_BLUE,                // set background color to blue
   aesBG_MAGENTA,             // set background color to magenta
   aesBG_CYAN,                // set background color to cyan
   aesBG_GREY,                // set background color to grey (dull white)
   aesBGb_BLACK,              // set background color to bright black (grey)
   aesBGb_RED,                // set background color to bright red
   aesBGb_GREEN,              // set background color to bright green
   aesBGb_BROWN,              // set background color to yellow (bright brown)
   aesBGb_BLUE,               // set background color to bright blue
   aesBGb_MAGENTA,            // set background color to bright magenta
   aesBGb_CYAN,               // set background color to bright cyan
   aesBGb_GREY,               // set background color to bright white
   aesBG_DFLT,                // set background color to terminal default

   //** Indexed Color Attribute Options (8-bit color and greyscale) **
   //*  0 - 7     basic standard color indices
   //*  8 - 15    basic intense colors indices
   //*  16 - 231  216 8-bit color indices
   //*  232 - 255 greyscale color indices (24 steps)
   aesFG_INDEX,               // set foreground color by lookup table index (0-255)
   aesBG_INDEX,               // set background color by lookup table index (0-255)

   //** Red/Green/Blue Color Attribute Options (24-bit color) **
   //*  Encoding. Fg: \x1B[38;2;(r);(g);(b)m  Bg: \x1B[48;2;(r);(g);(b)m
   aesFG_RGB,                 // set foreground color by r/g/b register number
   aesBG_RGB,                 // set background color by r/g/b register number

   //** Cursor-positioning Options **
   aesCUR_HOME,               // set cursor at 1,1
   aesCUR_ABSPOS,             // set (absolute) cursor at specified row;column position
   aesCUR_ROWSUP,             // move cursor upward by specified number of rows
   aesCUR_ROWSDN,             // move cursor downward by specified number of rows
   aesCUR_COLSRT,             // move cursor right by specified number of columns
   aesCUR_COLSLT,             // move cursor left by specified number of columns
   aesCUR_NEXTROW,            // move cursor to beginning of row below (or downward row count?)
   aesCUR_PREVROW,            // move cursor to beginning of row above (or upward row count?)
   aesCUR_ABSCOL,             // move cursor to specified column on current line
   aesCUR_REPORT,             // request current cursor position
   aesCUR_ROWDEL,             // delete the current row (scrolling the lower rows upward)
   aesCUR_SAVE_DEC,           // save current cursor position (DEC)
   aesCUR_RESTORE_DEC,        // restore cursor to last saved position (DEC)
   aesCUR_SAVE_SCO,           // save current cursor position (SCO, unreliable)
   aesCUR_RESTORE_SCO,        // restore cursor to last saved position (SCO, unreliable)

   //** Text Erasure Options **
   aesERASE_BOW,              // erase from cursor to bottom of window
   aesERASE_TOW,              // erase from cursor to top of window
   aesERASE_WIN,              // erase entire window ('clear')
   aesERASE_SAVED,            // erase "saved" lines
   aesERASE_EOL,              // erase from cursor to end of line
   aesERASE_BOL,              // erase from cursor to beginning of line
   aesERASE_LINE,             // erase entire line

   //** Alternate Font group **
   aesPRIMARY_FONT,           // primary font      (font 0)
   aesALTERNATE_FONT,         // alternate fonts (font 1 through font 9)

   //** ASCII Control Codes. **
   aesBELL,                   // '\a' terminal "bell" (beep, ping, etc.)
   aesBKSP,                   // '\b' backspace
   aesHTAB,                   // '\t' horizontal tab
   aesLINEFEED,               // '\n' linefeed (end-of-line)
   aesVTAB,                   // '\v' vertical tab
   aesFORMFEED,               // '\f' formfeed (new page)
   aesRETURN,                 // '\r' carriage return (used by Windoze)
   aesESC,                    // '\e' escape character ( ^[ or \x1B[ )
   aesDEL,                    // delete character under cursor

   //** Ideogram (logogram) group of options **
   aesIDEO_UNDER,             // ideogram underlined
   aesIDEO_UDOUBLE,           // ideogram double-underlined
   aesIDEO_OVER,              // ideogram overlined
   aesIDEO_ODOUBLE,           // ideogram double-overlined
   aesIDEO_STRESS,            // ideogram stress marking
   aesIDEO_OFF,               // all ideogram attributes off

   //* NOTE: The Aixterm-specific bright fg/bg codes are not  *
   //*       part of the ANSI standard, but may be supported. *
   aesAIXFGb_BLACK,           // set foreground color to bright black (grey)
   aesAIXFGb_RED,             // set foreground color to bright red
   aesAIXFGb_GREEN,           // set foreground color to bright green
   aesAIXFGb_BROWN,           // set foreground color to yellow (bright brown)
   aesAIXFGb_BLUE,            // set foreground color to bright blue
   aesAIXFGb_MAGENTA,         // set foreground color to bright magenta
   aesAIXFGb_CYAN,            // set foreground color to bright cyan
   aesAIXFGb_GREY,            // set foreground color to white (bright grey)

   aesAIXBGb_BLACK,           // set background color to bright black (grey)
   aesAIXBGb_RED,             // set background color to bright red
   aesAIXBGb_GREEN,           // set background color to bright green
   aesAIXBGb_BROWN,           // set background color to yellow (bright brown)
   aesAIXBGb_BLUE,            // set background color to bright blue
   aesAIXBGb_MAGENTA,         // set background color to bright magenta
   aesAIXBGb_CYAN,            // set background color to bright cyan
   aesAIXBGb_GREY,            // set background color to white (bright grey)

   aesCOUNT                   // number of items in enumerated type list
} ;   // enum aeSeq

//********************************************************************
//* acAttr (AnsiCmd Attributes) provides application-level color-    *
//* attribute and text-modifier configuration for Linux terminal     *
//* applications.                                                    *
//* Specify color attributes and text modifiers in a consistent      *
//* format which encapsulates the primitive ANSI escape sequences    *
//* defined for terminal-window text formatting. (see enum aeSeq)    *
//*                                                                  *
//* Within the AnsiCmd-class methods, these values are mapped to     *
//* their corresponding ANSI escape sequences (or group of           *
//* sequences). This allows the application to focus on algorithms   *
//* and user interaction rather than reading and writing binary data *
//* within the terminal window.                                      *
//*                                                                  *
//* acAttr is defined as an enumerated type, but is constructed      *
//* as a group of macros defining bitfields. This allows the macros  *
//* to be identified as a coherent group while also allowing         *
//* bitwise OR operations for the desired foreground and background  *
//* attributes including text-modifier attributes.                   *
//* For example, combine foreground, background attributes PLUS the  *
//* "bold" and "italic" text attributes, then pass the value as an   *
//* argument to one of the application-level methods.                *
//* acAttr txtAttr =                                                 *
//*        acAttr(acaFG_GREEN | acaBG_GREY | acaBOLD | acaITALIC) ;  *
//* acSetAttributes ( txtAttr ) ;                                    *
//*                                                                  *
//* -- Technical Note: The compiler will complain when performing    *
//*    mathematical operations on members of an enumerated type;     *
//*    therefore, be sure to cast the OR'd value back to the         *
//*    acAttr type. For instance, the following would cause a        *
//*    compiler warning: acAttr a;  a |= acaBOLD;                    *
//*    Instead, use the following syntax: a = acAttr(a | acaBOLD);   *
//*                                                                  *
//* See also the notes in the header of acaExpand-class 'decode()'   *
//* method for more details.                                         *
//********************************************************************
enum acAttr : uint32_t
{
   //** Foreground and Background color index fields **
   acaFG_INDEX    = 0x000000FF,  // Foreground index
   acaBG_INDEX    = 0x0000FF00,  // Background index

   //** Text modifier flags **
   acaPLAINTXT    = 0x00000000,  // plain text, no modifier flags set
   acaBOLD        = 0x00010000,  // bold-text modifier
   acaITALIC      = 0x00020000,  // italic-text modifier
   acaUNDERLINE   = 0x00040000,  // underline-text modifier
   acaOVERLINE    = 0x00080000,  // overline-text modifier
   acaXOUT        = 0x00100000,  // x-out (strikethrough) text modifier
   acaBLINK       = 0x00200000,  // blinking-text modifier
   acaCONCEAL     = 0x00400000,  // concealed (invisible) text
   acaREVERSE     = 0x00800000,  // reversed foreground/background color attributes

   //** Other flags **
   acaCLEAR_MODS  = 0x01000000,  // clear (reset) specified text modifiers (or all modifiers)
   acaFG_DFLT     = 0x02000000,  // Terminal default-foreground color attribute
                                 // (value in acaFG_INDEX field will be ignored)
   acaFG_RGB      = 0x04000000,  // if set, acaFG_INDEX contains the index of 
                                 // "web-safe" RGB color attribute for foreground
   acaBG_DFLT     = 0x08000000,  // Terminal default-foreground color attribute
                                 // (value in acaBG_INDEX field will be ignored)
   acaBG_RGB      = 0x10000000,  // if set, acaBG_INDEX contains the index of 
                                 // "web-safe" RGB color attribute for background
   acaRES1        = 0x20000000,  // reserved bit 1
   acaRES2        = 0x40000000,  // reserved bit 2
   acaUSEDFLT     = 0x80000000,  // placeholder for optional parameters (ignore the value)

   //** Foreground color attributes **
   acaFG_BLACK    = 0x00000000,  // foreground Black
   acaFG_RED      = 0x00000001,  // foreground Red
   acaFG_GREEN    = 0x00000002,  // foreground Green
   acaFG_BROWN    = 0x00000003,  // foreground Brown
   acaFG_BLUE     = 0x00000004,  // foreground Blue
   acaFG_MAGENTA  = 0x00000005,  // foreground Magenta
   acaFG_CYAN     = 0x00000006,  // foreground Cyan
   acaFG_GREY     = 0x00000007,  // foreground Grey
   acaFGb_BLACK   = 0x00000008,  // bold foreground Black
   acaFGb_RED     = 0x00000009,  // bold foreground Red
   acaFGb_GREEN   = 0x0000000A,  // bold foreground Green
   acaFGb_BROWN   = 0x0000000B,  // bold foreground Brown
   acaFGb_BLUE    = 0x0000000C,  // bold foreground Blue
   acaFGb_MAGENTA = 0x0000000D,  // bold foreground Magenta
   acaFGb_CYAN    = 0x0000000E,  // bold foreground Cyan
   acaFGb_GREY    = 0x0000000F,  // bold foreground Grey

   //** Background color attributes **
   acaBG_BLACK    = 0x00000000,  // background Black
   acaBG_RED      = 0x00000100,  // background Red
   acaBG_GREEN    = 0x00000200,  // background Green
   acaBG_BROWN    = 0x00000300,  // background Brown
   acaBG_BLUE     = 0x00000400,  // background Blue
   acaBG_MAGENTA  = 0x00000500,  // background Magenta
   acaBG_CYAN     = 0x00000600,  // background Cyan
   acaBG_GREY     = 0x00000700,  // background Grey
   acaBGb_BLACK   = 0x00000800,  // bold background Black
   acaBGb_RED     = 0x00000900,  // bold background Red
   acaBGb_GREEN   = 0x00000A00,  // bold background Green
   acaBGb_BROWN   = 0x00000B00,  // bold background Brown
   acaBGb_BLUE    = 0x00000C00,  // bold background Blue
   acaBGb_MAGENTA = 0x00000D00,  // bold background Magenta
   acaBGb_CYAN    = 0x00000E00,  // bold background Cyan
   acaBGb_GREY    = 0x00000F00,  // bold background Grey

   acaFG_MASK     = (acaFG_INDEX | acaFG_DFLT | acaFG_RGB), // Foreground bitmask
   acaBG_MASK     = (acaBG_INDEX | acaBG_DFLT | acaBG_RGB), // Background bitmask
   acaATTR_DFLT   = (acaFG_DFLT  | acaBG_DFLT), // Default fgnd OR'd with default bgnd
   acaFLAG_MASK   = 0xFFFF0000,  // All binary flags
   acaMOD_MASK    = 0x00FF0000,  // Text-modifier flags only

} ;   // enum acAttr

//********************************************************************
//* Definition of parameters for setting "web-safe" RGB foreground   *
//* and background. They are used for calls to the public methods:   *
//*      acSetWebFg( WebSafeRGB hue, uint8_t shade );                *
//*      acSetWebBg( WebSafeRGB hue, uint8_t shade );                *
//* and the private method:                                          *
//*      setRgbWeb( bool bgnd, WebSafeRGB hue, uint8_t shade );      *
//********************************************************************
enum WebSafeRGB : uint8_t
{  //* These two values are used for internal calculations. *
   wsrgbSHADES  = 36,            // number of shades in each web-safe color group
   wsrgbSTEP    = 6,             // "web-safe" color attribute step value

   //* These values define the range of the "shade" parameter for    *
   //* setting the RGB attributes via calls to the above RGB methods.*
   wsrgbSHADE_MIN = 0,               // minimum shade within a color block
   wsrgbSHADE_MAX = wsrgbSHADES - 1, // maximum shade within a color block

   //* These values designate the "hue" parameter (color block) for  *
   //* setting the RGB attributes via calls to the above RGB methods.*
   wsrgbMIN     = 0,                            // minimum web-safe color index
   wsrgbBLACK   = wsrgbMIN,                     // Rgb (black)
   wsrgbRED     = wsrgbBLACK + 1,               // base color index (red block)
   wsrgbGREEN   = wsrgbRED + wsrgbSHADES,       // base color index (green block)
   wsrgbBLUE    = wsrgbGREEN + wsrgbSHADES,     // base color index (blue block)
   wsrgbBROWN   = wsrgbBLUE + wsrgbSHADES,      // base color index (brown block)
   wsrgbMAGENTA = wsrgbBROWN + wsrgbSHADES,     // base color index (magenta block)
   wsrgbCYAN    = wsrgbMAGENTA + wsrgbSHADES,   // base color index (cyan block)
   wsrgbGREY    = wsrgbCYAN + wsrgbSHADES,      // base color index (grey block)
   wsrgbMAX     = wsrgbGREY + wsrgbSHADES - 1,  // maximum web-safe color index
} ;   // enum WebSafeRGB

//******************************************************
//* Decoded  acAttr bitfield into its component parts. *
//*     All methods and data members are private.      *
//******************************************************
class acaExpand
{
   //* -------------------------------- *
   //* All methods and data are public. *
   //* -------------------------------- *
   public:
   ~acaExpand ( void ) { /* nothing to do */ }     // destructor
   acaExpand ( void ) ;                            // default constructor
   //* Constructor: Decode the specified bitfield value.*
   acaExpand ( acAttr bits ) ;

   //* Decode the specified bitfield value.*
   void decode ( acAttr bits ) ;
   //* Add (or remove) one or more elements to (from) the bitfield value.*
   acAttr modify ( acAttr acaValue, bool add ) ;
   //* Reset (clear) all data members. Optionally, reset text-mod flags only.*
   void reset ( bool modsOnly = false ) ;
   //* Update attribute tracking data using a member of enum aeSeq.*
   void update ( aeSeq aesValue, uint8_t colorIndx = ZERO ) ;
   //* Construct an acAttr attribute bitmap from component values. *
   acAttr compose ( void ) ;
   //* Convert between acAttr (bitfield) and aeSeq (ANSI escape sequence name) *
   aeSeq bits2aeseq ( acAttr acaBits, uint8_t byteIndx, bool isFgnd, bool isRgb = false ) ;
   //* Convert between aeSeq (ANSI escape sequence name) and acAttr (bitfield) *
   acAttr aeseq2bits ( aeSeq aesCmd, bool& setFlag ) ;
   //* Convert web-safe RGB index into RGB component values *
   bool  web2regs ( uint8_t webrgb, bool fgnd ) ;

   //*************************
   //** Public Data Members **
   //*************************
   acAttr     acaVal ;     // full, encoded acAttr value
   uint32_t   fgBits ;     // nine bits defining foreground: index byte + fg default flag
   uint32_t   bgBits ;     // nine bits defining background: index byte + bg default flag
   uint32_t   allFlags ;   // all binary flags (excludes fg/bg indices)
   uint32_t   modFlags ;   // text-modification flags
   acAttr     acaFgnd ;    // the 8 bits that define the foreground index
   acAttr     acaBgnd ;    // the 8 bits that define the background index
   aeSeq      aesFgType ;  // Fgnd attr type: aesFG_DFLT, aesFG_INDEX, aesFG_RGB
   aeSeq      aesBgType ;  // Bgnd attr type: aesBG_DFLT, aesBG_INDEX, aesBG_RGB
   aeSeq      aesFgnd ;    // 4-bit foreground (member of enum aeSeq)
   aeSeq      aesBgnd ;    // 4-bit background (member of enum aeSeq)
   uint8_t    fgIndex ;    // foreground byte index
   uint8_t    bgIndex ;    // background byte index
   WebSafeRGB fgHue ;      // if RGB foreground, color group (black,red...)
   uint8_t    fgShade ;    // if RGB foreground, color shade (wsrgbSHADE_MIN-wsrgbSHADE_MAX)
   WebSafeRGB bgHue ;      // if RGB background, color group (black,red...)
   uint8_t    bgShade ;    // if RGB background, color shade (wsrgbSHADE_MIN-wsrgbSHADE_MAX)
   uint8_t    rgbFgR ;     // if RGB foreground, Red register value
   uint8_t    rgbFgG ;     // if RGB foreground, Green register value
   uint8_t    rgbFgB ;     // if RGB foreground, Blue register value
   uint8_t    rgbBgR ;     // if RGB background, Red register value
   uint8_t    rgbBgG ;     // if RGB background, Green register value
   uint8_t    rgbBgB ;     // if RGB background, Blue register value
   bool       fgDflt ;     // if set, terminal default foreground (ignore fg index bits)
   bool       bgDflt ;     // if set, terminal default background (ignore bg index bits)
   bool       boldMod ;    // if set, Bold (intense) text
   bool       italicMod ;  // if set, Italic text
   bool       ulineMod ;   // if set, Underlined text
   bool       olineMod ;   // if set, Overlined text
   bool       xoutMod ;    // if set, X-out (strikethrough) text
   bool       blinkMod ;   // if set, Blinking text
   bool       invisMod ;   // if set, Invisible (concealed) text
   bool       revMod ;     // if set, Reversed foreground and background
   bool       clrMods ;    // if set, Clear existing mods before setting new mods
   bool       fgRgb ;      // if set, rgbRed, rgbGreen,rgbBlue contain the foreground color
   bool       bgRgb ;      // if set, rgbRed, rgbGreen,rgbBlue contain the background color
} ;   // acaExpand

//***********************************************************
//* User input field definition. This defines the position  *
//* and dimensions of the field as well as the text within  *
//* the field and the cursor position within the field.     *
//***********************************************************
class skField
{
   public:
   ~skField ( void ) {}             // destructor

   skField ( void )                 // default constructor
   {
      this->orig = { 1, 1 } ;       // field origin
      this->hgt  = 1,               // initial height (rows)
      this->wid  = 5 ;              // initial width (columns)
      this->ip   = ZERO ;           // reference text origin
      this->cur  = { ZERO, ZERO } ; // cursor offset from origin
      this->off  = ZERO ;           // scroll offset zero
      this->ro   = false ;          // accept user input
      //* 'txt' is initially an empty string.                     *
      //* 'aca' is initialized to acaATTR_DFLT when acaExpand     *
      //*       object is instantiated, and is fully initialized  *
      //*       by the ACWin constructor.*
   }

   public:
   gString txt ;                    // text displayed within the field
   WinPos  orig,                    // field origin (upper left corner)
           cur ;                    // current cursor offset
   acaExpand aca ;                  // decoded color attributes for field
   short   hgt,                     // field height (rows)
           wid,                     // field width (columns)
           ip,                      // insertion point (index into text array)
           off ;                    // offset from top of text to first displayed char
   bool    ro ;                     // if 'true' field is read-only
                                    

   // Notes:
   // 'orig' : This is the 0-based offset from the skForm's 'base' coordinates,
   //          which in turn is the 1-based offset from the upper-left corner 
   //          of the terminal window.
   //          Note that within an ACWin object, 'skForm.base' is the same as 
   //          the 'txtBase' member of the ACWin object. 
   // 'cur'  : This is the 0-based offset from 'orig' coordinates which defines
   //          the position of the visible cursor. This is dynamically 
   //          calculated from the 'ip' member.
   // 'ip'   : text insertion point: index into 'txt.gstr()'
   // 'off'  : This is the offset from the head of the text to the first
   //          displayed character. It fascilitates horizontal scrolling of
   //          the text within the field. [currently ignored]
   // 'ro'   : If the field is set to read-only, then that field cannot receive
   //          input focus. This means that the contents of the field cannot be 
   //          modified under user control.
} ;   // skField

//***********************************************************
//* Define an array of skField objects.                     *
//* The user may edit each field indepentently and may move *
//* among the fields using the "special" keycodes.          *
//* -- The skForm class is integrated into the ACWin class, *
//*    defining the skField objects within the window.      *
//* -- The skForm class may also be used to implement a     *
//*    stand-alone form within the terminal window.         *
//***********************************************************
const CurStyle istyleDFLT = csBblock,     // default cursor style Insert mode
               ostyleDFLT = csBuline ;    // default cursor style Overstrike mode
   
class skForm
{
   public:
   //* Public Methods *
   ~skForm ( void )                       // destructor
   {
      if  ( (this->fCnt > ZERO) && (this->fld != NULL) )
         delete [] this->fld ;
   }

   //* Initialization Constructor *
   skForm ( short fieldCount, short rowOffset, short colOffset )
   {
      this->reset () ;              // set data members to defaults
      this->fCnt = fieldCount ;     // create the specified field(s)
      this->base = { rowOffset, colOffset } ; // origin (note: not range checked)
      this->fld = new skField[this->fCnt] ;
   }

   //* Clear text in current field, or all fields.*
   //* This method does not update the display.   *
   void clear ( bool all )
   {
      if ( all )
      {
         for ( short f = ZERO ; f < this->fCnt ; ++f )
            this->fld[f].txt.clear() ;
      }
      else
         this->fld[this->fi].txt.clear() ;
   }

   //* Copy data from another skForm object *
   void operator= ( const skForm *skfp )
   {
      this->fCnt    = skfp->fCnt ;
      this->fi      = skfp->fi ;
      this->pKey    = skfp->pKey ;
      this->base    = skfp->base ;
      this->ins     = skfp->ins ;
      this->insc    = skfp->insc ;
      this->togg    = skfp->togg ;
      this->beep    = skfp->beep ;
      for ( short f = ZERO ; f < this->fCnt ; ++f )
         this->fld[f] = skfp->fld[f] ;
   }

   private:
   //* Default constructor is private to prevent user embarrassment.*
   skForm ( void )            // default constructor
   {
      this->reset () ;        // set data members to defaults
      this->fCnt = ZERO ;     // no fields defined
      this->fld = NULL ;
   }

   void reset ( void )        // called only by constructors
   {
      this->pKey  = NULLCHAR ;   // no previous keycode
      this->fi    = ZERO ;       // index first field
      this->base  = { 1, 1 } ;   // terminal-window origin
      this->icur  = csBblock ;   // Insert Mode - default cursor style
      this->ocur  = csBuline ;   // Overstrike Mode - default cursor style
      this->ins   = true ;       // Insert mode
      this->insc  = false ;      // disable cursor style change
      this->togg  = true ;       // enable toggle of insert/overstrike
      this->beep  = true ;       // noisy mode
   }

   //* Public Data Members *
   public:
   skField *fld ;             // array of skField objects
   short   fCnt,              // number of objects in 'fld' array
           fi ;               // index of currently-active field
   WinPos  base ;             // upper left corner of form within terminal window (1-based)
   wchar_t pKey ;             // most recent special keycode processed
   CurStyle icur ;            // cursor style when in Insert mode
   CurStyle ocur ;            // cursor style when in Overstrike mode
   bool    ins ;              // insert/overstrike flag
                              // 'true'==insert, 'false'==overstrike
   bool    insc ;             // cursor shape changes with insert/overstrike toggle
                              // 'true'  == enable cursor shape change
                              // 'false' == disable cursor shape change
   bool    togg ;             // enable/disable the insert/overstrike toggle
                              // 'true'  == enable toggle
                              // 'false' == disable toggle
   bool    beep ;             // 'true' audible alert if keycode unprocessed
                              // 'false' silent
   // Notes:
   // 'fld'    An array of one or more skField objects which are allocated 
   //          by the initialization constructor. The fields are individually 
   //          initialized by the application, then the application's values 
   //          are range-checked and adjusted as necessary when application's
   //          copies of the object are copied into the ACWin-class skForm object.
   // 'fCnt'   This value is set during instantiation of the skForm object, and 
   //          may not be modified.
   // 'fi'     This is the index into the fld[] array which indicates the field 
   //          which currently has the input focus.
   // 'base'   This is the 1-based offset from the upper-left corner of the 
   //          terminal window to the upper-left corner of the area defined by 
   //          the skForm object. 
   //          Note: Within an ACWin object, skForm.base==ACWin.txtBase.
   // 'pKey'   The most recent "special key" keycode processed.
   //          (currently unused and may be removed)
   // 'ins'    This is the current state of the Insert/Overstrike flag for user 
   //          input into the fields.
   // 'togg'   If set, this locks the current value of the 'ins' member, 
   //          preventing the user from toggling the flag.
   // 'insc'   If set, then change the shape of the visible cursor when the 
   //          Insert key is pressed. This is a visual cue that insert/overstrike 
   //          mode has been toggled.
   // 'beep'   If set this flag activates an audible alert for invalid user 
   //          input or attempted overrun of the target field.
} ;   // skForm

//********************************************************************
//* The AC_Box class defines a mechanism for drawing a window-like   *
//* rectangle within the terminal window.                            *
//* This is a greatly-simplified version of the ACWin-class object.  *
//* Visually, it is identical; however, it has a much smaller        *
//* footprint and uses the ANSI escape sequence primitives directly. *
//* Therefore, it is the application's responsibility to ensure that *
//* the data are displayed in the intended format. Minimal range     *
//* checking and error correction are performed.                     *
//*                                                                  *
//* All data members except height and width and position values are *
//* public, so the object can be modified and redrawn as needed.     *
//*                    However:                                      *
//* "With great power comes great responsibility" - Old Uncle Ben    *
//* (Not all programmers are as skilled as you are, dear reader.)    *
//*                                                                  *
//* Specific AnsiCmd-class methods are available for the basic       *
//* AC_Box operations. These methods are located in AnsiCmdApi.cpp.  *
//* For examples, see the Test_Box() method in AnsiCmdTest.cpp.      *
//*                                                                  *
//*  acBoxDraw()     Draw the box in the terminal window.            *
//*  acBoxErase()    Erase the box from the terminal window.         *
//*  acBoxClear()    Erase the text from the interior of the box.    *
//*                                                                  *
//* For more advanced operations, use the standard AnsiCmd methods,  *
//* e.g. acWrite(), ttyWrite(), acSetCursor(), etc.                  *
//*                                                                  *
//* Two initialization constructors are defined. One uses terminal   *
//* default color attributes and default border style (ldSINGLE).    *
//* The other allows for direct specification of:                    *
//*  1) border and interior attributes,                              *
//*  2) line style for the border of either ltSINGLE or ltDUAL,      *
//*  3) text of title to be written in the top border                *
//*     (null pointer or empty string ("") indicates no title).      *
//*                                                                  *
//* (Note that the default constructor is pretty useless.)           *
//*                                                                  *
//********************************************************************
class AC_Box
{
   public:
   AC_Box ( void )               // default constructor
   {
      this->reset() ;                        // reset data members
   }

   //* Initialization constructor (basic) *
   AC_Box ( WinPos& base, short height, short width )
   {
      this->reset() ;                        // reset data members
      this->range ( base, height, width ) ;  // range check parameters
      this->orig   = base ;
      this->hgt    = height ;
      this->wid    = width ;
   }

   //* Initialization constructor (full) *
   AC_Box ( WinPos& base, short height, short width, LineType typ, 
            aeSeq bfg, aeSeq bbg, aeSeq tfg, aeSeq tbg, const char* tit )
   {
      this->reset() ;                        // reset data members
      this->range ( base, height, width ) ;  // range check parameters
      this->orig   = base ;
      this->hgt    = height ;
      this->wid    = width ;
      this->lnType = typ ;
      this->bFgnd  = bfg ;
      this->bBgnd  = bbg ;
      this->tFgnd  = tfg ;
      this->tBgnd  = tbg ;
      if ( tit != NULL && tit[0] != '\0' )
      {
         gString gs( tit ) ;
         gs.copy( this->title, gsMAXCHARS ) ;
      }
   }

   ~AC_Box ( void )              // destructor
   { /* nothing to do */ }

   //* Private Methods and Data *
   private:
   void reset ( void )           // data initialization
   {
      *this->title = '\0' ;
      this->orig = { 1, 1 } ;
      this->hgt    = 3 ;
      this->wid    = 3 ;
      this->lnType = ltSINGLE ;
      this->bFgnd  = aesFG_DFLT ;
      this->bBgnd  = aesBG_DFLT ;
      this->tFgnd  = aesFG_DFLT ;
      this->tBgnd  = aesBG_DFLT ;
   }
   void range ( WinPos& base, short& hgt, short& wid )
   {
      if ( base.row < 1 )  base.row = 1 ;
      if ( base.col < 1 )  base.col = 1 ;
      if ( hgt < 3 )       hgt = 3 ;
      if ( wid < 3 )       wid = 3 ;
   }

   WinPos orig ;                 // Box origin (terminal-relative)
   short hgt ;                   // Height of object (rows)
   short wid ;                   // Width of object (columns)

   //* Public Data Members *
   public:
   wchar_t title[gsMAXCHARS] ;   // Title to be displayed in top row of box
   LineType lnType ;             // line type for border
   aeSeq bFgnd ;                 // Border Foreground color attribute
   aeSeq bBgnd ;                 // Border Background color attribute
   aeSeq tFgnd ;                 // Interior Foreground color attribute
   aeSeq tBgnd ;                 // Interior Background color attribute

   friend class AnsiCmd ;  // AnsiCmd class has access to private members

   // Notes:
   // 'orig'   The 1-based offset from the upper-left corner of the terminal
   //          window to the upper-left corner of the box.
   // 'hgt'    Height of the box in rows.
   // 'wid'    Width of the box in columns.
   // 'title'  Title text (truncated if necessary to fit box width).
   // 'lnType' Member of enun LineType (limited to ltSINGLE or ltDUAL)
   // 'bFgnd'  Border foreground attribute
   // 'bBgnd'  Border background attribute
   // 'tFgnd'  Text foreground attribute
   // 'tBgnd'  Text background attribute
} ;   // AC_Box class

//*******************************************************************
//* Group of "Special Keycodes": These keycodes are translated from *
//* escape sequences or ASCII control codes and are used for cursor *
//* movement and text editing during user input.                    *
//*******************************************************************
const wchar_t wcsUARROW  = 0x002191 ;     // arrow pointng up     '↑'
const wchar_t wcsDARROW  = 0x002193 ;     // arrow pointng down   '↓'
const wchar_t wcsRARROW  = 0x002192 ;     // arrow pointng right  '→'
const wchar_t wcsLARROW  = 0x002190 ;     // arrow pointng left   '←'
const wchar_t wcsHOME    = 0x002196 ;     // Home key             '↖' (not ncurses)
const wchar_t wcsEND     = 0x002198 ;     // End key              '↘' (not ncurses)
const wchar_t wcsPGUP    = 0x002197 ;     // PageUp key           '↗' (not ncurses)
const wchar_t wcsPGDN    = 0x002199 ;     // PageDown key         '↙' (not ncurses)
const wchar_t wcsBKSP    = 0x0021B0 ;     // Backspace key        '↰' (not ncurses)
const wchar_t wcsTAB     = 0x0021B1 ;     // Tab key              '↱' (not ncurses)
const wchar_t wcsSTAB    = 0x0021B2 ;     // Shift + Tab key      '↲' (not ncurses)
const wchar_t wcsDELETE  = 0x0021B3 ;     // Delete key           '↳' (not ncurses)
const wchar_t wcsINSERT  = 0x0021A5 ;     // Insert key           '↥' (not ncurses)
const wchar_t wcsAENTER  = 0x0021A7 ;     // Alt + Enter key      '↧' (not ncurses)

//*******************************************************************
//* Line drawing characters and miscellaneous block characters.     *
//* These definitions are taken directly from the author's NcDialog *
//* API (NCurses.hpp).                                              *
//* --------------------------------------------------------------- *
//* The wcsXXXX group of character definitions are the wchar_t      *
//* character codes that correspond to the UTF-8 character encoding *
//* for line-drawing characters and other miscellaneous characters  *
//* found in the Linux/UNIX character set.                          *
//*******************************************************************

const wchar_t wcsSPACE   = 0x000020 ;     // space (blank)
const wchar_t wcsBLOCK   = 0x0025AE ;     // solid block
const wchar_t wcsDIAMOND = 0x0025C6 ;     // diamond (black)
const wchar_t wcsPATTERN = 0x002592 ;     // patterned block
const wchar_t wcsDEGREE  = 0x0000B0 ;     // degree symbol
const wchar_t wcsPLMINUS = 0x0000B1 ;     // plus-and-minus symbol
const wchar_t wcsBOARD   = 0x002592 ;     // pattern of squares
const wchar_t wcsLANTERN = 0x002603 ;     // light bulb
const wchar_t wcsLR      = 0x002518 ;     // drawing character - lower right corner
const wchar_t wcsUR      = 0x002510 ;     // drawing character - upper right corner
const wchar_t wcsUL      = 0x00250C ;     // drawing character - upper left corner
const wchar_t wcsLL      = 0x002514 ;     // drawing character - lower left corner
const wchar_t wcsINSECT  = 0x00253C ;     // drawing character - line intersection
const wchar_t wcsSCAN1   = 0x0023BA ;     // horizontal scanline #1
const wchar_t wcsSCAN3   = 0x0023BB ;     // horizontal scanline #3
const wchar_t wcsHORIZ   = 0x002500 ;     // drawing character - horizontal line
const wchar_t wcsSCAN7   = 0x0023BC ;     // horizontal scanline #7
const wchar_t wcsSCAN9   = 0x0023BD ;     // horizontal scanline #9
const wchar_t wcsLTEE    = 0x00251C ;     // drawing character - T for left-side connect
const wchar_t wcsRTEE    = 0x002524 ;     // drawing character - T for right-edge connect
const wchar_t wcsBTEE    = 0x002534 ;     // drawing character - T for bottom-edge connect
const wchar_t wcsTTEE    = 0x00252C ;     // drawing character - T for top-edge connect
const wchar_t wcsVERT    = 0x002502 ;     // drawing character - vertical line
const wchar_t wcsLEQUAL  = 0x002264 ;     // less-than-or-equal symbol
const wchar_t wcsGEQUAL  = 0x002265 ;     // greater-than-or equal symbol
const wchar_t wcsPI      = 0x0003C0 ;     // greek letter Pi
const wchar_t wcsNEQUAL  = 0x002260 ;     // not-equal-to symbol
const wchar_t wcsSTRLING = 0x0000A3 ;     // U.K Pound Sterling symbol
const wchar_t wcsBULLET  = 0x0000B7 ;     // bullet mark (centered dot)
//* Extra line-drawing characters defined in the Unicode character set.        *
const wchar_t wcsHORIZs  = wcsHORIZ ;  // single-line norm - horizontal line
const wchar_t wcsHORIZb  = 0x002501 ;  // single-line bold - horizontal line
const wchar_t wcsHORIZd  = 0x002550 ;  // dual-line        - horizontal line
const wchar_t wcsVERTs   = wcsVERT  ;  // single-line norm - vertical line
const wchar_t wcsVERTb   = 0x002503 ;  // single-line bold - vertical line
const wchar_t wcsVERTd   = 0x002551 ;  // dual-line        - vertical line
const wchar_t wcsULs     = wcsUL ;     // single-line norm - upper left corner 
const wchar_t wcsULb     = 0x00250F ;  // single-line bold - upper left corner
const wchar_t wcsULd     = 0x002554 ;  // dual-line        - upper left corner
const wchar_t wcsURs     = wcsUR ;     // single-line norm - upper right corner 
const wchar_t wcsURb     = 0x002513 ;  // single-line bold - upper right corner
const wchar_t wcsURd     = 0x002557 ;  // dual-line        - upper right corner
const wchar_t wcsLLs     = wcsLL ;     // single-line norm - lower left corner 
const wchar_t wcsLLb     = 0x002517 ;  // single-line bold - lower left corner
const wchar_t wcsLLd     = 0x00255A ;  // dual-line        - lower left corner
const wchar_t wcsLRs     = wcsLR ;     // single-line norm - lower right corner 
const wchar_t wcsLRb     = 0x00251B ;  // single-line bold - lower right corner
const wchar_t wcsLRd     = 0x00255D ;  // dual-line        - lower right corner
const wchar_t wcsLTEEs   = wcsLTEE  ;  // single-line norm - T for left-edge connect
const wchar_t wcsLTEEb   = 0x002523 ;  // single-line bold - T for left-edge connect
const wchar_t wcsLTEEd   = 0x002560 ;  // dual-line        - T for left-edge connect
const wchar_t wcsRTEEs   = wcsRTEE  ;  // single-line norm - T for right-edge connect
const wchar_t wcsRTEEb   = 0x00252B ;  // single-line bold - T for right-edge connect
const wchar_t wcsRTEEd   = 0x002563 ;  // dual-line        - T for right-edge connect
const wchar_t wcsBTEEs   = wcsBTEE  ;  // single-line norm - T for bottom-edge connect
const wchar_t wcsBTEEb   = 0x00253B ;  // single-line bold - T for bottom-edge connect
const wchar_t wcsBTEEd   = 0x002569 ;  // dual-line        - T for bottom-edge connect
const wchar_t wcsTTEEs   = wcsTTEE  ;  // single-line norm - T for top-edge connect
const wchar_t wcsTTEEb   = 0x002533 ;  // single-line bold - T for top-edge connect
const wchar_t wcsTTEEd   = 0x002566 ;  // dual-line        - T for top-edge connect
const wchar_t wcsINSECTs = wcsINSECT;  // single-line norm - line intersection
const wchar_t wcsINSECTb = 0x00254B ;  // single-line bold - line intersection
const wchar_t wcsINSECTd = 0x00256C ;  // dual-line        - line intersection
//* Dash (dotted) lines *
const wchar_t wcsHDASH2s = 0x00254C ;  // norm - 2-dash horizontal line
const wchar_t wcsHDASH2b = 0x00254D ;  // bold - 2-dash horizontal line
const wchar_t wcsVDASH2s = 0x00254E ;  // norm - 2-dash vertical line
const wchar_t wcsVDASH2b = 0x00254F ;  // bold - 2-dash vertical line
const wchar_t wcsHDASH3s = 0x002504 ;  // norm - 3-dash horizontal line
const wchar_t wcsHDASH3b = 0x002505 ;  // bold - 3-dash horizontal line
const wchar_t wcsVDASH3s = 0x002506 ;  // norm - 3-dash vertical line
const wchar_t wcsVDASH3b = 0x002507 ;  // bold - 3-dash vertical line
const wchar_t wcsHDASH4s = 0x002508 ;  // norm - 4-dash horizontal line
const wchar_t wcsHDASH4b = 0x002509 ;  // bold - 4-dash horizontal line
const wchar_t wcsVDASH4s = 0x00250A ;  // norm - 4-dash vertical line
const wchar_t wcsVDASH4b = 0x00250B ;  // bold - 4-dash vertical line

//* Miscellaneous 'special characters defined in the Unicode character set.    *
//* There are many additional text-mode drawing characters defined by the      *
//* Unicode standard. We define only a subset of them here. For a complete list*
//* visit the Unicode Consortium at  http://www.unicode.org - specifically the *
//* listings for 0x2500-0x257f and 0x2580-0x259F.                              *
const wchar_t wcsLTEEbv  = 0x002520 ;  // T for left-edge connect - bold vert. norm hori.
const wchar_t wcsLTEEbh  = 0x00251D ;  // T for left-edge connect - bold horiz. norm vert.
const wchar_t wcsLTEEdv  = 0x00255F ;  // T for left-edge connect - dual vert. single horiz.
const wchar_t wcsLTEEdh  = 0x00255E ;  // T for left-edge connect - dual horiz. single vert.

const wchar_t wcsRTEEbv  = 0x002528 ;  // T for right-edge connect - bold vert. norm hori.
const wchar_t wcsRTEEbh  = 0x002525 ;  // T for right-edge connect - bold horiz. norm vert.
const wchar_t wcsRTEEdv  = 0x002562 ;  // T for right-edge connect - dual vert. single horiz.
const wchar_t wcsRTEEdh  = 0x002561 ;  // T for right-edge connect - single vert. dual horiz.

const wchar_t wcsBTEEbv  = 0x002538 ;  // T for bottom-edge connect - bold vert. norm hori.
const wchar_t wcsBTEEbh  = 0x002537 ;  // T for bottom-edge connect - bold horiz. norm vert.
const wchar_t wcsBTEEdv  = 0x002568 ;  // T for bottom-edge connect - dual vert. single horiz.
const wchar_t wcsBTEEdh  = 0x002567 ;  // T for bottom-edge connect - single vert. dual horiz.

const wchar_t wcsTTEEbv  = 0x002530 ;  // T for top-edge connect - bold vert. norm hori.
const wchar_t wcsTTEEbh  = 0x00252F ;  // T for top-edge connect - bold horiz. norm vert.
const wchar_t wcsTTEEdv  = 0x002565 ;  // T for top-edge connect - dual vert. single horiz.
const wchar_t wcsTTEEdh  = 0x002564 ;  // T for top-edge connect - single vert. dual horiz.

const wchar_t wcsINSECTbv= 0x002542 ;  // line intersection - bold vert. norm horiz. 
const wchar_t wcsINSECTbh= 0x00253F ;  // line intersection - bold horiz. norm vert.
const wchar_t wcsINSECTdv= 0x00256B ;  // line intersection - dual vert. single horiz.
const wchar_t wcsINSECTdh= 0x00256A ;  // line intersection - single vert. dual horiz.
const wchar_t wcsBLOCKl  = 0x002591 ;  // block character - light pattern
const wchar_t wcsBLOCKm  = 0x002592 ;  // block character - medium pattern
const wchar_t wcsBLOCKd  = 0x002593 ;  // block character - dark pattern
const wchar_t wcsBLOCKs  = 0x002588 ;  // block character - solid black
const wchar_t wcsWCIRCLE = 0x0025CB ;  // white circle  ( ○ )
const wchar_t wcsBCIRCLE = 0x0025CF ;  // black circle  ( ● )
const wchar_t wcsFISHEYE = 0x0025C9 ;  // black circle within a white circle  ( ◉ )
const wchar_t wcsBULLSEYE= 0x0025CE ;  // small circle within a larger circle ( ◎ )
const wchar_t wcsWDIAMOND= 0x0025C7 ;  // white diamond ( ◇ see also wcsDIAMOND above)
const wchar_t wcsDIAMONDS= 0x0025C8 ;  // black diamond within a white diamond  ( ◈ )

